/*
 * This file contains two routines for parsing 
 *	ParseStartup()	- Parses the startup file .xfibs, initialising
 *			  Hostname, portnumber, username, password, startup
 *			  commands (such as connect, who ready etc.) And
 *			  finally creates a menubar if appropriate.
 *	ParseLine()	- Parses whatever comes from the server
*/
#include <limits.h>
#include <string.h>
#include "xfibs.h"
#include "Fibw.h"
#include "FibwP.h"
#include "parse.h"
#include "config.h"
#include "callbacks.h"
#include "graphics.h"

#define LINE_LENGTH	255
#define ITEM_STRINGS	400

char	 host_name[STRING_LENGTH];
int	 port_number;
char	 user_name[STRING_LENGTH];
char	 pass_word[STRING_LENGTH];
char	*item_string[ITEM_STRINGS];	/* pointers to the command strings */

Widget	 menu_item, pull_down, separator;
int	 item_counter = 0;
int	 on_login  = 0;
int	 autologin = 0;
int	 showNumbers = 1;
int	 onEdge = 0;
int	 logMode = -1;


struct logincmd {		/* line to send to server at login time */
    char cmd[STRING_LENGTH];
    struct logincmd *next;
};

extern int	 number_of_pending_entries;
extern String	 pending_list[];
struct logincmd	*first_logincmd=NULL;


static void add_to_login_cmds(line)
String line;
{
    struct logincmd *p;

    /* Don't add blank lines */
    if (*line == '\0')
	return;

    if (!first_logincmd) {
	first_logincmd = XtNew(struct logincmd);
	strcpy(first_logincmd->cmd, line);
	first_logincmd->next = (struct logincmd *)NULL;
    }
    else {
	p = first_logincmd;
	while (p->next)
	    p = p->next;

	p->next = XtNew(struct logincmd);
	strcpy(p->next->cmd, line);
	p->next->next = (struct logincmd *)NULL;
    }
}


void do_login_cmds()
{
    struct logincmd	*p;

    initPendingList();

    p = first_logincmd;
    if (p) {
	while (p) {
	    add_to_pending_list(p->cmd);
	    p = p->next;
	}
    }
}


static void sudden_death(filename, lineno, text)
char * filename;
int lineno;
char * text;
{
    fprintf(stderr,"%s at line %i in file %s\n",text, lineno, filename);
    exit(0);
}


static int get_word(word, line)
char *word, *line;
{
    /* copies characters from line to word. If line is started with a quote
     * copy continues to next quote, else to next space 
     * Returns 1 if word consists of several words, 0 if not
    */

    int i = 0;
    
    if (line[0] == '"') {
	while((line[i + 1] != '"') && (line[i + 1] != '\0')) {
	    word[i] = line[i + 1];
	    i++;
	}
	word[i] = '\0';
	return 1;
    }
    else {
	while( (!isspace(line[i])) && (line[i] != '\0')) {
	    word[i] = line[i];
	    i++;
	}
	word[i] = '\0';
	return 0;
    }
}


static void split_line(line, word1, word2, word3, word4)
char *line, *word1, *word2, *word3, *word4;
{
    /* Runs through line and splits it up in 4 words.
     * Substring enclosed in double-quotes (") are regarded as one word
     *
     * None of the commands need more than three parameters, so arg1 to arg3
     * is enough. Any text beyond that is simply ignored.
    */

    char * lastchar = &line[strlen(line)];
    word1[0] = '\0';
    word2[0] = '\0';
    word3[0] = '\0';
    word4[0] = '\0';

    while (isspace(line[0]) && (line[0] != '\0'))
	line++;
    if (get_word(word1, line))
	line += strlen(word1) + 2;
    else
	line += strlen(word1) + 1;
    while (isspace(line[0]) && (line[0] != '\0'))
	line++;
    if (line < lastchar) {
	if (get_word(word2, line))
	    line += strlen(word2) + 2;
	else
	    line += strlen(word2) + 1;
	while (isspace(line[0]) && (line[0] != '\0'))
	    line++;
	if (line < lastchar) {
	    if (get_word(word3, line))
		line += strlen(word3) + 2;
	    else
		line += strlen(word3) + 1;
	    while (isspace(line[0]) && (line[0] != '\0'))
		line++;
	    if (line < lastchar) {
		get_word(word3, line);
	    }
	}
    }
}


static int parseLine(filename, lineno, line)
char	*filename;
int	 lineno;
char	*line;
{
    char	command[STRING_LENGTH];
    char	arg1[STRING_LENGTH], arg2[STRING_LENGTH], arg3[STRING_LENGTH];
    Arg		args[3];
    int		temp;
    XmString	str;

    split_line(line, command, arg1, arg2, arg3);

    if (strcasecmp(command, "host") == 0) {
	if (!strcpy(host_name, arg1))
	    sudden_death(filename, lineno, "Missing host name");
    }
    else if (strcasecmp(command, "port") == 0) {
	if ((!arg1[0]) || ( !(port_number = atoi(arg1)) ) )
	    sudden_death(filename, lineno, "Strange port number");
    }
    else if (strcasecmp(command, "user") == 0) {
	if (!strcpy(user_name, arg1))
	    sudden_death(filename, lineno, "Missing user name");
    }
    else if (strcasecmp(command, "passwd") == 0) { 
	if (!strcpy(pass_word, arg1)) 
	    sudden_death(filename, lineno, "Missing password");
    }
    else if (strcasecmp(command,"gag") == 0) {
	if (arg1[0])
	    add_to_gag_list(arg1);
	else
	    sudden_death(filename, lineno, "Weird gag command");
    }
    else if ((strcasecmp(command, "colour") == 0) ||
	     (strcasecmp(command, "color" ) == 0))
	usecolor = 1;
    else if ((strcasecmp(command, "nocolour") == 0) ||
	     (strcasecmp(command, "nocolor" ) == 0))
	usecolor = 0;

    else if (strcasecmp(command, "scroll") == 0)
	scroll = 1;
    else if (strcasecmp(command, "noscroll") == 0)
	scroll = 0;

    else if (strcasecmp(command, "columns") == 0) {
	if ((!arg1[0]) || (!(temp = atoi(arg1))))
	    sudden_death(filename, lineno, "Strange columns number");
	columns = temp;
    }
    else if (strcasecmp(command, "fibslines") == 0) {
	if ((!arg1[0]) || (!(temp = atoi(arg1))))
	    sudden_death(filename, lineno, "Strange fibslines number");
	FIBSrows = temp;
    }
    else if (strcasecmp(command, "typelines") == 0) {
	if ((!arg1[0]) || (!(temp = atoi(arg1))))
	    sudden_death(filename, lineno, "Strange typelines number");
	typeRows = temp;
    }

    else if (strcasecmp(command, "hilite") == 0) {
	if (arg1[0])
	    add_to_hilite_list(arg1);
	else
	    sudden_death(filename, lineno, "Strange hilite command");
    }

    else if (strcasecmp(command, "autologin") == 0)
	autologin = 1;

    else if (strcasecmp(command, "on_login") == 0)
	on_login = 1;

    else if (strcasecmp(command, "logfile") == 0)
	openLog(arg1);
    else if (strcasecmp(command, "nologfile") == 0)
	closeLog();

    else if (strcasecmp(command, "logpath") == 0) {
	if (arg1[0])
	    strcpy(logpath, arg1);
	else 
	    sudden_death(filename, lineno, "Strange logpath command");
    }
    else if (strcasecmp(command, "nologpath") == 0)
	    strcpy(logpath, "");

    else if (strcasecmp(command, "log") == 0) {
	if (arg2[0]) {
	    if (strcasecmp(arg2, "+") != 0)
		sudden_death(filename, lineno, "Strange log style+");
	}
	if (arg1[0]) {
	    if (strcasecmp(arg1, "all") == 0)
		temp = LOGALL;
	    else if (strcasecmp(arg1, "input") == 0)
		temp = LOGINPUT;
	    else if (strcasecmp(arg1, "play") == 0)
		temp = LOGPLAY;
	    else if (strcasecmp(arg1, "board") == 0)
		temp = LOGBOARDS;
	    else if (strcasecmp(arg1, "none") == 0)
		temp = LOGNONE;
	    else
		sudden_death(filename, lineno, "Strange log style");

	    if (arg2[0])
		logMode |= temp;
	    else logMode = temp;
	}
	else
	    sudden_death(filename, lineno, "Strange log command");
    }

    else if (strcasecmp(command, "numbers") == 0)
	showNumbers = 1;
    else if (strcasecmp(command, "nonumbers") == 0)
	showNumbers = 0;

    else if ((strcasecmp(command, "frame") == 0) ||
	     (strcasecmp(command, "noside" ) == 0))
	onEdge = 1;
    else if ((strcasecmp(command, "noframe") == 0) ||
	     (strcasecmp(command, "side" ) == 0))
	onEdge = 0;

    /* These are purely for people who used Snoopy's patch on xfibs07 */
    /* add processing to toggle blank lines off and on */
    else if (strcasecmp(command,"blanks") == 0)
       blanklines = 1;
    else if (strcasecmp(command,"noblanks") == 0)
       blanklines = 0;
    /* add processing to toggle echoing of input commands off and on */
    else if (strcasecmp(command, "echo") == 0)
       echo_input = 1;
    else if (strcasecmp(command, "noecho") == 0)
       echo_input = 0;

    else
	return 0;

    return 1;
}

void setUserFlags()
{
    FibwWidget	fw = (FibwWidget) bgboard;

    use_numbers(showNumbers);
    use_board(onEdge);
    fw->fibw.logStyle = logMode;
}

static int parseMenu(parent, menu, filename, has_menu, lineno, line)
Widget	 parent;
Widget	*menu;
char	*filename;
int	*has_menu;
int	 lineno;
char	*line;
{
    char	command[80];
    char	arg1[80], arg2[80], arg3[80];
    Arg		args[3];
    XmString	str;

    split_line(line, command, arg1, arg2, arg3);

    if (strcasecmp(command, "MENU") == 0) {
	if (!arg1[0])
	    sudden_death(filename, lineno, "Missing menu name");
	if (!*has_menu) {
	    /* Create menu bar */
	    XtSetArg(args[0], XmNleftAttachment, XmATTACH_FORM);
	    XtSetArg(args[1], XmNrightAttachment, XmATTACH_FORM);
	    XtSetArg(args[2], XmNtopAttachment, XmATTACH_FORM);
	    *menu = XmCreateMenuBar(parent, "menuBar", args,3);
	    XtManageChild(*menu);
	    *has_menu=1;
	}
	/* Now create the pull-down menu */
	pull_down = XmCreatePulldownMenu(*menu, arg1, NULL, 0);
	str = XmStringCreateSimple(arg1);

	if (!str)
	    sudden_death(filename, lineno, "Couldn't create menu string");

	if (arg2[0]) /* Mnemonic ? */
	    (void *)XtVaCreateManagedWidget(arg1,
		xmCascadeButtonGadgetClass, *menu,
		XmNsubMenuId, pull_down,
		XmNlabelString, str,
		XmNmnemonic, arg2[0],
		NULL);
	else
	    (void *)XtVaCreateManagedWidget(arg1,
		xmCascadeButtonGadgetClass, *menu,
		XmNsubMenuId, pull_down,
		XmNlabelString, str,
		NULL);
	XmStringFree(str);
    }
    else if (strcasecmp(command,"item")==0) {
	if (!arg1[0])
	    sudden_death(filename, lineno, "Missing item name");
	if (!*has_menu) 
	    sudden_death(filename, lineno, "Missing 'Menu' before 'Item'");
	if (arg3[0]) {
	    /* There's text in arg3, so a menu accelerator has been given */
	    str = XmStringCreateSimple(arg1);
	    if (!str)
		sudden_death(filename, lineno, "Couldn't create item string");
	    menu_item = XtVaCreateManagedWidget(arg1,
		    xmPushButtonGadgetClass, pull_down,
		    XmNlabelString, str,
		    XmNmnemonic, arg2[0],NULL);
	    XmStringFree(str);
	    
	    /* Add a call-back function */
	    /* We've already checked that arg3 exists, so arg2 has to exist */
	    if (item_counter == ITEM_STRINGS)
		fprintf(stderr, "Too many menu items at line %i in %s",
		    lineno, filename);
	    else {
		item_string[item_counter] = XtNewString(arg3);
		if (!item_string[item_counter])
		    sudden_death(filename, lineno, "Couldn't create item");
		XtAddCallback(menu_item, XmNactivateCallback,
		    (XtCallbackProc)mnu_callback, item_string[item_counter]);
		item_counter++;
	    }
	}
	else { /* No accelerator given */

	    str = XmStringCreateSimple(arg1);
	    if (!str)
		sudden_death(filename, lineno, "Couldn't create item string");
	    menu_item = XtVaCreateManagedWidget(arg1,
		    xmPushButtonGadgetClass, pull_down,
		    XmNlabelString, str,
		    NULL);
	    XmStringFree(str);

	    if (!arg2[0])
		sudden_death(filename, lineno, "Missing item text");
	    if (item_counter == ITEM_STRINGS) 
		fprintf(stderr,"Too many menu items at line %i in %s", 
			lineno,filename);
	    else {
		item_string[item_counter] = XtNewString(arg2);
		if (!item_string[item_counter])
		    sudden_death(filename, lineno, "Couldn't create item");
		XtAddCallback(menu_item, XmNactivateCallback,
		    (XtCallbackProc)mnu_callback, item_string[item_counter]);
		item_counter++;
	    }
	}
    }
    else if (strcasecmp(command,"separator")==0) {
	if (!*has_menu)
	    sudden_death(filename, lineno, "Missing 'Menu' before 'Separator'");
	separator = XmCreateSeparatorGadget(pull_down, "separator", args, 0);
	XtManageChild(separator);
    }
    else		/* Not a menu line */
	return 0;

    return 1;
}


void parse_startup(parent, menu, justMenus)
Widget	 parent;
Widget	*menu;
int	 justMenus;
/* Need the widget stuff for creating menu bar, as the case may be */
{
    FILE	 *startup_file;

#ifdef PATH_MAX
    char	 rcfile[PATH_MAX];
#else
#ifdef _POSIX_PATH_MAX
    char	 rcfile[_POSIX_PATH_MAX];
#else
    char	 rcfile[STRING_LENGTH];
#endif
#endif

    char	 line[LINE_LENGTH];
    char	*lineptr;
    int		 lineno = 0;
    int		 has_menu = 0;

    if (!justMenus) {
	user_name[0] = '\0';
	pass_word[0] = '\0';
	strcpy(host_name, DEFAULT_HOST);
	port_number = DEFAULT_PORT;
    }

    strcpy(rcfile, getenv("HOME"));
    strcat(rcfile, "/");
    strcat(rcfile, STARTUPFILE);

    if ((startup_file = fopen(rcfile, "r")) == NULL) {
	fprintf(stderr, "Couldn't find startup-file!\n");
	fprintf(stderr, "Using standard setup.\n");
	parseMenu(parent, menu, "defaults", &has_menu, 1, "Menu File F");
	parseMenu(parent, menu, "defaults", &has_menu, 2, "ITEM Connect C connect");
	parseMenu(parent, menu, "defaults", &has_menu, 3, "ITEM Disconnect D disconnect");
	parseMenu(parent, menu, "defaults", &has_menu, 4, "SEPARATOR");
	parseMenu(parent, menu, "defaults", &has_menu, 5, "ITEM Exit x exit");
	return;
    }

    while (fgets(line, LINE_LENGTH, startup_file)) {
	lineptr = &line[0];
	lineno++;
	while((isspace((int) *lineptr)) && (*lineptr != '\0'))
	    lineptr++;
	if (*lineptr != '#') { /* comment ? */
	    if (!justMenus && on_login) { /* add to startup commands */
		add_to_login_cmds(lineptr);
	    }
	    else {
		if (lineptr[strlen(lineptr) - 1] == '\n')
		    lineptr[strlen(lineptr) - 1] =  '\0';
		if (strlen(lineptr) > 0) {
		    if (!parseMenu(parent, menu, rcfile, &has_menu, lineno, lineptr) && !justMenus)
			if (!parseLine(rcfile, lineno, lineptr)) {
			    fprintf(stderr, "Funny looking line at %i in %s:\n",
				lineno, rcfile);
			    fprintf(stderr,"     %s\n\n", lineptr);
			}
		}
	    }
	}
    }
    fclose(startup_file);
}
