#include "xfibs.h"
#include "FibwP.h"
#include "graphics.h"
#include <stdio.h>
#include "config.h"
#include "white.stpl"
#include "black.stpl"
#include "board.stpl"
#include "frame.stpl"
#include "bkgnd.stpl"


int usecolor = -1;

Pixmap	white_stipple, black_stipple, bkgnd_stipple, board_stipple,
	frame_stipple;
int	whites_height, blacks_height, bkgnds_height, boards_height,
	frames_height, whites_width, blacks_width, bkgnds_width, boards_width,
	frames_width;

void get_drawgc(fw)
FibwWidget fw;
{
    XGCValues values;
    XtGCMask mask = 0;

    fw->fibw.draw_gc = XtGetGC((Widget) fw, mask, &values);
}


void set_font(fw)
FibwWidget fw;
{
    Display *dpy = XtDisplay((Widget)fw);

    XSetFont(dpy, fw->fibw.draw_gc, fw->fibw.font);
    fw->fibw.pFont = XQueryFont(dpy, fw->fibw.font);
    fw->fibw.textheight = (fw->fibw.pFont->max_bounds.ascent +
	      fw->fibw.pFont->max_bounds.descent);
}


static void set_color(fw, pix)
FibwWidget fw;
Pixel pix;
{
    XSetForeground(XtDisplay((Widget)fw), fw->fibw.draw_gc, pix);
}


static void use_stipple(fw,stp)
FibwWidget fw;
Pixmap *stp;
{
    Display	*dpy = XtDisplay((Widget)fw);

    XSetBackground(dpy, fw->fibw.draw_gc, WhitePixel(dpy, DefaultScreen(dpy)));
    XSetForeground(dpy, fw->fibw.draw_gc, BlackPixel(dpy, DefaultScreen(dpy)));
    XSetFillStyle(dpy, fw->fibw.draw_gc, FillOpaqueStippled);
    XSetStipple(dpy, fw->fibw.draw_gc, *stp);
}


static void use_white(fw)
FibwWidget fw;
{
    Display	*dpy = XtDisplay((Widget)fw);

    XSetForeground(dpy, fw->fibw.draw_gc, WhitePixel(dpy, DefaultScreen(dpy)));
}

static void use_black(fw)
FibwWidget fw;
{
    Display	*dpy = XtDisplay((Widget)fw);

    XSetForeground(dpy, fw->fibw.draw_gc, BlackPixel(dpy, DefaultScreen(dpy)));
}

static void use_fore(fw)
FibwWidget fw;
{
    Display	*dpy = XtDisplay((Widget)fw);

    XSetForeground(dpy, fw->fibw.draw_gc, fw->fibw.foreground.pixel);
}

static void use_back(fw)
FibwWidget fw;
{
    Display	*dpy = XtDisplay((Widget)fw);

    XSetForeground(dpy, fw->fibw.draw_gc, fw->fibw.background.pixel);
}


/*
 * Pixmap functions, drawing into,  creating, etc...
 */
 
/*
 * Below are a lot of small routines used to calculate the position and
 * size of different board elements. Defined as functions so one may
 * easily change them if one doesn't care for the present look of the board
 */
 
 
/* gets the y_offset needed to display text in the graphics window. */
static int get_board_y_offset(fw)
FibwWidget fw;
{
    if (fw->fibw.numbers)
	return(3*fw->fibw.textheight/2);
    else return(fw->fibw.textheight/2);
}

/* Gets the x-offset of the board */
static int get_board_x_offset(fw)
FibwWidget fw;
{
    return(fw->fibw.piece_size/2); 
}


/* Get the frame size */
static int get_frame_size(fw)
FibwWidget fw;
{
int	temp;
    
    temp = (fw->fibw.pixmap_height - 2*fw->fibw.board_y_offset)/26;   
    if (fw->fibw.onFrame)
	return max(temp, fw->fibw.textheight + 4);
    else return temp;
}

/* Get a sensible size for the pieces */
static int get_piece_size(fw)
FibwWidget fw;
{
int	temp;

    temp = fw->fibw.pixmap_height - 2*fw->fibw.board_y_offset;

    /* Since onFrame adds + 4 to textheight, add that*2 to offset */
    if (fw->fibw.onFrame)
	temp -= 8;

    return (temp / 13);
}


/* Get the position for the Home area */
static int get_home_offset(fw)
FibwWidget fw;
{
    
    if (fw->fibw.fibs_rec.direction == 1)
	return(fw->fibw.board_x_offset + fw->fibw.frame_size);
    else
	return(fw->fibw.board_x_offset + 12*fw->fibw.piece_size +
		4*fw->fibw.frame_size);
}

/* Gets the x-offset of the playing field */
static int get_playfield_offset(fw)
FibwWidget fw;
{
    
    if (fw->fibw.fibs_rec.direction == 1) 
	return(2*fw->fibw.frame_size + fw->fibw.board_x_offset +
		fw->fibw.piece_size);
    else
	return(fw->fibw.frame_size + fw->fibw.board_x_offset);   
}


void create_pixmap(fw)
FibwWidget fw;
{
    Display	*dpy = XtDisplay(fw);
    Drawable	 drawable = RootWindow(dpy,DefaultScreen(dpy));
    unsigned int depth = DefaultDepthOfScreen(XtScreen(fw));

    fw->fibw.board_pixmap = XCreatePixmap(dpy,drawable,
	fw->fibw.pixmap_width, fw->fibw.pixmap_height,depth);
    fw->fibw.back_pixmap = XCreatePixmap(dpy,drawable,
	fw->fibw.pixmap_width, fw->fibw.pixmap_height,depth);
    /* init pixmap pointers to NULL so they will get allocated correctly */
    fw->fibw.dice_pixmap  = (Pixmap)NULL;
    fw->fibw.cube_pixmap  = (Pixmap)NULL;
    fw->fibw.small_pixmap = (Pixmap)NULL;

    return;
}

void completeRedraw(fw)
FibwWidget fw;
{
    /* complete redraw */
    if (fw->fibw.board_pixmap)
	XFreePixmap(XtDisplay(fw), fw->fibw.board_pixmap);
    create_pixmap(fw);
    draw_into_pixmap(fw);
    Redisplay((Widget)fw,NULL);
}

void use_colors(fw)
FibwWidget fw;
{
    Display *dpy = XtDisplay(fw);

    if (!usecolor) { /* Set all colors to black and white */
	XSetBackground(dpy, fw->fibw.draw_gc, BlackPixel(dpy, DefaultScreen(dpy)));
	XSetForeground(dpy, fw->fibw.draw_gc, WhitePixel(dpy, DefaultScreen(dpy)));
    }
    completeRedraw(fw);
}


void use_numbers(flag)
int flag;
{
    FibwWidget	 fw = (FibwWidget) bgboard;
    Display	*dpy = XtDisplay(fw);

    fw->fibw.numbers = flag;

    completeRedraw(fw);
}

void use_board(flag)
int flag;
{
    FibwWidget	 fw = (FibwWidget) bgboard;
    Display	*dpy = XtDisplay(fw);

    fw->fibw.onFrame = flag;

    completeRedraw(fw);
}


static int create_included_stipples(fw, stip, width, height, bm_bits,
				    bm_width, bm_height)
FibwWidget	 fw;
Pixmap		*stip;
unsigned int	*width, *height;
char		*bm_bits;
int		 bm_width, bm_height;
{
    Display *dpy = XtDisplay(fw);
    Drawable drawable = RootWindow(dpy, DefaultScreen(dpy));

    *stip = XCreateBitmapFromData(dpy, drawable, bm_bits, bm_width, bm_height);
    if (*stip == (Pixmap)NULL)
	return 0;

    *width = bm_width;
    *height = bm_height;
    return 1;
}


void create_stipples(fw)
FibwWidget fw;
{
    /* Must be able to create _all_ stipples to continue */
    if ((!create_included_stipples(fw, &bkgnd_stipple,
	(unsigned *)&bkgnds_width, (unsigned *)&bkgnds_height, 
	bkgnd_bits, bkgnd_width, bkgnd_height))			||
      (!create_included_stipples(fw, &board_stipple,
	(unsigned *)&boards_width, (unsigned *)&boards_height, 
	board_bits, board_width, board_height))			||
      (!create_included_stipples(fw, &frame_stipple,
	(unsigned *)&frames_width, (unsigned *)&frames_height, 
	frame_bits, frame_width, frame_height))			||
      (!create_included_stipples(fw, &black_stipple,
	(unsigned *)&blacks_width, (unsigned *)&blacks_height, 
	black_bits, black_width, black_height))			||
      (!create_included_stipples(fw, &white_stipple,
	(unsigned *)&whites_width, (unsigned *)&whites_height, 
	white_bits, white_width, white_height)))			{
	fprintf(stderr,"Couldn't create stipples! Exiting...\n");
	exit(1);
    }
}

void drawButton(fw, string, points, x, y, width, height, offset)
FibwWidget	 fw;
char		*string, *points;
int		 x, y, width, height, offset;
{
    Display	*dpy = XtDisplay((Widget)fw);
    void	(*color1)(), (*color2)();	/* Mono color routines */

    color1 = (usecolor) ? &use_back : &use_white;
    color2 = (usecolor) ? &use_fore : &use_black;

    if (!usecolor)
	XSetFillStyle(dpy, fw->fibw.draw_gc, FillSolid);

    color1(fw);
    XFillRectangle(dpy, fw->fibw.board_pixmap, fw->fibw.draw_gc,
	x, y, width, height + offset);

    color2(fw);
    XDrawRectangle(dpy, fw->fibw.board_pixmap, fw->fibw.draw_gc,
	x, y, width, height + offset);
    XDrawRectangle(dpy, fw->fibw.board_pixmap, fw->fibw.draw_gc,
	x - 1, y - 1, width + 2, height + offset + 2);
    XDrawString(dpy, fw->fibw.board_pixmap, fw->fibw.draw_gc,
	x + 4, y + height - 4, string, strlen(string));
    /* If there is a second line print it.. */
    if (points)
	XDrawString(dpy, fw->fibw.board_pixmap, fw->fibw.draw_gc,
	    x + 4, y + height + offset - 4, points, strlen(points));
    /* set_color(fw, colour); */
    use_white(fw);
    XDrawLine(dpy, fw->fibw.board_pixmap, fw->fibw.draw_gc,
	x, y, x + width, y);
    XDrawLine(dpy, fw->fibw.board_pixmap, fw->fibw.draw_gc,
	x - 1, y - 1, x + width + 1, y - 1);
    XDrawLine(dpy, fw->fibw.board_pixmap, fw->fibw.draw_gc,
	x, y, x, y + height + offset);
    XDrawLine(dpy, fw->fibw.board_pixmap, fw->fibw.draw_gc,
	x - 1, y - 1, x - 1, y + height + offset + 1);
}

/* Style 0 is the normal name-at-the-side-of-the-board style
 * Style 1 is Snoopy's on-the-frame-of-the-board style
 */
static void draw_text(fw, style)
FibwWidget	fw;
int		style;
{
    Display	*dpy = XtDisplay((Widget)fw);
    char	 name[40], string[100];
    char	 points[20];
    void	(*color1)(), (*color2)();	/* Mono color routines */
    int		 boardheight = 12*fw->fibw.piece_size;
    int		 boardwidth  =  6*fw->fibw.piece_size;
    int		 xPos, yPos, xOffset, skew, width;
    FIBS_Rec	*pFIBS = &fw->fibw.fibs_rec;

    color1 = (usecolor) ? &use_back : &use_white;
    color2 = (usecolor) ? &use_fore : &use_black;

    /* Find the right-hand edge of the board */
    xPos = fw->fibw.board_x_offset + 2*boardwidth + 5*fw->fibw.frame_size +
	    fw->fibw.piece_size;
    if (style == 1) {
	if (fw->fibw.fibs_rec.direction == -1)
	    xOffset = 0;
	else	/* All the way back to the left hand edge */
	    xOffset = fw->fibw.board_x_offset - xPos;
    }
    /* Add in offset to skip over cube */
    else xOffset = fw->fibw.frame_size + 2*fw->fibw.piece_size;

    yPos = fw->fibw.board_y_offset;

    if (style == 1 && fw->fibw.fibs_rec.direction == -1)
	sprintf(string,"%s : %i", pFIBS->opponentname, pFIBS->opponent_got);
    else sprintf(string,"%i : %s", pFIBS->opponent_got, pFIBS->opponentname);
    width = XTextWidth(fw->fibw.pFont, string, strlen(string)) + 8;
    /* Skew the text back into the board, so that it butts against the edge */
    skew = (style == 1 && fw->fibw.fibs_rec.direction == -1) ? width : 0;

    drawButton(fw, string, NULL, xPos + xOffset - skew, yPos,
	width, fw->fibw.textheight + 4, 0);

    /* Line up with the bottom of the frame */
    yPos = fw->fibw.board_y_offset + boardheight + 2*fw->fibw.frame_size -
		(fw->fibw.textheight + 4);
    if (style == 1 && fw->fibw.fibs_rec.direction == -1)
	sprintf(string,"%s : %i", pFIBS->playername, pFIBS->player_got);
    else sprintf(string,"%i : %s", pFIBS->player_got, pFIBS->playername);
    width = XTextWidth(fw->fibw.pFont, string, strlen(string)) + 8;
    skew = (style == 1 && fw->fibw.fibs_rec.direction == -1) ? width : 0;

    drawButton(fw, string, NULL, xPos + xOffset - skew, yPos,
	width, fw->fibw.textheight + 4, 0);

    yPos = fw->fibw.board_y_offset + (boardheight + fw->fibw.frame_size)/2;
    if (style == 1) {
	if (pFIBS->match_length == 9999)
	    strcpy(string, "UL");
	else sprintf(string, "%i", pFIBS->match_length);
	width = XTextWidth(fw->fibw.pFont, string, strlen(string)) + 8;
	width = max(width, fw->fibw.frame_size);
	skew = (style == 1 && fw->fibw.fibs_rec.direction == -1) ? width : 0;

	drawButton(fw, string, NULL, xPos + xOffset - skew, yPos,
	    width, fw->fibw.textheight + 4, 0);
    }
    else {
	if (strcmp("You", pFIBS->playername) == 0)
	    strcpy(string, "  Playing a");
	else strcpy(string, " Watching a");
	if (pFIBS->match_length == 9999) {
	    strcat(string, "n");
	    strcpy(points, "unlimited match");
	}
	else {
	    sprintf(points, "%i point ", pFIBS->match_length);
	    if (pFIBS->match_length == 1)
		strcat(points, "game");
	    else strcat(points, "match");
	}
	width = XTextWidth(fw->fibw.pFont, string, strlen(string));
	width = max(width, XTextWidth(fw->fibw.pFont, points, strlen(points)));

	drawButton(fw, string, points,
	    xPos + xOffset - skew, yPos - fw->fibw.textheight,
	    width + 9, fw->fibw.textheight + 4, fw->fibw.piece_size);
    }
    strncpy(name, pFIBS->playername, sizeof(name) - 5);
    sprintf(string, "%s : %i ", name, pFIBS->player_got);
    if (pFIBS->match_length == 9999)
	strcat(string, "(unlimited)");
    else {
	sprintf(points, "(%i)", pFIBS->match_length);
	strcat(string, points);
    }
    strncpy(name, pFIBS->opponentname, sizeof(name) - 5);
    sprintf(points," %i : ", pFIBS->opponent_got);
    strcat(string, points);
    strcat(string, name);
/* Anybody know why I can't call setTitle() here? This dumps core:
    setTitle(string);
 */
}


/* General routine to draw a quadrant of the board. If numBase is even,
 * we're drawing the lower half of the board. This means the point height
 * is inverted; the point colors are swapped; the point numbering decreases
 * instead of increases; and the x,y coords are obviously different.
 *
 * So all the stuff at the top works out the differences, then the loop just
 * writes everything for the quadrant.
 */
static void draw_quadrant(dpy, fw, xOffset, yOffset, numBase, revOffset)
Display		*dpy;
FibwWidget	 fw;
int		 xOffset, yOffset;
int		 numBase, revOffset;
{
    XPoint	 points[3];
    char	 posnum[3];
    int		 i, j, scalar;
    int		 pointHeight, textOffset;
    int		 point1, point2;
    Pixmap	*stipple1, *stipple2;		/* Pointers to stipple */
    Pixel	 color1, color2;		/* Color of points */
    void	(*mono1)(), (*mono2)();		/* Mono color routines */
    int		 yMin, yMax;

    if (fw->fibw.debug == 2)
	fprintf(stderr, "q @ %d, %d (%d %d)\n", xOffset, yOffset, numBase, revOffset);
    /* Bottom half of the board, point colours are swapped */
    stipple1 = (numBase % 2) ? &white_stipple : &black_stipple;
    stipple2 = (numBase % 2) ? &black_stipple : &white_stipple;
    color1 = (numBase % 2) ? fw->fibw.wpoint.pixel : fw->fibw.bpoint.pixel;
    color2 = (numBase % 2) ? fw->fibw.bpoint.pixel : fw->fibw.wpoint.pixel;
    mono1 = (numBase % 2) ? &use_black : &use_white;
    mono2 = (numBase % 2) ? &use_white : &use_black;
    point1 = (numBase % 2) ? numBase : revOffset;
    point2 = (numBase % 2) ? revOffset : numBase;

    /* Bottom half of the board goes up, not down */
    if (numBase % 2) {
	pointHeight =  fw->fibw.piece_size * 5;
	textOffset  = -fw->fibw.frame_size - fw->fibw.pFont->max_bounds.descent;
	scalar = 1;
    }
    else {
	pointHeight = -fw->fibw.piece_size * 5;
	textOffset  = fw->fibw.frame_size + fw->fibw.textheight;
	scalar = -1;
    }

    /* Y-coords are fixed for each quadrant */
    points[0].y = yOffset + fw->fibw.frame_size;
    points[1].y = points[0].y + pointHeight;
    points[2].y = points[0].y;

    yMin = min(points[0].y, points[0].y + pointHeight +
		fw->fibw.piece_size * scalar);
    yMax = max(points[0].y, points[0].y + pointHeight +
		fw->fibw.piece_size * scalar);

    for (i = 0; i < 6; i++) {
	points[0].x = xOffset + fw->fibw.piece_size * i;
	points[1].x = points[0].x + fw->fibw.piece_size / 2;
	points[2].x = points[0].x + fw->fibw.piece_size;

	if (!usecolor) 
	    use_stipple(fw, (i % 2) ? stipple1 : stipple2);
	else
	    set_color(fw, (i % 2) ? color1 : color2);

	XFillPolygon(dpy, fw->fibw.back_pixmap, fw->fibw.draw_gc,
		points, 3, Nonconvex, CoordModeOrigin);
	if (!usecolor) {
	    XSetFillStyle(dpy, fw->fibw.draw_gc, FillSolid);
	
	    if (i % 2)
		mono1(fw);
	    else mono2(fw);
	}
	else set_color(fw, (i % 2) ? color1 : color2);

	XDrawLines(dpy, fw->fibw.back_pixmap, fw->fibw.draw_gc,
		points, 3, CoordModeOrigin);

	/* Figure out the point number */
	if (fw->fibw.fibs_rec.direction == 1) 
	    j = point1 + i * scalar;
	else j = point2 + i * scalar;

	if (fw->fibw.numbers) {
	    /* Draw number */
	    sprintf(posnum," %i", j);

	    XDrawString(dpy, fw->fibw.back_pixmap, fw->fibw.draw_gc,
		(points[2].x + points[0].x) / 2
		-XTextWidth(fw->fibw.pFont, posnum, strlen(posnum)) / 2,
		points[0].y + textOffset, posnum, strlen(posnum));
	}

	if (fw->fibw.debug == 2)
	    fprintf(stderr, "%d >%d %d<,>%d %d<\n", j, points[0].x, points[2].x, yMin, yMax);
	fw->fibw.pos_rec[j].x1 = points[0].x;
	fw->fibw.pos_rec[j].x2 = points[2].x;
	fw->fibw.pos_rec[j].y1 = yMin;
	fw->fibw.pos_rec[j].y2 = yMax;
    }
}


/*
 * Draw the backgammon board,  and while we're at it - create a table of
 * coordinates used to quickly look up positiions on the board from
 * x-y coordinates (pixels) [Mostly done by draw_quadrant() now].
 */
static void draw_board(fw)
FibwWidget fw;
{
    Display	*dpy = XtDisplay((Widget)fw);
    
    char	posnum[3];
    int		i;
    int		xOffset, yOffset;
    int		boardheight = 12 * fw->fibw.piece_size;
    int		boardwidth  = 6 * fw->fibw.piece_size;

    /* draw the frame */
    if (!usecolor) 
	use_stipple(fw, &frame_stipple);
    else
	set_color(fw, fw->fibw.frame.pixel);
    /* The frame of the whole board */
    XFillRectangle(dpy, fw->fibw.back_pixmap, fw->fibw.draw_gc,
	    fw->fibw.board_x_offset, fw->fibw.board_y_offset, 
	    2*boardwidth + 5*fw->fibw.frame_size + fw->fibw.piece_size, 
	    boardheight + 2*fw->fibw.frame_size);
    if (!usecolor)
	XSetFillStyle(dpy, fw->fibw.draw_gc, FillSolid);

    /* draw the backgammon-board */	    
    if (!usecolor) 
	use_stipple(fw, &board_stipple);
    else
	set_color(fw, fw->fibw.base.pixel);
    /* The left-hand playing area */
    XFillRectangle(dpy, fw->fibw.back_pixmap, fw->fibw.draw_gc,
	    fw->fibw.playfield_offset,
	    fw->fibw.board_y_offset + fw->fibw.frame_size, 
	    boardwidth, boardheight);
    /* The right-hand playing area */
    XFillRectangle(dpy, fw->fibw.back_pixmap, fw->fibw.draw_gc, 
 	    fw->fibw.playfield_offset + boardwidth + fw->fibw.frame_size*2, 
	    fw->fibw.board_y_offset + fw->fibw.frame_size, 
	    boardwidth, boardheight);
    /* The home playing area */
    XFillRectangle(dpy, fw->fibw.back_pixmap, fw->fibw.draw_gc, 
	    fw->fibw.home_offset,
	    fw->fibw.board_y_offset + fw->fibw.frame_size, 
	    fw->fibw.piece_size, boardheight);
    if (!usecolor)
	XSetFillStyle(dpy, fw->fibw.draw_gc, FillSolid);

/* get the posision of the home fields, and the bars... */

    fw->fibw.pos_rec[26].x1 = fw->fibw.home_offset;
    fw->fibw.pos_rec[27].x1 = fw->fibw.home_offset;
    fw->fibw.pos_rec[26].x2 = fw->fibw.pos_rec[26].x1 + fw->fibw.piece_size;
    fw->fibw.pos_rec[27].x2 = fw->fibw.pos_rec[27].x1 + fw->fibw.piece_size;
    fw->fibw.pos_rec[0].x1  = fw->fibw.playfield_offset + boardwidth;
    fw->fibw.pos_rec[0].x2  = fw->fibw.pos_rec[0].x1 + fw->fibw.piece_size;
    fw->fibw.pos_rec[25].x1 = fw->fibw.pos_rec[0].x1;
    fw->fibw.pos_rec[25].x2 = fw->fibw.pos_rec[0].x2;
    if (fw->fibw.fibs_rec.direction == 1) {
	fw->fibw.pos_rec[25].y1 = fw->fibw.board_y_offset + fw->fibw.frame_size;
	fw->fibw.pos_rec[25].y2 = fw->fibw.pos_rec[25].y1 + boardwidth;
	fw->fibw.pos_rec[0].y1  = fw->fibw.pos_rec[25].y2;
	fw->fibw.pos_rec[0].y2  = fw->fibw.pos_rec[0].y1 + boardwidth;
    }
    else {
	fw->fibw.pos_rec[0].y1  = fw->fibw.board_y_offset + fw->fibw.frame_size;
	fw->fibw.pos_rec[0].y2  = fw->fibw.pos_rec[0].y1 + boardwidth;
	fw->fibw.pos_rec[25].y1 = fw->fibw.pos_rec[0].y2;
	fw->fibw.pos_rec[25].y2 = fw->fibw.pos_rec[25].y1 + boardwidth;
    }
    fw->fibw.pos_rec[26].y1 = fw->fibw.pos_rec[25].y1;
    fw->fibw.pos_rec[26].y2 = fw->fibw.pos_rec[25].y2;
    fw->fibw.pos_rec[27].y1 = fw->fibw.pos_rec[0].y1;
    fw->fibw.pos_rec[27].y2 = fw->fibw.pos_rec[0].y2;

    if (fw->fibw.debug >= 2) {
	fprintf(stderr, "%d >%d %d<,>%d %d<\n", 0, fw->fibw.pos_rec[0].x1,
		fw->fibw.pos_rec[0].x2, fw->fibw.pos_rec[0].y1,
		fw->fibw.pos_rec[0].y2);
	fprintf(stderr, "%d >%d %d<,>%d %d<\n", 25, fw->fibw.pos_rec[25].x1,
		fw->fibw.pos_rec[25].x2, fw->fibw.pos_rec[25].y1,
		fw->fibw.pos_rec[25].y2);
	fprintf(stderr, "%d >%d %d<,>%d %d<\n", 26, fw->fibw.pos_rec[26].x1,
		fw->fibw.pos_rec[26].x2, fw->fibw.pos_rec[26].y1,
		fw->fibw.pos_rec[26].y2);
	fprintf(stderr, "%d >%d %d<,>%d %d<\n", 27, fw->fibw.pos_rec[27].x1,
		fw->fibw.pos_rec[27].x2, fw->fibw.pos_rec[27].y1,
		fw->fibw.pos_rec[27].y2);
    }


    /* draw some lines to make the board look good... */
    if (!usecolor)
	use_black(fw);
    else
	use_fore(fw);
    /* rectangle around the whole board */
    XDrawRectangle(dpy, fw->fibw.back_pixmap, fw->fibw.draw_gc,
	    fw->fibw.board_x_offset, fw->fibw.board_y_offset, 
	    2*boardwidth + 5*fw->fibw.frame_size + fw->fibw.piece_size, 
	    boardheight + 2*fw->fibw.frame_size);

    /* left frame */
    XDrawRectangle(dpy, fw->fibw.back_pixmap, fw->fibw.draw_gc,
	fw->fibw.playfield_offset - 1,
	fw->fibw.board_y_offset + fw->fibw.frame_size - 1, 
	boardwidth + 2, boardheight + 1);
    /* right frame */
    XDrawRectangle(dpy, fw->fibw.back_pixmap, fw->fibw.draw_gc, 
	fw->fibw.playfield_offset + boardwidth + fw->fibw.frame_size*2 - 1, 
	fw->fibw.board_y_offset + fw->fibw.frame_size - 1, 
	boardwidth + 2, boardheight + 1);

    /* "inside" home */
    XDrawRectangle(dpy, fw->fibw.back_pixmap, fw->fibw.draw_gc, 
	    fw->fibw.home_offset,
	    fw->fibw.board_y_offset + fw->fibw.frame_size, 
	    fw->fibw.piece_size, boardheight);
    /* between each half */
    if (!usecolor)
	use_stipple(fw,&black_stipple);
    XFillRectangle(dpy, fw->fibw.back_pixmap, fw->fibw.draw_gc,
	    fw->fibw.playfield_offset + boardwidth +
		fw->fibw.frame_size - fw->fibw.frame_size/10, 
	    fw->fibw.board_y_offset, 
	    fw->fibw.frame_size/5, boardheight + fw->fibw.frame_size*2); 
    if (!usecolor)
	XSetFillStyle(dpy, fw->fibw.draw_gc, FillSolid);
	    
    /* draw the triangles */
    xOffset = fw->fibw.playfield_offset;
    yOffset = fw->fibw.board_y_offset;
    draw_quadrant(dpy, fw, xOffset, yOffset, 1, 13);

    xOffset = fw->fibw.playfield_offset + 2 * fw->fibw.frame_size + boardwidth;
    yOffset = fw->fibw.board_y_offset;
    draw_quadrant(dpy, fw, xOffset, yOffset, 7, 19);

    xOffset = fw->fibw.playfield_offset;
    yOffset = fw->fibw.board_y_offset + boardheight;
    draw_quadrant(dpy, fw, xOffset, yOffset, 12, 24);

    xOffset = fw->fibw.playfield_offset + 2 * fw->fibw.frame_size + boardwidth;
    yOffset = fw->fibw.board_y_offset + boardheight;
    draw_quadrant(dpy, fw, xOffset, yOffset, 6, 18);

    /* And copy background onto board_pixmap... */
    XCopyArea(XtDisplay(fw), fw->fibw.back_pixmap,
	fw->fibw.board_pixmap, fw->fibw.draw_gc,
	0, 0, fw->fibw.pixmap_width, fw->fibw.pixmap_height, 0, 0);

    if (fw->fibw.onFrame)
	draw_text(fw, fw->fibw.onFrame);
}


void draw_into_pixmap(fw)
FibwWidget fw;
{
    Display *dpy = XtDisplay(fw);
    int i;

    /* First fill in with background color */
    if (!usecolor)
	use_stipple(fw, &bkgnd_stipple);
    else
	set_color(fw, fw->fibw.table.pixel);

    XFillRectangle(dpy, fw->fibw.back_pixmap, fw->fibw.draw_gc,
	0, 0, fw->fibw.pixmap_width, fw->fibw.pixmap_height);
    if (!usecolor)
	XSetFillStyle(dpy, fw->fibw.draw_gc, FillSolid);
    /* before we do anything else, let's get the positions of different
	oard elements */
    fw->fibw.board_y_offset	= get_board_y_offset(fw);
    fw->fibw.piece_size		= get_piece_size(fw);
    fw->fibw.board_x_offset	= get_board_x_offset(fw);
    fw->fibw.frame_size		= get_frame_size(fw);
    fw->fibw.home_offset	= get_home_offset(fw);
    fw->fibw.playfield_offset	= get_playfield_offset(fw);
    if (fw->fibw.debug == 2)
	fprintf(stderr, "w, h: %d, %d. x, y: %d, %d. pc %d, fm %d, hm %d, fld %d\n",
	    fw->fibw.pixmap_width, fw->fibw.pixmap_height,
	    fw->fibw.board_x_offset, fw->fibw.board_y_offset,
	    fw->fibw.piece_size, fw->fibw.frame_size,
	    fw->fibw.home_offset, fw->fibw.playfield_offset);
    /* now we're ready to do the drawing... */
    draw_board(fw);
    draw_into_dicepixmap(fw);
    for (i=0;i<=27;i++)
	draw_pieces(fw,i);
    draw_roll(fw);
    draw_dblcube(fw);
    draw_text(fw, fw->fibw.onFrame);
    return;
}


static XPoint *get_piece_position(fw, pos, number, point)
FibwWidget fw;
unsigned int pos;
unsigned int number;
XPoint *point;

/* 
 * pos is the position of the piece: 0-25 as in FIBW_Rec, 
 * 26 and 27 is home. 
 * number is number of pieces on specified spot
 * 
 * If fw-fibw.fibs_rec.player_home is 0 
 *    pos 1 is down right 
 * else
 *    pos 1 is up left
 * 
 * returns position in pixels in point-variable
 */
{    
    int boardwidth = 6*fw->fibw.piece_size;

    /* First do the x-direction (that's the easy part :)) */
    if ((pos == 0) || (pos == 25))   /* bar */
	point->x = fw->fibw.playfield_offset + boardwidth;
    if (pos >= 26)	     /* home */
	point->x = fw->fibw.home_offset;

    /* the following calculations could probably be done in a more 
	compact and obscure way. I'll leave it as it is... */
    if (fw->fibw.fibs_rec.direction == 1) {    /* home on left side */
	if ((pos >= 1) && (pos <= 6))
	    point->x = fw->fibw.playfield_offset +
		     fw->fibw.piece_size*(pos - 1);
	if ((pos >= 7) && (pos <= 12))
	    point->x = fw->fibw.playfield_offset + fw->fibw.piece_size*6 +
		     fw->fibw.frame_size*2 + fw->fibw.piece_size*(pos - 7);
	if ((pos >= 13) && (pos <= 18))
	    point->x = fw->fibw.playfield_offset + fw->fibw.piece_size*6 +
		     fw->fibw.frame_size*2 + fw->fibw.piece_size*(18 - pos);
	if ((pos >= 19) && (pos <= 24))
	    point->x = fw->fibw.playfield_offset +
		     fw->fibw.piece_size*(24 - pos);
    }
    else {
	if ((pos >= 1) && (pos <= 6))
	    point->x = fw->fibw.playfield_offset + fw->fibw.piece_size*6 +
		     fw->fibw.frame_size*2 + fw->fibw.piece_size*(6 - pos);
	if ((pos >= 7) && (pos <= 12))
	    point->x = fw->fibw.playfield_offset +
		     fw->fibw.piece_size*(12 - pos);
	if ((pos >= 13) && (pos <= 18))
	    point->x = fw->fibw.playfield_offset +
		     fw->fibw.piece_size*(pos - 13);
	if ((pos >= 19) && (pos <= 24))
	    point->x = fw->fibw.playfield_offset + fw->fibw.piece_size*6 +
		     fw->fibw.frame_size*2 + fw->fibw.piece_size*(pos - 19);
    }
	    
    /* then do the y-direction */
    if ((fw->fibw.fibs_rec.direction == 1) && 
	(((pos > 0) && (pos <= 12)) || (pos == 25) || (pos == 26))) {
	point->y = fw->fibw.board_y_offset + fw->fibw.frame_size +
		fw->fibw.piece_size*((number - 1)%5) +  
		fw->fibw.piece_size/10 +
		((number - 1)/5)*fw->fibw.piece_size/4;
    }
    if ((fw->fibw.fibs_rec.direction == 1) && 
	(((pos >= 13) && (pos <= 24)) || (pos == 27) || (pos == 0))) {
	point->y = fw->fibw.board_y_offset + fw->fibw.frame_size +
		fw->fibw.piece_size*12 -
		fw->fibw.piece_size*((number - 1)%5 + 1) -
		((number - 1)/5)*fw->fibw.piece_size/4;
    }
   if ((fw->fibw.fibs_rec.direction == -1) &&
	(((pos > 0) && (pos <= 12)) || (pos == 25) || (pos == 26))) {
	point->y = fw->fibw.board_y_offset + fw->fibw.frame_size +
		fw->fibw.piece_size*12 -
		fw->fibw.piece_size*((number - 1)%5 + 1) -
		((number - 1)/5)*fw->fibw.piece_size/4;
    }
    if ((fw->fibw.fibs_rec.direction == -1) &&
	(((pos >= 13) && (pos <= 24)) || (pos == 27) || (pos == 0))) {
	point->y = fw->fibw.board_y_offset+fw->fibw.frame_size+
		fw->fibw.piece_size*((number - 1)%5) +  
		fw->fibw.piece_size/10 +
		((number - 1)/5)*fw->fibw.piece_size/4;
    }
}

/*
 * Calculates,  based on where you clicked or released the mouse button, 
 * on which triangle the piece should be dropped,  or from which it 
 * should be removed.
 */
int get_position_number(fw, point)
FibwWidget fw;
XPoint *point;
{
    int		 i;
    PosRec	*pos;

    pos = fw->fibw.pos_rec;
    for (i = 0; i < 28; i++) {
	if (((pos + i)->x1 <= point->x) && ((pos + i)->x2 >= point->x) &&    
	    ((pos + i)->y1 <= point->y) && ((pos + i)->y2 >= point->y)) {
	    if (fw->fibw.debug == 2)
		fprintf(stderr, "%d: %d < %d > %d, %d < %d > %d\n", i,
		    (pos + i)->x1, point->x, (pos + i)->x2,
		    (pos + i)->y1, point->y, (pos + i)->y2);
	    return (i);
	    }
    }
    return (-1);	/* outside board */
}


void draw_one_piece_at_loc(fw, x, y, color)
FibwWidget fw;
int x, y, color;
{
    Display	*dpy = XtDisplay(fw);
    Pixel	 piece1, piece2;		/* Color of pieces */
    Pixmap	*stipple;			/* Pointer to stipple */
    void	(*mono)();			/* Mono color routine */

    stipple = (color == 1) ? &white_stipple : &black_stipple;
    piece1 = (color == 1) ? fw->fibw.wpiece.pixel : fw->fibw.bpiece.pixel;
    piece2 = (color == 1) ? fw->fibw.bpiece.pixel : fw->fibw.wpiece.pixel;
    mono = (color == 1) ? &use_black : &use_white;

    if (!usecolor) 
	use_stipple(fw, stipple);
    else
	set_color(fw, piece1);

    XFillArc(dpy, fw->fibw.board_pixmap, fw->fibw.draw_gc, 
	x, y, 
	fw->fibw.piece_size, fw->fibw.piece_size,
	0, 360*64);
    if (!usecolor) {
	XSetFillStyle(dpy, fw->fibw.draw_gc, FillSolid);
	mono(fw);
    }
    else
	set_color(fw, piece2);

    XDrawArc(dpy, fw->fibw.board_pixmap, fw->fibw.draw_gc, 
	x, y, 
	fw->fibw.piece_size-1, fw->fibw.piece_size-1,
	0, 360*64);

    if (!usecolor) 
	use_stipple(fw, stipple);
    else
	set_color(fw, piece1);

    XFillArc(dpy, fw->fibw.board_pixmap, fw->fibw.draw_gc, 
	x, y-fw->fibw.piece_size/10, 
	fw->fibw.piece_size, fw->fibw.piece_size,
	0, 360*64);
    if (!usecolor) {
	XSetFillStyle(dpy, fw->fibw.draw_gc, FillSolid);
	mono(fw);
    }
    else
	set_color(fw, piece2);

    XDrawArc(dpy, fw->fibw.board_pixmap, fw->fibw.draw_gc, 
	x, y-fw->fibw.piece_size/10, 
	fw->fibw.piece_size-1, fw->fibw.piece_size-1,
	0, 360*64);
    return;	 
}


static void draw_one_piece_at_pos(fw, pos, number, color)
FibwWidget fw;
int pos, number, color;
{
    XPoint	 point;

    get_piece_position(fw, pos, number, &point);
    draw_one_piece_at_loc(fw, point.x, point.y, color);
}

void draw_pieces_at_pos(fw, pos, number, dir, color)
FibwWidget fw;
int pos, number, dir, color;
{ 
    int i, numberoffset;

    /* Have to draw the lower five first, then the next five, 
	then the last if any... 
	And,  I have to draw in the direction specified,  */
    numberoffset = 0;
    if (number > 10 ) {
	if (dir == 1) 
	    for (i = 5; i > 0; i--) 
		draw_one_piece_at_pos(fw, pos, i, color);
	else 
	    for (i = 1; i <= 5; i++) 
		draw_one_piece_at_pos(fw, pos, i, color);
	number -= 5;
	numberoffset += 5;
    }
    if (number > 5 ) {
	if (dir == 1) 
	    for (i = 5; i > 0; i--) 
		draw_one_piece_at_pos(fw, pos, i + numberoffset, color);
	else 
	    for (i = 1; i <= 5; i++)
		draw_one_piece_at_pos(fw, pos, i + numberoffset, color);
	number -= 5;
	numberoffset += 5;
    }
    if (dir == 1) 
	for (i = number; i > 0; i--)
	    draw_one_piece_at_pos(fw, pos, i + numberoffset, color);
    else
	for (i = 1; i <= number; i++)
	    draw_one_piece_at_pos(fw, pos, i + numberoffset, color);
	
    return;
}

void draw_pieces(fw, i)
FibwWidget fw;
int i;
{
    FIBS_Rec	*pFIBS = &fw->fibw.fibs_rec;
    int		 direction, number;

    /* check out coloring!!! */
    if (pFIBS->direction == 1 && ((i >= 1 && i <= 12) || (i == 25 || i == 26)))
	direction = -1;
    else if (pFIBS->direction == -1 &&
	((i >= 13 && i <= 24) || (i == 0 || i == 27)))
	direction = -1;
    else direction = 1;

    number = abs(pFIBS->board[i]);
    draw_pieces_at_pos(fw, i, number, direction, number == pFIBS->board[i]); 
    return;
}


void redrawBoard(fw)
FibwWidget	fw;
{
    XExposeEvent fake_event;
    int		 i;

    /* To keep it simple, let's just repaint the whole board... */
    fake_event.x = fw->fibw.board_x_offset;
    fake_event.y = fw->fibw.board_y_offset;
    fake_event.width = 13*fw->fibw.piece_size + 5*fw->fibw.frame_size;
    fake_event.height = 12*fw->fibw.piece_size + 2*fw->fibw.frame_size;

    /* copy blank background onto bgboard pixmap */
    XCopyArea(XtDisplay(fw), fw->fibw.back_pixmap,
	    fw->fibw.board_pixmap, fw->fibw.draw_gc,
	    fake_event.x, fake_event.y,
	    fake_event.width, fake_event.height,
	    fake_event.x, fake_event.y);

    /* draw pieces */
    for (i = 0; i <= 27; i++) 
	draw_pieces(fw, i);

    if (fw->fibw.onFrame)
	draw_text(fw, fw->fibw.onFrame);

    Redisplay((Widget)fw, &fake_event);
}


/* Routine to remove the current piece for the board, without redrawing it.
 * We only redraw things when we have a logical block, otherwise if we redraw
 * too many times the display flickers.
 */
void remove_current(dpy, fw, event)
Display		*dpy;
FibwWidget	 fw;
XButtonEvent	*event;
{
    XCopyArea(dpy, fw->fibw.small_pixmap, fw->fibw.board_pixmap,
	fw->fibw.draw_gc,
	0, 0,
	2 * fw->fibw.piece_size, 2 * fw->fibw.piece_size,
	fw->fibw.current_piece_pos.x - fw->fibw.piece_size,
	fw->fibw.current_piece_pos.y - fw->fibw.piece_size);

    XCopyArea(dpy, fw->fibw.board_pixmap, fw->fibw.small_pixmap,
	fw->fibw.draw_gc,
	event->x - fw->fibw.piece_size, event->y - fw->fibw.piece_size,
	2 * fw->fibw.piece_size, 2 * fw->fibw.piece_size,
	0, 0);
}


void undraw_current(dpy, fw, event)
Display		*dpy;
FibwWidget	 fw;
XButtonEvent	*event;
{
    XExposeEvent	fake_event;

    remove_current(dpy, fw, event);

    fake_event.x= fw->fibw.current_piece_pos.x - fw->fibw.piece_size;
    fake_event.y= fw->fibw.current_piece_pos.y - fw->fibw.piece_size;
    fake_event.width = 2 * fw->fibw.piece_size;
    fake_event.height = 2 * fw->fibw.piece_size;
    Redisplay((Widget)fw,&fake_event);
}


static void draw_one_dice_into_pixmap(fw, pixmap, fg, bg, x, y)
FibwWidget	fw;
Pixmap		pixmap;
Pixel		fg, bg;
int		x,y;
{
    Display *dpy = XtDisplay(fw);

    if (!usecolor) {
	if (fg == fw->fibw.bpiece.pixel)
	    use_stipple(fw, &black_stipple);
	else
	    use_stipple(fw, &white_stipple);
    }
    else
	set_color(fw, fg);
    XFillRectangle(dpy, pixmap, fw->fibw.draw_gc,
	x + fw->fibw.piece_size/6, y,
	4*fw->fibw.piece_size/6, fw->fibw.piece_size);
    XFillRectangle(dpy, pixmap, fw->fibw.draw_gc,
	x, y+fw->fibw.piece_size/6,
	fw->fibw.piece_size, 4*fw->fibw.piece_size/6); 
    XFillArc(dpy, pixmap, fw->fibw.draw_gc,
	x, y,
	2*fw->fibw.piece_size/6, 2*fw->fibw.piece_size/6, 0, 360*64);
    XFillArc(dpy, pixmap, fw->fibw.draw_gc,
	x + 4*fw->fibw.piece_size/6, y,
	2*fw->fibw.piece_size/6, 2*fw->fibw.piece_size/6, 0, 360*64);
    XFillArc(dpy, pixmap, fw->fibw.draw_gc,
	x, y + 4*fw->fibw.piece_size/6,
	2*fw->fibw.piece_size/6, 2*fw->fibw.piece_size/6, 0, 360*64);
    XFillArc(dpy, pixmap, fw->fibw.draw_gc,
	x + 4*fw->fibw.piece_size/6, y + 4*fw->fibw.piece_size/6,
	2*fw->fibw.piece_size/6, 2*fw->fibw.piece_size/6, 0, 360*64); 
    if (!usecolor) 
	XSetFillStyle(dpy, fw->fibw.draw_gc, FillSolid);

    if (!usecolor)
	if (fg == fw->fibw.bpiece.pixel)
	    use_white(fw);
	else use_black(fw);
    else
	set_color(fw, bg);
    XDrawLine(dpy, pixmap, fw->fibw.draw_gc,
	x + fw->fibw.piece_size/6, y,
	x + 5*fw->fibw.piece_size/6,y);
    XDrawLine(dpy, pixmap, fw->fibw.draw_gc,
	x + fw->fibw.piece_size/6, y + fw->fibw.piece_size,
	x + 5*fw->fibw.piece_size/6, y + fw->fibw.piece_size);
    XDrawLine(dpy, pixmap, fw->fibw.draw_gc,
	x, y + fw->fibw.piece_size/6,
	x, y + 5*fw->fibw.piece_size/6);
    XDrawLine(dpy, pixmap, fw->fibw.draw_gc,
	x + fw->fibw.piece_size, y + fw->fibw.piece_size/6,
	x + fw->fibw.piece_size, y + 5*fw->fibw.piece_size/6);
    XDrawArc(dpy, pixmap, fw->fibw.draw_gc,
	x, y,
	2*fw->fibw.piece_size/6, 2*fw->fibw.piece_size/6, 90*64, 90*64);
    XDrawArc(dpy, pixmap, fw->fibw.draw_gc,
	x + 4*fw->fibw.piece_size/6, y,
	2*fw->fibw.piece_size/6, 2*fw->fibw.piece_size/6, 0, 90*64);
    XDrawArc(dpy, pixmap, fw->fibw.draw_gc,
	x, y + 4*fw->fibw.piece_size/6,
	2*fw->fibw.piece_size/6, 2*fw->fibw.piece_size/6, 180*64, 90*64);
    XDrawArc(dpy, pixmap, fw->fibw.draw_gc,
	x + 4*fw->fibw.piece_size/6, y + 4*fw->fibw.piece_size/6,
	2*fw->fibw.piece_size/6, 2*fw->fibw.piece_size/6, 270*64, 90*64); 
    return;
}


void draw_into_cubepixmap(fw)
FibwWidget fw;
{
    Display	*dpy = XtDisplay(fw);
    Drawable	 drawable = RootWindow(dpy,DefaultScreen(dpy));
    unsigned int depth = DefaultDepthOfScreen(XtScreen(fw));
    char	 outstring[3];

    if (fw->fibw.cube_pixmap)
	XFreePixmap(dpy, fw->fibw.cube_pixmap);
    fw->fibw.cube_pixmap = XCreatePixmap(dpy, drawable,
	    fw->fibw.piece_size + 10, fw->fibw.piece_size + 10, depth);

    if (!usecolor) 
	use_stipple(fw, &bkgnd_stipple);
    else
	set_color(fw, fw->fibw.table.pixel);

    XFillRectangle(dpy, fw->fibw.cube_pixmap, fw->fibw.draw_gc,
	    0, 0, fw->fibw.piece_size + 10, fw->fibw.piece_size + 10); 
    if (!usecolor) 
	XSetFillStyle(dpy, fw->fibw.draw_gc, FillOpaqueStippled);
	    
    /* draw the dice */
    draw_one_dice_into_pixmap(fw, fw->fibw.cube_pixmap,
	    fw->fibw.dblcube.pixel, fw->fibw.dbltext.pixel, 5, 5);
    /* draw the number */
    sprintf(outstring, "%i", fw->fibw.fibs_rec.cube);
    if (!usecolor) /* monochrome */
	use_black(fw);
    else 
	set_color(fw, fw->fibw.dbltext.pixel);
    XDrawString(dpy, fw->fibw.cube_pixmap, fw->fibw.draw_gc,
	5 + fw->fibw.piece_size/2 -
	XTextWidth(fw->fibw.pFont, outstring, strlen(outstring))/2,
	5 + fw->fibw.piece_size/2 + fw->fibw.textheight/2,
	outstring, strlen(outstring));
    return;
}



void draw_into_dicepixmap(fw)
FibwWidget fw;
{
    Display	*dpy = XtDisplay(fw);
    Drawable	 drawable = RootWindow(dpy,DefaultScreen(dpy));
    unsigned int depth = DefaultDepthOfScreen(XtScreen(fw));
    int		 eyeSize = fw->fibw.piece_size/6;
    int		 i, j, xBase, yBase;

    if (fw->fibw.dice_pixmap)
	XFreePixmap(dpy, fw->fibw.dice_pixmap);

    /* entry 0 is background, 1 is dice for 1, etc.. */
    fw->fibw.dice_pixmap = XCreatePixmap(dpy,drawable,
	(fw->fibw.piece_size + 5)*2 + 5,(fw->fibw.piece_size + 5)*7 + 5, depth);
    if (!usecolor) 
	use_stipple(fw,&board_stipple);
    else
	set_color(fw, fw->fibw.base.pixel);
    XFillRectangle(dpy, fw->fibw.dice_pixmap, fw->fibw.draw_gc,
	0, 0, (fw->fibw.piece_size + 5)*2 + 5, (fw->fibw.piece_size + 5)*7 + 5);
    if (!usecolor) 
	XSetFillStyle(dpy, fw->fibw.draw_gc, FillSolid);

    for (i = 0 ; i <= 1; i++) {
	for (j = 1; j <= 6; j++) {
#ifdef OLDWAY
	    if (i == 0) 
		use_fore(fw);
	    else
		set_color(fw, fw->fibw.table.pixel);
#endif
	    if (!usecolor)
		if (i == 0)
		    use_white(fw);
		else use_black(fw);
	    else
		if (i == 0)
		    set_color(fw, fw->fibw.wpiece.pixel);
		else set_color(fw, fw->fibw.bpiece.pixel);
	    
	    xBase = i*(fw->fibw.piece_size + 5) + 5;
	    yBase = j*(fw->fibw.piece_size + 5) + 5;

	    /* First, draw the dice, without any eyes */
	    if (i == 0)
		draw_one_dice_into_pixmap(fw, fw->fibw.dice_pixmap,
		    fw->fibw.wpiece.pixel, fw->fibw.bpiece.pixel, xBase,
		    yBase);
	    else
		draw_one_dice_into_pixmap(fw, fw->fibw.dice_pixmap,
		    fw->fibw.bpiece.pixel, fw->fibw.wpiece.pixel, xBase,
		    yBase);

	    /* Now, draw the eyes of the cubes */
	    if (!usecolor)
		if (i == 0)
		    use_black(fw);
		else use_white(fw);
	    else
		if (i == 0)
		    set_color(fw, fw->fibw.bpiece.pixel);
		else set_color(fw, fw->fibw.wpiece.pixel);

	    if (j == 1 || j == 3 || j == 5) {	/* draw middle eye */
		XFillArc(dpy, fw->fibw.dice_pixmap, fw->fibw.draw_gc,
		    xBase + 5*fw->fibw.piece_size/12,
		    yBase + 5*fw->fibw.piece_size/12,
		    eyeSize, eyeSize,
		    0, 360*64);
	    }
	    if (j != 1) {	/* draw first diagonal */
		XFillArc(dpy, fw->fibw.dice_pixmap, fw->fibw.draw_gc,
		    xBase + fw->fibw.piece_size/6,
		    yBase + 4*fw->fibw.piece_size/6,
		    eyeSize, eyeSize,
		    0, 360*64);
		XFillArc(dpy, fw->fibw.dice_pixmap, fw->fibw.draw_gc,
		    xBase + 4*fw->fibw.piece_size/6,
		    yBase + fw->fibw.piece_size/6,
		    eyeSize, eyeSize,
		    0, 360*64);
	    }
	    if (j > 3) {	/* draw second diagonal */
		XFillArc(dpy, fw->fibw.dice_pixmap, fw->fibw.draw_gc,
		    xBase + fw->fibw.piece_size/6,
		    yBase + fw->fibw.piece_size/6,
		    eyeSize, eyeSize,
		    0, 360*64);
		XFillArc(dpy, fw->fibw.dice_pixmap, fw->fibw.draw_gc,
		    xBase + 4*fw->fibw.piece_size/6,
		    yBase + 4*fw->fibw.piece_size/6,
		    eyeSize, eyeSize,
		    0, 360*64);
	    }
	    if (j == 6) {	/* draw a couple of eyes more */
		XFillArc(dpy, fw->fibw.dice_pixmap, fw->fibw.draw_gc,
		    xBase + fw->fibw.piece_size/6,
		    yBase + 5*fw->fibw.piece_size/12,
		    eyeSize, eyeSize,
		    0, 360*64);
		XFillArc(dpy, fw->fibw.dice_pixmap, fw->fibw.draw_gc,
		    xBase + 4*fw->fibw.piece_size/6,
		    yBase + 5*fw->fibw.piece_size/12,
		    eyeSize, eyeSize,
		    0, 360*64);
	    }
	}
    }
}


void draw_dblcube(fw)
FibwWidget fw;
{
    Display	*dpy = XtDisplay((Widget)fw);
    XExposeEvent fake_event;
    int		 yPosition;
    FIBS_Rec	*pFIBS = &fw->fibw.fibs_rec;
    int		 boardwidth = 6*fw->fibw.piece_size;

    draw_into_cubepixmap(fw);
    /* fill area with background color */
    if (!usecolor) 
	use_stipple(fw, &bkgnd_stipple);
    else
	set_color(fw, fw->fibw.table.pixel);
    XFillRectangle(dpy, fw->fibw.board_pixmap, fw->fibw.draw_gc,
	fw->fibw.board_x_offset + 2*boardwidth + 7*fw->fibw.frame_size + 2,
	fw->fibw.board_y_offset,
	3*fw->fibw.piece_size, 13*fw->fibw.piece_size + 1);
    if (!usecolor) 
	XSetFillStyle(dpy, fw->fibw.draw_gc, FillSolid);

    if (pFIBS->was_doubled[1] == 1)
	/* just doubled by player. draw 1/4 down the board */
	yPosition = fw->fibw.board_y_offset + 3*fw->fibw.piece_size;
    else if (pFIBS->was_doubled[0] == 1)
	/* opponent just doubled. draw 3/4 down the board */
	yPosition = fw->fibw.board_y_offset + 9*fw->fibw.piece_size;
    else if (pFIBS->playerMayDouble && pFIBS->opponentMayDouble)
	/* both may double: cube in middle */
	yPosition = fw->fibw.board_y_offset + boardwidth +
			    fw->fibw.frame_size - fw->fibw.piece_size/2;
    else if (pFIBS->playerMayDouble && !pFIBS->opponentMayDouble)
	/* player may double, opponent may not. draw at bottom of board */
	yPosition = fw->fibw.board_y_offset + 12*fw->fibw.piece_size;
    else
	yPosition = fw->fibw.board_y_offset;

    XCopyArea(dpy, fw->fibw.cube_pixmap, fw->fibw.board_pixmap, 
	fw->fibw.draw_gc,
	0, 0, fw->fibw.piece_size + 10, fw->fibw.piece_size + 10,
	fw->fibw.board_x_offset + 2*boardwidth +
	7*fw->fibw.frame_size + fw->fibw.frame_size,
	yPosition - 5);
    /* Send expose event to myself... */
    fake_event.x = fw->fibw.board_x_offset + 2*boardwidth +
	7*fw->fibw.frame_size + 1;
    fake_event.y = fw->fibw.board_y_offset;
    fake_event.width = 2*fw->fibw.piece_size;
    fake_event.height = 13*fw->fibw.piece_size + 1; 
    Redisplay((Widget)fw, &fake_event);

    return;
}


/* Because we're drawing opponent's piece 1st, it must be die1 and roll1 */
static void draw_dice(fw, die1, die2)
FibwWidget fw;
int die1;
int die2;
{
    Display *dpy = XtDisplay((Widget)fw);
    XExposeEvent fake_event;
    int diceBlack, diceWhite;	/* Offset into dice pixmap for black/white */
    int diceOffset;		/* Offset between pixmaps for 1,2 .. 6 */
    int boardOffset;		/* Offset for different board directions */
    int roll1, roll2;		/* Where to put the roll */
    int yourColor, oppsColor;	/* Color for each player */
    int color1, color2;		/* What color are the dice? */

    FIBS_Rec	*pFIBS = &fw->fibw.fibs_rec;
    int		 boardwidth = 6*fw->fibw.piece_size;
    int		 barwidth = 2*fw->fibw.frame_size;
    int		 yOffset = fw->fibw.board_y_offset + boardwidth +
			    fw->fibw.frame_size - fw->fibw.piece_size/2;

    /* Check if nobody rolled yet */
    if (pFIBS->turn == 0)
	return;

    /* Set offset between black & white dice pixmaps, also used to
     * step between die number pixmaps ie pixmap for 1, then 2 ...
     */
    diceOffset = fw->fibw.piece_size + 5;	/* offset to black dice */

    diceBlack = diceOffset;		/* black dice */
    diceWhite = 0;			/* white dice */

    if (pFIBS->color == -1) {
	yourColor = diceBlack;
	oppsColor = diceWhite;
    }
    else {
	yourColor = diceWhite;
	oppsColor = diceBlack;
    }

    if (fw->fibw.first_roll) {
	/* Player die is always given first by FIBS */
	if (pFIBS->turn != pFIBS->color) {
	    /* Opponents roll - put his die closest to the bar */
	    roll1 = 9*fw->fibw.piece_size + barwidth;
	    roll2 = 4*fw->fibw.piece_size;
	}
	else {	/* Your roll, your die closest to the bar */
	    roll1 = 7*fw->fibw.piece_size + barwidth;
	    roll2 = 2*fw->fibw.piece_size;
	}
	color1 = yourColor;
	color2 = oppsColor;
    }
    else {
	if (pFIBS->turn != pFIBS->color) {
	    /* Opponents roll - put dice on the left of the bar */
	    roll1 = 2*fw->fibw.piece_size;
	    roll2 = 4*fw->fibw.piece_size;
	    color1 = color2 = oppsColor;
	}
	else {
	    /* Your roll - put dice on the right of the bar */
	    roll1 = 7*fw->fibw.piece_size + barwidth;
	    roll2 = 9*fw->fibw.piece_size + barwidth;
	    color1 = color2 = yourColor;
	}
    }

    if (fw->fibw.debug)
	fprintf(stderr, "roll %d %d, drawn at %d, %d\n", die1, die2,
		roll1 / fw->fibw.piece_size, roll2 / fw->fibw.piece_size);

    /* I have to draw it into both board_pixmap and back_pixmap */
    XCopyArea(dpy, fw->fibw.dice_pixmap, fw->fibw.back_pixmap,
	fw->fibw.draw_gc,
	color1 + 5, die1 * diceOffset + 5,
	fw->fibw.piece_size+1, fw->fibw.piece_size+1,
	fw->fibw.playfield_offset + roll1, yOffset);
    XCopyArea(dpy, fw->fibw.dice_pixmap, fw->fibw.board_pixmap,
	fw->fibw.draw_gc,
	color1 + 5, die1 * diceOffset + 5,
	fw->fibw.piece_size+1, fw->fibw.piece_size+1,
	fw->fibw.playfield_offset + roll1, yOffset);
    XCopyArea(dpy, fw->fibw.dice_pixmap, fw->fibw.back_pixmap,
	fw->fibw.draw_gc,
	color2 + 5, die2 * diceOffset + 5,
	fw->fibw.piece_size+1, fw->fibw.piece_size+1,
	fw->fibw.playfield_offset + roll2, yOffset);
    XCopyArea(dpy, fw->fibw.dice_pixmap, fw->fibw.board_pixmap,
	fw->fibw.draw_gc,
	color2 + 5, die2 * diceOffset + 5,
	fw->fibw.piece_size+1, fw->fibw.piece_size+1,
	fw->fibw.playfield_offset + roll2, yOffset);

    /* Find left-hand piece and draw from there to right-hand piece */
    if (roll2 > roll1) {
	fake_event.x     = fw->fibw.playfield_offset + roll1;
	fake_event.width = roll2 - roll1 + fw->fibw.piece_size + 1;
    }
    else {
	fake_event.width = roll1 - roll2 + fw->fibw.piece_size + 1;
	fake_event.x     = fw->fibw.playfield_offset + roll2;
    }
    fake_event.y      = yOffset;		/* Top of dice */
    fake_event.height = fw->fibw.piece_size+1;
    Redisplay((Widget)fw,&fake_event);
    return;
}


/* Must be called before next roll is read in */
void undraw_roll(fw)
FibwWidget fw;
{
    draw_dice(fw, 0, 0);
}

void draw_roll(fw)
FibwWidget fw;
{
    int		 die1, die2;
    FIBS_Rec	*pFIBS = &fw->fibw.fibs_rec;

    /* Player die is always given first by FIBS */
    if (fw->fibw.first_roll) {
	if (fw->fibw.first_roll == pFIBS->turn) {
	    die1 = fw->fibw.playerfirst;
	    die2 = fw->fibw.opponentfirst;
	}
	/* First turn has gone, waiting for someone to roll the dice */
	else die1 = die2 = 0;
    }
    else if (pFIBS->turn != pFIBS->color) {
	/* Opponents roll */
	die1 = pFIBS->opponentdice[0];
	die2 = pFIBS->opponentdice[1];
    }
    else {
	/* Your roll */
	die1 = pFIBS->playerdice[0];
	die2 = pFIBS->playerdice[1];
    }
    draw_dice(fw, die1, die2);
}
