/*
 * Copyright 1993 Torstein Hansen
 */

/*
 * Fibw.c - the Fabulously Impressive Backgammon board Widget
 *          implementation file
 */
#include <Xm/XmP.h>
#include <X11/StringDefs.h>
#include <stdio.h>
#include <string.h>
#include "xfibs.h"
#include "FibwP.h"
#include "callbacks.h"
#include "config.h"
#include "popup.h"
#include "mouse.h"
#include "graphics.h"

#define offset(field) XtOffsetOf(FibwRec, field)

static XtResource resources[] = {
     {
    XtNtoggleCallback, XtCToggleCallback, 
    XtRCallback, sizeof(XtPointer),
    offset(fibw.callback), 
    XtRCallback, NULL },
     {
    XtNpixmapWidth, XtCPixmapWidth,
    XtRDimension, sizeof(Dimension),
    offset(fibw.pixmap_width),
    XtRImmediate, (XtPointer)DEFAULT_PIXMAP_WIDTH },
     {
    XtNpixmapHeight, XtCPixmapWidth,
    XtRDimension, sizeof(Dimension),
    offset(fibw.pixmap_height),
    XtRImmediate, (XtPointer)DEFAULT_PIXMAP_HEIGHT },
     {
    XtNtable, XtCTable,
    XtRPixel, sizeof(Pixel),
    offset(fibw.table.pixel),
    XtRImmediate, (XtPointer)DEFAULT_TABLE },
     {
    XtNbase, XtCBase,
    XtRPixel, sizeof(Pixel),
    offset(fibw.base.pixel),
    XtRImmediate, (XtPointer)DEFAULT_BASE },
     {
    XtNframe, XtCFrame,
    XtRPixel, sizeof(Pixel),
    offset(fibw.frame.pixel),
    XtRImmediate, (XtPointer)DEFAULT_FRAME },
     {
    XtNwpoint, XtCWpoint,
    XtRPixel, sizeof(Pixel),
    offset(fibw.wpoint.pixel),
    XtRImmediate, (XtPointer)DEFAULT_WPOINT },
     {
    XtNbpoint, XtCBpoint,
    XtRPixel, sizeof(Pixel),
    offset(fibw.bpoint.pixel),
    XtRImmediate, (XtPointer)DEFAULT_BPOINT },
     {
    XtNwpiece, XtCWpiece,
    XtRPixel, sizeof(Pixel),
    offset(fibw.wpiece.pixel),
    XtRImmediate, (XtPointer)DEFAULT_WPIECE },
     {
    XtNbpiece, XtCBpiece,
    XtRPixel, sizeof(Pixel),
    offset(fibw.bpiece.pixel),
    XtRImmediate, (XtPointer)DEFAULT_BPIECE },
     {
    XtNdblcube, XtCDblcube,
    XtRPixel, sizeof(Pixel),
    offset(fibw.dblcube.pixel),
    XtRImmediate, (XtPointer)DEFAULT_DBLCUBE },
     {
    XtNdbltext, XtCDbltext,
    XtRPixel, sizeof(Pixel),
    offset(fibw.dbltext.pixel),
    XtRImmediate, (XtPointer)DEFAULT_DBLTEXT },
     {
    XtNforeground, XtCForeground,
    XtRPixel, sizeof(Pixel),
    offset(fibw.foreground.pixel),
    XtRImmediate, (XtPointer)DEFAULT_BTEXT },
     {
    XtNbackground, XtCBackground,
    XtRPixel, sizeof(Pixel),
    offset(fibw.background.pixel),
    XtRImmediate, (XtPointer)DEFAULT_WTEXT },
     {
    XtNfont, XtCFont,
    XtRFont, sizeof(Font),
    offset(fibw.font),
    XtRImmediate, (XtPointer)DEFAULT_FONT },
     {
    XtNnumbers, XtCNumbers,
    XtRBoolean, sizeof(Boolean),
    offset(fibw.numbers),
    XtRImmediate, (caddr_t)DEFAULT_NUMBERS },
     {
    XtNonFrame, XtCOnFrame,
    XtRBoolean, sizeof(Boolean),
    offset(fibw.onFrame),
    XtRImmediate, (caddr_t)DEFAULT_ONFRAME },
     {
    XtNdebug, XtCDebug,
    XtRBoolean, sizeof(Boolean),
    offset(fibw.debug),
    XtRImmediate, (caddr_t)DEFAULT_DEBUG },
     {
    XtNlogstyle, XtCLogstyle,
    XtRInt, sizeof(int),
    offset(fibw.logStyle),
    XtRImmediate, (caddr_t)DEFAULT_LOG },
};

String fallback_resources[] = {
    "*XmPushButton*marginHeight: 1",
    "*XmPushButton*marginWidth:	1",
    "*XmPushButton*marginLeft:	3",
    "*XmPushButton*marginRight:	3",
    /* Colors for the windows, menubar, buttons etc.... */
    "*XmScrollBar*Foreground:	gray69",
    "*background:		gray69",
    "*foreground:		black",
    "*XmText*Background:	gray69",
    "*XmSelectionBox*Background: gray69",
    /* Colors on different things on the bg-board */
    "*bgboard.table:		aquamarine4",
    "*bgboard.frame:		burlywood4",
    "*bgboard.base:		burlywood2",
    "*bgboard.wpoint:		burlywood1",
    "*bgboard.bpoint:		burlywood3",
    "*bgboard.wtext:		white",
    "*bgboard.btext:		blue",
    "*bgboard.wpiece:		antiquewhite2",
    "*bgboard.bpiece:		navajowhite4",
    "*bgboard.dblcube:		antiquewhite2",
    "*bgboard.dbltext:		burlywood4",
    "*bgboard.onBoard:		False",
    "*bgboard.numbers:		False",
    "*bgboard.debug:		False",
    NULL,
};

/* Declaration of methods */

static void		Initialize();
void			Redisplay();
static void		Destroy();
static void		Resize();
static Boolean		SetValues();
static XtGeometryResult	QueryGeometry();

static int	debug;
 
/* Double click applies 1st available roll to clicked piece,
 * click down to pickup a piece, click up to drop it.
 * Motion moves the selected piece.
 *
 * Due to the way X works, we can't distiguish between a 1st press of a double
 * and a single press, so we use a different button - 2 button mouse? Shame!
 */
static char defaultTranslations[] =
    "<Btn2Down>(2):	clickPiece()	\n\
     <Btn1Down>:	pickupPiece()	\n\
     <Btn1Up>:		dropPiece()	\n\
     <Btn1Motion>:	movePiece()";

XtActionsRec actions[] = {
        {"clickPiece", (XtActionProc) clickPiece},
        {"pickupPiece", (XtActionProc) pickupPiece},
        {"dropPiece", (XtActionProc) dropPiece},
        {"movePiece", (XtActionProc) movePiece},
};

FibwClassRec fibwClassRec = {
    {
    /* core_class fields */
    /* superclass        */	(WidgetClass) &xmPrimitiveClassRec,
    /* class_name        */	"Fibw",
    /* widget_size       */	sizeof(FibwRec),
    /* class_initialize      */	NULL,
    /* class_part_initialize */ NULL,
    /* class_inited          */	FALSE,
    /* initialize        */	Initialize,
    /* initialize_hook   */	NULL,
    /* realize           */	XtInheritRealize,
    /* actions           */	actions,
    /* num_actions       */	XtNumber(actions),
    /* resources         */	resources,
    /* num_resources     */	XtNumber(resources),
    /* xrm_class         */	NULLQUARK,
    /* compress_motion       */	TRUE,
    /* compress_exposure     */	XtExposeCompressMultiple,
    /* compress_enterleave   */	TRUE,
    /* visible_interest      */	FALSE,
    /* destroy           */	Destroy,
    /* resize            */	Resize,
    /* expose            */	Redisplay,
    /* set_values        */	SetValues,
    /* set_values_hook       */	NULL,
    /* set_values_almost     */	XtInheritSetValuesAlmost,
    /* get_values_hook       */	NULL,
    /* accept_focus	*/	NULL,
    /* version		*/	XtVersion,
    /* callback_private	*/	NULL,
    /* tm_table		*/	defaultTranslations,
    /* query_geometry	*/	QueryGeometry,
    /* display_accelerator */	XtInheritDisplayAccelerator,
    /* extension	*/	NULL
    },
    {  /* Primitive class fields */
    /* border_highlight   */	(XtWidgetProc) _XtInherit,       
    /* border_unhighlight */	(XtWidgetProc) _XtInherit,    
    /* translations       */	XtInheritTranslations,      
    /* arm_and_activate   */	NULL,             
    /* syn resources      */	NULL,           
    /* num_syn_resources  */	0, 
    /* extension          */	NULL,             
    },
    {
    /* extension          */	0,
    },
};

/*
 * The widget class declaration
 */
 
WidgetClass fibwWidgetClass = (WidgetClass) & fibwClassRec;

/*
 * Fibw methods definitions
 */

/* ARGSUSED */
static void Initialize(treq, tnew, args, num_args)
Widget treq, tnew;
ArgList args;
Cardinal *num_args;
{
    FibwWidget	 new = (FibwWidget) tnew;
    unsigned int depth = DefaultDepthOfScreen(XtScreen(new));
    char	 startupboard[150];
    
    if (usecolor == -1) 
        if (depth == 1)
            usecolor = 0;
        else
            usecolor = 1;
            
    strcpy(startupboard,"board:You:Unknown:9:0:0:0:-2:0:0:0:0:5:0:3:0:0:0:-5:5:0:0:0:-3:0:-5:0:0:0:0:2:0:0:6:6:6:6:1:1:1:0:1:-1:0:25:0:0:0:0:0:0:0:0");
    /*                                       ^ direction */

    strcpy(new->fibw.lastboard, startupboard);	/* Debug */
    new->fibw.depth = depth;
    new->core.width = 0;
    new->core.height = 0;

    if (new->core.width == 0) 
	new->core.width = new->fibw.pixmap_width;
    if (new->core.height == 0)
	new->core.height = new->fibw.pixmap_height;

    /* tell Primitive not to allow tabbing to this widget */ 
    XtVaSetValues((Widget) new, XmNtraversalOn, False, NULL);
    
    new->fibw.current_piece_pos.x = -1;
    new->fibw.current_piece_pos.y = -1;
    new->fibw.moving		  = 0;

    /* create gc and other graphics stuff */
    get_drawgc(new);
    set_font(new);
    new->fibw.first_roll = 0;
    create_stipples(new);
    create_pixmap(new);

    /* Initialize all the backgammon-specific stuff */
    new->fibw.fibs_rec.was_doubled[0] = 0;
    new->fibw.fibs_rec.was_doubled[1] = 0;
    new->fibw.move_off = 0;
    new->fibw.bear_off = 0;
    new->fibw.moves_left = 0;
    new->fibw.pieces_moved = 0;
    new->fibw.popupmenu = startup_id;
    FibwSetInfo(new, startupboard, fi_board); 

}


/* ARGSUSED */
void Redisplay(w, event)
Widget w;
XExposeEvent *event;
{
    FibwWidget	fw = (FibwWidget) w;
    Display	*dpy = XtDisplay(fw);
    int		x, y, width, height;

    if (!XtIsRealized(fw))
	return;

    if (event) {  /* called from btn-event or expose */
        x = event->x;
        y = event->y; 
        width = event->width;
        height =  event->height;
    } 
    else {        /* called because complete redraw */
        x = 0;
	y = 0;
	width = fw->fibw.pixmap_width;
	height = fw->fibw.pixmap_height;
    }
    XCopyArea(dpy,fw->fibw.board_pixmap,
	XtWindow(fw),fw->fibw.draw_gc,
	x,y,width,height,x,y);
}

/* ARGSUSED */
static Boolean SetValues(current, request, new, args, num_args)
Widget current, request, new;
ArgList args;
Cardinal *num_args;
{
    FibwWidget curfw = (FibwWidget) current;
    FibwWidget newfw = (FibwWidget) new;
    Boolean do_redisplay = False;

    if (curfw->primitive.foreground != newfw->primitive.foreground) {
        XtReleaseGC((Widget)curfw, curfw->fibw.draw_gc);
        get_drawgc(newfw);
	set_font(newfw);
        do_redisplay = True;
    }
    
    return do_redisplay;
}

/* ARGSUSED */
static void Resize(w, client_data, cbs)
Widget w;
XtPointer client_data;
XmDrawingAreaCallbackStruct *cbs;
{
    FibwWidget fw = (FibwWidget) w;
    
    fw->fibw.pixmap_width=fw->core.width;
    fw->fibw.pixmap_height=fw->core.height;
    if (fw->fibw.board_pixmap)
	XFreePixmap(XtDisplay(fw), fw->fibw.board_pixmap);
    create_pixmap(fw);
    draw_into_pixmap(fw);
}


/* ARGSUSED */
static XtGeometryResult QueryGeometry(w, proposed, answer)
Widget w;
XtWidgetGeometry *proposed, *answer;
{
    FibwWidget fw = (FibwWidget) w;

    /* set fields we care about */
    answer->request_mode = CWWidth | CWHeight;

    /* initial width and height */
    answer->width = fw->fibw.pixmap_width;
    answer->height = fw->fibw.pixmap_height;

    return XtGeometryYes;
}


/*
 * Destruction of GC and pixmap)
 */
 
static void Destroy(w)
Widget w;
{
    FibwWidget fw = (FibwWidget) w;
    Display *dpy = XtDisplay(fw);

    fprintf(stderr, "Destroy\n");
    if (fw->fibw.draw_gc)
        XFreeGC(dpy, fw->fibw.draw_gc);
    if (fw->fibw.board_pixmap)
	XFreePixmap(dpy,fw->fibw.board_pixmap);
    if (fw->fibw.back_pixmap)
	XFreePixmap(dpy,fw->fibw.back_pixmap);
    return;
}


/* 
 * Public functions, not static 
 */


void FibwDouble(fw,name1)
FibwWidget fw;
char *name1;
{
    FIBS_Rec	 *pFIBS = &fw->fibw.fibs_rec;

    if (strcmp(name1, pFIBS->playername) == 0) {
	/* Player doubled */
	pFIBS->was_doubled[0] = 0;
	pFIBS->was_doubled[1] = 1;
    }
    else {
	/* Opponent doubled */
	pFIBS->was_doubled[0] = 1;
	pFIBS->was_doubled[1] = 0;
	if (strcmp(pFIBS->playername, "You") == 0) 
            use_popup(fw, double_popup_id);
    }
    pFIBS->cube *= 2; /* increase the cube value */
    draw_dblcube(fw);
}


void FibwReject(fw,name1)
FibwWidget fw;
char *name1;
{
    FIBS_Rec	 *pFIBS = &fw->fibw.fibs_rec;

    if (fw->fibw.popupmenu == double_popup_id) 
	use_popup(fw, roll_popup_id);
 
    if (strcmp(name1, pFIBS->playername) == 0) {
	/* Player rejected */
	pFIBS->was_doubled[0] = 0;
	pFIBS->was_doubled[1] = 0;
    }
    else {
	/* Opponent rejected */
	pFIBS->was_doubled[0] = 0;
	pFIBS->was_doubled[1] = 0;
    }
    pFIBS->opponentMayDouble = 1;
    pFIBS->playerMayDouble   = 1;
    pFIBS->cube /= 2;			/* decrease the cube value */
    if (pFIBS->cube < 1)
	pFIBS->cube = 1;
    draw_dblcube(fw);
}


void FibwAccept(fw,name1)
FibwWidget fw;
char *name1;
{
    FIBS_Rec	 *pFIBS = &fw->fibw.fibs_rec;

    if (strcmp(name1, pFIBS->playername) == 0) {
	/* Player accepted */
	pFIBS->was_doubled[0]    = 1;
	pFIBS->was_doubled[1]    = 0;
	pFIBS->playerMayDouble   = 1;
	pFIBS->opponentMayDouble = 0;
    }
    else {
	/* Opponent accepted */
	pFIBS->was_doubled[0]    = 0;
	pFIBS->was_doubled[1]    = 1;
	pFIBS->playerMayDouble   = 0;
	pFIBS->opponentMayDouble = 1;
    }
    draw_dblcube(fw);
    use_popup(fw, roll_popup_id);
}


void FibwResign(fw,name1)
FibwWidget fw;
char *name1;
{
    FIBS_Rec	 *pFIBS = &fw->fibw.fibs_rec;

    if ((strcmp(name1, pFIBS->opponentname) == 0) &&
        (strcmp(pFIBS->playername, "You") == 0)) {
	/* Opponent resigns */
        use_popup(fw, resign_popup_id);
    }
}


void FibwRoll(fw)
FibwWidget fw;
{
    use_popup(fw, roll_popup_id);
    return;
}

void FibwJoin(fw,name)
FibwWidget fw;
char *name;
{
    if (name)
        strncpy(fw->fibw.invited_name, name, 80);
    else
        fw->fibw.invited_name[0] = '\0';
    use_popup(fw, join_popup_id);
    return;
}


void FibwSetMoves(fw, number_of_moves)
FibwWidget	fw;
int		number_of_moves;
{
    int		i, j, dummyPoint;
    int		highest_dice;
    Move	dummy;
    
    fw->fibw.moves_left = number_of_moves;
    /* Check bear-off status & update popup */
    checkBearOff(fw);
    fw->fibw.move_off = fw->fibw.bear_off;
    if (fw->fibw.popupmenu != 0)
	use_popup(fw, fw->fibw.popupmenu);

    /* if only 1 move allowed, check if we might move the highest number */
    if (number_of_moves != 1)
	return;

    highest_dice = max(fw->fibw.dice[0].value, fw->fibw.dice[1].value);

    if (fw->fibw.fibs_rec.direction == 1) {
	for (i = 0; i < 25; i++) {
	    if (fw->fibw.fibs_rec.board[i] * fw->fibw.fibs_rec.color <= 0)
		continue;

	    dummyPoint = i + highest_dice;
	    if (dummyPoint >= 25)
		dummyPoint = 27;

	    /* If it's legal, we're okay */
	    if (legal_move(fw, i, &dummyPoint, &dummy) > -1)
		return;

	    /* If we're on the bar, look no further */
	    /* Can't have 5 rolls, so this indicates a disabled roll */
	    if (i == 0) {
		fw->fibw.dice[0].used = 5;
		return;
	    }
	}
    }
    else {
	for (i = 25 ; i > 0; i--) {
	    if (fw->fibw.fibs_rec.board[i] * fw->fibw.fibs_rec.color <= 0)
		continue;

	    dummyPoint = i - highest_dice;
	    if (dummyPoint <= 0)
		dummyPoint = 26;

	    /* If it's legal, we're okay */
	    if (legal_move(fw, i, &dummyPoint, &dummy) > -1)
		return;

	    /* If we're on the bar, look no further */
	    /* Can't have 5 rolls, so this indicates a disabled roll */
	    if (i == 25) {
		fw->fibw.dice[0].used = 5;
		return;
	    }
	}
    }

    /* If we get here the highest die cannot move, so mark it as used. */
    if (highest_dice == fw->fibw.dice[0].value)
	fw->fibw.dice[0].used = 5;
}


void debugFirst(fw, value)
FibwWidget	fw;
int		value;
{
    if (value >= -1 && value <= 1)
	fw->fibw.first_roll = value;
    else fprintf(stderr, "First roll %d\n", fw->fibw.first_roll);
}

void debugAvail(fw)
FibwWidget fw;
{
    int		 i, j;
    FIBS_Rec	*pFIBS = &fw->fibw.fibs_rec;

    if (fw->fibw.moves_left == 0) {
	fprintf(stderr, "No rolls available\n");
	return;
    }

    j = fw->fibw.moves_left;
    fprintf(stderr, "%d rolls: %d moved\n", j, fw->fibw.pieces_moved);
    for (i = 0; i < j; i++) {
	fprintf(stderr, "Roll: %d, Used: %d", fw->fibw.dice[i].value,
		fw->fibw.dice[i].used);
	if (fw->fibw.dice[i].used == 5) {
	    fprintf(stderr, ": blocked");
	    j++;
	}
	else if (fw->fibw.dice[i].used)
	    fprintf(stderr, ": %s", fw->fibw.dice[i].move);
	fprintf(stderr, "\n");
    }
    if (fw->fibw.first_roll)
	fprintf(stderr, "1st roll: %s\n", (fw->fibw.first_roll == pFIBS->color)
		? pFIBS->playername : pFIBS->opponentname);
    fprintf(stderr, "%sBearing off\n", (fw->fibw.bear_off) ? "" : "Not ");
    fprintf(stderr, "%sMoving off\n", (fw->fibw.move_off) ? "" : "Not ");
}

	    
void debugBoard(board)
char *board;
{
    FibwWidget fw = (FibwWidget) bgboard;

    if (!board)
	fprintf(stderr, "%s\n", fw->fibw.lastboard);
    else FibwSetInfo(fw, board, fi_board);
}

	    
void FibwDiceRolled(fw, name1, name2, dice1, dice2)
FibwWidget fw;
char *name1;
char *name2;
int dice1;
int dice2;
{
    int		 i;
    FIBS_Rec	*pFIBS = &fw->fibw.fibs_rec;

    undraw_roll(fw);	/* Only really needed after opening rolls & dances */

    /* As someone has rolled, there's no doubling action at this time... */
    pFIBS->was_doubled[0] = 0;
    pFIBS->was_doubled[1] = 0;

    /* Let's store the numbers in case it's my roll... */
    if (dice1 == dice2) 
	for (i = 0; i < 4; i++) {
	    fw->fibw.dice[i].value = dice1;
	    fw->fibw.dice[i].used  = 0;
	}
    else {
	fw->fibw.dice[0].value = dice1;
	fw->fibw.dice[0].used  = 0;
	fw->fibw.dice[1].value = dice2;
	fw->fibw.dice[1].used  = 0;
	fw->fibw.dice[2].used  = 0;
	fw->fibw.dice[3].used  = 0;
    }
    if (name2 == (char *)NULL) { /* standard roll */
	if (strcmp(name1, pFIBS->playername) == 0) {
	    pFIBS->playerdice[0] = dice1;
	    pFIBS->playerdice[1] = dice2;
	    pFIBS->turn = pFIBS->color;
	}
	else {
	    pFIBS->opponentdice[0] = dice1;
	    pFIBS->opponentdice[1] = dice2;
	    pFIBS->turn = -pFIBS->color;
	}
	fw->fibw.first_roll = 0;
    }
    else { /* first roll */
	if (strcmp(name1, pFIBS->playername) == 0) {
	    fw->fibw.playerfirst   = dice1;
	    fw->fibw.opponentfirst = dice2;
	    pFIBS->playerdice[0]   = dice1;
	    pFIBS->opponentdice[0] = dice2;
	    if (dice1 > dice2) {
		pFIBS->playerdice[1] = dice2;
		pFIBS->turn = pFIBS->color;
	    }
	    else {
		pFIBS->opponentdice[1] = dice1;
		pFIBS->turn = -pFIBS->color;
	    }
	    if (dice1 == dice2)
		pFIBS->turn = 0;
	}
	else {
	    fw->fibw.playerfirst   = dice2;
	    fw->fibw.opponentfirst = dice1;
	    pFIBS->playerdice[0]   = dice2;
	    pFIBS->opponentdice[0] = dice1;
	    if (dice2 > dice1) {
		pFIBS->playerdice[1] = dice1;
		pFIBS->turn = pFIBS->color;
	    }
	    else {
		pFIBS->opponentdice[1] = dice2;
		pFIBS->turn = -pFIBS->color;
	    }
	    if (dice1 == dice2)
		pFIBS->turn = 0;
	}
	fw->fibw.first_roll = pFIBS->turn;
    }
    FibwSetMoves(fw, pFIBS->can_move);

    if (fw->fibw.debug)
     printf("%2d: %s: %d, %d. %s: %d, %d. Trn: %d, Col: %d, Dir: %d, Cub: %d\n",
	    fw->fibw.first_roll,
	    pFIBS->playername, pFIBS->playerdice[0], pFIBS->playerdice[1],
	    pFIBS->opponentname, pFIBS->opponentdice[0], pFIBS->opponentdice[1],
	    pFIBS->turn, pFIBS->color, pFIBS->direction, pFIBS->cube);
 
    /* redraw */
    draw_roll(fw);
    return;
}


void FibwSetInfo(fw, info, infotype)
FibwWidget fw;
char *info;
/* one of fi_double, fi_accept, fi_newboard, fi_redisplay, fi_board */
int infotype;
{
    char	 *infoword;
    int		 i, oldColor;
    XExposeEvent fake_event;
    FIBS_Rec	 *pFIBS = &fw->fibw.fibs_rec;
   
    if (infotype == fi_move) {
	return;
    }
    if (infotype == fi_newboard) {
	draw_into_pixmap(fw);
	FibwSetInfo(fw, (char *)NULL, fi_redisplay);
	return;
    }
    if (infotype == fi_redisplay) {
	fake_event.x = 0;
	fake_event.y = 0;
	fake_event.width  = fw->fibw.pixmap_width;
	fake_event.height = fw->fibw.pixmap_height;
	Redisplay((Widget) fw, &fake_event);
	return;
    }

    if (infotype == fi_board) {
	/* Remove any popupmenu handlers installed */
        if (fw->fibw.popupmenu != 0)
            use_popup(fw, roll_popup_id);

	oldColor = pFIBS->color;
	strcpy(fw->fibw.lastboard, info);	/* Make a copy of the board */

	infoword=strtok(info, ":");
	if (!infoword)
	    return;

	if (strcmp(infoword, "board")!=0)
	    return; 

	/* parse all this stuff to set all the other fields...*/ 
	memset(pFIBS->playername,'\0', sizeof(pFIBS->playername));
	memset(pFIBS->opponentname,'\0', sizeof(pFIBS->opponentname));
	strncpy(pFIBS->playername, strtok((char *)NULL, ":"), 80);
	strncpy(pFIBS->opponentname, strtok((char *)NULL, ":"), 80);
	/*  
	 * could use a union in the fibs_rec definition, then one could
	 * initialise all of the fields in a single loop. Nothing is 
	 * gained by it, though.
	 */
	pFIBS->match_length = atoi(strtok((char *)NULL, ":"));
	pFIBS->player_got   = atoi(strtok((char *)NULL, ":"));
	pFIBS->opponent_got = atoi(strtok((char *)NULL, ":"));

	for (i = 0; i < 26; i++)
	    pFIBS->board[i] = atoi(strtok((char *)NULL, ":"));

	pFIBS->turn = atoi(strtok((char *)NULL, ":"));

	pFIBS->playerdice[0]   = atoi(strtok((char *)NULL, ":"));
	pFIBS->playerdice[1]   = atoi(strtok((char *)NULL, ":"));
	pFIBS->opponentdice[0] = atoi(strtok((char *)NULL, ":"));
	pFIBS->opponentdice[1] = atoi(strtok((char *)NULL, ":"));

	pFIBS->cube              = atoi(strtok((char *)NULL, ":"));
	pFIBS->playerMayDouble   = atoi(strtok((char *)NULL, ":"));
	pFIBS->opponentMayDouble = atoi(strtok((char *)NULL, ":"));
	pFIBS->was_doubled[0]    = atoi(strtok((char *)NULL, ":"));

	pFIBS->color     = atoi(strtok((char *)NULL, ":"));
	pFIBS->direction = atoi(strtok((char *)NULL, ":"));

	/* Watch for the board suddenly turning round. Basically, FIBS
	 * sends the first roll followed by a new board, so we need to
	 * invert the sign of first_roll. If this happens after the 1st
	 * roll, -0 is still 0.
	 */
	if (pFIBS->color != oldColor)
		fw->fibw.first_roll = -fw->fibw.first_roll;
	/* next two are obsolete, so don't get used */
	pFIBS->home = atoi(strtok((char *)NULL, ":"));
	pFIBS->bar  = atoi(strtok((char *)NULL, ":"));

	/* get on with the useful ones... */
	pFIBS->playerOnHome   = atoi(strtok((char *)NULL, ":"));
	pFIBS->opponentOnHome = atoi(strtok((char *)NULL, ":"));
	pFIBS->playerOnBar    = atoi(strtok((char *)NULL, ":"));
	pFIBS->opponentOnBar  = atoi(strtok((char *)NULL, ":"));

	if (pFIBS->direction == 1) {
	    pFIBS->board[26] = - pFIBS->color * pFIBS->opponentOnHome;
	    pFIBS->board[27] =   pFIBS->color * pFIBS->playerOnHome;
	}
	else {
	    pFIBS->board[26] =   pFIBS->color * pFIBS->playerOnHome;
	    pFIBS->board[27] = - pFIBS->color * pFIBS->opponentOnHome;
	}

	pFIBS->can_move = atoi(strtok((char *)NULL, ":"));

	/* the last three doesn't get used by this program, but is included
	 * anyway... */
	pFIBS->forced_move  = atoi(strtok((char *)NULL, ":"));
	pFIBS->did_crawford = atoi(strtok((char *)NULL, ":"));
	pFIBS->redoubles    = atoi(strtok((char *)NULL, ":"));

    if (fw->fibw.debug)
      printf("b : %s: %d, %d. %s: %d, %d. Trn: %d, Col: %d, Dir: %d, Cub: %d\n",
	    pFIBS->playername, pFIBS->playerdice[0], pFIBS->playerdice[1],
	    pFIBS->opponentname, pFIBS->opponentdice[0], pFIBS->opponentdice[1],
	    pFIBS->turn, pFIBS->color, pFIBS->direction, pFIBS->cube);

	if (pFIBS->turn == pFIBS->color)
	    if (pFIBS->playerdice[0]) { /* has rolled */
		if (pFIBS->playerdice[0] == pFIBS->playerdice[1]) /* a double */
		    for (i = 0; i < 4; i++) {
			fw->fibw.dice[i].value  = pFIBS->playerdice[0];
			fw->fibw.dice[i].used   = 0;
		    }
		else {
		    fw->fibw.dice[0].value  = pFIBS->playerdice[0];
		    fw->fibw.dice[0].used   = 0;
		    fw->fibw.dice[1].value  = pFIBS->playerdice[1];
		    fw->fibw.dice[1].used   = 0;
		    fw->fibw.dice[2].used   = 0;
		    fw->fibw.dice[3].used   = 0;
		}
		FibwSetMoves(fw, pFIBS->can_move);
	    }
	/* Quite a success, really. At least i hope so */
	FibwSetInfo(fw,(char *)NULL, fi_newboard);
	fw->fibw.pieces_moved = 0;
    }
    return;
}
