
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "global.h"
#include "defs.h"
#include "struct.h"

#include "network.h"
#include "map.h"

/* wall shape based on neighbors */
#define W_TOP 1
#define W_RIGHT 2
#define W_BOT 4
#define W_LEFT 8

int wallshapes[16] =
{
    T_POST,
    T_WALL_T,
    T_WALL_R,
    T_WALL_TR,
    T_WALL_B,
    T_WALL_TB,
    T_WALL_BR,
    T_WALL_TBR,
    T_WALL_L,
    T_WALL_TL,
    T_WALL_RL,
    T_WALL_TRL,
    T_WALL_BL,
    T_WALL_TBL,
    T_WALL_BRL,
    T_WALL_TBRL,
};

/* map_iswall(v)

   Is the terrain value v a wall?

   return: true or false
   */
int map_iswall(char v)
{
    if (v >= T_GWALL && v <= T_WALL_TBRL)
        return 1;
    return 0;
}

/* makewall(x,y)

   Make a wall at (x,y).  Make sure the shape is correct

   Return: none
   */
static void makewall(int x,
                     int y)
{
    int walls = 0;

    if (map_iswall(GLO_map[x][y - 1]))
        walls |= W_TOP;
    if (map_iswall(GLO_map[x + 1][y]))
        walls |= W_RIGHT;
    if (map_iswall(GLO_map[x][y + 1]))
        walls |= W_BOT;
    if (map_iswall(GLO_map[x - 1][y]))
        walls |= W_LEFT;
    GLO_map[x][y] = wallshapes[walls];
}

/* map_makenewwall(x,y)

   Build a new wall at (x,y), update everyone

   Return: none
   */
void map_makenewwall(int x,
                     int y)
{
    int walls = 0;

    /* The terrain numbers for walls contain 4 bits that determine
       what type to draw, which depends on its four neighbors */
    if (map_iswall(GLO_map[x][y - 1]))
        walls |= W_TOP;
    if (map_iswall(GLO_map[x + 1][y]))
        walls |= W_RIGHT;
    if (map_iswall(GLO_map[x][y + 1]))
        walls |= W_BOT;
    if (map_iswall(GLO_map[x - 1][y]))
        walls |= W_LEFT;

    /* update the map array */
    GLO_map[x][y] = wallshapes[walls];

    /* Tell everyone baout it */
    NET_update_mapsquare(x, y);

    /* Adding this wall may have changed the shapes of neighboring walls,
       check and see */
    if (walls & W_TOP) {
        makewall(x, y - 1);
        NET_update_mapsquare(x, y - 1);
    }
    if (walls & W_RIGHT) {
        makewall(x + 1, y);
        NET_update_mapsquare(x + 1, y);
    }
    if (walls & W_BOT) {
        makewall(x, y + 1);
        NET_update_mapsquare(x, y + 1);
    }
    if (walls & W_LEFT) {
        makewall(x - 1, y);
        NET_update_mapsquare(x - 1, y);
    }
}

/* map_crater_square(x,y)

   Make a crater at (x,y)

   Return: none
*/
void map_crater_square(int x,
                       int y)
{
    int water = 0;

    /* don't crater a water square */
    if (GLO_map[x][y] == T_WATER)
        return;

    if (GLO_map[x - 1][y] == T_WATER)
        water++;
    if (GLO_map[x + 1][y] == T_WATER)
        water++;
    if (GLO_map[x][y - 1] == T_WATER)
        water++;
    if (GLO_map[x][y + 1] == T_WATER)
        water++;

    /* if two or more of this square's neighbors are water, make this square
       water instead of a crater */
    if (water >= 2 || GLO_map[x][y] == T_BRIDGE)
        GLO_map[x][y] = T_WATER;
    else
        GLO_map[x][y] = T_CRATER;

    /* Tell everyone what happened */
    NET_update_mapsquare(x, y);
}

/* read_map(name)

   read the named map into the global map array

   Return: true on success, false on failure
   */

static int read_map(char *name)
{
    FILE *mf;
    int w, h, x, y;
    char buf[512];

    /* count how many flags there are */
    GLO_maxflag = 0;

    /* Open the file */
    if (!(mf = fopen(name, "r"))) {
        fprintf(stderr, "Can't open map file %s\n", name);
        return 0;
    }
    /* Read the first line of the file: the map's name */
    fgets(buf, 512, mf);
    strncpy(GLO_map_info.m_name, buf, 32);

    /* make sure it's null terminated */
    if (strlen(buf) < 32)
        GLO_map_info.m_name[strlen(buf) - 1] = 0;
    else
        GLO_map_info.m_name[31] = 0;

    fprintf(stderr, "Map name is %s\n", GLO_map_info.m_name);

    /* read the second line, width and height */
    fgets(buf, 512, mf);
    if (sscanf(buf, "%d %d", &w, &h) != 2) {
        fprintf(stderr, "Error parsing width and height from %s", buf);
        fclose(mf);
        return 0;
    }
    /* both teams have 0 flags so far */
    GLO_num_flags[0] = GLO_num_flags[1] = 0;

    GLO_map_info.m_width = w;
    GLO_map_info.m_height = h;

    /* clear the global map array */
    memset(GLO_map, 0, MAXMAPSIZE * MAXMAPSIZE);

    /* read one line at a time */
    for (y = 0; y < h; y++) {

        /* read a line */
        if (!fgets(buf, 512, mf)) {
            perror("read_map");
            fclose(mf);
            return 0;
        }
        /* go across this line and change the text into our internal representation */
        for (x = 0; x < w; x++) {
            /* If it's a wall, it hasn't been damaged yet */
            GLO_mapdamage[x][y] = 0;
            switch (buf[x]) {
            case '.':
                GLO_map[x][y] = T_GRASS;
                break;
            case 'r':
                GLO_map[x][y] = T_ROAD;
                break;
            case 't':
                GLO_map[x][y] = T_TREE;
                break;
            case 'b':
                GLO_map[x][y] = T_GWALL;
                break;
            case 'w':
                GLO_map[x][y] = T_WATER;
                break;
            case '1':
                GLO_map[x][y] = T_BASE1;
                GLO_bases[0].b_x = x;
                GLO_bases[0].b_y = y;
                break;
            case 'O':           /* O for One, team one's flags */
                GLO_map[x][y] = T_ROAD;
                GLO_flags[GLO_maxflag].f_team = 0;
                GLO_flags[GLO_maxflag].f_x = (x * GRIDWIDTH + GRIDWIDTH / 2) << 4;
                GLO_flags[GLO_maxflag].f_y = (y * GRIDHEIGHT + GRIDHEIGHT / 2) << 4;
                GLO_flags[GLO_maxflag].f_carrier = -1;
                GLO_num_flags[0]++;
                GLO_maxflag++;
                break;

            case '2':
                GLO_map[x][y] = T_BASE2;
                GLO_bases[1].b_x = x;
                GLO_bases[1].b_y = y;
                break;
            case 'T':           /* T For Two, a dinkadoo */
                GLO_map[x][y] = T_ROAD;
                GLO_flags[GLO_maxflag].f_team = 1;
                GLO_flags[GLO_maxflag].f_x = (x * GRIDWIDTH + GRIDWIDTH / 2) << 4;
                GLO_flags[GLO_maxflag].f_y = (y * GRIDHEIGHT + GRIDHEIGHT / 2) << 4;
                GLO_flags[GLO_maxflag].f_carrier = -1;
                GLO_num_flags[1]++;
                GLO_maxflag++;
                break;
            }
        }
    }

    fclose(mf);

    /* post process */
    for (y = 0; y < h; y++) {
        for (x = 0; x < w; x++) {
            if (GLO_map[x][y] == T_GWALL) {
                /* make sure all the wall squares have the right shape */
                makewall(x, y);
            } else if (GLO_map[x][y] == T_ROAD) {
                int water = 0;

                if (GLO_map[x - 1][y] == T_WATER) {
                    water++;
                }
                if (GLO_map[x + 1][y] == T_WATER) {
                    water++;
                }
                if (GLO_map[x][y - 1] == T_WATER) {
                    water++;
                }
                if (GLO_map[x][y + 1] == T_WATER) {
                    water++;
                }
                /* draw roads as bridges when surrounded by 2 or more water squares */
                if (water >= 2) {
                    GLO_map[x][y] = T_BRIDGE;
                }
            }
        }
    }

    /* neither base has any flags yet */
    GLO_bases[0].b_numflags = 0;
    GLO_bases[1].b_numflags = 0;

    return 1;
}

/* map_pick_map(maplist)

   Read a list of names from the file specified, pick one at random, and read it in

   Return: True on success, false on failure
   */
int map_pick_map(char *maplist)
{
    FILE *ml;
    int num_lines, use_line, i;
    char buf[512];

    /* open the map list */
    if (!(ml = fopen(maplist, "r"))) {
        fprintf(stderr, "Can't open map list, trying map \"%s\"\n", GLO_mapname);
        /* Try the default map name if we can't find a list */
        return read_map(GLO_mapname);
    }
    /* Read the first line, should be number of maps contained here */
    if (fscanf(ml, "%d\n", &num_lines) != 1) {
        fprintf(stderr, "Error in map list! trying map \"%s\"\n", GLO_mapname);
        fclose(ml);
        return read_map(GLO_mapname);
    }
    /* pick a line number */
    use_line = rand() % num_lines;

    /* read down that many lines in the list file */
    for (i = 0; i <= use_line; i++)
        fgets(buf, 512, ml);

    fclose(ml);

    /* and try to read the map with the name found on that line */
    buf[strlen(buf) - 1] = 0;
    return read_map(buf);
}
