/*****************************************************************************
 *                                                                           *
 *  Copyright (c) 1993-1997 Elan Feingold (elan@jeeves.net)                  *
 *                                                                           *
 *     PERMISSION TO USE, COPY, MODIFY, AND TO DISTRIBUTE THIS SOFTWARE      *
 *     AND ITS DOCUMENTATION FOR ANY PURPOSE IS HEREBY GRANTED WITHOUT       *
 *     FEE, PROVIDED THAT THE ABOVE COPYRIGHT NOTICE APPEAR IN ALL           *
 *     COPIES AND MODIFIED COPIES AND THAT BOTH THAT COPYRIGHT NOTICE AND    *
 *     THIS PERMISSION NOTICE APPEAR IN SUPPORTING DOCUMENTATION.  THERE     *
 *     IS NO REPRESENTATIONS ABOUT THE SUITABILITY OF THIS SOFTWARE FOR      *
 *     ANY PURPOSE.  THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS       *
 *     OR IMPLIED WARRANTY.                                                  *
 *                                                                           *
 *****************************************************************************/

#include <X11/X.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>

#include "colormap.h"
#include "client.h"
#include "utils.h"
#include "riskgame.h"
#include "gui-vars.h"
#include "debug.h"

unsigned long   colormap[MAX_COLORS];
unsigned long   plCountryToColor[MAX_COLORS];
Int32           piColorToCountry[MAX_COLORS];
Colormap        cmapColormap = 0;
Color           pWorldColors[MAX_COLORS];
Int32           iNumColors;
Int32           COLOR_Depth;
static Flag     trueColor = FALSE;
static Flag	sixteenBitColor = FALSE;

/* Local variables for conversion between rgb and screen values. */
static int   redShift, greenShift, blueShift;
static int   redMove,  greenMove,  blueMove;
static Int32 redMask,  greenMask,  blueMask;

/************************************************************************
 *  FUNCTION: COLOR_InitRGB
 *  HISTORY:
 *     31.01.98	 JRXR Created.
 *  PURPOSE:
 *     To centralise the conversion between rgb values and screen
 *   values.
 *  NOTES:
 ************************************************************************/
Int32 COLOR_InitRGB(XVisualInfo *Info)
{
  Int32 r,g,b;
  int i;
  
  if (Info->class == TrueColor)
    {
      r = redMask   = Info->red_mask;
      g = greenMask = Info->green_mask;
      b = blueMask  = Info->blue_mask;
      
      for ( i=0 ; !(r & 1) ; r=r>>1,i++ )
	;
      redShift=i;
      for ( i=16 ; r ; r=r>>1,i-- )
	;
      redMove=i;
      for ( i=0 ; !(g & 1) ; g=g>>1,i++ )
	;
      greenShift=i;
      for ( i=16 ; g ; g=g>>1,i-- )
	;
      greenMove=i;
      for ( i=0 ; !(b & 1) ; b=b>>1,i++ )
	;
      blueShift=i;
      for ( i=16 ; b ; b=b>>1,i-- )
	;
      blueMove=i;
    }
  else if (Info->class == PseudoColor)
    {
      redMove    = greenMove = blueMove = 8;
      redMask    = 0x0000ff;
      greenMask  = 0x00ff00;
      blueMask   = 0xff0000;
      redShift   = 0;
      greenShift = 8;
      blueShift  = 16;
     }
}

/************************************************************************
 *  FUNCTION: COLOR_numTorgb
 *  HISTORY:
 *     31.01.98	 JRXR Created.
 *  PURPOSE:
 *     To centralise the conversion between rgb values and screen
 *   values.
 *  NOTES:
 ************************************************************************/
void COLOR_numTorgb( Int32 Num, UInt16 *Red, UInt16 *Green, UInt16 *Blue )
{
  Int32 temp;
  temp=(Num & redMask) >> redShift;
  *Red=temp << redMove;
  temp=(Num & greenMask) >> greenShift;
  *Green=temp << greenMove;
  temp=(Num & blueMask) >> blueShift;
  *Blue=temp << blueMove;
 }
 

/************************************************************************
 *  FUNCTION: COLOR_rgbToNum
 *  HISTORY:
 *     31.01.98	 JRXR Created.
 *  PURPOSE:
 *     To centralise the conversion between rgb values and screen
 *   values.
 *  NOTES:
 ************************************************************************/
Int32 COLOR_rgbToNum(Int32 Red, Int32 Green, Int32 Blue)
{
  Int32 c,r,g,b;
  r = Red   >> redMove;
  g = Green >> greenMove;
  b = Blue  >> blueMove;
  return (r<<redShift) + (g<<greenShift) + (b<<blueShift);
}
 

/************************************************************************ 
 *  FUNCTION: COLOR_IsTrueColor
 *  HISTORY: 
 *     07.08.95  JC  Created.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
Flag COLOR_IsTrueColor()
{
  return(trueColor);
}


/************************************************************************ 
 *  FUNCTION: COLOR_QueryColor
 *  HISTORY: 
 *     07.08.95  JC  Created.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
Int32 COLOR_QueryColor(Int32 iColor)
{
  if (!trueColor)
      return(iColor);
  return(colormap[iColor]);
}


/************************************************************************ 
 *  FUNCTION: COLOR_Init
 *  HISTORY: 
 *     09.08.94  ESF  Created.
 *     01.01.95  ESF  Added making the player turn indicator black.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void COLOR_Init(void)
{
  Int32 i;

  /* Set the initial player colors to be black */
  for (i=0; i!=MAX_PLAYERS; i++)
    COLOR_StoreNamedColor("Black", i);

  /* Set the initial player turn indicator to be black */
  COLOR_CopyColor(COLOR_PlayerToColor(0), COLOR_DieToColor(2));

  /* Set the initial player color edit color to black */
  COLOR_CopyColor(COLOR_PlayerToColor(0), COLOR_DieToColor(3));
}


/************************************************************************ 
 *  FUNCTION: COLOR_GetColormap
 *  HISTORY: 
 *     09.19.94  ESF  Moved this stuff over here from gui.c.
 *     10.08.94  ESF  Enhanced to reduce flashing with private colormap. 
 *     17.08.95  JC   True colors.
 *     29.08.95  JC   If the server support TrueColor, use it.
 *  PURPOSE: 
 *     Selects an appropriate visual if one is available, and then builds
 *   the required X args to pass to any top-level shells.  If no
 *   appropriate visual is available, then exit.  If a usable PseudoColor 
 *   visual exists, but there aren't enough colors available from the
 *   default colormap, then allocate a private colormap for the 
 *   application.  It allocates a colormap and sets up two mappings,
 *   from the index to the colors, and vice-versa.
 *  NOTES: 
 *     This function is specific to Frisk and sets global variables
 *   cmapColormap, hDisplay, etc.  Change it evantually.
 ************************************************************************/
void COLOR_GetColormap(void *pData, Int32 *piNumArgs, Int32 iNeededColors,
		       Int32 argc, CString *argv)
{
  Widget       wDummy;
  Arg         *pVisualArgs = (Arg *)pData; 
  XVisualInfo  Info;
  Int32        i;

  /* Create a dummy top level shell to learn more about the display */
  wDummy = XtAppInitialize(&appContext, "XFrisk", NULL, 0, 
			   &argc, argv, strResources, NULL, 0);
  hDisplay = XtDisplay(wDummy);
  trueColor = FALSE;

  /* See if there is a TrueColor visual with a depth of 24 bits */
  if (XMatchVisualInfo(hDisplay, DefaultScreen(hDisplay), 
	            	    24, TrueColor, &Info))
    {
      COLOR_InitRGB(&Info);

      trueColor = TRUE;
      COLOR_Depth = 24;
      cmapColormap = DefaultColormap(hDisplay, DefaultScreen(hDisplay));
      for(i=0; i!=MAX_COLORS; i++)
          plCountryToColor[i] = i;
    }

  /* See if there is a TrueColor visual with a depth of 16 bits */
  else if (XMatchVisualInfo(hDisplay, DefaultScreen(hDisplay), 
 			    16, TrueColor, &Info))
    {
      COLOR_InitRGB(&Info);
      trueColor = TRUE;
      COLOR_Depth = 16;
      sixteenBitColor = TRUE;
      cmapColormap = DefaultColormap(hDisplay, DefaultScreen(hDisplay));
      for(i=0; i!=MAX_COLORS; i++)
	plCountryToColor[i] = i;
    }
  
  /* See if there is a PseudoColor visual with a depth of 8 bits */
  else if (XMatchVisualInfo(hDisplay, DefaultScreen(hDisplay), 
		       8, PseudoColor, &Info))
    {
      /* Save the visual for use in the program -- this shouldn't be here,
       * once all of this gets cleaned up.
       */

      COLOR_InitRGB(&Info);
      pVisual = Info.visual;
      COLOR_Depth = 8;

      /* Try to allocate the needed colors from the default colormap.
       * If this fails, then try allocating a private colormap.  If
       * it fails, it is probably because the Display is TrueColor, 
       * or else because there aren't enough free colors in the
       * default colormap.
       */

      cmapColormap = DefaultColormap(hDisplay, DefaultScreen(hDisplay));
      if (!XAllocColorCells(hDisplay, cmapColormap, False, NULL, 0, 
			    plCountryToColor, iNeededColors))
	{
 	  XColor    xColor;

	  /* We must use a private colormap */
#ifdef ENGLISH
	  printf("CLIENT: Using a private colormap.\n");
#endif
#ifdef FRENCH
	  printf("CLIENT: Utilise une palette prive.\n");
#endif
	  cmapColormap = XCreateColormap(hDisplay,
					 RootWindowOfScreen(XtScreen(wDummy)),
					 Info.visual, AllocNone);

  	  /* Since we only need some of the colors, copy the first bunch
  	   * of colors from the default colormap, in the hope that we'll
  	   * get the window manager colors, so that nasty flashing won't
  	   * occur...
  	   */

  	  for (i=0; i!=256 - iNeededColors; i++)
  	    {
  	      xColor.pixel = i;
  	      xColor.flags = DoRed | DoGreen | DoBlue;
  	      XQueryColor(hDisplay, 
  			  XDefaultColormap(hDisplay, DefaultScreen(hDisplay)),
  			  &xColor);
  	      XAllocColor(hDisplay, cmapColormap, &xColor);
	    }

	  /* Allocate colors from this colormap */
	  if (!XAllocColorCells(hDisplay, cmapColormap, False, NULL, 0, 
				plCountryToColor, iNeededColors))
	    {
#ifdef ENGLISH
	      printf("CLIENT: Strange error, could not allocate colors.\n");
#endif
#ifdef FRENCH
	      printf("CLIENT: Erreur trange, pas d'allocation de couleurs.\n");
#endif
	      UTIL_ExitProgram(-1);
            }
	}
    }

  else

    {
      /* Print an error message, deregister the client, and get out! */
#ifdef ENGLISH
      printf("Fatal Error!  Could not find a PseudoColor visual to use,\n"
	     "              or the one found was not deep enough to\n"
	     "              allocate %d colors.\n", iNeededColors);
#endif
#ifdef FRENCH
      printf("Erreur fatale!  Impossible de trouver un PseudoColor visual\n"
	     "                 utiliser ou celui trouv ne permet pas\n"
	     "                d'utiliser %d couleurs.\n", iNeededColors);
#endif
      UTIL_ExitProgram(0);
    }

  /* See if there is a TrueColor visual with a depth of 24 bits */
  if (trueColor)
    {
      /* Set up the arguments */
      *piNumArgs = 0;
      XtSetArg(pVisualArgs[*piNumArgs], XtNvisual, Info.visual); 
      (*piNumArgs)++;
      XtSetArg(pVisualArgs[*piNumArgs], XtNdepth, Info.depth); 
      (*piNumArgs)++;
    }

  else
    {
      /* Set up the arguments */
      *piNumArgs = 0;
      XtSetArg(pVisualArgs[*piNumArgs], XtNvisual, pVisual); 
      (*piNumArgs)++;
      XtSetArg(pVisualArgs[*piNumArgs], XtNdepth, 8); 
      (*piNumArgs)++;  
      XtSetArg(pVisualArgs[*piNumArgs], XtNcolormap, cmapColormap);
      (*piNumArgs)++;
    }

  /* Set up mapping from color to country.  With this set up,
   * we have bidirectional mapping, from country to color and from
   * color to country. 
   */
  memset(piColorToCountry, (Byte)0, sizeof(Int32)*MAX_COLORS);
  for(i=0; i!=iNeededColors; i++)
      piColorToCountry[plCountryToColor[i]] = i;

  /* We don't need this anymore */
  XtDestroyWidget(wDummy);
}


/************************************************************************ 
 *  FUNCTION: COLOR_SetWorldColormap
 *  HISTORY: 
 *     05.12.94  ESF  Created.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void COLOR_SetWorldColormap(Color *cmap)
{
  memcpy(pWorldColors, cmap, sizeof(Color)*iNumColors);
}


/************************************************************************ 
 *  FUNCTION: COLOR_SetWorldColors
 *  HISTORY: 
 *     05.12.94  ESF  Created.
 *     08.03.94  ESF  Fixed loop bug.
 *     17.08.95  JC   true colors.
 *     29.08.95  JC   Moved.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void COLOR_SetWorldColors(void)
{
  XColor   xColor;
  Int32    i, c;

  /* Now read in the colormap, store it, and setup the screen.
   * Note that color == country here.
   */

  for (i=0; i!=iNumColors; i++)
    {
      xColor.flags = DoRed | DoGreen | DoBlue;
      xColor.pixel = COLOR_CountryToColor(i);
      xColor.red   = pWorldColors[i].r << 8;
      xColor.green = pWorldColors[i].g << 8;
      xColor.blue  = pWorldColors[i].b << 8;

      D_Assert(xColor.pixel<=MAX_COLORS, "Pixel out of range.");
      c = pWorldColors[i].b;
      c = (c << 8) + pWorldColors[i].g;
      c = (c << 8) + pWorldColors[i].r;
      colormap[i] = c;
      if (!trueColor)
	XStoreColor(hDisplay, cmapColormap, &xColor);
    }
  if (trueColor)
    {
      Int32   x, y, c;

      for (y = 0; y < pMapImage->height; y++)
	for (x = 0; x < pMapImage->width; x++)
	  {
	    c = COLOR_QueryColor(COLOR_ColorToCountry(XGetPixel(pMapImage, 
								x, y)));
	    XSetForeground(hDisplay, hGC, c);
	    XDrawPoint(hDisplay, pixMapImage, hGC, x, y);
	  }

      XCopyArea(hDisplay, pixMapImage, hWindow, hGC, 
	        0, 0, pMapImage->width, pMapImage->height, 0, 0);
    }
  XFlush(hDisplay);
}


/************************************************************************ 
 *  FUNCTION: COLOR_DieToColor 
 *  HISTORY: 
 *     03.04.94  ESF  Created.
 *     03.05.94  ESF  Fixed bug, wrong offset.
 *     08.28.94  ESF  Fixed bug, wrong argument.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
Int32 COLOR_DieToColor(Int32 iDie)
{
  D_Assert(iDie>=0 && iDie<MAX_PLAYERS, 
	   "Wrong range for die color.");
  return (plCountryToColor[NUM_COUNTRIES+2+MAX_PLAYERS+iDie]);
}


/************************************************************************ 
 *  FUNCTION: 
 *  HISTORY: 
 *     02.05.94  ESF  Created.
 *     05.05.94  ESF  Fixed for new colormap scheme.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
Int32 COLOR_CountryToColor(Int32 iCountry)
{
  /* It's +2 because of the two reserved colors for ocean and lines */
  D_Assert(iCountry>=0 && iCountry<NUM_COUNTRIES+2,
	   "Country out of range.");
  return(plCountryToColor[iCountry]);
}


/************************************************************************ 
 *  FUNCTION: 
 *  HISTORY: 
 *     02.05.94  ESF  Created.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
Int32 COLOR_PlayerToColor(Int32 iPlayer)
{
  D_Assert(iPlayer>=0 && iPlayer<MAX_PLAYERS,
	   "Player out of range.");
  return (plCountryToColor[NUM_COUNTRIES+2+iPlayer]);
}


/************************************************************************ 
 *  FUNCTION: COLOR_ColorCountry
 *  HISTORY: 
 *     02.05.94  ESF  Created.
 *     02.05.94  ESF  Factored out color changing code.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void COLOR_ColorCountry(Int32 iCountry, Int32 iPlayer)
{
  D_Assert(iCountry>=0 && iCountry<NUM_COUNTRIES &&
	   iPlayer>=0 && iPlayer<MAX_PLAYERS,
	   "Bad range for ColorCountry().");
  COLOR_CopyColor(COLOR_PlayerToColor(iPlayer), 
		  COLOR_CountryToColor(iCountry));

  if (COLOR_IsTrueColor())
    {
      Int32   x, y, i, j, c;

      c = COLOR_CountryToColor(iCountry);
      x = RISK_GetTextXOfCountry(iCountry) - 200;
      y = RISK_GetTextYOfCountry(iCountry) - 75;
      XSetForeground(hDisplay, hGC, COLOR_QueryColor(c));
      for (i = 0 ; i < 150; i++)
	for (j = 0; j < 300; j++)
	  {
	    if ((x+j<pMapImage->width) && (y+i<pMapImage->height))
	      if (XGetPixel(pMapImage, x+j, y+i) == c)
		XDrawPoint(hDisplay, pixMapImage, hGC, x+j, y+i);
	  }
      
      XCopyArea(hDisplay, pixMapImage, hWindow, hGC, 
	        x, y, 300, 150, x, y);
    }
}


/************************************************************************ 
 *  FUNCTION: COLOR_ColorToCountry
 *  HISTORY: 
 *     02.05.94  ESF  Created.
 *     05.05.94  ESF  Fixed for new colormap scheme.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
Int32 COLOR_ColorToCountry(Int32 iColor)
{
  D_Assert(iColor>=0 && iColor<MAX_COLORS,
	   "Color out of range.");
  return(piColorToCountry[iColor]);
}


/************************************************************************ 
 *  FUNCTION: COLOR_CopyColor
 *  HISTORY: 
 *     02.05.94  ESF  Created.
 *     17.08.95  JC   true colors.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void COLOR_CopyColor(Int32 iSrc, Int32 iDst)
{
  XColor  xColor;

  D_Assert(iSrc>=COLOR_CountryToColor(0) && 
	   iSrc<=COLOR_DieToColor(3), "Source color out of range.");
  D_Assert(iDst>=COLOR_CountryToColor(0) && 
	   iDst<=COLOR_DieToColor(3), "Dest. color out of range.");

  colormap[iDst] = colormap[iSrc];
  if (!trueColor)
    {
      xColor.flags = DoRed | DoGreen | DoBlue;
      xColor.pixel = iSrc;
      XQueryColor(hDisplay, cmapColormap, &xColor);

      xColor.pixel = iDst;
      XStoreColor(hDisplay, cmapColormap, &xColor);
    }

  XFlush(hDisplay);
}


/************************************************************************ 
 *  FUNCTION: COLOR_StoreNamedColor
 *  HISTORY: 
 *     05.03.94  ESF  Created.
 *     04.06.95  ESF  Fixed bug, store closest displayable color.
 *     17.08.95  JC   true colors.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void COLOR_StoreNamedColor(CString strPlayerColor, Int32 iPlayer) 
{
  XColor xColor, xColorClosest;

  D_Assert(iPlayer>=0 && iPlayer<MAX_PLAYERS,
	   "Player out of range.");
  D_Assert(cmapColormap!=0, "Colormap is not there!");

  /* Get the closest color we can display on this screen */
  XLookupColor(hDisplay, cmapColormap, strPlayerColor, 
	       &xColor, &xColorClosest);

  /* Store it */
  colormap[COLOR_PlayerToColor(iPlayer)] = 
    COLOR_rgbToNum(xColorClosest.red, xColorClosest.green,
		   xColorClosest.blue);

  if (!trueColor)
    {
      xColorClosest.flags = DoRed | DoGreen | DoBlue;
      xColorClosest.pixel = COLOR_PlayerToColor(iPlayer);
      XStoreColor(hDisplay, cmapColormap, &xColorClosest);
    }
}


/************************************************************************ 
 *  FUNCTION: COLOR_StorePlayerColor
 *  HISTORY: 
 *     01.22.95  ESF  Created.
 *     17.08.95  JC   true colors.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void COLOR_StoreColor(Int32 iColor, Int32 iRed, Int32 iGreen, Int32 iBlue)
{
  XColor xColor;
  Int32  c = COLOR_rgbToNum(iRed, iGreen, iBlue);

  D_Assert(iColor>=0 && iColor<=255, "Color out of range!");

  colormap[iColor] = c;

  if (!trueColor)
    {
      xColor.pixel = iColor;
      xColor.flags = DoRed | DoGreen | DoBlue;
      xColor.red   = iRed;
      xColor.green = iGreen;
      xColor.blue  = iBlue;
      XStoreColor(hDisplay, cmapColormap, &xColor);
    }
}


/************************************************************************ 
 *  FUNCTION: COLOR_GetColor
 *  HISTORY: 
 *     02.12.95  ESF  Created.
 *  PURPOSE: 
 *  NOTES: 
 ************************************************************************/
void COLOR_GetColor(Int32 iColor, UInt16 *iRed, UInt16 *iGreen, UInt16 *iBlue)
{
  Int32 c;

  D_Assert(iColor>=0 && iColor<=255, "Color out of range!");

  COLOR_numTorgb(colormap[iColor], iRed, iGreen, iBlue);
}
