/*
  gemdropx.c
  
  Gem Drop for X Window
  
  beta release 3
  
  based directly on GEMDROP.ACT, the original Atari 8-bit version of Gem Drop
  
  by Bill Kendrick
  kendrick@zippy.sonoma.edu
  http://zippy.sonoma.edu/kendrick/
  
  Atari version: August 17, 1997 - Sept. 24, 1997
  X Window version: November 5, 1997 - November 21, 1997
*/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <math.h>
#include <sys/types.h>
#include <sys/timeb.h>
#include "connect.h"
#include "window.h"
#include "hints.h"
#include "text.h"
#include "gc.h"
#include "color.h"
#include "visual.h"
#include "sleepfor.h"


/* Misc. value defines: */

#define NO 0
#define YES 1


/* Object specifications: */

#define NUM_NORMAL_BLOCKS 4
#define NUM_ALL_BLOCKS 8
#define BOMB NUM_ALL_BLOCKS + 1
#define CLOCK NUM_ALL_BLOCKS + 2
#define WILDCARD NUM_ALL_BLOCKS + 3
#define NUM_SPECIALS 3
#define NUM_EXPLOSION_GFX 4
#define EXPLOSION_SLOW_VAL 1
#define HAPPY_GFX NUM_ALL_BLOCKS + NUM_SPECIALS + NUM_EXPLOSION_GFX + 1
#define SAD_GFX HAPPY_GFX + 1
#define WIN1_GFX SAD_GFX + 1
#define WIN2_GFX WIN1_GFX + 1
#define WARNING_GFX WIN2_GFX + 1
#define BRICK_GFX WARNING_GFX + 1
#define TITLE_GFX BRICK_GFX + 1
#define AUTHOR_GFX TITLE_GFX + 1
#define NUM_OBJECTS AUTHOR_GFX


/* Game stuff: */

#define FRAME_RATE 100
#define HEIGHT 11
#define WIDTH 10
#define NUM_EXPLOSIONS 100


/* Number of colors for title animation: */

#define NUM_TITLE_COLORS 6


/* Explosion type: */

typedef struct explosion_type {
  int exist;     /* do I exist? */
  int x, y;      /* position */
  int anim;      /* animation frame counter */
  int animrate;  /* pause for animation */
} explosion_type;


/* Object (graphics) filenames: */

char * object_filenames[NUM_OBJECTS] = {"block", "star", "diamond", "disc",
					"dot", "face", "fuji", "triangle",
					"bomb", "stopwatch", "wildcard",
					"explode0", "explode1",
					"explode2", "explode3",
					"happy", "sad", "win1", "win2",
					"warning", "brick", "title", "author"};


/* Object color names: */

char * object_colors[NUM_OBJECTS] = {"green", "yellow", "blue", "red",
				     "red", "orange", "white", "blue",
				     "white", "yellow", "green",
				     "yellow", "yellow",
				     "orange", "red",
				     "yellow", "yellow", "yellow", "yellow",
				     "red", "red", "white", "white"};


/* Title color names: */

char * title_colors[NUM_TITLE_COLORS] = {"red", "orange", "yellow",
					 "green", "blue", "purple"};


/* Player color names: */

char * man1_color = "yellow";
char * man2_color = "blue";


/* Globals: */

explosion_type explosions[NUM_EXPLOSIONS];
int blocks[HEIGHT][WIDTH], killsx[100], killsy[100];
int scorevals[15];
int level, gameover, frozen, carrying, howmany, leveldone, warning,
  whichexplosion, lines, firstround, linesneeded, happy, score, playerx,
  use_sound;


/* X Window stuff: */

Display * display;
Colormap colormap;
Window root, window;
Pixmap object_pixmaps[NUM_OBJECTS];
XFontStruct * font;
int screen, black, white, fh, windowwidth, windowheight, has_color;
GC whitegc, blackgc;
GC colorgcs[NUM_OBJECTS], titlecolorgcs[NUM_TITLE_COLORS];
GC man1gc, man2gc;
XEvent event;
char key[1024];
KeySym keysym;
XComposeStatus composestatus;


#define EVENT_MASK (KeyPressMask | ExposureMask | ButtonPressMask)
#define VISUAL (Visual *)CopyFromParent


void playsound(char * aufile)
{
  char cmd[1024];
  
  if (use_sound == 1)
    {
       sprintf(cmd, "/bin/cat sounds/%s.au > /dev/audio &", aufile);
  
       system(cmd);
    }
}


/* Beep: (was flash and beep on the Atari) */

void complain(void)
{
  XBell(display, 0);
}


/* Clear the screen: */

void erasewindow(void)
{
  XFillRectangle(display, window, blackgc, 0, 0, windowwidth, windowheight);
}


/* Get a color, or some other color, or even some other color...
   it depends on what we have available to us: */

unsigned long MyAllocNamedColor(Display *display, Colormap colormap,
                                char* colorname, unsigned long default_color,
                                int has_color)
{
  if (has_color == 0)
    return(default_color);
  else
    return(AllocNamedColor(display, colormap, colorname, default_color));
}


/* Setup the application: */

void setup(char * server)
{
  int i, color, tempint, ret, tempwidth, tempheight;
  char title[1024], file[128];
  int status, temp_depth;
  Visual * temp_visual = CopyFromParent;
  FILE * fi;
  
  
  /* Connect to display: */
  
  display = ConnectToServer(server, &screen, &root);
  if (display == NULL)
    {
      perror(server);
      exit(1);
    }


  /* Get our primitve colors: */

  black = BlackPixel(display, screen);
  white = WhitePixel(display, screen);


  /* Load the text font: */
  
  font = LoadFont(display, "variable", "fixed");
  fh = FontHeight(font);


  /* Set the size of the window: */
  
  windowwidth = WIDTH * 32;
  windowheight = (HEIGHT + 1) * 32 + fh * 2;
  

  /* Open window: */

  window = OpenWindow(display, root, 10, 10, windowwidth, windowheight,
		      CopyFromParent, black, EVENT_MASK, VISUAL);
  
  sprintf(title, "Gem Drop X!");
  SetStandardHints(display, window, "Gem Drop X", title, 10, 10,
		   windowwidth, windowheight);
  
  
  /* Create GC's: */
  
  whitegc = CreateGC(display, window, white, black);
  blackgc = CreateGC(display, window, black, white);
  
  has_color = 0;
  
  if (SetUpVisual(display, screen, &temp_visual, &temp_depth))
    {
      if (!SetUpColormap(display, screen, window, temp_visual,
			 &colormap))
	{
	  fprintf(stderr, "Could not create a colormap!\n");
	  /* XCloseDisplay(display);
	     exit(1); */
	}
      else
	has_color = 1;
    }
  else
    {
      fprintf(stderr, "Could not find a PseudoColor visual!\n");
      /* XCloseDisplay(display);
	 exit(1); */
    }
  
  for (i = 0; i < NUM_OBJECTS; i++)
    {
      /* printf("Loading color: %s\n", object_colors[i]); */
      
      colorgcs[i] = CreateGC(display, window,
			     MyAllocNamedColor(display, colormap,
					       object_colors[i],
					       (unsigned long) white,
					       has_color),
			     black);
    }
  
  for (i = 0; i < NUM_TITLE_COLORS; i++)
    {
      /* printf("Loading color: %s\n", title_colors[i]); */
      
      titlecolorgcs[i] = CreateGC(display, window,
				  MyAllocNamedColor(display, colormap,
						    title_colors[i],
						    (unsigned long) white,
						    has_color),
				  black);
    }
  
  man1gc = CreateGC(display, window,
		    MyAllocNamedColor(display, colormap,
				      man1_color,
				      (unsigned long) white,
				      has_color),
		    black);
  
  man2gc = CreateGC(display, window,
		    MyAllocNamedColor(display, colormap,
				      man2_color,
				      (unsigned long) white,
				      has_color),
		    black);
  
  
  /* Load the object graphics: */
  
  for (i = 0; i < NUM_OBJECTS; i++)
    {
      sprintf(file, "gfx/%s.xbm", object_filenames[i]);
      
      /* printf("Loading image: %s\n", file); */
      
      ret = XReadBitmapFile(display, window, file, &tempwidth, &tempheight,
			    &object_pixmaps[i], &tempint, &tempint);
      
      if (ret != BitmapSuccess)
	{
	  perror(file);
	  exit(1);
	}
    }
  
  
  
  /* Initial level: */
  
  level = 1;
  
  
  /* Reset explosions: */
  
  for (i = 0; i < NUM_EXPLOSIONS; i++)
    {
      explosions[i].exist = NO;
    }
  
  
  /* Init. score value table: */
  
  scorevals[0] = 0;
  scorevals[1] = 0;
  scorevals[2] = 0;
  scorevals[3] = 10;
  scorevals[4] = 20;
  scorevals[5] = 50;
  scorevals[6] = 100;
  scorevals[7] = 150;
  scorevals[8] = 200;
  scorevals[9] = 250;
  scorevals[10] = 300;
  scorevals[11] = 500;
  scorevals[12] = 1000;
  scorevals[13] = 1000;
  scorevals[14] = 5000;
  
  
  /* Reset random number generator: */
  
  randinit();
  
  
  /* Put us up!: */
  
  XMapWindow(display, window);
  XMapRaised(display, window);
}


/* Update the score in the window: */

void drawscore()
{
  char string[128];
  
  sprintf(string, "Score:  %d  ", score);
  drawtext(display, window, whitegc, 0, (HEIGHT + 1) * 32 + fh, string);
}


/* Pick a random block: */

int randblock(void)
{
  int i;
  
  
  /* Pick one of the main four pieces in level 1-14: */
  
  i = randnum(NUM_NORMAL_BLOCKS + 1);
  
  
  /* Pick one of all eight pieces in levels 15-20: */
  
  if (level >= 15)
    i = randnum(NUM_ALL_BLOCKS + 1);
  
  
  /* Maybe pick a non-gem object: */
  
  if (randnum(40) < 1)
    i = randnum(NUM_SPECIALS) + NUM_ALL_BLOCKS + 1;
  
  return(i);
}


/* Draw an object graphic somewhere in the window: */

void drawblockgraphic(int x, int y, int c)
{
  if (c != 0)
    {
      XCopyPlane(display, object_pixmaps[c - 1], window, colorgcs[c - 1],
		0, 0, 32, 32, x, y, 1);
    }
  else
    {
      XFillRectangle(display, window, blackgc, x, y, 32, 32);
    }
}


/* Update the status icons at the bottom: */

void drawstatus()
{
  if (frozen != 0)
    drawblockgraphic((WIDTH / 2) * 32, (HEIGHT + 1) * 32, CLOCK);
  else
    drawblockgraphic((WIDTH / 2) * 32, (HEIGHT + 1) * 32, 0);
  
  if (warning != 0)
    drawblockgraphic((WIDTH / 2) * 32 + 32, (HEIGHT + 1) * 32, WARNING_GFX);
  else
    drawblockgraphic((WIDTH / 2) * 32 + 32, (HEIGHT + 1) * 32, 0);    
}


/* Draw the player (it's two colors): */

void drawmangraphic(int x, int y, int c)
{
  if (c != 0)
    {
      XCopyPlane(display, object_pixmaps[c - 1], window, man1gc,
		 0, 0, 32, 16, x, y, 1);
      XCopyPlane(display, object_pixmaps[c - 1], window, man2gc,
		 0, 16, 32, 16, x, y + 16, 1);
    }
}


/* Draw a block where it belongs: */

void drawblock(int x, int y)
{
  int c;
  
  
  /* Draw the block at (x, y): */
  
  c = blocks[y][x];
  
  drawblockgraphic(x * 32, y * 32, c);
}


/* Update the level number display: */

void updatelevel(void)
{
  char string[128];
  
  sprintf(string, "Level:  %d  ", level);
  drawtext(display, window, whitegc, 0, (HEIGHT + 1) * 32 + fh * 2, string);
}


/* Update the lines / needed lines displays: */

void updatelines(void)
{
  char string[128];
  int x, y;
  
  y = (HEIGHT + 1) * 32;
  
  x = WIDTH * 32 - XTextWidth(font, "xxxxxx: ---", 11);
  
  sprintf(string, "Lines:  %3d", lines);
  drawtext(display, window, whitegc, x, y + fh, string);
  
  sprintf(string, "Needed: %3d", linesneeded);
  drawtext(display, window, whitegc, x, y + fh * 2, string);
}


/* Get a key... mouse is seen as key, too!
   Maybe handle expose events
   (the original source that this was ported from was obviously
   not written for an event-based system): */

void getkey(int handle_expose)
{
  int x, y;
  
  
  strcpy(key, "");
  
  if (XPending(display))
    {
      XNextEvent(display, &event);
      
      if (event.type == KeyPress)
	{
	  /* Get the key's name: */
	  
	  XLookupString(&event.xkey, key, 1, &keysym, &composestatus);
	  
	  if (XKeysymToString(keysym) != NULL)
	    {
	      strcpy(key, XKeysymToString(keysym));
	    }
	}
      else if (event.type == Expose && handle_expose == 1)
	{
	  /* Redraw everything: */
	  
	  for (y = 0; y < HEIGHT; y++)
	    for (x = 0; x < WIDTH; x++)
	      drawblock(x, y);
	  
	  updatelines();
	  drawscore();
	  drawstatus();
	  updatelevel();
	}
      else if (event.type == ButtonPress)
	{
	  /* Mouse counts as keyboard: */
	  
	  if (event.xbutton.y >= HEIGHT * 32)
	    {
	      /* Left or right of man at the bottom counts as left/right: */
	      
	      if (event.xbutton.x < playerx * 32)
		strcpy(key, "Left");
	      else if (event.xbutton.x >= playerx * 32 + 31)
		strcpy(key, "Right");
	    }
	  else
	    {
	      /* Left or right click above man count as grab/throw: */
	      
	      if (event.xbutton.button == Button1)
		strcpy(key, "Down");
	      else
		strcpy(key, "Up");
	    }
	}
    }
}


/* Show the title... let them change the level: */

int title(void)
{
  int quit, ok, color, title_width, title_height, tempint, height,
    author_width, author_height, author_x, author_y, author,
    author_xm, author_ym;
  Window tempwindow;
  struct timeb timethen, timenow;
  
  
  /* Find out size of bitmaps we're using in the title screen: */
  
  XGetGeometry(display, object_pixmaps[TITLE_GFX - 1], &tempwindow,
	       &tempint, &tempint, &title_width, &title_height, &tempint,
	       &tempint);
  
  XGetGeometry(display, object_pixmaps[AUTHOR_GFX - 1], &tempwindow,
	       &tempint, &tempint, &author_width, &author_height, &tempint,
	       &tempint);
  
  author_x = WIDTH * 32 - author_width;
  author_y = HEIGHT * 32 - author_height - randnum(10) - 30;
  
  
  /* Clear the screen: */
  
  erasewindow();
  
  
  /* Draw credits / title / version: */
  
  drawcenteredtext(display, window, whitegc, 0,
		   windowwidth, windowheight / 2 + fh * 2, 
		   "by Bill Kendrick", font);
  
  drawcenteredtext(display, window, whitegc, 0,
		   windowwidth, windowheight / 2 + fh * 3,
		   "New Breed Software 1997", font);
  
  drawcenteredtext(display, window, whitegc, 0,
		   windowwidth, windowheight / 2 + fh * 5,
		   "beta release 3", font);
  
  
  /* Draw the currently-selected level: */
  
  updatelevel();
  
  XFlush(display);
  
  
  /* Init. animation variables: */
  
  author = 0;
  color = 0;
  height = 0;
  do
    {
      author_xm = -randnum(3);
    }
  while (author_xm == 0);
  author_ym = randnum(6) - 3;
  
  
  /* We haven't quit and we haven't started a game: */
  
  quit = NO;
  ok = NO;
  
  
  /* Title animation loop: */
  
  ftime(&timethen);
  
  do
    {
      ftime(&timenow);
      
      /* Do animation every 1/8th of a second: */
      
      if (timenow.time > timethen.time ||
	  timenow.millitm > timethen.millitm + 125)
	{
	  /* Change colors: */
	  
	  color = color + 1;
	  if (color >= NUM_TITLE_COLORS)
	    color = 0;
	  
	  
	  /* Show more and more of the title: */
	  
	  height = height + 10;
	  if (height > title_height)
	    height = title_height;
	  
	  
	  /* Draw it: */
	  
	  XCopyPlane(display, object_pixmaps[TITLE_GFX - 1], window,
		     titlecolorgcs[color],
		     0, 0, title_width, height,
		     (WIDTH * 32 - title_width) / 2, 0, 1);
	  
	  
	  /* Hmm!? */
	  
	  author = author + 1;
	  
	  if (author > 1000)
	    {
	      XFillRectangle(display, window, blackgc, author_x, author_y,
			     author_width, author_height);
	      
	      author_x = author_x + author_xm;
	      if (author_x < 0)
		{
		  author_x = 0;
		  do
		    {
		      author_xm = randnum(3);
		    }
		  while (author_xm == 0);
		}
	      if (author_x > WIDTH * 32 - author_width)
		{
		  author_x = WIDTH * 32 - author_width;
		  do
		    {
		      author_xm = -randnum(3);
		    }
		  while (author_xm == 0);
		}
	      
	      author_y = author_y + author_ym;
	      if (author_y < HEIGHT * 32 / 2)
		{
		  author_y = HEIGHT * 32 / 2;
		}
	      if (author_y > HEIGHT * 32 - author_height)
		{
		  author_y = HEIGHT * 32 - author_height;
		  author_ym = -author_ym;
		}
	      
	      author_ym = author_ym + 1;
	      if (author_ym > 10)
		author_ym = 10;
	      
	      if (author < 1500)
		{
		  XCopyPlane(display, object_pixmaps[AUTHOR_GFX - 1], window,
			     colorgcs[AUTHOR_GFX - 1],
			     0, 0, author_width, author_height,
			     author_x, author_y, 1);
		}
	      else
		{
		  author = 0;
		}
	    }

	  ftime(&timethen);
	}
      
      
      /* Check for keypresses: */
      
      getkey(0);
      
      if (strcmp(key, "Escape") == 0 ||
	  strcasecmp(key, "q") == 0)
	{
	  /* Escape to quit: */
	  
	  ok = YES;
	  quit = YES;
	}
      else if (strcasecmp(key, "L") == 0)
	{
	  /* L for level select: */
	  
	  level = level + 1;
	  if (level > 20)
	    level = 1;
	  
	  playsound("levelselect");
	  
	  updatelevel();
	}
      else if (strcmp(key, "Return") == 0 ||
	       strcmp(key, "space") == 0)
	{
	  /* Return or Space to begin: */
	  
	  ok = YES;
	}
      else if (strcasecmp(key, "A") == 0)
	{
	  /* Hmm!? */
	  
	  if (author < 1000)
	    author = 1000;
	  else
	    author = 1495;
	}
    }
  while (ok != YES);
  
  return(quit);
}


/* See if you're happy or sad...  Depends on if there are any blocks
   close to the bottom: */

void checkhappy()
{
  int x, y;
  
  happy = YES;
  for (y = HEIGHT - (HEIGHT / 3); y < HEIGHT; y++)
    {
      for (x = 0; x < WIDTH; x++)
	{
	  if (blocks[y][x] != 0)
	    happy = NO;
	}
    }
}


/* Init this level (randomly place blocks, etc.) */

void initlevel(void)
{
  int x, y, yy;
  
  
  playsound("begin");
  
  
  /* Erase all blocks: */
  
  for (y = 0; y < HEIGHT; y++)
    for (x = 0; x < WIDTH; x++)
      blocks[y][x] = 0;
  
  
  /* How low should they go? */
  
  yy = level;
  if (level > 14)
    yy = yy - (HEIGHT - 2);
  
  if (yy > HEIGHT - 2)
    yy = HEIGHT - 2;
  
  
  /* Place random blocks: */
  
  for (y = 0; y <= yy; y++)
    for (x = 0; x < WIDTH; x++)
      blocks[y][x] = randblock();
  
  
  /* Draw all blocks: */
  
  for (y = 0; y < HEIGHT; y++)
    for (x = 0; x < WIDTH; x++)
      drawblock(x, y);
  
  
  /* Init level variables: */
  
  leveldone = NO;
  
  carrying = 0;
  howmany = 0;
  frozen = 0;
  warning = NO;
  firstround = NO;
  
  
  /* Start out happy or not, depending on where the pieces are: */
  
  checkhappy();

  
  /* How many lines do you need / have? */
  
  lines = 0;
  linesneeded = (level * 3) + 2;
  
  
  /* Draw the rest of the game screen stuff: */
  
  drawscore();
  updatelines();
  drawstatus();
  updatelevel();
}


/* Erase player: */

void eraseyou(int x)
{
  XFillRectangle(display, window, blackgc, x * 32 - 32, HEIGHT * 32,
		 96, 32);
}


/* Draw player and objects he's carrying: */

void drawyou(int x)
{
  int y;
  
  
  y = HEIGHT * 32;
  
  if (howmany < 3)
    {
      /* Happy?  Or sad? */
      
      if (happy == YES)
	{
	  drawmangraphic(x * 32, y, HAPPY_GFX);
	}
      else
	{
	  drawmangraphic(x * 32, y, SAD_GFX);
	}
    }
  
  
  if (carrying != 0)
    {
      drawblockgraphic(x * 32 + 32, y, carrying);
      
      if (howmany >= 2)
	drawblockgraphic(x * 32 - 32, y, carrying);
      
      if (howmany >= 3)
	drawblockgraphic(x * 32, y, carrying);
    }
}


/* Assign an explosion: */

void explodeblock(int x, int y, int explosionwait)
{
  /* Pick an explosion entity to use: */
  
  whichexplosion = whichexplosion + 1;
  
  if (whichexplosion == NUM_EXPLOSIONS)
    whichexplosion = 0;
  
  
  /* Init. its values: */
  
  explosions[whichexplosion].exist = 1;
  explosions[whichexplosion].anim = -explosionwait;
  explosions[whichexplosion].animrate = 0;
  explosions[whichexplosion].x = x;
  explosions[whichexplosion].y = y;
}


/* See if two objects match: */

int same(int a, int b)
{
  int match;
  
  
  /* A match if it's the same, or the object (A) is a special object: */
  
  match = NO;
  
  if (a == b || a > NUM_ALL_BLOCKS || b > NUM_ALL_BLOCKS)
    match = YES;
  
  return(match);
}


/* Kill a block (may call itself recursively!) */

int killblock(int x, int y, int killed)
{
  int c, explosionwait;
  
  
  /* Values we use a lot later: */
  
  explosionwait = killed;
  c = blocks[y][x];
  
  
  if (c != 0)
    {
      /* Remove the block: */
      
      blocks[y][x] = 0;
      
      
      /* Put an explosion there: */
      
      explodeblock(x, y, explosionwait);
      
      
      if (c == BOMB)
	{
	  /* If it's a bomb, explode stuff next to it: */
	  
          playsound("bomb");
	  
	  if (y > 0)
	    {
	      blocks[y - 1][x] = 0;
	      explodeblock(x, y - 1, explosionwait);
	    }
	  
	  if (y < HEIGHT - 1)
	    {
	      blocks[y + 1][x] = 0;
	      explodeblock(x, y + 1, explosionwait);
	    }
	  
	  if (x > 0)
	    {
	      blocks[y][x - 1] = 0;
	      explodeblock(x - 1, y, explosionwait);
	    }
	  
	  if (x < WIDTH - 1)
	    {
	      blocks[y][x + 1] = 0;
	      explodeblock(x + 1, y, explosionwait);
	    }
	}
      else if (c == CLOCK)
	{
	  /* Hit a clock!  Freeze the game! */
	     
	  playsound("clock");
	  
	  frozen = 50;
	  drawstatus();
	}
      
      
      /* Kill stuff next to you if they match! */
      
      if (y > 0)
	{
	  if (same(blocks[y - 1][x], c) == YES)
	    killed = killblock(x, y - 1, killed + 1);
	}
      
      if (y < HEIGHT - 1)
	{
	  if (same(blocks[y + 1][x], c) == YES)
	    killed = killblock(x, y + 1, killed + 1);
	}
      
      if (x > 0)
	{
	  if (same(blocks[y][x - 1], c) == YES)
	    killed = killblock(x - 1, y, killed + 1);
	}
      
      if (x < WIDTH - 1)
	{
	  if (same(blocks[y][x + 1], c) == YES)
	    killed = killblock(x + 1, y, killed + 1);
	}
    }
  
  if (killed > 14)
    killed = 14;
  
  return(killed);
}


/* Throw your gems: */

void throw(int x)
{
  int lasty, killed;
  int y, c, last, nextlast, ytop, ybot, kx, ky, ok, doit;
  
  
  if (carrying != 0)
    {
      last = 0;
      nextlast = 0;
      lasty = -1;
      
      
      /* Find the lowest point that they'll attach to: */
      
      for (y = 0; y < HEIGHT; y++)
	{
	  c = blocks[y][x];
	  
	  if (c != 0)
	    {
	      nextlast = last;
	      last = c;
	      lasty = y;
	    }
	}
      
      
      /* See if there's a match: */
      
      ok = NO;
      
      if (same(last, carrying) == YES)
	{
	  if (same(nextlast, carrying) == YES || howmany > 1)
	    ok = YES;
	}
      
      doit = YES;
      
      ybot = lasty + 1 + howmany - 1;
      
      
      /* Don't go past the bottom: */
      
      if (ybot > 10)
	{
	  doit = NO;
	  ybot = 10;
	  if (ok == YES || howmany > 2)
	    doit = YES;
	}
      
      
      /* These'll fit?  Put them on the screen: */
      
      if (doit == YES)
	{
	  for (y = lasty + 1; y <= ybot; y++)
	    {
	      blocks[y][x] = carrying;
	      drawblock(x, y);
	    }
	}
      
      
      /* See if it's a match of 3 or more: */
      
      ok = NO;
      
      if (same(last, carrying) == YES)
	{
	  if (same(nextlast, carrying) == YES || howmany > 1)
	    ok = YES;
	}
      
      if (howmany > 2)
	ok = YES;
      
      
      /* It IS! */
      
      if (ok == YES)
	{
	  ytop = 0;
	  for (y = 0; y <= ybot; y++)
	    {
	      if (blocks[y][x] != carrying)
		ytop = y + 1;
	    }
	  
	  killed = killblock(x, ybot, 1);
	  
	  score = score + scorevals[killed];
	  
	  lines = lines + 1;
	  
	  drawscore();
	  updatelines();
	}
      
      if (doit == YES)
	{
	  carrying = 0;
	  howmany = 0;
	  
	  playsound("got");
	}
    }
  else
    {
      complain();
    }
  
  
  /* See if the level's done!? */
  
  if (lines >= linesneeded)
    leveldone = YES;
}


/* Grab some gems */

void grab(int x)
{
  int y, c, last, lasty, b, ok;
  
  
  /* Find the lowest gem on the column: */
  
  last = 0;
  lasty = 0;
  
  for (y = 0; y < HEIGHT; y++)
    {
      c = blocks[y][x];
      if (c != 0)
	{
	  last = c;
	  lasty = y;
	}
    }
  
  
  /* See if we got one: */
  
  if (last == 0 || last > NUM_ALL_BLOCKS)
    {
      /* Nope! */
      
      complain();
    }
  else
    {
      /* Yep!: */
      
      if (last != carrying && carrying != 0)
	{
	  /* Not the same as what we're carrying, though! */
	  
	  complain();
	}
      else
	{
	  /* Grab it: */
	  
	  carrying = last;
	  blocks[lasty][x] = 0;
	  drawblock(x, lasty);
	  howmany = howmany + 1;
	  
	  
	  /* Grab any of the same type that are above it: */
	  
	  do
	    {
	      ok=0;
	      lasty = lasty - 1;
	      if (lasty >= 0)
		{
		  if (blocks[lasty][x] == last)
		    {
		      blocks[lasty][x] = 0;
		      drawblock(x, lasty);
		      ok = 1;
		      howmany = howmany + 1;
		    }
		}
	    }
	  while (ok != 0);
	}
    }
}


/* Add more gems at the top of the screen: */

void addmore()
{
  int x, y;
  

  /* See if the game's over: */
  
  for (x = 0; x < WIDTH; x++)
    {
      if (blocks[HEIGHT - 1][x] != 0)
	gameover = YES;
    }
  
  
  if (gameover == NO)
    {
      /* Push the existing ones down: */
      
      for (y = HEIGHT - 1; y > 0; y--)
	{
	  for (x = 0; x < WIDTH; x++)
	    {
	      blocks[y][x] = blocks[y - 1][x];
	      drawblock(x, y);
	    }
	}
      
      
      /* Add random ones to the top: */
      
      for (x = 0; x < WIDTH; x++)
	{
	  blocks[0][x] = randblock();
	  drawblock(x, 0);
	}
    }
}


/* Update the explosion animations: */

int drawexplosions(void)
{
  int i, any;
  
  
  /* We'll say if there are any more or not: */
  
  any = 0;
  
  
  for (i = 0; i < NUM_EXPLOSIONS; i++)
    {
      if (explosions[i].exist == YES)
	{
	  /* (There are some!) */
	  
	  any = 1;
	  
	  
	  /* There's an animation slow-down... */
	  
	  explosions[i].animrate++;
	  
	  if (explosions[i].animrate >= EXPLOSION_SLOW_VAL)
	    {
	      explosions[i].animrate = 0;
	      explosions[i].anim++;
	      
	      if (explosions[i].anim >= NUM_EXPLOSION_GFX)
		{
		  /* It's gone!  Draw whatever was/is behind it: */
		  
		  explosions[i].exist = NO;
		  drawblock(explosions[i].x, explosions[i].y);
		}
	      else if (explosions[i].anim >= 0)
		{
		  /* Draw the animation frame: */
		  
		  drawblockgraphic(explosions[i].x * 32, explosions[i].y * 32,
				   NUM_ALL_BLOCKS + NUM_SPECIALS + 1 +
				   explosions[i].anim);
		}
	    }
	}
    }
  
  XFlush(display);
  
  return(any);
}      


/* End-of-level effect (major destruction, dude!) */

void levelendfx(int yourx)
{
  int x, y, i, any, win;


  /* Let all of old the explosions "fizzle out"... */
  
  do
    {
      /* Update explosions: */
      
      any = drawexplosions();
      
      XFlush(display);
      
      
      /* Wait a little: */
      
      sleepfor(FRAME_RATE);
    }
  while(any == 1);
  
  
  /* Toggle value for winning animation: */
  
  win = 0;
  
  
  /* Explode from the top to the bottom: */
  
  playsound("win");
  
  for (y = 0; y < HEIGHT; y++)
    {
      /* Erase this row: */
      
      for (x = 0; x < WIDTH; x++)
	{
	  blocks[y][x] = 0;
	  drawblock(x, y);
	}
      
      
      /* Add some explosions: */
      
      for (x = 0; x < WIDTH; x++)
	{
	  if (randnum(10) < 6)
	    explodeblock(x, y, -randnum(10));
	}
      
      
      /* Animate the dude: */
      
      win = 1 - win;
      drawmangraphic(yourx * 32, HEIGHT * 32, WIN1_GFX + win);
      
      
      /* Update the explosions: */
      
      drawexplosions();
      
      XFlush(display);
      
      
      /* Wait a little: */
      
      sleepfor(FRAME_RATE);
    }
  
  
  /* Let all of the explosions "fizzle out"... */
  
  do
    {
      /* Update explosions: */
      
      any = drawexplosions();
      
      
      /* Draw man: */
      
      win = 1 - win;
      drawmangraphic(yourx * 32, HEIGHT * 32, WIN1_GFX + win);
      
      XFlush(display);
      
      
      /* Wait a little: */
      
      sleepfor(FRAME_RATE);
    }
  while(any == 1);
}


/* Level 15 effect (none yet!) */

void level15fx()
{
  /* ??? level 15 effect */
}


/* MAIN GAME FUNCTION!!!! */

void play()
{
  int i, s, k, a, b, fire, segafire, activity, oact,
    clicks, q, loc, x, y, toggle;
  struct timeb timebefore, timenow;
  char string[128];
  
  
  /* Init the game variables: */
  
  initlevel();
  clicks = 0;
  playerx = 5;
  gameover = NO;
  oact = 0 ;
  score = 0;
  
  
  /* Draw you! */
  
  drawyou(playerx);
  
  
  /* MAIN GAME LOOP: */
  
  do
    {
      ftime(&timebefore);
      
      
      /* Event loop: */
      
      do
	{
	  /* Get any keys: */
	  
	  getkey(1);
	  
	  if (strlen(key) != 0)
	    {
	      if (strcmp(key, "Escape") == 0 ||
		  strcasecmp(key, "q") == 0)
		{
		  /* Escape or Q to quit: */
		  
		  gameover = YES;
		}
	      else if (strcmp(key, "KP_Right") == 0 ||
		       strcmp(key, "Right") == 0 ||
		       strcasecmp(key, "L") == 0)
		{
		  /* Right to move you right */
		  
		  eraseyou(playerx);
		  playerx++;
		  
		  
		  /* Wrap around: */
		  
		  if (playerx > WIDTH - 1)
		    playerx = 0;
		}
	      else if (strcmp(key, "KP_Left") == 0 ||
		       strcmp(key, "Left") == 0 ||
		       strcasecmp(key, "K") == 0)
		{
		  /* Left to move you left: */
		  
		  eraseyou(playerx);
		  playerx--;
		  
		  
		  /* Wrap around: */
		  
		  if (playerx < 0)
		    playerx = WIDTH - 1;
		}
	      else if (strcmp(key, "KP_Up") == 0 ||
		       strcmp(key, "Up") == 0 ||
		       strcasecmp(key, "A") == 0)
		{
		  /* Up to throw: */
		  
		  eraseyou(playerx);
		  throw(playerx);
		}
	      else if (strcmp(key, "KP_Down") == 0 ||
		       strcmp(key, "Down") == 0 ||
		       strcasecmp(key, "Z") == 0)
		{
		  /* Down to grab: */
		  
		  eraseyou(playerx);
		  grab(playerx);
		}
	      else if (strcmp(key, "Return") == 0)
		{
		  /* Return to get more gems NOW!: */
		  
		  firstround = YES;
		  frozen = NO;
		  clicks = 32767;
		  drawstatus();
		}
	      else if (strcasecmp(key, "S") == 0)
		{
		  use_sound = 1 - use_sound;
		  
		  if (use_sound == 0)
		    printf("(Sound off)\n");
		  else
		    printf("(Sound on)\n");
		}
	      else if (strcmp(key, "space") == 0 ||
		       strcasecmp(key, "P") == 0)
		{
		  /* Space or P to pause... wait for keypress to continue: */
		  
		  eraseyou(playerx);
		  
		  toggle = 0;
		  
		  do
		    {
		      /* Pause for a frame: */
		      
		      ftime(&timebefore);
		      do
			{
			  getkey(1);
			  ftime(&timenow);
			}
		      while (timenow.millitm < (timebefore.millitm +
						FRAME_RATE * 2) &&
			     timenow.time <= timebefore.time &&
			     strlen(key) == 0);
		      
		      
		      /* Flash the word "paused": */
		      
		      toggle = 1 - toggle;
		      
		      if (toggle == 1)
			{
			  drawcenteredtext(display, window, whitegc, 0,
					   windowwidth, (HEIGHT + 1) * 32 - fh,
					   "p a u s e d", font);
			}
		      else
			{
			  drawcenteredtext(display, window, blackgc, 0,
					   windowwidth, (HEIGHT + 1) * 32 - fh,
					   "p a u s e d", font);
			}
		    }
		  while (strlen(key) == 0);
		  
		  XFillRectangle(display, window, blackgc,
				 0, (HEIGHT + 1) * 32 - fh - fh,
				 WIDTH * 32, fh * 2);
		}
	      
	      
	      /* Draw you again, just in case: */
	      
	      drawyou(playerx);
	    }
	  
	  
	  /* Keep doing this until the framerate has passed: */
	  
	  ftime(&timenow);
	}
      while (timenow.millitm < timebefore.millitm + FRAME_RATE &&
	     timenow.time <= timebefore.time);
      
      
      /* Draw explosions: */
      
      drawexplosions();
      
      XFlush(display);
      
      
      /* Move stuff if need be: */
      
      if (frozen == 0)
	{
	  /* No clock... so keep counting til we move stuff down: */
	  
	  checkhappy();
	  
	  clicks = clicks + 1;
	  
	  
	  /* The higher the level, the faster stuff comes: */
	  
	  if (level < 5)
	    q = level * 80;
	  else if (level < 15)
	    q = level * 50;
	  else
	    q = (level - 10) * 40;
	  
	  
	  /* Warn them first... */
	  
	  if (clicks == 720 - q && firstround == YES)
	    {
	      warning = YES;
	      drawstatus();
	    }
	  
	  
	  /* Drop stuff down: */
	  
	  if (clicks > 750 - q)
	    {
	      if (firstround == YES)
		{
		  addmore();
		}
	      
	      clicks = 0;
	      firstround = YES;
	      
	      warning = NO;
	      drawstatus();
	    }
	}
      else
	{
	  /* Count down the clock... */
	  
	  happy = YES;
	  
	  frozen = frozen - 1;
	  
	  if (frozen == 0)
	    drawstatus();
	}
      
      
      /* See if the level's complete! */
      
      if (leveldone == YES)
	{
	  /* Do the end-of-level effect: */
	  
	  levelendfx(playerx);
	  
	  
	  /* Increment the level: */
	  
	  level = level + 1;
	  if (level > 20)
	    level = 20;
	  
	  
	  /* Do the level 15 effect: */
	  
	  if (level == 15)
	    level15fx();
	  
	  
	  /* Reset the level! */
	  
	  initlevel();
	  clicks = 0;
	  updatelevel();
	}
    }
  while (gameover == NO);
  
  
  /* End of game effect: */
  
  playsound("gameover");
  
  happy = NO;
  drawyou(playerx);
  
  
  /* Dump bricks down at you!: */
  
  for (y = 0; y < HEIGHT; y++)
    {
      for (x = 0; x < WIDTH; x++)
	{
	  drawblockgraphic(x * 32, y * 32, BRICK_GFX);
	}
      XFlush(display);
      
      if ((y % 2) == 0)
	sleepfor(FRAME_RATE);
    }
  
  
  /* Wait for a keypress: */
  
  do
    {
      getkey(0);
    }
  while (strlen(key) == 0);
}


/* Quit the application: */

void quitapp()
{
  XCloseDisplay(display);
}


/* MAIN FUNCTION */

int main(int argc, char * argv[])
{
  int quit, ready;
  char server[1024];
  

  use_sound = 0;
  
  
  if (argc == 1)
    {
      /* Default server: */
      
      strcpy(server, ":0.0");
    }
  else if (argc == 2)
    {
      /* -version or -help or a server specification?? */
      
      if (strcmp(argv[1], "-version") == 0)
	{
	  printf("Gem Drop X\n");
	  printf("version beta 3\n");
	  printf("November 21, 1997\n");
	  printf("by Bill Kendrick\n");
	  printf("New Breed Software, (c) 1997\n\n");
	  exit(0);
	}
      else if (strcmp(argv[1], "-help") == 0)
	{
	  printf("GEM DROP X HELP\n");
	  printf("How To Play:\n");
	  printf("  Use the man to grab and throw gems.\n");
	  printf("  Match 3+ in a column when you throw to get a line.\n");
	  printf("  Get enough lines to beat the level.\n");
	  printf("  Gems come down at you from the top.\n");
	  printf("  If the screen fills, the game ends.\n\n");
	  
	  printf("Special Pieces:  (You can't grab them)\n");
	  printf("  Activate them by including them in a match.\n");
	  printf("  Bombs explode.\n");
	  printf("  Clocks stop the gems from coming for a while.\n");
	  printf("  Wildcards (question-marks) make for more matches.\n\n");
	  
	  printf("Controls:\n");
	  printf("  Left/Right - Move left/right\n");
	  printf("  K/L        - Move left/right\n");
	  printf("  Up/Down    - Throw/Grab\n");
	  printf("  A/Z        - Throw/Grab\n");
	  printf("  Return     - Get more gems immediately\n");
	  printf("  Space / P  - Pause\n");
	  printf("  S          - Toggle sound\n");
	  printf("  Q/Escape   - Abort game\n");
	  
	  exit(0);
	}
      else if (strcmp(argv[1], "-sound") == 0)
	{
	  /* Wants sound, no sever specification: */
	  
	  use_sound = 1;
	  strcpy(server, ":0.0");
	}
      else
	{
	  /* Must be a server specification: */
	  
	  strcpy(server, argv[1]);
	}
    }
  else if (argc == 3)
    {
       if (strcmp(argv[1], "-sound") == 0)
         use_sound = 1;
       else
         {
           /* Oops!  Usage! */
	      
	   fprintf(stderr, "Usage: gemdropx [display | -version | -help]\n\n");
	   exit(0);
         }
       strcpy(server, argv[2]);
    }
  else
    {
      /* Oops!  Usage! */
      
      fprintf(stderr, "Usage: gemdropx {-sound} {display} | -version | -help");
      fprintf(stderr, "\n\n");
      exit(0);
    }
  
  
  /* Setup: */
  
  printf("Conneting to server \"%s\"\n", server);
  setup(server);
  
  
  /* Wait for window to appear: */
  
  ready = 0;
  
  do
    {
      XFlush(display);
      
      if (XPending(display))
	{
	  XNextEvent(display, &event);
	  
	  if (event.type == Expose)
	    ready = 1;
	}
    }
  while (ready == 0);
  
  
  /* Main loop: */
  
  do
    {
      quit = title();
      
      if (quit == NO)
	{
	  play();
	}
    }
  while (quit == NO);

  quitapp();
}
