/* Low level stuff needed to manage views
   Copyright (C) 1995 Viacom New Media.

	This file is part of e93.

	e93 is free software; you can redistribute it and/or modify
	it under the terms of the e93 LICENSE AGREEMENT.

	e93 is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	e93 LICENSE AGREEMENT for more details.

	You should have received a copy of the e93 LICENSE AGREEMENT
	along with e93; see the file "LICENSE.TXT".
*/

#include "includes.h"

#define	STAGINGBUFFERSIZE		4096									/* number of bytes used in the staging buffer, if we overflow, then we throw out characters! */
static char hexConv[]="0123456789ABCDEF";								/* used for speedy hex conversions (for display of non-ascii codes) */

/* when text is being built, or widths are being calculated, these globals
 * are used, they allow for much more efficient creation of text because things
 * are not being passed all over for each character
 */
static UINT8
	gceStagingBuffer[STAGINGBUFFERSIZE];									/* this is where get character equivalent builds its output */
static UINT32
	gceBufferIndex;															/* offset into the staging buffer */
static UINT32
	gceColumnNumber;														/* running column number */
static UINT16
	*gceWidthTable;															/* width table to use while calculating positions */
static UINT32
	gceTabSize;																/* number of spaces to use per tab */
static INT32
	gceRunningWidth;														/* running width */

/* pseudo globals (used within the scope of only a few routines) to allow
 * quick calculation of the part of the selection region that intersects the
 * view
 */

static UINT32
	csrWorkingLine;
static UINT32
	csrWorkingLineStartPosition,		/* global text position (in csrWorkingLine) where we currently are */
	csrWorkingLineEndPosition;			/* global text position of the end+1 of the line we are working on */
static CHUNKHEADER
	*csrWorkingChunk;					/* the chunk where csrWorkingLineStartPosition lives */
static UINT32
	csrWorkingChunkOffset;				/* the offset within csrWorkingChunk to csrWorkingLineStartPosition */

BOOLEAN PointInView(EDITORVIEW *theView,INT32 x,INT32 y)
/* return TRUE if parent window relative x/y are in theView
 * FALSE if not
 */
{
	return(PointInRECT(x,y,&(((GUIVIEWINFO *)theView->viewInfo)->bounds)));
}

void LocalClickToViewClick(EDITORVIEW *theView,INT32 clickX,INT32 clickY,INT32 *xPosition,INT32 *lineNumber)
/* given a point local to the window that contains theView,
 * convert it into a view position
 */
{
	GUIVIEWINFO
		*theViewInfo;
	INT32
		viewY;

	theViewInfo=(GUIVIEWINFO *)theView->viewInfo;				/* point at the information record for this view */
	(*xPosition)=clickX-theViewInfo->textBounds.x;
	viewY=clickY-theViewInfo->textBounds.y;
	if(viewY<0)
		{
		viewY-=theViewInfo->lineHeight;							/* account for cases about 0 */
		}
	(*lineNumber)=viewY/((INT32)(theViewInfo->lineHeight));
}

void HandleViewKeyEvent(EDITORVIEW *theView,XEvent *theEvent)
/* handle an incoming key event for theView
 */
{
	VIEWEVENT
		theViewEvent;
	EDITORKEY
		theKeyData;

	theViewEvent.theView=theView;							/* point at the view */
	theViewEvent.eventType=VET_KEYDOWN;						/* let it know what type */
	if(KeyEventToEditorKey(theEvent,&theKeyData))			/* fill in the key record */
		{
		theViewEvent.eventData=&theKeyData;
		HandleViewEvent(&theViewEvent);						/* pass this off to high level handler */
		}
}

static void InvertViewCursor(EDITORVIEW *theView)
/* invert the cursor wherever it happens to be in theView
 * theView's window's current graphics context will be used to constrain the draw
 */
{
	GUIVIEWINFO
		*theViewInfo;
	CHUNKHEADER
		*theChunk;
	UINT32
		theOffset;
	UINT32
		theLine,
		theLineOffset;
	INT32
		xPosition,
		maxXPosition;
	EDITORRECT
		cursorRect;
	Window
		xWindow;
	GC
		graphicsContext;

	theViewInfo=(GUIVIEWINFO *)theView->viewInfo;			/* point at the information record for this view */

	PositionToLinePosition(theView->editorUniverse->textUniverse,theView->editorUniverse->selectionUniverse->cursorPosition,&theLine,&theLineOffset,&theChunk,&theOffset);
	if((theLine>=theViewInfo->topLine)&&(theLine<theViewInfo->topLine+theViewInfo->numLines))	/* make sure cursor is on the view */
		{
		maxXPosition=theViewInfo->leftPixel+theViewInfo->textBounds.w;
		GetEditorViewTextToLimitedGraphicPosition(theView,theChunk,theOffset,theLineOffset,maxXPosition,&xPosition);
		if((xPosition>=theViewInfo->leftPixel)&&(xPosition<maxXPosition))	/* see if on view horizontally */
			{
			cursorRect.x=theViewInfo->textBounds.x+(xPosition-theViewInfo->leftPixel);
			cursorRect.w=1;
			cursorRect.y=theViewInfo->textBounds.y+theViewInfo->lineHeight*(theLine-theViewInfo->topLine);
			cursorRect.h=theViewInfo->lineHeight;

			xWindow=((WINDOWLISTELEMENT *)theView->parentWindow->userData)->xWindow;		/* get x window for this view */
			graphicsContext=((WINDOWLISTELEMENT *)theView->parentWindow->userData)->graphicsContext;	/* get graphics context for this window */

			XSetFunction(xDisplay,graphicsContext,GXxor);			/* prepare to fill in xor format */
			XSetFillStyle(xDisplay,graphicsContext,FillSolid);		/* use solid fill */

			XSetForeground(xDisplay,graphicsContext,theViewInfo->foregroundColor^theViewInfo->backgroundColor);	/* this is the value to xor with */

			XFillRectangle(xDisplay,xWindow,graphicsContext,cursorRect.x,cursorRect.y,cursorRect.w,cursorRect.h);

			XSetFunction(xDisplay,graphicsContext,GXcopy);			/* put function back */
			}
		}
}

static BOOLEAN CursorShowing(EDITORVIEW *theView)
/* returns TRUE if the cursor is showing
 */
{
	GUIVIEWINFO
		*theViewInfo;

	theViewInfo=(GUIVIEWINFO *)theView->viewInfo;			/* point at the information record for this view */
	if((!theView->editorUniverse->selectionUniverse->selectionChunks.totalElements)&&theViewInfo->cursorOn&&theViewInfo->blinkState&&theViewInfo->viewActive)
		{
		return(TRUE);
		}
	return(FALSE);
}

static void DrawViewCursor(EDITORVIEW *theView)
/* look at the cursor state for this view, and draw it as needed
 * it will be drawn into the current port, and current clip region
 */
{
	if(CursorShowing(theView))
		{
		InvertViewCursor(theView);								/* if it is showing, then draw it */
		}
}

static BOOLEAN GetCharacterEquiv(UINT8 theChar)
/* put the correct characters into the gceStagingBuffer at gceBufferIndex
 * update gceColumnNumber, gceBufferIndex, and gceRunningWidth
 * if an end of line character is seen, return TRUE, else return FALSE
 */
{
	UINT32
		i,
		numTabSpaces;

	if(theChar>=' '&&gceWidthTable[theChar])								/* make sure the font can display this, if not, display it as hex */
		{
		if(gceBufferIndex<STAGINGBUFFERSIZE)
			{
			gceStagingBuffer[gceBufferIndex++]=theChar;
			gceRunningWidth+=gceWidthTable[theChar];
			gceColumnNumber++;
			}
		}
	else
		{
		switch(theChar)
			{
			case '\n':
				return(TRUE);
				break;
			case '\t':
				if(gceTabSize)
					{
					numTabSpaces=gceTabSize-(gceColumnNumber%gceTabSize);
					if((gceBufferIndex+numTabSpaces)<STAGINGBUFFERSIZE)
						{
						for(i=0;i<numTabSpaces;i++)
							{
							gceStagingBuffer[gceBufferIndex++]=' ';			/* space over to tab stop */
							}
						gceRunningWidth+=gceWidthTable[' ']*numTabSpaces;	/* add in space to running width */
						gceColumnNumber+=numTabSpaces;						/* push to correct column */
						}
					}
				else
					{
					if((gceBufferIndex+4)<STAGINGBUFFERSIZE)
						{
						gceRunningWidth+=gceWidthTable[gceStagingBuffer[gceBufferIndex++]='['];
						gceRunningWidth+=gceWidthTable[gceStagingBuffer[gceBufferIndex++]=hexConv[theChar>>4]];
						gceRunningWidth+=gceWidthTable[gceStagingBuffer[gceBufferIndex++]=hexConv[theChar&0x0F]];
						gceRunningWidth+=gceWidthTable[gceStagingBuffer[gceBufferIndex++]=']'];
						gceColumnNumber+=4;
						}
					}
				break;
			default:
				/* this is a little scary looking, but it is fast (and needs to be) */
				if((gceBufferIndex+4)<STAGINGBUFFERSIZE)
					{
					gceRunningWidth+=gceWidthTable[gceStagingBuffer[gceBufferIndex++]='['];
					gceRunningWidth+=gceWidthTable[gceStagingBuffer[gceBufferIndex++]=hexConv[theChar>>4]];
					gceRunningWidth+=gceWidthTable[gceStagingBuffer[gceBufferIndex++]=hexConv[theChar&0x0F]];
					gceRunningWidth+=gceWidthTable[gceStagingBuffer[gceBufferIndex++]=']'];
					gceColumnNumber+=4;
					}
				break;
			}
		}
	return(FALSE);
}

static void CreateDrawText(TEXTUNIVERSE *theUniverse,CHUNKHEADER *theChunk,UINT32 theOffset,INT32 *scrollX,INT32 maxWidth,UINT32 *numBytes)
/* create a line of bytes that will be sent to DrawText to draw the given line
 * return the number of bytes created, and the offset needed to make them draw in the
 * correct spot
 */
{
	BOOLEAN
		atEOL,
		loopDone;
	INT32
		skippedWidth;
	UINT32
		lastColumnNumber;

	loopDone=atEOL=FALSE;
	gceColumnNumber=lastColumnNumber=0;

	skippedWidth=VIEWLEFTMARGIN;

	while(!loopDone&&!atEOL&&theChunk)				/* find the position where the text begins */
		{
		gceBufferIndex=0;
		gceRunningWidth=skippedWidth;
		if(!(atEOL=GetCharacterEquiv(theChunk->data[theOffset])))
			{
			if(gceRunningWidth>(*scrollX))
				{
				loopDone=TRUE;
				}
			else
				{
				lastColumnNumber=gceColumnNumber;
				skippedWidth=gceRunningWidth;
				theOffset++;
				if(theOffset>=theChunk->totalBytes)			/* see if time for a new chunk */
					{
					theChunk=theChunk->nextHeader;
					theOffset=0;
					}
				}
			}
		}

	gceBufferIndex=0;								/* begin at start of buffer */
	gceColumnNumber=lastColumnNumber;				/* point to the column we were in when we found the place */
	gceRunningWidth=skippedWidth;

	loopDone=FALSE;
	while(!loopDone&&theChunk)						/* fill in all of the text needed */
		{
		if(GetCharacterEquiv(theChunk->data[theOffset])||gceRunningWidth>(*scrollX)+maxWidth)
			{
			loopDone=TRUE;
			}
		else
			{
			theOffset++;
			if(theOffset>=theChunk->totalBytes)		/* push through end of chunk */
				{
				theChunk=theChunk->nextHeader;
				theOffset=0;
				}
			}
		}
	(*numBytes)=gceBufferIndex;						/* tell outside how many bytes we made */
	(*scrollX)-=skippedWidth;						/* adjust this to point to the start of the first character */
}

static void AddSelectionToRegion(UINT32 startLine,UINT32 endLine,INT32 startX,INT32 endX,INT32 leftX,INT32 rightX,UINT32 lineHeight,Region theRegion)
/* given the start and end positions (all view relative), add to theRegion
 * theRegion is view relative
 */
{
	XRectangle
		selectionRect;

	if(endLine>startLine)
		{
		selectionRect.y=startLine*lineHeight;
		selectionRect.height=lineHeight;
		selectionRect.x=startX;
		if(selectionRect.width=rightX-startX)
			{
			XUnionRectWithRegion(&selectionRect,theRegion,theRegion);
			}

		if(endLine>startLine+1)								/* see if any middle area */
			{
			selectionRect.y=(startLine+1)*lineHeight;
			selectionRect.height=endLine*lineHeight-selectionRect.y;
			selectionRect.x=leftX;
			if(selectionRect.width=rightX-leftX)
				{
				XUnionRectWithRegion(&selectionRect,theRegion,theRegion);
				}
			}
		selectionRect.y=endLine*lineHeight;
		selectionRect.height=lineHeight;
		selectionRect.x=leftX;
		if(selectionRect.width=endX-leftX)
			{
			XUnionRectWithRegion(&selectionRect,theRegion,theRegion);
			}
		}
	else
		{
		selectionRect.y=startLine*lineHeight;
		selectionRect.height=lineHeight;
		selectionRect.x=startX;
		if(selectionRect.width=endX-startX)
			{
			XUnionRectWithRegion(&selectionRect,theRegion,theRegion);
			}
		}
}

static BOOLEAN SelectionPointToLineOffset(EDITORVIEW *theView,UINT32 thePosition,UINT32 topLine,UINT32 numLines,INT32 leftPixel,UINT32 numPixels,UINT32 *theLine,INT32 *xOffset)
/* convert thePosition into a line and xOffset, update the csr globals as needed
 * return TRUE if the position was not past the end of the view, FALSE otherwise
 * if thePosition is before the current position, then the current position's line and xOffset will
 * be returned
 * if thePosition's xOffset would be past the right side of the view, then the xOffset of
 * the right will be returned
 * topLine and leftPixel are text relative
 */
{
	BOOLEAN
		pastEnd;
	CHUNKHEADER
		*tempChunk;
	UINT32
		tempChunkOffset;
	INT32
		maxX;

	pastEnd=FALSE;

	if(thePosition<csrWorkingLineStartPosition)				/* if the point is before the start, fudge it to the start */
		{
		thePosition=csrWorkingLineStartPosition;
		}
	else
		{
		if(thePosition>=csrWorkingLineEndPosition)			/* this position is on a different line than the current one, so change current line settings */
			{
			PositionToLinePosition(theView->editorUniverse->textUniverse,thePosition,&csrWorkingLine,&csrWorkingLineStartPosition,&csrWorkingChunk,&csrWorkingChunkOffset);
			if(csrWorkingLine<topLine+numLines)
				{
				csrWorkingLineStartPosition=thePosition-csrWorkingLineStartPosition;	/* get global position of start of this line */
				ChunkPositionToNextLine(theView->editorUniverse->textUniverse,csrWorkingChunk,csrWorkingChunkOffset,&tempChunk,&tempChunkOffset,&csrWorkingLineEndPosition);
				csrWorkingLineEndPosition+=csrWorkingLineStartPosition;					/* add distance moved to old position to get new position */
				gceColumnNumber=0;							/* reset current column, and width */
				gceRunningWidth=VIEWLEFTMARGIN;
				}
			else
				{
				(*theLine)=topLine+numLines;
				(*xOffset)=VIEWLEFTMARGIN;
				pastEnd=TRUE;
				}
			}
		}
	if(!pastEnd)											/* now, run across from the last position to locate the xOffset */
		{
		(*theLine)=csrWorkingLine;
		maxX=leftPixel+numPixels;
		while((gceRunningWidth<maxX)&&(csrWorkingLineStartPosition<thePosition))
			{
			if(csrWorkingChunk)
				{
				gceBufferIndex=0;		/* dont collect the bytes, just build them */
				GetCharacterEquiv(csrWorkingChunk->data[csrWorkingChunkOffset++]);
				csrWorkingLineStartPosition++;
				if(csrWorkingChunkOffset>=csrWorkingChunk->totalBytes)
					{
					csrWorkingChunk=csrWorkingChunk->nextHeader;
					csrWorkingChunkOffset=0;
					}
				}
			else
				{
				ReportMessage("Internal error, past end of chunk list\n");
				}
			}
		if(gceRunningWidth<maxX)
			{
			(*xOffset)=gceRunningWidth;
			}
		else
			{
			(*xOffset)=maxX;
			}
		}
	return(!pastEnd);
}

static BOOLEAN HandleSelectionEndPoints(EDITORVIEW *theView,UINT32 topLine,UINT32 numLines,INT32 leftPixel,UINT32 numPixels,UINT32 selectionStart,UINT32 selectionEnd,Region theRegion)
/* get the end points of the selection, add to theRegion
 * update variables used to determine what to do on next selection
 * return TRUE if we can keep going, FALSE if this selection passes
 * the end of the view
 * topLine and leftPixel are text relative
 * theRegion is view relative
 */
{
	UINT32
		selectionStartLine,
		selectionEndLine;
	INT32
		selectionStartXPosition,
		selectionEndXPosition;
	BOOLEAN
		stillInView;
	INT32
		leftEdge,
		rightEdge;
	GUIVIEWINFO
		*theViewInfo;

	theViewInfo=(GUIVIEWINFO *)theView->viewInfo;			/* point at the information record for this view */

	if(SelectionPointToLineOffset(theView,selectionStart,topLine,numLines,leftPixel,numPixels,&selectionStartLine,&selectionStartXPosition))
		{
		stillInView=SelectionPointToLineOffset(theView,selectionEnd,topLine,numLines,leftPixel,numPixels,&selectionEndLine,&selectionEndXPosition);
		selectionStartLine-=theViewInfo->topLine;			/* make these relative to the view */
		selectionEndLine-=theViewInfo->topLine;
		leftEdge=leftPixel-theViewInfo->leftPixel;			/* make left hand side view relative */
		rightEdge=leftEdge+numPixels;
		if(leftPixel<VIEWLEFTMARGIN)
			{
			leftEdge+=VIEWLEFTMARGIN-leftPixel;
			}
		if(leftEdge<0)
			{
			leftEdge=0;
			}
		if(rightEdge>leftEdge)
			{
			if(selectionStartLine!=selectionEndLine)
				{
				if((selectionEndXPosition-=theViewInfo->leftPixel)<0)				/* make X offsets relative to the view */
					{
					selectionEndXPosition=0;							/* pin to start of view */
					}
				if((selectionStartXPosition-=theViewInfo->leftPixel)<0)
					{
					selectionStartXPosition=0;							/* pin start to the view */
					}
				AddSelectionToRegion(selectionStartLine,selectionEndLine,selectionStartXPosition,selectionEndXPosition,leftEdge,rightEdge,theViewInfo->lineHeight,theRegion);
				}
			else
				{
				if((selectionEndXPosition-=theViewInfo->leftPixel)>0)
					{
					if((selectionStartXPosition-=theViewInfo->leftPixel)<0)
						{
						selectionStartXPosition=0;						/* pin start to the view */
						}
					AddSelectionToRegion(selectionStartLine,selectionEndLine,selectionStartXPosition,selectionEndXPosition,leftEdge,rightEdge,theViewInfo->lineHeight,theRegion);
					}
				}
			}
		return(stillInView);
		}
	return(FALSE);
}

static void CreateSelectionRegion(EDITORVIEW *theView,UINT32 topLine,UINT32 numLines,INT32 leftPixel,UINT32 numPixels,Region theRegion)
/* create the region that describes what is/should be currently selected in the view
 * the region will be relative to the view
 * this is used when a selection is changing in the view, and we need to know where
 * so it can be updated
 * topLine, and leftPixel are text relative (global) coordinates
 */
{
	GUIVIEWINFO
		*theViewInfo;
	CHUNKHEADER
		*tempChunk;
	UINT32
		tempChunkOffset;
	ARRAYCHUNKHEADER
		*selectionChunk;
	UINT32
		selectionOffset;
	UINT32
		currentPosition,
		nextPosition;
	BOOLEAN
		done;

	theViewInfo=(GUIVIEWINFO *)theView->viewInfo;			/* point at the information record for this view */
	if(theViewInfo->viewActive)								/* only do this if the view is active */
		{
		if(selectionChunk=theView->editorUniverse->selectionUniverse->selectionChunks.firstChunkHeader)
			{
			gceWidthTable=&(theViewInfo->charWidths[0]);		/* set these up for the view we are on */
			gceTabSize=theViewInfo->tabSize;

			csrWorkingLine=topLine;								/* start at the top line */
			LineToChunkPosition(theView->editorUniverse->textUniverse,csrWorkingLine,&csrWorkingChunk,&csrWorkingChunkOffset,&csrWorkingLineStartPosition);	/* get position where view begins */
			ChunkPositionToNextLine(theView->editorUniverse->textUniverse,csrWorkingChunk,csrWorkingChunkOffset,&tempChunk,&tempChunkOffset,&csrWorkingLineEndPosition);
			csrWorkingLineEndPosition+=csrWorkingLineStartPosition;			/* add distance moved to old position to get new position */
			gceColumnNumber=0;									/* clear current column, and current width */
			gceRunningWidth=VIEWLEFTMARGIN;
			done=FALSE;
			currentPosition=0;
			selectionOffset=0;
			while(selectionChunk&&!done)
				{
				currentPosition+=((SELECTIONELEMENT *)selectionChunk->data)[selectionOffset].startOffset;
				nextPosition=currentPosition+((SELECTIONELEMENT *)selectionChunk->data)[selectionOffset].endOffset;
				if((nextPosition)>csrWorkingLineStartPosition)	/* make sure this selection is not entirely before the current place we are looking */
					{
					done=!HandleSelectionEndPoints(theView,topLine,numLines,leftPixel,numPixels,currentPosition,currentPosition+((SELECTIONELEMENT *)selectionChunk->data)[selectionOffset].endOffset,theRegion);
					}
				currentPosition=nextPosition;
				selectionOffset++;
				if(selectionOffset>=selectionChunk->totalElements)
					{
					selectionChunk=selectionChunk->nextHeader;
					selectionOffset=0;
					}
				}
			XOffsetRegion(theRegion,theViewInfo->textBounds.x,theViewInfo->textBounds.y);	/* push over to view's location in the window */
			}
		}
}

static void DrawViewLines(EDITORVIEW *theView)
/* draw the lines into theView, obey the invalid region of the view
 */
{
	Window
		xWindow;
	GUIVIEWINFO
		*theViewInfo;
	INT32
		scrollX;
	UINT32
		numBytes;
	XRectangle
		lineRect,
		regionBox;
	Region
		invalidTextRegion,
		lineRegion,
		selectionRegion;
	UINT32
		topLineToDraw,
		numLinesToDraw;
	CHUNKHEADER
		*theChunk;
	UINT32
		theOffset,
		temp;
	UINT32
		i;
	GC
		graphicsContext,
		invertContext;

	theViewInfo=(GUIVIEWINFO *)theView->viewInfo;			/* point at the information record for this view */
	xWindow=((WINDOWLISTELEMENT *)theView->parentWindow->userData)->xWindow;					/* get x window for this window */
	graphicsContext=((WINDOWLISTELEMENT *)theView->parentWindow->userData)->graphicsContext;	/* get graphics context for this window */

	XSetFont(xDisplay,graphicsContext,theViewInfo->theFont->fid);	/* point to the current font for this view */

	XSetBackground(xDisplay,graphicsContext,theViewInfo->backgroundColor);
	XSetForeground(xDisplay,graphicsContext,theViewInfo->foregroundColor);
	gceWidthTable=&(theViewInfo->charWidths[0]);
	gceTabSize=theViewInfo->tabSize;

	invertContext=XCreateGC(xDisplay,xWindow,0,NULL);	/* create context in which to draw the inversion, (0 for mask)= use default */
	XSetFunction(xDisplay,invertContext,GXxor);			/* prepare to fill in xor format */
	XSetFillStyle(xDisplay,invertContext,FillSolid);	/* use solid fill */
	XSetForeground(xDisplay,invertContext,theViewInfo->foregroundColor^theViewInfo->backgroundColor);	/* this is the value to xor with */

	regionBox.x=lineRect.x=theViewInfo->textBounds.x;
	regionBox.width=lineRect.width=theViewInfo->textBounds.w;
	lineRect.height=theViewInfo->lineHeight;
	regionBox.y=theViewInfo->textBounds.y;
	regionBox.height=theViewInfo->textBounds.h;

	invalidTextRegion=XCreateRegion();
	XUnionRectWithRegion(&regionBox,invalidTextRegion,invalidTextRegion);		/* make rectangular region of entire text space */
	XIntersectRegion(invalidTextRegion,((WINDOWLISTELEMENT *)theView->parentWindow->userData)->invalidRegion,invalidTextRegion);		/* intersect with invalid region of window */

	XClipBox(invalidTextRegion,&regionBox);				/* get bounds for actual invalid text area */

	topLineToDraw=(regionBox.y-theViewInfo->textBounds.y)/theViewInfo->lineHeight;	/* screen relative top line to draw */
	numLinesToDraw=(regionBox.y-theViewInfo->textBounds.y+regionBox.height)/theViewInfo->lineHeight-topLineToDraw+1;	/* get number of lines to draw */

	LineToChunkPosition(theView->editorUniverse->textUniverse,theViewInfo->topLine+topLineToDraw,&theChunk,&theOffset,&temp);	/* point to data for top line */

	for(i=topLineToDraw;i<topLineToDraw+numLinesToDraw;i++)
		{
		lineRect.y=theViewInfo->textBounds.y+i*theViewInfo->lineHeight;
		lineRegion=XCreateRegion();
		XUnionRectWithRegion(&lineRect,lineRegion,lineRegion);		/* make region for the line we are about to draw */
		XIntersectRegion(lineRegion,invalidTextRegion,lineRegion);	/* see if it intersects with the invalid region */
		if(!XEmptyRegion(lineRegion))
			{
			XClipBox(lineRegion,&regionBox);						/* get bounds for this part */
			scrollX=theViewInfo->leftPixel+(regionBox.x-theViewInfo->textBounds.x);
			if(theChunk)
				{
				CreateDrawText(theView->editorUniverse->textUniverse,theChunk,theOffset,&scrollX,regionBox.width,&numBytes);
				if(numBytes)
					{
					XDrawImageString(xDisplay,xWindow,graphicsContext,regionBox.x-scrollX,lineRect.y+theViewInfo->theFont->ascent,(char *)gceStagingBuffer,numBytes);
					}
				}
			}
		ChunkPositionToNextLine(theView->editorUniverse->textUniverse,theChunk,theOffset,&theChunk,&theOffset,&temp);
		XDestroyRegion(lineRegion);
		}

	selectionRegion=XCreateRegion();
	CreateSelectionRegion(theView,theViewInfo->topLine+topLineToDraw,numLinesToDraw,theViewInfo->leftPixel,theViewInfo->textBounds.w,selectionRegion);
	XIntersectRegion(selectionRegion,invalidTextRegion,selectionRegion);		/* see if it intersects with the invalid region */
	if(!XEmptyRegion(selectionRegion))
		{
		XSetRegion(xDisplay,invertContext,selectionRegion);		/* clip to area we wish to invert */
		XClipBox(selectionRegion,&regionBox);					/* get bounds for this part */
		XFillRectangle(xDisplay,xWindow,invertContext,regionBox.x,regionBox.y,regionBox.width,regionBox.height);
		}
	XDestroyRegion(selectionRegion);
	XDestroyRegion(invalidTextRegion);
	XFreeGC(xDisplay,invertContext);							/* get rid of the graphics context */
}

void DrawView(EDITORVIEW *theView)
/* draw the given view in its parent window
 * also handle highlighting of text
 * NOTE: the invalidRegion of the view's parent
 * window should be set to the region that needs update
 */
{
	XRectangle
		regionBox;
	GUIVIEWINFO
		*theViewInfo;
	WINDOWLISTELEMENT
		*theWindowElement;
	GC
		graphicsContext;
	Region
		viewInvalidRegion;

	theViewInfo=(GUIVIEWINFO *)theView->viewInfo;						/* point at the information record for this view */

	theWindowElement=(WINDOWLISTELEMENT *)theView->parentWindow->userData;	/* get the x window information associated with this editor window */
	graphicsContext=theWindowElement->graphicsContext;					/* get graphics context for this window */

	viewInvalidRegion=XCreateRegion();
	regionBox.x=theViewInfo->bounds.x;
	regionBox.y=theViewInfo->bounds.y;
	regionBox.width=theViewInfo->bounds.w;
	regionBox.height=theViewInfo->bounds.h;
	XUnionRectWithRegion(&regionBox,viewInvalidRegion,viewInvalidRegion);	/* create a region which is the rectangle of the entire view */
	XIntersectRegion(viewInvalidRegion,theWindowElement->invalidRegion,viewInvalidRegion);	/* intersect with what is invalid in this window */

	if(!XEmptyRegion(viewInvalidRegion))								/* see if it has an invalid area */
		{
		XSetRegion(xDisplay,graphicsContext,viewInvalidRegion);			/* set clipping to what's invalid in the view */
		XSetFillStyle(xDisplay,graphicsContext,FillSolid);
		XSetForeground(xDisplay,graphicsContext,theViewInfo->backgroundColor);
		XClipBox(viewInvalidRegion,&regionBox);					/* get bounds of invalid region */
		XFillRectangle(xDisplay,theWindowElement->xWindow,graphicsContext,regionBox.x,regionBox.y,regionBox.width,regionBox.height);	/* fill with white */
		DrawViewLines(theView);
		DrawViewCursor(theView);
		XSetClipMask(xDisplay,graphicsContext,None);					/* get rid of clip mask */
		}
	XDestroyRegion(viewInvalidRegion);									/* get rid of invalid region for the view */
}

static void RecalculateViewFontInfo(EDITORVIEW *theView)
/* recalculate the cached font information for theView
 * from its current size and font
 */
{
	INT32
		viewHeight;
	GUIVIEWINFO
		*theViewInfo;

	theViewInfo=(GUIVIEWINFO *)theView->viewInfo;				/* point at the information record for this view */

	theViewInfo->textBounds=theViewInfo->bounds;				/* start with copy of the view bounds */
	theViewInfo->textBounds.y+=VIEWTOPMARGIN;					/* leave a small margin at the top */

	if(theViewInfo->bounds.h>VIEWTOPMARGIN+VIEWBOTTOMMARGIN)
		{
		viewHeight=theViewInfo->bounds.h-VIEWTOPMARGIN-VIEWBOTTOMMARGIN;	/* find out how much can be used for the text */
		theViewInfo->numLines=viewHeight/theViewInfo->lineHeight;	/* get integer number of lines in this view */
		theViewInfo->textBounds.h=theViewInfo->numLines*theViewInfo->lineHeight;	/* alter the textBounds to be the exact rectangle to hold the text */
		}
	else
		{
		theViewInfo->numLines=0;
		theViewInfo->textBounds.h=0;							/* no room to draw text */
		}
}

void SetViewBounds(EDITORVIEW *theView,EDITORRECT *newBounds)
/* set the bounds of theView to newBounds
 * invalidate area where the view used to be but now is not
 * and invalidate new areas of the view
 */
{
	GUIVIEWINFO
		*theViewInfo;
	EDITORRECT
		*viewBounds;
	unsigned int
		theWidth,
		theHeight;

	theViewInfo=(GUIVIEWINFO *)theView->viewInfo;				/* point at the information record for this view */

	viewBounds=&(theViewInfo->bounds);							/* point at the view bounds rect */

	if((newBounds->x!=viewBounds->x)||(newBounds->y!=viewBounds->y)||(newBounds->w!=viewBounds->w)||(newBounds->h!=viewBounds->h))
		{
		InvalidateWindowRect(theView->parentWindow,viewBounds);			/* invalidate old area */
		InvalidateWindowRect(theView->parentWindow,newBounds);			/* invalidate new area */

		(*viewBounds)=*newBounds;									/* copy new rect into place */
		RecalculateViewFontInfo(theView);							/* update additional information */

		if(!(theWidth=theViewInfo->textBounds.w))
			{
			theWidth=1;
			}
		if(!(theHeight=theViewInfo->textBounds.h))
			{
			theHeight=1;
			}
		XMoveResizeWindow(xDisplay,theViewInfo->cursorWindow,theViewInfo->textBounds.x,theViewInfo->textBounds.y,theWidth,theHeight);	/* adjust the cursor change window */
		}
}

static void UnSetViewFont(EDITORVIEW *theView)
/* free the font currently in use by theView
 */
{
	GUIVIEWINFO
		*theViewInfo;

	theViewInfo=(GUIVIEWINFO *)theView->viewInfo;				/* point at the information record for this view */
	if(theViewInfo->theFont)
		{
		XFreeFont(xDisplay,theViewInfo->theFont);
		}
}

static void SetViewFont(EDITORVIEW *theView,char *fontName)
/* set the font used to draw theView
 */
{
	GUIVIEWINFO
		*theViewInfo;
	int
		i,j;

	theViewInfo=(GUIVIEWINFO *)theView->viewInfo;				/* point at the information record for this view */

	if(theViewInfo->theFont=LoadFont(fontName))
		{
		theViewInfo->lineHeight=theViewInfo->theFont->ascent+theViewInfo->theFont->descent;		/* get total height of line from font */
		for(i=0;i<256;i++)
			{
			theViewInfo->charWidths[i]=0;							/* clear widths for non-existent characters */
			}
		if(theViewInfo->theFont->per_char)							/* see if there is information for individual characters */
			{
			j=0;
			for(i=theViewInfo->theFont->min_char_or_byte2;i<=theViewInfo->theFont->max_char_or_byte2;i++)
				{
				theViewInfo->charWidths[i]=theViewInfo->theFont->per_char[j++].width;	/* set widths for existing characters */
				}
			}
		else
			{
			for(i=theViewInfo->theFont->min_char_or_byte2;i<=theViewInfo->theFont->max_char_or_byte2;i++)
				{
				theViewInfo->charWidths[i]=theViewInfo->theFont->max_bounds.width;		/* set widths for existing characters */
				}
			}
		RecalculateViewFontInfo(theView);							/* update additional information */
		}
	else
		{
		ReportMessage("Failed to load view font!\n");
		}
}

static void UnSetViewColors(EDITORVIEW *theView)
/* free the colors currently in use by theView
 */
{
	GUIVIEWINFO
		*theViewInfo;

	theViewInfo=(GUIVIEWINFO *)theView->viewInfo;				/* point at the information record for this view */
	FreeColor(theViewInfo->backgroundColor);
	FreeColor(theViewInfo->foregroundColor);
}

static void SetViewColors(EDITORVIEW *theView,EDITORCOLOR foreground,EDITORCOLOR background)
/* set the colors used to draw theView
 */
{
	GUIVIEWINFO
		*theViewInfo;

	theViewInfo=(GUIVIEWINFO *)theView->viewInfo;				/* point at the information record for this view */

	theViewInfo->backgroundColor=AllocColor(background);
	theViewInfo->foregroundColor=AllocColor(foreground);
}

static void SetViewTabSize(EDITORVIEW *theView,UINT32 theSize)
/* set the tabSize for theView
 */
{
	if(theSize>255)
		{
		theSize=255;					/* limit this, or we could crash when drawing */
		}
	((GUIVIEWINFO *)(theView->viewInfo))->tabSize=theSize;
}

EDITORVIEW *CreateEditorView(EDITORWINDOW *theWindow,EDITORVIEWDESCRIPTOR *theDescriptor)
/* create a view of theUniverse in theWindow using theDescriptor
 * if there is a problem, SetError, and return NULL
 */
{
	EDITORVIEW
		*theView;
	GUIVIEWINFO
		*viewInfo;
	UINT16
		theWidth,
		theHeight;

	if(theView=(EDITORVIEW *)MNewPtr(sizeof(EDITORVIEW)))
		{
		if(viewInfo=(GUIVIEWINFO *)MNewPtr(sizeof(GUIVIEWINFO)))
			{
			/* create a window which has the sole purpose of changing the cursor to a caret when it enters the view */
			theView->viewInfo=viewInfo;

			viewInfo->topLine=theDescriptor->topLine;
			viewInfo->leftPixel=theDescriptor->leftPixel;		/* initial pixel to display */
			SetViewTabSize(theView,theDescriptor->tabSize);		/* set, and limit the tab size */
			viewInfo->bounds=theDescriptor->theRect;			/* copy bounds rectangle */

			SetViewColors(theView,theDescriptor->foregroundColor,theDescriptor->backgroundColor);

			viewInfo->viewActive=theDescriptor->active;
			viewInfo->cursorOn=TRUE;							/* the cursor is turned on */
			viewInfo->blinkState=FALSE;							/* next time it blinks, it will blink to on */
			viewInfo->nextBlinkTime=0;							/* next time it checks, it will blink */
			LinkViewToEditorUniverse(theView,theDescriptor->theUniverse);
			theView->viewTextChangedVector=theDescriptor->viewTextChangedVector;
			theView->viewSelectionChangedVector=theDescriptor->viewSelectionChangedVector;
			theView->viewPositionChangedVector=theDescriptor->viewPositionChangedVector;
			theView->parentWindow=theWindow;
			SetViewFont(theView,theDescriptor->fontName);	/* set up for the chosen font */

			if(!(theWidth=viewInfo->textBounds.w))
				{
				theWidth=1;
				}
			if(!(theHeight=viewInfo->textBounds.h))
				{
				theHeight=1;
				}
			viewInfo->cursorWindow=XCreateWindow(xDisplay,((WINDOWLISTELEMENT *)theWindow->userData)->xWindow,viewInfo->textBounds.x,viewInfo->textBounds.y,theWidth,theHeight,0,0,InputOnly,DefaultVisual(xDisplay,xScreenNum),0,NULL);
			XDefineCursor(xDisplay,viewInfo->cursorWindow,caretCursor);			/* this is the cursor to use in the view */
			XMapRaised(xDisplay,viewInfo->cursorWindow);						/* map window to the top of the pile */

			return(theView);
			}
		MDisposePtr(theView);
		}
	return(NULL);
}

void DisposeEditorView(EDITORVIEW *theView)
/* unlink theView from everywhere, and dispose of it
 */
{
	UnSetViewColors(theView);
	UnSetViewFont(theView);
	XDestroyWindow(xDisplay,((GUIVIEWINFO *)theView->viewInfo)->cursorWindow);				/* tell x to make the cursor window go away */
	UnlinkViewFromEditorUniverse(theView);
	MDisposePtr(theView->viewInfo);
	MDisposePtr(theView);
}

void InvalidateViewPortion(EDITORVIEW *theView,UINT32 startLine,UINT32 endLine,UINT32 startPixel,UINT32 endPixel)
/* invalidate the area of the view between startLine, endLine, and startPixel, endPixel
 * by numLines, and numPixels
 * NOTE: startLine, endLine, startPixel, and endPixel are all relative to the displayed view
 * so that startLine of 0 always means the top line, and startPixel of 0 always
 * means the left-most pixel
 * NOTE: startLine must be <= endLine, startPixel must be <= endPixel
 * the invalidated area is exclusive of endLine, and endPixel
 */
{
	UINT32
		totalPixels;
	GUIVIEWINFO
		*theViewInfo;
	EDITORRECT
		invalidRect;

	theViewInfo=(GUIVIEWINFO *)theView->viewInfo;						/* point at the information record for this view */

	totalPixels=theViewInfo->textBounds.w;
	if(theViewInfo->numLines&&totalPixels)								/* make sure we can actually invalidate */
		{
		if(startLine>theViewInfo->numLines)
			{
			startLine=theViewInfo->numLines;
			}
		if(endLine>theViewInfo->numLines)
			{
			endLine=theViewInfo->numLines;
			}

		if(startPixel>totalPixels)
			{
			startPixel=totalPixels;
			}
		if(endPixel>totalPixels)
			{
			endPixel=totalPixels;
			}
		invalidRect.x=theViewInfo->textBounds.x+startPixel;
		invalidRect.y=theViewInfo->textBounds.y+startLine*theViewInfo->lineHeight;
		invalidRect.w=endPixel-startPixel;
		invalidRect.h=(endLine-startLine)*theViewInfo->lineHeight;
		InvalidateWindowRect(theView->parentWindow,&invalidRect);
		}
}

static void InvalidateView(EDITORVIEW *theView)
/* invalidate the entire area of the view
 */
{
	GUIVIEWINFO
		*theViewInfo;

	theViewInfo=(GUIVIEWINFO *)theView->viewInfo;						/* point at the information record for this view */
	InvalidateWindowRect(theView->parentWindow,&theViewInfo->bounds);
}

void ScrollViewPortion(EDITORVIEW *theView,UINT32 startLine,UINT32 endLine,INT32 numLines,UINT32 startPixel,UINT32 endPixel,INT32 numPixels)
/* scroll the area of the view between startLine, endLine, and startPixel, endPixel
 * by numLines, and numPixels
 * if numLines > 0, the data is scrolled down, else it is scrolled up
 * if numPixels > 0, the data is scrolled right, else it is scrolled left
 * any area left open by the scroll will be left erased, and invalidated
 * NOTE: startLine, endLine, startPixel, and endPixel are all relative to the displayed view
 * so that startLine of 0 always means the top line, and startPixel of 0 always
 * means the left-most pixel
 * NOTE: startLine must be <= endLine, startPixel must be <= endPixel
 * the scrolled area is exclusive of endLine, and endPixel
 */
{
	Window
		xWindow;
	GC
		graphicsContext;
	UINT32
		totalPixels;
	INT32
		copySrcX,					/* source x and y for copy */
		copySrcY,
		copyDestX,
		copyDestY,
		invalidX,
		invalidY;
	UINT32
		copyWidth,
		copyHeight,
		invalidWidth,
		invalidHeight;
	EDITORRECT
		invalidRect;
	GUIVIEWINFO
		*theViewInfo;

	theViewInfo=(GUIVIEWINFO *)theView->viewInfo;						/* point at the information record for this view */
	xWindow=((WINDOWLISTELEMENT *)theView->parentWindow->userData)->xWindow;					/* get x window for this view */
	graphicsContext=((WINDOWLISTELEMENT *)theView->parentWindow->userData)->graphicsContext;	/* get graphicsContext for this view */

	totalPixels=theViewInfo->textBounds.w;
	if(theViewInfo->numLines&&totalPixels)								/* make sure we can actually scroll */
		{
		if(startLine>theViewInfo->numLines)
			{
			startLine=theViewInfo->numLines;
			}
		if(endLine>theViewInfo->numLines)
			{
			endLine=theViewInfo->numLines;
			}

		if(startPixel>totalPixels)
			{
			startPixel=totalPixels;
			}
		if(endPixel>totalPixels)
			{
			endPixel=totalPixels;
			}

		if(numLines>((INT32)(endLine-startLine)))						/* allow a scroll that erases everything */
			{
			numLines=endLine-startLine;
			}
		else
			{
			if(numLines<((INT32)startLine-(INT32)endLine))
				{
				numLines=(INT32)startLine-(INT32)endLine;
				}
			}

		if(numPixels>((INT32)(endPixel-startPixel)))
			{
			numPixels=endPixel-startPixel;
			}
		else
			{
			if(numPixels<(INT32)startPixel-(INT32)endPixel)
				{
				numPixels=(INT32)startPixel-(INT32)endPixel;
				}
			}
		if(numLines||numPixels)
			{
			if(numLines>=0)												/* see if scrolling down, or not scrolling */
				{
				invalidY=copySrcY=startLine;
				copyDestY=startLine+numLines;
				copyHeight=(endLine-startLine)-numLines;
				invalidHeight=numLines;
				}
			else														/* scrolling up */
				{
				copySrcY=startLine-numLines;
				copyDestY=startLine;
				copyHeight=endLine-copySrcY;
				invalidY=endLine+numLines;
				invalidHeight=-numLines;
				}
			if(numPixels>=0)											/* scrolling right, or not scrolling horizontally */
				{
				invalidX=copySrcX=startPixel;
				copyDestX=startPixel+numPixels;
				copyWidth=(endPixel-startPixel)-numPixels;
				invalidWidth=numPixels;
				}
			else														/* left */
				{
				copySrcX=startPixel-numPixels;
				copyDestX=startPixel;
				copyWidth=endPixel-copySrcX;
				invalidX=endPixel+numPixels;
				invalidWidth=-numPixels;
				}
			if(copyWidth&&copyHeight)
				{
				XSetGraphicsExposures(xDisplay,graphicsContext,True);	/* tell x to give events when part of scroll is revealed */
				XCopyArea(xDisplay,xWindow,xWindow,graphicsContext,theViewInfo->textBounds.x+copySrcX,theViewInfo->textBounds.y+copySrcY*theViewInfo->lineHeight,copyWidth,copyHeight*theViewInfo->lineHeight,theViewInfo->textBounds.x+copyDestX,theViewInfo->textBounds.y+copyDestY*theViewInfo->lineHeight);
				XSetGraphicsExposures(xDisplay,graphicsContext,False);
				}
			if(invalidWidth)
				{
				invalidRect.x=theViewInfo->textBounds.x+invalidX;
				invalidRect.y=theViewInfo->textBounds.y+startLine*theViewInfo->lineHeight;
				invalidRect.w=invalidWidth;
				invalidRect.h=(endLine-startLine)*theViewInfo->lineHeight;
				InvalidateWindowRect(theView->parentWindow,&invalidRect);
				}
			if(invalidHeight)
				{
				invalidRect.x=theViewInfo->textBounds.x+startPixel;
				invalidRect.y=theViewInfo->textBounds.y+invalidY*theViewInfo->lineHeight;
				invalidRect.w=endPixel-startPixel;
				invalidRect.h=invalidHeight*theViewInfo->lineHeight;
				InvalidateWindowRect(theView->parentWindow,&invalidRect);
				}
			}
		}
}

void SetViewTopLeft(EDITORVIEW *theView,UINT32 newTopLine,INT32 newLeftPixel)
/* set the upper left corner of theView
 * NOTE: newTop, and newLeft will be forced into bounds by this routine
 */
{
	UINT32
		oldTopLine;
	INT32
		oldLeftPixel;
	UINT32
		numLines;
	GUIVIEWINFO
		*theViewInfo;

	theViewInfo=(GUIVIEWINFO *)theView->viewInfo;						/* point at the information record for this view */
	numLines=theView->editorUniverse->textUniverse->totalLines;
	oldTopLine=theViewInfo->topLine;
	oldLeftPixel=theViewInfo->leftPixel;

	if(newTopLine>numLines)
		{
		newTopLine=numLines;
		}
	if(newLeftPixel>EDITMAXWIDTH)
		{
		newLeftPixel=EDITMAXWIDTH;
		}
	if(newLeftPixel<-EDITMAXWIDTH)
		{
		newLeftPixel=-EDITMAXWIDTH;
		}
	ScrollViewPortion(theView,0,theViewInfo->numLines,oldTopLine-newTopLine,0,theViewInfo->textBounds.w,oldLeftPixel-newLeftPixel);
	theViewInfo->topLine=newTopLine;
	theViewInfo->leftPixel=newLeftPixel;
	if(theView->viewPositionChangedVector)								/* see if someone needs to be called because of my change */
		{
		theView->viewPositionChangedVector(theView);					/* call him with my pointer */
		}
	UpdateEditorWindows();												/* redraw anything that is needed */
}

void GetEditorViewGraphicToTextPosition(EDITORVIEW *theView,CHUNKHEADER *theChunk,UINT32 theOffset,INT32 xPosition,UINT32 *betweenOffset,UINT32 *charOffset)
/* given theChunk and theOffset point to the start of a line within theView, and xPosition
 * as a "pixel" position along the line, determine two offsets:
 * betweenOffset is the offset of the character boundary closest to the xPosition,
 * charOffset is the offset of the character closest to the xPosition
 * if theChunk is passed as NULL, it is assumed to indicate a past EOF condition
 * if this happens, both offsets will be returned as 0
 */
{
	GUIVIEWINFO
		*theViewInfo;
	UINT32
		charNumber;
	INT32
		lastWidth;
	BOOLEAN
		done;

	theViewInfo=(GUIVIEWINFO *)theView->viewInfo;			/* point at the information record for this view */
	gceWidthTable=&(theViewInfo->charWidths[0]);
	gceTabSize=theViewInfo->tabSize;

	charNumber=0;
	gceColumnNumber=0;
	lastWidth=gceRunningWidth=VIEWLEFTMARGIN;
	done=FALSE;
	while(theChunk&&!done)
		{
		gceBufferIndex=0;
		lastWidth=gceRunningWidth;							/* remember the width of the line BEFORE we hit a match */
		if(!GetCharacterEquiv(theChunk->data[theOffset]))
			{
			if(gceRunningWidth<=xPosition)
				{
				charNumber++;
				theOffset++;
				if(theOffset>=theChunk->totalBytes)
					{
					theChunk=theChunk->nextHeader;
					theOffset=0;
					}
				}
			else
				{
				if((xPosition>lastWidth)&&((gceRunningWidth-xPosition)<=(xPosition-lastWidth)))
					{
					(*betweenOffset)=charNumber+1;			/* x was closer to next character than to last */
					}
				else
					{
					(*betweenOffset)=charNumber;			/* x was closer to last width than to current width */
					}
				done=TRUE;
				}
			}
		else
			{
			(*betweenOffset)=charNumber;
			done=TRUE;										/* reached end of line, so done */
			}
		}
	(*charOffset)=charNumber;
	if(!done)
		{
		(*betweenOffset)=charNumber;						/* if we ran out of data, then place between at the end */
		}
}

void GetEditorViewTextToGraphicPosition(EDITORVIEW *theView,CHUNKHEADER *theChunk,UINT32 theOffset,UINT32 linePosition,INT32 *xPosition)
/* given theChunk and theOffset point to the start of a line within theView, and linePosition
 * as a character position along the line, determine the "pixel" offset to the start of the character
 * if theChunk is passed as NULL, it is assumed to indicate a past EOF condition
 * if this happens, xPosition will be returned as 0
 */
{
	GUIVIEWINFO
		*theViewInfo;
	UINT32
		charNumber;
	BOOLEAN
		atEOL;

	theViewInfo=(GUIVIEWINFO *)theView->viewInfo;			/* point at the information record for this view */
	gceWidthTable=&(theViewInfo->charWidths[0]);
	gceTabSize=theViewInfo->tabSize;

	charNumber=0;
	gceColumnNumber=0;
	gceRunningWidth=VIEWLEFTMARGIN;
	atEOL=FALSE;
	while(!atEOL&&(charNumber<linePosition)&&theChunk)
		{
		while(!atEOL&&(charNumber<linePosition)&&(theOffset<theChunk->totalBytes))
			{
			gceBufferIndex=0;
			atEOL=GetCharacterEquiv(theChunk->data[theOffset]);
			theOffset++;
			charNumber++;
			}
		theChunk=theChunk->nextHeader;
		theOffset=0;
		}
	(*xPosition)=gceRunningWidth;
}

void GetEditorViewTextToLimitedGraphicPosition(EDITORVIEW *theView,CHUNKHEADER *theChunk,UINT32 theOffset,UINT32 linePosition,INT32 maxXPosition,INT32 *xPosition)
/* given theChunk and theOffset point to the start of a line within theView, and linePosition
 * as a character position along the line, determine the "pixel" offset to the start of the character
 * For efficiency, if xPosition would be larger than maxXPosition, this is allowed
 * to return any xPosition >= maxXPosition
 * if theChunk is passed as NULL, it is assumed to indicate a past EOF condition
 * if this happens, xPosition will be returned as 0
 */
{
	GUIVIEWINFO
		*theViewInfo;
	UINT32
		charNumber;
	BOOLEAN
		atEOL;

	theViewInfo=(GUIVIEWINFO *)theView->viewInfo;			/* point at the information record for this view */
	gceWidthTable=&(theViewInfo->charWidths[0]);
	gceTabSize=theViewInfo->tabSize;

	charNumber=0;
	gceColumnNumber=0;
	gceRunningWidth=VIEWLEFTMARGIN;
	atEOL=FALSE;
	while(!atEOL&&(charNumber<linePosition)&&(gceRunningWidth<maxXPosition)&&theChunk)
		{
		while(!atEOL&&(charNumber<linePosition)&&(gceRunningWidth<maxXPosition)&&(theOffset<theChunk->totalBytes))
			{
			gceBufferIndex=0;
			atEOL=GetCharacterEquiv(theChunk->data[theOffset]);
			theOffset++;
			charNumber++;
			}
		theChunk=theChunk->nextHeader;
		theOffset=0;
		}
	(*xPosition)=gceRunningWidth;
}

void GetEditorViewTextInfo(EDITORVIEW *theView,UINT32 *topLine,UINT32 *numLines,INT32 *leftPixel,UINT32 *numPixels)
/* get information about the given view
 */
{
	GUIVIEWINFO
		*theViewInfo;

	theViewInfo=(GUIVIEWINFO *)theView->viewInfo;
	*topLine=theViewInfo->topLine;
	*numLines=theViewInfo->numLines;
	*leftPixel=theViewInfo->leftPixel;
	*numPixels=theViewInfo->textBounds.w;
}

void SetEditorViewTopLine(EDITORVIEW *theView,UINT32 lineNumber)
/* set the line number of the top line that is
 * displayed in the given edit view
 * NOTE: this does no invalidation of the view
 */
{
	((GUIVIEWINFO *)(theView->viewInfo))->topLine=lineNumber;
}

UINT32 GetEditorViewTabSize(EDITORVIEW *theView)
/* get the current tab size for the given view
 */
{
	return(((GUIVIEWINFO *)(theView->viewInfo))->tabSize);
}

BOOLEAN SetEditorViewTabSize(EDITORVIEW *theView,UINT32 theSize)
/* set the tab size to be used in theView
 * this will force an update of theView
 * if there is a problem, leave the tab size unchanged, set the error
 * and return FALSE
 */
{
	SetViewTabSize(theView,theSize);
	InvalidateView(theView);
	return(TRUE);
}

BOOLEAN GetEditorViewFont(EDITORVIEW *theView,char *theFont,UINT32 stringBytes)
/* get the font that is in use in theView
 * if the font will not fit within string bytes, return FALSE
 */
{
	GUIVIEWINFO
		*theViewInfo;
	unsigned long
		theValue;
	char
		*atomName;

	if(stringBytes)
		{
		theViewInfo=(GUIVIEWINFO *)theView->viewInfo;
		if(theViewInfo->theFont)
			{
			if(XGetFontProperty(theViewInfo->theFont,XA_FONT,&theValue))
				{
				if(atomName=XGetAtomName(xDisplay,theValue))
					{
					strncpy(theFont,atomName,stringBytes);
					theFont[stringBytes-1]='\0';
					return(TRUE);
					}
				}
			}
		}
	return(FALSE);
}

BOOLEAN SetEditorViewFont(EDITORVIEW *theView,char *theFont)
/* set the font to be used in theView
 * this will force an update of theView
 * if there is a problem, leave the window font unchanged, set the error
 * and return FALSE
 */
{
	UnSetViewFont(theView);										/* get rid of old font for this view */
	SetViewFont(theView,theFont);								/* set new font */
	InvalidateView(theView);
	return(TRUE);
}

BOOLEAN GetEditorViewColors(EDITORVIEW *theView,EDITORCOLOR *foregroundColor,EDITORCOLOR *backgroundColor)
/* get the colors in use by theView
 * if there is a problem, SetError, and return FALSE
 */
{
	GUIVIEWINFO
		*theViewInfo;
	XColor
		theColor;

	theViewInfo=(GUIVIEWINFO *)theView->viewInfo;				/* point at the information record for this view */
	theColor.pixel=theViewInfo->backgroundColor;				/* get pixel of background for this view */
	XQueryColor(xDisplay,DefaultColormap(xDisplay,xScreenNum),&theColor);
	XColorToEditorColor(&theColor,backgroundColor);
	theColor.pixel=theViewInfo->foregroundColor;				/* get pixel of foreground for this view */
	XQueryColor(xDisplay,DefaultColormap(xDisplay,xScreenNum),&theColor);
	XColorToEditorColor(&theColor,foregroundColor);
	return(TRUE);
}

BOOLEAN SetEditorViewColors(EDITORVIEW *theView,EDITORCOLOR foregroundColor,EDITORCOLOR backgroundColor)
/* set the colors to be used in theView
 * this will force an update of theView
 * if there is a problem, leave the colors unchanged, set the error
 * and return FALSE
 */
{
	UnSetViewColors(theView);											/* get rid of old colors for this view */
	SetViewColors(theView,foregroundColor,backgroundColor);				/* set new ones */
	InvalidateView(theView);											/* redraw it all */
	return(TRUE);
}

static void ViewStartCursorChange(EDITORVIEW *theView)
/* when the state (not position) of the cursor
 * is about to change, call this
 */
{
	GUIVIEWINFO
		*theViewInfo;

	theViewInfo=(GUIVIEWINFO *)theView->viewInfo;				/* point at the information record for this view */
	theViewInfo->oldCursorShowing=CursorShowing(theView);		/* remember if it was showing */
}

static void ViewEndCursorChange(EDITORVIEW *theView)
/* when the state (not position) of the cursor
 * has changed, call this, and it will update the cursor as needed
 */
{
	GUIVIEWINFO
		*theViewInfo;

	theViewInfo=(GUIVIEWINFO *)theView->viewInfo;				/* point at the information record for this view */
	if(theViewInfo->oldCursorShowing!=CursorShowing(theView))	/* see if the state has changed */
		{
		InvertViewCursor(theView);								/* if it is showing, then draw it */
		}
}

static void ViewStartSelectionChange(EDITORVIEW *theView)
/* the selection in this view is about to change, so prepare for it
 * since this could move the cursor, turn it off
 */
{
	GUIVIEWINFO
		*theViewInfo;

	theViewInfo=(GUIVIEWINFO *)theView->viewInfo;		/* point at the information record for this view */
	theViewInfo->oldSelectedRegion=XCreateRegion();		/* create a region to hold what is currently selected */
	CreateSelectionRegion(theView,theViewInfo->topLine,theViewInfo->numLines,theViewInfo->leftPixel,theViewInfo->textBounds.w,theViewInfo->oldSelectedRegion);
	ViewStartCursorChange(theView);						/* begin cursor update */
	theViewInfo->cursorOn=FALSE;						/* turn cursor off */
	ViewEndCursorChange(theView);
}

static void ViewEndSelectionChange(EDITORVIEW *theView)
/* the selection in this view has changed, update it
 */
{
	GUIVIEWINFO
		*theViewInfo;
	GC
		graphicsContext;
	Window
		xWindow;
	Region
		newSelectionRegion;

	newSelectionRegion=XCreateRegion();					/* create a region to hold what is currently selected */
	theViewInfo=(GUIVIEWINFO *)theView->viewInfo;			/* point at the information record for this view */
	CreateSelectionRegion(theView,theViewInfo->topLine,theViewInfo->numLines,theViewInfo->leftPixel,theViewInfo->textBounds.w,newSelectionRegion);
	XXorRegion(newSelectionRegion,theViewInfo->oldSelectedRegion,newSelectionRegion);

	xWindow=((WINDOWLISTELEMENT *)theView->parentWindow->userData)->xWindow;		/* get x window for this view */
	graphicsContext=XCreateGC(xDisplay,xWindow,0,NULL);	/* create context in which to draw the inversion, (0 for mask)= use default */

	XSetRegion(xDisplay,graphicsContext,newSelectionRegion);	/* clip to area we wish to invert */
	XSetFunction(xDisplay,graphicsContext,GXxor);		/* prepare to fill in xor format */
	XSetFillStyle(xDisplay,graphicsContext,FillSolid);	/* use solid fill */
	XSetForeground(xDisplay,graphicsContext,theViewInfo->foregroundColor^theViewInfo->backgroundColor);	/* this is the value to xor with */

	XFillRectangle(xDisplay,xWindow,graphicsContext,theViewInfo->textBounds.x,theViewInfo->textBounds.y,theViewInfo->textBounds.w,theViewInfo->textBounds.h);

	XFreeGC(xDisplay,graphicsContext);					/* get rid of the graphics context */
	XDestroyRegion(theViewInfo->oldSelectedRegion);
	XDestroyRegion(newSelectionRegion);					/* get rid of the region we created */
	ViewStartCursorChange(theView);						/* begin cursor update */
	theViewInfo->cursorOn=TRUE;							/* turn cursor back on */
	ViewEndCursorChange(theView);
}

static void BlinkViewCursor(EDITORVIEW *theView)
/* calling this will cause the cursor in theView to change blink states, updating it as needed
 */
{
	GUIVIEWINFO
		*theViewInfo;

	theViewInfo=(GUIVIEWINFO *)theView->viewInfo;			/* point at the information record for this view */
	ViewStartCursorChange(theView);
	theViewInfo->blinkState=!theViewInfo->blinkState;		/* invert blink state */
	ViewEndCursorChange(theView);
}

void ResetEditorViewCursorBlink(EDITORVIEW *theView)
/* when this is called, it will reset the cursor blinking, forcing the
 * cursor to the blink-on state, and resetting the blink timeout
 */
{
	GUIVIEWINFO
		*theViewInfo;

	theViewInfo=(GUIVIEWINFO *)theView->viewInfo;			/* point at the information record for this view */
	if(!theViewInfo->blinkState)							/* if it is currently off, blink it to on */
		{
		BlinkViewCursor(theView);
		}
	ResetCursorBlinkTime();
}

void ViewCursorTask(EDITORVIEW *theView)
/* handle blinking the cursor for theView
 */
{
	BlinkViewCursor(theView);
}

void ViewsStartSelectionChange(EDITORUNIVERSE *theUniverse)
/* the selection in theUniverse is about to be changed in some way, do whatever is needed
 * NOTE: a change in the selection could mean something as simple as a cursor position change
 * NOTE also: selection changes cannot be nested, and nothing except the selection is allowed
 * to be altered during a selection change
 */
{
	EDITORVIEW
		*currentView;

	currentView=theUniverse->firstView;
	while(currentView)									/* walk through all views, update each one as needed */
		{
		ViewStartSelectionChange(currentView);
		currentView=currentView->nextUniverseView;		/* locate next view of this edit universe */
		}
}

void ViewsEndSelectionChange(EDITORUNIVERSE *theUniverse)
/* the selection in theUniverse has finished being changed
 * do whatever needs to be done
 */
{
	EDITORVIEW
		*currentView;

	currentView=theUniverse->firstView;
	while(currentView)									/* walk through all views, update each one as needed */
		{
		ViewEndSelectionChange(currentView);
		if(currentView->viewSelectionChangedVector)		/* see if someone needs to be called because of my change */
			{
			currentView->viewSelectionChangedVector(currentView);	/* call him with my pointer */
			}
		currentView=currentView->nextUniverseView;		/* locate next view of this edit universe */
		}
}

void ViewsStartTextChange(EDITORUNIVERSE *theUniverse)
/* the text in theUniverse is about to be changed in some way, do whatever is needed
 * NOTE: when the text changes, it is possible that the selection information
 * will also change, so that updates will be handled correctly
 */
{
	EDITORVIEW
		*currentView;

	currentView=theUniverse->firstView;
	while(currentView)									/* walk through all views, update each one as needed */
		{
		ViewStartCursorChange(currentView);
		((GUIVIEWINFO *)(currentView->viewInfo))->cursorOn=FALSE;	/* turn cursor off while it is moved */
		ViewEndCursorChange(currentView);
		currentView=currentView->nextUniverseView;		/* locate next view of this edit universe */
		}
}

void ViewsEndTextChange(EDITORUNIVERSE *theUniverse)
/* the text in theUniverse has finished being changed, and the views have been
 * updated, do whatever needs to be done
 */
{
	EDITORVIEW
		*currentView;

	currentView=theUniverse->firstView;
	while(currentView)								/* walk through all views, update each one as needed */
		{
		ViewStartCursorChange(currentView);
		((GUIVIEWINFO *)(currentView->viewInfo))->cursorOn=TRUE;	/* turn cursor back on when done */
		ViewEndCursorChange(currentView);
		if(currentView->viewTextChangedVector)		/* see if someone needs to be called because of my change */
			{
			currentView->viewTextChangedVector(currentView);	/* call him with my pointer */
			}
		currentView=currentView->nextUniverseView;		/* locate next view of this edit universe */
		}
}

BOOLEAN TrackView(EDITORVIEW *theView,XEvent *theEvent)
/* call this to see if the click was in the view, and if so, track it
 * if the click was not in the view, return FALSE
 */
{
	VIEWEVENT
		theRecord;
	VIEWCLICKEVENTDATA
		theClickData;
	GUIVIEWINFO
		*theViewInfo;
	Window
		root,
		child;
	int
		rootX,
		rootY,
		windowX,
		windowY;
	unsigned int
		state;
	UINT32
		numClicks,
		keyCode,
		modifiers;

	if(PointInView(theView,theEvent->xbutton.x,theEvent->xbutton.y))
		{
		numClicks=HandleMultipleClicks(theEvent);
		XStateToEditorModifiers(theEvent->xbutton.state,numClicks,&modifiers);
		switch(theEvent->xbutton.button)
			{
			case Button1:
				keyCode=0;
				break;
			case Button2:
				keyCode=1;
				break;
			case Button3:
				keyCode=2;
				break;
			default:
				keyCode=0;
				break;
			}
		theClickData.keyCode=keyCode;
		theClickData.modifiers=modifiers;

		theRecord.eventType=VET_CLICK;
		theRecord.theView=theView;
		theRecord.eventData=&theClickData;
		theViewInfo=(GUIVIEWINFO *)(theView->viewInfo);
		LocalClickToViewClick(theRecord.theView,theEvent->xbutton.x,theEvent->xbutton.y,&theClickData.xClick,&theClickData.yClick);
		if((theClickData.xClick>=0)&&(theClickData.xClick<theViewInfo->textBounds.w))	/* make sure initial click is in the text part of the view */
			{
			if((theClickData.yClick>=0)&&(theClickData.yClick<theViewInfo->numLines))	/* make sure initial click is in the view */
				{
				HandleViewEvent(&theRecord);
				while(StillDown(theEvent->xbutton.button,1))							/* send repeat events for the view click */
					{
					if(XQueryPointer(xDisplay,((WINDOWLISTELEMENT *)theView->parentWindow->userData)->xWindow,&root,&child,&rootX,&rootY,&windowX,&windowY,&state))
						{
						theRecord.theView=theView;
						theRecord.eventType=VET_CLICKHOLD;
						theRecord.eventData=&theClickData;
						theClickData.modifiers=keyCode;
						theClickData.modifiers=modifiers;
						LocalClickToViewClick(theRecord.theView,windowX,windowY,&theClickData.xClick,&theClickData.yClick);
						HandleViewEvent(&theRecord);
						}
					UpdateEditorWindows();
					}
				}
			}
		return(TRUE);
		}
	return(FALSE);
}

void EditorActivateView(EDITORVIEW *theView)
/* activate theView (this is not nestable)
 */
{
	ViewStartSelectionChange(theView);
	((GUIVIEWINFO *)(theView->viewInfo))->viewActive=TRUE;
	ViewEndSelectionChange(theView);
}

void EditorDeactivateView(EDITORVIEW *theView)
/* deactivate theView (this is not nestable)
 */
{
	ViewStartSelectionChange(theView);
	((GUIVIEWINFO *)(theView->viewInfo))->viewActive=FALSE;
	ViewEndSelectionChange(theView);
}
