/* Search/replace functions
   Copyright (C) 1995 Viacom New Media.

	This file is part of e93.

	e93 is free software; you can redistribute it and/or modify
	it under the terms of the e93 LICENSE AGREEMENT.

	e93 is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	e93 LICENSE AGREEMENT for more details.

	You should have received a copy of the e93 LICENSE AGREEMENT
	along with e93; see the file "LICENSE.TXT".
*/

#include "includes.h"

static UINT8
	upperCaseTable[]=				/* used to convert character to upper case quickly */
		{
		0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0A,0x0B,0x0C,0x0D,0x0E,0x0F,
		0x10,0x11,0x12,0x13,0x14,0x15,0x16,0x17,0x18,0x19,0x1A,0x1B,0x1C,0x1D,0x1E,0x1F,
		0x20,0x21,0x22,0x23,0x24,0x25,0x26,0x27,0x28,0x29,0x2A,0x2B,0x2C,0x2D,0x2E,0x2F,
		0x30,0x31,0x32,0x33,0x34,0x35,0x36,0x37,0x38,0x39,0x3A,0x3B,0x3C,0x3D,0x3E,0x3F,
		0x40,0x41,0x42,0x43,0x44,0x45,0x46,0x47,0x48,0x49,0x4A,0x4B,0x4C,0x4D,0x4E,0x4F,
		0x50,0x51,0x52,0x53,0x54,0x55,0x56,0x57,0x58,0x59,0x5A,0x5B,0x5C,0x5D,0x5E,0x5F,
		0x60,0x41,0x42,0x43,0x44,0x45,0x46,0x47,0x48,0x49,0x4A,0x4B,0x4C,0x4D,0x4E,0x4F,
		0x50,0x51,0x52,0x53,0x54,0x55,0x56,0x57,0x58,0x59,0x5A,0x7B,0x7C,0x7D,0x7E,0x7F,
		0x80,0x81,0x82,0x83,0x84,0x85,0x86,0x87,0x88,0x89,0x8A,0x8B,0x8C,0x8D,0x8E,0x8F,
		0x90,0x91,0x92,0x93,0x94,0x95,0x96,0x97,0x98,0x99,0x9A,0x9B,0x9C,0x9D,0x9E,0x9F,
		0xA0,0xA1,0xA2,0xA3,0xA4,0xA5,0xA6,0xA7,0xA8,0xA9,0xAA,0xAB,0xAC,0xAD,0xAE,0xAF,
		0xB0,0xB1,0xB2,0xB3,0xB4,0xB5,0xB6,0xB7,0xB8,0xB9,0xBA,0xBB,0xBC,0xBD,0xBE,0xBF,
		0xC0,0xC1,0xC2,0xC3,0xC4,0xC5,0xC6,0xC7,0xC8,0xC9,0xCA,0xCB,0xCC,0xCD,0xCE,0xCF,
		0xD0,0xD1,0xD2,0xD3,0xD4,0xD5,0xD6,0xD7,0xD8,0xD9,0xDA,0xDB,0xDC,0xDD,0xDE,0xDF,
		0xE0,0xE1,0xE2,0xE3,0xE4,0xE5,0xE6,0xE7,0xE8,0xE9,0xEA,0xEB,0xEC,0xED,0xEE,0xEF,
		0xF0,0xF1,0xF2,0xF3,0xF4,0xF5,0xF6,0xF7,0xF8,0xF9,0xFA,0xFB,0xFC,0xFD,0xFE,0xFF
		};

static char localErrorFamily[]="search";

enum
	{
	MATCHEDNOTHING,
	TCLERROR
	};

static char *errorMembers[]=
	{
	"MatchedNothing",
	"TclError"
	};

static char *errorDescriptions[]=
	{
	"Find matched nothing -- could not continue",
	"Tcl error"
	};

typedef struct
	{
	BOOLEAN
		ignoreCase;					/* TRUE if we should ignore case, FALSE if not */
	TEXTUNIVERSE
		*searchForText;				/* pointer to the search for text universe */
	TEXTUNIVERSE
		*replaceWithText;			/* pointer to the replace with text universe */
	COMPILEDEXPRESSION
		*expression;				/* if type is regex, this points to the compiled expression, otherwise it is NULL */
	char
		*procedure;					/* if non NULL, this points to a procedure to use instead of the replace text */
	} SEARCHRECORD;

BOOLEAN LiteralMatch(CHUNKHEADER *theChunk,UINT32 theOffset,CHUNKHEADER *forTextChunk,UINT32 forTextOffset,UINT32 numToSearch,CHUNKHEADER **endChunk,UINT32 *endOffset)
/* see if the text at theChunk/theOffset matches the text in forTextChunk/forTextOffset
 * if there is a match, return TRUE with endChunk, and endOffset filled in to point to one
 * past the last place looked in inText
 * NOTE: there must be at least numToSearch bytes remaining in both theChunk/theOffset and forTextChunk/forTextOffset
 */
{
	UINT8
		*inPointer,
		*forPointer;
	UINT32
		inTotal,
		forTotal;

	if(numToSearch)
		{
		if(forTextChunk->data[forTextOffset]==theChunk->data[theOffset])		/* for speed, make initial check to see if this can match */
			{
			inTotal=theChunk->totalBytes;										/* cache total bytes, and pointer */
			inPointer=&(theChunk->data[theOffset+1]);
			forTotal=forTextChunk->totalBytes;
			forPointer=&(forTextChunk->data[forTextOffset+1]);
			if((++theOffset)>=inTotal)											/* move past bytes just checked */
				{
				theOffset=0;
				if(theChunk=theChunk->nextHeader)
					{
					inTotal=theChunk->totalBytes;
					inPointer=theChunk->data;
					}
				}
			if((++forTextOffset)>=forTotal)
				{
				forTextOffset=0;
				if(forTextChunk=forTextChunk->nextHeader)
					{
					forTotal=forTextChunk->totalBytes;
					forPointer=forTextChunk->data;
					}
				}
			numToSearch--;
			while(numToSearch&&(*inPointer++==*forPointer++))					/* go until mismatch, or until we run out of for data */
				{
				if((++theOffset)>=inTotal)
					{
					theOffset=0;
					if(theChunk=theChunk->nextHeader)
						{
						inTotal=theChunk->totalBytes;
						inPointer=theChunk->data;
						}
					}
				if((++forTextOffset)>=forTotal)
					{
					forTextOffset=0;
					if(forTextChunk=forTextChunk->nextHeader)
						{
						forTotal=forTextChunk->totalBytes;
						forPointer=forTextChunk->data;
						}
					}
				numToSearch--;
				}
			if(!numToSearch)
				{
				*endChunk=theChunk;
				*endOffset=theOffset;
				return(TRUE);
				}
			}
		}
	else
		{
		*endChunk=theChunk;
		*endOffset=theOffset;
		return(TRUE);
		}
	return(FALSE);
}

BOOLEAN LiteralMatchTT(CHUNKHEADER *theChunk,UINT32 theOffset,CHUNKHEADER *forTextChunk,UINT32 forTextOffset,UINT32 numToSearch,UINT8 *translateTable,CHUNKHEADER **endChunk,UINT32 *endOffset)
/* see if the text at theChunk/theOffset matches the text in forTextChunk/forTextOffset
 * if there is a match, return TRUE with endChunk, and endOffset filled in to point to one
 * past the last place looked in inText
 * NOTE: there must be at least numToSearch bytes remaining in both theChunk/theOffset and forTextChunk/forTextOffset
 * translate characters when matching
 */
{
	UINT8
		*inPointer,
		*forPointer;
	UINT32
		inTotal,
		forTotal;

	if(numToSearch)
		{
		if(translateTable[forTextChunk->data[forTextOffset]]==translateTable[theChunk->data[theOffset]])		/* for speed, make initial check to see if this can match */
			{
			inTotal=theChunk->totalBytes;										/* cache total bytes, and pointer */
			inPointer=&(theChunk->data[theOffset+1]);
			forTotal=forTextChunk->totalBytes;
			forPointer=&(forTextChunk->data[forTextOffset+1]);
			if((++theOffset)>=inTotal)											/* move past bytes just checked */
				{
				theOffset=0;
				if(theChunk=theChunk->nextHeader)
					{
					inTotal=theChunk->totalBytes;
					inPointer=theChunk->data;
					}
				}
			if((++forTextOffset)>=forTotal)
				{
				forTextOffset=0;
				if(forTextChunk=forTextChunk->nextHeader)
					{
					forTotal=forTextChunk->totalBytes;
					forPointer=forTextChunk->data;
					}
				}
			numToSearch--;
			while(numToSearch&&(translateTable[*inPointer++]==translateTable[*forPointer++]))	/* go until mismatch, or until we run out of for data */
				{
				if((++theOffset)>=inTotal)
					{
					theOffset=0;
					if(theChunk=theChunk->nextHeader)
						{
						inTotal=theChunk->totalBytes;
						inPointer=theChunk->data;
						}
					}
				if((++forTextOffset)>=forTotal)
					{
					forTextOffset=0;
					if(forTextChunk=forTextChunk->nextHeader)
						{
						forTotal=forTextChunk->totalBytes;
						forPointer=forTextChunk->data;
						}
					}
				numToSearch--;
				}
			if(!numToSearch)
				{
				*endChunk=theChunk;
				*endOffset=theOffset;
				return(TRUE);
				}
			}
		}
	else
		{
		*endChunk=theChunk;
		*endOffset=theOffset;
		return(TRUE);
		}
	return(FALSE);
}

static BOOLEAN SearchForwardLiteral(CHUNKHEADER *theChunk,UINT32 theOffset,CHUNKHEADER *forTextChunk,UINT32 forTextOffset,UINT32 forTextLength,UINT32 numToSearch,BOOLEAN ignoreCase,UINT32 *matchOffset,CHUNKHEADER **startChunk,UINT32 *startOffset,CHUNKHEADER **endChunk,UINT32 *endOffset)
/* search forward from theChunk/theOffset for forTextChunk/forTextOffset
 * look at up to numToSearch bytes of source data
 * if a match is located, return TRUE, with matchOffset set to the number
 * of bytes in from theChunk/theOffset where the match was found, and
 * startChunk/startOffset set to the start of the match, and
 * endChunk/endOffset set to one past the last byte of source that was matched
 */
{
	UINT32
		numSearched;
	BOOLEAN
		found;

	numSearched=0;
	found=FALSE;
	if(ignoreCase)
		{
		while(!found&&((forTextLength+numSearched)<=numToSearch))
			{
			if(LiteralMatchTT(theChunk,theOffset,forTextChunk,forTextOffset,forTextLength,upperCaseTable,endChunk,endOffset))
				{
				*matchOffset=numSearched;
				*startChunk=theChunk;
				*startOffset=theOffset;
				found=TRUE;
				}
			else
				{
				if((++theOffset)>=theChunk->totalBytes)			/* move forward in the input */
					{
					theOffset=0;
					theChunk=theChunk->nextHeader;
					}
				numSearched++;
				}
			}
		}
	else
		{
		while(!found&&((forTextLength+numSearched)<=numToSearch))
			{
			if(LiteralMatch(theChunk,theOffset,forTextChunk,forTextOffset,forTextLength,endChunk,endOffset))
				{
				*matchOffset=numSearched;
				*startChunk=theChunk;
				*startOffset=theOffset;
				found=TRUE;
				}
			else
				{
				if((++theOffset)>=theChunk->totalBytes)			/* move forward in the input */
					{
					theOffset=0;
					theChunk=theChunk->nextHeader;
					}
				numSearched++;
				}
			}
		}
	return(found);
}

static BOOLEAN SearchBackwardLiteral(CHUNKHEADER *theChunk,UINT32 theOffset,CHUNKHEADER *forTextChunk,UINT32 forTextOffset,UINT32 forTextLength,UINT32 numToSearch,BOOLEAN ignoreCase,UINT32 *matchOffset,CHUNKHEADER **startChunk,UINT32 *startOffset,CHUNKHEADER **endChunk,UINT32 *endOffset)
/* search backward from theChunk/theOffset for forTextChunk/forTextOffset
 * look at up to numToSearch bytes of source data
 * if a match is located, return TRUE, with matchOffset set to the number
 * of bytes back from theChunk/theOffset where the match was found, and
 * startChunk/startOffset set to theChunk/theOffset at the point of the match
 * endChunk/endOffset set to position just after the last match
 */
{
	UINT32
		numSearched;
	UINT32
		amountToBackUp;
	BOOLEAN
		found;

	numSearched=0;
	found=FALSE;
	if(forTextLength<=numToSearch)
		{
		amountToBackUp=forTextLength;				/* back up in the source by forTextLength */
		while(amountToBackUp)
			{
			if(theOffset>=amountToBackUp)
				{
				theOffset-=amountToBackUp;
				amountToBackUp=0;
				}
			else
				{
				amountToBackUp-=(theOffset+1);
				if(theChunk=theChunk->previousHeader)
					{
					theOffset=theChunk->totalBytes-1;
					}
				}
			}
		numSearched=forTextLength;
		if(ignoreCase)
			{
			while(!found&&(numSearched<=numToSearch))
				{
				if(LiteralMatchTT(theChunk,theOffset,forTextChunk,forTextOffset,forTextLength,upperCaseTable,endChunk,endOffset))
					{
					*matchOffset=numSearched;
					*startChunk=theChunk;
					*startOffset=theOffset;
					found=TRUE;
					}
				else
					{
					if(!theOffset)					/* move backward in the input */
						{
						if(theChunk=theChunk->previousHeader)
							{
							theOffset=theChunk->totalBytes;
							}
						}
					theOffset--;
					numSearched++;
					}
				}
			}
		else
			{
			while(!found&&(numSearched<=numToSearch))
				{
				if(LiteralMatch(theChunk,theOffset,forTextChunk,forTextOffset,forTextLength,endChunk,endOffset))
					{
					*matchOffset=numSearched;
					*startChunk=theChunk;
					*startOffset=theOffset;
					found=TRUE;
					}
				else
					{
					if(!theOffset)					/* move backward in the input */
						{
						if(theChunk=theChunk->previousHeader)
							{
							theOffset=theChunk->totalBytes;
							}
						}
					theOffset--;
					numSearched++;
					}
				}
			}
		}
	return(found);
}

static BOOLEAN SearchForwardRE(CHUNKHEADER *theChunk,UINT32 theOffset,COMPILEDEXPRESSION *theCompiledExpression,BOOLEAN leftEdge,BOOLEAN rightEdge,UINT32 numToSearch,BOOLEAN ignoreCase,BOOLEAN *foundMatch,UINT32 *matchOffset,UINT32 *numMatched,CHUNKHEADER **startChunk,UINT32 *startOffset,CHUNKHEADER **endChunk,UINT32 *endOffset)
/* search for the expression return TRUE in foundMatch if it is found
 * This repeatedly calls REMatch
 * if there is a hard error, SetError will be called, and FALSE returned
 */
{
	BOOLEAN
		fail;

	(*foundMatch)=FALSE;
	(*matchOffset)=0;
	fail=!REMatch(theChunk,theOffset,theCompiledExpression,leftEdge,rightEdge,numToSearch,ignoreCase?upperCaseTable:NULL,foundMatch,numMatched,endChunk,endOffset);
	while(!fail&&!(*foundMatch)&&(*matchOffset)++<numToSearch)
		{
		if((++theOffset)>=theChunk->totalBytes)			/* move forward in the input, and try a match there */
			{
			theChunk=theChunk->nextHeader;
			theOffset=0;
			}
		fail=!REMatch(theChunk,theOffset,theCompiledExpression,FALSE,rightEdge,numToSearch-(*matchOffset),ignoreCase?upperCaseTable:NULL,foundMatch,numMatched,endChunk,endOffset);
		}
	*startChunk=theChunk;
	*startOffset=theOffset;
	return(!fail);
}

static BOOLEAN SearchBackwardRE(CHUNKHEADER *theChunk,UINT32 theOffset,COMPILEDEXPRESSION *theCompiledExpression,BOOLEAN leftEdge,BOOLEAN rightEdge,UINT32 numToSearch,BOOLEAN ignoreCase,BOOLEAN *foundMatch,UINT32 *matchOffset,UINT32 *numMatched,CHUNKHEADER **startChunk,UINT32 *startOffset,CHUNKHEADER **endChunk,UINT32 *endOffset)
/* search backward for the expression return TRUE in foundMatch if it is found
 * This repeatedly calls REMatch
 * if there is a hard error, SetError will be called, and FALSE returned
 */
{
	BOOLEAN
		fail;

	(*foundMatch)=FALSE;
	(*matchOffset)=0;
	fail=!REMatch(theChunk,theOffset,theCompiledExpression,((*matchOffset)==numToSearch)?leftEdge:FALSE,rightEdge,*matchOffset,ignoreCase?upperCaseTable:NULL,foundMatch,numMatched,endChunk,endOffset);
	while(!fail&&!(*foundMatch)&&(*matchOffset)++<numToSearch)
		{
		if(!theOffset)					/* move backward in the input */
			{
			theChunk=theChunk->previousHeader;
			theOffset=theChunk->totalBytes;
			}
		theOffset--;
		fail=!REMatch(theChunk,theOffset,theCompiledExpression,((*matchOffset)==numToSearch)?leftEdge:FALSE,rightEdge,*matchOffset,ignoreCase?upperCaseTable:NULL,foundMatch,numMatched,endChunk,endOffset);
		}
	*startChunk=theChunk;
	*startOffset=theOffset;
	return(!fail);
}

static BOOLEAN SearchForward(CHUNKHEADER *theChunk,UINT32 theOffset,SEARCHRECORD *theRecord,BOOLEAN leftEdge,BOOLEAN rightEdge,UINT32 numToSearch,BOOLEAN *foundMatch,UINT32 *matchOffset,UINT32 *numMatched,CHUNKHEADER **startChunk,UINT32 *startOffset,CHUNKHEADER **endChunk,UINT32 *endOffset)
/* search forward given theRecord
 * return foundMatch TRUE if a match was found
 * return FALSE if there was some sort of hard failure
 * if a match is found, startChunk/startOffset point to the start of the match
 * matchOffset is the number of bytes searched
 * endChunk/endOffset point to one past the end of the matched text
 */
{
	BOOLEAN
		fail;

	fail=FALSE;
	if(theRecord->expression)
		{
		fail=!SearchForwardRE(theChunk,theOffset,theRecord->expression,leftEdge,rightEdge,numToSearch,theRecord->ignoreCase,foundMatch,matchOffset,numMatched,startChunk,startOffset,endChunk,endOffset);
		}
	else
		{
		*foundMatch=SearchForwardLiteral(theChunk,theOffset,theRecord->searchForText->firstChunkHeader,0,theRecord->searchForText->totalBytes,numToSearch,theRecord->ignoreCase,matchOffset,startChunk,startOffset,endChunk,endOffset);
		*numMatched=theRecord->searchForText->totalBytes;
		}
	return(!fail);
}

static BOOLEAN SearchBackward(CHUNKHEADER *theChunk,UINT32 theOffset,SEARCHRECORD *theRecord,BOOLEAN leftEdge,BOOLEAN rightEdge,UINT32 numToSearch,BOOLEAN *foundMatch,UINT32 *matchOffset,UINT32 *numMatched,CHUNKHEADER **startChunk,UINT32 *startOffset,CHUNKHEADER **endChunk,UINT32 *endOffset)
/* search backward given theRecord
 * return foundMatch TRUE if a match was found
 * return FALSE if there was some sort of hard failure
 * if a match is found, startChunk/startOffset point to the start of the match
 * matchOffset is the number of bytes searched
 * endChunk/endOffset point to one past the end of the matched text
 */
{
	BOOLEAN
		fail;

	fail=FALSE;
	if(theRecord->expression)
		{
		fail=!SearchBackwardRE(theChunk,theOffset,theRecord->expression,leftEdge,rightEdge,numToSearch,theRecord->ignoreCase,foundMatch,matchOffset,numMatched,startChunk,startOffset,endChunk,endOffset);
		}
	else
		{
		*foundMatch=SearchBackwardLiteral(theChunk,theOffset,theRecord->searchForText->firstChunkHeader,0,theRecord->searchForText->totalBytes,numToSearch,theRecord->ignoreCase,matchOffset,startChunk,startOffset,endChunk,endOffset);
		*numMatched=theRecord->searchForText->totalBytes;
		}
	return(!fail);
}

static void DisposeSearchRecord(SEARCHRECORD *theRecord)
/* get rid of theRecord, and anything allocated under it
 */
{
	if(theRecord->expression)			/* if there is a selection expression, get rid of it */
		{
		REFree(theRecord->expression);
		}
	if(theRecord->procedure)			/* if there is a procedure, get rid of it */
		{
		MDisposePtr(theRecord->procedure);
		}
	MDisposePtr(theRecord);
}

static SEARCHRECORD *CreateSearchRecord(TEXTUNIVERSE *searchForText,TEXTUNIVERSE *replaceWithText,BOOLEAN selectionExpr,BOOLEAN ignoreCase,BOOLEAN replaceProc)
/* create a search record that describes the search, and remembers some stuff about it
 * if there is a problem, SetError, and return NULL
 */
{
	SEARCHRECORD
		*theRecord;
	COMPILEDEXPRESSION
		*theCompiledExpression;
	UINT8
		*theRE;
	CHUNKHEADER
		*theChunk;
	UINT32
		theOffset;
	BOOLEAN
		fail;

	fail=FALSE;
	if(theRecord=(SEARCHRECORD *)MNewPtr(sizeof(SEARCHRECORD)))
		{
		theRecord->searchForText=searchForText;			/* remember these for later */
		theRecord->replaceWithText=replaceWithText;
		theRecord->ignoreCase=ignoreCase;
		theRecord->expression=NULL;						/* assume no selection expression yet */
		theRecord->procedure=NULL;						/* assume no procedure yet */
		if(replaceProc)									/* if treating replace as procedure, we need to grab the text from the replace buffer, and place it into memory */
			{
			if(theRecord->procedure=(char *)MNewPtr(replaceWithText->totalBytes+1))	/* make a buffer */
				{
				fail=!ExtractUniverseText(replaceWithText,replaceWithText->firstChunkHeader,0,(UINT8 *)theRecord->procedure,replaceWithText->totalBytes,&theChunk,&theOffset);
				theRecord->procedure[replaceWithText->totalBytes]='\0';				/* terminate the string */
				}
			else
				{
				fail=TRUE;
				}
			}
		if(!fail)
			{
			if(selectionExpr)
				{
				if(theRE=(UINT8 *)MNewPtr(searchForText->totalBytes))	/* make a buffer to temporarily hold the expression text */
					{
					if(ExtractUniverseText(searchForText,searchForText->firstChunkHeader,0,theRE,searchForText->totalBytes,&theChunk,&theOffset))
						{
						if(theCompiledExpression=RECompile(&(theRE[0]),searchForText->totalBytes,ignoreCase?upperCaseTable:NULL))
							{
							theRecord->expression=theCompiledExpression;
							MDisposePtr(theRE);				/* get rid of temporary buffer */
							return(theRecord);
							}
						}
					MDisposePtr(theRE);
					}
				}
			else
				{
				return(theRecord);
				}
			}
		if(theRecord->procedure)			/* if there is a procedure, get rid of it */
			{
			MDisposePtr(theRecord->procedure);
			}
		MDisposePtr(theRecord);
		}
	return(NULL);
}

BOOLEAN EditorFind(EDITORUNIVERSE *theEditorUniverse,TEXTUNIVERSE *searchForText,BOOLEAN backward,BOOLEAN wrapAround,BOOLEAN selectionExpr,BOOLEAN ignoreCase,BOOLEAN *foundMatch,UINT32 *matchPosition)
/* find searchForText in theEditorUniverse, starting at the cursor position, or the end of
 * the current selection
 * if searchForText is not located, return FALSE in foundMatch
 * if it is located, cancel any selection, and select the located text, return TRUE in foundMatch
 * and return the position of the match in matchPosition
 * if there is some problem during the search, SetError, return FALSE
 */
{
	CHUNKHEADER
		*inChunk,
		*startChunk,
		*endChunk;
	UINT32
		inOffset,
		startOffset,
		endOffset;
	SEARCHRECORD
		*theRecord;
	UINT32
		startPosition,
		endPosition;
	UINT32
		locatedPosition,
		locatedLength;
	TEXTUNIVERSE
		*searchInText;
	BOOLEAN
		fail;

	fail=*foundMatch=FALSE;
	if(theRecord=CreateSearchRecord(searchForText,NULL,selectionExpr,ignoreCase,FALSE))
		{
		ShowBusy();
		searchInText=theEditorUniverse->textUniverse;

		GetSelectionEndPositions(theEditorUniverse->selectionUniverse,&startPosition,&endPosition);		/* find ends of old selection, or cursor position */
		if(!backward)
			{
			PositionToChunkPosition(searchInText,endPosition,&inChunk,&inOffset);		/* locate place to start the search */
			if(SearchForward(inChunk,inOffset,theRecord,FALSE,FALSE,searchInText->totalBytes-endPosition,foundMatch,&locatedPosition,&locatedLength,&startChunk,&startOffset,&endChunk,&endOffset))
				{
				locatedPosition+=endPosition;											/* make located position absolute */
				if(!(*foundMatch)&&wrapAround)
					{
					if(!SearchForward(searchInText->firstChunkHeader,0,theRecord,FALSE,FALSE,startPosition,foundMatch,&locatedPosition,&locatedLength,&startChunk,&startOffset,&endChunk,&endOffset))
						{
						fail=TRUE;
						}
					}
				}
			else
				{
				fail=TRUE;
				}
			}
		else
			{
			PositionToChunkPositionPastEnd(searchInText,startPosition,&inChunk,&inOffset);	/* locate place to start the search */
			if(SearchBackward(inChunk,inOffset,theRecord,FALSE,FALSE,startPosition,foundMatch,&locatedPosition,&locatedLength,&startChunk,&startOffset,&endChunk,&endOffset))
				{
				locatedPosition=startPosition-locatedPosition;					/* make located position absolute */
				if(!(*foundMatch)&&wrapAround)
					{
					PositionToChunkPositionPastEnd(searchInText,searchInText->totalBytes,&inChunk,&inOffset);	/* locate place to start the search */
					if(SearchBackward(inChunk,inOffset,theRecord,FALSE,FALSE,searchInText->totalBytes-endPosition,foundMatch,&locatedPosition,&locatedLength,&startChunk,&startOffset,&endChunk,&endOffset))
						{
						locatedPosition=searchInText->totalBytes-locatedPosition;
						}
					else
						{
						fail=TRUE;
						}
					}
				}
			else
				{
				fail=TRUE;
				}
			}
		if(!fail&&(*foundMatch))
			{
			*matchPosition=locatedPosition;
			EditorSetNormalSelection(theEditorUniverse,locatedPosition,locatedPosition+locatedLength);
			}
		DisposeSearchRecord(theRecord);
		ShowNotBusy();
		}
	else
		{
		fail=TRUE;
		}
	return(!fail);
}

static BOOLEAN FindAllForward(TEXTUNIVERSE *searchInText,BOOLEAN leftEdge,BOOLEAN rightEdge,UINT32 lowPosition,UINT32 highPosition,SEARCHRECORD *theRecord,SELECTIONUNIVERSE *theSelectionUniverse,ARRAYCHUNKHEADER *selectionChunk,UINT32 selectionOffset,UINT32 *selectionEndPosition,BOOLEAN *foundMatch,UINT32 *firstMatchPosition)
/* Find all occurrences of theRecord between lowPosition, and highPosition, adding the selections
 * to theSelectionUniverse before selectionChunk, selectionOffset
 * if there is a problem, SetError and return FALSE
 * if a matches were found, return TRUE in foundMatch, and the position of the first match found
 * in firstMatchPosition
 * NOTE: even if FALSE is returned, foundMatch, firstMatchPosition, and theSelectionUniverse are
 * valid
 * This routine calls CheckAbort, but ClearAbort should be called externally
 */
{
	BOOLEAN
		fail;
	CHUNKHEADER
		*inChunk,
		*startChunk;
	UINT32
		inOffset,
		startOffset;
	BOOLEAN
		matching;
	UINT32
		nextSearchPosition,
		locatedPosition,
		locatedLength;
	UINT32
		newEndPosition;

	fail=*foundMatch=FALSE;
	nextSearchPosition=lowPosition;
	PositionToChunkPosition(searchInText,nextSearchPosition,&inChunk,&inOffset);		/* locate place to start the search */
	matching=TRUE;
	newEndPosition=*selectionEndPosition;
	while(!fail&&matching)
		{
		if(!(fail=CheckAbort()))
			{
			if(SearchForward(inChunk,inOffset,theRecord,leftEdge,rightEdge,highPosition-nextSearchPosition,&matching,&locatedPosition,&locatedLength,&startChunk,&startOffset,&inChunk,&inOffset))
				{
				if(matching)
					{
					if(locatedLength)											/* finding an empty string is fatal to find all */
						{
						leftEdge=FALSE;											/* no matter what, we just moved off the left edge */
						locatedPosition+=nextSearchPosition;					/* make located position absolute */
						nextSearchPosition=locatedPosition+locatedLength;		/* move to next place to search */
						if(!(*foundMatch))
							{
							(*firstMatchPosition)=locatedPosition;
							(*foundMatch)=TRUE;
							}
						if(InsertUniverseSelection(theSelectionUniverse,selectionChunk,selectionOffset,1,&selectionChunk,&selectionOffset))		/* add a new selection to the end */
							{
							((SELECTIONELEMENT *)selectionChunk->data)[selectionOffset].startOffset=locatedPosition-newEndPosition;
							((SELECTIONELEMENT *)selectionChunk->data)[selectionOffset].endOffset=locatedLength;
							selectionOffset++;									/* move to the next one */
							if(selectionOffset>=selectionChunk->totalElements)
								{
								selectionChunk=selectionChunk->nextHeader;
								selectionOffset=0;
								}
							newEndPosition=nextSearchPosition;
							}
						else
							{
							fail=TRUE;
							}
						}
					else
						{
						SetError(localErrorFamily,errorMembers[MATCHEDNOTHING],errorDescriptions[MATCHEDNOTHING]);
						fail=TRUE;
						}
					}
				}
			else
				{
				fail=TRUE;
				}
			}
		}
	if((*foundMatch)&&selectionChunk)				/* adjust selections below the ones we just added to account for additions */
		{
		((SELECTIONELEMENT *)selectionChunk->data)[selectionOffset].startOffset-=newEndPosition-(*selectionEndPosition);
		}
	*selectionEndPosition=newEndPosition;
	return(!fail);
}

static BOOLEAN FindAllBackward(TEXTUNIVERSE *searchInText,BOOLEAN leftEdge,BOOLEAN rightEdge,UINT32 lowPosition,UINT32 highPosition,SEARCHRECORD *theRecord,SELECTIONUNIVERSE *theSelectionUniverse,ARRAYCHUNKHEADER *selectionChunk,UINT32 selectionOffset,UINT32 *selectionEndPosition,BOOLEAN *foundMatch,UINT32 *firstMatchPosition)
/* Find all occurrences of theRecord between lowPosition, and highPosition, adding the selections
 * to theSelectionUniverse before selectionChunk, selectionOffset
 * if there is a problem, SetError and return FALSE
 * if a matches were found, return TRUE in foundMatch, and the position of the first match found
 * in firstMatchPosition
 * NOTE: even if FALSE is returned, foundMatch, firstMatchPosition, and theSelectionUniverse are
 * valid
 * This routine calls CheckAbort, but ClearAbort should be called externally
 */
{
	BOOLEAN
		fail;
	UINT32
		nextSearchPosition;
	CHUNKHEADER
		*inChunk,
		*endChunk;
	UINT32
		inOffset,
		endOffset;
	BOOLEAN
		matching;
	UINT32
		locatedPosition,
		locatedLength;
	ARRAYCHUNKHEADER
		*nextSelectionChunk;
	UINT32
		nextSelectionOffset;
	UINT32
		newEndPosition;

	fail=*foundMatch=FALSE;
	nextSearchPosition=highPosition;
	PositionToChunkPositionPastEnd(searchInText,nextSearchPosition,&inChunk,&inOffset);			/* locate place to start the search */
	newEndPosition=*selectionEndPosition;
	matching=TRUE;
	while(!fail&&matching)
		{
		if(!(fail=CheckAbort()))
			{
			if(SearchBackward(inChunk,inOffset,theRecord,leftEdge,rightEdge,nextSearchPosition-lowPosition,&matching,&locatedPosition,&locatedLength,&inChunk,&inOffset,&endChunk,&endOffset))
				{
				if(matching)
					{
					if(locatedLength)										/* finding an empty string is fatal to find all */
						{
						rightEdge=FALSE;									/* no matter what, just left the right edge */
						locatedPosition=nextSearchPosition-locatedPosition;	/* make located position absolute */
						nextSearchPosition=locatedPosition;
						if(!(*foundMatch))
							{
							(*firstMatchPosition)=locatedPosition;
							newEndPosition=locatedPosition+locatedLength;
							(*foundMatch)=TRUE;
							}
						if(InsertUniverseSelection(theSelectionUniverse,selectionChunk,selectionOffset,1,&selectionChunk,&selectionOffset))		/* add a new selection before the current one */
							{
							((SELECTIONELEMENT *)selectionChunk->data)[selectionOffset].startOffset=locatedPosition-*selectionEndPosition;
							((SELECTIONELEMENT *)selectionChunk->data)[selectionOffset].endOffset=locatedLength;
							nextSelectionChunk=selectionChunk;
							nextSelectionOffset=selectionOffset;
							nextSelectionOffset++;									/* move to the next one */
							if(nextSelectionOffset>=nextSelectionChunk->totalElements)
								{
								nextSelectionChunk=nextSelectionChunk->nextHeader;
								nextSelectionOffset=0;
								}
							if(nextSelectionChunk)							/* adjust selection below this one to account for amount just added */
								{
								((SELECTIONELEMENT *)nextSelectionChunk->data)[nextSelectionOffset].startOffset-=(locatedPosition-*selectionEndPosition)+locatedLength;
								}
							}
						else
							{
							fail=TRUE;
							}
						}
					else
						{
						SetError(localErrorFamily,errorMembers[MATCHEDNOTHING],errorDescriptions[MATCHEDNOTHING]);
						fail=TRUE;
						}
					}
				}
			else
				{
				fail=TRUE;
				}
			}
		}
	*selectionEndPosition=newEndPosition;
	return(!fail);
}

BOOLEAN EditorFindAll(EDITORUNIVERSE *theEditorUniverse,TEXTUNIVERSE *searchForText,BOOLEAN backward,BOOLEAN wrapAround,BOOLEAN selectionExpr,BOOLEAN ignoreCase,BOOLEAN limitScope,BOOLEAN *foundMatch,UINT32 *firstMatchPosition)
/* find all occurrences of searchForText using the given rules, if there is an error,
 * or the user aborts, SetError, and return FALSE. If nothing is found, set found to FALSE
 * if found returns TRUE, firstMatchPosition reflects the position in the text where the first match
 * was located
 * This routine calls CheckAbort, but ClearAbort should be called externally
 */
{
	SEARCHRECORD
		*theRecord;
	UINT32
		startPosition,
		endPosition;
	UINT32
		tempFirstMatchPosition,
		selectionEndPosition;
	TEXTUNIVERSE
		*searchInText;
	BOOLEAN
		hadMatch,
		fail;
	SELECTIONUNIVERSE
		*tempSelectionUniverse,
		*newSelectionUniverse;
	ARRAYCHUNKHEADER
		*previousChunk,
		*theChunk;
	UINT32
		theOffset;

	fail=*foundMatch=FALSE;
	if(theRecord=CreateSearchRecord(searchForText,NULL,selectionExpr,ignoreCase,FALSE))
		{
		ShowBusy();
		if(newSelectionUniverse=OpenSelectionUniverse())
			{
			searchInText=theEditorUniverse->textUniverse;
			*firstMatchPosition=0;
			EditorStartSelectionChange(theEditorUniverse);
			if(!backward)
				{
				if(!limitScope)
					{
					GetSelectionEndPositions(theEditorUniverse->selectionUniverse,&startPosition,&endPosition);		/* find ends of old selection, or cursor position */
					selectionEndPosition=0;
					if(FindAllForward(searchInText,FALSE,FALSE,endPosition,searchInText->totalBytes,theRecord,newSelectionUniverse,NULL,0,&selectionEndPosition,foundMatch,&*firstMatchPosition))
						{
						if(wrapAround)
							{
							selectionEndPosition=0;
							fail=!FindAllForward(searchInText,FALSE,FALSE,0,startPosition,theRecord,newSelectionUniverse,newSelectionUniverse->selectionChunks.firstChunkHeader,0,&selectionEndPosition,&hadMatch,&tempFirstMatchPosition);
							if(!(*foundMatch))
								{
								*foundMatch=hadMatch;
								*firstMatchPosition=tempFirstMatchPosition;
								}
							}
						}
					else
						{
						fail=TRUE;
						}
					}
				else
					{
					if(theEditorUniverse->selectionUniverse->selectionChunks.totalElements)
						{
						if(theChunk=theEditorUniverse->selectionUniverse->selectionChunks.firstChunkHeader)
							{
							theOffset=0;
							startPosition=0;
							selectionEndPosition=0;
							while(theChunk&&!fail)
								{
								startPosition+=((SELECTIONELEMENT *)theChunk->data)[theOffset].startOffset;
								endPosition=startPosition+((SELECTIONELEMENT *)theChunk->data)[theOffset].endOffset;
								if(FindAllForward(searchInText,TRUE,TRUE,startPosition,endPosition,theRecord,newSelectionUniverse,NULL,0,&selectionEndPosition,&hadMatch,&tempFirstMatchPosition))
									{
									theOffset++;
									if(theOffset>=theChunk->totalElements)
										{
										theChunk=theChunk->nextHeader;
										theOffset=0;
										}
									startPosition=endPosition;
									}
								else
									{
									fail=TRUE;
									}
								if(!(*foundMatch))
									{
									*foundMatch=hadMatch;
									*firstMatchPosition=tempFirstMatchPosition;
									}
								}
							}
						}
					}
				}
			else
				{
				if(!limitScope)
					{
					GetSelectionEndPositions(theEditorUniverse->selectionUniverse,&startPosition,&endPosition);		/* find ends of old selection, or cursor position */
					selectionEndPosition=0;
					if(FindAllBackward(searchInText,FALSE,FALSE,0,startPosition,theRecord,newSelectionUniverse,NULL,0,&selectionEndPosition,foundMatch,&*firstMatchPosition))
						{
						if(wrapAround)
							{
							fail=!FindAllBackward(searchInText,FALSE,FALSE,endPosition,searchInText->totalBytes,theRecord,newSelectionUniverse,NULL,0,&selectionEndPosition,&hadMatch,&tempFirstMatchPosition);
							if(!(*foundMatch))
								{
								*foundMatch=hadMatch;
								*firstMatchPosition=tempFirstMatchPosition;
								}
							}
						}
					else
						{
						fail=TRUE;
						}
					}
				else
					{
					if(theEditorUniverse->selectionUniverse->selectionChunks.totalElements)
						{
						if(theChunk=theEditorUniverse->selectionUniverse->selectionChunks.firstChunkHeader)
							{
							theOffset=0;
							startPosition=0;
							previousChunk=NULL;
							while(theChunk)			/* run through to the end of the selection list */
								{
								startPosition+=((SELECTIONELEMENT *)theChunk->data)[theOffset].startOffset+((SELECTIONELEMENT *)theChunk->data)[theOffset].endOffset;
								theOffset++;
								if(theOffset>=theChunk->totalElements)
									{
									previousChunk=theChunk;
									theChunk=theChunk->nextHeader;
									theOffset=0;
									}
								}
							theChunk=previousChunk;
							theOffset=theChunk->totalElements-1;
							while(theChunk&&!fail)
								{
								selectionEndPosition=0;
								endPosition=startPosition;
								startPosition-=((SELECTIONELEMENT *)theChunk->data)[theOffset].endOffset;
								if(FindAllBackward(searchInText,TRUE,TRUE,startPosition,endPosition,theRecord,newSelectionUniverse,newSelectionUniverse->selectionChunks.firstChunkHeader,0,&selectionEndPosition,&hadMatch,&tempFirstMatchPosition))
									{
									startPosition-=((SELECTIONELEMENT *)theChunk->data)[theOffset].startOffset;
									if(!theOffset)
										{
										if(theChunk=theChunk->previousHeader)
											{
											theOffset=theChunk->totalElements;
											}
										}
									theOffset--;
									}
								else
									{
									fail=TRUE;
									}
								if(!(*foundMatch))
									{
									*foundMatch=hadMatch;
									*firstMatchPosition=tempFirstMatchPosition;
									}
								}
							}
						}
					}
				}
			if(*foundMatch)																				/* got something, so change selection universe */
				{
				tempSelectionUniverse=theEditorUniverse->selectionUniverse;								/* swap universes */
				theEditorUniverse->selectionUniverse=newSelectionUniverse;
				newSelectionUniverse=tempSelectionUniverse;
				}
			EditorEndSelectionChange(theEditorUniverse);
			CloseSelectionUniverse(newSelectionUniverse);												/* close this universe */
			}
		else
			{
			fail=TRUE;
			}
		DisposeSearchRecord(theRecord);
		ShowNotBusy();
		}
	else
		{
		fail=TRUE;
		}
	return(!fail);
}

static BOOLEAN ReplaceRegisterReferences(TEXTUNIVERSE *replaceWithUniverse,SEARCHRECORD *searchedForText)
/* run through replaceWithUniverse looking for register specifications, if any are seen,
 * replace them with what searchForText found
 * if there is a problem, SetError, and return FALSE
 * NOTE: registers are specified in replaceWithUniverse using \0 through \9
 * ignore any other \ in replaceWithUniverse
 */
{
	CHUNKHEADER
		*currentChunk,
		*nextChunk;
	UINT32
		currentOffset,
		currentPosition,
		nextOffset;
	UINT8
		theChar;
	BOOLEAN
		fail;

	fail=FALSE;
	currentChunk=replaceWithUniverse->firstChunkHeader;
	currentPosition=currentOffset=0;
	while(currentChunk&&!fail)
		{
		if(currentChunk->data[currentOffset]!='\\')
			{
			currentPosition++;
			currentOffset++;
			if(currentOffset>currentChunk->totalBytes)
				{
				currentChunk=currentChunk->nextHeader;
				currentOffset=0;
				}
			}
		else
			{
			nextChunk=currentChunk;
			nextOffset=currentOffset+1;
			if(nextOffset>nextChunk->totalBytes)
				{
				nextChunk=nextChunk->nextHeader;
				nextOffset=0;
				}
			if(nextChunk)
				{
				theChar=nextChunk->data[nextOffset];
				if(theChar>='0'&&theChar<='9')
					{
					theChar-='0';														/* make into an index */
					if(DeleteUniverseText(replaceWithUniverse,currentPosition,2))		/* remove the register specification */
						{
						if(searchedForText->expression->theRegisters[theChar].theChunk)	/* see if anything to insert */
							{
							fail=!InsertUniverseChunks(replaceWithUniverse,currentPosition,searchedForText->expression->theRegisters[theChar].theChunk,searchedForText->expression->theRegisters[theChar].theOffset,searchedForText->expression->theRegisters[theChar].numBytes);
							currentPosition+=searchedForText->expression->theRegisters[theChar].numBytes;	/* move past stuff just inserted */
							}
						PositionToChunkPosition(replaceWithUniverse,currentPosition,&currentChunk,&currentOffset);		/* locate place to continue looking */
						}
					else
						{
						fail=TRUE;
						}
					}
				else
					{
					currentChunk=nextChunk;		/* push past the quote character */
					currentOffset=nextOffset;
					currentPosition++;
					}
				}
			else
				{
				currentChunk=nextChunk;			/* \ as last character, just terminate scan */
				currentOffset=nextOffset;
				}
			}
		}
	return(!fail);
}

static BOOLEAN CreateFoundVariable(EDITORUNIVERSE *replaceInUniverse,UINT32 replaceOffset,UINT32 replaceLength,SEARCHRECORD *searchRecord)
/* create the "found" variable, and $n variables which are passed to the Tcl procedure as
 * globals.
 * If there is a problem, SetError, and return NULL
 * NOTE: this routine, and ReplaceSelectedText are the ONLY routines in this file that
 * rely on Tcl...
 * if there is a problem, SetError, and return FALSE
 */
{
	CHUNKHEADER
		*theChunk;
	UINT32
		theOffset;
	char
		*theLocatedText;
	int
		varCount;
	char
		varName[32];
	BOOLEAN
		fail;

	fail=FALSE;
	if(theLocatedText=(char *)MNewPtr(replaceLength+1))	/* make pointer to text that was located (this pointer will used to hold all the extracted text) */
		{
		PositionToChunkPosition(replaceInUniverse->textUniverse,replaceOffset,&theChunk,&theOffset);		/* locate place where match was found */
		if(ExtractUniverseText(replaceInUniverse->textUniverse,theChunk,theOffset,(UINT8 *)theLocatedText,replaceLength,&theChunk,&theOffset))
			{
			theLocatedText[replaceLength]='\0';			/* terminate the string */
			if(!Tcl_SetVar(theTclInterpreter,"found",theLocatedText,TCL_LEAVE_ERR_MSG)!=TCL_OK)
				{
				SetError(localErrorFamily,errorMembers[TCLERROR],theTclInterpreter->result);
				fail=TRUE;
				}
			}
		else
			{
			fail=TRUE;
			}
		MDisposePtr(theLocatedText);
		}
	else
		{
		fail=TRUE;
		}

	if(!fail&&searchRecord->expression)			/* if it is an expression, make a list of 10 more variables (one for each tagged subexpression) */
		{
		for(varCount=0;!fail&&varCount<10;varCount++)
			{
			sprintf(varName,"%d",varCount);		/* create variable name */
			if(searchRecord->expression->theRegisters[varCount].theChunk)	/* see if anything to add */
				{
				if(theLocatedText=(char *)MNewPtr(replaceLength+1))		/* make pointer to the text */
					{
					if(ExtractUniverseText(replaceInUniverse->textUniverse,searchRecord->expression->theRegisters[varCount].theChunk,searchRecord->expression->theRegisters[varCount].theOffset,(UINT8 *)theLocatedText,searchRecord->expression->theRegisters[varCount].numBytes,&theChunk,&theOffset))
						{
						theLocatedText[searchRecord->expression->theRegisters[varCount].numBytes]='\0';	/* terminate the string */
						if(!Tcl_SetVar(theTclInterpreter,varName,theLocatedText,TCL_LEAVE_ERR_MSG)!=TCL_OK)
							{
							SetError(localErrorFamily,errorMembers[TCLERROR],theTclInterpreter->result);
							fail=TRUE;
							}
						}
					else
						{
						fail=TRUE;
						}
					MDisposePtr(theLocatedText);
					}
				else
					{
					fail=TRUE;
					}
				}
			else
				{
				if(!Tcl_SetVar(theTclInterpreter,varName,"",TCL_LEAVE_ERR_MSG)!=TCL_OK)
					{
					SetError(localErrorFamily,errorMembers[TCLERROR],theTclInterpreter->result);
					fail=TRUE;
					}
				}
			}
		}
	return(!fail);
}

static BOOLEAN ReplaceSearchedText(EDITORUNIVERSE *replaceInUniverse,UINT32 replaceOffset,UINT32 replaceLength,CHUNKHEADER **replaceAtChunk,UINT32 *replaceAtChunkOffset,SEARCHRECORD *searchRecord,UINT32 *bytesReplaced)
/* given text that was just searched for by searchForText, replace it
 * based on the search type
 * if there is a problem, SetError, and return FALSE
 */
{
	BOOLEAN
		fail;
	UINT32
		theLength;
	TEXTUNIVERSE
		*alternateUniverse;

	fail=FALSE;
	if(searchRecord->procedure)								/* handle replacement by calling Tcl procedure */
		{
		if(CreateFoundVariable(replaceInUniverse,replaceOffset,replaceLength,searchRecord))
			{
			if(Tcl_Eval(theTclInterpreter,searchRecord->procedure)==TCL_OK)
				{
				theLength=strlen(theTclInterpreter->result);
				fail=!ReplaceEditorText(replaceInUniverse,replaceOffset,replaceOffset+replaceLength,(UINT8 *)theTclInterpreter->result,theLength);
				(*bytesReplaced)=theLength;
				}
			else
				{
				SetError(localErrorFamily,errorMembers[TCLERROR],theTclInterpreter->result);
				fail=TRUE;
				}
			Tcl_ResetResult(theTclInterpreter);				/* do not leave any Tcl result around (other stuff may append to it) */
			}
		else
			{
			fail=TRUE;
			}
		}
	else
		{
		if(searchRecord->expression)
			{
			if(alternateUniverse=OpenTextUniverse())		/* create alternate universe where we make the real text to replace */
				{
				if(InsertUniverseChunks(alternateUniverse,alternateUniverse->totalBytes,searchRecord->replaceWithText->firstChunkHeader,0,searchRecord->replaceWithText->totalBytes))	/* copy replace with text into alternate universe */
					{
					if(ReplaceRegisterReferences(alternateUniverse,searchRecord))
						{
						fail=!ReplaceEditorChunks(replaceInUniverse,replaceOffset,replaceOffset+replaceLength,alternateUniverse->firstChunkHeader,0,alternateUniverse->totalBytes);
						(*bytesReplaced)=alternateUniverse->totalBytes;
						}
					else
						{
						fail=TRUE;
						}
					}
				else
					{
					fail=TRUE;
					}
				CloseTextUniverse(alternateUniverse);
				}
			else
				{
				fail=TRUE;
				}
			}
		else
			{
			fail=!ReplaceEditorChunks(replaceInUniverse,replaceOffset,replaceOffset+replaceLength,searchRecord->replaceWithText->firstChunkHeader,0,searchRecord->replaceWithText->totalBytes);
			(*bytesReplaced)=searchRecord->replaceWithText->totalBytes;
			}
		}
	return(!fail);
}

BOOLEAN EditorReplace(EDITORUNIVERSE *theEditorUniverse,TEXTUNIVERSE *searchForText,TEXTUNIVERSE *replaceWithText,BOOLEAN backward,BOOLEAN wrapAround,BOOLEAN selectionExpr,BOOLEAN ignoreCase,BOOLEAN replaceProc,BOOLEAN *foundMatch,UINT32 *matchPosition)
/* find searchForText in theEditorUniverse, starting at the cursor position, or the _start_ of
 * the current selection
 * if searchForText is not located, return FALSE in foundMatch
 * if it is located, cancel any selection, replace what was found with replaceWithText, return TRUE in foundMatch
 * with the start position of the match in matchPosition
 * NOTE: this is subtly different from find, in that it searches the selection as well
 * if there is a problem, SetError, and return FALSE
 */
{
	CHUNKHEADER
		*inChunk,
		*startChunk,
		*endChunk;
	UINT32
		inOffset,
		startOffset,
		endOffset;
	SEARCHRECORD
		*theRecord;
	UINT32
		startPosition,
		endPosition,
		bytesReplaced;
	UINT32
		locatedPosition,
		locatedLength;
	TEXTUNIVERSE
		*searchInText;
	BOOLEAN
		fail;

	fail=*foundMatch=FALSE;
	if(theRecord=CreateSearchRecord(searchForText,replaceWithText,selectionExpr,ignoreCase,replaceProc))
		{
		ShowBusy();
		searchInText=theEditorUniverse->textUniverse;

		GetSelectionEndPositions(theEditorUniverse->selectionUniverse,&startPosition,&endPosition);		/* find ends of old selection, or cursor position */
		if(!backward)
			{
			PositionToChunkPosition(searchInText,startPosition,&inChunk,&inOffset);		/* locate place to start the search */
			if(SearchForward(inChunk,inOffset,theRecord,FALSE,FALSE,searchInText->totalBytes-startPosition,foundMatch,&locatedPosition,&locatedLength,&startChunk,&startOffset,&endChunk,&endOffset))
				{
				locatedPosition+=startPosition;											/* make located position absolute */
				if(!(*foundMatch)&&wrapAround)
					{
					if(!SearchForward(searchInText->firstChunkHeader,0,theRecord,FALSE,FALSE,startPosition,foundMatch,&locatedPosition,&locatedLength,&startChunk,&startOffset,&endChunk,&endOffset))
						{
						fail=TRUE;
						}
					}
				}
			else
				{
				fail=TRUE;
				}
			}
		else
			{
			PositionToChunkPositionPastEnd(searchInText,endPosition,&inChunk,&inOffset);	/* locate place to start the search */
			if(SearchBackward(inChunk,inOffset,theRecord,FALSE,FALSE,endPosition,foundMatch,&locatedPosition,&locatedLength,&startChunk,&startOffset,&endChunk,&endOffset))
				{
				locatedPosition=endPosition-locatedPosition;					/* make located position absolute */
				if(!(*foundMatch)&&wrapAround)
					{
					PositionToChunkPositionPastEnd(searchInText,searchInText->totalBytes,&inChunk,&inOffset);	/* locate place to start the search */
					if(SearchBackward(inChunk,inOffset,theRecord,FALSE,FALSE,searchInText->totalBytes-endPosition,foundMatch,&locatedPosition,&locatedLength,&startChunk,&startOffset,&endChunk,&endOffset))
						{
						locatedPosition=searchInText->totalBytes-locatedPosition;
						}
					else
						{
						fail=TRUE;
						}
					}
				}
			else
				{
				fail=TRUE;
				}
			}
		if(!fail&&(*foundMatch))
			{
			*matchPosition=locatedPosition;										/* going to return this */
			EditorSetNormalSelection(theEditorUniverse,locatedPosition,locatedPosition+locatedLength);	/* highlight the thing we are about to write over (if we fail to write, it will remain highlighted) */
			BeginUndoGroup(theEditorUniverse);
			EditorStartReplace(theEditorUniverse);
			fail=!ReplaceSearchedText(theEditorUniverse,locatedPosition,locatedLength,&startChunk,&startOffset,theRecord,&bytesReplaced);
			EditorEndReplace(theEditorUniverse);
			StrictEndUndoGroup(theEditorUniverse);
			}
		DisposeSearchRecord(theRecord);
		ShowNotBusy();
		}
	else
		{
		fail=TRUE;
		}
	return(!fail);
}

static BOOLEAN ReplaceAllForward(EDITORUNIVERSE *theEditorUniverse,BOOLEAN leftEdge,BOOLEAN rightEdge,UINT32 lowPosition,UINT32 *highPosition,SEARCHRECORD *theRecord,SELECTIONUNIVERSE *theSelectionUniverse,ARRAYCHUNKHEADER *selectionChunk,UINT32 selectionOffset,UINT32 *selectionEndPosition,BOOLEAN *foundMatch,UINT32 *firstMatchPosition)
/* Replace all occurrences of theRecord between lowPosition, and highPosition, adding the selections
 * to theSelectionUniverse before selectionChunk, selectionOffset
 * if there is a problem, SetError and return FALSE
 * if a matches were found, return TRUE in foundMatch, and the position of the first match found
 * in firstMatchPosition
 * highPosition is modified to reflect the new highPosition after all replacements have been made
 * NOTE: even if FALSE is returned, foundMatch, firstMatchPosition, highPosition, and theSelectionUniverse are
 * valid
 * This routine calls CheckAbort, but ClearAbort should be called externally
 */
{
	BOOLEAN
		fail;
	CHUNKHEADER
		*inChunk,
		*startChunk;
	UINT32
		inOffset,
		startOffset;
	BOOLEAN
		matching;
	TEXTUNIVERSE
		*searchInText;
	UINT32
		nextSearchPosition,
		locatedPosition,
		locatedLength,
		bytesReplaced;
	UINT32
		newHighPosition,
		newEndPosition;

	fail=*foundMatch=FALSE;
	searchInText=theEditorUniverse->textUniverse;
	nextSearchPosition=lowPosition;
	matching=TRUE;
	newHighPosition=*highPosition;
	newEndPosition=*selectionEndPosition;
	while(!fail&&matching)
		{
		if(!(fail=CheckAbort()))
			{
			PositionToChunkPosition(searchInText,nextSearchPosition,&inChunk,&inOffset);	/* locate place to start the search */
			if(SearchForward(inChunk,inOffset,theRecord,leftEdge,rightEdge,newHighPosition-nextSearchPosition,&matching,&locatedPosition,&locatedLength,&startChunk,&startOffset,&inChunk,&inOffset))
				{
				if(matching)													/* see if a match was found, if not, we are done */
					{
					if(locatedLength)											/* finding an empty string is fatal to replace all */
						{
						leftEdge=FALSE;											/* no longer at left edge */
						locatedPosition+=nextSearchPosition;					/* make located position absolute */
						if(!(*foundMatch))
							{
							(*firstMatchPosition)=locatedPosition;				/* remember the first match position so we can home to it when done */
							(*foundMatch)=TRUE;
							}
						if(ReplaceSearchedText(theEditorUniverse,locatedPosition,locatedLength,&startChunk,&startOffset,theRecord,&bytesReplaced))
							{
							inChunk=startChunk;
							inOffset=startOffset;
							nextSearchPosition=locatedPosition+bytesReplaced;	/* move to the new position after the replacement */
							newHighPosition-=locatedLength;						/* subtract off bytes that were replaced */
							newHighPosition+=bytesReplaced;						/* add back number that replaced them */
							if(bytesReplaced)									/* select what was added if something was added */
								{
								if(InsertUniverseSelection(theSelectionUniverse,selectionChunk,selectionOffset,1,&selectionChunk,&selectionOffset))		/* add a new selection to the end */
									{
									((SELECTIONELEMENT *)selectionChunk->data)[selectionOffset].startOffset=locatedPosition-newEndPosition;
									((SELECTIONELEMENT *)selectionChunk->data)[selectionOffset].endOffset=bytesReplaced;
									selectionOffset++;									/* move to the next one */
									if(selectionOffset>=selectionChunk->totalElements)
										{
										selectionChunk=selectionChunk->nextHeader;
										selectionOffset=0;
										}
									newEndPosition=nextSearchPosition;
									}
								else
									{
									fail=TRUE;
									}
								}
							}
						else
							{
							fail=TRUE;
							}
						}
					else
						{
						SetError(localErrorFamily,errorMembers[MATCHEDNOTHING],errorDescriptions[MATCHEDNOTHING]);
						fail=TRUE;
						}
					}
				}
			else
				{
				fail=TRUE;
				}
			}
		}
	if((*foundMatch)&&selectionChunk)				/* adjust selections below the ones we just added to account for additions, and text changes */
		{
		((SELECTIONELEMENT *)selectionChunk->data)[selectionOffset].startOffset+=newHighPosition-(*highPosition);
		((SELECTIONELEMENT *)selectionChunk->data)[selectionOffset].startOffset-=newEndPosition-(*selectionEndPosition);
		}
	*highPosition=newHighPosition;
	*selectionEndPosition=newEndPosition;
	return(!fail);
}

static BOOLEAN ReplaceAllBackward(EDITORUNIVERSE *theEditorUniverse,BOOLEAN leftEdge,BOOLEAN rightEdge,UINT32 lowPosition,UINT32 *highPosition,SEARCHRECORD *theRecord,SELECTIONUNIVERSE *theSelectionUniverse,ARRAYCHUNKHEADER *selectionChunk,UINT32 selectionOffset,UINT32 *selectionEndPosition,BOOLEAN *foundMatch,UINT32 *firstMatchPosition)
/* Replace all occurrences of theRecord between lowPosition, and highPosition, adding the selections
 * to theSelectionUniverse before selectionChunk, selectionOffset
 * if there is a problem, SetError and return FALSE
 * if a matches were found, return TRUE in foundMatch, and the position of the first match found
 * in firstMatchPosition
 * highPosition is modified to reflect the new highPosition after all replacements have been made
 * NOTE: even if FALSE is returned, foundMatch, firstMatchPosition, highPosition, and theSelectionUniverse are
 * valid
 * This routine calls CheckAbort, but ClearAbort should be called externally
 */
{
	BOOLEAN
		fail;
	CHUNKHEADER
		*inChunk,
		*endChunk;
	UINT32
		inOffset,
		endOffset;
	BOOLEAN
		matching;
	TEXTUNIVERSE
		*searchInText;
	UINT32
		nextSearchPosition,
		locatedPosition,
		locatedLength,
		bytesReplaced;
	ARRAYCHUNKHEADER
		*nextSelectionChunk;
	UINT32
		nextSelectionOffset;
	UINT32
		newHighPosition,
		newEndPosition;

	fail=*foundMatch=FALSE;
	searchInText=theEditorUniverse->textUniverse;
	nextSearchPosition=*highPosition;
	matching=TRUE;
	newHighPosition=*highPosition;
	newEndPosition=*selectionEndPosition;
	while(!fail&&matching)
		{
		if(!(fail=CheckAbort()))
			{
			PositionToChunkPositionPastEnd(searchInText,nextSearchPosition,&inChunk,&inOffset);		/* locate place to start the search */
			if(SearchBackward(inChunk,inOffset,theRecord,leftEdge,rightEdge,nextSearchPosition-lowPosition,&matching,&locatedPosition,&locatedLength,&inChunk,&inOffset,&endChunk,&endOffset))
				{
				if(matching)													/* see if a match was found, if not, we are done */
					{
					if(locatedLength)											/* finding an empty string is fatal to replace all */
						{
						rightEdge=FALSE;										/* no longer at the right edge */
						locatedPosition=nextSearchPosition-locatedPosition;		/* make located position absolute */
						nextSearchPosition=locatedPosition;
						if(!(*foundMatch))										/* if no matches found so far ... */
							{
							newEndPosition=locatedPosition+locatedLength;		/* remember end of selections (sort of, will be adjusted at the end) */
							(*foundMatch)=TRUE;
							}
						if(ReplaceSearchedText(theEditorUniverse,locatedPosition,locatedLength,&inChunk,&inOffset,theRecord,&bytesReplaced))
							{
							newHighPosition-=locatedLength;						/* subtract off bytes that were replaced */
							newHighPosition+=bytesReplaced;						/* add back number that replaced them */
							if(bytesReplaced)									/* add selection if something was replaced */
								{
								if(InsertUniverseSelection(theSelectionUniverse,selectionChunk,selectionOffset,1,&selectionChunk,&selectionOffset))		/* add a new selection to the end */
									{
									((SELECTIONELEMENT *)selectionChunk->data)[selectionOffset].startOffset=locatedPosition-*selectionEndPosition;
									((SELECTIONELEMENT *)selectionChunk->data)[selectionOffset].endOffset=bytesReplaced;
									nextSelectionChunk=selectionChunk;
									nextSelectionOffset=selectionOffset;
									nextSelectionOffset++;									/* move to the next one */
									if(nextSelectionOffset>=nextSelectionChunk->totalElements)
										{
										nextSelectionChunk=nextSelectionChunk->nextHeader;
										nextSelectionOffset=0;
										}
									if(nextSelectionChunk)							/* adjust selection below this one to account for amount just added, and the text just changed */
										{
										((SELECTIONELEMENT *)nextSelectionChunk->data)[nextSelectionOffset].startOffset-=(locatedPosition-*selectionEndPosition)+locatedLength;
										}
									}
								else
									{
									fail=TRUE;
									}
								}
							else
								{
								if(selectionChunk)								/* adjust selections below for amount removed */
									{
									((SELECTIONELEMENT *)selectionChunk->data)[selectionOffset].startOffset-=locatedLength;
									}
								}
							}
						else
							{
							fail=TRUE;
							}
						}
					else
						{
						SetError(localErrorFamily,errorMembers[MATCHEDNOTHING],errorDescriptions[MATCHEDNOTHING]);
						fail=TRUE;
						}
					}
				}
			else
				{
				fail=TRUE;
				}
			}
		}
	*selectionEndPosition=newEndPosition+newHighPosition-(*highPosition);
	*firstMatchPosition=*selectionEndPosition;
	*highPosition=newHighPosition;
	return(!fail);
}

BOOLEAN EditorReplaceAll(EDITORUNIVERSE *theEditorUniverse,TEXTUNIVERSE *searchForText,TEXTUNIVERSE *replaceWithText,BOOLEAN backward,BOOLEAN wrapAround,BOOLEAN selectionExpr,BOOLEAN ignoreCase,BOOLEAN limitScope,BOOLEAN replaceProc,BOOLEAN *foundMatch,UINT32 *firstMatchPosition)
/* replace all occurrences of searchForText with replaceWithText using the given rules, if none are found,
 * return FALSE in foundMatch
 * if foundMatch is TRUE, firstMatchPosition contains the position of the start of the first replacement made
 * if there is a problem, SetError, return FALSE
 * This routine calls CheckAbort, but ClearAbort should be called externally
 * NOTE: this hurt my brain.
 */
{
	SEARCHRECORD
		*theRecord;
	UINT32
		startPosition,
		endPosition;
	UINT32
		tempFirstMatchPosition,
		selectionEndPosition,
		highPosition;
	TEXTUNIVERSE
		*searchInText;
	BOOLEAN
		hadMatch,
		fail;
	SELECTIONUNIVERSE
		*emptySelectionUniverse,
		*oldSelectionUniverse,
		*newSelectionUniverse;
	ARRAYCHUNKHEADER
		*previousChunk,
		*theChunk;
	UINT32
		theOffset;

	fail=*foundMatch=FALSE;
	if(theRecord=CreateSearchRecord(searchForText,replaceWithText,selectionExpr,ignoreCase,replaceProc))
		{
		ShowBusy();
		if(newSelectionUniverse=OpenSelectionUniverse())
			{
			if(emptySelectionUniverse=OpenSelectionUniverse())
				{
				searchInText=theEditorUniverse->textUniverse;
				*firstMatchPosition=0;
				EditorStartSelectionChange(theEditorUniverse);
				oldSelectionUniverse=theEditorUniverse->selectionUniverse;								/* get the universe to read from while replacing */
				theEditorUniverse->selectionUniverse=emptySelectionUniverse;							/* temporarily clear the current selection universe */
				EditorEndSelectionChange(theEditorUniverse);											/* ### this will cause a sort of flash if nothing is replaced */
				BeginUndoGroup(theEditorUniverse);
				EditorStartReplace(theEditorUniverse);
				if(!backward)
					{
					if(!limitScope)
						{
						GetSelectionEndPositions(oldSelectionUniverse,&startPosition,&endPosition);		/* find ends of old selection, or cursor position */
						selectionEndPosition=0;
						highPosition=searchInText->totalBytes;
						if(ReplaceAllForward(theEditorUniverse,FALSE,FALSE,startPosition,&highPosition,theRecord,newSelectionUniverse,NULL,0,&selectionEndPosition,foundMatch,&*firstMatchPosition))
							{
							if(wrapAround)
								{
								selectionEndPosition=0;
								highPosition=startPosition;
								fail=!ReplaceAllForward(theEditorUniverse,FALSE,FALSE,0,&highPosition,theRecord,newSelectionUniverse,newSelectionUniverse->selectionChunks.firstChunkHeader,0,&selectionEndPosition,&hadMatch,&tempFirstMatchPosition);
								if(!(*foundMatch))
									{
									*foundMatch=hadMatch;
									*firstMatchPosition=tempFirstMatchPosition;
									}
								else
									{
									*firstMatchPosition+=highPosition-startPosition;	/* move home position due to text changes above */
									}
								}
							}
						else
							{
							fail=TRUE;
							}
						}
					else
						{
						if(oldSelectionUniverse->selectionChunks.totalElements)
							{
							if(theChunk=oldSelectionUniverse->selectionChunks.firstChunkHeader)
								{
								theOffset=0;
								startPosition=0;
								selectionEndPosition=0;
								while(theChunk&&!fail)
									{
									startPosition+=((SELECTIONELEMENT *)theChunk->data)[theOffset].startOffset;
									endPosition=startPosition+((SELECTIONELEMENT *)theChunk->data)[theOffset].endOffset;
									if(ReplaceAllForward(theEditorUniverse,TRUE,TRUE,startPosition,&endPosition,theRecord,newSelectionUniverse,NULL,0,&selectionEndPosition,&hadMatch,&tempFirstMatchPosition))
										{
										theOffset++;
										if(theOffset>=theChunk->totalElements)
											{
											theChunk=theChunk->nextHeader;
											theOffset=0;
											}
										startPosition=endPosition;
										}
									else
										{
										fail=TRUE;
										}
									if(!(*foundMatch))
										{
										*foundMatch=hadMatch;
										*firstMatchPosition=tempFirstMatchPosition;
										}
									}
								}
							}
						}
					}
				else
					{
					if(!limitScope)
						{
						GetSelectionEndPositions(oldSelectionUniverse,&startPosition,&endPosition);		/* find ends of old selection, or cursor position */
						highPosition=endPosition;
						selectionEndPosition=0;
						if(ReplaceAllBackward(theEditorUniverse,FALSE,FALSE,0,&highPosition,theRecord,newSelectionUniverse,NULL,0,&selectionEndPosition,foundMatch,&*firstMatchPosition))
							{
							if(wrapAround)
								{
								endPosition=highPosition;
								highPosition=searchInText->totalBytes;
								fail=!ReplaceAllBackward(theEditorUniverse,FALSE,FALSE,endPosition,&highPosition,theRecord,newSelectionUniverse,NULL,0,&selectionEndPosition,&hadMatch,&tempFirstMatchPosition);
								if(!(*foundMatch))
									{
									*foundMatch=hadMatch;
									*firstMatchPosition=tempFirstMatchPosition;
									}
								}
							}
						else
							{
							fail=TRUE;
							}
						}
					else
						{
						if(oldSelectionUniverse->selectionChunks.totalElements)
							{
							if(theChunk=oldSelectionUniverse->selectionChunks.firstChunkHeader)
								{
								theOffset=0;
								startPosition=0;
								previousChunk=NULL;
								while(theChunk)			/* run through to the end of the selection list */
									{
									startPosition+=((SELECTIONELEMENT *)theChunk->data)[theOffset].startOffset+((SELECTIONELEMENT *)theChunk->data)[theOffset].endOffset;
									theOffset++;
									if(theOffset>=theChunk->totalElements)
										{
										previousChunk=theChunk;
										theChunk=theChunk->nextHeader;
										theOffset=0;
										}
									}
								theChunk=previousChunk;
								theOffset=theChunk->totalElements-1;
								while(theChunk&&!fail)
									{
									selectionEndPosition=0;
									endPosition=startPosition;
									startPosition-=((SELECTIONELEMENT *)theChunk->data)[theOffset].endOffset;
									highPosition=endPosition;
									if(ReplaceAllBackward(theEditorUniverse,TRUE,TRUE,startPosition,&highPosition,theRecord,newSelectionUniverse,newSelectionUniverse->selectionChunks.firstChunkHeader,0,&selectionEndPosition,&hadMatch,&tempFirstMatchPosition))
										{
										startPosition-=((SELECTIONELEMENT *)theChunk->data)[theOffset].startOffset;
										if(!theOffset)
											{
											if(theChunk=theChunk->previousHeader)
												{
												theOffset=theChunk->totalElements;
												}
											}
										theOffset--;
										}
									else
										{
										fail=TRUE;
										}
									if(!(*foundMatch))
										{
										*foundMatch=hadMatch;
										*firstMatchPosition=tempFirstMatchPosition;
										}
									else
										{
										*firstMatchPosition+=highPosition-endPosition;		/* adjust for any replacements just made */
										}
									}
								}
							}
						}
					}
				EditorEndReplace(theEditorUniverse);
				StrictEndUndoGroup(theEditorUniverse);
				EditorStartSelectionChange(theEditorUniverse);
				if(*foundMatch)																	/* got something, so change selection universe */
					{
					theEditorUniverse->selectionUniverse=newSelectionUniverse;
					theEditorUniverse->selectionUniverse->cursorPosition=*firstMatchPosition;	/* place the cursor at the start of the first change */
					newSelectionUniverse=oldSelectionUniverse;									/* destroy this one */
					}
				else
					{
					theEditorUniverse->selectionUniverse=oldSelectionUniverse;
					}
				EditorEndSelectionChange(theEditorUniverse);
				CloseSelectionUniverse(emptySelectionUniverse);									/* close this universe */
				}
			else
				{
				fail=TRUE;
				}
			CloseSelectionUniverse(newSelectionUniverse);										/* close this universe */
			}
		else
			{
			fail=TRUE;
			}
		DisposeSearchRecord(theRecord);
		ShowNotBusy();
		}
	else
		{
		fail=TRUE;
		}
	return(!fail);
}
