/* Regular expression handling
   Copyright (C) 1995 Viacom New Media.

	This file is part of e93.

	e93 is free software; you can redistribute it and/or modify
	it under the terms of the e93 LICENSE AGREEMENT.

	e93 is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	e93 LICENSE AGREEMENT for more details.

	You should have received a copy of the e93 LICENSE AGREEMENT
	along with e93; see the file "LICENSE.TXT".
*/

#include "includes.h"

static char localErrorFamily[]="regex";

enum
	{
	UNBALANCEDPAREN,
	UNBALANCEDSQUAREBRACKET,
	BADRANGEEXPRESSION,
	EMPTYEXPRESSION,
	BADREPETITION,
	INVALIDREGISTER,
	DANGLINGQUOTE,
	INVALIDDIGIT,
	MISSINGDIGIT,
	STACKOVERFLOW
	};

static char *errorMembers[]=
	{
	"UnbalancedParen",
	"UnbalancedSquareBracket",
	"BadRangeExpression",
	"EmptyExpression",
	"BadRepetition",
	"InvalidRegister",
	"DanglingQuote",
	"InvalidDigit",
	"MissingDigit",
	"StackOverflow"
	};

static char *errorDescriptions[]=
	{
	"Unbalanced ()",
	"Unbalanced [ or [^",
	"Invalid {} expression",
	"Empty expression or group",
	"Repetition operator *, +, ?, or {} applied to nothing",
	"Invalid register specification",
	"Incomplete \\ expression",
	"Invalid digit encountered",
	"Digit expected, but not found",
	"Expression stack overflow"
	};

typedef struct
	{
	BOOLEAN
		leftEdge,					/* tells if search is starting on logical left edge (used for ^) */
		rightEdge;					/* tells if search is starting on logical right edge (used for $) */
	UINT32
		initialNumToSearch;			/* tells how many bytes we were asked to search through at the start */
	CHUNKHEADER
		*endChunk;					/* these are passed in everywhere */
	UINT32
		endOffset;
	} RECURSEGLOBALS;

static EXPRESSIONNODE *REAllocateNode()
/* allocate, and return an expression node
 * if there is a problem, SetError, and return NULL
 */
{
	return((EXPRESSIONNODE *)MNewPtr(sizeof(EXPRESSIONNODE)));
}

static void REFreeNode(EXPRESSIONNODE *theExpressionNode)
/* free the node given by theExpressionNode
 */
{
	MDisposePtr(theExpressionNode);
}

static EXPRESSIONLISTHEADER *REAllocateList()
/* allocate, and return an expression list header
 * if there is a problem, SetError, and return NULL
 */
{
	return((EXPRESSIONLISTHEADER *)MNewPtr(sizeof(EXPRESSIONLISTHEADER)));
}

static void REFreeList(EXPRESSIONLISTHEADER *theExpressionList)
/* free the list given by theExpressionList
 */
{
	MDisposePtr(theExpressionList);
}

static EXPRESSIONNODE *REAddString(EXPRESSIONNODE *theParent,EXPRESSIONNODE *currentNode,UINT8 theChar,UINT8 *translateTable)
/* add theChar to currentNode. If currentNode is NULL, or full, or not of the correct type
 * then create a new node, and make theChar the first of the string
 * NOTE: if currentNode was full, the new node is linked to it's nextExpression
 * return the node that theChar was added to.
 * if there is a problem, SetError, and return NULL
 */
{
	EXPRESSIONNODE
		*theNode;

	if(currentNode&&(currentNode->theType==PT_STRING)&&(!currentNode->matchSpecified)&&(currentNode->dataLength<MAXSTRINGPERNODE))
		{
		theNode=currentNode;							/* there is space in the current node, so add this in */
		if(translateTable)
			{
			theNode->pt.theData[theNode->dataLength++]=translateTable[theChar];	/* drop in the data, bump the amount */
			}
		else
			{
			theNode->pt.theData[theNode->dataLength++]=theChar;	/* drop in the data, bump the amount */
			}
		}
	else
		{
		if(theNode=REAllocateNode())
			{
			theNode->theType=PT_STRING;					/* create a node, and apply the correct type to it */
			theNode->parentExpression=theParent;		/* set the parent */
			theNode->dataLength=1;						/* so far, it contains one byte of string data */
			theNode->pt.theData[0]=theChar;				/* drop in the data */
			theNode->matchSpecified=FALSE;				/* no match count specified for this node yet */
			theNode->matchMin=1;						/* until further notice, this must match exactly once */
			theNode->matchMax=1;
			theNode->nextExpression=NULL;				/* this is now the last expression */
			if(currentNode)
				{
				currentNode->nextExpression=theNode;	/* link to previous node if one existed */
				}
			}
		}
	return(theNode);
}

#define	ADDCHARBIT(theArray,theChar)	((theArray)[(theChar>>3)]|=(1<<(theChar&0x07)))		/* set bit in field */
#define	CLRCHARBIT(theArray,theChar)	((theArray)[(theChar>>3)]&=~(1<<(theChar&0x07)))	/* clear bit in field */
#define	GETCHARBIT(theArray,theChar)	((theArray)[(theChar>>3)]&(1<<(theChar&0x07)))		/* get bit from field */

static BOOLEAN REGetHexDigit(UINT8 *theExpression,UINT32 *index,UINT32 numBytes,UINT8 *theNibble)
/* get a hex digit from the input stream, and convert it to a nibble
 * if there is a problem, set the error, and return FALSE
 */
{
	BOOLEAN
		fail;
	UINT8
		theChar;

	fail=FALSE;
	if(*index<numBytes)
		{
		theChar=theExpression[(*index)++];
		if(theChar>='a')
			{
			theChar-='a'-'A';			/* make lower case into upper case */
			}
		if(theChar>='A')
			{
			theChar-='A'-10;			/* make letter into number */
			if(theChar>15)
				{
				SetError(localErrorFamily,errorMembers[INVALIDDIGIT],errorDescriptions[INVALIDDIGIT]);
				fail=TRUE;
				}
			}
		else
			{
			if(theChar>='0')
				{
				theChar-='0';
				if(theChar>9)
					{
					SetError(localErrorFamily,errorMembers[INVALIDDIGIT],errorDescriptions[INVALIDDIGIT]);
					fail=TRUE;
					}
				}
			else
				{
				SetError(localErrorFamily,errorMembers[INVALIDDIGIT],errorDescriptions[INVALIDDIGIT]);
				fail=TRUE;
				}
			}
		}
	else
		{
		SetError(localErrorFamily,errorMembers[MISSINGDIGIT],errorDescriptions[MISSINGDIGIT]);
		fail=TRUE;
		}
	*theNibble=theChar;
	return(!fail);
}

static BOOLEAN REConvertQuotedCharacter(UINT8 *theExpression,UINT32 *index,UINT32 numBytes,UINT8 *theChar)
/* convert character after a \ into what it actually represents
 * (for instance \n means newline)
 * this can fail if there are no more characters after the \, or
 * if the characters do not make sense
 */
{
	BOOLEAN
		fail;
	UINT8
		highNibble,
		lowNibble;

	fail=FALSE;
	if(*index<numBytes)
		{
		*theChar=theExpression[(*index)++];
		switch(*theChar)
			{
			case 'b':
				*theChar=0x08;
				break;
			case 'n':
				*theChar='\n';
				break;
			case 'r':
				*theChar='\r';
				break;
			case 't':
				*theChar='\t';
				break;
			case 'x':
				if(REGetHexDigit(theExpression,index,numBytes,&highNibble))
					{
					if(REGetHexDigit(theExpression,index,numBytes,&lowNibble))
						{
						*theChar=highNibble*16+lowNibble;			/* create actual character */
						}
					else
						{
						fail=TRUE;
						}
					}
				else
					{
					fail=TRUE;
					}
				break;
			}
		}
	else
		{
		SetError(localErrorFamily,errorMembers[DANGLINGQUOTE],errorDescriptions[DANGLINGQUOTE]);
		fail=TRUE;
		}
	return(!fail);
}

static void REAddRangeToList(EXPRESSIONNODE *theNode,UINT8 rangeLow,UINT8 rangeHigh,UINT8 *translateTable)
/* add the bytes between rangeLow, and rangeHigh (inclusive) to the list in theNode
 */
{
	UINT32
		i;

	if(translateTable)
		{
		for(i=rangeLow;i<=(UINT32)rangeHigh;i++)
			{
			ADDCHARBIT(&(theNode->pt.theBits[0]),translateTable[i]);
			}
		}
	else
		{
		for(i=rangeLow;i<=(UINT32)rangeHigh;i++)
			{
			ADDCHARBIT(&(theNode->pt.theBits[0]),i);
			}
		}
}

static EXPRESSIONNODE *REAddList(EXPRESSIONNODE *theParent,EXPRESSIONNODE *currentNode,UINT8 *theExpression,UINT32 *index,UINT32 numBytes,UINT8 *translateTable)
/* list operator has started, parse it, and create a list node in the list
 * if there is a problem, SetError, and return NULL
 */
{
	EXPRESSIONNODE
		*theNode;
	UINT8
		theChar,
		lastChar;
	BOOLEAN
		not,
		done,
		fail;
	UINT32
		i;

	if(*index<numBytes)
		{
		if(theNode=REAllocateNode())
			{
			theNode->theType=PT_LIST;						/* create a node, and apply the correct type to it */
			theNode->parentExpression=theParent;			/* set the parent */
			theNode->matchSpecified=FALSE;					/* no match count specified for this node yet */
			theNode->matchMin=1;							/* until further notice, this must match exactly once */
			theNode->matchMax=1;
			theNode->nextExpression=NULL;					/* this is now the last expression */
			for(i=0;i<32;i++)								/* clear the array */
				{
				theNode->pt.theBits[i]=0;
				}
			done=fail=FALSE;
			not=FALSE;
			i=*index;										/* remember where we started */
			lastChar=0;										/* start range at 0 */
			while(!done&&!fail&&(*index<numBytes))
				{
				switch(theChar=theExpression[(*index)++])
					{
					case '\\':
						if(REConvertQuotedCharacter(theExpression,index,numBytes,&theChar))
							{
							if(translateTable)
								{
								ADDCHARBIT(&(theNode->pt.theBits[0]),translateTable[theChar]);
								}
							else
								{
								ADDCHARBIT(&(theNode->pt.theBits[0]),theChar);
								}
							}
						else
							{
							fail=TRUE;
							}
						break;
					case '^':
						if(*index-1==i)						/* at start?? */
							{
							not=TRUE;						/* set not flag */
							theChar=0;						/* keep last char pointed to the start */
							}
						else
							{
							if(translateTable)
								{
								ADDCHARBIT(&(theNode->pt.theBits[0]),translateTable[theChar]);
								}
							else
								{
								ADDCHARBIT(&(theNode->pt.theBits[0]),theChar);
								}
							}
						break;
					case '-':
						if(*index<numBytes)					/* get the next character for the range (if there is one) */
							{
							theChar=theExpression[(*index)++];
							switch(theChar)
								{
								case '\\':
									if(REConvertQuotedCharacter(theExpression,index,numBytes,&theChar))
										{
										REAddRangeToList(theNode,lastChar,theChar,translateTable);
										}
									else
										{
										fail=TRUE;
										}
									break;
								case ']':
									theChar=255;
									REAddRangeToList(theNode,lastChar,theChar,translateTable);		/* range pointing into space is assumed max value */
									done=TRUE;
									break;
								default:
									REAddRangeToList(theNode,lastChar,theChar,translateTable);		/* range pointing into space is assumed max value */
									break;
								}
							}
						else
							{
							SetError(localErrorFamily,errorMembers[UNBALANCEDSQUAREBRACKET],errorDescriptions[UNBALANCEDSQUAREBRACKET]);
							fail=TRUE;
							}
						break;
					case ']':
						done=TRUE;							/* scanning is done (empty lists are allowed) */
						break;
					default:								/* otherwise it is normal, and just gets added in */
						if(translateTable)
							{
							ADDCHARBIT(&(theNode->pt.theBits[0]),translateTable[theChar]);
							}
						else
							{
							ADDCHARBIT(&(theNode->pt.theBits[0]),theChar);
							}
						break;
					}
				lastChar=theChar;							/* remember this for next time around the loop */
				}
			if(!done&&!fail)								/* ran out of things to scan */
				{
				SetError(localErrorFamily,errorMembers[UNBALANCEDSQUAREBRACKET],errorDescriptions[UNBALANCEDSQUAREBRACKET]);
				fail=TRUE;
				}
			if(!fail)
				{
				if(not)										/* see if the table needs to be inverted */
					{
					for(i=0;i<32;i++)
						{
						theNode->pt.theBits[i]^=0xFF;		/* invert all the bits in the table if doing "not" search */
						}
					}
				if(currentNode)
					{
					currentNode->nextExpression=theNode;	/* link to previous node if one existed */
					}
				return(theNode);
				}
			REFreeNode(theNode);
			}
		}
	else
		{
		SetError(localErrorFamily,errorMembers[UNBALANCEDSQUAREBRACKET],errorDescriptions[UNBALANCEDSQUAREBRACKET]);
		}
	return(NULL);
}

static EXPRESSIONNODE *REAddAny(EXPRESSIONNODE *theParent,EXPRESSIONNODE *currentNode,UINT8 *theExpression,UINT32 *index,UINT32 numBytes)
/* create a list that matches any character except \n
 * if there is a problem, SetError, and return NULL
 */
{
	UINT32
		i;
	EXPRESSIONNODE
		*theNode;

	if(theNode=REAllocateNode())
		{
		theNode->theType=PT_LIST;						/* create a node, and apply the correct type to it */
		theNode->parentExpression=theParent;			/* set the parent */
		theNode->matchSpecified=FALSE;					/* no match count specified for this node yet */
		theNode->matchMin=1;							/* until further notice, this must match exactly once */
		theNode->matchMax=1;
		theNode->nextExpression=NULL;					/* this is now the last expression */
		for(i=0;i<32;i++)								/* clear the array */
			{
			theNode->pt.theBits[i]=0xFF;				/* select ALL characters */
			}
		CLRCHARBIT(&(theNode->pt.theBits[0]),'\n');		/* remove newline from the list */
		if(currentNode)
			{
			currentNode->nextExpression=theNode;		/* link to previous node if one existed */
			}
		return(theNode);
		}
	return(NULL);
}

static EXPRESSIONNODE *REAddPreviousMatch(EXPRESSIONNODE *theParent,EXPRESSIONNODE *currentNode,REGISTER *theRegister)
/* link in a new node that matches previously matched text
 * if there is a problem, SetError, and return NULL
 */
{
	EXPRESSIONNODE
		*theNode;

	if(theNode=REAllocateNode())
		{
		theNode->theType=PT_PREVIOUSMATCH;				/* create a node, and apply the correct type to it */
		theNode->parentExpression=theParent;			/* set the parent */
		theNode->matchSpecified=FALSE;					/* no match count specified for this node yet */
		theNode->matchMin=1;							/* until further notice, this must match exactly once */
		theNode->matchMax=1;
		theNode->nextExpression=NULL;					/* this is now the last expression */
		theNode->pt.theRegister=theRegister;			/* set pointer to the register that needs to be matched */
		if(currentNode)
			{
			currentNode->nextExpression=theNode;		/* link to previous node if one existed */
			}
		return(theNode);
		}
	return(NULL);
}

static EXPRESSIONNODE *REAddStartOfLineMatch(EXPRESSIONNODE *theParent,EXPRESSIONNODE *currentNode)
/* link in a new node that matches the start of a line
 * if there is a problem, SetError, and return NULL
 */
{
	EXPRESSIONNODE
		*theNode;

	if(theNode=REAllocateNode())
		{
		theNode->theType=PT_STARTOFLINE;				/* create a node, and apply the correct type to it */
		theNode->parentExpression=theParent;			/* set the parent */
		theNode->matchSpecified=FALSE;					/* no match count specified for this node yet */
		theNode->matchMin=1;							/* until further notice, this must match exactly once */
		theNode->matchMax=1;
		theNode->nextExpression=NULL;					/* this is now the last expression */
		if(currentNode)
			{
			currentNode->nextExpression=theNode;		/* link to previous node if one existed */
			}
		return(theNode);
		}
	return(NULL);
}

static EXPRESSIONNODE *REAddEndOfLineMatch(EXPRESSIONNODE *theParent,EXPRESSIONNODE *currentNode)
/* link in a new node that matches the end of a line
 * if there is a problem, SetError, and return NULL
 */
{
	EXPRESSIONNODE
		*theNode;

	if(theNode=REAllocateNode())
		{
		theNode->theType=PT_ENDOFLINE;					/* create a node, and apply the correct type to it */
		theNode->parentExpression=theParent;			/* set the parent */
		theNode->matchSpecified=FALSE;					/* no match count specified for this node yet */
		theNode->matchMin=1;							/* until further notice, this must match exactly once */
		theNode->matchMax=1;
		theNode->nextExpression=NULL;					/* this is now the last expression */
		if(currentNode)
			{
			currentNode->nextExpression=theNode;		/* link to previous node if one existed */
			}
		return(theNode);
		}
	return(NULL);
}

static void REInhaleDigits(UINT8 *theString,UINT32 *index,UINT32 numBytes,UINT32 *theResult)
/* inhale as many digits as possible from theString, starting at index, and possibly reading up to
 * offset numBytes
 * return theResult
 * also return with index updated
 * NOTE: it is ok for there to be no digits during the read, in this case
 * theResult will return as 0, and index will not be changed
 */
{
	UINT8
		theChar;

	*theResult=0;
	while(*index<numBytes&&(theChar=theString[*index])&&theChar>='0'&&theChar<='9')
		{
		*theResult=*theResult*10+(theChar-'0');			/* add this digit to the result */
		(*index)++;
		}
}

static BOOLEAN REParseRange(UINT8 *theExpression,UINT32 *index,UINT32 numBytes,UINT32 *rangeLow,UINT32 *rangeHigh)
/* repetition range operator has started, read in the ranges, and return them in
 * rangeLow, and rangeHigh
 * if there is a problem, SetError, and return FALSE
 */
{
	*rangeHigh=0;											/* this means infinity */
	REInhaleDigits(theExpression,index,numBytes,rangeLow);	/* attempt to read the low number in the range */
	if(*index<numBytes)										/* see if any more characters (looking for , or }) */
		{
		switch(theExpression[(*index)++])
			{
			case ',':
				REInhaleDigits(theExpression,index,numBytes,rangeHigh);	/* attempt to read the high number in the range */
				if(*index<numBytes&&theExpression[(*index)++]=='}')		/* see if any more characters (looking for }) */
					{
					if((*rangeHigh==0)||(*rangeHigh>=*rangeLow))			/* make sure low is <= high */
						{
						return(TRUE);
						}
					}
				SetError(localErrorFamily,errorMembers[BADRANGEEXPRESSION],errorDescriptions[BADRANGEEXPRESSION]);
				break;
			case '}':
				if(*rangeHigh=*rangeLow)					/* range with just one number means exactly that many times (0 is invalid) */
					{
					return(TRUE);
					}
				else
					{
					SetError(localErrorFamily,errorMembers[BADRANGEEXPRESSION],errorDescriptions[BADRANGEEXPRESSION]);
					}
				break;
			default:
				SetError(localErrorFamily,errorMembers[BADRANGEEXPRESSION],errorDescriptions[BADRANGEEXPRESSION]);
				break;
			}
		}
	else
		{
		SetError(localErrorFamily,errorMembers[BADRANGEEXPRESSION],errorDescriptions[BADRANGEEXPRESSION]);
		}
	return(FALSE);
}

/* this is called recursively from below, so we must prototype it */
static EXPRESSIONLISTHEADER *REParseSubExpressionList(COMPILEDEXPRESSION *theCompiledExpression,EXPRESSIONNODE *theParent,UINT8 *theExpression,UINT32 *index,UINT32 numBytes,UINT32 currentRegister,UINT8 *translateTable);	/* prototype this to keep compiler happy */

static EXPRESSIONNODE *REAddSubExpression(COMPILEDEXPRESSION *theCompiledExpression,EXPRESSIONNODE *theParent,EXPRESSIONNODE *currentNode,UINT8 *theExpression,UINT32 *index,UINT32 numBytes,UINT32 currentRegister,UINT8 *translateTable)
/* create a subexpression node, and build the subexpression into it
 * if there is a problem, SetError, and return NULL
 */
{
	EXPRESSIONNODE
		*startNode,
		*endNode;

	if(startNode=REAllocateNode())
		{
		startNode->theType=PT_SUBEXPRESSIONSTART;			/* create a start node, and apply the correct type to it */
		startNode->parentExpression=theParent;
		startNode->matchSpecified=FALSE;					/* no match count specified for this node yet (none ever will be) */
		startNode->matchMin=1;								/* set to some default values (never looked at) */
		startNode->matchMax=1;
		if(startNode->nextExpression=endNode=REAllocateNode())
			{
			endNode->theType=PT_SUBEXPRESSIONEND;			/* create a node, and apply the correct type to it */
			endNode->parentExpression=theParent;
			endNode->matchSpecified=FALSE;					/* no match count specified for this node yet */
			endNode->matchMin=1;							/* until further notice, this must match exactly once */
			endNode->matchMax=1;
			endNode->nextExpression=NULL;					/* this is now the last expression */
			if(currentRegister<MAXREGISTERS)				/* see if there is a register */
				{
				endNode->pt.se.theRegister=&(theCompiledExpression->theRegisters[currentRegister]);		/* remember the register */
				}
			else
				{
				endNode->pt.se.theRegister=NULL;			/* no register for this one */
				}
			if(endNode->pt.se.subExpressionList=REParseSubExpressionList(theCompiledExpression,endNode,theExpression,index,numBytes,currentRegister,translateTable))		/* get subexpression */
				{
				if(currentNode)
					{
					currentNode->nextExpression=startNode;		/* link to previous node if one existed */
					}
				return(startNode);
				}
			REFreeNode(endNode);
			}
		REFreeNode(startNode);
		}
	return(NULL);
}

static void REFreeLists(EXPRESSIONLISTHEADER *theList)
/* given a pointer to the top of an expression list, free
 * all subnodes, and all alternates below
 * then free theList
 * NOTE: this calls itself recursively for sublists
 */
{
	EXPRESSIONLISTHEADER
		*currentList,
		*nextAlternate;
	EXPRESSIONNODE
		*currentNode,
		*nextNode;

	currentList=theList;
	while(currentList)
		{
		nextAlternate=currentList->nextAlternate;
		currentNode=currentList->firstExpression;
		while(currentNode)
			{
			nextNode=currentNode->nextExpression;
			if(currentNode->theType==PT_SUBEXPRESSIONEND)		/* free sub-list */
				{
				REFreeLists(currentNode->pt.se.subExpressionList);		/* free all the nodes of this subexpression */
				}
			REFreeNode(currentNode);
			currentNode=nextNode;								/* move to the next horizontally */
			}
		REFreeList(currentList);
		currentList=nextAlternate;
		}
}

static EXPRESSIONLISTHEADER *REParseSubExpressionList(COMPILEDEXPRESSION *theCompiledExpression,EXPRESSIONNODE *theParent,UINT8 *theExpression,UINT32 *index,UINT32 numBytes,UINT32 currentRegister,UINT8 *translateTable)
/* parse theExpression, and build lists of nodes for theCompiledExpression
 * (if theCompiledExpression is NULL, then it is assumed that we are compiling a sublist)
 * return the head of the created list of expression nodes
 * if there is a problem, SetError, and return NULL
 */
{
	EXPRESSIONLISTHEADER
		*startList,
		*currentList;						/* current list in the list of alternates */
	EXPRESSIONNODE
		*startNode,
		*currentNode;
	UINT8
		theChar;
	UINT32
		theRegister;
	BOOLEAN
		done,
		fail;

	fail=done=FALSE;
	if(startList=REAllocateList())
		{
		startList->firstExpression=NULL;
		startList->nextAlternate=NULL;
		currentList=startList;							/* point to the current list that is being built */
		startNode=currentNode=NULL;
		while(!fail&&!done&&((*index)<numBytes))
			{
			switch(theChar=theExpression[(*index)++])
				{
				case '(':
					theCompiledExpression->numGroups++;	/* increment the number of groups */
					fail=!((currentNode=REAddSubExpression(theCompiledExpression,theParent,currentNode,theExpression,index,numBytes,theCompiledExpression->numGroups-1,translateTable)));
					break;
				case ')':
					if(theParent)						/* if not at the top, then take this as the end of our group */
						{
						if(currentRegister<MAXREGISTERS)	/* define this register now */
							{
							theCompiledExpression->registerDefined[currentRegister]=TRUE;
							}
						done=TRUE;
						}
					else
						{
						SetError(localErrorFamily,errorMembers[UNBALANCEDPAREN],errorDescriptions[UNBALANCEDPAREN]);
						fail=TRUE;
						}
					break;
				case '{':
					if(currentNode&&!currentNode->matchSpecified&&currentNode->theType!=PT_STARTOFLINE&&currentNode->theType!=PT_ENDOFLINE)
						{
						currentNode->matchSpecified=TRUE;	/* match has been specified */
						fail=!REParseRange(theExpression,index,numBytes,&currentNode->matchMin,&currentNode->matchMax);
						}
					else
						{
						SetError(localErrorFamily,errorMembers[BADREPETITION],errorDescriptions[BADREPETITION]);
						fail=TRUE;
						}
					break;
				case '*':								/* match 0 or more */
					if(currentNode&&!currentNode->matchSpecified&&currentNode->theType!=PT_STARTOFLINE&&currentNode->theType!=PT_ENDOFLINE)
						{
						currentNode->matchSpecified=TRUE;	/* match has been specified */
						currentNode->matchMin=0;		/* no requirement to match any */
						currentNode->matchMax=0;		/* match as many as possible */
						}
					else
						{
						SetError(localErrorFamily,errorMembers[BADREPETITION],errorDescriptions[BADREPETITION]);
						fail=TRUE;
						}
					break;
				case '+':								/* match 1 or more */
					if(currentNode&&!currentNode->matchSpecified&&currentNode->theType!=PT_STARTOFLINE&&currentNode->theType!=PT_ENDOFLINE)
						{
						currentNode->matchSpecified=TRUE;	/* match has been specified */
						currentNode->matchMin=1;		/* match at least 1 */
						currentNode->matchMax=0;		/* match as many as possible */
						}
					else
						{
						SetError(localErrorFamily,errorMembers[BADREPETITION],errorDescriptions[BADREPETITION]);
						fail=TRUE;
						}
					break;
				case '?':								/* match 0 or 1 */
					if(currentNode&&!currentNode->matchSpecified&&currentNode->theType!=PT_STARTOFLINE&&currentNode->theType!=PT_ENDOFLINE)
						{
						currentNode->matchSpecified=TRUE;	/* match has been specified */
						currentNode->matchMin=0;		/* no requirement to match any */
						currentNode->matchMax=1;		/* match at most 1 */
						}
					else
						{
						SetError(localErrorFamily,errorMembers[BADREPETITION],errorDescriptions[BADREPETITION]);
						fail=TRUE;
						}
					break;
				case '[':
					fail=!(currentNode=REAddList(theParent,currentNode,theExpression,index,numBytes,translateTable));
					break;
				case '|':
					if(currentList->firstExpression)		/* make sure there is something in the current list before moving on */
						{
						if(currentList=currentList->nextAlternate=REAllocateList())	/* create a new list, link it to the old, and make it current */
							{
							currentList->nextAlternate=NULL;
							currentList->firstExpression=NULL;
							currentNode=NULL;				/* no longer in the middle of a node */
							}
						else
							{
							fail=TRUE;
							}
						}
					else
						{
						SetError(localErrorFamily,errorMembers[EMPTYEXPRESSION],errorDescriptions[EMPTYEXPRESSION]);
						fail=TRUE;
						}
					break;
				case '.':
					fail=!(currentNode=REAddAny(theParent,currentNode,theExpression,index,numBytes));
					break;
				case '^':
					fail=!(currentNode=REAddStartOfLineMatch(theParent,currentNode));
					break;
				case '$':
					fail=!(currentNode=REAddEndOfLineMatch(theParent,currentNode));
					break;
				case '\\':
					if(*index<=numBytes)
						{
						theChar=theExpression[*index];
						if(theChar>='0'&&theChar<='9')									/* see if sub pattern match */
							{
							(*index)++;
							theRegister=theChar-'0';									/* make ASCII digit into binary number */
							if(theRegister<MAXREGISTERS&&theCompiledExpression->registerDefined[theRegister])	/* make sure it is in range, and that the given register has already been defined (this will guarantee that it is matched before we need it) */
								{
								fail=!(currentNode=REAddPreviousMatch(theParent,currentNode,&theCompiledExpression->theRegisters[theRegister]));
								}
							else
								{
								SetError(localErrorFamily,errorMembers[INVALIDREGISTER],errorDescriptions[INVALIDREGISTER]);
								fail=TRUE;
								}
							}
						else
							{
							if(REConvertQuotedCharacter(theExpression,index,numBytes,&theChar))
								{
								fail=!(currentNode=REAddString(theParent,currentNode,theChar,translateTable));
								}
							else
								{
								fail=TRUE;
								}
							}
						}
					else
						{
						SetError(localErrorFamily,errorMembers[DANGLINGQUOTE],errorDescriptions[DANGLINGQUOTE]);
						fail=TRUE;
						}
					break;
				default:
					fail=!(currentNode=REAddString(theParent,currentNode,theChar,translateTable));	/* just take the character as part of a string */
					break;
				}
			if(currentNode)
				{
				if(!currentList->firstExpression)
					{
					currentList->firstExpression=currentNode;
					}
				if(currentNode->theType==PT_SUBEXPRESSIONSTART)							/* if just had start of expression, move to the end */
					{
					currentNode=currentNode->nextExpression;
					}
				}
			}
		if(!fail&&!done&&theParent)														/* reached the end, but in middle of group somewhere */
			{
			SetError(localErrorFamily,errorMembers[UNBALANCEDPAREN],errorDescriptions[UNBALANCEDPAREN]);
			fail=TRUE;
			}
		if(!fail)
			{
			if(currentList->firstExpression)
				{
				return(startList);
				}
			else
				{
				SetError(localErrorFamily,errorMembers[EMPTYEXPRESSION],errorDescriptions[EMPTYEXPRESSION]);
				}
			}
		REFreeLists(startList);
		}
	return(NULL);
}

void REFree(COMPILEDEXPRESSION *theExpression)
/* free all nodes used by theExpression
 * this must be called for every call to RECompile to deallocate the
 * memory used by theExpression
 */
{
	REFreeLists(theExpression->firstList);											/* free the lists */
	MDisposePtr(theExpression);														/* remove the expression header */
}

static void MoveOver(EXPRESSIONLISTHEADER *theList,UINT32 depth)
/* move over by depth spaces
 */
{
	UINT32
		i;

	for(i=0;i<depth;i++)
		{
		printf(" ");
		}
}

static void PrintNode(EXPRESSIONLISTHEADER *theList,EXPRESSIONNODE *theNode,UINT32 depth)
/* print formatted node information
 */
{
	UINT32
		i;

	switch(theNode->theType)
		{
		case PT_STRING:
			MoveOver(theList,depth);
			printf("STRING %d/%d '",theNode->matchMin,theNode->matchMax);
			for(i=0;i<theNode->dataLength;i++)
				{
				printf("%c",theNode->pt.theData[i]);
				}
			printf("'\n");
			break;
		case PT_LIST:
			MoveOver(theList,depth);
			printf("LIST %d/%d ",theNode->matchMin,theNode->matchMax);
			for(i=0;i<32;i++)
				{
				printf("%02X ",theNode->pt.theBits[i]);
				}
			printf("\n");
			break;
		case PT_PREVIOUSMATCH:
			MoveOver(theList,depth);
			printf("PREVIOUSMATCH %d/%d %X\n",theNode->matchMin,theNode->matchMax,theNode->pt.theRegister);
			break;
		case PT_SUBEXPRESSIONSTART:
			MoveOver(theList,depth);
			printf("SUBEXPRESSIONSTART %d/%d\n",theNode->matchMin,theNode->matchMax);
			break;
		case PT_SUBEXPRESSIONEND:
			MoveOver(theList,depth);
			printf("SUBEXPRESSIONEND %d/%d\n",theNode->matchMin,theNode->matchMax);
			break;
		case PT_STARTOFLINE:
			MoveOver(theList,depth);
			printf("STARTOFLINE %d/%d\n",theNode->matchMin,theNode->matchMax);
			break;
		case PT_ENDOFLINE:
			MoveOver(theList,depth);
			printf("ENDOFLINE %d/%d\n",theNode->matchMin,theNode->matchMax);
			break;
		default:
			MoveOver(theList,depth);
			printf("????? (%d)\n",theNode->theType);
			break;
		}
}

void REPrintExpressionList(EXPRESSIONLISTHEADER *theList,UINT32 depth)
/* dump debugging output of the compiled expression
 * NOTE: this calls itself
 */
{
	EXPRESSIONLISTHEADER
		*currentList;
	EXPRESSIONNODE
		*currentNode;

	currentList=theList;
	while(currentList)
		{
		currentNode=currentList->firstExpression;
		while(currentNode)
			{
			if(currentNode->theType==PT_SUBEXPRESSIONEND)
				{
				REPrintExpressionList(currentNode->pt.se.subExpressionList,depth+1);		/* print the sublist */
				}
			PrintNode(currentList,currentNode,depth);
			currentNode=currentNode->nextExpression;
			}
		currentList=currentList->nextAlternate;
		}
}

void REPrintCompiledExpression(COMPILEDEXPRESSION *theExpression)
/* dump debugging output of the compiled expression
 */
{
	REPrintExpressionList(theExpression->firstList,0);
}

COMPILEDEXPRESSION *RECompile(UINT8 *theExpression,UINT32 numBytes,UINT8 *translateTable)
/* compile numBytes of theExpression, and return a pointer to the
 * node of the start of the expression.
 * if there is a problem, call SetError, and return NULL
 */
{
	COMPILEDEXPRESSION
		*theCompiledExpression;
	UINT32
		i,
		index;

	if(theCompiledExpression=(COMPILEDEXPRESSION *)MNewPtr(sizeof(COMPILEDEXPRESSION)))
		{
		theCompiledExpression->numGroups=0;							/* no groups yet */
		for(i=0;i<MAXREGISTERS;i++)
			{
			theCompiledExpression->registerDefined[i]=FALSE;		/* no registers have been defined yet */
			}
		index=0;
		if(theCompiledExpression->firstList=REParseSubExpressionList(theCompiledExpression,NULL,theExpression,&index,numBytes,0,translateTable))
			{
			return(theCompiledExpression);
			}
		MDisposePtr(theCompiledExpression);
		}
	return(NULL);
}

/* MATCHING FUNCTIONS -----------------------------------------------------------------------------------------------------------------
 */

static UINT32
	matchRecursionDepth;					/* used to keep track of how far we are recursing during match attempts, allows us to fail without running out of stack */

static EXPRESSIONNODE *REGetNextNode(EXPRESSIONNODE *theNode)
/* get the next expression node at this level, or move up a level if possible
 */
{
	EXPRESSIONNODE
		*newNode;

	if(newNode=theNode->nextExpression)
		{
		return(newNode);
		}
	newNode=theNode->parentExpression;
	return(newNode);
}

/* these are called recursively from below, so they must be prototyped */
static BOOLEAN REMatchAlternatives(CHUNKHEADER *theChunk,UINT32 theOffset,UINT32 numToSearch,EXPRESSIONLISTHEADER *theList,BOOLEAN *foundMatch,UINT32 *numMatched,RECURSEGLOBALS *recurseGlobals);
static BOOLEAN REMatchStringNode(CHUNKHEADER *theChunk,UINT32 theOffset,UINT32 numToSearch,EXPRESSIONNODE *theNode,BOOLEAN *foundMatch,UINT32 *numMatched,RECURSEGLOBALS *recurseGlobals);
static BOOLEAN REMatchStringNodeTT(CHUNKHEADER *theChunk,UINT32 theOffset,UINT32 numToSearch,EXPRESSIONNODE *theNode,BOOLEAN *foundMatch,UINT32 *numMatched,RECURSEGLOBALS *recurseGlobals);
static BOOLEAN REMatchListNode(CHUNKHEADER *theChunk,UINT32 theOffset,UINT32 numToSearch,EXPRESSIONNODE *theNode,BOOLEAN *foundMatch,UINT32 *numMatched,RECURSEGLOBALS *recurseGlobals);
static BOOLEAN REMatchListNodeTT(CHUNKHEADER *theChunk,UINT32 theOffset,UINT32 numToSearch,EXPRESSIONNODE *theNode,BOOLEAN *foundMatch,UINT32 *numMatched,RECURSEGLOBALS *recurseGlobals);
static BOOLEAN REMatchPreviousNode(CHUNKHEADER *theChunk,UINT32 theOffset,UINT32 numToSearch,EXPRESSIONNODE *theNode,BOOLEAN *foundMatch,UINT32 *numMatched,RECURSEGLOBALS *recurseGlobals);
static BOOLEAN REMatchPreviousNodeTT(CHUNKHEADER *theChunk,UINT32 theOffset,UINT32 numToSearch,EXPRESSIONNODE *theNode,BOOLEAN *foundMatch,UINT32 *numMatched,RECURSEGLOBALS *recurseGlobals);
static BOOLEAN REMatchSubStartNode(CHUNKHEADER *theChunk,UINT32 theOffset,UINT32 numToSearch,EXPRESSIONNODE *theNode,BOOLEAN *foundMatch,UINT32 *numMatched,RECURSEGLOBALS *recurseGlobals);
static BOOLEAN REMatchSubEndNode(CHUNKHEADER *theChunk,UINT32 theOffset,UINT32 numToSearch,EXPRESSIONNODE *theNode,BOOLEAN *foundMatch,UINT32 *numMatched,RECURSEGLOBALS *recurseGlobals);
static BOOLEAN REMatchStartLineNode(CHUNKHEADER *theChunk,UINT32 theOffset,UINT32 numToSearch,EXPRESSIONNODE *theNode,BOOLEAN *foundMatch,UINT32 *numMatched,RECURSEGLOBALS *recurseGlobals);
static BOOLEAN REMatchEndLineNode(CHUNKHEADER *theChunk,UINT32 theOffset,UINT32 numToSearch,EXPRESSIONNODE *theNode,BOOLEAN *foundMatch,UINT32 *numMatched,RECURSEGLOBALS *recurseGlobals);

/* array of pointers to exact matching routines */
static BOOLEAN
	(*exactRoutines[])(CHUNKHEADER *,UINT32,UINT32,EXPRESSIONNODE *,BOOLEAN *,UINT32 *,RECURSEGLOBALS *)=
		{
		REMatchStringNode,
		REMatchListNode,
		REMatchPreviousNode,
		REMatchSubStartNode,
		REMatchSubEndNode,
		REMatchStartLineNode,
		REMatchEndLineNode
		};

/* array of pointers to translated matching routines */
static BOOLEAN
	(*translatedRoutines[])(CHUNKHEADER *,UINT32,UINT32,EXPRESSIONNODE *,BOOLEAN *,UINT32 *,RECURSEGLOBALS *)=
		{
		REMatchStringNodeTT,
		REMatchListNodeTT,
		REMatchPreviousNodeTT,
		REMatchSubStartNode,
		REMatchSubEndNode,
		REMatchStartLineNode,
		REMatchEndLineNode
		};

/* this points to the array that tells what routines to call for each node type */
static BOOLEAN
	(**matchRoutines)(CHUNKHEADER *,UINT32,UINT32,EXPRESSIONNODE *,BOOLEAN *,UINT32 *,RECURSEGLOBALS *);

/* points to the translation table to use during translated compares */
static UINT8
	*theTranslationTable;

/* macro function that branches to next match routine from current one */
#define	RECallNextNode(theChunk,theOffset,numToSearch,theNode,foundMatch,numMatched,recurseGlobals)						\
{																														\
	fail=!matchRoutines[theNode->theType](theChunk,theOffset,numToSearch,theNode,foundMatch,numMatched,recurseGlobals);	\
}

static BOOLEAN REMatchStringNode(CHUNKHEADER *theChunk,UINT32 theOffset,UINT32 numToSearch,EXPRESSIONNODE *theNode,BOOLEAN *foundMatch,UINT32 *numMatched,RECURSEGLOBALS *recurseGlobals)
/* attempt to match theNode with the data at theChunk/theOffset
 * if the pattern could be matched, return TRUE in foundMatch with numMatched as the number of bytes matched
 * if no match was located, return FALSE in foundMatch
 * NOTE: do not search through more than numToSearch bytes
 * endChunk/endOffset will point to the end of the match if one was located
 * if there is some sort of hard failure, this routine will SetError, and return FALSE
 */
{
	CHUNKHEADER
		*lastChunk;												/* need to remember this in case we fall off the edge */
	UINT32
		i,
		nodeMatched;
	BOOLEAN
		fail,
		matching;
	UINT8
		theChar;
	EXPRESSIONNODE
		*nextExpression;

	if(++matchRecursionDepth<MAXRECURSIONDEPTH)
		{
		fail=FALSE;
		matching=(theNode->dataLength-1<=numToSearch);			/* make sure there are enough bytes to even allow a match */
		for(i=0;matching&&(i<theNode->dataLength-1);i++)		/* match the constant part of the string (if one exists) */
			{
			matching=(theChunk->data[theOffset]==theNode->pt.theData[i]);	/* check for match */
			if((++theOffset)>=theChunk->totalBytes)				/* move forward in the input */
				{
				lastChunk=theChunk;
				theChunk=theChunk->nextHeader;
				theOffset=0;
				}
			}
		if(matching)											/* make sure we made it through the first part */
			{
			numToSearch-=(theNode->dataLength-1);				/* matched the constant part of the string, now try to get up to matchMax of last character */
			theChar=theNode->pt.theData[theNode->dataLength-1];
			i=0;
			while(matching&&(i<numToSearch)&&((!theNode->matchMax)||(i<theNode->matchMax)))	/* read as many as possible until we hit max, or fail */
				{
				if(matching=(theChunk->data[theOffset]==theChar))	/* check for match */
					{
					if((++theOffset)>=theChunk->totalBytes)			/* move forward in the input iff there was a match */
						{
						lastChunk=theChunk;
						theChunk=theChunk->nextHeader;
						theOffset=0;
						}
					i++;
					}
				}
			if(i>=theNode->matchMin)							/* see if we got the minimum number of matches */
				{
				if(nextExpression=REGetNextNode(theNode))		/* see if there is another node after this one (if so, recurse into it) */
					{
					RECallNextNode(theChunk,theOffset,numToSearch-i,nextExpression,&matching,&nodeMatched,recurseGlobals);
					while(!fail&&!matching&&i>theNode->matchMin)	/* see if possible to backup and try again */
						{
						i--;
						if(theChunk)							/* back up 1 character */
							{
							if(theOffset)
								{
								theOffset--;
								}
							else
								{
								theChunk=theChunk->previousHeader;
								theOffset=theChunk->totalBytes-1;
								}
							}
						else
							{
							theChunk=lastChunk;
							theOffset=theChunk->totalBytes-1;
							}
						RECallNextNode(theChunk,theOffset,numToSearch-i,nextExpression,&matching,&nodeMatched,recurseGlobals);
						}
					if(!fail&&matching)							/* match was located!! */
						{
						(*numMatched)=theNode->dataLength-1+i+nodeMatched;		/* return number of characters that matched */
						}
					}
				else											/* no other nodes, this was at the end, so it is done */
					{
					(*numMatched)=theNode->dataLength-1+i;		/* return number of characters that matched */
					recurseGlobals->endChunk=theChunk;
					recurseGlobals->endOffset=theOffset;
					matching=TRUE;								/* if no more nodes, then this list is complete, and it matched */
					}
				}
			else
				{
				matching=FALSE;									/* did not find minimum number of matches, so this fails here and now */
				}
			}
		(*foundMatch)=matching;
		matchRecursionDepth--;
		}
	else
		{
		SetError(localErrorFamily,errorMembers[STACKOVERFLOW],errorDescriptions[STACKOVERFLOW]);
		fail=TRUE;
		}
	return(!fail);
}

static BOOLEAN REMatchStringNodeTT(CHUNKHEADER *theChunk,UINT32 theOffset,UINT32 numToSearch,EXPRESSIONNODE *theNode,BOOLEAN *foundMatch,UINT32 *numMatched,RECURSEGLOBALS *recurseGlobals)
/* Exactly like REMatchStringNode except that this one uses a translation table during the search
 */
{
	CHUNKHEADER
		*lastChunk;												/* need to remember this in case we fall off the edge */
	UINT32
		i,
		nodeMatched;
	BOOLEAN
		fail,
		matching;
	UINT8
		theChar;
	EXPRESSIONNODE
		*nextExpression;

	if(++matchRecursionDepth<MAXRECURSIONDEPTH)
		{
		fail=FALSE;
		matching=(theNode->dataLength-1<=numToSearch);			/* make sure there are enough bytes to even allow a match */
		for(i=0;matching&&(i<theNode->dataLength-1);i++)		/* match the constant part of the string (if one exists) */
			{
			matching=(theTranslationTable[theChunk->data[theOffset]]==theTranslationTable[theNode->pt.theData[i]]);	/* check for match */
			if((++theOffset)>=theChunk->totalBytes)				/* move forward in the input */
				{
				lastChunk=theChunk;
				theChunk=theChunk->nextHeader;
				theOffset=0;
				}
			}
		if(matching)											/* make sure we made it through the first part */
			{
			numToSearch-=(theNode->dataLength-1);				/* matched the constant part of the string, now try to get up to matchMax of last character */
			theChar=theTranslationTable[theNode->pt.theData[theNode->dataLength-1]];
			i=0;
			while(matching&&(i<numToSearch)&&((!theNode->matchMax)||(i<theNode->matchMax)))	/* read as many as possible until we hit max, or fail */
				{
				if(matching=(theTranslationTable[theChunk->data[theOffset]]==theChar))	/* check for match */
					{
					if((++theOffset)>=theChunk->totalBytes)			/* move forward in the input iff there was a match */
						{
						lastChunk=theChunk;
						theChunk=theChunk->nextHeader;
						theOffset=0;
						}
					i++;
					}
				}
			if(i>=theNode->matchMin)							/* see if we got the minimum number of matches */
				{
				if(nextExpression=REGetNextNode(theNode))		/* see if there is another node after this one (if so, recurse into it) */
					{
					RECallNextNode(theChunk,theOffset,numToSearch-i,nextExpression,&matching,&nodeMatched,recurseGlobals);
					while(!fail&&!matching&&i>theNode->matchMin)	/* see if possible to backup and try again */
						{
						i--;
						if(theChunk)							/* back up 1 character */
							{
							if(theOffset)
								{
								theOffset--;
								}
							else
								{
								theChunk=theChunk->previousHeader;
								theOffset=theChunk->totalBytes-1;
								}
							}
						else
							{
							theChunk=lastChunk;
							theOffset=theChunk->totalBytes-1;
							}
						RECallNextNode(theChunk,theOffset,numToSearch-i,nextExpression,&matching,&nodeMatched,recurseGlobals);
						}
					if(!fail&&matching)							/* match was located!! */
						{
						(*numMatched)=theNode->dataLength-1+i+nodeMatched;		/* return number of characters that matched */
						}
					}
				else											/* no other nodes, this was at the end, so it is done */
					{
					(*numMatched)=theNode->dataLength-1+i;		/* return number of characters that matched */
					recurseGlobals->endChunk=theChunk;
					recurseGlobals->endOffset=theOffset;
					matching=TRUE;								/* if no more nodes, then this list is complete, and it matched */
					}
				}
			else
				{
				matching=FALSE;									/* did not find minimum number of matches, so this fails here and now */
				}
			}
		(*foundMatch)=matching;
		matchRecursionDepth--;
		}
	else
		{
		SetError(localErrorFamily,errorMembers[STACKOVERFLOW],errorDescriptions[STACKOVERFLOW]);
		fail=TRUE;
		}
	return(!fail);
}

static BOOLEAN REMatchListNode(CHUNKHEADER *theChunk,UINT32 theOffset,UINT32 numToSearch,EXPRESSIONNODE *theNode,BOOLEAN *foundMatch,UINT32 *numMatched,RECURSEGLOBALS *recurseGlobals)
/* attempt to match theNode with the data at theChunk/theOffset
 * if the pattern could be matched, return TRUE in foundMatch with numMatched as the number of bytes matched
 * if no match was located, return FALSE in foundMatch
 * NOTE: do not search through more than numToSearch bytes
 * endChunk/endOffset will point to the end of the match if one was located
 * if there is some sort of hard failure, this routine will SetError, and return FALSE
 */
{
	CHUNKHEADER
		*lastChunk;												/* need to remember this in case we fall off the edge */
	UINT32
		i,
		nodeMatched;
	BOOLEAN
		fail,
		matching;
	UINT8
		theChar;
	EXPRESSIONNODE
		*nextExpression;

	if(++matchRecursionDepth<MAXRECURSIONDEPTH)
		{
		fail=FALSE;
		matching=TRUE;											/* start by assuming a match */
		i=0;
		while(matching&&(i<numToSearch)&&((!theNode->matchMax)||(i<theNode->matchMax)))	/* read as many as possible until we hit max, or fail */
			{
			theChar=theChunk->data[theOffset];
			if(matching=(GETCHARBIT(theNode->pt.theBits,theChar)))	/* check for match */
				{
				if((++theOffset)>=theChunk->totalBytes)			/* move forward in the input iff there was a match */
					{
					lastChunk=theChunk;
					theChunk=theChunk->nextHeader;
					theOffset=0;
					}
				i++;
				}
			}
		if(i>=theNode->matchMin)							/* see if we got the minimum number of matches */
			{
			if(nextExpression=REGetNextNode(theNode))		/* see if there is another node after this one (if so, recurse into it) */
				{
				RECallNextNode(theChunk,theOffset,numToSearch-i,nextExpression,&matching,&nodeMatched,recurseGlobals);
				while(!fail&&!matching&&i>theNode->matchMin)	/* see if possible to backup and try again */
					{
					i--;
					if(theChunk)							/* back up 1 character */
						{
						if(theOffset)
							{
							theOffset--;
							}
						else
							{
							theChunk=theChunk->previousHeader;
							theOffset=theChunk->totalBytes-1;
							}
						}
					else
						{
						theChunk=lastChunk;
						theOffset=theChunk->totalBytes-1;
						}
					RECallNextNode(theChunk,theOffset,numToSearch-i,nextExpression,&matching,&nodeMatched,recurseGlobals);
					}
				if(!fail&&matching)							/* match was located!! */
					{
					(*numMatched)=i+nodeMatched;			/* return number of characters that matched */
					}
				}
			else											/* no other nodes, this was at the end, so it is done */
				{
				(*numMatched)=i;							/* return number of characters that matched */
				recurseGlobals->endChunk=theChunk;
				recurseGlobals->endOffset=theOffset;
				matching=TRUE;								/* if no more nodes, then this list is complete, and it matched */
				}
			}
		else
			{
			matching=FALSE;									/* did not find minimum number of matches, so this fails here and now */
			}
		(*foundMatch)=matching;
		matchRecursionDepth--;
		}
	else
		{
		SetError(localErrorFamily,errorMembers[STACKOVERFLOW],errorDescriptions[STACKOVERFLOW]);
		fail=TRUE;
		}
	return(!fail);
}

static BOOLEAN REMatchListNodeTT(CHUNKHEADER *theChunk,UINT32 theOffset,UINT32 numToSearch,EXPRESSIONNODE *theNode,BOOLEAN *foundMatch,UINT32 *numMatched,RECURSEGLOBALS *recurseGlobals)
/* exactly like REMatchListNode but uses a translation table during the compare
 */
{
	CHUNKHEADER
		*lastChunk;												/* need to remember this in case we fall off the edge */
	UINT32
		i,
		nodeMatched;
	BOOLEAN
		fail,
		matching;
	UINT8
		theChar;
	EXPRESSIONNODE
		*nextExpression;

	if(++matchRecursionDepth<MAXRECURSIONDEPTH)
		{
		fail=FALSE;
		matching=TRUE;											/* start by assuming a match */
		i=0;
		while(matching&&(i<numToSearch)&&((!theNode->matchMax)||(i<theNode->matchMax)))	/* read as many as possible until we hit max, or fail */
			{
			theChar=theTranslationTable[theChunk->data[theOffset]];
			if(matching=(GETCHARBIT(theNode->pt.theBits,theChar)))	/* check for match */
				{
				if((++theOffset)>=theChunk->totalBytes)			/* move forward in the input iff there was a match */
					{
					lastChunk=theChunk;
					theChunk=theChunk->nextHeader;
					theOffset=0;
					}
				i++;
				}
			}
		if(i>=theNode->matchMin)							/* see if we got the minimum number of matches */
			{
			if(nextExpression=REGetNextNode(theNode))		/* see if there is another node after this one (if so, recurse into it) */
				{
				RECallNextNode(theChunk,theOffset,numToSearch-i,nextExpression,&matching,&nodeMatched,recurseGlobals);
				while(!fail&&!matching&&i>theNode->matchMin)	/* see if possible to backup and try again */
					{
					i--;
					if(theChunk)							/* back up 1 character */
						{
						if(theOffset)
							{
							theOffset--;
							}
						else
							{
							theChunk=theChunk->previousHeader;
							theOffset=theChunk->totalBytes-1;
							}
						}
					else
						{
						theChunk=lastChunk;
						theOffset=theChunk->totalBytes-1;
						}
					RECallNextNode(theChunk,theOffset,numToSearch-i,nextExpression,&matching,&nodeMatched,recurseGlobals);
					}
				if(!fail&&matching)							/* match was located!! */
					{
					(*numMatched)=i+nodeMatched;			/* return number of characters that matched */
					}
				}
			else											/* no other nodes, this was at the end, so it is done */
				{
				(*numMatched)=i;							/* return number of characters that matched */
				recurseGlobals->endChunk=theChunk;
				recurseGlobals->endOffset=theOffset;
				matching=TRUE;								/* if no more nodes, then this list is complete, and it matched */
				}
			}
		else
			{
			matching=FALSE;									/* did not find minimum number of matches, so this fails here and now */
			}
		(*foundMatch)=matching;
		matchRecursionDepth--;
		}
	else
		{
		SetError(localErrorFamily,errorMembers[STACKOVERFLOW],errorDescriptions[STACKOVERFLOW]);
		fail=TRUE;
		}
	return(!fail);
}

static BOOLEAN REMatchPreviousNode(CHUNKHEADER *theChunk,UINT32 theOffset,UINT32 numToSearch,EXPRESSIONNODE *theNode,BOOLEAN *foundMatch,UINT32 *numMatched,RECURSEGLOBALS *recurseGlobals)
/* attempt to match theNode with the data at theChunk/theOffset
 * if the pattern could be matched, return TRUE in foundMatch with numMatched as the number of bytes matched
 * if no match was located, return FALSE in foundMatch
 * NOTE: do not search through more than numToSearch bytes
 * endChunk/endOffset will point to the end of the match if one was located
 * if there is some sort of hard failure, this routine will SetError, and return FALSE
 */
{
	UINT32
		i,
		totalMatched,
		nodeMatched;
	CHUNKHEADER
		*lastChunk;
	UINT32
		lastOffset;
	BOOLEAN
		fail,
		matching;
	EXPRESSIONNODE
		*nextExpression;

	if(++matchRecursionDepth<MAXRECURSIONDEPTH)
		{
		fail=FALSE;
		matching=TRUE;											/* start by assuming a match */
		i=totalMatched=0;
		while(matching&&((!theNode->matchMax)||(i<theNode->matchMax)))	/* match as many as possible until we hit max, or fail */
			{
			if(theNode->pt.theRegister->theChunk&&numToSearch-totalMatched>=theNode->pt.theRegister->numBytes)	/* see if it is defined, and enough bytes to test, if not, matching fails */
				{
				if(matching=LiteralMatch(theChunk,theOffset,theNode->pt.theRegister->theChunk,theNode->pt.theRegister->theOffset,theNode->pt.theRegister->numBytes,&(recurseGlobals->endChunk),&(recurseGlobals->endOffset)))
					{
					i++;
					totalMatched+=theNode->pt.theRegister->numBytes;
					lastChunk=theChunk;							/* keep these in case we step off the end */
					lastOffset=theOffset;
					theChunk=recurseGlobals->endChunk;			/* move to new position */
					theOffset=recurseGlobals->endOffset;
					}
				}
			else
				{
				matching=FALSE;
				}
			}
		if(i>=theNode->matchMin)							/* see if we got the minimum number of matches */
			{
			if(nextExpression=REGetNextNode(theNode))		/* see if there is another node after this one (if so, recurse into it) */
				{
				RECallNextNode(theChunk,theOffset,numToSearch-totalMatched,nextExpression,&matching,&nodeMatched,recurseGlobals);
				while(!fail&&!matching&&i>theNode->matchMin)	/* see if possible to backup and try again */
					{
					i--;
					nodeMatched=theNode->pt.theRegister->numBytes;
					totalMatched-=nodeMatched;
					if(theChunk)							/* back up nodeMatched characters */
						{
						while(nodeMatched>theOffset)		/* move back through the chunks until we use up all of the bytes matched */
							{
							theChunk=theChunk->previousHeader;	/* step back to the previous chunk */
							nodeMatched-=theOffset;
							theOffset=theChunk->totalBytes;
							}
						theOffset-=nodeMatched;
						}
					else
						{
						theChunk=lastChunk;
						theOffset=lastOffset;
						}
					RECallNextNode(theChunk,theOffset,numToSearch-totalMatched,nextExpression,&matching,&nodeMatched,recurseGlobals);
					}
				if(!fail&&matching)							/* match was located!! */
					{
					(*numMatched)=totalMatched+nodeMatched;	/* return number of characters that matched */
					}
				}
			else											/* no other nodes, this was at the end, so it is done */
				{
				(*numMatched)=totalMatched;					/* return number of characters that matched */
				recurseGlobals->endChunk=theChunk;
				recurseGlobals->endOffset=theOffset;
				matching=TRUE;								/* if no more nodes, then this list is complete, and it matched */
				}
			}
		else
			{
			matching=FALSE;									/* did not find minimum number of matches, so this fails here and now */
			}
		(*foundMatch)=matching;
		matchRecursionDepth--;
		}
	else
		{
		SetError(localErrorFamily,errorMembers[STACKOVERFLOW],errorDescriptions[STACKOVERFLOW]);
		fail=TRUE;
		}
	return(!fail);
}

static BOOLEAN REMatchPreviousNodeTT(CHUNKHEADER *theChunk,UINT32 theOffset,UINT32 numToSearch,EXPRESSIONNODE *theNode,BOOLEAN *foundMatch,UINT32 *numMatched,RECURSEGLOBALS *recurseGlobals)
/* exactly like REMatchPreviousNode except that this one uses a translation table
 */
{
	UINT32
		i,
		totalMatched,
		nodeMatched;
	CHUNKHEADER
		*lastChunk;
	UINT32
		lastOffset;
	BOOLEAN
		fail,
		matching;
	EXPRESSIONNODE
		*nextExpression;

	if(++matchRecursionDepth<MAXRECURSIONDEPTH)
		{
		fail=FALSE;
		matching=TRUE;											/* start by assuming a match */
		i=totalMatched=0;
		while(matching&&((!theNode->matchMax)||(i<theNode->matchMax)))	/* match as many as possible until we hit max, or fail */
			{
			if(theNode->pt.theRegister->theChunk&&numToSearch-totalMatched>=theNode->pt.theRegister->numBytes)	/* see if it is defined, and enough bytes to test, if not, matching fails */
				{
				if(matching=LiteralMatchTT(theChunk,theOffset,theNode->pt.theRegister->theChunk,theNode->pt.theRegister->theOffset,theNode->pt.theRegister->numBytes,theTranslationTable,&(recurseGlobals->endChunk),&(recurseGlobals->endOffset)))
					{
					i++;
					totalMatched+=theNode->pt.theRegister->numBytes;
					lastChunk=theChunk;							/* keep these in case we step off the end */
					lastOffset=theOffset;
					theChunk=recurseGlobals->endChunk;			/* move to new position */
					theOffset=recurseGlobals->endOffset;
					}
				}
			else
				{
				matching=FALSE;
				}
			}
		if(i>=theNode->matchMin)							/* see if we got the minimum number of matches */
			{
			if(nextExpression=REGetNextNode(theNode))		/* see if there is another node after this one (if so, recurse into it) */
				{
				RECallNextNode(theChunk,theOffset,numToSearch-totalMatched,nextExpression,&matching,&nodeMatched,recurseGlobals);
				while(!fail&&!matching&&i>theNode->matchMin)	/* see if possible to backup and try again */
					{
					i--;
					nodeMatched=theNode->pt.theRegister->numBytes;
					totalMatched-=nodeMatched;
					if(theChunk)							/* back up nodeMatched characters */
						{
						while(nodeMatched>theOffset)		/* move back through the chunks until we use up all of the bytes matched */
							{
							theChunk=theChunk->previousHeader;	/* step back to the previous chunk */
							nodeMatched-=theOffset;
							theOffset=theChunk->totalBytes;
							}
						theOffset-=nodeMatched;
						}
					else
						{
						theChunk=lastChunk;
						theOffset=lastOffset;
						}
					RECallNextNode(theChunk,theOffset,numToSearch-totalMatched,nextExpression,&matching,&nodeMatched,recurseGlobals);
					}
				if(!fail&&matching)							/* match was located!! */
					{
					(*numMatched)=totalMatched+nodeMatched;	/* return number of characters that matched */
					}
				}
			else											/* no other nodes, this was at the end, so it is done */
				{
				(*numMatched)=totalMatched;					/* return number of characters that matched */
				recurseGlobals->endChunk=theChunk;
				recurseGlobals->endOffset=theOffset;
				matching=TRUE;								/* if no more nodes, then this list is complete, and it matched */
				}
			}
		else
			{
			matching=FALSE;									/* did not find minimum number of matches, so this fails here and now */
			}
		(*foundMatch)=matching;
		matchRecursionDepth--;
		}
	else
		{
		SetError(localErrorFamily,errorMembers[STACKOVERFLOW],errorDescriptions[STACKOVERFLOW]);
		fail=TRUE;
		}
	return(!fail);
}

static BOOLEAN REMatchSubStartNode(CHUNKHEADER *theChunk,UINT32 theOffset,UINT32 numToSearch,EXPRESSIONNODE *theNode,BOOLEAN *foundMatch,UINT32 *numMatched,RECURSEGLOBALS *recurseGlobals)
/* attempt to match theNode with the data at theChunk/theOffset
 * if the pattern could be matched, return TRUE in foundMatch with numMatched as the number of bytes matched
 * if no match was located, return FALSE in foundMatch
 * NOTE: do not search through more than numToSearch bytes
 * endChunk/endOffset will point to the end of the match if one was located
 * if there is some sort of hard failure, this routine will SetError, and return FALSE
 */
{
	UINT32
		oldCount;
	BOOLEAN
		fail;

	if(++matchRecursionDepth<MAXRECURSIONDEPTH)
		{
		fail=FALSE;
		theNode->nextExpression->pt.se.justEntered=TRUE;	/* set up the end of the subexpression so that it knows we just hit the start */
		oldCount=theNode->nextExpression->pt.se.currentCount;	/* keep track of how many times this expression matched last time */
		RECallNextNode(theChunk,theOffset,numToSearch,theNode->nextExpression,foundMatch,numMatched,recurseGlobals);
		theNode->nextExpression->pt.se.currentCount=oldCount;	/* restore match count */
		matchRecursionDepth--;
		}
	else
		{
		SetError(localErrorFamily,errorMembers[STACKOVERFLOW],errorDescriptions[STACKOVERFLOW]);
		fail=TRUE;
		}
	return(!fail);
}

static BOOLEAN REMatchSubEndNode(CHUNKHEADER *theChunk,UINT32 theOffset,UINT32 numToSearch,EXPRESSIONNODE *theNode,BOOLEAN *foundMatch,UINT32 *numMatched,RECURSEGLOBALS *recurseGlobals)
/* attempt to match theNode with the data at theChunk/theOffset
 * if the pattern could be matched, return TRUE in foundMatch with numMatched as the number of bytes matched
 * if no match was located, return FALSE in foundMatch
 * NOTE: do not search through more than numToSearch bytes
 * endChunk/endOffset will point to the end of the match if one was located
 * if there is some sort of hard failure, this routine will SetError, and return FALSE
 */
{
	BOOLEAN
		fail,
		matching;
	EXPRESSIONNODE
		*nextExpression;
	REGISTER
		oldRegister;
	CHUNKHEADER
		*oldLastChunk;									/* used to keep the contents of the old register in case we back up */
	UINT32
		oldLastOffset,
		oldLastNumToSearch;

	if(++matchRecursionDepth<MAXRECURSIONDEPTH)
		{
		if(theNode->pt.se.theRegister)			/* see if this is one that is actually remembered */
			{
			oldRegister=*theNode->pt.se.theRegister;
			oldLastChunk=theNode->pt.se.lastChunk;
			oldLastOffset=theNode->pt.se.lastOffset;
			oldLastNumToSearch=theNode->pt.se.lastNumToSearch;
			}
		fail=matching=FALSE;
		if(theNode->pt.se.justEntered)
			{
			theNode->pt.se.currentCount=0;					/* start off the counter */
			theNode->pt.se.lastChunk=theChunk;				/* remember where we were, so the register can be set if needed */
			theNode->pt.se.lastOffset=theOffset;
			theNode->pt.se.lastNumToSearch=numToSearch;
			theNode->pt.se.justEntered=FALSE;				/* next time this is seen, we will know it came from a successful completion! */
			if(REMatchAlternatives(theChunk,theOffset,numToSearch,theNode->pt.se.subExpressionList,&matching,numMatched,recurseGlobals))
				{
				if(!matching)								/* if could not match, then see if we can move forward from here */
					{
					if(!theNode->matchMin)					/* if no matches needed, then we have success */
						{
						if(nextExpression=REGetNextNode(theNode))		/* see if there is another node after this one (if so, recurse into it) */
							{
							RECallNextNode(theChunk,theOffset,numToSearch,nextExpression,&matching,numMatched,recurseGlobals);
							}
						else
							{
							matching=TRUE;
							*numMatched=0;					/* this node is done here and now */
							recurseGlobals->endChunk=theChunk;
							recurseGlobals->endOffset=theOffset;
							}
						}
					}
				}
			else
				{
				fail=TRUE;
				}
			}
		else
			{
			theNode->pt.se.currentCount++;					/* made another successful match */
			if(theNode->pt.se.theRegister)				/* see if this is one that is actually remembered */
				{
				theNode->pt.se.theRegister->theChunk=theNode->pt.se.lastChunk;
				theNode->pt.se.theRegister->theOffset=theNode->pt.se.lastOffset;
				theNode->pt.se.theRegister->numBytes=theNode->pt.se.lastNumToSearch-numToSearch;
				theNode->pt.se.lastChunk=theChunk;
				theNode->pt.se.lastOffset=theOffset;
				theNode->pt.se.lastNumToSearch=numToSearch;
				}
			if((!theNode->matchMax)||theNode->pt.se.currentCount<theNode->matchMax)
				{
				if(REMatchAlternatives(theChunk,theOffset,numToSearch,theNode->pt.se.subExpressionList,&matching,numMatched,recurseGlobals))
					{
					if(!matching)								/* if could not match, then see if we can move forward from here */
						{
						if(theNode->pt.se.currentCount>=theNode->matchMin)
							{
							if(nextExpression=REGetNextNode(theNode))		/* see if there is another node after this one (if so, recurse into it) */
								{
								RECallNextNode(theChunk,theOffset,numToSearch,nextExpression,&matching,numMatched,recurseGlobals);
								}
							else
								{
								matching=TRUE;
								*numMatched=0;					/* this node is done here and now */
								recurseGlobals->endChunk=theChunk;
								recurseGlobals->endOffset=theOffset;
								}
							}
						}
					}
				else
					{
					fail=TRUE;
					}
				}
			else
				{
				if(nextExpression=REGetNextNode(theNode))		/* see if there is another node after this one (if so, recurse into it) */
					{
					RECallNextNode(theChunk,theOffset,numToSearch,nextExpression,&matching,numMatched,recurseGlobals);
					}
				else
					{
					matching=TRUE;
					*numMatched=0;
					recurseGlobals->endChunk=theChunk;
					recurseGlobals->endOffset=theOffset;
					}
				}
			if(!matching)
				{
				theNode->pt.se.currentCount--;				/* this match failed, so back out */
				}
			}
		if(!matching)
			{
			if(theNode->pt.se.theRegister)					/* see if this is one that is actually remembered */
				{
				theNode->pt.se.lastChunk=oldLastChunk;
				theNode->pt.se.lastOffset=oldLastOffset;
				theNode->pt.se.lastNumToSearch=oldLastNumToSearch;
				*theNode->pt.se.theRegister=oldRegister;
				}
			}
		(*foundMatch)=matching;
		matchRecursionDepth--;
		}
	else
		{
		SetError(localErrorFamily,errorMembers[STACKOVERFLOW],errorDescriptions[STACKOVERFLOW]);
		fail=TRUE;
		}
	return(!fail);
}


static BOOLEAN REMatchStartLineNode(CHUNKHEADER *theChunk,UINT32 theOffset,UINT32 numToSearch,EXPRESSIONNODE *theNode,BOOLEAN *foundMatch,UINT32 *numMatched,RECURSEGLOBALS *recurseGlobals)
/* attempt to match theNode with the data at theChunk/theOffset
 * if the pattern could be matched, return TRUE in foundMatch with numMatched as the number of bytes matched
 * if no match was located, return FALSE in foundMatch
 * NOTE: do not search through more than numToSearch bytes
 * endChunk/endOffset will point to the end of the match if one was located
 * if there is some sort of hard failure, this routine will SetError, and return FALSE
 */
{
	CHUNKHEADER
		*lastChunk;												/* need to remember this in case we fall off the edge */
	UINT32
		lastOffset;
	BOOLEAN
		fail,
		matching;
	EXPRESSIONNODE
		*nextExpression;

	if(++matchRecursionDepth<MAXRECURSIONDEPTH)
		{
		fail=FALSE;
		matching=TRUE;										/* start by assuming a match */
		(*numMatched)=0;									/* return number of characters that matched */
		recurseGlobals->endChunk=lastChunk=theChunk;		/* return position of match */
		recurseGlobals->endOffset=lastOffset=theOffset;
		if(lastOffset)										/* back up from current position, and look for start of line */
			{
			lastOffset--;
			}
		else
			{
			if(lastChunk&&(lastChunk=lastChunk->previousHeader))
				{
				lastOffset=lastChunk->totalBytes-1;
				}
			}
		if(lastChunk)										/* if no chunk back, then we were at the start of text which is considered the start of a line */
			{
			if(lastChunk->data[lastOffset]!='\n')			/* see if not start of line */
				{
				if(!(recurseGlobals->leftEdge)||(numToSearch<recurseGlobals->initialNumToSearch))	/* make sure we have not been told that this is an edge */
					{
					matching=FALSE;
					}
				}
			}
		if(matching&&(nextExpression=REGetNextNode(theNode)))	/* see if there is another node after this one (if so, recurse into it) */
			{
			RECallNextNode(theChunk,theOffset,numToSearch,nextExpression,&matching,numMatched,recurseGlobals);
			}
		(*foundMatch)=matching;
		matchRecursionDepth--;
		}
	else
		{
		SetError(localErrorFamily,errorMembers[STACKOVERFLOW],errorDescriptions[STACKOVERFLOW]);
		fail=TRUE;
		}
	return(!fail);
}

static BOOLEAN REMatchEndLineNode(CHUNKHEADER *theChunk,UINT32 theOffset,UINT32 numToSearch,EXPRESSIONNODE *theNode,BOOLEAN *foundMatch,UINT32 *numMatched,RECURSEGLOBALS *recurseGlobals)
/* attempt to match theNode with the data at theChunk/theOffset
 * if the pattern could be matched, return TRUE in foundMatch with numMatched as the number of bytes matched
 * if no match was located, return FALSE in foundMatch
 * NOTE: do not search through more than numToSearch bytes
 * endChunk/endOffset will point to the end of the match if one was located
 * if there is some sort of hard failure, this routine will SetError, and return FALSE
 */
{
	BOOLEAN
		fail,
		matching;
	EXPRESSIONNODE
		*nextExpression;

	if(++matchRecursionDepth<MAXRECURSIONDEPTH)
		{
		fail=FALSE;
		matching=TRUE;										/* start by assuming a match */
		(*numMatched)=0;									/* return number of characters that matched */
		recurseGlobals->endChunk=theChunk;					/* return position of match */
		recurseGlobals->endOffset=theOffset;
		if(theChunk)										/* if no chunk, then we are at the end of text which is considered the end of a line */
			{
			if(theChunk->data[theOffset]!='\n')				/* see if not end of line */
				{
				if(!(recurseGlobals->rightEdge)||numToSearch)	/* make sure we have not been told that this is an edge */
					{
					matching=FALSE;
					}
				}
			}
		if(matching&&(nextExpression=REGetNextNode(theNode)))	/* see if there is another node after this one (if so, recurse into it) */
			{
			RECallNextNode(theChunk,theOffset,numToSearch,nextExpression,&matching,numMatched,recurseGlobals);
			}
		(*foundMatch)=matching;
		matchRecursionDepth--;
		}
	else
		{
		SetError(localErrorFamily,errorMembers[STACKOVERFLOW],errorDescriptions[STACKOVERFLOW]);
		fail=TRUE;
		}
	return(!fail);
}

static BOOLEAN REMatchAlternatives(CHUNKHEADER *theChunk,UINT32 theOffset,UINT32 numToSearch,EXPRESSIONLISTHEADER *theList,BOOLEAN *foundMatch,UINT32 *numMatched,RECURSEGLOBALS *recurseGlobals)
/* attempt to match alternatives at the given level
 * if the pattern could be matched, return TRUE in foundMatch with numMatched as the number of bytes matched
 * if no match was located, return FALSE in foundMatch
 * NOTE: do not search through more than numToSearch bytes
 * endChunk/endOffset will point to the end of the match if one was located
 * this tries each alternative in order until a match is located, or no match could be found
 * if there is some sort of hard failure, this routine will SetError, and return FALSE
 */
{
	EXPRESSIONLISTHEADER
		*currentList;
	BOOLEAN
		fail;

	if(++matchRecursionDepth<MAXRECURSIONDEPTH)
		{
		currentList=theList;
		(*foundMatch)=FALSE;
		fail=FALSE;
		while(currentList&&!(*foundMatch)&&!fail)
			{
			RECallNextNode(theChunk,theOffset,numToSearch,currentList->firstExpression,foundMatch,numMatched,recurseGlobals);
			currentList=currentList->nextAlternate;
			}
		matchRecursionDepth--;
		}
	else
		{
		SetError(localErrorFamily,errorMembers[STACKOVERFLOW],errorDescriptions[STACKOVERFLOW]);
		fail=TRUE;
		}
	return(!fail);
}

BOOLEAN REMatch(CHUNKHEADER *theChunk,UINT32 theOffset,COMPILEDEXPRESSION *theCompiledExpression,BOOLEAN leftEdge,BOOLEAN rightEdge,UINT32 numToSearch,UINT8 *translateTable,BOOLEAN *foundMatch,UINT32 *numMatched,CHUNKHEADER **endChunk,UINT32 *endOffset)
/* this is the regular expression match routine
 * it will return TRUE in foundMatch if a match is found
 * if there is a hard failure, SetError will be called, and FALSE returned
 */
{
	UINT32
		i;
	RECURSEGLOBALS
		recurseGlobals;

	for(i=0;i<MAXREGISTERS;i++)
		{
		theCompiledExpression->theRegisters[i].theChunk=NULL;			/* clear registers */
		}
	matchRecursionDepth=0;												/* clear recursion depth */
	if(translateTable)
		{
		matchRoutines=translatedRoutines;
		theTranslationTable=translateTable;								/* point at the table to use */
		}
	else
		{
		matchRoutines=exactRoutines;
		}

/* to save time/stack while calling recursively, this structure is used to hold variables that are global to the recursive search routines, and only a pointer to this is passed */
	recurseGlobals.leftEdge=leftEdge;
	recurseGlobals.rightEdge=rightEdge;
	recurseGlobals.initialNumToSearch=numToSearch;
	recurseGlobals.endChunk=NULL;
	recurseGlobals.endOffset=0;

	if(REMatchAlternatives(theChunk,theOffset,numToSearch,theCompiledExpression->firstList,foundMatch,numMatched,&recurseGlobals))
		{
		*endChunk=recurseGlobals.endChunk;
		*endOffset=recurseGlobals.endOffset;
		return(TRUE);
		}
	return(FALSE);
}
