
#include "xplore.h"
#include "callbacks.h"

#include "curdir.h"
#include "devmount.h"
#include "dialogs.h"
#include "dirs.h"
#include "error.h"
#include "icons.h"
#include "fileops.h"
#include "ftype.h"
#include "interface.h"
#include "menus.h"
#include "util.h"

#include <Xm/XmAll.h>
#include <Tabs.h>

/* private functions */

/* update directory information and redisplay the panes */

static void Update(void)
{
  update();
  UpdatePanes(shelf_modified, root_modified, cur_modified,
	      True, True, !cur_changed);
}

static void Restat(void)
{
  restat();
  UpdatePanes(shelf_modified, root_modified, cur_modified,
	      True, True, !cur_changed);
}

/* determine selected files in the file pane or shelf */

static int FindSelected(DirPtr dir)
{
  int i;
    
  for (i = 0; i < dirNFiles(dir); i++)
    if (fileSelected(dirFile(dir, i)))
      return i;
  return NONE;
}

static IconPtr sel_icon(DirPtr dir)
{
  int i;

  if (dirNFilesSelected(dir) == 1 && (i = FindSelected(dir)) != NONE)
    return fileLargeIcon(dirFile(dir, i));
  else
    return &std_icons[ICON_FILES];
}

/* ----------------------------------------------------------------------- */

void FileMapCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  lock = True;
  if (filepane && dirNFilesSelected(curdir) == 1) {
    XtSetSensitive(open_button, True);
    XtSetSensitive(rename_button, True);
  } else {
    XtSetSensitive(open_button, False);
    XtSetSensitive(rename_button, False);
  }
  if (filepane && dirNFilesSelected(curdir) > 0) {
    XtSetSensitive(move_button, True);
    XtSetSensitive(copy_button, True);
    XtSetSensitive(link_button, True);
    XtSetSensitive(delete_button, True);
    XtSetSensitive(properties_button, True);
  } else {
    XtSetSensitive(move_button, False);
    XtSetSensitive(copy_button, False);
    XtSetSensitive(link_button, False);
    XtSetSensitive(delete_button, False);
    XtSetSensitive(properties_button, False);
  }
  if (filepane) {
    XtSetSensitive(mkdir_button, True);
    XtSetSensitive(chdir_button, True);
    XtSetSensitive(parent_button, True);
    XtSetSensitive(home_button, True);
    XtSetSensitive(filter_button, True);
    XtSetSensitive(select_button, True);
    XtSetSensitive(select_all_button, True);
    XtSetSensitive(invert_all_button, True);
  } else {
    if (treepane) {
      XtSetSensitive(command_button, True);
      XtSetSensitive(mkdir_button, True);
      XtSetSensitive(chdir_button, True);
      XtSetSensitive(parent_button, True);
      XtSetSensitive(home_button, True);
    } else {
      XtSetSensitive(command_button, False);
      XtSetSensitive(mkdir_button, False);
      XtSetSensitive(chdir_button, False);
      XtSetSensitive(parent_button, False);
      XtSetSensitive(home_button, False);
    }
    XtSetSensitive(filter_button, False);
    XtSetSensitive(select_button, False);
    XtSetSensitive(select_all_button, False);
    XtSetSensitive(invert_all_button, False);
  }
}

void ShelfMapCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  lock = True;
  if (shelfpane && dirNFilesSelected(shelfdir) == 1) {
    XtSetSensitive(shelf_open_button, True);
    XtSetSensitive(shelf_rename_button, True);
  } else {
    XtSetSensitive(shelf_open_button, False);
    XtSetSensitive(shelf_rename_button, False);
  }
  if (shelfpane && dirNFilesSelected(shelfdir) > 0) {
    XtSetSensitive(shelf_move_button, True);
    XtSetSensitive(shelf_copy_button, True);
    XtSetSensitive(shelf_link_button, True);
    XtSetSensitive(shelf_delete_button, True);
    XtSetSensitive(shelf_properties_button, True);
  } else {
    XtSetSensitive(shelf_move_button, False);
    XtSetSensitive(shelf_copy_button, False);
    XtSetSensitive(shelf_link_button, False);
    XtSetSensitive(shelf_delete_button, False);
    XtSetSensitive(shelf_properties_button, False);
  }
  if (shelfpane) {
    XtSetSensitive(shelf_select_button, True);
    XtSetSensitive(shelf_select_all_button, True);
    XtSetSensitive(shelf_invert_all_button, True);
  } else {
    XtSetSensitive(shelf_select_button, False);
    XtSetSensitive(shelf_select_all_button, False);
    XtSetSensitive(shelf_invert_all_button, False);
  }
}

void ViewMapCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  lock = True;
  if (filepane) {
    XtSetSensitive(icons_button, True);
    XtSetSensitive(text_button, True);
    XtSetSensitive(sort_by_name_button, True);
    XtSetSensitive(sort_by_size_button, True);
    XtSetSensitive(sort_by_date_button, True);
    XtSetSensitive(reverse_button, True);
    XtSetSensitive(dirs_first_button, True);
    XtSetSensitive(show_parent_dir_button, True);
    XtSetSensitive(show_files_button, True);
    XtSetSensitive(show_dirs_button, True);
  } else {
    XtSetSensitive(icons_button, False);
    XtSetSensitive(text_button, False);
    XtSetSensitive(sort_by_name_button, False);
    XtSetSensitive(sort_by_size_button, False);
    XtSetSensitive(sort_by_date_button, False);
    XtSetSensitive(reverse_button, False);
    XtSetSensitive(dirs_first_button, False);
    XtSetSensitive(show_parent_dir_button, False);
    XtSetSensitive(show_files_button, False);
    XtSetSensitive(show_dirs_button, False);
  }
  if (filepane || treepane) {
    XtSetSensitive(rescan_files_button, True);
    XtSetSensitive(magic_headers_button, True);
    XtSetSensitive(update_button, True);
    XtSetSensitive(reread_button, True);
    XtSetSensitive(unmount_button, True);
    XtSetSensitive(unmount_all_button, True);
  } else {
    XtSetSensitive(rescan_files_button, False);
    XtSetSensitive(magic_headers_button, False);
    XtSetSensitive(update_button, False);
    XtSetSensitive(reread_button, False);
    XtSetSensitive(unmount_button, False);
    XtSetSensitive(unmount_all_button, False);
  }
  if ((filepane || treepane) &&
      (dirOptions(curdir) & (INCLUDE_FILES | INCLUDE_DIRS)))
    XtSetSensitive(show_hidden_button, True);
  else
    XtSetSensitive(show_hidden_button, False);
  if (viewtype == IconView) {
    XtVaSetValues(icons_button, XmNset, XmSET, NULL);
    XtVaSetValues(text_button, XmNset, XmUNSET, NULL);
  } else {
    XtVaSetValues(icons_button, XmNset, XmUNSET, NULL);
    XtVaSetValues(text_button, XmNset, XmSET, NULL);
  }
  if (treepane)
    XtVaSetValues(tree_pane_button, XmNset, XmSET, NULL);
  else
    XtVaSetValues(tree_pane_button, XmNset, XmUNSET, NULL);
  if (filepane)
    XtVaSetValues(file_pane_button, XmNset, XmSET, NULL);
  else
    XtVaSetValues(file_pane_button, XmNset, XmUNSET, NULL);
  if (shelfpane)
    XtVaSetValues(shelf_pane_button, XmNset, XmSET, NULL);
  else
    XtVaSetValues(shelf_pane_button, XmNset, XmUNSET, NULL);
  if (dirOptions(curdir) & SORT_BY_NAME)
    XtVaSetValues(sort_by_name_button, XmNset, XmSET, NULL);
  else
    XtVaSetValues(sort_by_name_button, XmNset, XmUNSET, NULL);
  if (dirOptions(curdir) & SORT_BY_SIZE)
    XtVaSetValues(sort_by_size_button, XmNset, XmSET, NULL);
  else
    XtVaSetValues(sort_by_size_button, XmNset, XmUNSET, NULL);
  if (dirOptions(curdir) & SORT_BY_DATE)
    XtVaSetValues(sort_by_date_button, XmNset, XmSET, NULL);
  else
    XtVaSetValues(sort_by_date_button, XmNset, XmUNSET, NULL);
  if (dirOptions(curdir) & REVERSE)
    XtVaSetValues(reverse_button, XmNset, XmSET, NULL);
  else
    XtVaSetValues(reverse_button, XmNset, XmUNSET, NULL);
  if (dirOptions(curdir) & DIRS_FIRST)
    XtVaSetValues(dirs_first_button, XmNset, XmSET, NULL);
  else
    XtVaSetValues(dirs_first_button, XmNset, XmUNSET, NULL);
  if (dirOptions(curdir) & INCLUDE_HIDDEN)
    XtVaSetValues(show_hidden_button, XmNset, XmSET, NULL);
  else
    XtVaSetValues(show_hidden_button, XmNset, XmUNSET, NULL);
  if (dirOptions(curdir) & INCLUDE_UPDIR)
    XtVaSetValues(show_parent_dir_button, XmNset, XmSET, NULL);
  else
    XtVaSetValues(show_parent_dir_button, XmNset, XmUNSET, NULL);
  if (dirOptions(curdir) & INCLUDE_FILES)
    XtVaSetValues(show_files_button, XmNset, XmSET, NULL);
  else
    XtVaSetValues(show_files_button, XmNset, XmUNSET, NULL);
  if (dirOptions(curdir) & INCLUDE_DIRS)
    XtVaSetValues(show_dirs_button, XmNset, XmSET, NULL);
  else
    XtVaSetValues(show_dirs_button, XmNset, XmUNSET, NULL);
  if (dirOptions(curdir) & CHECK_FILES)
    XtVaSetValues(rescan_files_button, XmNset, XmSET, NULL);
  else
    XtVaSetValues(rescan_files_button, XmNset, XmUNSET, NULL);
  if (dirOptions(curdir) & NOMAGIC)
    XtVaSetValues(magic_headers_button, XmNset, XmUNSET, NULL);
  else
    XtVaSetValues(magic_headers_button, XmNset, XmSET, NULL);
}

void OptionsMapCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  lock = True;
  if (update > 0)
    XtSetSensitive(auto_updates_button, True);
  else
    XtSetSensitive(auto_updates_button, False);
  if (absolute)
    XtVaSetValues(absolute_paths_button, XmNset, XmSET, NULL);
  else
    XtVaSetValues(absolute_paths_button, XmNset, XmUNSET, NULL);
  if (dirtarget)
    XtVaSetValues(target_dir_button, XmNset, XmSET, NULL);
  else
    XtVaSetValues(target_dir_button, XmNset, XmUNSET, NULL);
  if (pushdir)
    XtVaSetValues(push_dir_button, XmNset, XmSET, NULL);
  else
    XtVaSetValues(push_dir_button, XmNset, XmUNSET, NULL);
  if (echo)
    XtVaSetValues(echo_commands_button, XmNset, XmSET, NULL);
  else
    XtVaSetValues(echo_commands_button, XmNset, XmUNSET, NULL);
  if (check)
    XtVaSetValues(check_mounts_button, XmNset, XmSET, NULL);
  else
    XtVaSetValues(check_mounts_button, XmNset, XmUNSET, NULL);
  if (backups)
    XtVaSetValues(backups_button, XmNset, XmSET, NULL);
  else
    XtVaSetValues(backups_button, XmNset, XmUNSET, NULL);
  if (confirm_drop)
    XtVaSetValues(confirm_drop_button, XmNset, XmSET, NULL);
  else
    XtVaSetValues(confirm_drop_button, XmNset, XmUNSET, NULL);
  if (confirm_move)
    XtVaSetValues(confirm_move_button, XmNset, XmSET, NULL);
  else
    XtVaSetValues(confirm_move_button, XmNset, XmUNSET, NULL);
  if (confirm_copy)
    XtVaSetValues(confirm_copy_button, XmNset, XmSET, NULL);
  else
    XtVaSetValues(confirm_copy_button, XmNset, XmUNSET, NULL);
  if (confirm_link)
    XtVaSetValues(confirm_link_button, XmNset, XmSET, NULL);
  else
    XtVaSetValues(confirm_link_button, XmNset, XmUNSET, NULL);
  if (confirm_delete)
    XtVaSetValues(confirm_delete_button, XmNset, XmSET, NULL);
  else
    XtVaSetValues(confirm_delete_button, XmNset, XmUNSET, NULL);
  if (confirm_deldir)
    XtVaSetValues(confirm_deldir_button, XmNset, XmSET, NULL);
  else
    XtVaSetValues(confirm_deldir_button, XmNset, XmUNSET, NULL);
  if (confirm_delfile)
    XtVaSetValues(confirm_delfile_button, XmNset, XmSET, NULL);
  else
    XtVaSetValues(confirm_delfile_button, XmNset, XmUNSET, NULL);
  if (confirm_overwrt)
    XtVaSetValues(confirm_overwrite_button, XmNset, XmSET, NULL);
  else
    XtVaSetValues(confirm_overwrite_button, XmNset, XmUNSET, NULL);
  if (confirm_quit)
    XtVaSetValues(confirm_quit_button, XmNset, XmSET, NULL);
  else
    XtVaSetValues(confirm_quit_button, XmNset, XmUNSET, NULL);
  if (updates)
    XtVaSetValues(auto_updates_button, XmNset, XmSET, NULL);
  else
    XtVaSetValues(auto_updates_button, XmNset, XmUNSET, NULL);
}

void HelpMapCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  lock = True;
}

/* ----------------------------------------------------------------------- */

void FileUnmapCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  lock = False;
  XtSetSensitive(command_button, True);
  XtSetSensitive(mkdir_button, True);
  XtSetSensitive(chdir_button, True);
  XtSetSensitive(parent_button, True);
  XtSetSensitive(home_button, True);
  XtSetSensitive(open_button, True);
  XtSetSensitive(rename_button, True);
  XtSetSensitive(move_button, True);
  XtSetSensitive(copy_button, True);
  XtSetSensitive(link_button, True);
  XtSetSensitive(delete_button, True);
  XtSetSensitive(properties_button, True);
  XtSetSensitive(filter_button, True);
  XtSetSensitive(select_button, True);
  XtSetSensitive(select_all_button, True);
  XtSetSensitive(invert_all_button, True);
}

void ShelfUnmapCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  lock = False;
  XtSetSensitive(shelf_open_button, True);
  XtSetSensitive(shelf_rename_button, True);
  XtSetSensitive(shelf_move_button, True);
  XtSetSensitive(shelf_copy_button, True);
  XtSetSensitive(shelf_link_button, True);
  XtSetSensitive(shelf_delete_button, True);
  XtSetSensitive(shelf_properties_button, True);
  XtSetSensitive(shelf_select_button, True);
  XtSetSensitive(shelf_select_all_button, True);
  XtSetSensitive(shelf_invert_all_button, True);
}

void ViewUnmapCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  lock = False;
  XtSetSensitive(icons_button, True);
  XtSetSensitive(text_button, True);
  XtSetSensitive(sort_by_name_button, True);
  XtSetSensitive(sort_by_size_button, True);
  XtSetSensitive(sort_by_date_button, True);
  XtSetSensitive(reverse_button, True);
  XtSetSensitive(dirs_first_button, True);
  XtSetSensitive(show_parent_dir_button, True);
  XtSetSensitive(show_files_button, True);
  XtSetSensitive(show_dirs_button, True);
  XtSetSensitive(show_hidden_button, True);
  XtSetSensitive(rescan_files_button, True);
  XtSetSensitive(magic_headers_button, True);
  XtSetSensitive(update_button, True);
  XtSetSensitive(reread_button, True);
  XtSetSensitive(unmount_button, True);
  XtSetSensitive(unmount_all_button, True);
}

void OptionsUnmapCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  lock = False;
  XtSetSensitive(auto_updates_button, True);
}

void UnmapCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  lock = False;
}

/* ----------------------------------------------------------------------- */

void OpenCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  int i;

  if (filepane && dirNFilesSelected(curdir) == 1 &&
      (i = FindSelected(curdir)) != NONE) {
    cur_push(i, absolute, pushdir);
    UpdatePanes(shelf_modified, root_modified, cur_modified,
		True, True, !cur_changed);
  }
}

#define MAX_BUF 10000

void CommandActionCB(Widget w, int i, XtPointer widget_data)
{
  char buf[MAX_BUF+1];

  if (!filepane && !treepane)
    return;
  XmUpdateDisplay(app_shell);
  if (cmd_action(i) && *cmd_action(i) &&
      param_dlg(&std_icons[ICON_APPLY], cmd_label(i), cmd_action(i), buf,
		MAX_BUF) == OK) {
    SelPtr sel = filepane ? selFromDir(curdir) : NULL;

    fileApply(sel, buf, dirName(curdir), absolute);
    selFree(sel);
    Update();
  }
}

void MkdirCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  char name[MAXPATHLEN+1];

  if (!filepane && !treepane)
    return;
  XmUpdateDisplay(app_shell);
  *name = '\0';
  if (mkdir_dlg(&std_icons[ICON_DIR], name, MAXPATHLEN) == OK) {
    cur_mkdir(name);
    UpdatePanes(shelf_modified, root_modified, cur_modified,
		True, True, !cur_changed);
  }
}

void ChdirCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  char name[MAXPATHLEN+1], path[MAXPATHLEN+1];

  if (!filepane && !treepane)
    return;
  XmUpdateDisplay(app_shell);
  *name = '\0';
  if (chdir_dlg(&std_icons[ICON_DIR], name, MAXPATHLEN) == OK) {
    if (!cur_chdir(abspath(path, dirName(curdir), name)))
      xplerr1(ERR_FIND_DIR, path);
    UpdatePanes(shelf_modified, root_modified, cur_modified,
		True, True, !cur_changed);
  }
}

void ParentCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  char path[MAXPATHLEN+1];

  if (!filepane && !treepane)
    return;
  XmUpdateDisplay(app_shell);
  if (!cur_chdir(abspath(path, dirName(curdir), "..")))
    xplerr1(ERR_FIND_DIR, path);
  UpdatePanes(shelf_modified, root_modified, cur_modified,
	      True, True, !cur_changed);
}

void HomeCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  char path[MAXPATHLEN+1];

  if (!filepane && !treepane)
    return;
  XmUpdateDisplay(app_shell);
  if (!cur_chdir(fnexpand(path, "~")))
    xplerr1(ERR_FIND_DIR, path);
  UpdatePanes(shelf_modified, root_modified, cur_modified,
	      True, True, !cur_changed);
}

void RenameCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  int i;

  if (filepane && dirNFilesSelected(curdir) == 1 &&
      (i = FindSelected(curdir)) != NONE) {
    FilePtr file = dirFile(curdir, i);
    char target[MAXPATHLEN+1], abstarget[MAXPATHLEN+1];
    StatRec stats;

    XmUpdateDisplay(app_shell);
    strcpy(target, fileName(file));
    if (rename_dlg(fileLargeIcon(file), target, MAXPATHLEN) == OK &&
	strcmp(target, fileName(file)))
      if (!stat(abspath(abstarget, dirName(curdir), target), &stats) &&
	  S_ISDIR(stats.st_mode))
	xplerr1(ERR_TARGET_IS_DIR, target);
      else {
	cur_move(target, absolute);
	UpdatePanes(shelf_modified, root_modified, cur_modified,
		    True, True, !cur_changed);
      }
  }
}

void MoveCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  if (filepane && dirNFilesSelected(curdir)) {
    char target[MAXPATHLEN+1];

    XmUpdateDisplay(app_shell);
    *target = '\0';
    if (move_dlg(sel_icon(curdir), target, MAXPATHLEN) == OK) {
      cur_move(target, absolute);
      UpdatePanes(shelf_modified, root_modified, cur_modified,
		  True, True, !cur_changed);
    }
  }
}

void CopyCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  if (filepane && dirNFilesSelected(curdir)) {
    char target[MAXPATHLEN+1];

    XmUpdateDisplay(app_shell);
    *target = '\0';
    if (copy_dlg(sel_icon(curdir), target, MAXPATHLEN) == OK) {
      cur_copy(target, absolute);
      UpdatePanes(shelf_modified, root_modified, cur_modified,
		  True, True, !cur_changed);
    }
  }
}

void LinkCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  if (filepane && dirNFilesSelected(curdir)) {
    char target[MAXPATHLEN+1];

    XmUpdateDisplay(app_shell);
    *target = '\0';
    if (link_dlg(sel_icon(curdir), target, MAXPATHLEN) == OK) {
      cur_link(target, absolute);
      UpdatePanes(shelf_modified, root_modified, cur_modified,
		  True, True, !cur_changed);
    }
  }
}

void DeleteCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  if (filepane && dirNFilesSelected(curdir)) {
    XmUpdateDisplay(app_shell);
    cur_delete();
    UpdatePanes(shelf_modified, root_modified, cur_modified,
		True, True, !cur_changed);
  }
}

void PropertiesCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  IconPtr icon = NULL;
  int nFiles = dirNFilesSelected(curdir);
  int nBytes = dirNBytesSelected(curdir);
  String dirname = dirName(curdir), filename = NULL, linkname = NULL;
  String desc = NULL, comm = NULL;
  String last_acc = NULL, last_mod = NULL, last_chg = NULL;
  uid_t uid;
  gid_t gid;
  Boolean uid_mask = True, gid_mask = True;
  umode_t mode, mode_mask = S_IRWXU|S_IRWXG|S_IRWXO;
  char user[100], group[100];
  char ctime_acc[100], ctime_mod[100], ctime_chg[100];
  char path[MAXPATHLEN+1], buf[MAXPATHLEN+1];
  struct passwd *pw;
  struct group *gr;

  if (filepane && nFiles > 0) {
    int i, i0 = FindSelected(curdir);
    FilePtr file = dirFile(curdir, i0);
    StatPtr lstats = fileLstats(file);

    Wait();
    if (nFiles == 1) {
      icon = fileLargeIcon(file);
      filename = fileName(file);
      comm = comment(fileType(file));
      if (S_ISLNK(lstats->st_mode)) {
	int len = readlink(pathname(path, dirname, filename), buf, MAXPATHLEN);
	if (len >= 0) {
	  buf[len] = '\0';
	  linkname = buf;
	} else
	  linkname = NULL;
	desc = "Symbolic link";
      } else if (S_ISDIR(lstats->st_mode))
	desc = "Directory";
      else if (S_ISCHR(lstats->st_mode))
	desc = "Character special file";
      else if(S_ISBLK(lstats->st_mode))
	desc = "Block special file";
      else if(S_ISSOCK(lstats->st_mode))
	desc = "Socket";
      else if(S_ISFIFO(lstats->st_mode))
	desc = "Pipe or FIFO special file";
      else
	desc = "Ordinary file";
      strcpy(ctime_acc, ctime(&lstats->st_atime));
      strcpy(ctime_mod, ctime(&lstats->st_mtime));
      strcpy(ctime_chg, ctime(&lstats->st_ctime));
      if (*ctime_acc) ctime_acc[strlen(ctime_acc)-1] = '\0';
      if (*ctime_mod) ctime_mod[strlen(ctime_mod)-1] = '\0';
      if (*ctime_chg) ctime_chg[strlen(ctime_chg)-1] = '\0';
      last_acc = ctime_acc;
      last_mod = ctime_mod;
      last_chg = ctime_chg;
    } else
      icon = &std_icons[ICON_FILES];
    uid = lstats->st_uid;
    gid = lstats->st_gid;
    mode = lstats->st_mode;
    for (i = i0+1; i < dirNFiles(curdir); i++) {
      file = dirFile(curdir, i);
      if (fileSelected(file)) {
	umode_t m;

	lstats = fileLstats(file);
	if (uid_mask && lstats->st_uid != uid)
	  uid_mask = False;
	if (gid_mask && lstats->st_gid != gid)
	  gid_mask = False;
	if ((m = (mode_mask & lstats->st_mode) ^ (mode_mask & mode)))
	  mode_mask &= ~m;
      }
    }
    if (uid_mask)
      if ((pw = getpwuid(uid)) == NULL)
	sprintf(user, "%lu", (unsigned long) uid);
      else
	strcpy(user, pw->pw_name);
    else
      *user = '\0';
    if (gid_mask)
      if ((gr = getgrgid(gid)) == NULL)
	sprintf(group, "%lu", (unsigned long) gid);
      else
	strcpy(group, gr->gr_name);
    else
      *group = '\0';
    if (props_dlg(icon, nFiles, nBytes, dirname, filename, linkname,
		  desc, comm, last_acc, last_mod, last_chg,
		  user, group, 100, &mode, &mode_mask) != CANCEL) {
      /* the following stuff should really be replaced by a fileChmod
	 operation in the fileops module sometime */
      long l;
      String p;

      if (*user)
	if ((l = strtol(user, &p, 10)) >= 0 && *p == '\0')
	  uid = l;
	else if ((pw = getpwnam(user))) {
	  uid = pw->pw_uid;
	  uid_mask = True;
	} else {
	  xplerr1(ERR_INVALID_USR, user);
	  goto exit;
	}
      else
	uid_mask = False;
      if (*group)
	if ((l = strtol(group, &p, 10)) >= 0 && *p == '\0')
	  gid = l;
	else if ((gr = getgrnam(group))) {
	  gid = gr->gr_gid;
	  gid_mask = True;
	} else {
	  xplerr1(ERR_INVALID_GRP, group);
	  goto exit;
	}
      else
	gid_mask = False;
      for (i = i0; i < dirNFiles(curdir); i++) {
	file = dirFile(curdir, i);
	
	if (fileSelected(file)) {
	  umode_t mode1, mode2;

	  lstats = fileLstats(file);
	  pathname(path, dirname, fileName(file));
	  mode1 = lstats->st_mode;
	  mode2 = (mode1 & ~mode_mask) | (mode & mode_mask);
	  if (mode2 != mode1 && chmod(path, mode2)) {
	    syserr1(ERR_CHMOD, path);
	    goto exit;
	  }
	  if ((uid_mask && lstats->st_uid != uid ||
	       gid_mask && lstats->st_gid != gid) &&
	      chown(path,
		    uid_mask?uid:lstats->st_uid, 
		    gid_mask?gid:lstats->st_gid)) {
	    syserr1(ERR_CHOWN, path);
	    goto exit;
	  }
	}
      }
    }
  exit:
    update();
    UpdatePanes(shelf_modified, root_modified, cur_modified,
		True, True, !cur_changed);
    Done();
  }
}

#define MAXPATLEN 5000

void SelectCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  if (filepane) {
    char pattern[MAXPATLEN+1];

    XmUpdateDisplay(app_shell);
    *pattern = '\0';
    switch (select_dlg(&std_icons[ICON_FILES], pattern, MAXPATLEN)) {
    case REPLACE:
      Wait();
      dirDeselect(curdir, ALL, NULL);
      dirSelect(curdir, ALL, pattern);
      break;
    case ADD:
      Wait();
      dirSelect(curdir, ALL, pattern);
      break;
    case REMOVE:
      Wait();
      dirDeselect(curdir, ALL, pattern);
      break;
    default:
      return;
    }
    UpdatePanes(False, False, True, True, True, True);
    Done();
  }
}

void SelectAllCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  if (filepane) {
    Wait();
    dirSelect(curdir, ALL, "*");
    UpdatePanes(False, False, True, True, True, True);
    Done();
  }
}

void InvertAllCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  if (filepane) {
    Wait();
    dirInvert(curdir, ALL, "*");
    UpdatePanes(False, False, True, True, True, True);
    Done();
  }
}

void FilterCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  if (filepane) {
    int ret;
    char pattern[MAXPATLEN+1];

    XmUpdateDisplay(app_shell);
    if (dirFilter(curdir))
      strcpy(pattern, dirFilter(curdir));
    else
      *pattern = '\0';
    if ((ret = filter_dlg(&std_icons[ICON_FILES], pattern, MAXPATLEN))
	!= CANCEL) {
      Wait();
      dirSetFilter(curdir, (ret == CLEAR)?NULL:pattern);
      UpdatePanes(False, True, True, True, True, False);
      Done();
    }
  }
}

void QuitCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  XmUpdateDisplay(app_shell);
  if (confirm_quit_dlg(NULL) == OK)
    quit();
}

void MenuActionCB(Widget w, int j, XtPointer widget_data)
{
  int i;

  if (filepane && dirNFilesSelected(curdir) == 1 &&
      (i = FindSelected(curdir)) != NONE) {
    XmUpdateDisplay(app_shell);
    fileMenu(curdir, i, j, absolute, False);
    Update();
  }
}

/* ----------------------------------------------------------------------- */

void ShelfOpenCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  int i;

  if (shelfpane && dirNFilesSelected(shelfdir) == 1 &&
      (i = FindSelected(shelfdir)) !=
      NONE) {
    shelf_push(i, absolute, pushdir);
    UpdatePanes(shelf_modified, root_modified, cur_modified,
		True, True, !cur_changed);
  }
}

void ShelfRenameCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  int i;

  if (shelfpane && dirNFilesSelected(shelfdir) == 1 &&
      (i = FindSelected(shelfdir)) !=
      NONE) {
    FilePtr file = dirFile(shelfdir, i);
    char target[MAXPATHLEN+1], abstarget[MAXPATHLEN+1];
    StatRec stats;

    XmUpdateDisplay(app_shell);
    strcpy(target, fileName(file));
    if (rename_dlg(fileLargeIcon(file), target, MAXPATHLEN) == OK &&
	strcmp(target, fileName(file)))
      if (!stat(abspath(abstarget, dirName(shelfdir), target), &stats) &&
	  S_ISDIR(stats.st_mode))
	xplerr1(ERR_TARGET_IS_DIR, target);
      else {
	shelf_move(target, absolute);
	UpdatePanes(shelf_modified, root_modified, cur_modified,
		    True, True, !cur_changed);
      }
  }
}

void ShelfMoveCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  if (shelfpane && dirNFilesSelected(shelfdir)) {
    char target[MAXPATHLEN+1];

    XmUpdateDisplay(app_shell);
    *target = '\0';
    if (move_dlg(sel_icon(shelfdir), target, MAXPATHLEN) == OK) {
      shelf_move(target, absolute);
      UpdatePanes(shelf_modified, root_modified, cur_modified,
		  True, True, !cur_changed);
    }
  }
}

void ShelfCopyCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  if (shelfpane && dirNFilesSelected(shelfdir)) {
    char target[MAXPATHLEN+1];

    XmUpdateDisplay(app_shell);
    *target = '\0';
    if (copy_dlg(sel_icon(shelfdir), target, MAXPATHLEN) == OK) {
      shelf_copy(target, absolute);
      UpdatePanes(shelf_modified, root_modified, cur_modified,
		  True, True, !cur_changed);
    }
  }
}

void ShelfLinkCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  if (shelfpane && dirNFilesSelected(shelfdir)) {
    char target[MAXPATHLEN+1];

    XmUpdateDisplay(app_shell);
    *target = '\0';
    if (link_dlg(sel_icon(shelfdir), target, MAXPATHLEN) == OK) {
      shelf_link(target, absolute);
      UpdatePanes(shelf_modified, root_modified, cur_modified,
		  True, True, !cur_changed);
    }
  }
}

void ShelfDeleteCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  if (shelfpane && dirNFilesSelected(shelfdir)) {
    XmUpdateDisplay(app_shell);
    shelf_delete();
    UpdatePanes(shelf_modified, root_modified, cur_modified,
		True, True, !cur_changed);
  }
}

void ShelfPropertiesCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  IconPtr icon = NULL;
  int nFiles = dirNFilesSelected(shelfdir);
  int nBytes = dirNBytesSelected(shelfdir);
  String dirname = dirName(shelfdir), filename = NULL, linkname = NULL;
  String desc = NULL, comm = NULL;
  String last_acc = NULL, last_mod = NULL, last_chg = NULL;
  uid_t uid;
  gid_t gid;
  Boolean uid_mask = True, gid_mask = True;
  umode_t mode, mode_mask = S_IRWXU|S_IRWXG|S_IRWXO;
  char user[100], group[100];
  char ctime_acc[100], ctime_mod[100], ctime_chg[100];
  char path[MAXPATHLEN+1], buf[MAXPATHLEN+1];
  struct passwd *pw;
  struct group *gr;

  if (shelfpane && nFiles > 0) {
    int i, i0 = FindSelected(shelfdir);
    FilePtr file = dirFile(shelfdir, i0);
    StatPtr lstats = fileLstats(file);

    Wait();
    if (nFiles == 1) {
      icon = fileLargeIcon(file);
      filename = fileName(file);
      comm = comment(fileType(file));
      if (S_ISLNK(lstats->st_mode)) {
	int len = readlink(pathname(path, dirname, filename), buf, MAXPATHLEN);
	if (len >= 0) {
	  buf[len] = '\0';
	  linkname = buf;
	} else
	  linkname = NULL;
	desc = "Symbolic link";
      } else if (S_ISDIR(lstats->st_mode))
	desc = "Directory";
      else if (S_ISCHR(lstats->st_mode))
	desc = "Character special file";
      else if(S_ISBLK(lstats->st_mode))
	desc = "Block special file";
      else if(S_ISSOCK(lstats->st_mode))
	desc = "Socket";
      else if(S_ISFIFO(lstats->st_mode))
	desc = "Pipe or FIFO special file";
      else
	desc = "Ordinary file";
      strcpy(ctime_acc, ctime(&lstats->st_atime));
      strcpy(ctime_mod, ctime(&lstats->st_mtime));
      strcpy(ctime_chg, ctime(&lstats->st_ctime));
      if (*ctime_acc) ctime_acc[strlen(ctime_acc)-1] = '\0';
      if (*ctime_mod) ctime_mod[strlen(ctime_mod)-1] = '\0';
      if (*ctime_chg) ctime_chg[strlen(ctime_chg)-1] = '\0';
      last_acc = ctime_acc;
      last_mod = ctime_mod;
      last_chg = ctime_chg;
    } else
      icon = &std_icons[ICON_FILES];
    uid = lstats->st_uid;
    gid = lstats->st_gid;
    mode = lstats->st_mode;
    for (i = i0+1; i < dirNFiles(shelfdir); i++) {
      file = dirFile(shelfdir, i);
      if (fileSelected(file)) {
	umode_t m;

	lstats = fileLstats(file);
	if (uid_mask && lstats->st_uid != uid)
	  uid_mask = False;
	if (gid_mask && lstats->st_gid != gid)
	  gid_mask = False;
	if ((m = (mode_mask & lstats->st_mode) ^ (mode_mask & mode)))
	  mode_mask &= ~m;
      }
    }
    if (uid_mask)
      if ((pw = getpwuid(uid)) == NULL)
	sprintf(user, "%lu", (unsigned long) uid);
      else
	strcpy(user, pw->pw_name);
    else
      *user = '\0';
    if (gid_mask)
      if ((gr = getgrgid(gid)) == NULL)
	sprintf(group, "%lu", (unsigned long) gid);
      else
	strcpy(group, gr->gr_name);
    else
      *group = '\0';
    if (props_dlg(icon, nFiles, nBytes, dirname, filename, linkname,
		  desc, comm, last_acc, last_mod, last_chg,
		  user, group, 100, &mode, &mode_mask) != CANCEL) {
      /* the following stuff should really be replaced by a fileChmod
	 operation in the fileops module sometime */
      long l;
      String p;

      if (*user)
	if ((l = strtol(user, &p, 10)) >= 0 && *p == '\0')
	  uid = l;
	else if ((pw = getpwnam(user))) {
	  uid = pw->pw_uid;
	  uid_mask = True;
	} else {
	  xplerr1(ERR_INVALID_USR, user);
	  goto exit;
	}
      else
	uid_mask = False;
      if (*group)
	if ((l = strtol(group, &p, 10)) >= 0 && *p == '\0')
	  gid = l;
	else if ((gr = getgrnam(group))) {
	  gid = gr->gr_gid;
	  gid_mask = True;
	} else {
	  xplerr1(ERR_INVALID_GRP, group);
	  goto exit;
	}
      else
	gid_mask = False;
      for (i = i0; i < dirNFiles(shelfdir); i++) {
	file = dirFile(shelfdir, i);
	
	if (fileSelected(file)) {
	  umode_t mode1, mode2;

	  lstats = fileLstats(file);
	  pathname(path, dirname, fileName(file));
	  mode1 = lstats->st_mode;
	  mode2 = (mode1 & ~mode_mask) | (mode & mode_mask);
	  if (mode2 != mode1 && chmod(path, mode2)) {
	    syserr1(ERR_CHMOD, path);
	    goto exit;
	  }
	  if ((uid_mask && lstats->st_uid != uid ||
	       gid_mask && lstats->st_gid != gid) &&
	      chown(path,
		    uid_mask?uid:lstats->st_uid, 
		    gid_mask?gid:lstats->st_gid)) {
	    syserr1(ERR_CHOWN, path);
	    goto exit;
	  }
	}
      }
    }
  exit:
    update();
    UpdatePanes(shelf_modified, root_modified, cur_modified,
		True, True, !cur_changed);
    Done();
  }
}

void ShelfSelectCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  if (shelfpane) {
    char pattern[MAXPATLEN+1];

    XmUpdateDisplay(app_shell);
    *pattern = '\0';
    switch (select_dlg(&std_icons[ICON_FILES], pattern, MAXPATLEN)) {
    case REPLACE:
      Wait();
      dirDeselect(shelfdir, ALL, NULL);
      dirSelect(shelfdir, ALL, pattern);
      break;
    case ADD:
      Wait();
      dirSelect(shelfdir, ALL, pattern);
      break;
    case REMOVE:
      Wait();
      dirDeselect(shelfdir, ALL, pattern);
      break;
    default:
      return;
    }
    UpdatePanes(True, False, False, True, True, True);
    Done();
  }
}

void ShelfSelectAllCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  if (shelfpane) {
    Wait();
    dirSelect(shelfdir, ALL, "*");
    UpdatePanes(True, False, False, True, True, True);
    Done();
  }
}

void ShelfInvertAllCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  if (shelfpane) {
    Wait();
    dirInvert(shelfdir, ALL, "*");
    UpdatePanes(True, False, False, True, True, True);
    Done();
  }
}

void ShelfMenuActionCB(Widget w, int j, XtPointer widget_data)
{
  int i;

  if (shelfpane && dirNFilesSelected(shelfdir) == 1 &&
      (i = FindSelected(shelfdir)) !=
      NONE) {
    XmUpdateDisplay(app_shell);
    fileMenu(shelfdir, i, j, absolute, True);
    Update();
  }
}

/* ----------------------------------------------------------------------- */

void IconsCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  if (filepane && viewtype != IconView) {
    viewtype = IconView;
    Wait();
    UpdatePanes(False, False, True, True, True, False);
    Done();
  }
}

void TextCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  if (filepane && viewtype != TextView) {
    viewtype = TextView;
    Wait();
    UpdatePanes(False, False, True, True, True, False);
    Done();
  }
}

void TreePaneCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  if (treepane)
    HideTreePane();
  else
    ShowTreePane();
}

void FilePaneCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  if (filepane)
    HideFilePane();
  else
    ShowFilePane();
}

void ShelfPaneCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  if (shelfpane)
    HideShelfPane();
  else
    ShowShelfPane();
}

void SortByNameCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  if (filepane) {
    Wait();
    if ((dirOptions(curdir) & SORT_MASK) != SORT_BY_NAME) {
      dirSetOptions(curdir, SORT_BY_NAME);
      UpdatePanes(False, True, True, True, True, False);
    }
    Done();
  }
}

void SortBySizeCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  if (filepane) {
    Wait();
    if ((dirOptions(curdir) & SORT_MASK) != SORT_BY_SIZE) {
      dirSetOptions(curdir, SORT_BY_SIZE);
      UpdatePanes(False, True, True, True, True, False);
    }
    Done();
  }
}

void SortByDateCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  if (filepane) {
    Wait();
    if ((dirOptions(curdir) & SORT_MASK) != SORT_BY_DATE) {
      dirSetOptions(curdir, SORT_BY_DATE);
      UpdatePanes(False, True, True, True, True, False);
    }
    Done();
  }
}

void ReverseCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  if (filepane) {
    Wait();
    if (dirOptions(curdir) & REVERSE)
      dirClearOptions(curdir, REVERSE);
    else
      dirSetOptions(curdir, REVERSE);
    UpdatePanes(False, True, True, True, True, False);
    Done();
  }
}

void DirsFirstCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  if (filepane) {
    Wait();
    if (dirOptions(curdir) & DIRS_FIRST)
      dirClearOptions(curdir, DIRS_FIRST);
    else
      dirSetOptions(curdir, DIRS_FIRST);
    UpdatePanes(False, True, True, True, True, False);
    Done();
  }
}

void ShowHiddenFilesCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  if (filepane || treepane) {
    Wait();
    if (dirOptions(curdir) & INCLUDE_HIDDEN)
      dirClearOptions(curdir, INCLUDE_HIDDEN);
    else
      dirSetOptions(curdir, INCLUDE_HIDDEN);
    UpdatePanes(False, True, True, True, True, False);
    Done();
  }
}

void ShowParentDirCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  if (filepane) {
    Wait();
    if (dirOptions(curdir) & INCLUDE_UPDIR)
      dirClearOptions(curdir, INCLUDE_UPDIR);
    else
      dirSetOptions(curdir, INCLUDE_UPDIR);
    UpdatePanes(False, True, True, True, True, False);
    Done();
  }
}

void ShowFilesCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  if (filepane) {
    Wait();
    if (dirOptions(curdir) & INCLUDE_FILES)
      dirClearOptions(curdir, INCLUDE_FILES);
    else
      dirSetOptions(curdir, INCLUDE_FILES);
    UpdatePanes(False, True, True, True, True, False);
    Done();
  }
}

void ShowDirsCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  if (filepane) {
    Wait();
    if (dirOptions(curdir) & INCLUDE_DIRS)
      dirClearOptions(curdir, INCLUDE_DIRS);
    else
      dirSetOptions(curdir, INCLUDE_DIRS);
    UpdatePanes(False, True, True, True, True, False);
    Done();
  }
}

void RescanFilesCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  if (!filepane && !treepane)
    return;
  Wait();
  if (dirOptions(curdir) & CHECK_FILES)
    dirClearOptions(curdir, CHECK_FILES);
  else
    dirSetOptions(curdir, CHECK_FILES);
  UpdatePanes(False, True, True, True, True, True);
  Done();
}

void MagicHeadersCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  if (!filepane && !treepane)
    return;
  Wait();
  if (dirOptions(curdir) & NOMAGIC)
    dirClearOptions(curdir, NOMAGIC);
  else
    dirSetOptions(curdir, NOMAGIC);
  UpdatePanes(False, True, True, True, True, True);
  Done();
}

void UpdateCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  if (!filepane && !treepane)
    return;
  XmUpdateDisplay(app_shell);
  if (updates)
    XtRemoveTimeOut(timer_id);
  Update();
  if (updates)
    timer_id = XtAppAddTimeOut(app, update_time, TimerCB, NULL);
}

void RereadCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  if (!filepane && !treepane)
    return;
  XmUpdateDisplay(app_shell);
  if (updates)
    XtRemoveTimeOut(timer_id);
  Restat();
  if (updates)
    timer_id = XtAppAddTimeOut(app, update_time, TimerCB, NULL);
}

void UnmountCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  DirPtr dir = curdir;

  if (!filepane && !treepane)
    return;
  XmUpdateDisplay(app_shell);
  Wait();
  if (dirDev(dir) == NONE || dirDev(dir) == ANY) return;
  while (dirParent(dir) && dirDev(dirParent(dir)) == dirDev(dir))
    dir = dirParent(dir);
  dirUnmountAll(dir);
  shelf_modified = dirModified(shelfdir);
  if (shelf_modified)
    dirUpdate(shelfdir);
  cur_check();
  UpdatePanes(shelf_modified || !dirIsReadable(shelfdir), True, cur_changed ||
	      !dirIsReadable(curdir), True, True, !cur_changed &&
	      dirIsReadable(curdir));
  Done();
}

void UnmountAllCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  if (!filepane && !treepane)
    return;
  XmUpdateDisplay(app_shell);
  Wait();
  dirUnmountAll(rootdir);
  dirUnmountAll(shelfdir);
  cur_check();
  UpdatePanes(!dirIsReadable(shelfdir), True, cur_changed ||
	      !dirIsReadable(curdir), True, True, !cur_changed &&
	      dirIsReadable(curdir));
  Done();
}

/* ----------------------------------------------------------------------- */

void AbsolutePathsCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  absolute = !absolute;
}

void TargetDirCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  dirtarget = !dirtarget;
}

void PushDirCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  pushdir = !pushdir;
}

void EchoCommandsCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  echo = !echo;
}

void CheckMountsCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  check = !check;
}

void BackupsCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  backups = !backups;
}

void AutoUpdatesCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  if (updates) {
    XtRemoveTimeOut(timer_id);
    updates = False;
  } else {
    updates = update > 0;
    if (updates)
      timer_id = XtAppAddTimeOut(app, update_time, TimerCB, NULL);
  }
}

void ConfirmDropCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  confirm_drop = !confirm_drop;
}

void ConfirmMoveCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  confirm_move = !confirm_move;
}

void ConfirmCopyCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  confirm_copy = !confirm_copy;
}

void ConfirmLinkCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  confirm_link = !confirm_link;
}

void ConfirmDeleteCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  confirm_delete = !confirm_delete;
}

void ConfirmDelDirCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  confirm_deldir = !confirm_deldir;
}

void ConfirmDelFileCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  confirm_delfile = !confirm_delfile;
}

void ConfirmOverwriteCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  confirm_overwrt = !confirm_overwrt;
}

void ConfirmQuitCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  confirm_quit = !confirm_quit;
}

#define MAXLEN 1024
#define IDLINE_START \
  "!!! start of xplore resource section -- DO NOT EDIT THIS LINE !!!\n"
#define IDLINE_END \
  "!!! end of xplore resource section -- DO NOT EDIT THIS LINE !!!\n"

void SaveSetupCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  String tmpfile = tmpnam(NULL);
  char buf[MAXLEN+1];
  FILE *src, *dest;
  Dimension height, width;

  Wait();
  if (!tmpfile || !(dest = fopen(tmpfile, "w"))) {
    syserr(ERR_CREATE_RESFILE);
    Done();
    return;
  }
  if ((src = fopen(resfile, "r"))) {
    while (!ferror(src) && !feof(src) && !ferror(dest) &&
	   fgets(buf, MAXLEN, src) && strcmp(buf, IDLINE_START))
      fputs(buf, dest);
    while (!ferror(src) && !feof(src) && fgets(buf, MAXLEN, src) &&
	   strcmp(buf, IDLINE_END))
      ;
    if (ferror(src) || ferror(dest)) {
      syserr(ERR_CREATE_RESFILE);
      fclose(src); fclose(dest);
      Done();
      return;
    }
  }
  fputs(IDLINE_START, dest);
  fprintf(dest, "! geometry\n");
  XtVaGetValues(app_shell, XmNheight, &height, XmNwidth, &width, NULL);
  fprintf(dest, "xplore.width: %d\n", width);
  fprintf(dest, "xplore.height: %d\n", height);
  XtVaGetValues(tree_pane, XmNheight, &height, XmNwidth, &width, NULL);
  fprintf(dest, "xplore*tree_pane.width: %d\n", width);
  fprintf(dest, "xplore*tree_pane.height: %d\n", height);
  XtVaGetValues(file_pane, XmNheight, &height, XmNwidth, &width, NULL);
  fprintf(dest, "xplore*file_pane.width: %d\n", width);
  fprintf(dest, "xplore*file_pane.height: %d\n", height);
  XtVaGetValues(shelf_pane, XmNheight, &height, XmNwidth, &width, NULL);
  fprintf(dest, "xplore*shelf_pane.width: %d\n", width);
  fprintf(dest, "xplore*shelf_pane.height: %d\n", height);
  fprintf(dest, "! current directory and shelf\n");
  fprintf(dest, "xplore.curdir: %s\n", dirName(curdir));
  fprintf(dest, "xplore.curshelf: %s\n", shelf_label(shelf_no));
  fprintf(dest, "! view options\n");
  fprintf(dest, "xplore.updir: %s\n", (dirOptions(curdir) & INCLUDE_UPDIR)?
	  "True":"False");
  fprintf(dest, "xplore.hidden: %s\n", (dirOptions(curdir) & INCLUDE_HIDDEN)?
	  "True":"False");
  fprintf(dest, "xplore.files: %s\n", (dirOptions(curdir) & INCLUDE_FILES)?
	  "True":"False");
  fprintf(dest, "xplore.dirs: %s\n", (dirOptions(curdir) & INCLUDE_DIRS)?
	  "True":"False");
  fprintf(dest, "xplore.sort: %s\n",
	  (dirOptions(curdir) & SORT_BY_SIZE)?"BySize":
	  ((dirOptions(curdir) & SORT_BY_DATE)?"ByDate":
	   "ByName"));
  fprintf(dest, "xplore.reverse: %s\n", (dirOptions(curdir) & REVERSE)?
	  "True":"False");
  fprintf(dest, "xplore.dirsfirst: %s\n", (dirOptions(curdir) & DIRS_FIRST)?
	  "True":"False");
  fprintf(dest, "xplore.rescan: %s\n", (dirOptions(curdir) & CHECK_FILES)?
	  "True":"False");
  fprintf(dest, "xplore.magic: %s\n", (dirOptions(curdir) & NOMAGIC)?
	  "False":"True");
  fprintf(dest, "! global options\n");
  fprintf(dest, "xplore.updates: %s\n", updates?"True":"False");
  fprintf(dest, "xplore.view: %s\n", (viewtype == IconView)?"Icons":"Text");
  fprintf(dest, "xplore.tree: %s\n", treepane?"True":"False");
  fprintf(dest, "xplore.file: %s\n", filepane?"True":"False");
  fprintf(dest, "xplore.shelf: %s\n", shelfpane?"True":"False");
  fprintf(dest, "xplore.absolute: %s\n", absolute?"True":"False");
  fprintf(dest, "xplore.pushdir: %s\n", pushdir?"True":"False");
  fprintf(dest, "xplore.dirtarget: %s\n", dirtarget?"True":"False");
  fprintf(dest, "xplore.check: %s\n", check?"True":"False");
  fprintf(dest, "xplore.echo: %s\n", echo?"True":"False");
  fprintf(dest, "xplore.backups: %s\n", backups?"True":"False");
  fprintf(dest, "! confirmation options\n");
  fprintf(dest, "xplore.drop: %s\n", confirm_drop?"True":"False");
  fprintf(dest, "xplore.move: %s\n", confirm_move?"True":"False");
  fprintf(dest, "xplore.copy: %s\n", confirm_copy?"True":"False");
  fprintf(dest, "xplore.link: %s\n", confirm_link?"True":"False");
  fprintf(dest, "xplore.delete: %s\n", confirm_delete?"True":"False");
  fprintf(dest, "xplore.deldir: %s\n", confirm_deldir?"True":"False");
  fprintf(dest, "xplore.delfile: %s\n", confirm_delfile?"True":"False");
  fprintf(dest, "xplore.overwrt: %s\n", confirm_overwrt?"True":"False");
  fprintf(dest, "xplore.quit: %s\n", confirm_quit?"True":"False");
  fputs(IDLINE_END, dest);
  if (src) {
    while (!ferror(src) && !feof(src) && !ferror(dest) &&
	   fgets(buf, MAXLEN, src))
      fputs(buf, dest);
    if (ferror(src) || ferror(dest)) {
      syserr(ERR_CREATE_RESFILE);
      fclose(src); fclose(dest);
      Done();
      return;
    }
    fclose(src);
  }
  if (fclose(dest))
    syserr(ERR_CREATE_RESFILE);
  else {
    unlink(resfile);
    if (rename(tmpfile, resfile))
      syserr(ERR_CREATE_RESFILE);
    else if (rescmd && *rescmd) {
      /* merge the new resources into the database: */
      char *cmd = alloca(strlen(rescmd)+strlen(resfile)+1);
      sprintf(cmd, rescmd, resfile);
      fileExec(cmd, NULL, NULL);
    }
  }
  Done();
}

static Boolean ReloadWorkProc(XtPointer p)
{
  /* looks like we have to invoke this as a background procedure s.t. old
     icons don't get disposed before the display is refreshed */
  refresh_icons();
  return True;
}

void ReloadConfigCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  int d;
  
  Wait();
  dirUnmountAll(rootdir);
  dirUnmountAll(shelfdir);
  cur_check();
  if (exists(magicfile))
    magic_parse_file(magicfile);
  else
    magic_parse_file(libmagic);
  reinit_icons();
  initdevs(); init_file_types(); init_menus();
  if (exists(configfile))
    parse_config(configfile, CPP, cpp_options);
  else
    parse_config(libconfig, CPP, cpp_options);
  if (!n_cmd_actions())
    default_cmd_menu();
  if (!n_shelves())
    default_shelf_menu(curshelf);
  restat();
  reinit();
  Done();
  XtAppAddWorkProc(app, ReloadWorkProc, NULL);
}

/* ----------------------------------------------------------------------- */

void AboutCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  about_dlg(&std_icons[ICON_XPLORE_LOGO]);
}

/* ----------------------------------------------------------------------- */

void ShelfCB(Widget w, XtPointer app_data, XgTabsCallbackStruct *tcs)
{
  int i = tcs->tab;
  if (shelfpane && i != shelf_no) {
    shelf_goto(i);
    UpdatePanes(shelf_modified, root_modified, cur_modified,
		False, True, True);
  }
}

/* ----------------------------------------------------------------------- */

void FileActionCB(Widget w, XtPointer app_data,
		  XmContainerSelectCallbackStruct *cscs)
{
  if (cscs->selected_item_count == 1) {
    int i;

    XtVaGetValues(cscs->selected_items[0], XmNuserData, &i, NULL);
    cur_push(i, absolute, pushdir);
    UpdatePanes(shelf_modified, root_modified, cur_modified,
		True, True, !cur_changed);
  }
}

void FileSelectionCB(Widget w, XtPointer app_data,
		     XmContainerSelectCallbackStruct *cscs)
{
  int i, j;

  dirDeselect(curdir, ALL, NULL);
  for (j = 0; j < cscs->selected_item_count; j++) {
    XtVaGetValues(cscs->selected_items[j], XmNuserData, &i, NULL);
    dirSelect(curdir, i, NULL);
  }
  UpdateStatusLine();
}

void ShelfActionCB(Widget w, XtPointer app_data,
		   XmContainerSelectCallbackStruct *cscs)
{
  if (cscs->selected_item_count == 1) {
    int i;

    XtVaGetValues(cscs->selected_items[0], XmNuserData, &i, NULL);
    shelf_push(i, absolute, pushdir);
    UpdatePanes(shelf_modified, root_modified, cur_modified,
		True, True, !cur_changed);
  }
}

void ShelfSelectionCB(Widget w, XtPointer app_data,
		      XmContainerSelectCallbackStruct *cscs)
{
  int i, j;

  dirDeselect(shelfdir, ALL, NULL);
  for (j = 0; j < cscs->selected_item_count; j++) {
    XtVaGetValues(cscs->selected_items[j], XmNuserData, &i, NULL);
    dirSelect(shelfdir, i, NULL);
  }
  UpdateStatusLine();
}

void TreeOutlineCB(Widget w, XtPointer app_data,
		   XmContainerOutlineCallbackStruct *cocs)
{
  DirPtr dir;

  XtVaGetValues(cocs->item, XmNuserData, &dir, NULL);

  if (dir)
    if (cocs->new_outline_state) {
      if (!dirHasSubdirInfo(dir)) {
	Wait();
	if (dirReadSubdirInfo(dir))
	  Expand(cocs->item);
	Done();
      }
    } else if (dirHasSubdirInfo(dir)) {
      Wait();
      dirFreeSubdirInfo(dir);
      cur_check();
      if (cur_changed)
	UpdatePanes(False, True, True, True, True, False);
      else
	Collapse(cocs->item);
      Done();
    }
}

void TreeSelectionCB(Widget w, XtPointer app_data,
		     XmContainerSelectCallbackStruct *cscs)
{
  if (cscs->selected_item_count == 1) {
    Widget gadget = cscs->selected_items[0];
    DirPtr dir;

    XtVaGetValues(gadget, XmNuserData, &dir, NULL);
    if (dir)
      if (dir == curdir)
	Update();
      else {
	curgadget = gadget;
	cur_chdir(dirName(dir));
	UpdatePanes(shelf_modified, root_modified, cur_modified,
		    True, True, !cur_changed);
      }
  }
}

/* ----------------------------------------------------------------------- */

void FileConvertCB(Widget w, XtPointer app_data,
		   XmConvertCallbackStruct *ccs)
{
  Atom FILE_NAME = XInternAtom(display, "FILE_NAME", False);
  Atom FILES = XInternAtom(display, "FILES", False);
  Atom TARGETS = XInternAtom(display, "TARGETS", False);
  Atom _MOTIF_EXPORT_TARGETS = XInternAtom(display, XmS_MOTIF_EXPORT_TARGETS,
					   False);

  if (ccs->target == _MOTIF_EXPORT_TARGETS ||
      ccs->target == TARGETS) {
    Atom *targets = (Atom*)MALLOC(2 * sizeof(Atom));

    targets[0] = FILES;
    targets[1] = FILE_NAME;
    ccs->value = (XtPointer) targets;
    ccs->type = XA_ATOM;
    ccs->length = 2;
    ccs->format = 8 * sizeof(XtPointer);
    ccs->status = XmCONVERT_DONE;
  } else if (ccs->target == FILES) {
    SelPtr sel = selFromDir(curdir);

    if (sel) {
      String buf = selBuffer(sel);

      ccs->value = (XtPointer) buf;
      ccs->type = FILES;
      ccs->length = strlen(buf);
      ccs->format = 8;
      ccs->status = XmCONVERT_DONE;
      selFree(sel);
    } else
      ccs->status = XmCONVERT_REFUSE;
  } else if (ccs->target == FILE_NAME) {
    SelPtr sel = selFromDir(curdir);

    if (sel && selNFiles(sel) == 1) {
      String buf = selName(sel);

      ccs->value = (XtPointer) buf;
      ccs->type = FILE_NAME;
      ccs->length = strlen(buf);
      ccs->format = 8;
      ccs->status = XmCONVERT_DONE;
      selFree(sel);
    } else
      ccs->status = XmCONVERT_REFUSE;
  } else
    ccs->status = XmCONVERT_REFUSE;
}

void ShelfConvertCB(Widget w, XtPointer app_data,
		    XmConvertCallbackStruct *ccs)
{
  Atom FILE_NAME = XInternAtom(display, "FILE_NAME", False);
  Atom FILES = XInternAtom(display, "FILES", False);
  Atom TARGETS = XInternAtom(display, "TARGETS", False);
  Atom _MOTIF_EXPORT_TARGETS = XInternAtom(display, XmS_MOTIF_EXPORT_TARGETS,
					   False);

  if (ccs->target == _MOTIF_EXPORT_TARGETS ||
      ccs->target == TARGETS) {
    Atom *targets = (Atom*)MALLOC(2 * sizeof(Atom));

    targets[0] = FILES;
    targets[1] = FILE_NAME;
    ccs->value = (XtPointer) targets;
    ccs->type = XA_ATOM;
    ccs->length = 2;
    ccs->format = 8 * sizeof(XtPointer);
    ccs->status = XmCONVERT_DONE;
  } else if (ccs->target == FILES) {
    SelPtr sel = selFromDir(shelfdir);

    if (sel) {
      String buf = selBuffer(sel);

      ccs->value = (XtPointer) buf;
      ccs->type = FILES;
      ccs->length = strlen(buf);
      ccs->format = 8;
      ccs->status = XmCONVERT_DONE;
      selFree(sel);
    } else
      ccs->status = XmCONVERT_REFUSE;
  } else if (ccs->target == FILE_NAME) {
    SelPtr sel = selFromDir(shelfdir);

    if (sel && selNFiles(sel) == 1) {
      String buf = selName(sel);

      ccs->value = (XtPointer) buf;
      ccs->type = FILE_NAME;
      ccs->length = strlen(buf);
      ccs->format = 8;
      ccs->status = XmCONVERT_DONE;
      selFree(sel);
    } else
      ccs->status = XmCONVERT_REFUSE;
  } else
    ccs->status = XmCONVERT_REFUSE;
}

void TreeConvertCB(Widget w, XtPointer app_data,
		   XmConvertCallbackStruct *ccs)
{
  Atom FILE_NAME = XInternAtom(display, "FILE_NAME", False);
  Atom FILES = XInternAtom(display, "FILES", False);
  Atom TARGETS = XInternAtom(display, "TARGETS", False);
  Atom _MOTIF_EXPORT_TARGETS = XInternAtom(display, XmS_MOTIF_EXPORT_TARGETS,
					   False);

  if (ccs->target == _MOTIF_EXPORT_TARGETS ||
      ccs->target == TARGETS) {
    Atom *targets = (Atom*)MALLOC(2 * sizeof(Atom));

    targets[0] = FILES;
    targets[1] = FILE_NAME;
    ccs->value = (XtPointer) targets;
    ccs->type = XA_ATOM;
    ccs->length = 2;
    ccs->format = 8 * sizeof(XtPointer);
    ccs->status = XmCONVERT_DONE;
  } else if (ccs->target == FILES) {
    DirPtr dir = (DirPtr) ccs->location_data;
    SelPtr sel = selFromName(dirName(dir));

    if (sel) {
      String buf = selBuffer(sel);

      ccs->value = (XtPointer) buf;
      ccs->type = FILES;
      ccs->length = strlen(buf);
      ccs->format = 8;
      ccs->status = XmCONVERT_DONE;
      selFree(sel);
    } else
      ccs->status = XmCONVERT_REFUSE;
  } else if (ccs->target == FILE_NAME) {
    DirPtr dir = (DirPtr) ccs->location_data;
    String buf = NEWSTRING(dirName(dir));

    ccs->value = (XtPointer) buf;
    ccs->type = FILE_NAME;
    ccs->length = strlen(buf);
    ccs->format = 8;
    ccs->status = XmCONVERT_DONE;
  } else
    ccs->status = XmCONVERT_REFUSE;
}

typedef struct {
  int op;
  int item;
  FilePtr file;
  SelPtr sel;
  Boolean valid;
} FileTransferRec;

typedef struct {
  int op;
  DirPtr dir;
  SelPtr sel;
  Boolean valid;
} TreeTransferRec;

static Boolean DoFileTransfer(FileTransferRec *t)
{
  if (!t || !t->valid) return True;
  cur_drop(t->item, t->op, t->sel, absolute, dirtarget);
  selFree(t->sel);
  FREE(t);
  lock = False;
  UpdatePanes(shelf_modified, root_modified, cur_modified,
	      True, True, !cur_changed);
  return True;
}

static Boolean DoShelfTransfer(FileTransferRec *t)
{
  if (!t || !t->valid) return True;
  shelf_drop(t->item, t->op, t->sel, absolute, dirtarget);
  selFree(t->sel);
  FREE(t);
  lock = False;
  UpdatePanes(shelf_modified, root_modified, cur_modified,
	      True, True, !cur_changed);
  return True;
}

#define fileMov(op) ((op==OP_COPY)?fileCopy:(op==OP_LINK)?fileLink:fileMove)

static Boolean DoTreeTransfer(TreeTransferRec *t)
{
  if (!t || !t->valid) return True;
  fileMov(t->op)(t->sel, dirName(t->dir), absolute);
  selFree(t->sel);
  FREE(t);
  lock = False;
  Update();
  return True;
}

static void FileTransferCB(Widget w, FileTransferRec *t,
			   XmSelectionCallbackStruct *scs)
{
  Atom FILE_NAME = XInternAtom(display, "FILE_NAME", False);
  Atom FILES = XInternAtom(display, "FILES", False);

  if (scs->target == FILES && t && t->valid && t->op != NONE) {
    String buf = (String) scs->value;

    t->sel = selFromBuffer(buf);
    FREE(buf);
    lock = True;
    XtAppAddWorkProc(app, (XtWorkProc)DoFileTransfer, (XtPointer)t);
    XmTransferDone(scs->transfer_id, XmTRANSFER_DONE_SUCCEED);
  } else if (scs->target == FILE_NAME && t && t->valid && t->op != NONE) {
    String buf = (String) scs->value;

    t->sel = selFromName(buf);
    FREE(buf);
    lock = True;
    XtAppAddWorkProc(app, (XtWorkProc)DoFileTransfer, (XtPointer)t);
    XmTransferDone(scs->transfer_id, XmTRANSFER_DONE_SUCCEED);
  } else {
    FREE(t);
    XmTransferDone(scs->transfer_id, XmTRANSFER_DONE_FAIL);
  }
  RestoreStatusLine();
  XmUpdateDisplay(w);
}

static void ShelfTransferCB(Widget w, FileTransferRec *t,
			   XmSelectionCallbackStruct *scs)
{
  Atom FILE_NAME = XInternAtom(display, "FILE_NAME", False);
  Atom FILES = XInternAtom(display, "FILES", False);

  if (scs->target == FILES && t && t->valid && t->op != NONE) {
    String buf = (String) scs->value;

    t->sel = selFromBuffer(buf);
    FREE(buf);
    lock = True;
    XtAppAddWorkProc(app, (XtWorkProc)DoShelfTransfer, (XtPointer)t);
    XmTransferDone(scs->transfer_id, XmTRANSFER_DONE_SUCCEED);
  } else if (scs->target == FILE_NAME && t && t->valid && t->op != NONE) {
    String buf = (String) scs->value;

    t->sel = selFromName(buf);
    FREE(buf);
    lock = True;
    XtAppAddWorkProc(app, (XtWorkProc)DoShelfTransfer, (XtPointer)t);
    XmTransferDone(scs->transfer_id, XmTRANSFER_DONE_SUCCEED);
  } else {
    FREE(t);
    XmTransferDone(scs->transfer_id, XmTRANSFER_DONE_FAIL);
  }
  RestoreStatusLine();
  XmUpdateDisplay(w);
}

static void TreeTransferCB(Widget w, TreeTransferRec *t,
			   XmSelectionCallbackStruct *scs)
{
  Atom FILE_NAME = XInternAtom(display, "FILE_NAME", False);
  Atom FILES = XInternAtom(display, "FILES", False);

  if (scs->target == FILES && t && t->valid && t->op != NONE) {
    String buf = (String) scs->value;

    t->sel = selFromBuffer(buf);
    FREE(buf);
    lock = True;
    XtAppAddWorkProc(app, (XtWorkProc)DoTreeTransfer, (XtPointer)t);
    XmTransferDone(scs->transfer_id, XmTRANSFER_DONE_SUCCEED);
  } else if (scs->target == FILE_NAME && t && t->valid && t->op != NONE) {
    String buf = (String) scs->value;

    t->sel = selFromName(buf);
    FREE(buf);
    lock = True;
    XtAppAddWorkProc(app, (XtWorkProc)DoTreeTransfer, (XtPointer)t);
    XmTransferDone(scs->transfer_id, XmTRANSFER_DONE_SUCCEED);
  } else {
    FREE(t);
    XmTransferDone(scs->transfer_id, XmTRANSFER_DONE_FAIL);
  }
  RestoreStatusLine();
  XmUpdateDisplay(w);
}

void FileDestinationCB(Widget w, XtPointer app_data,
		       XmDestinationCallbackStruct *dcs)
{
  Atom FILE_NAME = XInternAtom(display, "FILE_NAME", False);
  Atom FILES = XInternAtom(display, "FILES", False);
  XmDropProcCallbackStruct *ds = dcs->destination_data;
  Widget target = XmObjectAtPoint(w, ds->x, ds->y);
  FileTransferRec *t = (FileTransferRec*) MALLOC(sizeof(FileTransferRec));
  Atom *exportTargets;
  Cardinal numExportTargets;
  int i;

  switch (dcs->operation) {
  case XmMOVE:
    t->op = OP_MOVE; break;
  case XmCOPY:
    t->op = OP_COPY; break;
  case XmLINK:
    t->op = OP_LINK; break;
  default:
    t->op = NONE; break;
  }
  t->sel = NULL;
  t->valid = True;
  t->file = NULL;
  if (target) {
    XtVaGetValues(target, XmNuserData, &t->item, NULL);
    if (!cur_is_drop_target(t->item))
      if (dcs->flags != XmCONVERTING_SAME)
	t->item = NONE;
      else
	t->valid = False;
    else if (dcs->flags == XmCONVERTING_SAME &&
	     fileSelected(dirFile(curdir, t->item)))
      t->valid = False;
    else
      t->file = dirFile(curdir, t->item);
  } else if (dcs->flags != XmCONVERTING_SAME)
    t->item = NONE;
  else
    t->valid = False;
  XtVaGetValues(ds->dragContext, XmNexportTargets, &exportTargets,
		XmNnumExportTargets, &numExportTargets, NULL);
  for (i = 0; i < numExportTargets; i++)
    if (exportTargets[i] == FILES || exportTargets[i] == FILE_NAME)
      break;
  if (i < numExportTargets)
    XmTransferValue(dcs->transfer_id, exportTargets[i],
		    (XtCallbackProc)FileTransferCB,
		    (XtPointer)t, CurrentTime);
  else {
    FREE(t);
    XmTransferDone(dcs->transfer_id, XmTRANSFER_DONE_FAIL);
  }
}

void ShelfDestinationCB(Widget w, XtPointer app_data,
		       XmDestinationCallbackStruct *dcs)
{
  Atom FILE_NAME = XInternAtom(display, "FILE_NAME", False);
  Atom FILES = XInternAtom(display, "FILES", False);
  XmDropProcCallbackStruct *ds = dcs->destination_data;
  Widget target = XmObjectAtPoint(w, ds->x, ds->y);
  FileTransferRec *t = (FileTransferRec*) MALLOC(sizeof(FileTransferRec));
  Atom *exportTargets;
  Cardinal numExportTargets;
  int i;

  switch (dcs->operation) {
  case XmMOVE:
    t->op = OP_MOVE; break;
  case XmCOPY:
    t->op = OP_COPY; break;
  case XmLINK:
    t->op = OP_LINK; break;
  default:
    t->op = NONE; break;
  }
  t->sel = NULL;
  t->valid = True;
  t->file = NULL;
  if (target) {
    XtVaGetValues(target, XmNuserData, &t->item, NULL);
    if (!shelf_is_drop_target(t->item))
      if (dcs->flags != XmCONVERTING_SAME)
	t->item = NONE;
      else
	t->valid = False;
    else if (dcs->flags == XmCONVERTING_SAME &&
	     fileSelected(dirFile(shelfdir, t->item)))
      t->valid = False;
    else
      t->file = dirFile(shelfdir, t->item);
  } else if (dcs->flags != XmCONVERTING_SAME)
    t->item = NONE;
  else
    t->valid = False;
  XtVaGetValues(ds->dragContext, XmNexportTargets, &exportTargets,
		XmNnumExportTargets, &numExportTargets, NULL);
  for (i = 0; i < numExportTargets; i++)
    if (exportTargets[i] == FILES || exportTargets[i] == FILE_NAME)
      break;
  if (i < numExportTargets)
    XmTransferValue(dcs->transfer_id, exportTargets[i],
		    (XtCallbackProc)ShelfTransferCB,
		    (XtPointer)t, CurrentTime);
  else {
    FREE(t);
    XmTransferDone(dcs->transfer_id, XmTRANSFER_DONE_FAIL);
  }
}

void TreeDestinationCB(Widget w, XtPointer app_data,
		       XmDestinationCallbackStruct *dcs)
{
  Atom FILE_NAME = XInternAtom(display, "FILE_NAME", False);
  Atom FILES = XInternAtom(display, "FILES", False);
  XmDropProcCallbackStruct *ds = dcs->destination_data;
  Widget target = XmObjectAtPoint(w, ds->x, ds->y);
  TreeTransferRec *t = (TreeTransferRec*) MALLOC(sizeof(TreeTransferRec));
  Atom *exportTargets;
  Cardinal numExportTargets;
  int i;

  switch (dcs->operation) {
  case XmMOVE:
    t->op = OP_MOVE; break;
  case XmCOPY:
    t->op = OP_COPY; break;
  case XmLINK:
    t->op = OP_LINK; break;
  default:
    t->op = NONE; break;
  }
  t->sel = NULL;
  t->valid = True;
  if (target) {
    XtVaGetValues(target, XmNuserData, &t->dir, NULL);
    if (dcs->flags == XmCONVERTING_SAME && t->dir == tree_drag_dir)
      t->valid = False;
  } else
    t->valid = False;
  XtVaGetValues(ds->dragContext, XmNexportTargets, &exportTargets,
		XmNnumExportTargets, &numExportTargets, NULL);
  for (i = 0; i < numExportTargets; i++)
    if (exportTargets[i] == FILES || exportTargets[i] == FILE_NAME)
      break;
  if (i < numExportTargets)
    XmTransferValue(dcs->transfer_id, exportTargets[i],
		    (XtCallbackProc)TreeTransferCB,
		    (XtPointer)t, CurrentTime);
  else {
    FREE(t);
    XmTransferDone(dcs->transfer_id, XmTRANSFER_DONE_FAIL);
  }
}

void DragProcCB(Widget w, XtPointer app_data,
		XmDragProcCallbackStruct *dpcs)
{
  static unsigned char saveStatus = XmDROP_SITE_VALID;
  static unsigned char prevStatus = XmDROP_SITE_VALID;
  static DirPtr prev_dir = NULL;
  static int prev_item = NONE;
  Widget gadget = XmObjectAtPoint(w, dpcs->x, dpcs->y);
  Boolean update = False;
  DirPtr dir = NULL;
  int item = NONE;

  switch (dpcs->reason) {
  case XmCR_DROP_SITE_LEAVE_MESSAGE:
    RestoreStatusLine();
    break;
  case XmCR_DROP_SITE_ENTER_MESSAGE:
    saveStatus = dpcs->dropSiteStatus = XmDROP_SITE_VALID;
    update = True;
  case XmCR_DROP_SITE_MOTION_MESSAGE:
    if (saveStatus == XmDROP_SITE_VALID) {
      if (w == tree_scroll) {
	if (gadget)
	  XtVaGetValues(gadget, XmNuserData, &dir, NULL);
	if (dir)
	  dpcs->dropSiteStatus = XmDROP_SITE_VALID;
	else
	  dpcs->dropSiteStatus = XmDROP_SITE_INVALID;
      } else if (w == file_scroll || w == shelf_scroll) {
	dir = (w == file_scroll)?curdir:shelfdir;
	if (gadget)
	  XtVaGetValues(gadget, XmNuserData, &item, NULL);
	if (item != NONE && !(((dir==curdir)?cur_is_drop_target:
			       shelf_is_drop_target)(item)))
	  item = NONE;
	dpcs->dropSiteStatus = XmDROP_SITE_VALID;
      } else
	dpcs->dropSiteStatus = XmDROP_SITE_INVALID;
    }
    update = update || dpcs->dropSiteStatus != prevStatus ||
      dpcs->dropSiteStatus == XmDROP_SITE_VALID &&
      (dir != prev_dir || item != prev_item);
    if (update) {
      if (dpcs->dropSiteStatus == XmDROP_SITE_VALID) {
	String name, desc;
	if (item != NONE) {
	  name = fileName(dirFile(dir, item));
	  desc = comment(fileType(dirFile(dir, item)));
	} else {
	  name = dirName(dir);
	  desc = comment(dirType(dir));
	}
	if (desc) {
	  int l = strlen(name) + strlen(desc) + 3;
	  String msg = alloca((l+1) * sizeof(char));
	  sprintf(msg, "%s (%s)", name, desc);
	  StatusMessage(msg);
	} else
	  StatusMessage(name);
	prev_dir = dir;
	prev_item = item;
      } else
	RestoreStatusLine();
      prevStatus = dpcs->dropSiteStatus;
    }
    break;
  }
}

void DropFinishCB(Widget w, XtPointer app_data,
		  XmDropFinishCallbackStruct *dfcs)
{
  lock = False;
}

/* ----------------------------------------------------------------------- */

Boolean lock = False;

void TimerCB(XtPointer data, XtIntervalId *id)
{
  if (!lock) Update();
  timer_id = XtAppAddTimeOut(app, update_time, TimerCB, NULL);
}

/* ----------------------------------------------------------------------- */

void DeleteWindowCB(Widget w, XtPointer app_data, XtPointer widget_data)
{
  quit();
}

