/*
 * xvtarga.c - load routine for 'targa' format pictures
 *
 * written and submitted by:
 *     Derek Dongray    (dongray@genrad.com)
 *
 * The format read/written is actually Targa type 2 uncompressed as
 * produced by POVray 1.0
 *
 * LoadTarga(fname, pinfo)
 */
/* Modified by Leo [Leonid V. KHramov] (old@sunct2.jinr.dubna.su)
 * for using with Image Engine.
 */

/*
 * Targa Format (near as I can tell)
 *   0:
 *   1: colormap type
 *   2: image type  (1=colmap RGB, 2=uncomp RGB, 3=uncomp gray)
 *   3: 
 *   4: 
 *   5: colormap_length, low byte
 *   6: colormap_length, high byte
 *   7: bits per cmap entry     (8, 24, 32)
 *
 *  12: width, low byte
 *  13: width, high byte
 *  14: height, low byte
 *  15: height, high byte
 *  16: bits per pixel (8, 24)
 *  17: flags  
 */



#include "picinfo.h"



/*******************************************/
int LoadTarga(char *fname,PICINFO* pinfo)
/*******************************************/
{
  /* returns '1' on success */

  long filesize;
  char *bname;
  FILE  *fp;
  int    i, row, c, c1, w, h, r, g, b, flags, intlace, topleft, trunc;
  byte *pic24, *pp;

  bname = fname;

  pinfo->pic     = (byte *) NULL;
  pinfo->comment = (char *) NULL;

  fp=fopen(fname,"r");
  if (!fp) {
     fprintf(stderr,"TARGA: %s:  %s", bname, "can't open file");
     return 0;
   }

  /* compute file length */
  fseek(fp, 0L, 2);
  filesize = ftell(fp);
  fseek(fp, 0L, 0);

  if (filesize < 18) {
     fclose(fp);
     fprintf(stderr,"TARGA: %s:  %s", bname, "file is too short");
     return 0;
   }

  /* Discard the first few bytes of the file.
     The format check has already been done or we wouldn't be here. */

  for (i=0; i<12; i++) {
    c=getc(fp);
  }


  /* read in header information */
  c=getc(fp); c1=getc(fp);
  w = c1*256 + c;

  c=getc(fp); c1=getc(fp);
  h = c1*256 + c;

  if (w<1 || h<1) {
    fclose(fp);
    fprintf(stderr,"TARGA: %s:  error in Targa header (bad image size)", bname);
    return 0;
  }

  c=getc(fp);
  if (c!=24)  {
    fclose(fp);
    fprintf(stderr,"TARGA: %s:  unsupported type (not 24-bit)", bname);
    return 0;
  }

  flags   = getc(fp);
  topleft = (flags & 0x20) >> 5;
  intlace = (flags & 0xc0) >> 6;


  pic24 = (byte *) calloc((size_t) w*h*3, (size_t) 1);
  if (!pic24) {fprintf(stderr,"TARGA: couldn't malloc 'pic24'");return 0;}


  trunc = 0;

  /* read the data */
  for (i=0; i<h; i++) {
    if (intlace == 2) {        /* four pass interlace */
      if      (i < (1*h) / 4) row = 4 * i;
      else if (i < (2*h) / 4) row = 4 * (i - ((1*h)/4)) + 1;
      else if (i < (3*h) / 4) row = 4 * (i - ((2*h)/4)) + 2;
      else                    row = 4 * (i - ((3*h)/4)) + 3;
    }

    else if (intlace == 1) {   /* two pass interlace */
      if      (i < h / 2) row = 2 * i;
      else                row = 2 * (i - h/2) + 1;
    }
    
    else row = i;              /* no interlace */



    if (!topleft) row = (h - row - 1);     /* bottom-left origin: invert y */


    c = fread(pic24 + (row*w*3), (size_t) 1, (size_t) w*3, fp);
    if (c != w*3) trunc=1;
  }

  if (trunc) fprintf(stderr,"TARGA: %s:  File appears to be truncated.",bname);


  /* swap R,B values (file is in BGR, pic24 should be in RGB) */
  for (i=0, pp=pic24; i<w*h; i++, pp+=3) {
    c = pp[0];  pp[0] = pp[2];  pp[2] = c;
  }


  pinfo->pic     = pic24;
  pinfo->type    = PIC24;
  pinfo->w       = w;
  pinfo->h       = h;
  pinfo->frmType = F_TARGA;
  sprintf(pinfo->fullInfo,"Targa, uncompressed RGB. %dx%d (%ld bytes)",w,h, filesize);
  sprintf(pinfo->shrtInfo,"%dx%d Targa.", w,h);
  pinfo->colType = F_FULLCOLOR;

  fclose(fp);

  return 1;
}


