/* $XConsortium: queue-test.cc /main/2 1996/12/30 16:34:43 swick $ */

/*
Copyright (c) 1996  X Consortium

Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE X CONSORTIUM BE LIABLE FOR ANY CLAIM, DAMAGES OR
OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
OTHER DEALINGS IN THE SOFTWARE.

Except as contained in this notice, the name of the X Consortium shall
not be used in advertising or otherwise to promote the sale, use or
other dealings in this Software without prior written authorization
from the X Consortium.
*/

#include <Xmd.h>
#include <Xa/atomdefs.h>
#include <Xa/errors.h>
#ifdef sun
#include <memory.h>
#endif
#include "intbuffer.h"
#include <stdio.h>

#define DATA_SIZE 8000
char dataBuf[DATA_SIZE];

void
QueueInitTestData(char *testData, int testDataSize)
{
    // sawtooth wave, should be sufficient
    for (int i=0;i<testDataSize;i++)
    {
	testData[i] = i & 0xFF;
    }
}

/*
  Methodology:
  Create an audio buffer collection (queue). Add to the queue a series of
  audio data buckets, ascending in time. Read the buckets out again; expect
  to receive the buckets. After reading out all buckets, ensure that the
  reference count for the underlying internal buffer is 1.
*/
XaBoolean
QueueSerialInOutTest(void)
{
    QueueInitTestData(dataBuf, DATA_SIZE);

    XaInternalBuffer *buf = new XaInternalBuffer(DATA_SIZE, dataBuf);
    XaAudioBufferCltn *cltn = new XaAudioBufferCltn;
    XaAudioBuffer *aBuf;

    XaBoolean testPassed = XaTrue;

    // Feed buckets to the queue
    XaTime currTime;
    XaTime maxTime = 50000;
    XaTime stepTime = 500;
    for(currTime = 0; (currTime < maxTime) && testPassed; currTime += stepTime)
    {
	// Create a new audio buffer based on the internal buffer.
	aBuf = new XaAudioBuffer(buf, currTime, stepTime, DATA_SIZE,
				 0);
	if (aBuf == NULL)
	{
	    // Couldn't create the new audio buffer, fail.
	    fprintf(stderr,"New XaAudioBuffer at time %ld was NULL!\n",
		    currTime);
	    testPassed = XaFalse;
	}
	else
	{
	    // Add the buffer to the queue.
	    cltn->insert(*aBuf);
	}
    }

    // Now attempt to remove the audio buffers.
    for(currTime = 0; (currTime < maxTime) && testPassed; currTime += stepTime)
    {
	aBuf = cltn->remove(currTime);
	if (aBuf == NULL)
	{
	    // Couldn't get the audio buffer, fail.
	    fprintf(stderr,"XaAudioBuffer at time %ld in the queue was NULL.\n",
		    currTime);
	    testPassed = XaFalse;
	}
	else
	{
	    // Got the audio buffer, compare values and delete the buffer.
	    if (aBuf->AtTime() != currTime)
	    {
		fprintf(stderr,"XaAudioBuffer at time %ld had time %ld.\n",
			currTime, aBuf->AtTime());
		testPassed = XaFalse;
	    }

	    if (aBuf->Duration() != stepTime)
	    {
		fprintf(stderr,"XaAudioBuffer at time %ld had duration %ld.\n",
			currTime, aBuf->Duration());
		testPassed = XaFalse;
	    }

	    if (aBuf->DurationInBits() != DATA_SIZE)
	    {
		fprintf(stderr,"XaAudioBuffer at time %ld had size %ld.\n",
			currTime, aBuf->DurationInBits());
		testPassed = XaFalse;
	    }

	    if (aBuf->LeftPad() != 0)
	    {
		fprintf(stderr,"XaAudioBuffer at time %ld had leftpad %d.\n",
			currTime, aBuf->LeftPad());
		testPassed = XaFalse;
	    }

	    delete aBuf;
	}
    }

    // Destroy the audio buffer collection.
    delete cltn;

    // Check the reference count of the underlying data.
    if (buf->GetRefCount() != 1)
    {
	fprintf(stderr,"Internal buffer had refcount %ld (expected 1).\n",
		buf->GetRefCount());
	testPassed = XaFalse;
    }

    return testPassed;
}


/*
  Methodology:
  Create an audio buffer collection (queue). Add to the queue a series of
  audio data buckets, ascending in time. Do the same thing, except writing
  between the time values of the initial run. Read the buckets out again;
  expect to receive the buckets. After reading out all buckets, ensure that the
  reference count for the underlying internal buffer is 1.
*/
XaBoolean
QueueInterleaveInOutTest(void)
{
    QueueInitTestData(dataBuf, DATA_SIZE);

    XaInternalBuffer *buf = new XaInternalBuffer(DATA_SIZE, dataBuf);
    XaAudioBufferCltn *cltn = new XaAudioBufferCltn;
    XaAudioBuffer *aBuf;

    XaBoolean testPassed = XaTrue;

    // Feed buckets to the queue
    XaTime currTime;
    XaTime maxTime = 50000;
    XaTime stepTime = 1000;
    XaTime duration = stepTime / 2;
    for(int i = 0; i <= 1; i++)
    {
	for(currTime = (duration * i);
	    (currTime < maxTime) && testPassed;
	    currTime += stepTime)
	{
	    // Create a new audio buffer based on the internal buffer.
	    aBuf = new XaAudioBuffer(buf, currTime, duration, DATA_SIZE,
				     0);
	    if (aBuf == NULL)
	    {
		// Couldn't create the new audio buffer, fail.
		fprintf(stderr,"New XaAudioBuffer at time %ld was NULL!\n",
			currTime);
		testPassed = XaFalse;
	    }
	    else
	    {
		// Add the buffer to the queue.
		cltn->insert(*aBuf);
	    }
	}
    }
	
    // Now attempt to remove the audio buffers.
    stepTime = duration;
    for(currTime = 0; (currTime < maxTime) && testPassed; currTime += stepTime)
    {
	aBuf = cltn->remove(currTime);
	if (aBuf == NULL)
	{
	    // Couldn't get the audio buffer, fail.
	    fprintf(stderr,"XaAudioBuffer at time %ld in the queue was NULL.\n",
		    currTime);
	    testPassed = XaFalse;
	}
	else
	{
	    // Got the audio buffer, compare values and delete the buffer.
	    if (aBuf->AtTime() != currTime)
	    {
		fprintf(stderr,"XaAudioBuffer at time %ld had time %ld.\n",
			currTime, aBuf->AtTime());
		testPassed = XaFalse;
	    }

	    if (aBuf->Duration() != duration)
	    {
		fprintf(stderr,"XaAudioBuffer at time %ld had duration %ld.\n",
			currTime, aBuf->Duration());
		testPassed = XaFalse;
	    }

	    if (aBuf->DurationInBits() != DATA_SIZE)
	    {
		fprintf(stderr,"XaAudioBuffer at time %ld had size %ld.\n",
			currTime, aBuf->DurationInBits());
		testPassed = XaFalse;
	    }

	    if (aBuf->LeftPad() != 0)
	    {
		fprintf(stderr,"XaAudioBuffer at time %ld had leftpad %d.\n",
			currTime, aBuf->LeftPad());
		testPassed = XaFalse;
	    }

	    delete aBuf;
	}
    }

    // Destroy the audio buffer collection.
    delete cltn;
    
    // Check the reference count of the underlying data.
    if (buf->GetRefCount() != 1)
    {
	fprintf(stderr,"Internal buffer had refcount %ld (expected 1).\n",
		buf->GetRefCount());
	testPassed = XaFalse;
    }

    return testPassed;
}

XaBoolean
QueueOverlayInOutTest(void)
{
    return XaTrue;
}

main (int argc, char **argv)
{
    printf("Serial queue test %s.\n", QueueSerialInOutTest() ?
	"PASSED" : "FAILED");
    printf("Interleaved queue test %s.\n", QueueInterleaveInOutTest() ?
	"PASSED" : "FAILED");
}

