/* $XConsortium: tagrange.cc /main/4 1996/12/30 16:34:25 swick $ */

/*
Copyright (c) 1996  X Consortium

Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE X CONSORTIUM BE LIABLE FOR ANY CLAIM, DAMAGES OR
OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
OTHER DEALINGS IN THE SOFTWARE.

Except as contained in this notice, the name of the X Consortium shall
not be used in advertising or otherwise to promote the sale, use or
other dealings in this Software without prior written authorization
from the X Consortium.
*/

// tagrange.cc
//
// Interface for handing out tags.
//
//
// A note on implementation:
//
// Tag ranges are a factory of XaTags
// managing tags between a range of tag endpoints.
// getTag obtains a new tag not already gotten,
// releaseTag makes the releaseTag available
// for getting again.
//
// Tags ranges are monotonic increasing from the
// start to the stop tag. To implement the get
// release behaviour two variables are used.
// A "current tag" which represents the highwater
// mark on returned tags,  and a "reuse bag" which
// is a collection of released tags.
// When a tag is released it is put into the reuse bag.
// When a new tag is obtained, the reuse bag is checked
// first to see if there are tags available for reuse
// and if not then the "current pointer" is used and
// then incremented.
//
// The factory interface works analogously:
// The current pointer represents the start tag
// for a new range, and the reuseBag keeps reusable
// start tags. There is no "release" method for
// ranges; The XaRange destructor
// takes care of adding start tags to the
// rangeReuse bag.
//
// Note: this design requires constant sized ranges,
// for the factory interface to work properly.
//
// Expected Useage paterns
// If Tags in a particular range for fairly long lived
// objects, then Tags should get parcelled out incrementally
// as the "current pointer" gets bumped up. If the TAG
// pointer overflows then TAGS have to be released to
// get more.
//
// If tags are gotten then quickly released then many of
// the tags will be reused, and the "current pointer" will
// only grow slowly.
//
// Once the "current pointer" overflows all tags are obtained
// from the reuse bag.
//
// The reason we don't create all of the TAGS and just put them
// in the bag, is because of the overhaed of creating all of the
// tags. The time for manipulating 1 tag in a bag is negligeable,
// as is the time for get a new tag off of the "current pointer".
//
// Since a XaCtag has to be created for each bag entry, the size
// of the XaTagRange is based on the number of number of releases
// versus gets done at any particular time.
// If a range has a large number of gets done, then a large number
// of releaes without accompanying gets the reuse bag will get big.
//

#include "tagrange.h"
#include "class.h"
#include <assert.h>
#include "clientconn.h"
#include <memory.h>


/////////////////////////////////////////////////////////////////////
//
// XaTagRange
//
//////////////////////////////////////////////////////////////////////

static void *
XaTagRangeCreate(void *conn, void *c, XaTag objectTag, XaTag classID, XaTag name)
{
    if (objectTag != XaTnone)
	return NULL;

    XaMasterConnection * masterConnection;
    if (((XaConnection *)conn)->Narrow("XaMasterConnection")) {
	masterConnection = (XaMasterConnection *) conn;
    } else if (((XaConnection *)conn)->Narrow("XaClientConnection")) {
	masterConnection = ((XaClientConnection *) conn)->MasterConnection();
    } else
	return NULL;

    return masterConnection->NewTagRange((XaConnection *)conn, (XaClass*)c,
					 classID, name);
}


static XaAttrInitRec XaTagRangeAttrInit[] =
{
    { 
        XaNempty,        	XaNbool,
                                XA_NULL,        
                                (void *)XaFalse,
                                XaMODE_G
    }
};


static XaClassInitRec XaTagRangeClassInit =
{
    XaNtagRange, XaNobject,
    XaTagRangeCreate,
    XaNumber(XaTagRangeAttrInit),
    XaMODE_CSG, XaMODE_CSG,
    XA_NULL, XA_NULL, XA_NULL, XA_NULL, XA_NULL
};

XaErrorCode XaCreateClassTagRange(XaConnection *conn)
{
    XaClass * c = new XaClass(conn, XaAtagRange, &XaTagRangeClassInit,
			      XaTagRangeAttrInit);
    return (c ? XaESuccess : XaEFailure);
}


XaTagRange::XaTagRange(XaConnection * connection, XaClass * classPtr,
		       XaTag objectTag, XaTag theClassID, XaTag name,
		       XaTagRangeFactory & fact) :
    XaObject(connection, classPtr, objectTag, theClassID, name),
    start(objectTag + 1), stop(start + fact.RangeSize()), factory(fact)
{
    memset(inUseMap, 0, sizeof(inUseMap));
}

XaTagRange::~XaTagRange()
{
    // XXX tkr what should happen to all the objects in a tagrange when that
    // range is destroyed?
    factory.ReleaseTagRange(Tag());
}

void XaTagRange::ReleaseTag(XaTag t)
{
    MarkInUse(t, False);
}

XaBoolean
XaTagRange::IsTagInUse(XaTag tag)
{
    tag -= start;
    int index = tag / 32;
    int bit = tag % 32;
    return (inUseMap[index] & (1 << bit)) != 0;
}

void
XaTagRange::MarkInUse(XaTag tag, XaBoolean inUse)
{
    tag -= start;
    int index = tag / 32;
    int bit = tag % 32;
    if (inUse)
	inUseMap[index] |= (1 << bit);
    else
	inUseMap[index] &= ~(1 << bit);
}

//
// TagRange Factory
//
XaTagRangeFactory::XaTagRangeFactory(XaTag theBase, XaTag theEnd,
				     CARD32 size) :
    base(theBase), end(theEnd),
    rangeSize(size), currentBase(theBase), rangeOverFlow(XaFalse) {}



// Get a new tag range - resuing oldones as you go along.
XaErrorCode XaTagRangeFactory::NewTagRange(XaTag & newTagRange)
{
    XaErrorCode e = XA_SUCCESS;
    
    if(rangeReuse.entries() == 0) {	
	if( rangeOverFlow) {
	    e = XA_FAILURE;
	    newTagRange = XaTnone;
	} else {
	    newTagRange = currentBase;

	    // Check for overflow.
	    if( (currentBase + rangeSize) < currentBase) {
		rangeOverFlow = XaTrue;
	    } else {
		currentBase += rangeSize;
	    }
	}
    } else {
	newTagRange = rangeReuse.any();
	rangeReuse.remove(newTagRange);
    }
    
    return e;
}

void XaTagRangeFactory::ReleaseTagRange(XaTag oldRange)
{
    // make sure this is on a rangeSize boundary because if it's
    // not then there's an error somewhere because all TagRanges
    // exist on a rangeSize boundary
    assert (((oldRange - base) % rangeSize) == 0);

    // let's see if this range is the last range that we handed out
    if (oldRange + rangeSize == currentBase) {
	currentBase = oldRange;
    } else {
	rangeReuse.insert(oldRange);
    }
}

XaTagRangeFactory::~XaTagRangeFactory() {
}
