/* $XConsortium: tagset.cc /main/3 1996/12/30 16:31:51 swick $ */

/*
Copyright (c) 1996  X Consortium

Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE X CONSORTIUM BE LIABLE FOR ANY CLAIM, DAMAGES OR
OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
OTHER DEALINGS IN THE SOFTWARE.

Except as contained in this notice, the name of the X Consortium shall
not be used in advertising or otherwise to promote the sale, use or
other dealings in this Software without prior written authorization
from the X Consortium.
*/

#include "tagset.h"
#include "atomdefs.h"
#include "memory.h"

//////////////////////////////////////////////////////////////////////
//
// XaTagSet
//
//////////////////////////////////////////////////////////////////////


XaTagSet::XaTagSet(unsigned numBuckets) :
    allocSize(numBuckets),
    tagArray(new XaTag[numBuckets]),
    nEntries(0)
{
    nBits = 1;
    while ((numBuckets & 1) == 0) {
	nBits <<= 1;
	numBuckets >>= 1;
    }
    clear();
}


XaTagSet::XaTagSet(const XaTagSet & other) :
    tagArray(new XaTag[other.allocSize]),
    allocSize(other.allocSize),
    nBits(other.nBits),
    nEntries(0)
{
    for (int i = 0; i < allocSize; i++) {
	if (other.tagArray[i])
	    insert(other.tagArray[i]);
    }
}


XaTagSet::~XaTagSet()
{
    delete [] tagArray;
}


XaTag
XaTagSet::any() const
{
    if (nEntries == 0)
	return XaTnone;

    int i = 0;
    while (1) {
	if (tagArray[i])
	    return tagArray[i];
	i++;
    }
}

unsigned
XaTagSet::find(XaTag theTag) const
{
    unsigned mask = allocSize - 1;
    unsigned index = hash(theTag);
    index &= mask;
    XaTag entry;

    // search until we find an empty slot or the tag.
    while (entry = tagArray[index]) {
	if (entry == theTag)
	    return index;
	index = (index + 1) & mask;
    }
    return XaNotFound;
}

XaBoolean   XaTagSet::contains(XaTag theTag) const
{
    return find(theTag) != XaNotFound;
}

XaBoolean	XaTagSet::insert( XaTag theTag)
{
    if (contains(theTag))
	return XaFalse;
    if (4 * nEntries > allocSize * 3) { // enlarge when 3/4 full
	doubleSize();
    }
    nEntries++;
    unsigned mask = allocSize - 1;
    unsigned index = hash(theTag);
    index &= mask;
    while (tagArray[index] != XaTnone) {
	index = (index + 1) & mask;
    }
    tagArray[index] = theTag;
    return XaTrue;
}

XaBoolean
XaTagSet::remove(XaTag theTag)
{
    unsigned index = find(theTag);
    if (index == XaNotFound)
	return XaFalse;

    // first remove the entry we are looking for
    unsigned mask = allocSize - 1;
    unsigned startSlot = hash(theTag) & mask;;
    unsigned emptySlot = index;
    XaTag entry;

    // first find the first empty slot before the hash slot of the tag we are removing
    while (tagArray[startSlot]) {
	startSlot = (startSlot - 1) & mask;
    }

    tagArray[index] = XaTnone;

    // walk over the array and move tags back to the last
    //empty slot if they could have hashed into that slot.
    // stop when you hit an empty slot
    index = (index + 1) & mask;
    while (entry = tagArray[index]) {
	unsigned h = hash(entry) & mask;
	if (startSlot > emptySlot) {
	    if (h >= startSlot || h <= emptySlot) {
		tagArray[emptySlot] = entry;
		tagArray[index] = XaTnone;
		emptySlot = index;
	    }
	} else {
	    if (h >= startSlot && h <= emptySlot) {
		tagArray[emptySlot] = entry;
		tagArray[index] = XaTnone;
		emptySlot = index;
	    }
	}
	
	index = (index + 1) & mask;
    }
    nEntries--;
    return validate();
}

void 	XaTagSet::clear(void)
{
    memset(tagArray, 0, sizeof(XaTag) * allocSize);
    nEntries = 0;
}

XaTagSet&
XaTagSet::operator=(const XaTagSet& other)
{
    this->clear();
    for (int i = 0; i < allocSize; i++)
	if (other.tagArray[i])
	    this->insert(other.tagArray[i]);
    return *this;
}

XaBoolean
XaTagSet::validate()
{
    int entries = 0;
    for (int i = 0; i < allocSize; i++) {
	if (tagArray[i]) {
	    entries ++;
	    if (find(tagArray[i]) == XaNotFound) {
		return XaFalse;
	    }
	}
    }
    if (entries != nEntries)
	return XaFalse;
    return XaTrue;
}

void 	XaTagSet::doubleSize(void)
{
    int oldnEntries = nEntries;
    XaTag * oldArray = tagArray;

    tagArray = new XaTag[allocSize * 2];
    allocSize *= 2;
    nBits << 1;
    nEntries = 0;
    clear();

    int i = 0;
    while (oldnEntries) {
	if (oldArray[i]) {
	    oldnEntries--;
	    insert(oldArray[i]);
	}
	i++;
    }

    delete [] oldArray;
}

XaTag    	XaTagSet::at(unsigned index) const
{
    return tagArray[index];
}

XaTag     XaTagSet::replaceAtIndex(unsigned index, 
				   XaTag theTag)
{
    XaTag old = tagArray[index];
    tagArray[index] = theTag;
    return old;
}

unsigned
XaTagSet::hash(XaTag t) const
{
    return t;
}
