/* $XConsortium: wavfile.c /main/3 1996/12/30 16:30:02 swick $ */

/*
Copyright (c) 1996  X Consortium

Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE X CONSORTIUM BE LIABLE FOR ANY CLAIM, DAMAGES OR
OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
OTHER DEALINGS IN THE SOFTWARE.

Except as contained in this notice, the name of the X Consortium shall
not be used in advertising or otherwise to promote the sale, use or
other dealings in this Software without prior written authorization
from the X Consortium.
*/

/*////////////////////////////////////////////////////////
//
// Implementation of XaWavFile
//
///////////////////////////////////////////////////////*/

#include <malloc.h>
#include <math.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <Xa/Xalib.h>
#include <Xa/atomdefs.h>
#include "cobject.h"
#include "filedefs.h"
#include "wav.h"
#include "errors.h"


static const char endianTest[4] = { 0, 0, 0, 1 };


/* Read long, little-endian: little end first. VAX/386 style. */
Card32
rllong (int fd)
{
    if (*(Int32*)endianTest == 1)
    {
	unsigned char uc, uc2, uc3, uc4;

	read (fd, &uc, 1);
	read (fd, &uc2, 1);
	read (fd, &uc3, 1);
	read (fd, &uc4, 1);
	return ((Card32)uc4 << 24) | ((Card32)uc3 << 16)
			| ((Card32)uc2 << 8) | (Card32)uc;
    }
    else
    {
	Card32 retVal;
	read (fd, &retVal, 4);
	return retVal;
    }
}

/* Read short, little-endian: little end first. VAX/386 style. */
unsigned short
rlshort(int fd)
{
    if (*(Int32*)endianTest == 1)
    {
	unsigned char uc, uc2;
	read (fd, &uc, 1);
	read (fd, &uc2, 1);
	return (uc2 << 8) | uc;
    }
    else
    {
	unsigned short retVal;
	read (fd, &retVal, 2);
	return retVal;
    }
}

/* Write short, little-endian: little end first. VAX/386 style. */
unsigned short
wlshort(int fd, unsigned short us)
{
    if (*(Int32*)endianTest == 1)
    {
        unsigned char datum[2];

	datum[0] = (us) & 0xff;
	datum[1] = us >> 8;
	if (write (fd, &datum, 2) != 2)
	  return 0;
	else
	  return 1;
    }
    else
    {
	if (write (fd, &us, 2) != 2)
	  return 0;
	else
	  return 1;
    }
}

/* Write long, little-endian: little end first. VAX/386 style. */
Card32
wllong(int fd, Card32 ul)
{
    if (*(Int32*)endianTest == 1)
    {
        unsigned char datum[4];

        datum[0] = (ul) & 0xff;
	datum[1] = (ul >> 8) & 0xff;
	datum[2] = (ul >> 16) & 0xff;
	datum[3] = (ul >> 24) & 0xff;
        if (write (fd, &datum, 4) != 4)
          return 0;
        else
	  return 1;
    }
    else
    {
        if (write (fd, &ul, 4) != 4)
          return 0;
        else
	  return 1;
    }
}

static int
decode_wav_header (int fd, Audio_hdr *hdr)
{
    unsigned char   buf[sizeof (BUFSIZ)];
    Int32           length, samples;
    unsigned short  nbits;
    unsigned        nread = 0;
    unsigned        wav_type;
    off_t           err;
    int             nbytes;
    char            c;
    
    /*    
    // Read the magic number.
    */
    nbytes = read (fd, (char *)buf, 4);  nread += 4;
    if (nbytes != 4 || strncmp ("RIFF", (char *)buf, 4))
      return 0;
    
    /*
    // Read the next 4 bytes in little-endian
    */
    hdr->data_size = rllong(fd);  nread +=4;
    
    /*    
    // Read another magic number.
    */
    nbytes = read (fd, (char *)buf, 4);  nread += 4;
    if (nbytes != 4 || strncmp ("WAVE", (char *)buf, 4))
      return 0;
    
    /*
    // Skip to the next "fmt " or end of file.
    */
    while (1) {
	nbytes = read (fd, buf, 4);  nread += 4;
	 if (nbytes == 0)
           return 0;
	 else if ((nbytes == 4) && !strncmp ("fmt ", (char *)buf, 4))
           break;
	 length = rllong (fd);  nread += 4;
	 for (length; length > 0; length--) {
           read (fd, &c, 1); nread += 1;
         }
       }
    
    /*
    // Read the data size.
    */
    length = rllong (fd);  nread += 4;
    
    /*
    // Read the wave format type.
    // We can only do WAVE_FORMAT_PCM now, the
    // rest are proprietary.
    */
    wav_type = (unsigned)rlshort (fd);  nread += 2;
    if (wav_type != WAVE_FORMAT_PCM)
        return 0;
    /*
    // Assume PCM for now.
    */
    else
      hdr->encoding = XaAencodeLinear;

    /*
    // Assume 1 sample per unit for now.
    */
    hdr->samples_per_unit = 1;
    
    /*
    // Read the number of channels.
    */
    hdr->channels = (unsigned)rlshort(fd);  nread += 2;
    /*
    // Read the sample_rate
    */
    hdr->sample_rate = (unsigned )rllong(fd);  nread += 4;
    
    rllong (fd);  nread += 4;
    rlshort (fd);  nread += 2;
    nbits = rlshort (fd);  nread += 2;
    hdr->bytes_per_unit = nbits / 8;
    
    /*
    // Go to the end of this data.
    */
    length -= 16;
    while (--length >= 0) {
        read (fd, &c, 1);  nread += 1;
      } 
    /*
    // Read the magic number for the start of the data
    */
    nbytes = read (fd, (char *)buf, 4);  nread += 4;

    /*
    // Skip over INFO field if found 
    */
    if (strncmp ("INFO", (char *)buf, 4) == 0) {
      length = rllong (fd); nread += 4;
      while (--length >= 0) {
	read (fd, &c, 1); nread += 1;
      }
      read (fd, buf, 4); nread += 4; /* read next 4 bytes */
    }

    if (strncmp ("data", (char *)buf, 4))
        return 0;
    /*
    // Samples and data_size.
    */
    samples = rllong (fd);  nread += 4;
    hdr->data_size = samples;
    
    /*
    // Set header size for easier reset later
    */
    hdr->hdr_size = nread;
    /*
    // Set endian
    */
    hdr->endian = XaAendianSmall;
    return 1;
}      

static int
encode_wav_header (int fd, Audio_hdr *hdr)
{
    int   samsize = (hdr->bytes_per_unit * 8);

    write (fd, "RIFF", 4);
    wllong (fd, hdr->data_size + 6 + 16 + 12);  /* Wavform chunk size: FIXUP(4) */
    write (fd, "WAVE", 4);
    write (fd, "fmt ", 4);
    wllong (fd, (Int32)16);                 /* fmt chunk size */
    wlshort (fd, WAVE_FORMAT_PCM);         /* FormatTag: WAV_FORMAT_PCM */
    wlshort (fd, hdr->channels);           /* channels */
    wllong (fd, (Card32 )hdr->sample_rate);  /* SamplesPerSec */
    wllong (fd, ((Card32)hdr->sample_rate * hdr->channels * samsize + 7) / 8); 
				/* Avg Bytes/sec, nBlockAlign */
    wlshort (fd, hdr->bytes_per_unit * samsize + 7 / 8); /* BlockAlign *?
    wlshort (fd, samsize);                 /* Sample size */
    write (fd, "data", 4); 
    wllong (fd, hdr->data_size);           /* data chunk size: FIXUP(40) */
    return 1;
}

static XaErrorCode 
wavInit(int fd, void *file, XaTag fileMode)
{
    Audio_hdr          hdr;  
    int                status;
    XaAttributeCBData  cbd;
    XaErrorCode        error;

/*
// If file was opened for reading, read in header and
// set properties.
// If file was opened for writing, write out header
// based on current properties set.
*/
    switch (fileMode) {
    case XaAopenForRead:
      /*
      // Decode wav properties into header
      */
      if (!decode_wav_header (fd, &hdr))
	return XA_FAILURE;
      
      /*
      // Store the wav properties
      */
      cbd.name = XaAendian;
      cbd.value = (void *)hdr.endian;
      error = XaSetAttributes (file, &cbd, 1);

      cbd.name = XaAsampleRate;
      cbd.value = (void *)hdr.sample_rate;
      error = XaSetAttributes (file, &cbd, 1);

      cbd.name = XaAsamplesPerUnit;
      cbd.value = (void *)hdr.samples_per_unit;
      error = XaSetAttributes (file, &cbd, 1);

      cbd.name = XaAbitsPerSample;
      cbd.value =  (void *)(hdr.bytes_per_unit * 8);
      error = XaSetAttributes (file, &cbd, 1);

      cbd.name = XaAnumChannels;
      cbd.value = (void *)hdr.channels;
      error = XaSetAttributes (file, &cbd, 1);

      cbd.name = XaAencoding;
      cbd.value = (void *)hdr.encoding;
      error = XaSetAttributes (file, &cbd, 1);

      cbd.name = XaAdataSize;
      cbd.value = (void *)hdr.data_size;
      error = XaSetAttributes (file, &cbd, 1);

      cbd.name = XaAheaderSize;
      cbd.value = (void *)hdr.hdr_size;
      error = XaSetAttributes (file, &cbd, 1);

      break;

    case XaAopenForWrite:
      /*
      // Get the properties for wav header.
      */
      cbd.name = XaAsampleRate;
      error = XaGetAttributes (file, &cbd, 1);
      if (error != XaESuccess)
	return XA_FAILURE;
      else 
        hdr.sample_rate = (unsigned)cbd.value;

      cbd.name = XaAencoding;
      error = XaGetAttributes (file, &cbd, 1);
      if (error != XaESuccess)
	return XA_FAILURE;
      else 
        hdr.encoding = (unsigned)cbd.value;
      
      cbd.name = XaAnumChannels;
      error = XaGetAttributes (file, &cbd, 1);
      if (error != XaESuccess)
	return XA_FAILURE;
      else 
        hdr.channels = (unsigned)cbd.value;
      
      cbd.name = XaAdataSize;
      error = XaGetAttributes (file, &cbd, 1);
      if (error != XaESuccess)
	return XA_FAILURE;
      else
        hdr.data_size = (unsigned)cbd.value;
      
      cbd.name = XaAbitsPerSample;
      error = XaGetAttributes (file, &cbd, 1);
      if (error != XaESuccess)
	return XA_FAILURE;
      else
	hdr.bytes_per_unit = (unsigned)cbd.value/8;

      /*
      // Write out wav header
      */
      if (!encode_wav_header (fd, &hdr))
	return XA_FAILURE;

      break;

    default:
      break;
    }

    return XA_SUCCESS;
}

fileInit (int fd, void *file)
{
    XaAttributeCBData  cbd;
    XaErrorCode        error;
    XaTag              fileMode;

    cbd.name = XaAfileMode;
    error = XaGetAttributes (file, &cbd, 1);
    if (error != XaESuccess)
      return XA_FAILURE;
    else
      fileMode = (unsigned)cbd.value;

    return wavInit(fd, file, fileMode);
}

