/* $XConsortium: sunfile.c /main/2 1996/12/30 16:29:56 swick $ */

/*
Copyright (c) 1996  X Consortium

Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE X CONSORTIUM BE LIABLE FOR ANY CLAIM, DAMAGES OR
OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
OTHER DEALINGS IN THE SOFTWARE.

Except as contained in this notice, the name of the X Consortium shall
not be used in advertising or otherwise to promote the sale, use or
other dealings in this Software without prior written authorization
from the X Consortium.
*/

/*/////////////////////////////////////////////////////////
//
// Implementation of XaSunFile
//
////////////////////////////////////////////////////////*/

#include <malloc.h>
#include <math.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include "filedefs.h"
#include "au.h"
#include "errors.h"
#include <Xa/atomdefs.h>
#include "cobject.h"

/* By convention, this should be a multiple of 4. */
#define AUDIO_INFO_FIELD       "Generated by X Audio"

/*
 * Try to decode buffer containing a Sun audio file header 
 * into an audio header.
 */
static int
decode_sun_filehdr(unsigned char *buf, Audio_hdr *hdrp, unsigned int *isize)
				        /* buffer address */
	         	      		/* output audio header */
	        	       		/* output size of info */
{
    SunFileHeader fhdr;		/* temporary file header storage */
    off_t         err;
  
    /* Decode the 32-bit integer header fields. */
    DECODE_LONG(buf, &fhdr.magic); buf += 4;
    DECODE_LONG(buf, &fhdr.hdr_size); buf += 4;
    DECODE_LONG(buf, &fhdr.data_size); buf += 4;
    DECODE_LONG(buf, &fhdr.encoding); buf += 4;
    DECODE_LONG(buf, &fhdr.sample_rate); buf += 4;
    DECODE_LONG(buf, &fhdr.channels);
    
    /* Check the magic number. */
    if (fhdr.magic == AU_SUN_MAGIC || fhdr.magic == AU_DEC_MAGIC)
	hdrp->endian = XaAendianBig;
    else if (fhdr.magic == AU_SUN_INV_MAGIC || fhdr.magic == AU_DEC_INV_MAGIC)
	hdrp->endian = XaAendianSmall;
    else
	return (1);
    
    /* Decode into the audio header structure. */
    hdrp->hdr_size = fhdr.hdr_size;
    hdrp->data_size = fhdr.data_size;
    hdrp->sample_rate = fhdr.sample_rate;
    hdrp->channels = fhdr.channels;
    
    /* Set the info field size (ie, number of bytes left before data). */
    *isize = fhdr.hdr_size - sizeof (SunFileHeader);
    
    /* Check the data encoding. */
    switch (fhdr.encoding) {
    case AUDIO_FILE_ENCODING_MULAW_8:
      hdrp->encoding = XaAencodeUlaw;
      hdrp->bytes_per_unit = 1;
      hdrp->samples_per_unit = 1;
      break;
    case AUDIO_FILE_ENCODING_ALAW_8:
      hdrp->encoding = XaAencodeAlaw;
      hdrp->bytes_per_unit = 1;
      hdrp->samples_per_unit = 1;
      break;
    case AUDIO_FILE_ENCODING_LINEAR_8:
      hdrp->encoding = XaAencodeLinear;
      hdrp->bytes_per_unit = 1;
      hdrp->samples_per_unit = 1;
      break;
    case AUDIO_FILE_ENCODING_LINEAR_16:
      hdrp->encoding = XaAencodeLinear;
      hdrp->bytes_per_unit = 2;
      hdrp->samples_per_unit = 1;
      break;
    case AUDIO_FILE_ENCODING_LINEAR_24:
      hdrp->encoding = XaAencodeLinear;
      hdrp->bytes_per_unit = 3;
      hdrp->samples_per_unit = 1;
      break;
    case AUDIO_FILE_ENCODING_LINEAR_32:
      hdrp->encoding = XaAencodeLinear;
      hdrp->bytes_per_unit = 4;
      hdrp->samples_per_unit = 1;
      break;
    case AUDIO_FILE_ENCODING_FLOAT:
      hdrp->encoding = XaAencodeFloat;
      hdrp->bytes_per_unit = 4;
      hdrp->samples_per_unit = 1;
      break;
    case AUDIO_FILE_ENCODING_DOUBLE:
      hdrp->encoding = XaAencodeFloat;
      hdrp->bytes_per_unit = 8;
      hdrp->samples_per_unit = 1;
      break;
    case AUDIO_FILE_ENCODING_ADPCM_G721:
      hdrp->encoding = XaAencodeG721;
      hdrp->bytes_per_unit = 1;
      hdrp->samples_per_unit = 2;
      break;
    case AUDIO_FILE_ENCODING_ADPCM_G723_3:
      hdrp->encoding = XaAencodeG723;
      hdrp->bytes_per_unit = 3;
      hdrp->samples_per_unit = 8;
      break;
    case AUDIO_FILE_ENCODING_ADPCM_G723_5:
      hdrp->encoding = XaAencodeG723;
      hdrp->bytes_per_unit = 5;
      hdrp->samples_per_unit = 8;
      break;
      
    default:
      return (1);
    }
    return (0);
}

static CARD32
convertEncodingValue (XaTag encoding, 
		      unsigned bytes_per_unit, 
		      unsigned samples_per_unit)
{
    CARD32 value = AUDIO_FILE_ENCODING_MULAW_8;

    if (encoding == XaAencodeUlaw  && 
	bytes_per_unit == 1 &&
	samples_per_unit == 1)
      value = AUDIO_FILE_ENCODING_MULAW_8;
    else if (encoding == XaAencodeAlaw &&
             bytes_per_unit == 1 &&
             samples_per_unit == 1)
      value = AUDIO_FILE_ENCODING_ALAW_8;
    else if (encoding == XaAencodeLinear &&
             bytes_per_unit == 1 &&
       	     samples_per_unit == 1)
      value = AUDIO_FILE_ENCODING_LINEAR_8;
    else if (encoding == XaAencodeLinear &&
             bytes_per_unit == 2 &&
             samples_per_unit == 1)
      value = AUDIO_FILE_ENCODING_LINEAR_16;
    else if (encoding == XaAencodeLinear &&
             bytes_per_unit == 3 &&
             samples_per_unit == 1)
      value = AUDIO_FILE_ENCODING_LINEAR_24;
    else if (encoding == XaAencodeLinear &&
             bytes_per_unit == 4 &&
             samples_per_unit == 1)
      value = AUDIO_FILE_ENCODING_LINEAR_32;
    else if (encoding == XaAencodeFloat &&
             bytes_per_unit == 4 &&
	     samples_per_unit == 1)
      value = AUDIO_FILE_ENCODING_FLOAT;
    else if (encoding == XaAencodeFloat &&
	     bytes_per_unit == 8 &&
	     samples_per_unit == 1)
      value = AUDIO_FILE_ENCODING_DOUBLE;
    else if (encoding == XaAencodeG721 &&
	     bytes_per_unit == 1 &&
	     samples_per_unit == 2)
      value = AUDIO_FILE_ENCODING_ADPCM_G721;
    else if (encoding == XaAencodeG723 &&
	     bytes_per_unit == 3 &&
	     samples_per_unit == 8)
      value = AUDIO_FILE_ENCODING_ADPCM_G723_3;
    else if (encoding == XaAencodeG723 &&
	     bytes_per_unit == 5 &&
	     samples_per_unit == 8)
      value = AUDIO_FILE_ENCODING_ADPCM_G723_5;

    return value;

}

static XaErrorCode
sunInit(int fd, void *file, XaTag fileMode)
{
    Audio_hdr     hdr;  
    SunFileHeader shdr;  
    unsigned char buf[sizeof (SunFileHeader)];
    off_t         err;
    XaErrorCode   error;
    unsigned      isize;
    int           nbytes;
    XaAttributeCBData cbd;

/*
// If file was opened for reading, read in header and
// set properties.
// If file was opened for writing, write out header
// based on current properties set.
*/
    switch (fileMode) {
    case XaAopenForRead:  
      /*    
      // Read the header (but not the text info).
      // Decode the hdr into an Audio_hdr.
      // Leave the fd at the beginning of the data.
      */
      nbytes = read (fd, (char *)buf, sizeof(buf));
      if ((nbytes != sizeof (buf)) || 
	  (decode_sun_filehdr (buf, &hdr, &isize) != 0)) {
	return XA_FAILURE;
      }
      /*
      // Skip over info field and
      // Set the fd at the beginning of the data.
      */
      err = lseek (fd, isize, SEEK_CUR);
      if (err < 0)
 	return XA_FAILURE;

      /*
      // Store the properties
      */
      cbd.name = XaAendian;
      cbd.value = (void *)hdr.endian;
      error = XaSetAttributes (file, &cbd, 1);

      cbd.name = XaAsampleRate;
      cbd.value = (void *)hdr.sample_rate;
      error = XaSetAttributes (file, &cbd, 1);

      cbd.name = XaAsamplesPerUnit;
      cbd.value = (void *)hdr.samples_per_unit;
      error = XaSetAttributes (file, &cbd, 1);

      cbd.name = XaAbitsPerSample;
      cbd.value =  (void *)(hdr.bytes_per_unit * 8);
      error = XaSetAttributes (file, &cbd, 1);

      cbd.name = XaAnumChannels;
      cbd.value = (void *)hdr.channels;
      error = XaSetAttributes (file, &cbd, 1);

      cbd.name = XaAencoding;
      cbd.value = (void *)hdr.encoding;
      error = XaSetAttributes (file, &cbd, 1);

      cbd.name = XaAdataSize;
      cbd.value = (void *)hdr.data_size;
      error = XaSetAttributes (file, &cbd, 1);

      cbd.name = XaAheaderSize;
      cbd.value = (void *)hdr.hdr_size;
      error = XaSetAttributes (file, &cbd, 1);

      break;

    case XaAopenForWrite:
      /*
      // Get the properties and write out header and info field
      */
      shdr.magic = AU_SUN_MAGIC;

      cbd.name = XaAsampleRate;
      error = XaGetAttributes (file, &cbd, 1);
      if (error != XaESuccess)
	return XA_FAILURE;
      else 
        shdr.sample_rate = (unsigned)cbd.value;

      cbd.name = XaAnumChannels;
      error = XaGetAttributes (file, &cbd, 1);
      if (error != XaESuccess)
	return XA_FAILURE;
      else 
        shdr.channels = (unsigned)cbd.value;
      
      cbd.name = XaAdataSize;
      error = XaGetAttributes (file, &cbd, 1);
      if (error != XaESuccess)
	shdr.data_size = ~0;
      else
        shdr.data_size = (unsigned)cbd.value;

      cbd.name = XaAsamplesPerUnit;
      error = XaGetAttributes (file, &cbd, 1);
      if (error != XaESuccess)
	return XA_FAILURE;
      else
        hdr.samples_per_unit = (unsigned)cbd.value;

      cbd.name = XaAbitsPerSample;
      error = XaGetAttributes (file, &cbd, 1);
      if (error != XaESuccess)
	return XA_FAILURE;
      else
	hdr.bytes_per_unit = (unsigned)cbd.value / 8;
    
      cbd.name = XaAencoding;
      error = XaGetAttributes (file, &cbd, 1);
      if (error != XaESuccess)
	return XA_FAILURE;
      else 
        shdr.encoding = convertEncodingValue ((XaTag)cbd.value, 
					      hdr.bytes_per_unit,
					      hdr.samples_per_unit);

      /* By convention, this should be a multiple of 4. */
      shdr.hdr_size = sizeof (SunFileHeader) + strlen (AUDIO_INFO_FIELD);
      nbytes = write (fd, (void *)&shdr, sizeof (shdr));
      if (nbytes != sizeof (shdr))
	return XA_FAILURE;
      nbytes = write (fd, (char *)AUDIO_INFO_FIELD, strlen (AUDIO_INFO_FIELD));
      if (nbytes != strlen (AUDIO_INFO_FIELD))
	return XA_FAILURE;

      break;

    default:
      break;
    }

    return XA_SUCCESS;
}

fileInit (int fd, void *file)
{
    XaAttributeCBData  cbd;
    XaErrorCode	      error;
    XaTag	      fileMode;

    cbd.name = XaAfileMode;
    error = XaGetAttributes (file, &cbd, 1);
    if (error != XaESuccess)
      return XA_FAILURE;
    else 
      fileMode = (unsigned)cbd.value;

    return sunInit(fd, file, fileMode);
}
