//This line lets emacs recognize this as -*- C++ -*- Code
/* $XConsortium: bag.h /main/5 1996/12/30 16:29:15 swick $ */

/*
Copyright (c) 1996  X Consortium

Permission is hereby granted, free of charge, to any person obtaining
a copy of this software and associated documentation files (the
"Software"), to deal in the Software without restriction, including
without limitation the rights to use, copy, modify, merge, publish,
distribute, sublicense, and sell copies of the Software, and to
permit persons to whom the Software is furnished to do so, subject to
the following conditions:

The above copyright notice and this permission notice shall be included
in all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
IN NO EVENT SHALL THE X CONSORTIUM BE LIABLE FOR ANY CLAIM, DAMAGES OR
OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE,
ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
OTHER DEALINGS IN THE SOFTWARE.

Except as contained in this notice, the name of the X Consortium shall
not be used in advertising or otherwise to promote the sale, use or
other dealings in this Software without prior written authorization
from the X Consortium.
*/

//------------------------------------------------------------------------

//   "@(#)XaBag.h	1.21\t94/06/15  "

#ifndef XA_BAG_H
#define XA_BAG_H

#include "xadefines.h"
#include "collectable.h"

// make sure this is > 2 or you may have problems in XaDictionary
#define XA_DEFAULT_COLLECTION_SIZE 16

// REMIND: this is really true only for 32-bit architectures
// REMIND: collections will fail (quietly) at 4 billion elements
const unsigned XA_COLLECTION_BAD_INDEX = 0xffffffff;

class XaBag {

  public:
    // 
    // --------- constructors and destructors
    // 

    XaBag(unsigned N = XA_DEFAULT_COLLECTION_SIZE);// Construct a bag of N items
    XaBag(const XaBag&);  
    virtual ~XaBag(void);

    // 
    // --------- Informational
    // 

    virtual unsigned	entries(void) const { return nEntries; };
    virtual XaBoolean	isEmpty(void) const;
    virtual unsigned	occurrencesOf(const XaCollectable& theObject) const;
    virtual XaBoolean   contains(const XaCollectable& theObject) const;    

    //
    // --------- Bag manipulation
    //

    // finds an instance in the bag of an object that equals() theObject
    virtual XaCollectable*	find(const XaCollectable& theObject) const;

    // inserts theObject and returns a pointer to it.  returns NULL if
    // insertion fails.
    virtual XaCollectable*	insert(XaCollectable& theObject);

    // removes object which equals() theObject and returns the bag's 
    // reference to it.  returns NULL if no such object.
    virtual void*	remove(const XaCollectable& theObject);

    // removes and delete the object.
    virtual void	removeAndDestroy(const XaCollectable& theObject);

    // Non destructive removal of all objects in bag.
    virtual void 	clear(void);

    // Destructive (delete on each object) remove of all
    // objects in bag.
    virtual void       clearAndDelete(void);

    // 
    // --------- Copy support
    // 

    //By default do a shallow copy.
    virtual XaBag&	operator=(const XaBag&);

  private:
    void 		finishConstructor(void);
    friend 		class XaBagIterator;
    friend 		class XaScalarBagIterator;
    friend 		class XaDictionary;

  protected:
    // doubles the allocated size
    XaErrorCode 	doubleSize(void);

    // returns object at the index or NULL if no object there
    virtual XaCollectable*    	at(unsigned index) const;

    // friend declaration to keep the compiler happy
    friend class XaDictionaryIterator;

    // replaces the object at the specified index
    virtual XaCollectable*     replaceAtIndex(unsigned index, 
					       XaCollectable& theObject);

    XaCollectable**   	objectArray; // array of XaCollectable object pointers
    unsigned 		allocSize;   // size of the array 
    unsigned 		nBits;	     // log2(allocSize)
    unsigned 		nEntries;    // number of entries in the bag
};


class XaBagIterator {
  public:
    XaBagIterator(const XaBag& );
    virtual ~XaBagIterator(void);

    //
    // --------- Iteration and contained object operations
    //

    // Advances iterator to the next item and returns it.  Returns NULL when
    // the end of the collection has been reached.
    virtual XaCollectable *       operator () (void);

    // Moves iterator to the next item which equals the object pointed
    // to by target, and returns it.  If no such item exists, NULL is returned
    // and the state of the iterator is then undefined.
    virtual XaCollectable *       findNext(const XaCollectable& target);
 
    // Return the item at the current iterator position.
    virtual XaCollectable *       key(void) const;

    // Reset the iterator to its initial state.
    virtual void        reset(void);
 
  protected:
    XaBag*		pBag;
    int 		currentIndex;
};

// The scalar bag class uses the hash value as a key for operations, instead
// of a collectable. Otherwise the semantics are the same as XaBag.
// This class is appropriate when the uniqueness of the object can be expressed
// as a scalar (the hash value). 

class XaScalarBag : public XaBag {
  public:
    XaScalarBag(unsigned N = XA_DEFAULT_COLLECTION_SIZE);// Construct a bag of N items
    XaScalarBag(const XaBag&);  
    ~XaScalarBag(void);

    virtual XaScalarBag&	operator=(const XaScalarBag&);
    virtual unsigned	occurrencesOf(unsigned scalar) const;
    virtual unsigned	occurrencesOf(const XaCollectable& theObject ) const;
    virtual XaBoolean   contains(unsigned scalar) const;    
    virtual XaBoolean   contains(const XaCollectable& theObject) const;    

    //
    // --------- Bag manipulation
    //

    // finds an instance in the bag of an object that equals() theObject
    virtual XaCollectable*	find(unsigned scalar) const;
    virtual XaCollectable*	find(const XaCollectable& theObject) const;

    // removes object which equals() theObject and returns the bag's 
    // reference to it.  returns NULL if no such object.
    virtual void*	remove(unsigned scalar);
    virtual void*	remove(const XaCollectable& theObject);

    // removes and delete the object.
    virtual void	removeAndDestroy(unsigned scalar);
    virtual void	removeAndDestroy(const XaCollectable& theObject);

};

// Mirrors XaBag operations
class XaScalarBagIterator : public XaBagIterator {
  public:
    XaScalarBagIterator(const XaBag& );

    // Moves iterator to the next item which equals the object pointed
    // to by target, and returns it.  If no such item exists, NULL is returned
    // and the state of the iterator is then undefined.
    virtual XaCollectable *       findNext(unsigned scalar);
    virtual XaCollectable *       findNext(const XaCollectable& theObject);
};

#endif /* XA_BAG_H */
