/******************************************************************************
 * $RCSfile: xwhodconfig.c,v $   $Revision: 1.16 $
 * Copyright (C) 1994  Technische Universitaet Muenchen
 ******************************************************************************/

#include "xwho.h"


/*----------------------------------------------------------------------------*
 * functions
 *----------------------------------------------------------------------------*/

/*--- xwhodconfig ------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
xwhodconfig
    New_xwhodconfig()
{
	xwhodconfig c;

	c = (xwhodconfig)TrueReAlloc(NULL, 1, sizeof (struct xwhodconfig));
	c->domain = c->gateways = New_xwhodaemons();
	c->seektree = New_tree();
	return (c);
}


/*----------------------------------------------------------------------------*/
xwhodaemon
    SeekServer_xwhodconfig(xwhodconfig c,
			   string subhost)
{
	xwhoseekserver entry;

	entry = SeekServer_xwhoseekserver(c->seektree, subhost);
	if (entry != NULL)
	    return (entry->daemon);
	else
	    return (NULL);
}


/*----------------------------------------------------------------------------*/
void
    Free_xwhodconfig(void** c_ref)
{
	if (*c_ref != NULL) {
		xwhodconfig c = *c_ref;

		Free_xwhodaemons((void**)&(c->domain));
		Free_xwhodaemons((void**)&(c->gateways));
		Free_tree(&(c->seektree), Free_xwhoseekserver);
		Free(c_ref);
	}
}


/*--- xwhoseekserver ---------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
xwhoseekserver
    New_xwhoseekserver()
{
	xwhoseekserver s;

	s = (xwhoseekserver)
	    TrueReAlloc(NULL, 1, sizeof (struct xwhoseekserver));
	s->subhost = NULL;
	s->daemon = NULL;
	return (s);
}


/*----------------------------------------------------------------------------*/
int
    Cmp_xwhoseekserver(void* seekserver1,
		       void* seekserver2)
{
	xwhoseekserver s1 = seekserver1;
	xwhoseekserver s2 = seekserver2;

	return ( strcmp(s1->subhost, s2->subhost) );
}


/*----------------------------------------------------------------------------*/
xwhoseekserver
    SeekServer_xwhoseekserver(tree seektree,
			      string subhost)
{
	xwhoseekserver dummy = New_xwhoseekserver();

	dummy->subhost = subhost;
	return ( Seek_tree(seektree, dummy, NULL, Cmp_xwhoseekserver) );
}


/*----------------------------------------------------------------------------*/
void
    Free_xwhoseekserver(void** s_ref)
{
	xwhoseekserver s = *s_ref;

	if (s != NULL) {
		s->subhost = NULL;
		s->daemon = NULL;
		Free(s_ref);
	}
}


/*--- xwhodaemons ------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
xwhodaemons
    New_xwhodaemons()
{
	return (NULL);
}


/*----------------------------------------------------------------------------*/
void
    Add_xwhodaemons(xwhodaemons* dl_ref,
		    xwhodaemon new)
{
	AddTrueNewSorted_list((list*)dl_ref, (void*)new, Cmp_xwhodaemon);
}


/*----------------------------------------------------------------------------*/
xwhodaemon
    SeekServer_xwhodaemons(xwhodaemons dl,
			   string subhost)
{
	list l;

	for (l = (list)dl; ! Empty_list(l); l = Next_list(l)) {
		xwhodaemon d = (xwhodaemon)First_list(l);

		if (HasSubhost_xwhodaemon(d,subhost))
		    return (d);
	}
	return (NULL);
}


/*----------------------------------------------------------------------------*/
void
    Del_xwhodaemons(xwhodaemons* dl_ref,
		    xwhodaemon d)
{
	xwhodaemon del;

	del = (xwhodaemon)DelSorted_list((list*)dl_ref, (void*)d, NULL,
					 Cmp_xwhodaemon);
	Free_xwhodaemon((void**)&del);
}


/*----------------------------------------------------------------------------*/
void
    Show_xwhodaemons(xwhodaemons dl)
{
	puts("daemon = <<<");
	InOrder_list((list)dl, Show_xwhodaemon);
	puts(">>>");
}


/*----------------------------------------------------------------------------*/
void
    Free_xwhodaemons(void** dl_ref)
{
	Free_list((list*)dl_ref, Free_xwhodaemon);
}


/*--- xwhodaemon -------------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
xwhodaemon
    New_xwhodaemon()
{
	xwhodaemon d;

	d = (xwhodaemon)TrueReAlloc(NULL, 1, sizeof (struct xwhodaemon));
	d->hostname = NULL;
	d->port = 0;
	d->subhosts = New_xwhodservedhosts();
	return (d);
}


/*----------------------------------------------------------------------------*/
xwhodaemon
    Init_xwhodaemon(string hostname)
{
	xwhodaemon d = New_xwhodaemon();

	if (hostname == NULL)
	    hostname = myhostname;
	StrTrueReAlloc(&(d->hostname), GetHostnameByName(hostname));
	d->port = XWHOD_PORT;
	Add_xwhodservedhosts(&(d->subhosts), Init_xwhodservedhost(hostname));
	return (d);
}


/*----------------------------------------------------------------------------*/
void
    SetPort_xwhodaemon(xwhodaemon d,
		       int port)
{
	d->port = port;
}


/*----------------------------------------------------------------------------*/
void
    AddSubhost_xwhodaemon(xwhodaemon d,
			  xwhodservedhost subhost)
{
	Add_xwhodservedhosts(&(d->subhosts), subhost);
}


/*----------------------------------------------------------------------------*/
xwhodaemon
    Copy_xwhodaemon(xwhodaemon d)
{
	xwhodaemon new;

	new = Init_xwhodaemon(d->hostname);
	SetPort_xwhodaemon(new, d->port);
	Free_xwhodservedhosts((void**)&(new->subhosts));
	new->subhosts = Copy_xwhodservedhosts(d->subhosts);
	return (new);
}


/*----------------------------------------------------------------------------*/
int
    Cmp_xwhodaemon(void* daemon1,
		   void* daemon2)
{
	xwhodaemon d1 = (xwhodaemon)daemon1;
	xwhodaemon d2 = (xwhodaemon)daemon2;

	return ( strcmp((char*)d1->hostname, (char*)d2->hostname) );
}


/*----------------------------------------------------------------------------*/
int
    HasSubhost_xwhodaemon(xwhodaemon d,
			  string subhost)
{
	return (HasSubhost_xwhodservedhosts(d->subhosts, subhost));
}


/*----------------------------------------------------------------------------*/
void
    Show_xwhodaemon(void* daemon)
{
	xwhodaemon d = (xwhodaemon)daemon;

	printf("   host \"%s\" on port \"%s\"\n",
	       (char*)d->hostname, (char*)d->port);
	Show_xwhodservedhosts(d->subhosts);
}


/*----------------------------------------------------------------------------*/
void
    Free_xwhodaemon(void** d_ref)
{
	xwhodaemon d = *d_ref;

	if (d != NULL) {
		StrFree(&(d->hostname));
		Free_xwhodservedhosts((void**)&(d->subhosts));
		Free(d_ref);
	}
}


/*--- xwhodservedhosts -------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
xwhodservedhosts
    New_xwhodservedhosts()
{
	return (NULL);
}


/*----------------------------------------------------------------------------*/
void
    Add_xwhodservedhosts(xwhodservedhosts* hl_ref,
			 xwhodservedhost h)
{
	xwhodservedhost old;

	old = (xwhodservedhost)DelSorted_list((list*)hl_ref, (void*)h, NULL,
					      Cmp_xwhodservedhost);
	if (old != NULL)
	    Free_xwhodservedhost((void**)&old);
	AddTrueSorted_list((list*)hl_ref, (void*)h, Cmp_xwhodservedhost);
}


/*----------------------------------------------------------------------------*/
xwhodservedhosts
    Copy_xwhodservedhosts(xwhodservedhosts hl)
{
	xwhodservedhosts new = New_xwhodservedhosts();
	list l;

	for (l = (list)hl; ! Empty_list(l); l = Next_list(l)) {
		Add_xwhodservedhosts(&new,
				     Copy_xwhodservedhost((xwhodservedhost)
							  First_list(l)));
	}
	return (new);
}


/*----------------------------------------------------------------------------*/
xwhodservedhost
    TakeOut_xwhodservedhosts(xwhodservedhosts* hl_ref)
{
	return ( (xwhodservedhost)TakeOut_list((list*)hl_ref) );
}


/*----------------------------------------------------------------------------*/
int
    HasSubhost_xwhodservedhosts(xwhodservedhosts hl,
				string subhost)
{
	static struct xwhodservedhost hs;
	int has_subhost = 0;

	hs.hostname = subhost;
	if (SeekSorted_list((list)hl, (void*)&hs, NULL, Cmp_xwhodservedhost)
	    != NULL)
	    has_subhost = 1;
	return (has_subhost);
}


/*----------------------------------------------------------------------------*/
void
    Update_xwhodservedhosts(xwhodservedhosts hosts)
{
	InOrder_list((list)hosts, Update_xwhodservedhost);
}


/*----------------------------------------------------------------------------*/
xwhodservedhosts
    Changed_xwhodservedhosts(xwhodservedhosts hl)
{
	xwhodservedhosts changed = New_xwhodservedhosts();
	list l;

	for (l = (list)hl; ! Empty_list(l); l = Next_list(l)) {
		xwhodservedhost h = (xwhodservedhost)First_list(l);

		if (HasChanged_xwhodservedhost(h))
		    Add_xwhodservedhosts(&changed, h);
	}
	return (changed);
}


/*----------------------------------------------------------------------------*/
void
    Send_xwhodservedhosts(conn c,
			  xwhodservedhosts hl)
{
	list l;

	for (l = (list)hl; ! Empty_list(l); l = Next_list(l)) {
		ProtocolNext(c);
		Send_xwhodservedhost(c, (xwhodservedhost)First_list(l));
	}
	ProtocolEnd(c);
}


/*----------------------------------------------------------------------------*/
xwhodservedhosts
    Recv_xwhodservedhosts(conn c)
{
	xwhodservedhosts hl = New_xwhodservedhosts();

	while (ProtocolHasNext(c))
	    Add_xwhodservedhosts(&hl, Recv_xwhodservedhost(c));
	return (hl);
}


/*----------------------------------------------------------------------------*/
void
    Show_xwhodservedhosts(xwhodservedhosts hl)
{
	puts("hosts = {");
	InOrder_list((list)hl, Show_xwhodservedhost);
	puts("}");
}


/*----------------------------------------------------------------------------*/
void
    Free_xwhodservedhosts(void** hl_ref)
{
	Free_list((list*)hl_ref, Free_xwhodservedhost);
}


/*--- xwhodservedhost --------------------------------------------------------*/

/*----------------------------------------------------------------------------*/
xwhodservedhost
    New_xwhodservedhost()
{
	xwhodservedhost h;

	h = (xwhodservedhost)TrueReAlloc(NULL, 1,
					 sizeof (struct xwhodservedhost));
	h->hostname = h->utmpfile = h->localtty = NULL;
	h->logins = h->lastlogins = New_xwhologins();
	return (h);
}


/*----------------------------------------------------------------------------*/
xwhodservedhost
    Init_xwhodservedhost(string hostname)
{
	xwhodservedhost h = New_xwhodservedhost();

	if (hostname == NULL)
	    hostname = myhostname;
	else
	    hostname = (string)GetHostnameByName(hostname);
	StrTrueReAlloc(&(h->hostname), hostname);
	StrTrueReAlloc(&(h->utmpfile), UTMP_FILE);
	StrTrueReAlloc(&(h->localtty), DEFAULT_LOCALTTY);
	return (h);
}


/*----------------------------------------------------------------------------*/
void
    SetUtmpFile_xwhodservedhost(xwhodservedhost h,
				string utmpfile)
{
	StrTrueReAlloc(&(h->utmpfile), utmpfile);
}


/*----------------------------------------------------------------------------*/
void
    SetLocalTty_xwhodservedhost(xwhodservedhost h,
				string ttyline)
{
	StrTrueReAlloc(&(h->localtty), ttyline);
}


/*----------------------------------------------------------------------------*/
xwhodservedhost
    Copy_xwhodservedhost(xwhodservedhost h)
{
	xwhodservedhost new;

	new = Init_xwhodservedhost(h->hostname);
	SetUtmpFile_xwhodservedhost(new, h->utmpfile);
	SetLocalTty_xwhodservedhost(new, h->localtty);
	return (new);
}


/*----------------------------------------------------------------------------*/
int
    Cmp_xwhodservedhost(void* host1,
			void* host2)
{
	xwhodservedhost h1 = (xwhodservedhost)host1;
	xwhodservedhost h2 = (xwhodservedhost)host2;

	return ( strcmp((char*)h1->hostname, (char*)h2->hostname) );
}


/*----------------------------------------------------------------------------*/
void
    Update_xwhodservedhost(void* host)
{
	xwhodservedhost h = (xwhodservedhost)host;

	Free_xwhologins((void**)&(h->lastlogins));
	h->lastlogins = h->logins;
	h->logins = Get_xwhologins(h);
}


/*----------------------------------------------------------------------------*/
int
    HasChanged_xwhodservedhost(xwhodservedhost h)
{
	return ( Cmp_xwhologins((void*)h->logins, (void*)h->lastlogins) );
}


/*----------------------------------------------------------------------------*/
void
    Send_xwhodservedhost(conn c,
			 xwhodservedhost h)
{
	SendHostData(c, h->hostname);
	Send_xwhologins(c, h->logins);
}


/*----------------------------------------------------------------------------*/
xwhodservedhost
    Recv_xwhodservedhost(conn c)
{
	xwhodservedhost h = New_xwhodservedhost();

	RecvHostData(c, &(h->hostname));
	h->logins = Recv_xwhologins(c);
	return (h);
}


/*----------------------------------------------------------------------------*/
void
    Show_xwhodservedhost(void* host)
{
	xwhodservedhost h = (xwhodservedhost)host;

	printf("   host \"%s\" has utmp-file \"%s\" with tty \"%s\"\n",
	       (char*)h->hostname, (char*)h->utmpfile, (char*)h->localtty);
}


/*----------------------------------------------------------------------------*/
void
    Free_xwhodservedhost(void** h_ref)
{
	xwhodservedhost h = *h_ref;

	if (h != NULL) {
		StrFree(&(h->hostname));
		StrFree(&(h->utmpfile));
		StrFree(&(h->localtty));
		Free_xwhologins((void**)&(h->logins));
		Free_xwhologins((void**)&(h->lastlogins));
		Free(h_ref);
	}
}
