/******************************************************************************
 * $RCSfile: list.h,v $   $Revision: 1.1.1.1 $
 * Copyright (C) 1993, 1994  A.Michael.Leliveld@Informatik.TU-Muenchen.De
 *
 * data structure list
 * ===================
 *
 * This data structure is implemented as doubly-linked list.
 *                      #####    #####           #####
 *              HEAD--->##1##--->##2##--->...--->##n##--->NULL
 *                   +--#####<---#####<---...<---#####<-+
 *                   |  #####    #####           #####  |
 *                   +----------------------------------+
 * l is empty list  <==>  Empty_list(l)  <==>  l == NULL  <==>  n == 0
 *                              HEAD--->NULL
 *
 * n > 0
 * -----
 *
 * first element of l  ==  First_list(l)  ==  l->data
 * l not a sublist :  last element of l  ==  Last_list(l)  ==  l->prev
 * rest of l  ==  first sublist of l  ==  Next_list(l)  ==  l->next
 * s sublist :  preceding sublist  ==  Previous_list(s)  ==  s->prev
 *
 * sorted list
 * -----------
 *
 * l is sorted list  <==>
 *       forall s sublist in l :  First_list(Previous_list(s)) <= First_list(s)
 * A sorted list needs a compare function for list-elements cmp():
 *    int    < 0		e1 < e2
 *           == 0		e1 == e2
 *           > 0		e1 > e2
 *    cmp(
 *       void*,			element e1
 *       void*			element e2
 *    );
 *
 * operations
 * ----------
 *
 * elementary:
 *    New_list(), Empty_list(), First_list(), Last_list(), Next_list(),
 *    Previous_list()
 *
 * normal:
 *    Append_list(), Add_list(), Seek_list(), Del_list(), TakeOut_list(),
 *    InOrder_list(), Free_list()
 ******************************************************************************/

#ifndef _MISC_LIST_H
#define _MISC_LIST_H


#include <misc/mem.h>


/*----------------------------------------------------------------------------*
 * typedefs
 *----------------------------------------------------------------------------*/

struct list {
	struct list * next; /* next element */
	struct list * prev; /* previous element */
	void* data;	    /* data of the element */
};
typedef struct list * list;


/*----------------------------------------------------------------------------*
 * extern prototypes
 *----------------------------------------------------------------------------*/

/* Returns an empty list */
extern list
    New_list();


/* Is the list empty? */
extern int
    Empty_list(list);


/* Retrieves the data of the first element on the list.
 * If the list is empty NULL will be returned. */
extern void*
    First_list(list);


/* Retrieves the data of the last element on the list.
 * If the list is empty NULL will be returned. */
extern void*
    Last_list(list);


/* Returns the first sublist of the list.
 * If the list is empty NULL will be returned. */
extern list
    Next_list(list);


/* Returns the preceding list or NULL, if there isn't one */
extern list
    Previous_list(list);


/* Append an element to a list as last element */
extern void
    AppendTrue_list(list*, /* pointer to the header of the list */
		    void*  /* data of the element to be added to the list */
		    );


/* Add an element to a sorted list */
extern void
    AddTrueSorted_list(list*, /* pointer to the header of the sorted list */
		       void*, /* data of the element to be added to the list */
		       int (*)(void *, void *) /* compare function */
		       );


/* Add an element ne to a sorted list, only if there is no other element e
 * on the list with cmp(ne,e) == 0 */
extern int /* 0 ok, 1 element was in list */
    AddTrueNewSorted_list(
			  list*, /* pointer to the header of the sorted list */
			  void*, /* data of the element to be added to the list
				  */
			  int (*)(void*, void*) /* compare function */
			  );


/* Seek the first occurrence of an element on a list.
 * If the element does not appear at all, then NULL is returned */
extern void* /* pointer to the seeked data from the list  or  NULL */
    Seek_list(list,  /* list to be searched */
	      void*, /* data to be seeked */
	      list*, /* pointer to the rest of the list, if the pointer is NULL
		      * this parameter will be ignored */
	      int (*)(void*, void*) /* compare function */
	      );


/* Seek the first occurrence of an element on a sorted list.
 * If the element does not appear at all, then NULL is returned */
extern void* /* pointer to the seeked data from the list  or  NULL */
    SeekSorted_list(list,  /* list to be searched */
		    void*, /* data to be seeked */
		    list*, /* pointer to the rest of the list, if the pointer 
			    * is NULL this parameter will be ignored */
		    int (*)(void *, void *) /* compare function */
		    );


/* Delete an element from a list */
extern void* /* pointer to the deleted data from the list or NULL */
    Del_list(list*, /* pointer to the header of the list */
	     void*, /* data to be deleted from the list */
	     list*, /* pointer to the rest of the list, if the pointer is NULL 
		     * this parameter will be ignored */
	     int (*)(void *, void *) /* compare function */
	     );


/* Delete an element from a sorted list */
extern void* /* pointer to the deleted data from the list or NULL */
    DelSorted_list(list*, /* pointer to the header of the list */
		   void*, /* data to be deleted from the list */
		   list*, /* pointer to the rest of the list, if the pointer is
			   * NULL this parameter will be ignored */
		   int (*)(void*, void*) /* compare function */
		   );


/* Get the first element from a list and remove it */
extern void* /* pointer to the first element or NULL */
    TakeOut_list(list* /* the list */
		 );


/* Uses the function in the order of occurrence on every element of the list */
extern void
    InOrder_list(list, /* header of the list */
		 void (*)(void*) /* function to apply to the element */
		 );


/* Frees the whole list */
extern void
    Free_list(list*, /* pointer to the header of the list */
	      void (*)(void**) /* function to free an element */
	      );



#endif	 /* _MISC_LIST_H */
