/* xtea - distribute beverages and other resources over the network
 *
 * Copyright (c) 1994 Henning Spruth (spruth@regent.e-technik.tu-muenchen.de)
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.  No representations are made about the suitability of this
 * software for any purpose.  It is provided "as is" without express or 
 * implied warranty.
 */

#include <stdio.h>
#include <unistd.h>

#include <X11/StringDefs.h>
#include <X11/Intrinsic.h>
#include <X11/Shell.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/SmeBSB.h>
#include <X11/Xaw/SimpleMenu.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/AsciiText.h>

#include "config.h"
#include "alloc.h"
#include "sockio.h"
#include "userlist.h"
#include "message.h"
#include "xresources.h"

extern Widget toplevel;
extern appresStruct appres;
extern char my_uname[];
extern int verbose;
extern int restypes;

/*
 * This structure contains data on a user who appears in a userlist popup.
*/ 
typedef struct 
{
  char *uname;
  char *fullname;
  struct ResourceData *resourcedata;
  int index;
  enum {UNSELECTED,SELECTED,CONFIRMED} status;
  int active;			/* is the user still active ? */
  int amountordered;			/* amount ordered */
  Widget command;
  int gotdefaults;
  Pixel defaultbg, defaultfg;
} UserlistPopup;


/*
 * This structure contains the status of all commodities we are offering 
*/
typedef struct ResourceData
{
  int resource;
  int local_amount;
  int userlistcount;
  UserlistPopup *userlistdata;
  Widget userlistpopup;
  Widget amount, info, invite, selectall, cancel;
  enum {SELECT,WAIT} state;
  char amountval[5];	/* string used by the AsciiText widget */
/*  Widget userlistamount;*/	/* the text widget conaining the amount */
} ResourceData;

static ResourceData resourcedata[MAX_RESOURCES];

/* ************************** userbutton_callback **************************
 *
 * One of the candidates for invitation has been selected. Change the
 * status of the button, depending how far we are with providing the
 * commodity.
 *
*/
static void userbutton_callback(Widget w,  UserlistPopup *p, void *dummy)
{
  int n;
  Arg arg[8];
  char buf[200];
  
  Pixel bg,fg;

  if(!p->gotdefaults)
  {
    n=0;
    XtSetArg(arg[n],XtNbackground,&p->defaultbg); n++;
    XtSetArg(arg[n],XtNforeground,&p->defaultfg); n++;
    XtGetValues(w,arg,n);
    p->gotdefaults=1;
  }

  /* We are still choosing users to be invited, so pressing the button just
     toggles the status between SELECTED and UNSELECTED. */
  if(p->resourcedata->state==SELECT)
    switch(p->status)
    {
    case UNSELECTED:
      p->status=SELECTED;
      break;
    case SELECTED:
      p->status=UNSELECTED;
      break;
    default:
      break;
    }

  n=0;

  switch(p->status)
  {
  case UNSELECTED:
    bg=p->defaultbg;
    fg=p->defaultfg;
    break;
  case SELECTED:
    bg=appres.selectbgcolor;
    fg=appres.selectfgcolor;
    break;
  case CONFIRMED:
    if(p->amountordered>0)
    {
      sprintf(buf,"%s: %d",p->fullname,p->amountordered);
      XtSetArg(arg[n], XtNlabel, buf); n++;
    }
    bg=appres.orderbgcolor;
    fg=appres.orderfgcolor;
    break;
  }

  XtSetArg(arg[n],XtNbackground,bg); n++;
  XtSetArg(arg[n],XtNforeground,fg); n++;
  XtSetValues(w,arg,n);
}


/* **************************** quitbutton_callback() ************************
 *
 * The user selection process is cancelled. No invitations have been sent
 * yet, so simply free the memory.
*/
static void quitbutton_callback(Widget w, ResourceData *r, void *dummy)
{
  int i;
  XtPopdown(r->userlistpopup);
  for(i=0;i<r->userlistcount;i++)
  {
    free(r->userlistdata[i].uname);
    free(r->userlistdata[i].fullname);
  }
  free(r->userlistdata);
  r->userlistdata=NULL;
  r->userlistcount=0;
  XtDestroyWidget(r->userlistpopup);
  r->userlistpopup=NULL;
}  


/* **************************** send_informserver() **************************
 *
 * We have decided which users to invite, so send an INFORMSERVER message to
 * the xtead server.
 *
*/
static void send_informserver(ResourceData *r)
{
  int i,n,sock;
  char buf[80];
  Arg arg[5];
  char *widgetinfo,*info;

  n=0;
  XtSetArg(arg[n],XtNstring,&widgetinfo); n++;
  XtGetValues(r->info,arg,n);

  info=strdup(widgetinfo);

  n=0;
  for(i=0;i<r->userlistcount;i++) 
    if(r->userlistdata[i].active &&
       (r->userlistdata[i].status==SELECTED ||
       r->userlistdata[i].status==CONFIRMED)) n++;

  start_servermessage(INFORMSERVER,&sock);
  put_string(sock,my_uname);
  sprintf(buf,"%d",r->resource);
  put_string(sock,buf);
  sprintf(buf,"%d",r->local_amount);
  put_string(sock,buf);
  sprintf(buf,"%d",n);
  put_string(sock,buf);
  put_string(sock,info);
  for(i=0;i<r->userlistcount;i++)
    if(r->userlistdata[i].active &&
       (r->userlistdata[i].status==SELECTED ||
       r->userlistdata[i].status==CONFIRMED))
      put_string(sock,r->userlistdata[i].uname);
  put_string(sock,"END");
  free(info);
  close(sock);
}


/* ************************* quitbutton_callback2() *************************
 *
 * When the cancel button is activated after the invitiation has been sent,
 * the invitees have to be informed that nothing is left.
 *
*/
static void quitbutton_callback2(Widget w, ResourceData *r, void *dummy)
{
  r->local_amount=0;
  send_informserver(r);
  /* now call the 'local' callback function */
  quitbutton_callback(w,r,dummy);
}


/* **************************** invite_callback() **************************
 *
 * We have selected the invitees, so now send the invitation.
 *
*/
static void invite_callback(Widget w, ResourceData *r, void *dummy)
{
  int i,n;
  Arg arg[5];

  r->state=WAIT;

  if(sscanf(r->amountval,"%d",&(r->local_amount))!=1)
    r->local_amount=1;

  /* count # of invitees */
  n=0;
  for(i=0;i<r->userlistcount;i++) 
    if(r->userlistdata[i].status==SELECTED) n++;
  
  /* no invitations: quit */
  if(n==0)
    quitbutton_callback(w,r,dummy);

  /* send the invitation */
  send_informserver(r);

  n=0;
  XtSetArg(arg[n], XtNsensitive, False); n++;
  XtSetValues(r->invite, arg, n);
  XtSetValues(r->selectall, arg, n);

  /* make the text fields read-ounly */
  n=0;
  XtSetArg(arg[n], XtNeditType,XawtextRead); n++;
  XtSetValues(r->amount, arg, n);
  XtSetValues(r->info, arg, n);


  /* change the callback of the quitbutton so that the popups are removed */
  XtRemoveCallback(r->cancel,XtNcallback,
		   (XtCallbackProc) quitbutton_callback,r);
  XtAddCallback(r->cancel, XtNcallback,
		(XtCallbackProc) quitbutton_callback2, r);
}


/* ************************** inviteall_callback() ***************************
 *
 * Mark all users as invitees.
 *
*/
static void inviteall_callback(Widget w, ResourceData *r, void *dummy)
{
  int i;
  for(i=0; i<r->userlistcount;i++)
  {
    if(r->userlistdata[i].status==UNSELECTED)
    {
      userbutton_callback(r->userlistdata[i].command, 
			  &(r->userlistdata[i]), dummy);
    }
  }
}



/* ************************* send_confirmsever() ***************************
 *
 * A user has responded to the invitation, so tell him whether something is
 * still left.
 *
*/
static void send_confirmserver(char *uname, int resource, int status)
{
  char buf[5];
  int sock;
  /* send a confirmation to the requesting user */
  start_servermessage(CONFIRMSERVER,&sock);
  sprintf(buf,"%d",resource);
  put_string(sock,buf);
  put_string(sock,uname);
   sprintf(buf,"%d",status);
  put_string(sock,buf);
  put_string(sock,my_uname);
  put_string(sock,"END");
  close(sock);
}


/* ****************************** process_order() *************************
 *
 * A user responds to the invitation. Update his button and decide if the
 * request can be fulfilled.
 *
*/
void process_order(int sock)
{
  int resource,status,amount;
  char *temp,*uname;
  int i,n;
  Arg arg[5];
  ResourceData *r;
  UserlistPopup *p;

  if(get_string(sock,&temp)) return;
  status=0; if(sscanf(temp,"%d",&resource)!=1) status=1;
  free(temp);
  if(status) return;

  if(get_string(sock,&uname)) return;

  if(get_string(sock,&temp)) return;
  status=0; if(sscanf(temp,"%d",&amount)!=1) status=1;
  free(temp);
  if(status) { free(uname); return; }

  r=&resourcedata[resource];

  if(r->userlistcount==0)
  {
    /* There is currently no userlist for this resource, which should
     * not happen. Anyway, ignore the rest of the message and proceed.
     */
    if(verbose) 
      printf("Warning: Bad request for resource %d (%s) from user %s\n",
	     resource, appres.resourcename[resource], uname);
  }

  for(i=0;i<r->userlistcount; i++)
    if(strcmp(r->userlistdata[i].uname,uname)==0) break;

  if(i>=r->userlistcount) 
  {
    send_confirmserver(uname,resource,2);
    free(uname);
    return;
  }
  free(uname);

  p=&(r->userlistdata[i]);
  if(amount!=0)
  {
    p->status=2;
      if(r->local_amount>0)
    {
      (r->local_amount)--;
      
      /* send positive confirmation */
      send_confirmserver(p->uname,resource,0);
      
      /* update other popups */
      send_informserver(r);
      sprintf(r->amountval,"%d",r->local_amount);
      n=0;
      XtSetArg(arg[n], XtNstring, r->amountval); n++;
      XtSetValues(r->amount, arg, n);
      
      (p->amountordered)++;

    }
    else send_confirmserver(p->uname,resource,1);
    userbutton_callback(p->command, p,NULL);
  }
  else /* amount == 0 */
  {
    n=0;
    XtSetArg(arg[n], XtNsensitive, False); n++;
    XtSetValues(p->command, arg, n);
    p->active=0;
  }
}


/* ***************************** popup_userlist() ****************************
 *
 * The server sends us a list of possible invitees for our commodity. Create
 * the popup to select the invitees.
 *
*/
void popup_userlist(int sock)
{
  Arg arg[8];
  char buf[40];
  int n;
  Widget popup,menu,command,button,text,label;
  int i,j;
  int resource, usercount;
  char *s;
  UserlistPopup *ul;

  if(get_string(sock,&s)) return;
  if(sscanf(s,"%d",&resource)!=1 || resource>=restypes 
     || resourcedata[resource].userlistpopup)
  {
    free(s);
    return;
  }
  free(s);

  resourcedata[resource].state = SELECT;
  if(get_string(sock,&s)) return;
  if(sscanf(s,"%d",&usercount)!=1)
  {
    free(s);
    return;
  }
  free(s);
  if(usercount<1) return;

  ul=myalloc(usercount*sizeof(UserlistPopup));
  for(i=0;i<usercount;i++)
  {
    if(get_string(sock,&ul[i].uname)) break;
    if(get_string(sock,&ul[i].fullname))
    {
      free(ul[i].uname);
      break;
    }
  }
  if(i<usercount)
    for(j=0;j<i;j++){ free(ul[j].uname); free(ul[j].fullname);}

  n=0;
  popup= XtCreatePopupShell("invitepopup",transientShellWidgetClass,
			    toplevel,arg,n);

  for(i=0;i<usercount;i++)
  {
    ul[i].index=i;
    ul[i].status=UNSELECTED;
    ul[i].active = 1;
    ul[i].amountordered=0;
    ul[i].resourcedata=&resourcedata[resource];
    ul[i].gotdefaults=0;
  }

  resourcedata[resource].userlistcount = usercount;
  resourcedata[resource].userlistpopup= popup;
  resourcedata[resource].userlistdata = ul;
  resourcedata[resource].resource = resource;

  n=0;
  menu=XtCreateManagedWidget("form",formWidgetClass,popup,arg,n);

  sprintf(buf,appres.makeoffer,appres.resourcename[resource]);
  n=0;
  XtSetArg(arg[n], XtNlabel, buf); n++;
  command=XtCreateManagedWidget("reslabel",labelWidgetClass,menu,arg,n); 

  for(i=0;i<usercount;i++)
  {
    if(verbose) printf("User %s %s\n",ul[i].uname, ul[i].fullname);
    n=0;
    XtSetArg(arg[n], XtNlabel, ul[i].fullname); n++;
    XtSetArg(arg[n], XtNleft, XawChainLeft); n++;
    XtSetArg(arg[n], XtNright, XawChainRight); n++;
    XtSetArg(arg[n], XtNfromVert, command); n++;
    XtSetArg(arg[n], XtNresizable, True); n++;
    command=XtCreateManagedWidget("userbutton",commandWidgetClass,menu,arg,n); 
    XtAddCallback(command,XtNcallback,(XtCallbackProc) userbutton_callback,
		  &ul[i]);
    ul[i].command=command;
  }

  strcpy(resourcedata[resource].amountval,"10 ");

  n=0;
  XtSetArg(arg[n], XtNfromVert, command); n++;
/*  XtSetArg(arg[n], XtNlabel, "Amount:"); n++;*/
  label=XtCreateManagedWidget("amountlabel",labelWidgetClass,menu,arg,n); 

  n=0;
  XtSetArg(arg[n], XtNfromVert, command); n++;
  XtSetArg(arg[n], XtNfromHoriz, label); n++;
  XtSetArg(arg[n], XtNeditType, XawtextEdit); n++;
  XtSetArg(arg[n], XtNwidth, 40); n++;
  XtSetArg(arg[n], XtNuseStringInPlace, True); n++;
  XtSetArg(arg[n], XtNstring, resourcedata[resource].amountval); n++;
  text=XtCreateManagedWidget("amounttext",asciiTextWidgetClass,menu,arg,n); 
  resourcedata[resource].amount=text;

  n=0;
  XtSetArg(arg[n], XtNfromVert, text); n++;
  label=XtCreateManagedWidget("infolabel",labelWidgetClass,menu,arg,n); 

  n=0;
  XtSetArg(arg[n], XtNfromVert,text ); n++;
  XtSetArg(arg[n], XtNfromHoriz, label); n++;
  XtSetArg(arg[n], XtNeditType, XawtextEdit); n++;
  XtSetArg(arg[n], XtNscrollHorizontal, XawtextScrollWhenNeeded); n++;
  XtSetArg(arg[n], XtNwidth, 250); n++;
  text=XtCreateManagedWidget("infotext",asciiTextWidgetClass,menu,arg,n); 
  resourcedata[resource].info=text;

  n=0;
  XtSetArg(arg[n], XtNfromVert, text); n++;
  button=XtCreateManagedWidget("invite",commandWidgetClass,menu,arg,n); 
  XtAddCallback(button,XtNcallback,(XtCallbackProc) invite_callback,
		&resourcedata[resource]);
  resourcedata[resource].invite=button;

  n=0;
  XtSetArg(arg[n], XtNfromVert, text); n++;
  XtSetArg(arg[n], XtNfromHoriz, button); n++;
  button=XtCreateManagedWidget("selectall",commandWidgetClass,menu,arg,n); 
  XtAddCallback(button,XtNcallback,(XtCallbackProc) inviteall_callback,
		&resourcedata[resource]);
  resourcedata[resource].selectall=button;

  n=0;
  XtSetArg(arg[n], XtNfromVert, text); n++;
  XtSetArg(arg[n], XtNfromHoriz, button); n++;
  button=XtCreateManagedWidget("cancel",commandWidgetClass,menu,arg,n); 
  XtAddCallback(button,XtNcallback, (XtCallbackProc) quitbutton_callback,
		&resourcedata[resource]);
  resourcedata[resource].cancel=button;

  XtPopup(popup,XtGrabNone);
}

