/* xtea - distribute beverages and other resources over the network
 *
 * Copyright (c) 1994 Henning Spruth (spruth@regent.e-technik.tu-muenchen.de)
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.  No representations are made about the suitability of this
 * software for any purpose.  It is provided "as is" without express or 
 * implied warranty.
 */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>

#include <string.h>
#include <signal.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <time.h>

#include "config.h"
#include "sockio.h"
#include "version.h"
#include "message.h"
#include "alloc.h"

#ifdef NONO__alpha
int accept(int,struct sockaddr_in *,int *);
void bcopy(char *,char*,int);
void bzero(char *,int);
int socket(int , int,int);
int bind(int,struct sockaddr_in *, int);
int getsockname(int,struct sockaddr_in *,int *);
int gethostname(char *,int len);
int listen(int, int);
#endif

extern char *msgtype[];
extern char *xtea_resources[];


/*
 * This structure holds data on every user in the xtea network.
*/
typedef struct Consumer
{
  char *uname;
  char *host;
  char *fullname;
  int port;
  int pflags;
  int cflags;
  int mark;
  int group;
  time_t timestamp;
  struct Consumer *next;
  struct Consumer *prev;
} Consumer;


static Consumer *firstconsumer, *lastconsumer;
static int consumercount;
static int verbose;
int restypes;

/* ******************************** print_resources() **********************
 *
 * Print all resources whose flag bits are set.
 *
*/
static void print_resources(int flag)
{
  int i;
  for(i=0;i<restypes;i++)
  {
    if(flag & 1<<i) printf(" %s",xtea_resources[i]);
  }
  printf("\n");
}


/* ******************************* new_consumer() ***************************
 *
 * Create a new consumer structure and put it into the linked list.
 *
*/
Consumer *new_consumer(char *uname, char *host, char *fullname, 
			      int port, int pflags, int cflags,
			      int group)
{
  Consumer *c;
  c=myalloc(sizeof(Consumer));
  c->uname=uname;
  c->host = host;
  c->fullname=fullname;
  c->port=port;
  c->pflags=pflags;
  c->cflags=cflags;
  c->group=group;
  c->mark=0;
  c->timestamp=time(NULL);
  if(verbose)
  {
    printf("  New consumer %s %s at %s, port %d flags %d group %d taking\n  ",
	 c->uname, c->fullname, c->host, c->port, c->cflags, c->group);
    print_resources(c->cflags);
  }
  c->next = NULL;
  c->prev = lastconsumer;
  if(lastconsumer) lastconsumer->next = c;
  lastconsumer = c;
  if(firstconsumer==NULL) firstconsumer=c;
  consumercount++;
  return c;
}


/* ****************************** search_consumer() **************************
 *
 * Check if we already have the consumer 'uname'.
 *
*/
Consumer *search_consumer(char *uname)
{
  Consumer *c;
  for(c=firstconsumer; c!=NULL; c=c->next)
    if(strcmp(uname,c->uname)==0) return c;
  return NULL;
}


/* ****************************** remove_consumer() *************************
 *
 * Remove a consumer from the link list and free the memory.
 *
*/
void remove_consumer(Consumer *c)
{
  if(firstconsumer==c) firstconsumer=c->next;
  if(lastconsumer==c) lastconsumer=c->prev;
  if(c->prev) (c->prev)->next = c->next;
  if(c->next) (c->next)->prev = c->prev;
  free(c->uname);
  free(c->host);
  free(c->fullname);
  free(c);
  consumercount--;
}


/* ****************************** purge_consumers() ************************
 *
 * Remove all consumers that were idle for more then TIMEOUT seconds.
 *
*/
void purge_consumers()
{
  time_t now;
  Consumer *c,*cnext;
  now=time(NULL);
  for(c=firstconsumer; c!=NULL; c=cnext)
  {
    cnext=c->next;
    if(now - c->timestamp > TIMEOUT)
    {
      if(verbose)
	printf("purge_consumers(): User %s has been quiet for %d seconds - removed\n",
	  c->uname,(int) (now - c->timestamp));
      remove_consumer(c);
    }
  }
}


/* ******************************** send_status() ***************************
 *
 * Send the list of consumers to an xteaq client.
 *
*/
static int send_status(int con)
{
  int i,status,port,sendsock;
  char *hostname, *temp;
  char buf[300];
  Consumer *c;

  if(get_string(con,&hostname)) return 1;

  if(get_string(con,&temp)) return 1;
  status=0; if(sscanf(temp,"%d",&port)!=1) status=1;
  free(temp);
  if(status) return 1;

  end_of_message(con);

  status=start_message(hostname,htons(port),STATUSDATA,&sendsock);
  free(hostname);
  if(status) return 1;

  sprintf(buf,"xtea daemon version %s reporting",XTEA_VERSION);
  put_string(sendsock,buf);

  sprintf(buf,"Resources:");
  for(i=0;i<restypes;i++)
  {
    strcat(buf," ");
    strcat(buf,xtea_resources[i]);
  }
  put_string(sendsock,buf);

  sprintf(buf,"Client count: %d",consumercount);
  put_string(sendsock,buf);
  for(c=firstconsumer; c!=NULL; c=c->next)
  {
    sprintf(buf,"  %s (%s@%s), group %d",
       c->fullname, c->uname, c->host, c->group);  
    put_string(sendsock,buf);

    strcpy(buf,"    taking");
    for(i=0;i<restypes;i++)
    {
      if(c->cflags & 1<<i) 
      { 
	strcat(buf," ");
	strcat(buf,xtea_resources[i]);
      }
    }
    put_string(sendsock,buf);
    strcpy(buf,"    providing");
    for(i=0;i<restypes;i++)
    {
      if(c->pflags & 1<<i) 
      { 
	strcat(buf," ");
	strcat(buf,xtea_resources[i]);
      }
    }
    put_string(sendsock,buf);
  }    
  put_string(sendsock,"END");
  close(sendsock);
  return 0;
}



/* ********************************** get_consumer_data() ********************
 *
 * Parse the STARTUP message of a consumer.
 *
*/
static Consumer *get_consumer_data(int con)
{
  Consumer *c,*c1;
  char *s,*s1,*s2,*s3;
  int i1,i2,i3,i4;

  /* uname */
  if(get_string(con,&s1)) return NULL;
  /* host */
  if(get_string(con,&s2))
  { free(s1);
    return NULL;
  }
  /* fullname */
  if(get_string(con,&s3))
  { free(s1); free(s2);
    return NULL;
  }
  /* port number */
  if(get_string(con,&s))
   { free(s1); free(s2); free(s3);
     return NULL;
   }
  if(sscanf(s,"%d",&i1)!=1)
  { free(s1); free(s2); free(s3); free(s);
    return NULL;
  }
  free(s);
  /* pflags */
  if(get_string(con,&s))
   { free(s1); free(s2); free(s3); free(s);
     return NULL;
   }
  if(sscanf(s,"%d",&i2)!=1)
  { free(s1); free(s2); free(s3); free(s);
    return NULL;
  }
  free(s);
  /* cflags */
  if(get_string(con,&s))
   { free(s1); free(s2); free(s3); free(s);
     return NULL;
   }
  if(sscanf(s,"%d",&i3)!=1)
  { free(s1); free(s2); free(s3); free(s);
    return NULL;
  }
  free(s);
  /* group */
  if(get_string(con,&s))
   { free(s1); free(s2); free(s3); free(s);
     return NULL;
   }
  if(sscanf(s,"%d",&i4)!=1)
  { free(s1); free(s2); free(s3); free(s);
    return NULL;
  }
  free(s);
  end_of_message(con);

  if((c1=search_consumer(s1)))
  {
    if(verbose)
      printf("  Consumer %s %s at %s port %d already known - removed\n",
	   c1->uname, c1->fullname, c1->host, c1->port);
    remove_consumer(c1);
  }

  c=new_consumer(s1,s2,s3,i1,i2,i3,i4);
  return c;
}  


/* ******************************* propagate_order() **************************
 *
 * Relay an order message from an invitee to the inviter.
 *
*/
static void propagate_order(int sock)
{
  char *temp;
  int resource, amount;
  int status,sendsock;
  Consumer *csrc,*ctarget;
  char buf[10];

  if(get_string(sock,&temp)) return;
  status=0; if(sscanf(temp,"%d",&resource)!=1) status=1;
  free(temp);
  if(status) return;

  if(get_string(sock,&temp)) return;
  status =0; if(!(ctarget=search_consumer(temp))) status=1;
  free(temp);
  if(status) return;

  if(get_string(sock,&temp)) return;
  status =0; if(!(csrc=search_consumer(temp))) status=1;
  free(temp);
  if(status) return;

  if(get_string(sock,&temp)) return;
  status=0; if(sscanf(temp,"%d",&amount)!=1) status=1;
  free(temp);
  if(status) return;

  end_of_message(sock);

  if(verbose)
    printf("  %s orders %d %s from %s\n",csrc->uname, amount, 
	 xtea_resources[resource], ctarget->uname);

  if(start_message(ctarget->host, htons(ctarget->port), ORDER, &sendsock))
    return;
  sprintf(buf,"%d",resource);
  put_string(sendsock,buf);
  put_string(sendsock,csrc->uname);
  sprintf(buf,"%d",amount);
  put_string(sendsock,buf);
  put_string(sendsock,"END");
  close(sendsock);
}


/* ******************************** consumer_quits() ************************
 *
 * An xtea client informs us that it's about to exit.
 *
*/
static void consumer_quits(int sock)
{
  char *temp;
  int status;
  Consumer *c;

  if(get_string(sock,&temp)) return;
  status =0; if(!(c=search_consumer(temp))) status=1;
  free(temp);
  if(status) return;
  end_of_message(sock);
  if(verbose)
    printf("  Consumer %s (%s) quits\n",c->uname,c->fullname);
  remove_consumer(c);
}
  

/* ******************************* progapate_confirm() **********************
 *
 * Relay a confirm message from the inviter to the invitee.
 *
*/
static void propagate_confirm(int sock)
{
  char *temp;
  int resource,result;
  int status,sendsock;
  Consumer *csrc,*ctarget;
  char buf[10];

  if(get_string(sock,&temp)) return;
  status=0; if(sscanf(temp,"%d",&resource)!=1) status=1;
  free(temp);
  if(status) return;

  if(get_string(sock,&temp)) return;
  status =0; if(!(ctarget=search_consumer(temp))) status=1;
  free(temp);
  if(status) return;

  if(get_string(sock,&temp)) return;
  status=0; if(sscanf(temp,"%d",&result)!=1) status=1;
  free(temp);
  if(status) return;

  if(get_string(sock,&temp)) return;
  status =0; if(!(csrc=search_consumer(temp))) status=1;
  free(temp);
  if(status) return;
  end_of_message(sock);

  if(verbose)
    printf("  %s confirms request of %s for %s with status %d\n",
	 csrc->uname, ctarget->uname, xtea_resources[resource], result);

  if(start_message(ctarget->host, htons(ctarget->port), CONFIRM, &sendsock))
    return;
  sprintf(buf,"%d",resource);
  put_string(sendsock,buf);
  sprintf(buf,"%d",result);
  put_string(sendsock,buf);
  put_string(sendsock,csrc->fullname);
  put_string(sendsock,"END");
  close(sendsock);
}


/* ******************************* distribute_inform ************************  
 *
 * Relay INFORM messages from the inviter to a list of invitees.
 *
*/
static void distribute_inform(int con)
{
  char *uname,*temp, *info;
  Consumer *c,*c1;
  int flag,resource,amount,usercount,i,status;
  int sendsock;
  char buf[10];

  if(get_string(con,&uname)) return;
  if(!(c=search_consumer(uname)))
  {
    free(uname);
    return;
  }
  free(uname);

  if(get_string(con,&temp)) return;
  status=0; if(sscanf(temp,"%d",&resource)!=1) status=1;
  free(temp);
  if(status) return;

  if(get_string(con,&temp)) return;
  status=0; if(sscanf(temp,"%d",&amount)!=1) status=1;
  free(temp);
  if(status) return;

  if(get_string(con,&temp)) return;
  status=0; if(sscanf(temp,"%d",&usercount)!=1) status=1;
  free(temp);
  if(status) return;

  if(get_string(con,&info)) return;

  for(c1=firstconsumer; c1!=NULL; c1=c1->next) c1->mark=0;

  for(i=0;i<usercount;i++)
  {
    if(get_string(con,&temp)) return;
    flag=0; if(!(c1=search_consumer(temp))) flag=1;
    if(flag&&verbose)
      printf("  Oops: user %s unknown, he probably logged off\n",temp);
    free(temp);
    if(c1) c1->mark=1;
  }

  end_of_message(con);

  if(verbose)
  {
    printf("  %s informs availability of %d units of %s:\n",
	 c->uname, amount, xtea_resources[resource]);
    printf("  extra info: %s\n",info);
  }
  
  for(c1=firstconsumer;c1!=NULL;c1=c1->next)
    if(c1->mark)
    {
      if(verbose) printf("  %s",c1->uname);
      if(!start_message(c1->host, htons(c1->port), INFORM, &sendsock))
      {
	put_string(sendsock,c->fullname);
	put_string(sendsock,c->uname);
	sprintf(buf,"%d",resource);
	put_string(sendsock,buf);
	sprintf(buf,"%d",amount);
	put_string(sendsock,buf);
	put_string(sendsock,info);
	put_string(sendsock,"END");
	close(sendsock);
      }
      else printf("-ERR");
    }
  if(verbose) printf("\n");
  free(info);
}


/* ********************************* send_userlist() *************************
 *
 * Send a list of candidate invitees to a requesting candidate inviter.
 *
*/
static void send_userlist(int con)
{
  char *uname, *temp;
  Consumer *c,*c1;
  int resflag,n,sendsock;
  char buf[10];
  if(get_string(con,&uname)) return;
  if(get_string(con,&temp))
  {
    free(uname);
    return;
  }
  c=search_consumer(uname);
  if(!c)
  {
    if(verbose) printf("  Consumer %s not found\n",uname);
    free(uname);
    free(temp);
    return;
  }
  free(uname);
  if(sscanf(temp,"%d",&resflag)!=1)
  {
    if(verbose) printf("  Can't scan resource field %s\n",temp);
    free(temp);
    return;
  }
  free(temp);
  end_of_message(con);

  if(verbose) 
  {
    printf("  user %s queries list of users taking %s:\n",
	   c->uname, xtea_resources[resflag]);
    printf("   ");
  }

  n=0;
  for(c1=firstconsumer; c1!=NULL; c1=c1->next)
    if(c->group==c1->group && (c1->cflags & 1<<resflag)) 
    {
      n++;
      if(verbose) printf(" %s",c1->uname);
    }
  if(verbose)
    printf("\n  Found %d users with flag %d set\n",n,resflag);

  if(start_message(c->host, htons(c->port), USERLIST, &sendsock)) return;
  sprintf(buf,"%d",resflag);
  put_string(sendsock,buf);
  sprintf(buf,"%d",n);
  put_string(sendsock,buf);
  for(c1=firstconsumer; c1!=NULL; c1=c1->next)
    if(c->group==c1->group && (c1->cflags & 1<<resflag))
    { put_string(sendsock,c1->uname);
      put_string(sendsock,c1->fullname);
    }
  put_string(sendsock,"END");
  close(sendsock);
}


/* ********************************* save_state() ****************************
 *
 * We are about to exit. Save the internal state in file STATE_FILE.
 *
*/
static void save_state()
{
  int n;
  FILE *fp;
  Consumer *c;
  if(consumercount)
  {
    n=0;
    for(c=firstconsumer;c!=NULL;c=c->next) n++;
    if(n!=consumercount)
    {
      fprintf(stderr,"xtead warning: internal consumercount %d != actual value %d\n",
	     consumercount, n);
    }
    fp=fopen(STATE_FILE,"w");
    if(fp==NULL)
    {
      perror("Panic: can't create state file");
    }
    else
    {
      fprintf(fp,"%d\n",n);
      for(c=firstconsumer;c!=NULL;c=c->next)
      {
	 fprintf(fp,"%s\n",c->uname);
	 fprintf(fp,"%s\n",c->host);
	 fprintf(fp,"%s\n",c->fullname);
	 fprintf(fp,"%d\n",c->port);
	 fprintf(fp,"%d\n",c->pflags);
	 fprintf(fp,"%d\n",c->cflags);
	 fprintf(fp,"%d\n",c->group);
       }
      fclose(fp);
    }
  }
}


/* ****************************** read_string() *****************************
 *
 * Read a string from the state file.
 *
*/
static void read_string(FILE *fp, char **s)
{
  char buf[200];
  int l;
  if(fgets(buf,199,fp)==NULL)
  {
    perror("Error reading state file");
    fatal_error(1);
  }
  l=strlen(buf);
  /* remove closing newline */
  if(l>0 && buf[l-1]=='\n')
  {
    buf[l-1]=0;
    l--;
  }
  (*s)=myalloc(l+1);
  strcpy(*s,buf);
}


/* **************************** restore_state() ***************************
 *
 * Read an earlier saved state file to recover from a signal.
 *
*/
static void restore_state()
{
  int i,port,pflags,cflags,group,count;
  FILE *fp;
  char *s,*uname,*host,*fullname;

  fprintf(stderr, "Restoring saved state from %s\n",STATE_FILE);
  fp=fopen(STATE_FILE,"r");
  if(fp==NULL)
  {
    perror("can't open state file");
    fatal_error(1);
  }
  
  read_string(fp,&s);
  if(sscanf(s,"%d",&count)!=1)
  {
    fprintf(stderr,"  Error: can't parse consumercount in state file\n");
    fatal_error(1);
  }
  free(s);
  
  for(i=0;i<count;i++)
  {
    read_string(fp,&uname);
    read_string(fp,&host);
    read_string(fp,&fullname);
    read_string(fp,&s);
    if(sscanf(s,"%d",&port)!=1)
    {
      fprintf(stderr,"  Error: can't parse port for consumer #%d\n",i);
      fatal_error(1);
    }
    free(s);
    read_string(fp,&s);
    if(sscanf(s,"%d",&pflags)!=1)
    {
      fprintf(stderr,"  Error: can't parse pflags for consumer #%d\n",i);
      fatal_error(1);
    }
    free(s);
    read_string(fp,&s);
    if(sscanf(s,"%d",&cflags)!=1)
    {
      fprintf(stderr,"  Error: can't parse cflags for consumer #%d\n",i);
      fatal_error(1);
    }
    free(s);
    read_string(fp,&s);
    if(sscanf(s,"%d",&group)!=1)
    {
      fprintf(stderr,"  Error: can't parse group for consumer #%d\n",i);
      fatal_error(1);
    }
    free(s);
    new_consumer(uname,host,fullname,port,pflags,cflags,group);
  }
  fprintf(stderr,"  Re-initialized with %d consumers\n",consumercount);
  fclose(fp);
}
  

/* *************************** catchint() **********************************
 * 
 * We got a signal - dump internal tables and exit.
 *
*/
static void catchint()
{
  signal(SIGINT,  SIG_DFL);
  signal(SIGQUIT, SIG_DFL);
  signal(SIGPIPE, SIG_DFL);
  signal(SIGTERM, SIG_DFL);
  fprintf(stderr, "\n\nxtead got signal - saving state and aborting\n");
  save_state();
  exit(1);
}


/* ********************************* fatal_error() **************************
 *
 * We are panicking.
 *
*/
void fatal_error(int code)
{
  fprintf(stderr, "\n\nfatal internal error - saving state and aborting\n");
  save_state();
  exit(code);
}


void main(int argc, char **argv)
{
  struct servent *se;
  struct sockaddr_in acc_sin;
  int acc_sin_len;
  int con,sock,sendsock,port;
  char *s;
  int type;
  Consumer *c;
  int ch,errflg,restore;
  extern int optind;

  firstconsumer=lastconsumer=NULL;
  consumercount=0;

  restypes=resource_count();

  errflg=0;
  restore=0;
  verbose=0;
  while((ch=getopt(argc,argv,"rv"))!=EOF)
    switch(ch)
    {
    case 'r':
      restore=1;
      break;
    case 'v':
      verbose=1;
      break;
    case '?':
      errflg++;
      break;
    }
  if(errflg || optind!=argc)
  {
    fprintf(stderr,"usage: %s [-r] [-v]\n",argv[0]);
    fatal_error(1);
  }

  if(verbose) fprintf(stderr,"xtea daemon version %s\n",XTEA_VERSION);

  if(restore)
    restore_state();

  signal(SIGINT, catchint);
  signal(SIGQUIT, catchint);
  signal(SIGPIPE, catchint);
  signal(SIGTERM, catchint);
  
#ifndef PORTNUMBER
  se=getservbyname(XTEASERVICE,"tcp");
  if(se==NULL)
  { perror("getservbyname");
    fatal_error(2);
  }
  port=se->s_port;
#else
  port=PORTNUMBER;
#endif
  init_listen(port, &sock);

  while(1)
  {
    acc_sin_len=sizeof(acc_sin); 
    con = accept(sock,(struct sockaddr *) &acc_sin, &acc_sin_len);
    if(con<0)
    { perror("accept");
      fatal_error(2);
    }
    
    if(!get_string(con,&s))
    {
      if((type=parse_msgtype(s))>=0)
      {
	if(verbose)
	  printf("Got message type %d = %s\n",type,msgtype[type]);
	switch(type)
	{
	case STARTUP:
	  c=get_consumer_data(con);
	  if(!c)
	  { printf("Can't get consumer data!\n");
	    break;
	  }
	  if(!start_message(c->host, htons(c->port), ACK, &sendsock))
	  {
	    put_string(sendsock,"END");
	    close(sendsock);
	  }
	  break;
	case INFORMSERVER:
	  distribute_inform(con);
	  break;
	case ORDERSERVER:
	  propagate_order(con);
	  break;
	case CONFIRMSERVER:
	  propagate_confirm(con);
	  break;
	case OFFER:
	  send_userlist(con);
	  break;
	case GOODBYE:
	  consumer_quits(con);
	  break;
	case QUERYSTATUS:
	  send_status(con);
	  break;
	default:
	  while(!get_string(con,&s) && parse_msgtype(s)!=END)
	  {
	    puts(s);
	    free(s);
	  }
	  close(con);
	  break;
	}
      }
      else
      {
	fprintf(stderr,"xtead: Unknown message %s\n",s);
	close(con);
      }
      free(s);
      purge_consumers();
    }
    else
      close(con);
  }
}


