/*
  Xquote Copyright (c) 1997 Mark Buser,
  All Rights Reserved.

  Permission is hereby granted to copy
  and freely distribute copies of this
  program for non-commercial purposes
  without fee, provided that this notice
  appears in all copies.
  
  All redistributions must be in their
  entirety as originally distributed.
  Feel free to modify Xquote, but
  modified versions may not be distributed
  without prior consent of the author.
  
  This software is provided 'as-is'
  without any express or implied warranty.
  In no event will the author be held
  liable for any damages resulting from
  the use of this software.

  $Revision: 1.6 $ $Date: 1997/11/02 16:46:51 $
*/

#include <ctype.h>
#include <stdlib.h>
#include <string.h>

#include <Xm/XmAll.h>

#include "optdetail.h"
#include "parse.h"
#include "pixmap.h"
#include "server.h"
#include "status.h"
#include "tape.h"
#include "xquote.h"
#include "xutil.h"

/* Externs */
extern Widget Toplevel;

typedef struct tick_struct {
  char *name;
  int type;
  QUERY_STRUCT *detail;
} TICK_STRUCT;

/* Globals */
static XmStringTable save;
static int  saveCount = 0;
static TICK_STRUCT **tickTable;
static int numTick = 0;
static int curType = 0;

static Widget menu;

#define TICKFIELDS 5

/*
** Ticker Symbol Support
*/

/* 
** Session save and restore functions 
*/
char *tickGetItem (int pos)
{
  char *item;
  char *ticker = NULL;
  char *next;

  if (pos >= 0 && pos < numTick) {
    XmStringGetLtoR ( save[pos], XmFONTLIST_DEFAULT_TAG, &item);
    ticker = XtCalloc ( strlen(item)+1, sizeof (char) );

    next = strchr ( item, ' ');             /* Find delimiter */
    *next++ = '\0';
    while ( isspace(*next) ) 
      next++;
    sprintf ( ticker, "%s %s", item, next); /* Strip spaces */

    XtFree (item);
  }
  return (ticker);   
}


void tickAddItem (char *item)
{
  char *line, *next;

  save = (XmStringTable) XtRealloc ( (char *)save, 
		                     (saveCount+1) * sizeof(XmString) );
  if (save) {
    next = strchr ( item, ' ');  /* break item into ticker and type */
    *next++ = '\0';

    line = XtCalloc ( 7 + strlen(next) +1, sizeof(char) );
    sprintf ( line, "%-6.6s %s", item, next);
    save[saveCount++] = XmStringCreateLocalized (line);
    XtFree (line);
  }
}

int tickGetNum ()
{
  return (numTick);
}

/* Get saved ticker type */
int tickGetType ( int which )
{
  if (which < numTick)
    return ( tickTable[which]->type );
  else
    return (0);
}

/* Get saved ticker name */
char *tickGetName ( int which )
{
  if (which < numTick)
    return ( tickTable[which]->name );
  else
    return ((char *)NULL);
}

/* Get saved data retrieved from net for displaying */
QUERY_STRUCT *tickGetDetail ( int which )
{
  if (which < numTick)
    return ( tickTable[which]->detail );
  else
    return ((QUERY_STRUCT *)NULL);
}


/* Save data retrieved from net for displaying */
void tickSetDetail ( int which, QUERY_STRUCT *data )
{

  /* Update the data structure */
  if ( which < numTick  ) {                   /* Replace existing */
    freePattern (tickTable[which]->detail);
    tickTable[which]->detail = data;
  } 
  
  /* Update the main window display */
}

/* Add everthing from the tick editor list to the main window list */
void tickAddMainWindow ()
{
  int  i, j;
  char *next;
  char *ticker;
  char *type;
  char *item;

  for (i=0; i < saveCount; i++) {

    /* Get ticker symbol */
    XmStringGetLtoR ( save[i], XmFONTLIST_DEFAULT_TAG, &item);
    
    /* I hate C string functions */
    next = strchr ( item, ' ');       /* Find delimiter */
    *next++ = '\0';
    while ( isspace(*next) ) 
      next++;
    ticker = XtNewString (item);
    type = XtNewString (next);        /* Spaces here aren't delimiters */

    XtFree (item);

    /* Get ticker type */
    /* 
    ** !Doesn't get possible second word of server type title!
    */
    for (j=0; j < numServerType(); j++) {
      setServerType(j);
      if (strcmp (type, getServer(TITLE)) == 0)
        break;
    }
    if (j == numServerType()) {
      sprintf ( errmsg,
	        "Can not find server type '%s', using first type.\n"
	        "This may cause table values to not be found.",
		type);
      write_status (errmsg, WARN);
      j = 0;
    }

    /* Make a structure with name and details for each ticker symbol */

    /* If ticker name changed reuse, else leave alone */
    if ( i < numTick && strcmp(ticker, tickTable[i]->name) ) {
      freePattern (tickTable[i]->detail);
      XtFree (tickTable[i]->name);

      tickTable[i]->name = XtNewString (ticker);
      tickTable[i]->type = j;
      tickTable[i]->detail = (QUERY_STRUCT *)NULL;
    }

    /* Add new ticker */
    if ( i >= numTick ) {
      tickTable = (TICK_STRUCT **) XtRealloc ( (char *)tickTable, 
                                               (i+1) * sizeof(TICK_STRUCT **) );
      tickTable[numTick++] = (TICK_STRUCT *) XtCalloc ( 1, sizeof(TICK_STRUCT));

      tickTable[i]->name = XtNewString (ticker);
      tickTable[i]->type = j;
      tickTable[i]->detail = (QUERY_STRUCT *)NULL;
    }

    XtFree (ticker);
    XtFree (type);
  }

  for (; i < numTick; i++) {                  /* Blow away extras */
    freePattern (tickTable[i]->detail);
    XtFree (tickTable[i]->name);
  }

  numTick = saveCount;
  detailUpdateMainWindow ();
  makeTextTape();
}


/* If symbol is in list, return its position, else return 0 */
static int tickIsDuplicate ( Widget list, char *symbol )
{
  XmStringTable ticklist;
  char *item;
  char name[7];
  int i, count;

  XtVaGetValues ( list, XmNitemCount, &count,
		        XmNitems,     &ticklist,
	          NULL);
  for ( i=0; i < count; i++) {
    XmStringGetLtoR ( ticklist[i], XmFONTLIST_DEFAULT_TAG, &item);
    sscanf ( item, "%6s", name );
    if (strcasecmp ( name, symbol) == 0 ) {
      XtFree(item);
      return (i+1);
    }
    XtFree(item);
  }
  return 0;  
}

/* Callback for ticker symbol dialog */
/* ARGSUSED */
static void procTick (Widget w, int client_data, XtPointer call_data)
{
  static Widget list = (Widget) NULL;
  static WidgetList field = (WidgetList)NULL;
  char *symbol[TICKFIELDS];

  XmStringTable orig;
  int origCount;

  if (list == (Widget) NULL) {
    list = XtNameToWidget (GetTopShell(w), 
                 "StockPane.TickForm.StockListSW.StockList");
    if (list) 
      /* So we can find the text fields to refill */
      XtVaGetValues (list, XmNuserData, &field, NULL);
  }

  switch (client_data) {
 
    case 0: /* Ok */
            XtPopdown (GetTopShell(w));

	    /* Save list contents */
	    if (saveCount) {
	      for (origCount=0; origCount < saveCount; origCount++)
	        XmStringFree (save[origCount]);
	      XtFree ((char *)save);
	      saveCount = 0;
	    }
	    XtVaGetValues (list, 
			   XmNitemCount, &saveCount, 
			   XmNitems, &orig, 
			   NULL);
	    if (saveCount) {
	      save = (XmStringTable) XtCalloc (saveCount, sizeof(XmString));
	      if (save) {
	        for (origCount=0; origCount < saveCount; origCount++)
	          save[origCount] = XmStringCopy (orig[origCount]);
	      } else
		saveCount = 0;
	    }

            /* Update main window table */
	    tickAddMainWindow ();

            break;

    case 1: /* Cancel */
            XtPopdown (GetTopShell(w));

	    /* Restore previous list */
	    XmListDeleteAllItems (list);
	    if (saveCount)
	      XmListAddItems (list, save, saveCount, 1);
            break;

    case 2: /* Add */
            {
              XmString listItem;
              char *line;

              /* Get ticker symbol */
              if (field[0]) {
                symbol[0] = XmTextFieldGetString (field[0]);

                /* If not blank or duplicate, add to list */
                if (strlen(symbol[0])) {
		  /* Not Blank */
	          if ( !tickIsDuplicate (list, symbol[0]) ) {
		    /* and Not Duplicate */
		    setServerType (curType);
		    symbol[1] = getServer(TITLE);

                    line = XtCalloc (7 + strlen(symbol[1]) +1, sizeof (char));
                    sprintf (line, "%-6.6s %s", symbol[0], symbol[1]);
                    listItem = XmStringCreateLocalized (line);
                    XmListAddItem (list, listItem, 0);
		    XmListSetBottomPos (list, 0);
                    XtFree (line);
                    XmStringFree (listItem);
                  } else {
		    /* and Duplicate */
		    sprintf (errmsg, "Symbol '%s' is already in the list.", 
			     symbol[0]);
		    write_status (errmsg, ERR);
		  }
	        } else {
		  /* Blank */
	          write_status ("Symbol is empty.", ERR);
	        }

                XtFree (symbol[0]);
                XmTextFieldSetSelection (field[0], 
                                         0, XmTextGetLastPosition(field[0]),
                                         CurrentTime);
              } else
	        write_status ("Internal error, no ticker text widget!", ERR);
            }
            break;

    case 3: /* Remove */
            {
              char line[80];
	      int pos;

              /* Get ticker symbol */
              if (field[0]) {
                symbol[0] = XmTextFieldGetString (field[0]);

                /* If not blank and exists, remove from list */
                if (strlen(symbol[0]) && 
	  	     (pos = tickIsDuplicate (list, symbol[0])) ) {
	          XmListDeletePos (list, pos); 
	        } else {
		  sprintf (line, "Symbol '%s' is not in the list.", symbol[0]);
		  write_status (line, ERR);
	        }

                XtFree (symbol[0]);
                XmTextFieldSetSelection (field[0], 
                                         0, XmTextGetLastPosition(field[0]),
                                         CurrentTime);
              } else
	        write_status ("Internal error, no ticker text widget!", ERR);
            }
            break;

#if 0
    case 4: /* Change */
            {
              XmString listItem;
              char line[80];
              int i, pos;

              /* Get ticker symbol */
              symbol[0] = XmTextFieldGetString (field[0]);

              /* If not blank and exists, change it */
              if ( strlen(symbol[0]) && 
		   (pos = tickIsDuplicate (list, symbol[0])) ) {

                for (i=1; i < TICKFIELDS; i++) {
		  if (field[i]) {
                    symbol[i] = XmTextFieldGetString (field[i]);

		    /* Error check for numbers */
		    if (i > 1 && strlen(symbol[i])) {
		      double number; /* converted number */
		      char *end;     /* pointer to last char converted */

		      number = strtod ( symbol[i], &end );
                      if ( end >= symbol[i] &&
		  	   end < (symbol[i] + strlen (symbol[i])) ) {
                        XmTextSetSelection ( field[i], 0, 
				             XmTextGetLastPosition( field[i] ), 
		  			     CurrentTime );
		        sprintf (line, "'%s' is not a valid number.", symbol[i]);
                        write_status (line, ERR);
                        for (; i >= 0; i--)
                          XtFree (symbol[i]);
                        return;
                      }
		    }
		  } else 
		    symbol[i] = XtNewString("");
		}

	        XmListDeletePos (list, pos); 

                sprintf ( line, "%-9.9s %-9.9s %9.9s %9.9s %9.9s ", symbol[0],
                          strlen(symbol[1])?symbol[1]:" ", 
                          strlen(symbol[2])?symbol[2]:" ", 
                          strlen(symbol[3])?symbol[3]:" ", 
                          strlen(symbol[4])?symbol[4]:" "
                        );
                listItem = XmStringCreateLocalized ( line );
                XmListAddItem ( list, listItem, pos);
                XmStringFree (listItem);

                for (i=1; i < TICKFIELDS; i++)
                  XtFree (symbol[i]);
              } else {
		sprintf ( line, "Symbol '%s' is not in the list.", 
			  symbol[0]);
		write_status ( line, ERR );
	      }
              XtFree( symbol[0] );
            }
            break;
#endif

    default: break;
  }

}


/* ARGSUSED */
static void textTick (Widget w, XtPointer client_data, XtPointer call_data)
{
#if 0
  /* Move to next text field when 'enter' hit */
  XmProcessTraversal (w, XmTRAVERSE_NEXT);
#else
  /* Simulate add button */
  procTick (w, 2, (XtPointer) NULL);
#endif

}


/* Read list params back into text fields when list item clicked on */
/* ARGSUSED */
static void loadTick (Widget w, XtPointer client_data, 
                      XmListCallbackStruct *call_data)
{
  int i;
  char *next;
  char field[TICKFIELDS][10];
  WidgetList textfield;

  /* Get the text field widgets */
  XtVaGetValues (w, XmNuserData, &textfield, NULL);

  XmStringGetLtoR ( call_data->item, XmFONTLIST_DEFAULT_TAG, &next );
  sscanf ( next, "%10c%10c%10c%10c%10c",         /* 9 for symbol, 1 space */
	   &(field[0][0]), &(field[1][0]), &(field[2][0]), 
           &(field[3][0]), &(field[4][0]) );

  /* Assume that the ticker symbol is never NULL or blanks */
  for ( i=0; i < TICKFIELDS; i++) {
    char temp[9];

    field[i][9] = '\0';                          /* end string */

    if (textfield[i]) {
      if (sscanf ( field[i], "%s", temp) == 1)   /* skip blanks */
        XmTextFieldSetString ( textfield[i], temp );
      else
        XmTextFieldSetString ( textfield[i], "" );
    }
  }
  XtFree (next);
}

/* ARGSUSED */
static void tickTypeCB ( Widget w, XtPointer client_data, XtPointer call_data)
{
  curType = (int)client_data;
}

void tickMakeTypeMenu()
{
  extern Widget menu;
  WidgetList buttons;
  int num;
  Widget button;
  int types;

  if (menu == (Widget) NULL)
    return;

  /* Out with the old... */
  XtVaGetValues ( menu,
		  XmNchildren, &buttons,
		  XmNnumChildren, &num,
		  NULL );
  for (types=0; types < num; types++)
    XtDestroyWidget ( buttons[types] );

  /* in with the new. */
  for (types = 0; types < numServerType(); types++) {
    setServerType(types);
    button = XtVaCreateManagedWidget ( getServer(TITLE),
		                       xmPushButtonGadgetClass, menu, NULL);
    XtAddCallback ( button, XmNactivateCallback, tickTypeCB, (XtPointer)types);
  }
  setServerType (0);
}


Widget createTickDialog ()
{
  Widget dialog, form, pane, row, list, button, option;
  static Widget text[] = { (Widget)NULL, (Widget)NULL, (Widget)NULL, 
                           (Widget)NULL, (Widget)NULL };

  Dimension width, height, border;

  int num;
  Arg args[10];

#if 0  /* alarm, exchange */
  char *labels[] =  { "label_0", "label_1", "label_2", 
                      "label_3", "label_4" };
  char *texts[] =  { "text_0", "text_1", "text_2", 
                     "text_3", "text_4" };
  char *buttons[] = { "button_0", "button_1", "button_2", 
                      "button_3", "button_4" };
#else /* just ticker name */
  char *labels[] = { "label_0", "label_1"};
  char *texts[] = { "text_0"};
  char *buttons[] = { "button_0","button_1","button_2","button_3"};
#endif


  dialog = XtVaCreatePopupShell ("OptionTick", 
                            xmDialogShellWidgetClass,
                            GetTopShell (Toplevel),
                            NULL);

  pane = XtVaCreateWidget ("StockPane", xmPanedWindowWidgetClass, 
                            dialog,
                            XmNsashWidth, 1,
                            XmNsashHeight, 1,
                            NULL);

  /* form to hold rowcolumn form, scrolled list */
  form = XtVaCreateWidget ("TickForm", xmFormWidgetClass, pane, NULL);

  row = XtVaCreateWidget ("TickRow", xmRowColumnWidgetClass, form,
                            XmNnavigationType, XmNONE,
                            XmNtopAttachment, XmATTACH_FORM,
                            XmNleftAttachment, XmATTACH_FORM,
                            XmNnumColumns, XtNumber(labels),
                            XmNorientation, XmHORIZONTAL,
                            XmNpacking, XmPACK_COLUMN,
                            NULL);
                
  for (num=0; num < XtNumber(labels); num++) {
    /* Label describing entry */
    XtVaCreateManagedWidget (labels[num], xmLabelGadgetClass, row, NULL);

    /* Text widget for number entry */
    if (num == 0) {
      text[num] = XtVaCreateManagedWidget (texts[num],
                            xmTextFieldWidgetClass, row,
                            XmNnavigationType, XmNONE,
                            NULL);
      XtAddCallback (text[num], XmNactivateCallback,
                     (XtCallbackProc) textTick, (XtPointer) num);
    } else {
      menu = XmCreatePulldownMenu (row, "TickOptionMenu_pulldown", NULL, 0);
      XtSetArg (args[0], XmNsubMenuId, menu);
      option = XmCreateOptionMenu (row, "TickOptionMenu", args, 1);
      tickMakeTypeMenu();
      XtManageChild (option);
    }
  }
  XtManageChild (row);

  /* List holding what's been added to date */
  num = 0;
  XtSetArg(args[num], XmNscrollBarDisplayPolicy, XmSTATIC); num++;
  XtSetArg(args[num], XmNvisibleItemCount, 5); num++;
  XtSetArg(args[num], XmNselectionPolicy,  XmSINGLE_SELECT); num++;
  XtSetArg(args[num], XmNtopAttachment,    XmATTACH_WIDGET); num++;
  XtSetArg(args[num], XmNtopWidget,        row); num++;
  XtSetArg(args[num], XmNleftAttachment,   XmATTACH_FORM); num++;
  XtSetArg(args[num], XmNrightAttachment,  XmATTACH_FORM); num++;
  /* Needed when we restore ticker symbols and then try to delete them 
     (loadTick). procTick is not called yet, so fields[] is not valid. */
  XtSetArg(args[num], XmNuserData, text); num++;
  list = XmCreateScrolledList (form, "StockList", args, num);

  XtAddCallback (list, XmNdefaultActionCallback,
                        (XtCallbackProc) loadTick, NULL);
  XtManageChild (list);
  XtManageChild (form);

  /* if we have restored items, load em up */
  if (saveCount)
    XmListAddItems (list, save, saveCount, 1);

  /* Buttons to add delete cancel exit */
  form = XtVaCreateWidget ("ButForm", xmFormWidgetClass, pane,
                            XmNfractionBase, XtNumber(buttons),
                            NULL);

  for (num=0; num < XtNumber(buttons); num++) {
    button = XtVaCreateManagedWidget (buttons[num], 
                            xmPushButtonWidgetClass, form,
                            XmNtopAttachment, XmATTACH_FORM,
                            XmNbottomAttachment, XmATTACH_FORM,
                            XmNleftAttachment, XmATTACH_POSITION,
                            XmNleftPosition, num,
                            XmNrightAttachment, XmATTACH_POSITION,
                            XmNrightPosition, num+1,
                            XmNshowAsDefault, (num==0)?True:False,
                            XmNdefaultButtonShadowThickness, 1,
                            NULL);
    XtAddCallback (button, XmNactivateCallback,
                            (XtCallbackProc) procTick, (XtPointer)num);
  }
  XtManageChild (form);
  XtManageChild (pane);

  /* Prevent pane from changing size */
  XtVaGetValues (dialog, 
                 XmNwidth, &width,
                 XmNheight, &height,
                 XmNborderWidth, &border,
                 NULL);

  XtVaSetValues (dialog,
                 XmNminWidth,  width +  border,
                 XmNmaxWidth,  width +  border,
                 XmNminHeight, height + border,
                 XmNmaxHeight, height + border,
                 NULL);
  return (dialog);
}
