/* mif.c -- routines for producing MIF output from xdatplot		*/
/*
 * Copyright (c) 1993  Leon Avery
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Send questions or comments on xdatplot to:
 *
 * Leon Avery
 * Department of Biochemistry
 * University of Texas Southwestern Medical Center
 * 5323 Harry Hines Blvd
 * Dallas, TX  75235-9038
 *
 * leon@eatworms.swmed.edu
 */

#include "xdatplot.h"
#include <pwd.h>

#define	PLSIZE	50
#define	HEADER	"<MIFFile 3.00> # Generated by xdatplot %s\n"
#define	FONT_FAMILY	"Courier"

#ifdef	__STDC__
static	void	MIFplot_data(FILE *);
static	void	MIFplot_marks(FILE *);
static	void	MIFplot_axes(FILE *);
#else	/* __STDC__ */
static	void	MIFplot_data();
static	void	MIFplot_marks();
static	void	MIFplot_axes();
#endif	/* __STDC__ */

static	int	group_id;

/* MIFplot -- write the current plot to the named file			*/
void
MIFplot(fname)
String		fname;
{
    char	ebuf[LLEN];
    FILE	*os;

    if (NULL == (os = fopen(fname, "w"))) {
	sprintf(ebuf, "unable to open %s for printing", fname);
	PU_error(ebuf, "file.html#MIF");
	return;
    }
    MIFplot_stream(os);
    fclose(os);
}

/* MIFplot_stream -- write the current plot to the open stream		*/
void
MIFplot_stream(os)
FILE		*os;
{
    struct passwd	*pswd;

    fprintf(os, HEADER, VERSION);
    fprintf(os, "# File %s, t in [%ld, %ld)\n", DATA_FILE,
	    (long) TL, (long) TR);
    pswd = getpwuid(getuid());
    fprintf(os, "# for %s \"%s\"\n", pswd->pw_name, pswd->pw_gecos);
    group_id = 1;
    fprintf(os, " <Units Upt>\n");
    MIFplot_data(os);
    MIFplot_marks(os);
    MIFplot_axes(os);
}

static void
MIFplot_data(os)
FILE		*os;
{
    register TIME	i;
    register TIME	j;
    int			ggid = group_id++;
    char		god[LLEN];	/* Generic Object Data		*/
    TIME		n;

    for(j = TL + 1; j < TR; j += PLSIZE) {
	sprintf(god, "  <GroupID %d>\n", group_id);
	strcat(god, "  <PenWidth 0.25>\n");
	strcat(god, "  <NumPoints 2>\n");
	strcat(god, "  <HeadCap Round>\n");
	strcat(god, "  <TailCap Round>\n");
	n = min(TR, j + PLSIZE);
	for(i=j; i<n; i++) {
	    fprintf(os, " <PolyLine\n%s  <Point %g %g>\n  <Point %g %g>\n > # end of PolyLine\n",
		    god,
		    (double) Map_TIME_to_Dim_c(i-1),
		    (double) Map_VOLTAGE_to_Dim_c(Vf(i-1)),
		    (double) Map_TIME_to_Dim_c(i),
		    (double) Map_VOLTAGE_to_Dim_c(Vf(i)));
	    if (debug) fflush(os);
	}
	fprintf(os, " <Group\n  <GroupID %d>\n  <ID %d>\n >\n",
		ggid, group_id);
	group_id++;
    }
    fprintf(os, " <Group\n  <ID %d>\n >\n", ggid);
}

static void
MIFplot_marks(os)
FILE		*os;
{
    MARK		*mp;
    int			n;
    register int	i;
    double		x, y;
    double		st, en;

    n = lookup_interval(Map_TIME_to_MARK(TL), Map_TIME_to_MARK(TR), &mp);
    if (n > 0) {
	fprintf(os, " <Font\n");
	fprintf(os, "  <FFamily %s>\n", FONT_FAMILY);
	fprintf(os, "  <FSize %d>\n", PFINFO->ascent + PFINFO->descent);
	fprintf(os, " >\n");
	if (debug) fflush(os);

	st = -MARK_SIZE / 2;
	en = st + MARK_SIZE;
	for(i=0; i<n; i++) {
	    x = Map_MARK_to_Dim_c(mp[i].t);
	    y = Map_VOLTAGE_to_Dim_c(Vf(Map_MARK_to_TIME(mp[i].t)));
	    fprintf(os, " <PolyLine\n");
	    fprintf(os, "  <GroupID %d>\n", group_id);
	    fprintf(os, "  <PenWidth %d>\n", MARK_THICKNESS);
	    fprintf(os, "  <NumPoints 2>\n");
	    fprintf(os, "  <HeadCap Round>\n");
	    fprintf(os, "  <TailCap Round>\n");
	    fprintf(os, "  <Point %g %g>\n",
		    (double) (x + st), (double) (y + st));
	    fprintf(os, "  <Point %g %g>\n",
		    (double) (x + en), (double) (y + en));
	    fprintf(os, " > #end of PolyLine\n");
	    if (debug) fflush(os);

	    fprintf(os, " <PolyLine\n");
	    fprintf(os, "  <GroupID %d>\n", group_id);
	    fprintf(os, "  <PenWidth %d>\n", MARK_THICKNESS);
	    fprintf(os, "  <NumPoints 2>\n");
	    fprintf(os, "  <HeadCap Round>\n");
	    fprintf(os, "  <TailCap Round>\n");
	    fprintf(os, "  <Point %g %g>\n",
		    (double) (x + en), (double) (y + st));
	    fprintf(os, "  <Point %g %g>\n",
		    (double) (x + st), (double) (y + en));
	    fprintf(os, " > #end of PolyLine\n");
	    if (debug) fflush(os);

	    y = PRY + PFINFO->ascent + 2;
	    fprintf(os, " <TextLine\n");
	    fprintf(os, "  <GroupID %d>\n", group_id);
	    fprintf(os, "  <TLOrigin %g %g>\n", x, y);
	    fprintf(os, "  <TLAlignment Center>\n");
	    fprintf(os, "  <String %c>\n", mp[i].val);
	    fprintf(os, " > # end of TextLine\n");
	    if (debug) fflush(os);
	}
	fprintf(os, " <Group\n");
	fprintf(os, "  <ID %d>\n", group_id);
	fprintf(os, " > # end of Group\n");
	if (debug) fflush(os);
	group_id++;
    }
}

static void
MIFplot_axes(os)
FILE		*os;
{
    register int	i;
    register double	tp;
    TICS		tics;
    double		x, y;
    double		xmin;
    char		lbuf[LLEN];

    fprintf(os, " <Font\n");
    fprintf(os, "  <FFamily %s>\n", FONT_FAMILY);
    fprintf(os, "  <FSize %d>\n", PFINFO->ascent + PFINFO->descent);
    fprintf(os, " >\n");
    if (debug) fflush(os);

    fprintf(os, " <Rectangle\n");
    fprintf(os, "  <GroupID %d>\n", group_id);
    fprintf(os, "  <PenWidth 0.25>\n");
    fprintf(os, "  <Fill 15>\n");
    fprintf(os, "  <BRect %d %d %d %d>\n", PRX-1, PRY-1, PRW+1, PRH+1);
    fprintf(os, " > # end of Rectangle\n");
    if (debug) fflush(os);
    /*
     * put tic marks on the t axis
     */
    y = PRY + PRH + PFINFO->ascent + 2;
    tics.tcmin = T_TICS;
    tics.lo = Map_TIME_to_t(TL + V_TICLEN / TIME_to_Dim_sf);
    tics.hi = Map_TIME_to_t(TR - V_TICLEN / TIME_to_Dim_sf);
    find_tics(&tics);
    for(i = tics.first; i < tics.n + tics.first; i++) {
	tp = i * tics.space;
	x = Map_t_to_Dim_c(tp);
	fprintf(os, " <TextLine\n");
	fprintf(os, "  <GroupID %d>\n", group_id);
	fprintf(os, "  <TLOrigin %g %g>\n", x, y);
	fprintf(os, "  <TLAlignment Center>\n");
	fprintf(os, "  <String %g>\n", tp);
	fprintf(os, " > # end of TextLine\n");
	if (debug) fflush(os);
	
	fprintf(os, " <PolyLine\n");
	fprintf(os, "  <GroupID %d>\n", group_id);
	fprintf(os, "  <PenWidth 0.25>\n");
	fprintf(os, "  <NumPoints 2>\n");
	fprintf(os, "  <HeadCap Round>\n");
	fprintf(os, "  <TailCap Round>\n");
	fprintf(os, "  <Point %g %g>\n",
		(double) Map_t_to_Dim_c(tp), (double) PRY);
	fprintf(os, "  <Point %g %g>\n",
		(double) Map_t_to_Dim_c(tp), (double) (PRY + T_TICLEN));
	fprintf(os, " > #end of PolyLine\n");
	if (debug) fflush(os);

	fprintf(os, " <PolyLine\n");
	fprintf(os, "  <GroupID %d>\n", group_id);
	fprintf(os, "  <PenWidth 0.25>\n");
	fprintf(os, "  <NumPoints 2>\n");
	fprintf(os, "  <HeadCap Round>\n");
	fprintf(os, "  <TailCap Round>\n");
	fprintf(os, "  <Point %g %g>\n",
		(double) Map_t_to_Dim_c(tp), (double) (PRY + PRH - 1));
	fprintf(os, "  <Point %g %g>\n",
		(double) Map_t_to_Dim_c(tp),
		(double) (PRY + PRH - 1 - T_TICLEN));
	fprintf(os, " > #end of PolyLine\n");
	if (debug) fflush(os);
    }
    /*
     * put tic marks on the v axis
     */
    y = (PFINFO->ascent - PFINFO->descent)/2;
    tics.tcmin = V_TICS;
    tics.lo = Map_VOLTAGE_to_v(VB + T_TICLEN / VOLTAGE_to_Dim_sf);
    tics.hi = Map_VOLTAGE_to_v(VT - T_TICLEN / VOLTAGE_to_Dim_sf);
    find_tics(&tics);
    xmin = PRX;
    for(i = tics.first; i < tics.n + tics.first; i++) {
	tp = i * tics.space;
	x = PRX - PFINFO->min_bounds.width;
	sprintf(lbuf, "%g", tp);
	xmin = min(xmin, x - XTextWidth(PFINFO, lbuf, strlen(lbuf)));
	if (x < 0) x = 0;
	fprintf(os, " <TextLine\n");
	fprintf(os, "  <GroupID %d>\n", group_id);
	fprintf(os, "  <TLOrigin %g %g>\n", x, y + Map_v_to_Dim_c(tp));
	fprintf(os, "  <TLAlignment Right>\n");
	fprintf(os, "  <String %s>\n", lbuf);
	fprintf(os, " > # end of TextLine\n");
	if (debug) fflush(os);

	fprintf(os, " <PolyLine\n");
	fprintf(os, "  <GroupID %d>\n", group_id);
	fprintf(os, "  <PenWidth 0.25>\n");
	fprintf(os, "  <NumPoints 2>\n");
	fprintf(os, "  <HeadCap Round>\n");
	fprintf(os, "  <TailCap Round>\n");
	fprintf(os, "  <Point %g %g>\n",
		(double) PRX, (double) Map_v_to_Dim_c(tp));
	fprintf(os, "  <Point %g %g>\n",
		(double) (PRX + V_TICLEN), (double) Map_v_to_Dim_c(tp));
	fprintf(os, " > #end of PolyLine\n");
	if (debug) fflush(os);

	fprintf(os, " <PolyLine\n");
	fprintf(os, "  <GroupID %d>\n", group_id);
	fprintf(os, "  <PenWidth 0.25>\n");
	fprintf(os, "  <NumPoints 2>\n");
	fprintf(os, "  <HeadCap Round>\n");
	fprintf(os, "  <TailCap Round>\n");
	fprintf(os, "  <Point %g %g>\n",
		(double) (PRX + PRW - 1), (double) Map_v_to_Dim_c(tp));
	fprintf(os, "  <Point %g %g>\n",
		(double) (PRX + PRW - 1 - V_TICLEN),
		(double) Map_v_to_Dim_c(tp));
	fprintf(os, " > #end of PolyLine\n");
	if (debug) fflush(os);
    }
    /*
     * place all the various headers
     */
    x = PRX + PRW/2;
    y = PRY + PRH + PFINFO->ascent + 2 + PFINFO->ascent + PFINFO->descent;
    y = max(y, 0);
    fprintf(os, " <TextLine\n");
    fprintf(os, "  <GroupID %d>\n", group_id);
    fprintf(os, "  <TLOrigin %g %g>\n", x, y);
    fprintf(os, "  <TLAlignment Center>\n");
    fprintf(os, "  <String %s>\n", T_UNITS);
    fprintf(os, " > # end of TextLine\n");
    if (debug) fflush(os);
	
    x = xmin - PFINFO->min_bounds.width;
    x = max(x, 0);
    y = PRY + PRH/2 + (PFINFO->ascent - PFINFO->descent)/2;
    y = max(y, 0);
    fprintf(os, " <TextLine\n");
    fprintf(os, "  <GroupID %d>\n", group_id);
    fprintf(os, "  <TLOrigin %g %g>\n", x, y);
    fprintf(os, "  <TLAlignment Right>\n");
    fprintf(os, "  <String %s>\n", V_UNITS);
    fprintf(os, " > # end of TextLine\n");
    if (debug) fflush(os);
	
    x = PRX + PRW/2;
    y = PRY - PHFINFO->descent - 2;
    y = max(y, 0);
    fprintf(os, " <TextLine\n");
    fprintf(os, "  <GroupID %d>\n", group_id);
    fprintf(os, "  <TLOrigin %g %g>\n", x, y);
    fprintf(os, "  <TLAlignment Center>\n");
    fprintf(os, "  <Font\n");
    fprintf(os, "   <FSize %d>\n", PHFINFO->ascent + PHFINFO->descent);
    fprintf(os, "  >\n");
    fprintf(os, "  <String %s>\n", DATA_FILE);
    fprintf(os, " > # end of TextLine\n");
    if (debug) fflush(os);
	
    fprintf(os, " <Group\n");
    fprintf(os, "  <ID %d>\n", group_id);
    fprintf(os, " > # end of Group\n");
    if (debug) fflush(os);
    group_id++;
}
