/* marks.c -- handle marks for xdatplot					*/
/*
 * Copyright (c) 1993  Leon Avery
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Send questions or comments on xdatplot to:
 *
 * Leon Avery
 * Department of Biochemistry
 * University of Texas Southwestern Medical Center
 * 5323 Harry Hines Blvd
 * Dallas, TX  75235-9038
 *
 * leon@eatworms.swmed.edu
 */

#include "xdatplot.h"

#define	MBLOCK	64			/* # marks to allocate at once	*/
#define	OUTBLK	256			/* max # chars to send at once	*/

#define	M_IN	0			/* lookup_marks return values	*/
#define	M_BEGIN	-1
#define	M_END	-2
#define	M_EMPTY	-3

#ifdef	__STDC__
static	void	marks_filter_ok(Widget, XtPointer,
				XmFileSelectionBoxCallbackStruct *);
static	void	marks_filter_cancel(Widget, XtPointer,
				    XmFileSelectionBoxCallbackStruct *);
static	void	write_fstdin(XtPointer, int *, XtInputId *);
static	void	read_fstdout(XtPointer, int *, XtInputId *);
static	void	read_fstderr(XtPointer, int *, XtInputId *);
static	void	except_f(XtPointer, int *, XtInputId *);
static	void	filter_finish(Bool);
#else	/* __STDC__ */
static	void	marks_filter_ok();
static	void	marks_filter_cancel();
static	void	write_fstdin();
static	void	read_fstdout();
static	void	read_fstderr();
static	void	filter_finish();
#endif	/* __STDC__ */

/*
 * stuff for running a subprocess to filter a mark list
 */
					/* dialog box			*/
static	Widget	marks_filter_pu = NO_WIDGET;
static	MARKLIST *fml = NULL;		/* MARKLIST being filtered	*/
static	String	filname = NULL;		/* filter command name		*/
static	pid_t	fpid = EOF;		/* filter process ID		*/
					/* I/O descriptors of process	*/
static	int	fpipes[3] = {EOF, EOF, EOF};
static	XtInputId stdin_ID = NO_INPUTID;  /* input function ID		*/
static	XtInputId estdin_ID = NO_INPUTID;  /* exception func ID		*/
static	String	fstdin = NULL;		/* string to send to stdin	*/
static	int	tstdin = 0;		/* total length			*/
static	int	nstdin = 0;		/* # chars already sent		*/
static	XtInputId stdout_ID = NO_INPUTID;  /* input function ID		*/
static	XtInputId estdout_ID = NO_INPUTID;  /* exception func ID	*/
static	String	fstdout = NULL;		/* string rcvd from stdout	*/
static	int	nstdout = 0;		/* # chars rcvd			*/
static	XtInputId stderr_ID = NO_INPUTID;  /* input function ID		*/
static	XtInputId estderr_ID = NO_INPUTID;  /* exception func ID	*/
static	String	fstderr = NULL;		/* string rcvd from stderr	*/
static	int	nstderr = 0;		/* # chars rcvd			*/
static	SIGRET	(*sigpipsav)() = NULL;	/* save for SIGPIPE handler	*/

/*
 * Marks are maintained as a single array of MARK structures allocated
 * from the heap and sorted by time.
 */
/* clear_all_marks_in_list -- get rid of all marks			*/
void
clear_all_marks_in_list(ml)
MARKLIST	*ml;
{
    int			i;
    MARK		*mp;

    for(i=0, mp=ml->marks; i<ml->nmarks; i++, mp++) {
	Nfree(mp->comment);
    }
    Nfree(ml->marks);
    ml->nmalloc = ml->nmarks = 0;
}

void
clear_all_marks()
{
    clear_all_marks_in_list(&app_data.mark_list);
}

/* lookup_marks_in_list -- find marks at a particular time
 *
 * TIME		when;
 * MARK		*where;
 * MARKLIST	list;
 * int		howmany;
 * howmany = lookup_marks_in_list(when, &where, &list);
 *
 * lookup_marks_in_list looks for marks at the specified time.  If it
 * finds them, it sets where pointing to the first, and returns the
 * number of marks at that time.  If it finds none, it returns an
 * integer <= 0.  This is the only behavior routines outside the mark
 * package should rely on.
 *
 * Routines within the mark package get further information from the
 * failure return.  If the time requested is before the first mark in
 * the mark list, the return value is -1 and where is set pointing to
 * the beginning of the mark list.  If the time requested is after the
 * last mark the return is -2 and where is set pointing after the last
 * mark in the list.  If the list is empty, the return value is -3 and
 * where is NULL.  If when is between two marks already in the list,
 * the return value is 0 and where points to the first mark with time
 * value >when.
 */
int
lookup_marks_in_list(t, wh, ml)
TIME		t;
MARK		**wh;
MARKLIST	*ml;
{
    unsigned		l = 0;
    unsigned		u = ml->nmarks;
    unsigned		m;

    while(u > l) {
	m = (l + u) >> 1;
	if (t <= ml->marks[m].t) {
	    u = m;
	}
	else {
	    l = m+1;
	}
    }
    if (0 == ml->nmarks) {
	*wh = NULL;
	return(M_EMPTY);
    }
    *wh = &ml->marks[l];
    if (ml->nmarks <= l) return(M_END);
    if (t == ml->marks[l].t) {
	while((u < ml->nmarks) && (t == ml->marks[u].t)) u++;
	return(u - l);
    }
    if (0 >= l) return(M_BEGIN);
    return(M_IN);
}

int
lookup_marks(t, wh)
TIME		t;
MARK		**wh;
{
    return(lookup_marks_in_list(t, wh, &app_data.mark_list));
}

/* lookup_interval_in_list -- find marks in a time interval
 *
 * TIME		start;
 * TIME		end;
 * MARK		*where;
 * MARKLIST	list;
 * int		howmany;
 * howmany = lookup_interval_in_list(start, end, &where, &list);
 *
 * Just like lookup_marks_in_list, except that it returns an array of
 * all the marks in the specified time interval (specifically, all >=
 * start and < end).
 */
int
lookup_interval_in_list(t0, t1, wh, ml)
TIME		t0, t1;
MARK		**wh;
MARKLIST	*ml;
{
    int			n0, n1;
    MARK		*mp;

    n0 = lookup_marks_in_list(t0, wh, ml);
    if (M_EMPTY == n0) return(n0);
    n1 = lookup_marks_in_list(t1, &mp, ml);
    if (mp > *wh) return(mp - *wh);
    return(n0);
}

int
lookup_interval(t0, t1, wh)
TIME		t0, t1;
MARK		**wh;
{
    return(lookup_interval_in_list(t0, t1, wh, &app_data.mark_list));
}

/* find_mark_in_list -- look for and if necessary create mark
 *
 * TIME		time;
 * int		value;
 * MARKLIST	list;
 * MARK		*m;
 * m = find_mark_in_list(time, value, &list);
 *
 * find_mark_in_list returns a pointer to a mark in list at the
 * specified time and with the specified value.  It always returns a
 * pointer to a mark: if no such mark existed, one is created.
 */
MARK *
find_mark_in_list(t, val, ml)
TIME		t;
int		val;
MARKLIST	*ml;
{
    register int	i;
    int			n;
    MARK		*m;

    n = lookup_marks_in_list(t, &m, ml);
    if (n > 0) {
	for(i=0; i<n; i++, m++) {
	    if (val <= m->val) break;
	}
	if (val == m->val) return(m);	/* found it!			*/
    }
    if (NULL == m) {			/* list void: allocate it	*/
	ml->marks = (MARK *) XtCalloc(MBLOCK, sizeof(MARK));
	ml->nmalloc = MBLOCK;
	ml->nmarks = 0;
	m = ml->marks;
    }
    if (ml->nmarks >= ml->nmalloc) {	/* full: add some space		*/
	i = m - ml->marks;
	ml->marks = (MARK *) Realloc(ml->marks,
	    (ml->nmalloc + MBLOCK) * sizeof(MARK));
/* 	bzero(ml->marks + ml->nmalloc * sizeof(MARK), MBLOCK * sizeof(MARK)); */
	ml->nmalloc += MBLOCK;
	m = ml->marks + i;
    }
    if (ml->marks + ml->nmarks > m) {	/* make room for new mark	*/
	bcopy((VOIDST) m,
	      (VOIDST) (m+1),
	      (ml->marks + ml->nmarks - m) * sizeof(MARK));
    }
    m->t = t;
    m->val = val;
    m->comment = NULL;
    ml->nmarks++;
    return(m);
}

MARK *
find_mark(t, val)
TIME		t;
int		val;
{
    return(find_mark_in_list(t, val, &app_data.mark_list));
}

/* delete_marks_in_list -- delete all marks at a particular time
 *
 * If any of the marks have comments, the comments are freed.
 */
void
delete_marks_in_list(t, ml)
TIME		t;
MARKLIST	*ml;
{
    MARK		*m;
    int			n;
    int			i;

    if (0 >= (n = lookup_marks_in_list(t, &m, ml)))
        return;
    for(i=0; i<n; i++) Nfree(m[i].comment);
    if (ml->marks + ml->nmarks > m + n) {
	bcopy(m + n, m, (ml->marks + ml->nmarks - m - n) * sizeof(MARK));
    }
    ml->nmarks -= n;
}

void
delete_marks(t)
TIME		t;
{
    delete_marks_in_list(t, &app_data.mark_list);
}

/* delete_interval_in_list -- delete all marks in a time interval
 *
 * If any of the marks have comments, the comments are freed.
 */
void
delete_interval_in_list(t0, t1, ml)
TIME		t0, t1;
MARKLIST	*ml;
{
    MARK		*m;
    int			n;
    int			i;

    if (0 >= (n = lookup_interval_in_list(t0, t1, &m, ml)))
        return;
    for(i=0; i<n; i++) Nfree(m[i].comment);
    if (ml->marks + ml->nmarks > m + n) {
	bcopy(m + n, m, (ml->marks + ml->nmarks - m - n) * sizeof(MARK));
    }
    ml->nmarks -= n;
}

void
delete_interval(t0, t1)
TIME		t0, t1;
{
    delete_interval_in_list(t0, t1, &app_data.mark_list);
}

/* delete_mark_in_list -- delete a mark at a particular time
 *
 * If any of the marks have comments, the comments are freed.
 */
void
delete_mark_in_list(t, val, ml)
TIME		t;
int		val;
MARKLIST	*ml;
{
    MARK		*m;
    int			n;
    int			i;

    if (0 >= (n = lookup_marks_in_list(t, &m, ml)))
        return;
    for(i=0; i<n; i++, m++) {
	if (val == m->val) break;
    }
    if (i >= n) return;
    Nfree(m->comment);
    if (ml->marks + ml->nmarks > m + 1) {
	bcopy(m + 1, m, (ml->marks + ml->nmarks - m - 1) * sizeof(MARK));
    }
    ml->nmarks--;
}

void
delete_mark(t, val)
TIME		t;
int		val;
{
    delete_mark_in_list(t, val, &app_data.mark_list);
}

/* next_marks_in_list -- find the next marks at or after given time
 *
 * TIME		time;
 * MARK		*marks;
 * MARKLIST	list;
 * int		n;
 * n = next_marks_in_list(time, &marks, &list);
 *
 * next_marks_in_list finds the next marks at or after the specified
 * time, if there are any.  marks is set pointing to the first, and n
 * is set to the number of marks at that time (>= time argument).  If
 * there are no more marks, the function returns 0.
 */
int
next_marks_in_list(t, m, ml)
TIME		t;
MARK		**m;
MARKLIST	*ml;
{
    int			n;
    MARK		*mp;

    n = lookup_marks_in_list(t, m, ml);
    switch(n) {
    case M_END:
    case M_EMPTY:
	return(0);

    case M_IN:
    case M_BEGIN:
	mp = *m;
	t = mp->t;
	mp++;
	n = 1;
	while((mp < ml->marks + ml->nmarks) && (t == mp->t)) {
	    mp++;
	    n++;
	}
	return(n);

    default:
	return(n);
    }
}

int
next_marks(t, m)
TIME		t;
MARK		**m;
{
    return(next_marks_in_list(t, m, &app_data.mark_list));
}

/* prev_marks_in_list -- find marks before a particular time		*/
int
prev_marks_in_list(t, m, ml)
TIME		t;
MARK		**m;
MARKLIST	*ml;
{
    register int	i;
    int			n;
    MARK		*mp;

    n = lookup_marks_in_list(t, m, ml);
    switch(n) {
    case M_BEGIN:
    case M_EMPTY:
	return(0);

    default:
	if (*m <= ml->marks) return(0);
	/* else drop through...		*/

    case M_IN:
    case M_END:
	mp = *m - 1;
	t = mp->t;
	mp--;
	n = 1;
	while((mp >= ml->marks) && (t == mp->t)) {
	    mp--;
	    n++;
	}
	mp++;
	*m = mp;
	return(n);
    }
}

int
prev_marks(t, m)
TIME		t;
MARK		**m;
{
    return(prev_marks_in_list(t, m, &app_data.mark_list));
}

/* next_mark_in_list -- find mark with specific value at or after time	*/
MARK *
next_mark_in_list(t, val, ml)
TIME		t;
int		val;
MARKLIST	*ml;
{
    int			n;
    MARK		*mp;

    n = lookup_marks_in_list(t, &mp, ml);
    if ((M_END == n) || (M_EMPTY == n)) return(NULL);
    while(mp < ml->marks + ml->nmarks) {
	if (mp->val == val) return(mp);
	mp++;
    }
    return(NULL);
}

MARK *
next_mark(t, val)
TIME		t;
int		val;
{
    return(next_mark_in_list(t, val, &app_data.mark_list));
}

/* prev_mark_in_list -- find mark with specific value before time	*/
MARK *
prev_mark_in_list(t, val, ml)
TIME		t;
int		val;
MARKLIST	*ml;
{
    int			n;
    MARK		*mp;

    n = lookup_marks_in_list(t, &mp, ml);
    if ((M_BEGIN == n) || (M_EMPTY == n) || (mp <= ml->marks)) return(NULL);
    mp--;
    while(mp >= ml->marks) {
	if (mp->val == val) return(mp);
	mp--;
    }
    return(NULL);
}

MARK *
prev_mark(t, val)
TIME		t;
int		val;
{
    return(prev_mark_in_list(t, val, &app_data.mark_list));
}

Bool
write_marks_in_list(fname, ml)
String		fname;
MARKLIST	*ml;
{
    int			zero = 0;
    String		s;
    XrmValue		from;
    XrmValue		to;
    Bool		ret;
    FILE		*fp;

    from.addr = (caddr_t) ml;
    from.size = sizeof(*ml);
    to.addr = (caddr_t) XtMalloc(LLEN);
    to.size = LLEN;
    ret = CvtMarkListToString(NULL, NULL, &zero, &from, &to);
    if (!ret) {
	to.addr = (caddr_t) XtRealloc(to.addr, to.size);
	ret = CvtMarkListToString(NULL, NULL, &zero, &from, &to);
    }
    s = (String) to.addr;
    if (!ret) {
	Free(s);
	return(TRUE);
    }
    if (NULL == (fp = fopen(fname, "w"))) return(TRUE);
    ret = fputs(s, fp);
    Free(s);
    fclose(fp);
    return(EOF == ret);
}

Bool
write_marks(fname)
String		fname;
{
    return(write_marks_in_list(fname, &app_data.mark_list));
}

/* merge_marklists -- combine two MARKLISTs into one
 *
 * MARKLIST	ml1;
 * MARKLIST	ml2;
 * merge_marklists(&ml1, &ml2);
 *
 * The marks in ml2 are added to those in ml1.  If any marks in ml1
 * and ml2 have identical t and val fields, the one in ml1 is
 * overwritten by that from ml2 (only significant if one or both has a
 * comment).  All the comments from ml2 are copied with XtNewString.
 */
MARKLIST *
merge_marklists(ml1, ml2)
MARKLIST	*ml1;
MARKLIST	*ml2;
{
    int			n;
    MARK		*m1;
    MARK		*m2;
    MARK		*newmarks;
    int			newnmalloc;
    MARK		*mp;
    Bool		m1_is_valid;
    Bool		m2_is_valid;
    Bool		m1_is_less;
    Bool		m2_is_less;

    if (NULL == ml2->marks) return(ml1);
    newnmalloc = ml1->nmarks + ml2->nmarks;
    newmarks = (MARK *) XtCalloc(newnmalloc, sizeof(MARK));
    m1 = ml1->marks;
    m2 = ml2->marks;
    mp = newmarks;
    while(
	(m1_is_valid = ((NULL != m1) && (m1 < ml1->marks + ml1->nmarks))),
	(m2_is_valid = (m2 < ml2->marks + ml2->nmarks)),
	(m1_is_valid || m2_is_valid)
    ) {
	m1_is_less = m1_is_valid && m2_is_valid && (
	    (m1->t < m2->t) ||
	    ((m1->t == m2->t) && (m1->val < m2->val))
	);
	m2_is_less = m1_is_valid && m2_is_valid && (
	    (m2->t < m1->t) ||
	    ((m2->t == m1->t) && (m2->val < m1->val))
	);
	if (!m1_is_valid || m2_is_less) {
	    bcopy(m2, mp, sizeof(MARK));
	    if (NULL != m2->comment) mp->comment = XtNewString(m2->comment);
	    m2++;
	}
	else if (!m2_is_valid || m1_is_less) {
	    bcopy(m1, mp, sizeof(MARK));
	    if (NULL != m1->comment) mp->comment = XtNewString(m1->comment);
	    m1++;
	}
	else {				/* both valid and equal		*/
	    bcopy(m2, mp, sizeof(MARK));
	    if (NULL != m2->comment) mp->comment = XtNewString(m2->comment);
	    m1++; m2++;
	}
	mp++;
    }
    clear_all_marks_in_list(ml1);
    ml1->nmalloc = newnmalloc;
    ml1->marks = newmarks;
    ml1->nmarks = mp - newmarks;
    return(ml1);
}

Bool
read_marks_in_list(fname, ml)
String		fname;
MARKLIST	*ml;
{
    char		lbuf[LLEN];
    int			zero = 0;
    String		s = XtNewString("");
    int			nc = 0;
    XrmValue		from;
    XrmValue		to;
    Bool		ret;
    FILE		*fp;
    MARKLIST		fml;

    if (NULL == (fp = fopen(fname, "r"))) return(TRUE);
    while(NULL != fgets(lbuf, LLEN, fp)) {
	s = append_string(s, lbuf, &nc);
    }
    fclose(fp);
    from.addr = (caddr_t) s;
    from.size = nc + 1;
    to.addr = (caddr_t) &fml;
    to.size = sizeof(fml);
    ret = CvtStringToMarkList(NULL, NULL, &zero, &from, &to);
    Free(s);
    if (!ret) return(TRUE);
    merge_marklists(ml, &fml);
    clear_all_marks_in_list(&fml);
    return(FALSE);
}

Bool
read_marks(fname)
String		fname;
{
    return(read_marks_in_list(fname, &app_data.mark_list));
}

Bool
filter_marks_in_list(cname, ml)
String		cname;
MARKLIST	*ml;
{
    char		lbuf[LLEN];
    int			zero = 0;
    XrmValue		from;
    XrmValue		to;
    Bool		ret;

    Nfree(fstdin);
    from.addr = (caddr_t) ml;
    from.size = sizeof(*ml);
    to.addr = (caddr_t) XtMalloc(LLEN);
    to.size = LLEN;
    ret = CvtMarkListToString(NULL, NULL, &zero, &from, &to);
    if (!ret) {
	to.addr = (caddr_t) XtRealloc(to.addr, to.size);
	ret = CvtMarkListToString(NULL, NULL, &zero, &from, &to);
    }
    if (!ret) {
	Free(to.addr);
	return(TRUE);
    }
    fstdin = (String) to.addr;
    tstdin = to.size - 1;
    nstdin = 0;
    Nfree(fstdout);
    fstdout = XtNewString("");
    nstdout = 0;
    Nfree(fstderr);
    fstderr = XtNewString("");
    nstderr = 0;
    Nfree(filname);
    filname = XtNewString(cname);
    sigpipsav = signal(SIGPIPE, SIG_IGN); /* ignore SIGPIPE till done	*/
    fpid = run(cname, fpipes);
    if (EOF == fpid) {
	Nfree(fstdin);
	Nfree(fstdout);
	Nfree(fstderr);
	Nfree(filname);
	sprintf(lbuf, "unable to run filter %s", cname);
	PU_error(lbuf, "marks.html#FILTER");
	return(TRUE);
    }
    fml = ml;
    clear_all_marks_in_list(fml);
    stdin_ID = AppAddInput(app, fpipes[0], XtInputWriteMask,
			   write_fstdin, NULL);
    estdin_ID = AppAddInput(app, fpipes[0],
			    XtInputExceptMask, except_f, NULL);
    stdout_ID = AppAddInput(app, fpipes[1], XtInputReadMask,
			    read_fstdout, NULL);
    estdout_ID = AppAddInput(app, fpipes[1],
			     XtInputExceptMask, except_f, NULL);
    stderr_ID = AppAddInput(app, fpipes[2], XtInputReadMask,
			    read_fstderr, NULL);
    estderr_ID = AppAddInput(app, fpipes[2],
			     XtInputExceptMask, except_f, NULL);
    return(FALSE);
}

static void
filter_finish(err)
Bool		err;
{
    char		lbuf[LLEN];
    int			ret;
    MARKLIST		nml;
    int			zero = 0;
    XrmValue		from;
    XrmValue		to;

    if (err) kill(fpid, SIGTERM);
    if (NO_INPUTID != stdin_ID) XtRemoveInput(stdin_ID);
    if (NO_INPUTID != stdout_ID) XtRemoveInput(stdout_ID);
    if (NO_INPUTID != stderr_ID) XtRemoveInput(stderr_ID);
    if (NO_INPUTID != estdin_ID) XtRemoveInput(estdin_ID);
    if (NO_INPUTID != estdout_ID) XtRemoveInput(estdout_ID);
    if (NO_INPUTID != estderr_ID) XtRemoveInput(estderr_ID);
    stdin_ID = stdout_ID = stderr_ID = NO_INPUTID;
    estdin_ID = estdout_ID = estderr_ID = NO_INPUTID;
    if (EOF != fpipes[0]) close(fpipes[0]);
    if (EOF != fpipes[1]) close(fpipes[1]);
    if (EOF != fpipes[2]) close(fpipes[2]);
    fpipes[0] = fpipes[1] = fpipes[2] = EOF;
    for(				/*  wait for filter to exit	*/
	errno = 0;
	(EOF == waitpid(fpid, &ret, 0)) && (EINTR == errno);
	errno = 0
    );
    fpid = -1;
    if (NULL != sigpipsav) signal(SIGPIPE, sigpipsav);
    sigpipsav = NULL;
    err = (err || (0 != (ret & 0xff)) || (0 != (ret >> 8)));
    if (err) {
	if (nstderr > 0) {
	    PU_error(fstderr, NULL);
	}
	else if (0 != (ret >> 8)) {
	    sprintf(lbuf, "%s exited with error status %d", filname, ret >> 8);
	    PU_error(lbuf, NULL);
	}
	else {
	    sprintf(lbuf, "error in excuting filter %s", filname);
	    PU_error(lbuf, NULL);
	}
	Nfree(fstdout);
	fstdout = fstdin;
	nstdout = tstdin;
	fstdin = NULL;
    }
    Nfree(fstdin);
    Nfree(fstderr);
    from.addr = (caddr_t) fstdout;
    from.size = nstdout + 1;
    to.addr = (caddr_t) &nml;
    to.size = sizeof(nml);
    ret = CvtStringToMarkList(NULL, NULL, &zero, &from, &to);
    Nfree(fstdout);
    if (!ret) {
	PU_error("The filtered list couldn't be converted to marks",
		 "marks.html#FILTER");
    }
    else {
	merge_marklists(fml, &nml);
	clear_all_marks_in_list(&nml);
    }
    XtSetSensitive(
	XmSelectionBoxGetChild(marks_filter_pu, XmDIALOG_OK_BUTTON),
	TRUE
    );
    XtUnmanageChild(marks_filter_pu);
    XtPopdown(XtParent(marks_filter_pu));
    replot_whole();
}

static void
write_fstdin(data, fdp, idp)
XtPointer	data;
int		*fdp;
XtInputId	*idp;
{
    int			ret;

    ret = write(fpipes[0], fstdin + nstdin, min(OUTBLK, tstdin - nstdin));
    if (EOF != ret) nstdin += ret;
    if ((EOF == ret) || (nstdin >= tstdin)) {
	close(fpipes[0]);
	fpipes[0] = EOF;
	XtRemoveInput(stdin_ID);
	XtRemoveInput(estdin_ID);
	stdin_ID = estdin_ID = NO_INPUTID;
    }
    if ((EOF == fpipes[0]) && (EOF == fpipes[1]) && (EOF == fpipes[2]))
	filter_finish(FALSE);
}

static void
read_fstdout(data, fdp, idp)
XtPointer	data;
int		*fdp;
XtInputId	*idp;
{
    char		lbuf[LLEN];
    int			ret;

    ret = read(fpipes[1], lbuf, LLEN);
    if (0 >= ret) {
	close(fpipes[1]);
	fpipes[1] = EOF;
	XtRemoveInput(stdout_ID);
	XtRemoveInput(estdout_ID);
	stdout_ID = estdout_ID = NO_INPUTID;
	stdout_ID = NO_INPUTID;
	if ((EOF == fpipes[0]) && (EOF == fpipes[1]) && (EOF == fpipes[2]))
	    filter_finish(FALSE);
	return;
    }
    fstdout = XtRealloc(fstdout, nstdout + ret + 1);
    bcopy(lbuf, fstdout + nstdout, ret);
    nstdout += ret;
    fstdout[nstdout] = '\0';
}

static void
read_fstderr(data, fdp, idp)
XtPointer	data;
int		*fdp;
XtInputId	*idp;
{
    char		lbuf[LLEN];
    int			ret;

    ret = read(fpipes[2], lbuf, LLEN);
    if (0 >= ret) {
	close(fpipes[2]);
	fpipes[2] = EOF;
	XtRemoveInput(stderr_ID);
	XtRemoveInput(estderr_ID);
	stderr_ID = estderr_ID = NO_INPUTID;
	stderr_ID = NO_INPUTID;
	if ((EOF == fpipes[0]) && (EOF == fpipes[1]) && (EOF == fpipes[2]))
	    filter_finish(FALSE);
	return;
    }
    fstderr = XtRealloc(fstderr, nstderr + ret + 1);
    bcopy(lbuf, fstderr + nstderr, ret);
    nstderr += ret;
    fstderr[nstderr] = '\0';
}

static void
except_f(data, fdp, idp)
XtPointer	data;
int		*fdp;
XtInputId	*idp;
{
    filter_finish(TRUE);
}

Bool
filter_marks(cname)
String		cname;
{
    return(filter_marks_in_list(cname, &app_data.mark_list));
}

#define	done(type, value)\
{									\
    if (to->addr != NULL) {						\
	if (to->size < sizeof(type)) {					\
	    to->size = sizeof(type);					\
	    return(False);						\
	}								\
	*((type *) to->addr) = (value);					\
    }									\
    else {								\
	static type	static_val;					\
	static_val = (value);						\
	to->addr = (XtPointer) &static_val;				\
    }									\
    to->size = sizeof(type);						\
    return(True);							\
}

Boolean
CvtStringToMarkList(disp, args, nargs, from, to)
Display		*disp;
XrmValuePtr	args;
Cardinal	*nargs;
XrmValuePtr	from;
XrmValuePtr	to;
{
    char		lbuf[LLEN];
    MARKLIST		ml;
    MARK		*m;
    TIME		t;
    int			val;
    String		fc;
    String		s;
    int			nc;

    if (0 != *nargs) {
	XtWarningMsg("wrongParameters", "cvtStringToMarkList",
		     "XtToolkitError", "String to MarkList\
		     conversion needs no extra arguments",
		     NULL, NULL);
    }
    ml.nmalloc = 0;
    ml.nmarks = 0;
    ml.marks = NULL;
    for(
	fc = (String) from->addr;
	(((String) NULL + 1) != fc) && ('\0' != fc[strspn(fc, WHITE)]);
	fc = STRCHR(fc, '\n') + 1
    ) {
	fc += strspn(fc, LWHITE);	/* extract value...		*/
	val = *fc;
	fc += strcspn(fc, WHITE);
	t = strtol(fc, &s, 0);		/* extract time			*/
	fc = s + strspn(s, LWHITE);
					/* add mark to list		*/
	m = find_mark_in_list(t, val, &ml);
	nc = strcspn(fc, "\n");		/* extract comment		*/
	if (nc > 0) {
	    nc = min(nc, LLEN - 1);
	    bcopy(fc, lbuf, nc);
	    lbuf[nc] = '\0';
	    Nfree(m->comment);
	    m->comment = XtNewString(lbuf);
	}
	fc += nc;
    }
    done(MARKLIST, ml);
}

Boolean
CvtMarkListToString(disp, args, nargs, from, to)
Display		*disp;
XrmValuePtr	args;
Cardinal	*nargs;
XrmValuePtr	from;
XrmValuePtr	to;
{
    char		lbuf[LLEN];
    MARK		*m;
    int			n;
    String		s;
    String		fc = XtNewString("");
    int			nc = 0;

    if (0 != *nargs) {
	XtWarningMsg("wrongParameters", "cvtMarkListToString",
		     "XtToolkitError", "MarkList to String\
		     conversion needs no extra arguments",
		     NULL, NULL);
    }
    m = ((MARKLIST *) from->addr)->marks;
    n = ((MARKLIST *) from->addr)->nmarks;
    while(n-- > 0) {
	sprintf(lbuf, "%c %ld ", m->val, (long) m->t);
	if (NULL != m->comment)
	    strncat(lbuf, m->comment, LLEN - 2 - strlen(lbuf));
	for(				/* turn embedded \n to space	*/
	    s = STRCHR(lbuf, '\n');
	    NULL != s;
	    s = STRCHR(s+1, '\n')
	) *s = ' ';
	strcat(lbuf, "\n");
	fc = append_string(fc, lbuf, &nc);
	m++;
    }
    if (NULL == to->addr) {
	static String		scratch = NULL;

	Nfree(scratch);
	scratch = fc;
	to->addr = (caddr_t) scratch;
	to->size = nc + 1;
	return(True);
    }
    else if (to->size <= nc) {
	to->size = nc + 1;
	Free(fc);
	return(False);
    }
    else {
	strcpy((String) to->addr, fc);
	to->size = nc + 1;
	Free(fc);
	return(True);
    }
}

Widget
make_marks_filter_dialog()
{
    char		lbuf[LLEN];
    String		s;
    XmString		title;
    Arg			args[1];

    XtSetArg(args[0], XmNautoUnmanage, False);
    marks_filter_pu = XmCreateSelectionDialog(toplevel, "marksFilter",
					      args, 1);
    XtVaGetValues(toplevel, XmNtitle, &s, NULL);
    sprintf(lbuf, "%s Marks Filter", s);
    title = XmStringCreateSimple(lbuf);
    XtVaSetValues(marks_filter_pu,
	XmNdeleteResponse, XmUNMAP,
	XtVaTypedArg, XmNselectionLabelString, XmRString,
	"Filter to process marks through:",
	strlen("Filter to process marks through:")+1,
	XmNdialogTitle, title,
    NULL);
    XmStringFree(title);
    XtUnmanageChild(
	XmSelectionBoxGetChild(marks_filter_pu, XmDIALOG_APPLY_BUTTON)
    );
    XtSetSensitive(
	XmSelectionBoxGetChild(marks_filter_pu, XmDIALOG_HELP_BUTTON),
	FALSE
    );
    AddCallback(marks_filter_pu, XmNokCallback, marks_filter_ok, NULL);
    AddCallback(marks_filter_pu, XmNcancelCallback, marks_filter_cancel, NULL);
    return(marks_filter_pu);
}

static void
marks_filter_ok(popup, data, cbs)
Widget					popup;
XtPointer				data;
XmFileSelectionBoxCallbackStruct	*cbs;
{
    int			i;
    int			c = 1;
    char		lbuf[LLEN];
    String		cname;
    XmStringTable	xmclist;
    String		*clist;
    int			nc;

    if (!XmStringGetLtoR(cbs->value, XmSTRING_DEFAULT_CHARSET, &cname))
	error("marks_filter_ok: can't happen");
    dprintf1("marks_filter_ok: %s\n", cname);
    XtVaGetValues(popup,
	XmNlistItemCount, &nc,
	XmNlistItems, &xmclist,
    NULL);
    clist = (String *) XtCalloc(nc+1, sizeof(String));
    for(i=0; i<nc; i++) {		/* compound strings...		*/
	XmStringGetLtoR(xmclist[i], XmSTRING_DEFAULT_CHARSET, &clist[i]);
    }
    for(i=0; i<nc; i++) {
	if (0 >= (c = strcmp(cname, clist[i]))) break;
    }
    if (0 != c) {
	if (i < nc)
	    bcopy(&clist[i], &clist[i+1], (nc-i) * sizeof(String));
	clist[i] = XtNewString(cname);
	nc++;
    }
    xmclist = (XmStringTable) XtCalloc(nc, sizeof(XmString));
    for(i=0; i<nc; i++) {		/* ...are a pain...		*/
	xmclist[i] = XmStringCreateSimple(clist[i]);
	Free(clist[i]);
    }
    Free(clist);
    XtVaSetValues(popup,
	XmNlistItemCount, nc,
	XmNlistItems, xmclist,
    NULL);
    for(i=0; i<nc; i++) {		/* ...in the ass		*/
	Free(xmclist[i]);
    }
    Free(xmclist);
    if (EOF != fpid) {
	PU_error("There is already a filter running", "marks.html#FILTER");
	Free(cname);
	return;
    }
    filter_marks(cname);
    Free(cname);
    XtSetSensitive(
	XmSelectionBoxGetChild(marks_filter_pu, XmDIALOG_OK_BUTTON),
	FALSE
    );
    replot_whole();
}

static void
marks_filter_cancel(popup, data, cbs)
Widget					popup;
XtPointer				data;
XmFileSelectionBoxCallbackStruct	*cbs;
{
    marks_filter_end();
    XtUnmanageChild(popup);
    XtPopdown(XtParent(popup));
}

void
marks_filter_end()
{
    if (EOF != fpid) filter_finish(TRUE);
}

void
marks_filter_drop()
{
    if (EOF == fpid) return;
    if (NO_INPUTID != stdin_ID) XtRemoveInput(stdin_ID);
    if (NO_INPUTID != stdout_ID) XtRemoveInput(stdout_ID);
    if (NO_INPUTID != stderr_ID) XtRemoveInput(stderr_ID);
    if (NO_INPUTID != estdin_ID) XtRemoveInput(estdin_ID);
    if (NO_INPUTID != estdout_ID) XtRemoveInput(estdout_ID);
    if (NO_INPUTID != estderr_ID) XtRemoveInput(estderr_ID);
    stdin_ID = stdout_ID = stderr_ID = NO_INPUTID;
    estdin_ID = estdout_ID = estderr_ID = NO_INPUTID;
    if (EOF != fpipes[0]) close(fpipes[0]);
    if (EOF != fpipes[1]) close(fpipes[1]);
    if (EOF != fpipes[2]) close(fpipes[2]);
    fpipes[0] = fpipes[1] = fpipes[2] = EOF;
    if (NULL != sigpipsav) signal(SIGPIPE, sigpipsav);
    sigpipsav = NULL;
    Nfree(fstdin);
    Nfree(fstdout);
    Nfree(fstderr);
    if (NO_WIDGET != marks_filter_pu) {
	XtSetSensitive(
	    XmSelectionBoxGetChild(marks_filter_pu, XmDIALOG_OK_BUTTON),
	    TRUE
	);
	XtUnmanageChild(marks_filter_pu);
	XtPopdown(XtParent(marks_filter_pu));
    }
}
