/********************************************************************\
 * Transaction.h -- defines transaction for xacc (X-Accountant)     *
 * Copyright (C) 1997 Robin D. Clark                                *
 * Copyright (C) 1997 Linas Vepstas                                 *
 *                                                                  *
 * This program is free software; you can redistribute it and/or    *
 * modify it under the terms of the GNU General Public License as   *
 * published by the Free Software Foundation; either version 2 of   *
 * the License, or (at your option) any later version.              *
 *                                                                  *
 * This program is distributed in the hope that it will be useful,  *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of   *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the    *
 * GNU General Public License for more details.                     *
 *                                                                  *
 * You should have received a copy of the GNU General Public License*
 * along with this program; if not, write to the Free Software      *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.        *
 *                                                                  *
 *   Author: Rob Clark                                              *
 * Internet: rclark@cs.hmc.edu                                      *
 *  Address: 609 8th Street                                         *
 *           Huntington Beach, CA 92648-4632                        *
\********************************************************************/

#ifndef __XACC_TRANSACTION_H__
#define __XACC_TRANSACTION_H__

#include <Xm/Xm.h>  /* for String */
#include "date.h"   /* for Date */

#include "config.h"


/* Values for the reconciled field in Transaction: */
#define CREC 'c'              /* The transaction has been cleared        */
#define YREC 'y'              /* The transaction has been reconciled     */
#define NREC 'n'              /* not reconciled or cleared               */

/** STRUCTS *********************************************************/
/* The debit & credit pointers are used to implement a double-entry 
 * accounting system.  Basically, the idea with double entry is that
 * there is always an account that is debited, and another that is
 * credited.  These two pointers identify the two accounts. 
 */

typedef struct _transaction {
  struct _account *credit;     /* back-pointer to credited account          */
  struct _account *debit;      /* back-pointer to debited account           */
  String  num;                 /* transaction id                            */
  Date    date;                /* transaction date                          */
  String  description;         
  String  memo;
  String  action;              /* Buy, Sell, Div, etc.                      */
  int     catagory;
  char    reconciled;
  double  damount;             /* num-shares, if positive, deposit, else payment */
  double  share_price;         /* the share price, ==1.0 for bank account */

  /* the "balance" is the sum of all of the transactions, up to and including
   * this transaction.  Since this transaction belong to two accounts, there
   * are, respectively, two partial sums, one for each account */
  double  credit_balance;
  double  debit_balance;       
  double  credit_cleared_balance;
  double  debit_cleared_balance;       
  double  credit_reconciled_balance;
  double  debit_reconciled_balance;       

  double  credit_share_balance;
  double  debit_share_balance;       
  double  credit_share_cleared_balance;
  double  debit_share_cleared_balance;       
  double  credit_share_reconciled_balance;
  double  debit_share_reconciled_balance;       

  /* some private data */
  char    write_flag;          /* used only during file IO */

} Transaction;

/** PROTOTYPES ******************************************************/

Transaction * mallocTransaction (void);           /* mallocs and inits */
void          initTransaction (Transaction *);    /* clears a transaction struct */
void          freeTransaction (Transaction *);

/********************************************************************\
 * sorting comparison function
 *
 * returns a negative value if transaction a is dated earlier than b,        
 * returns a positive value if transaction a is dated later than b,
 * returns zero if both transactions are on the same date.
 *
\********************************************************************/

int  xaccTransOrder (Transaction **ta, Transaction **tb);

/*
 * count the number of transactions in the null-terminated array
 */
int xaccCountTransactions (Transaction **tarray);

/** GLOBALS *********************************************************/

#endif /* __XACC_TRANSACTION_H__ */
