/*
 * Copyright (c) 1992 The Regents of the University of California.
 * All rights reserved.
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose, without fee, and without written agreement is
 * hereby granted, provided that the above copyright notice and the following
 * two paragraphs appear in all copies of this software.
 * 
 * IN NO EVENT SHALL THE UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR
 * DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF THE UNIVERSITY OF
 * CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 * AND FITNESS FOR A PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND THE UNIVERSITY OF CALIFORNIA HAS NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

/*
 * Portions of this software Copyright (c) 1995 Brown University.
 * All rights reserved.
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose, without fee, and without written agreement
 * is hereby granted, provided that the above copyright notice and the
 * following two paragraphs appear in all copies of this software.
 * 
 * IN NO EVENT SHALL BROWN UNIVERSITY BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT 
 * OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF BROWN
 * UNIVERSITY HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * BROWN UNIVERSITY SPECIFICALLY DISCLAIMS ANY WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE.  THE SOFTWARE PROVIDED HEREUNDER IS ON AN "AS IS"
 * BASIS, AND BROWN UNIVERSITY HAS NO OBLIGATION TO PROVIDE MAINTENANCE,
 * SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.
 */

#include <stdlib.h>
#include "video.h"
#include "dither.h"
#include "mpeg.h"		/* for ImageDesc */

#define NUM_COLORS 256		/* number of entries in colormap */
				/* for gray-scale dithering */

/* Range values for lum, cr, cb. */
extern int LUM_RANGE;
extern int CR_RANGE;
extern int CB_RANGE;

/* Array that remaps color numbers to actual pixel values used by X server. */

extern unsigned char pixel[256];

/* Arrays holding quantized value ranged for lum, cr, and cb. */

extern int *lum_values;
extern int *cr_values;
extern int *cb_values;


/* ----------------------------- MNI Header -----------------------------------
@NAME       : InitColormap
@INPUT      : (none)
@OUTPUT     : *NumColors - number of entries in the newly-created colormap
              *Map - an array of colourmap entries; each one contains a 
                     red, green, and blue byte-values (0 .. 255).  
		     *Map[i] gives the colour to display a pixel value i.
@RETURNS    : (none)
@DESCRIPTION: Creates a colour map used for most dithering methods 
              (everything except full-colour, gray, and monochrome).
	      The colour map itself is pretty self-explanatory -- a 
	      pixel with value i is to be displayed using the red, green
	      and blue values in *Map[i] after InitColormap() is done.
@METHOD     : 
@GLOBALS    : 
@CALLS      : 
@CREATED    : 95/3/4, Greg Ward: based on InitColorDisplay(), from gdith.c
                                 in the original Berkeley player
@MODIFIED   : 
---------------------------------------------------------------------------- */
#ifndef DISABLE_DITHER
static void
InitColormap (int *NumColors, ColormapEntry **Map)
{
   int i, lum_num, cr_num, cb_num;
   unsigned char r, g, b; /* Hack -- DCR 96-04-12 */

   *NumColors =  LUM_RANGE*CB_RANGE*CR_RANGE;
   *Map = (ColormapEntry *) malloc (*NumColors * sizeof (ColormapEntry));

   for (i = 0; i < *NumColors; i++) 
   {
      lum_num = (i / (CR_RANGE*CB_RANGE))%LUM_RANGE;
      cr_num = (i / CB_RANGE)%CR_RANGE;
      cb_num = i % CB_RANGE;

      ConvertColor(lum_values[lum_num], cr_values[cr_num], cb_values[cb_num], 
		   &r, &g, &b);

      (*Map)[i].red   = r;
      (*Map)[i].green = g;
      (*Map)[i].blue  = b;

      pixel[i] = i;
   }
}
#endif


/* ----------------------------- MNI Header -----------------------------------
@NAME       : InitGrayColormap
@INPUT      : (none)
@OUTPUT     : *NumColors - number of entries in the newly-created colormap
              *Map - an array of colourmap entries
@RETURNS    : (none)
@DESCRIPTION: Creates a colour map used for gray-scale dithering, i.e.
              the red/green/blue values are the same for any given
	      pixel value.
@METHOD     : 
@GLOBALS    : 
@CALLS      : 
@CREATED    : 95/3/4, Greg Ward: based on InitGrayDisplay(), from gdith.c
                                 in the original Berkeley player
@MODIFIED   : 
---------------------------------------------------------------------------- */
#ifndef DISABLE_DITHER
static void
InitGrayColormap (int *NumColors, ColormapEntry **Map)
{
   int  i;
   
   *NumColors =  NUM_COLORS;
   *Map = (ColormapEntry *) malloc (*NumColors * sizeof (ColormapEntry));

   for (i = 0; i < *NumColors; i++) 
   {
      (*Map)[i].red = (*Map)[i].green = (*Map)[i].blue = i;
      pixel[i] = i;
   }
}
#endif


/* ----------------------------- MNI Header -----------------------------------
@NAME       : InitDither
@INPUT      : Image - pointer to the image descriptor for the current MPEG
@OUTPUT     : Image->ColormapSize, Image->Colormap - the colour map for
              this movie, as initialized by either InitColormap or
	      InitGrayColormap (unless the current dithering scheme
	      is full colour, in which case there is no colour map)
@RETURNS    : (none)
@DESCRIPTION: Does all initialization particular to the type of dithering
              being used.  Basically, sets up the internal data structures
	      needed by the dithering code, and then sets up a colour map
	      needed to display the pixels output by the ditherers.
@METHOD     : 
@GLOBALS    : 
@CALLS      : InitColor     (for most dithering methods)
              InitColormap  (for most dithering methods)
	      InitGrayColormap (for gray-scale dithering)
              Init(..)Dither  (.. = the current dithering method)
@CREATED    : 95/3/3, Greg Ward: taken mostly from main() in the original 
                                 Berkeley player
@MODIFIED   : 
---------------------------------------------------------------------------- */
void
InitDither (ImageDesc *Image)
{
   LUM_RANGE = 8;
   CR_RANGE = 4;
   CB_RANGE = 4;
   /* Allocate/initialize tables used for dithering (?) */
   lum_values = (int *) malloc(LUM_RANGE*sizeof(int));
   cr_values = (int *) malloc(CR_RANGE*sizeof(int));
   cb_values = (int *) malloc(CB_RANGE*sizeof(int));
   switch (Image->vid_stream->ditherType)
   {
#ifndef DISABLE_DITHER
      case HYBRID_DITHER:
	 InitColor ();
	 InitHybridDither ();
	 InitColormap (&Image->ColormapSize, &Image->Colormap);
	 break;

      case HYBRID2_DITHER:
	 InitColor ();
	 InitHybridErrorDither ();
	 InitColormap (&Image->ColormapSize, &Image->Colormap);
	 break;

      case FS4_DITHER:
	 InitColor ();
	 InitFS4Dither ();
	 InitColormap (&Image->ColormapSize, &Image->Colormap);
	 break;

      case FS2_DITHER:
	 InitColor ();
	 InitFS2Dither ();
	 InitColormap (&Image->ColormapSize, &Image->Colormap);
	 break;

      case FS2FAST_DITHER:
	 InitColor ();
	 InitFS2FastDither ();
	 InitColormap (&Image->ColormapSize, &Image->Colormap);
	 break;

      case Twox2_DITHER:
	 InitColor ();
	 Init2x2Dither ();
	 InitColormap (&Image->ColormapSize, &Image->Colormap);
	 PostInit2x2Dither ();
	 break;

      case GRAY_DITHER:
	 InitGrayColormap (&Image->ColormapSize, &Image->Colormap);
	 break;
#endif
      case FULL_COLOR_DITHER:
         wpixel[0] = 0xff;
	 wpixel[1] = 0xff00;
	 wpixel[2] = 0xff0000;
	 Image->vid_stream->matched_depth=24;
	 InitColorDither(1);
	 Image->ColormapSize = -1;
	 Image->Colormap = NULL;
	 break;

#ifndef DISABLE_DITHER
      case NO_DITHER:
	 break;

      case ORDERED_DITHER:
	 InitColor ();
	 InitOrderedDither ();
	 InitColormap (&Image->ColormapSize, &Image->Colormap);
	 break;

      case MONO_DITHER:
      case MONO_THRESHOLD:
	 break;

      case ORDERED2_DITHER:
	 InitColor ();
	 InitColormap (&Image->ColormapSize, &Image->Colormap);
	 InitOrdered2Dither ();
	 break;

      case MBORDERED_DITHER:
	 InitColor ();
	 InitColormap (&Image->ColormapSize, &Image->Colormap);
	 InitMBOrderedDither ();
	 break;

      case PPM_DITHER:
	 Image->ColormapSize = -1;
	 Image->Colormap = NULL;
         wpixel[0] = 0xff;
	 wpixel[1] = 0xff00;
	 wpixel[2] = 0xff0000;
	 Image->vid_stream->matched_depth=24;
	 InitColorDither(TRUE);
	 break;
#endif
   }
}   

