/* worldmap - a world map display program
   Copyright (C) 1998 Andrew Skypeck
  
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
 * resedit.c
 */

#include <stdio.h>
#include <ctype.h>
#include <assert.h>
#include <stdlib.h>		/* atof() */
#include <string.h>		/* strcpy() ... */

#include <X11/Intrinsic.h>
#include <X11/Shell.h>

#include <Xm/Form.h>
#include <Xm/Label.h>
#include <Xm/DialogS.h>
#include <Xm/Frame.h>
#include <Xm/PanedW.h>
#include <Xm/PushB.h>
#include <Xm/RowColumn.h>
#include <Xm/ScrolledW.h>
#include <Xm/TextF.h>
#include <Xm/ToggleB.h>

#include "message.h"
#include "help.h"
#include "xtutil.h"
#include "Map.h"

/* this dialogs widgets for i/o */
static Widget toggle[5];
static Widget textf[8];
static XmProjection projection = XmMILLER_PROJECTION;

static void Ok(Widget dialog, XtPointer client_data, XtPointer call_data);
static void Apply(Widget dialog, XtPointer client_data, XtPointer call_data);
static void Cancel(Widget dialog, XtPointer client_data, XtPointer call_data);

static Widget
CreateGridFrame(Widget parent)
{
  XmString xms;
  Widget rowcol, grid_frame, grid_frame_label;
  Widget label;

  grid_frame = \
    XtVaCreateManagedWidget("gridFrame",
			    xmFrameWidgetClass,      parent,
			    NULL);

  xms = XmStringCreateSimple("Map Grid");
  grid_frame_label = \
    XtVaCreateManagedWidget("gridFrameLabel",
			    xmLabelWidgetClass, grid_frame,
			    XmNchildType, XmFRAME_TITLE_CHILD,
			    XmNchildVerticalAlignment,
			    XmALIGNMENT_BASELINE_TOP,
			    XmNlabelString, xms,
			    NULL);
  XmStringFree(xms);

  rowcol = \
    XtVaCreateWidget("gridRowcol",
		     xmRowColumnWidgetClass,  grid_frame,
		     XmNpacking,              XmPACK_COLUMN,
		     XmNnumColumns,           2,
		     XmNorientation,          XmVERTICAL,
		     XmNisAligned,            True,
		     XmNentryAlignment,       XmALIGNMENT_BEGINNING,
		     NULL);

  xms = XmStringCreateSimple("Draw Geographic Grid");
  toggle[0] = \
    XtVaCreateManagedWidget("gridToggle",
			    xmToggleButtonWidgetClass,   rowcol,
			    XmNlabelString, xms,
			    NULL);
  assert(XmIsToggleButton(toggle[0]));
  XmStringFree(xms);

  xms = XmStringCreateSimple("Major Grid Size (deg)");
  XtVaCreateManagedWidget("majorGridLabel",
			  xmLabelWidgetClass, rowcol,
			  XmNlabelString, xms,
			  NULL);
  XmStringFree(xms);

  xms = XmStringCreateSimple("Minor Grid Size (deg)");
  XtVaCreateManagedWidget("minorGridLabel",
			  xmLabelWidgetClass, rowcol,
			  XmNlabelString, xms,
			  NULL);
  XmStringFree(xms);

  xms = XmStringCreateSimple("");
  label = \
    XtVaCreateManagedWidget("label",
			    xmLabelWidgetClass, rowcol,
			    XmNlabelString, xms,
			    NULL);
  XmStringFree(xms);

  textf[0] = \
    XtVaCreateManagedWidget("textf",
			    xmTextFieldWidgetClass,   rowcol,
			    XmNvalue, "10",
			    NULL);
  assert(XmIsTextField(textf[0]));

  textf[1] = \
    XtVaCreateManagedWidget("textf",
			    xmTextFieldWidgetClass,   rowcol,
			    XmNvalue, "2",
			    NULL);
  assert(XmIsTextField(textf[1]));

  XtManageChild(rowcol);

  return grid_frame;
}

static Widget
CreateMapDetailFrame(Widget parent)
{
  Widget frame, frame_label;
  Widget rowcol;
  XmString xms;
  int i;

  String item_labels[] = {
    "Land", "Ocean", "Coastlines", "Lakes", "Rivers", "Boundaries",
  };

  String value[] = {
    "forest green", "navy", "blue", "blue", "blue", "red"
  };

  frame = \
    XtVaCreateManagedWidget("frame",
			    xmFrameWidgetClass,      parent,
			    NULL);

  xms = XmStringCreateSimple("Map Detail");
  frame_label = 
    XtVaCreateManagedWidget("frameLabel",
			    xmLabelWidgetClass, frame,
			    XmNchildType, XmFRAME_TITLE_CHILD,
			    XmNchildVerticalAlignment,
			    XmALIGNMENT_BASELINE_TOP,
			    XmNlabelString, xms,
			    NULL);
  XmStringFree(xms);

  rowcol = \
    XtVaCreateWidget("rowcol",
		     xmRowColumnWidgetClass,  frame,
		     XmNpacking,              XmPACK_COLUMN,
		     XmNnumColumns,           XtNumber(item_labels),
		     XmNorientation,          XmHORIZONTAL,
		     XmNisAligned,            True,
		     XmNentryAlignment,       XmALIGNMENT_BEGINNING,
		     NULL);

  for (i = 0; i < XtNumber(item_labels); i++)
    {
      if (i == 0 || i == 1)
	{
	  xms = XmStringCreateSimple(item_labels[i]);
	  XtVaCreateManagedWidget("label",
				  xmLabelWidgetClass, rowcol,
				  XmNlabelString, xms,
				  NULL);
	  XmStringFree(xms);
	}
      else
	{
	  toggle[i - 1] = \
	    XtVaCreateManagedWidget(item_labels[i],
				    xmToggleButtonWidgetClass,   rowcol,
				    XmNset, True,
				    NULL);
	  assert(XmIsToggleButton(toggle[i - 1]));
	}

      textf[i + 2] = \
	XtVaCreateManagedWidget("textf",
				xmTextFieldWidgetClass,   rowcol,
				XmNvalue, value[i],
				NULL);
      assert(XmIsTextField(textf[i + 2]));
    }

  XtManageChild(rowcol);

  return frame;
}

static void
ProjectionOption(Widget menu, XtPointer client_data, XtPointer call_data)
{
  int item_no = (int) client_data;

  projection = (XmProjection) item_no;
}

Widget
CreateBasemapDialog(Widget widget)
{
  Widget dialog;
  Widget pane;
  Widget option_menu;
  XmString option_label, cyl_eq, eckert_iv, hammer, mercator, miller;
  XmString mollweide, orthographic, sinusoidal;
  Widget grid_frame, map_detail_frame;
  Widget rowcol;
  Widget l_form, ok_pushb, apply_pushb, cancel_pushb;
  XmString xms;

  /* do not position the dialog now, let a map action do it from main() */
  dialog = \
    XtVaCreatePopupShell("basemapDialog",
			 xmDialogShellWidgetClass, TopLevel(widget),
			 XmNdeleteResponse,       XmDESTROY,
			 XmNtitle,                "Basemap Resources",
			 XmNdefaultPosition,      False,
			 NULL);
  assert(XmIsDialogShell(dialog));
  XtAddCallback(dialog, XmNpopupCallback, CenterDialogOnce, NULL);

  pane = \
    XtVaCreateWidget("pane",
		     xmPanedWindowWidgetClass, dialog,
		     XmNsashWidth,             1,
		     XmNsashHeight,            1,
		     NULL);
  assert(XmIsPanedWindow(pane));

  rowcol = \
    XtVaCreateWidget("rowcol",
		     xmRowColumnWidgetClass,  pane,
		     XmNisAligned,            True,
		     XmNentryAlignment,       XmALIGNMENT_BEGINNING,
		     NULL);


  option_label = XmStringCreateSimple("Projection:");
  cyl_eq = XmStringCreateSimple("Cylindrical Equidistant");
  eckert_iv = XmStringCreateSimple("Eckert IV");
  hammer = XmStringCreateSimple("Hammer");
  mercator = XmStringCreateSimple("Mercator");
  miller = XmStringCreateSimple("Miller");
  mollweide = XmStringCreateSimple("Mollweide");
  orthographic = XmStringCreateSimple("Orthographic");
  sinusoidal = XmStringCreateSimple("Sinusoidal");

  option_menu = \
    XmVaCreateSimpleOptionMenu(rowcol, "optionMenu",
			       option_label, (KeySym) NULL,
			       (int) projection, ProjectionOption,
			       XmNtopAttachment, XmATTACH_FORM,
			       XmNleftAttachment, XmATTACH_FORM,
			       XmNrightAttachment, XmATTACH_FORM,
			       XmVaPUSHBUTTON, cyl_eq, 'C', NULL, NULL,
			       XmVaPUSHBUTTON, eckert_iv, 'E', NULL, NULL,
			       XmVaPUSHBUTTON, hammer, 'H', NULL, NULL,
			       XmVaPUSHBUTTON, mercator, 'M', NULL, NULL,
			       XmVaPUSHBUTTON, miller, 'I', NULL, NULL,
			       XmVaPUSHBUTTON, mollweide, 'O', NULL, NULL,
			       XmVaPUSHBUTTON, orthographic, 'R', NULL, NULL,
			       XmVaPUSHBUTTON, sinusoidal, 'S', NULL, NULL,
			       NULL);

  XtManageChild(option_menu);
  XmStringFree(cyl_eq);
  XmStringFree(eckert_iv);
  XmStringFree(hammer);
  XmStringFree(mercator);
  XmStringFree(miller);
  XmStringFree(mollweide);
  XmStringFree(orthographic);
  XmStringFree(sinusoidal);

  grid_frame = CreateGridFrame(rowcol);
  assert(XmIsFrame(grid_frame));

  map_detail_frame = CreateMapDetailFrame(rowcol);
  assert(XmIsFrame(map_detail_frame));

  /* rather crucial */
  XtManageChild(rowcol);

  l_form = \
    XtVaCreateWidget("l_form",
		     xmFormWidgetClass,        pane,
		     XmNfractionBase,          5,
		     XmNpositionIndex,         1,
		     NULL);
  assert(XmIsForm(l_form));

  xms = XmStringCreateSimple("OK");
  ok_pushb = \
    XtVaCreateManagedWidget("ok_pushb",
			    xmPushButtonWidgetClass,  l_form,
			    XmNtopAttachment,         XmATTACH_FORM,
			    XmNbottomAttachment,      XmATTACH_FORM,
			    XmNleftAttachment,        XmATTACH_POSITION,
			    XmNleftPosition,          0,
			    XmNrightAttachment,       XmATTACH_POSITION,
			    XmNrightPosition,         1,
			    XmNshowAsDefault,         True,
			    XmNdefaultButtonShadowThickness, 1,
			    XmNlabelString,           xms,
			    NULL);
  assert(XmIsPushButton(ok_pushb));
  XmStringFree(xms);
  XtAddCallback(ok_pushb, XmNactivateCallback, Ok, dialog);

  xms = XmStringCreateSimple("Apply");
  apply_pushb = \
    XtVaCreateManagedWidget("apply_pushb",
			    xmPushButtonWidgetClass,  l_form,
			    XmNtopAttachment,         XmATTACH_FORM,
			    XmNbottomAttachment,      XmATTACH_FORM,
			    XmNleftAttachment,        XmATTACH_POSITION,
			    XmNleftPosition,          2,
			    XmNrightAttachment,       XmATTACH_POSITION,
			    XmNrightPosition,         3,
			    XmNdefaultButtonShadowThickness, 1,
			    XmNlabelString,           xms,
			    NULL);
  assert(XmIsPushButton(apply_pushb));
  XmStringFree(xms);
  XtAddCallback(apply_pushb, XmNactivateCallback, Apply, NULL);

  xms = XmStringCreateSimple("Cancel");
  cancel_pushb = \
    XtVaCreateManagedWidget("cancel_pushb",
			    xmPushButtonWidgetClass,  l_form,
			    XmNtopAttachment,         XmATTACH_FORM,
			    XmNbottomAttachment,      XmATTACH_FORM,
			    XmNleftAttachment,        XmATTACH_POSITION,
			    XmNleftPosition,          4,
			    XmNrightAttachment,       XmATTACH_POSITION,
			    XmNrightPosition,         5,
			    XmNshowAsDefault,         False,
			    XmNdefaultButtonShadowThickness, 1,
			    XmNlabelString,           xms,
			    NULL);
  assert(XmIsPushButton(cancel_pushb));
  XtAddCallback(cancel_pushb, XmNactivateCallback, Cancel, dialog);
  XmStringFree(xms);

  XtManageChild(l_form);
  XtManageChild(pane);

  return dialog;
}

static void
SetMapValues(Widget w)
{
  Widget map = (Widget) XtNameToWidget(TopLevel(w), "*map");
  String major_grid_string, minor_grid_string;
  String ocean_color, land_color, coastline_color;
  String lake_color, river_color, boundary_color;
  Boolean draw_grid;
  Boolean draw_coastline, draw_lakes, draw_rivers, draw_boundaries;
  FloatArg major, minor;

  assert(XmIsMap(map));

  /* get the toggle values */
  draw_grid       = XmToggleButtonGetState(toggle[0]);
  draw_coastline  = XmToggleButtonGetState(toggle[1]);
  draw_lakes      = XmToggleButtonGetState(toggle[2]);
  draw_rivers     = XmToggleButtonGetState(toggle[3]);
  draw_boundaries = XmToggleButtonGetState(toggle[4]);

  /* get the color values */
  XtVaGetValues(textf[0], XmNvalue, &major_grid_string, NULL);
  XtVaGetValues(textf[1], XmNvalue, &minor_grid_string, NULL);
  XtVaGetValues(textf[2], XmNvalue, &land_color, NULL);
  XtVaGetValues(textf[3], XmNvalue, &ocean_color, NULL);
  XtVaGetValues(textf[4], XmNvalue, &coastline_color, NULL);
  XtVaGetValues(textf[5], XmNvalue, &lake_color, NULL);
  XtVaGetValues(textf[6], XmNvalue, &river_color, NULL);
  XtVaGetValues(textf[7], XmNvalue, &boundary_color, NULL);

  /* convert the floating point resources */
  major.float_value = (float) fabs(atof(major_grid_string));
  minor.float_value = (float) fabs(atof(minor_grid_string));

  /* set the map resources */
  XtVaSetValues(map,
		XmNprojection, projection,
		XmNdrawGrid, draw_grid,
		XmNdrawCoastline, draw_coastline,
		XmNdrawLakes, draw_lakes,
		XmNdrawRivers, draw_rivers,
		XmNdrawBoundaries, draw_boundaries,
		VaFloatArg(XmNmajorGridSize, major),
		VaFloatArg(XmNminorGridSize, minor),
		VaStringArg(XmNlandColor, land_color),
		VaStringArg(XmNoceanColor, ocean_color),
		VaStringArg(XmNcoastlineColor, coastline_color),
		VaStringArg(XmNlakeColor, lake_color),
		VaStringArg(XmNriverColor, river_color),
		VaStringArg(XmNboundaryColor, boundary_color),
		NULL);

  XtFree(major_grid_string);
  XtFree(minor_grid_string);
  XtFree(ocean_color);
  XtFree(land_color);
  XtFree(lake_color);
  XtFree(river_color);
  XtFree(boundary_color);

  SetMessageArea("Map resources reset");
}

static void
Ok(Widget ok_pushb, XtPointer client_data, XtPointer call_data)
{
  Widget dialog = (Widget) client_data;

  SetMapValues(ok_pushb);
  XtPopdown(dialog);
}

static void
Apply(Widget apply_pushb, XtPointer client_data, XtPointer call_data)
{
  SetMapValues(apply_pushb);
}

static void
Cancel(Widget apply_pushb, XtPointer client_data, XtPointer call_data)
{
  Widget dialog = (Widget) client_data;

  XtPopdown(dialog);
}
