/*--------------------------------*-C-*---------------------------------*
 * File:	utmp.c
 *
 * Public:
 *    makeutent ();
 *    cleanutent ();
 *
 * Private:
 *	get_tslot ();
 *	write_utmp ();
 *	write_wtmp ();
 *
 * As usual, the author accepts no responsibility for anything, nor does
 * he guarantee anything whatsoever.
 * ---------------------------------------------------------------------*
 *
 * Revision 1.6  07DEC95:16	Piet W. Plomp,
 * wtmp support for (at least linux)
 *
 * Revision 1.55  1995/10/16  16:04:16  rgg
 * now works in Solaris 2.x and Linux 1.2.x
 *
 * Revision 1.5 08/09/1993 stempien
 * Something was wrong with the Linux support!
 * I fixed it using the provided utmp manipulation routines.
 * I also surrounded many varables that were not needed for Linux
 * in the BSD defines to reduce the memory needed to run.
 * I didn't touch the Sun part of the code so it should still work.
 *
 * $Log: utmp.c,v $
 * Revision 1.4  1993/08/09  11:54:15  lipka
 * now works both on Linux and SunOs 4.1.3.
 * Brians clean-ups incorporated
 *
 * Revision 1.3  1993/08/09  07:16:42  lipka
 * nearly correct (running) linux-version.
 * Delete-Window is still a problem
 *
 * Revision 1.1  1993/08/07  18:03:53  lipka
 * Initial revision
 *
 * Clean-ups according to suggestions of Brian Stempien <stempien@cs.wmich.edu>
 *
 *    Bugs:
 *	UTMP should be locked from call to utmp_end() 'til makeutent() (?).
 *	Maybe the child should tell the parent, where the entry is made.
 *	Tested only on Linux.
 *
 *	Gives weird inital idle times. They go away after one uses the
 *	window, but......
 * ------------------------------------------------------------------------
 * This version has wtmp support for (at least Linux) added by:
 *    Piet W. Plomp,
 *    ICCE - Institute for educational technology,
 *    State University of Groningen, The Netherlands,
 *    <piet@icce.rug.nl> or (faster) <piet@idefix.icce.rug.nl>
 *
 *  All additions were put in the non-BSD branch
 *  Function added:
 *      write_wtmp ();
 *  Functions changed:
 *      makeutent ();
 *      cleanutent ();
 *  all WTMP specific code is #ifdef'd WTMP_SUPPORT, which currently depends
 *  on UTMP_SUPPORT.
 * This code is valid and tested on linux (libc.so.4.6.27), but is
 * POSIX compliant and should work on SYSVR4, except possibly the use
 * of EAGAIN, which may be called EACCESS in SVR4. Linux has only EAGAIN,
 * POSIX allows either.
 *
 * My additions are tagged with an entry like "... pwp 95-12-07", where the
 * former are my initials and the latter the date in yy-mm-dd format.
 *----------------------------------------------------------------------*/
#include "rxvt.h"

#ifdef UTMP_SUPPORT
#ifdef HAVE_UTMPX_H
# include <utmpx.h>
#endif
#include <sys/types.h>
#include <sys/stat.h>

#ifdef TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# ifdef HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif

#include <utmp.h>
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#ifdef HAVE_LASTLOG_H
# include <lastlog.h>
#endif
#include <pwd.h>

/* WTMP_SUPPORT added pwp 95-12-07 */
#ifdef UTMP_SUPPORT
# define WTMP_SUPPORT
# include <errno.h>
# ifdef HAVE_FCNTL_H
#  include <fcntl.h>
# endif
# if !defined (EAGAIN) && defined (EACCESS)
#   define EAGAIN EACCESS
# endif
#endif

/*----------------------------------------------------------------------*
 * extern variables referenced
 */
extern char *display_name;

/*----------------------------------------------------------------------*
 * extern variables declared here
 */
int utmp_inhibit = 0;

/*----------------------------------------------------------------------*
 * local variables
 */
static unsigned char utmp_made = 0;	/* marks if an entry has been made */
static int utmp_pos = 0;		/* position of utmp-stamp */

/*----------------------------------------------------------------------*
 * Code to make utmp entries really starts here !
 */

#if !defined (USE_SYSV_UTMP) && !defined (USE_BSD_UTMP)
#  undef UTMP_SUPPORT
#endif

#ifndef UTMP_FILENAME
#  ifdef UTMP_FILE
#    define UTMP_FILENAME UTMP_FILE
#  else
#    ifdef _PATH_UTMP
#      define UTMP_FILENAME _PATH_UTMP
#    else
#      define UTMP_FILENAME "/etc/utmp"
#    endif
#  endif
#endif

#ifndef LASTLOG_FILENAME
#  ifdef _PATH_LASTLOG
#    define LASTLOG_FILENAME _PATH_LASTLOG
#  else
#    define LASTLOG_FILENAME "/usr/adm/lastlog"  /* only on BSD systems */
#  endif
#endif

#ifndef WTMP_FILENAME
#  ifdef WTMP_FILE
#    define WTMP_FILENAME WTMP_FILE
#  else
#    ifdef _PATH_WTMP
#      define WTMP_FILENAME _PATH_WTMP
#    else
#      ifdef SYSV
#        define WTMP_FILENAME "/etc/wtmp"
#      else
#        define WTMP_FILENAME "/usr/adm/wtmp"
#      endif
#    endif
#  endif
#endif

#ifndef TTYTAB_FILENAME
#  ifdef TTYTAB
#    define TTYTAB_FILENAME TTYTAB_FILENAME
#  else
#    define TTYTAB_FILENAME "/etc/ttytab"
#  endif
#endif

#ifndef USER_PROCESS
#  define USER_PROCESS 7
#endif
#ifndef DEAD_PROCESS
#  define DEAD_PROCESS 8
#endif

/*
 * The code was pretty ugly when BSD and SYSV style utmp stuff was mixed,
 * so I just made a huge ifdef for BSD, and another for SYSV
 *
 * SYSV is also divided on utmpx support (Solaris 2.x)
 * and normal SVR4 utmp support -- rgg 16/10/95
 */
#ifdef USE_SYSV_UTMP
#ifdef HAVE_UTMPX_H
#ifndef WTMPX_FILE
Error WTMPX_FILE must be defined on Solaris 2.x
#endif
/*
 * write a utmp entry to the utmp file
 */
static void
write_utmp (struct utmpx *ux)
{
  struct utmp u;

  utmpname (UTMP_FILENAME);
  getutmp (ux, &u);
  pututline (&u);
  pututxline (ux);
  updwtmpx (WTMPX_FILE, ux);
  endutent ();
  utmp_made = 1;
}

/*
 * make a utmp entry
 */
void
makeutent (char *ttyname)
{
   struct passwd *pwent;
   struct utmpx ux;
   int num;
   char *hostname = display_name;

   if (utmp_inhibit)
     return;

   pwent = getpwuid (getuid ());
   memset (&ux, 0, sizeof(ux));

   /* get the host: */
   strncpy (ux.ut_host, hostname, sizeof(ux.ut_host));
   ux.ut_syslen = strlen (ux.ut_host) + 1;

   /* and now the line: */
   if (sscanf (ttyname, "pts/%d", &num) != 1)
     {
	print_error ("can't parse tty name \"%s\"", ttyname);
	return;
     }
   sprintf (ux.ut_id, "vt%02x", num);
   sprintf (ux.ut_line, "pts/%d", num);

   time (&ux.ut_xtime);
   strncpy (ux.ut_user, pwent->pw_name, sizeof (ux.ut_user));
   ux.ut_type = USER_PROCESS;
   ux.ut_pid = getpid ();

   write_utmp (&ux);
}

/*
 * remove a utmp entry
 */
void
cleanutent (void)
{
   struct utmp u;

   if (!utmp_made)
     return;

   utmpname (UTMP_FILENAME);
   setutent ();
   if (getutid (&u) == NULL)
     return;
   u.ut_type = DEAD_PROCESS;
   time (&u.ut_time);
   pututline (&u);
   updwtmp (WTMP_FILENAME, &u);
}
#else /* not HAVE_UTMPX_H */

/*
 * write a utmp entry to the utmp file
 */
static void
write_utmp (struct utmp * u)
{
   utmpname (UTMP_FILENAME);
   setutent ();		/* HAVE_setutent */
   pututline (u);
   endutent ();
   utmp_made = 1;
}

/*
 * write_wtmp added by pwp 95 12 07
 * called by makeutent() and cleanutent() below
 */
static void
write_wtmp (struct utmp * wtmp)
{
#ifdef WTMP_SUPPORT
   int fd, retry = 10;		/* 10 attempts at locking */
   struct flock lck;		/* fcntl locking scheme */

   if ((fd = open (WTMP_FILENAME, O_WRONLY|O_APPEND, 0)) < 0)
     return;

   lck.l_whence = SEEK_END;	/* start lock at current eof    */
   lck.l_len    = 0;		/* end at ``largest possible eof'' */
   lck.l_start  = 0;
   lck.l_type   = F_WRLCK;	/* we want a write lock         */

   /* attempt lock - use F_SETLK, F_SETLKW would cause a deadlock! */
   while (retry--)
     {
	if ((fcntl (fd, F_SETLK, &lck) < 0) && errno != EAGAIN)
	  {
	     close (fd);
	     return;	/* failed for unknown reason: give up */
	  }
     }
   write (fd, wtmp, sizeof(struct utmp));

   /* unlocking the file */
   lck.l_type = F_UNLCK;
   fcntl (fd, F_SETLK, &lck);

   close (fd);
#endif	/* WTMP_SUPPORT */
}

/*
 * make a utmp entry
 */
void
makeutent (char *ttyname)
{
   struct passwd *pwent;
   struct utmp u;
   char *hostname = display_name;

   if (utmp_inhibit)
     return;

   pwent = getpwuid (getuid ());
   memset (&u, 0, sizeof(u));

   /* get the host */
   strncpy (u.ut_host, hostname, sizeof(u.ut_host));

   /* and now the line: */
   strcpy (u.ut_line, ttyname);	/* ttyname always nul-terminated? */

   time (&u.ut_time);
   strncpy (u.ut_user, pwent->pw_name, sizeof(u.ut_user));
   u.ut_type = USER_PROCESS;
   u.ut_pid = getpid ();
   strncpy (u.ut_id, (ttyname+3), 2);

   write_utmp (&u);
   write_wtmp (&u);
}

/*----------------------------------------------------------------------*
 * remove a utmp entry
 *
 * there is a nice function "endutent" defined in <utmp.h>;
 * like "setutent" it takes no arguments, so I think it gets all information
 * from library-calls.
 * That's why "setutent" has to be called by the child-process with
 * file-descriptors 0/1 set to the pty. But that child execs to the
 * application-program and therefore can't clean it's own utmp-entry!(?)
 * The master on the other hand doesn't have the correct process-id
 * and io-channels... I'll do it by hand:
 * (what's the position of the utmp-entry, the child wrote? :-)
 *----------------------------------------------------------------------*/
void
cleanutent (void)
{
   int pid;
   struct utmp *u;

   if (!utmp_made)
     return;

   utmpname (UTMP_FILENAME);
   setutent ();
   pid = getpid ();
   /* The following 11 lines of code were copied from the
    * poeigl-1.20 login/init package. Special thanks to
    * poe for the code examples.
    */
   while ((u = getutent ()))
     {
	if (u->ut_pid == pid)
	  {
	     time (&u->ut_time);
	     memset (&u->ut_user, 0, sizeof(u->ut_user));
	     memset (&u->ut_host, 0, sizeof(u->ut_host));
	     u->ut_type = DEAD_PROCESS;
	     u->ut_pid = 0;
#if !defined (_AIX) && !defined (SVR4)
	     u->ut_addr = 0;
#endif
	     pututline (u);
	     endutent ();
             /* write_wtmp () added by pwp 95 12 07 */
             write_wtmp (u);  /* add empty entry to wtmp: line has closed */
	  }
     }
}
#endif /* HAVE_UTMPX_H */
#endif	/* USE_SYSV_UTMP */

#ifdef USE_BSD_UTMP
/*----------------------------------------------------------------------*
 * get_tslot() - grabbed from xvt-1.0 - modified by David Perry
 *
 * find ttyname in /etc/ttytab and return a slot number that can be used to
 * access the utmp file.  Can't use ttyslot() because the tty name is not
 * that of fd 0.
 *----------------------------------------------------------------------*/
static int
get_tslot (char *ttyname)
{
   FILE *fd;
   char buf [256], name [256];
   int i;

   if ((fd = fopen (TTYTAB_FILENAME, "r")) != NULL)
     {
	for (i = 1; fgets (buf, sizeof(buf), fd) != NULL; i++)
	  {
	     if (*buf == '#' || sscanf (buf, "%s", name) != 1)
	       continue;
	     if (!strcmp (ttyname, name))
	       {
		  fclose (fd);
		  return i;
	       }
	  }
	fclose (fd);
     }
   return -1;
}

/*
 * write a utmp entry to the utmp file
 */
static void
write_utmp (char *ttyname, struct utmp * u)
{
   FILE *fd;

   if ((fd = fopen (UTMP_FILENAME, "r+")) == NULL)
     return;
   utmp_pos = get_tslot (ttyname) * sizeof(struct utmp);
   if (utmp_pos >= 0)
     {
	fseek (fd, utmp_pos, 0);
	fwrite (u, sizeof(struct utmp), 1, fd);
	utmp_made = 1;
     }
   fclose (fd);
}

/*
 * make a utmp entry
 */
void
makeutent (char *ttyname)
{
   struct utmp u;
   struct timeval tp;
   struct timezone tzp;
   struct passwd *pwent;

   char *hostname = display_name;

   if (utmp_inhibit)
     return;

   pwent = getpwuid (getuid ());
   memset (&u, 0, sizeof(u));

   /* get the host, the line, the time and the name: */
   strncpy (u.ut_host, hostname, sizeof(u.ut_host));
   strncpy (u.ut_line, ttyname, sizeof(u.ut_line));
   if (!gettimeofday (&tp, &tzp))
     u.ut_time = tp.tv_sec;
   strncpy (u.ut_name, pwent->pw_name, sizeof(u.ut_name));

   write_utmp (ttyname, &u);
}

/*
 * remove a utmp entry
 */
void
cleanutent (void)
{
   FILE *fd;
   struct utmp u;

   if (!utmp_made || (fd = fopen (UTMP_FILENAME, "r+")) == NULL)
     return;
   fseek (fd, utmp_pos, 0);
   memset (&u, 0, sizeof(u));
   fwrite (&u, sizeof(struct utmp), 1, fd);
   fclose (fd);
}
#endif	/* USE_BSD_UTMP */
#endif	/* UTMP_SUPPORT */

/*
 * Dummy routines to use if utmp support is not wanted/needed
 */
#ifndef UTMP_SUPPORT
void makeutent (char *ttyname) {}
void cleanutent (void) {}
#endif /* UTMP_SUPPORT */
/*----------------------- end-of-file (C source) -----------------------*/
