/*--------------------------------*-C-*---------------------------------*
 * File:	resources.c
 *
 * obtain resources from ~/.Xdefaults or ~/.Xresources using either
 * REAL_RESOURCES (XGetDefault) or for memory savings FAKE_RESOURCES.
 *
 * This module is all new by Rob Nation
 *
 * You can do what you like with this source code provided you don't make
 * money from it and you include an unaltered copy of this message
 * (including the copyright).  As usual, the author accepts no
 * responsibility for anything, nor does he guarantee anything whatsoever.
 * ---------------------------------------------------------------------*
 * Heavily modified by mj olesen <olesen@me.QueensU.CA>
 * No additional restrictions.
 *----------------------------------------------------------------------*/
#include "rxvt.h"

#include <ctype.h>
#include <X11/Xlib.h>

#include "command.h"
#include "xsetup.h"
#include "screen.h"
#include "debug.h"

#ifdef NO_RESOURCES
void extract_resources (char *name){}	/* dummy function */
#else	/* NO_RESOURCES */
/*----------------------------------------------------------------------*
 * extern variables referenced
 */

/*----------------------------------------------------------------------*
 * local variables
 */

#ifdef WITH_7BIT_MODE
static char *rs_bits = NULL;
#endif
static char *rs_loginShell = NULL;
#ifdef UTMP_SUPPORT
static char *rs_utmpInhibit = NULL;
#endif
#ifdef SCROLLBAR_ANY
static char *rs_scrollBar = NULL;
#endif
#ifdef GREEK_SUPPORT
static char *rs_greektoggle_key = NULL;
#endif
#ifdef PRINTPIPE
static char *rs_printscreen_key = NULL;
#endif
#if (NFONTS > 1)
static char *rs_bigfont_key = NULL;
static char *rs_smallfont_key = NULL;
#endif
static char *rs_pageup_key = NULL;
static char *rs_pagedown_key = NULL;

#ifdef MAPALERT
static char *rs_mapAlert = NULL;
#endif
#ifdef VISUAL_BELL
static char *rs_visualBell = NULL;
#endif
static char *rs_saveLines = NULL;
#ifdef REFRESH_RESOURCE
static char *rs_refreshPeriod = NULL;
#endif
#ifndef META_ESCAPE
static char *rs_meta = NULL;
#endif
#ifdef KANJI
static char *rs_kanji_encoding = NULL;
#endif

/* list of resource strings */
struct {
   char *kw;			/* keyword */
   char **ptr;			/* data pointer */
}  rs_list [] = {
#ifdef PRINTPIPE
     { "print-pipe",	& rs_print_pipe },
#endif
     { "geometry",	& rs_geometry },
     { "font",		& rs_font [0] },
#if (NFONTS > 1)
     { "font1",		& rs_font [1] },
     { "font2",		& rs_font [2] },
     { "font3",		& rs_font [3] },
     { "font4",		& rs_font [4] },
#endif
#ifdef KANJI
     { "kfont",		& rs_kfont [0] },
#if (NFONTS > 1)
     { "kfont1",	& rs_kfont [1] },
     { "kfont2",	& rs_kfont [2] },
     { "kfont3",	& rs_kfont [3] },
     { "kfont4",	& rs_kfont [4] },
#endif
     { "kanji_encoding",& rs_kanji_encoding },
#endif	/* KANJI */
     { "title",		& rs_title },
     { "iconName",	& rs_iconName },
#ifdef WITH_7BIT_MODE
     { "bits",		& rs_bits },
#endif
     { "loginShell",	& rs_loginShell },
#ifdef UTMP_SUPPORT
     { "utmpInhibit",	& rs_utmpInhibit },
#endif
#ifdef SCROLLBAR_ANY
     { "scrollBar",	& rs_scrollBar },
#endif
     { "saveLines",	& rs_saveLines },
#ifdef REFRESH_RESOURCE
     { "refreshPeriod", & rs_refreshPeriod },
#endif
#ifndef META_ESCAPE
     { "meta",		& rs_meta },
#endif
#ifdef GREEK_SUPPORT
     { "greektoggle_key",	& rs_greektoggle_key },
#endif
#ifdef PRINTPIPE
     { "printscreen_key",	& rs_printscreen_key },
#endif
#if (NFONTS > 1)
     { "bigfont_key",	& rs_bigfont_key },
     { "smallfont_key",	& rs_smallfont_key },
#endif
     { "pageup_key",	& rs_pageup_key },
     { "pagedown_key",	& rs_pagedown_key },
#ifdef MAPALERT
     { "mapAlert",	& rs_mapAlert },
#endif
#ifdef VISUAL_BELL
     { "visualBell",	& rs_visualBell },
#endif
#ifndef NO_MULTIPLE_CLICK
     { "cutchars",	& rs_cutchars },
#endif
     { "foreground",	& rs_color [COLORFG] },
     { "background",	& rs_color [COLORBG] },
#ifndef NO_COLOR
     { "color0",	& rs_color [COLOR0] },	/* black	*/
     { "color1",	& rs_color [COLOR1] },	/* red		*/
     { "color2",	& rs_color [COLOR2] },	/* green	*/
     { "color3",	& rs_color [COLOR3] },	/* yellow	*/
     { "color4",	& rs_color [COLOR4] },	/* blue		*/
     { "color5",	& rs_color [COLOR5] },	/* magenta	*/
     { "color6",	& rs_color [COLOR6] },	/* cyan		*/
     { "color7",	& rs_color [COLOR7] },	/* white	*/
#ifndef USE_FAKE_BOLD
     { "color10",	& rs_color [BOLD0] },	/* bright black		*/
     { "color11",	& rs_color [BOLD1] },	/* bright red		*/
     { "color12",	& rs_color [BOLD2] },	/* bright green		*/
     { "color13",	& rs_color [BOLD3] },	/* bright yellow	*/
     { "color14",	& rs_color [BOLD4] },	/* bright blue		*/
     { "color15",	& rs_color [BOLD5] },	/* bright magenta	*/
     { "color16",	& rs_color [BOLD6] },	/* bright cyan		*/
     { "color17",	& rs_color [BOLD7] },	/* bright white		*/
#endif
#endif
     { NULL,			NULL}
};

/*----------------------------------------------------------------------*
 * local functions
 */
static int get_Xdefaults (char *name);

/*----------------------------------------------------------------------*/

/*
 * a replacement for strcasecmp() to avoid linking in the entire library
 */
static int
my_strcasecmp (const char *s1, const char *s2)
{
   for (/*nil*/; (*s1 && *s2); s1++, s2++)
     {
	register int c1, c2;
	c1 = tolower (*s1);
	c2 = tolower (*s2);
	if (c1 != c2)
	  return (c1 - c2);
     }
   return (int) (*s1 - *s2);
}

#define is_boolean_true(x)	(!my_strcasecmp ((x), "true"))

/*
 * read the resources files
 */
void
extract_resources (char *name)
{
   const char *fn = "resources";

   /*
    * don't assume anything about memory for Xdefaults(),
    * but we are safe to free any memory allocated using FAKE_RESOURCES
    */
#ifndef FAKE_RESOURCES
# undef FREE
# define FREE(ptr,id,fn)	((void)0)
#endif

   if (!get_Xdefaults (name)) return;

#ifdef KANJI
   if (rs_kanji_encoding != NULL) {
      set_kanji_encoding (rs_kanji_encoding);
      FREE (rs_kanji_encoding, "kanji_encoding", fn);
   }
#endif
#ifdef WITH_7BIT_MODE
   if (rs_bits != NULL) {
      hibit_mask = (rs_bits [0] == '7' ? 0x7F : 0xFF);
      FREE (rs_bits, "bits", fn);
   }
#endif
#ifdef MAPALERT
   if (rs_mapAlert != NULL) {
      map_alert = is_boolean_true (rs_mapAlert);
      FREE (rs_mapAlert, "mapAlert", fn);
   }
#endif
#ifdef VISUAL_BELL
   if (rs_visualBell != NULL) {
      visual_bell = is_boolean_true (rs_visualBell);
      FREE (rs_visualBell, "visualBell", fn);
   }
#endif

   if (rs_loginShell != NULL) {
      login_shell = is_boolean_true (rs_loginShell);
      FREE (rs_loginShell, "loginShell", fn);
   }
#ifdef UTMP_SUPPORT
   if (rs_utmpInhibit != NULL) {
      utmp_inhibit = is_boolean_true (rs_utmpInhibit);
      FREE (rs_utmpInhibit, "utmpInhibit", fn);
   }
#endif
#ifdef SCROLLBAR_ANY
   if (rs_scrollBar != NULL) {
      if (is_boolean_true (rs_scrollBar))
	sbar.type = SBAR_XTERM;
      else if (!my_strcasecmp (rs_scrollBar, "arrows"))
	sbar.type = SBAR_ARROWS;
      else
	sbar.type = SBAR_NONE;
      FREE (rs_scrollBar, "scrollBar", fn);
   }
#endif
   if (rs_saveLines != NULL)
     {
	TermWin.hist_size = atoi (rs_saveLines);
	if (TermWin.hist_size < 0)
	  TermWin.hist_size = 0;
	FREE (rs_saveLines, "saveLines", fn);
     }
#ifdef REFRESH_RESOURCE
     if (rs_refreshPeriod) {
	refresh_period = atoi (rs_refreshPeriod);
        FREE (rs_refreshPeriod, "refreshPeriod", fn);
     }
#endif
#ifdef PRINTPIPE
   String2Keysym ( &ks_printscreen,	rs_printscreen_key);
#endif
#ifdef GREEK_SUPPORT
   String2Keysym ( &ks_greektoggle,	rs_greektoggle_key);
#endif
   String2Keysym ( &ks_pageup,		rs_pageup_key);
   String2Keysym ( &ks_pagedown,	rs_pagedown_key);
#if (NFONTS > 1)
   String2Keysym ( &ks_bigfont,		rs_bigfont_key);
   String2Keysym ( &ks_smallfont,	rs_smallfont_key);
#endif
#ifndef META_ESCAPE
   if (rs_meta != NULL) {
      meta_char = (is_boolean_true (rs_meta) ? 033 :
		   (rs_meta [0] == '8' ? 0x80 : 0));
      FREE (rs_meta, "meta", fn);
   }
#endif
}

/*----------------------------------------------------------------------*/
#ifdef FAKE_RESOURCES
#ifdef KEYSYM_RESOURCE

/*
 * interprete backslash-escaped strings
 */
static char *
escaped_string (char *str)
{
   register char *p, ch;
   int i, len;

   if (str == NULL || (len = strlen (str)) == 0)
     return NULL;

   /* use 'i' to increment through destination and p through source */
   for (p = str, i = 0; i < len; i++)
     {
	ch = *p++;
	if (ch == '\\')
	  {
	     ch = *p;
	     if (isdigit (ch))	/* octal */
	       {
		  int j, num = 0;
		  for (j = 0; j < 3 && (ch >= '0' && ch <= '7'); j++)
		    {
		       num = num * 010 + (ch - '0');
		       p++; len--;
		       ch = *p;
		    }
		  ch = (unsigned char) num;
	       }
	     else
	       {
		  p++; len--;	/* one fewer char remains */
		  switch (ch) {
		   case 'b': ch = '\b';	break;	/* backspace */
		   case 'E':
		   case 'e': ch =  033;	break;	/* escape */
		   case 'n': ch = '\n';	break;	/* new-line */
		   case 'r': ch = '\r';	break;	/* carriage-return */
		   case 't': ch = '\t';	break;	/* tab */
		  }
	       }
	  }
	str [i] = ch;
     }
   str [len] = '\0';

   return strlen (str) ? str : NULL;
}
#endif

/*
 * remove leading/trailing space and strip-off leading/trailing quotes
 */
static char *
trim_string (char *str)
{
   int n;

   if (str == NULL || *str == '\0')
     return NULL;

   while (*str && isspace (*str)) str++;	/* skip leading spaces */

   /* trim trailing space */
   n = strlen (str) - 1;
   while (n > 0 && isspace (str [n])) n--;
   str [n+1] = '\0';

   if (str [0] == '"')		/* strip leading/trailing quotes */
     {
	str++; n--;
	if (str [n] == '"')
	  str [n--] = '\0';
     }
   return n >= 0 ? str : NULL;
}

/*
 * the matching algorithm used by for FAKE_RESOURCES
 */
static int
find_match (char *name, FILE *stream)
{
   int len, found = 0;
   char *text, buffer [256];

   if (stream == NULL) return 0;
   len = strlen (name);
   while ((text = fgets (buffer, sizeof(buffer), stream)) != NULL)
     {
	int i, n;

	while (*text && isspace (*text)) text++;    /* leading whitespace */

	if (((text [len] != '*' && text [len] != '.')) ||
	    (len && strncmp (text, name, len)))
	  continue;
	text += (len + 1);	/* skip `name*' or `name.' */

	/*
	 * look for something like this (XK_Delete)
	 * rxvt*keysym.0xFFFF: "\177"
	 */
#ifdef KEYSYM_RESOURCE
#define KEYSYM_kw		"keysym"	/* keyword */
	n = strlen (KEYSYM_kw);
	if ((text [n] == '*' || text [n] == '.') &&
	    !strncmp (text, KEYSYM_kw, n))
	  {
	     int sym;
	     text += (n + 1);	/* skip `keyword*' or `keyword.' */
	     if (sscanf (text, "%x:", &sym) == 1)
	       {
		  if (sym >= 0xFF00)	/* only do extended keys */
		    sym -= 0xFF00;
		  if (sym < 0 || sym > 0xFF) continue;

		  /* cue to ':' , it is there if sscanf() worked */
		  if ((text = strchr (text, ':')) == NULL) continue;

		  text++;		/* skip `:' */
		  text = escaped_string (trim_string (text));
		  n = (text == NULL) ? 0 : strlen (text);
		  if (n && KeySym_map [sym] == NULL)
		    {
		       /* only if not previously set */
		       KeySym_map [sym] = MALLOC (n+1, "match");
		       strcpy (KeySym_map [sym], text);
		       found++;
		    }
	       }
	  }
	else
#endif
	  for (i = 0; rs_list[i].kw; i++)
	  {
	     n = strlen (rs_list[i].kw);
	     if ((text [n] == ':') &&
		 !strncmp (text, rs_list[i].kw, n))
	       {
		  text += (n + 1);	/* skip `keyword:' */
		  text = trim_string (text);
		  n = (text == NULL) ? 0 : strlen (text);
		  if (n && *(rs_list[i].ptr) == NULL)
		    {
		       /* only if not previously set */
		       *(rs_list[i].ptr) = MALLOC (n+1, "match");
		       strcpy (*rs_list[i].ptr, text);
		       found++;
		    }
		  break;
	       }
	  }
     }
   rewind (stream);
   return found;
}
#endif	/* FAKE_RESOURCES */

/*
 * get X resources using XGetDefault() or the hand-rolled replacement
 */
static int
get_Xdefaults (char *name)
{
   int i, found = 0;
#if defined (REAL_RESOURCES)
   /* get resources using the X library function */
   for (i = 0; rs_list [i].kw; i++)
     {
	char *p;
	if (*(rs_list[i].ptr) != NULL) continue;	/* previously set */
	if ((p = XGetDefault (Xdisplay, name, rs_list[i].kw)) != NULL ||
#ifdef APL_SUBCLASS
	    (p = XGetDefault (Xdisplay, APL_SUBCLASS, rs_list[i].kw)) != NULL ||
#endif
	    (p = XGetDefault (Xdisplay, APL_CLASS, rs_list[i].kw)) != NULL
	    /* is this allowed, for partial matches? -- don't know */
	    /* || (p = XGetDefault (Xdisplay, NULL, rs_list[i].kw)) != NULL */
	    )
	  {
	     *rs_list [i].ptr = p;
	     found++;
	  }
     }
#elif defined (FAKE_RESOURCES)
   /* get resources the hard way, but save lots of memory */
   char *home, *fname[] = { ".Xdefaults", ".Xresources", NULL };
   FILE *ad, *fd = NULL;

   /*
    * The normal order to match resources is the following:
    * - global resources (partial match on ~/.Xdefaults)
    * - application file resources (XAPPLOADDIR/RXvt)
    * - class resources (in ~/.Xdefaults)
    * - private resources (in ~/.Xdefaults)
    *
    * However, for FAKE_RESOURCES, the matching algorithm checks if a
    * resource string value has already been allocated and won't over-write
    * it with (in this case) a less specific resource value.
    *
    * This avoids multiple allocation.  Also, when we've called this
    * routine command-line string options have already been applied so we
    * needn't to allocate for those resources.
    *
    * So, search in resources from most to least specific.
    *
    * Also, use a special sub-class so that we can use either or both of
    * "XTerm" and "RXvt" as class names.
    */

   if ((home = getenv ("HOME")) != NULL)
     {
	int len = strlen (home) + 2;
	for (i = 0; fname [i]; i++)
	  {
	     char * f = MALLOC (len + strlen (fname[i]), "fname");
	     sprintf (f, "%s/%s", home, fname [i]);

	     fd = fopen (f, "r");
	     FREE (f, "fname", "Xdefaults");
	     if (fd != NULL)
	       break;
	  }
     }

   found += find_match (name, fd);
#ifdef APL_SUBCLASS
   found += find_match (APL_SUBCLASS, fd);
#endif
   found += find_match (APL_CLASS, fd);

#ifdef XAPPLOADDIR
#ifndef APL_SUBCLASS
#  define APL_SUBCLASS	"RXvt"
#endif
   ad = fopen (XAPPLOADDIR "/" APL_SUBCLASS, "r");
   if (ad != NULL)
     {
	found += find_match ("", ad);
	fclose (ad);
     }
#endif

   found += find_match ("", fd);		/* partial match */
   if (fd != NULL)
     fclose (fd);
#endif	/* FAKE_RESOURCES */
   return found;
}
#endif	/* NO_RESOURCES */
/*----------------------- end-of-file (C source) -----------------------*/
