/*--------------------------------*-C-*---------------------------------*
 * File:	debug.c
 *
 * This module is all original work by Robert Nation
 * <nation@rocket.sanders.lockheed.com>
 *
 * Copyright 1994, Robert Nation, no rights reserved.
 * The author guarantees absolutely nothing about anything, anywhere, anytime
 * Use this code at your own risk for anything at all.
 *
 * Modified by mj olesen <olesen@me.QueensU.CA>
 * No additional restrictions.
 *
 * Modifications by Raul Garcia Garcia <rgg@tid.es>
 *----------------------------------------------------------------------*/
#include "rxvt.h"
#include "debug.h"

#ifndef DEBUG_MALLOC

/*
 * malloc and check for NULL return
 */
void *
my_malloc (int size)
{
   void *ptr;
   if ((ptr = malloc (size)) == NULL)
     {
	printf ("malloc error: size 0x%X\n", size);
	abort ();
     }
   return ptr;
}

void
my_free (void *ptr)
{
   if (ptr != NULL)		/* in case free() is non-compilant */
     free (ptr);
}

#else	/* DEBUG_MALLOC */

#define MAX_MALLOCS	100
/* Number of pad characters to use when checking for out-of-bounds */
#define NPAD	4
/* pad chars */
#define PAD1		0x3E
#define PAD2		0x3F

static void *ptr_list [MAX_MALLOCS];
static char *ptr_name [MAX_MALLOCS];
static int alloc_init = 0;

/*
 * Malloc that checks for NULL return, and adds out-of-bounds checking if
 * DEBUG_MALLOC is set
 */
void *
safe_malloc (int size, char *id)
{
   void *ptr;
#ifdef DEBUG_MALLOC
   char *tmp_ptr;
   int *l, i;

   if (!alloc_init)
     {
	for (i = 0; i < MAX_MALLOCS; i++)
	  ptr_list [i] = NULL;
	alloc_init = 1;
     }

   if ((ptr = malloc (size + 2 * NPAD * sizeof(int))) == NULL)
     {
	abort ();
     }

   tmp_ptr = ptr;
   for (i = 0; i < (NPAD-1) * sizeof(int); i++)
     tmp_ptr [i] = PAD1;
   for (i = size + (NPAD * sizeof(int));
	i < (size + (2 * NPAD) * sizeof(int));
	i++)
     tmp_ptr [i] = PAD2;

   l = ((int *)ptr + NPAD - 1);
   *l = size;

   fprintf (stderr, "malloc %d bytes 0x%08X: \"%s\"\n", size, ptr, id);

   for (i = 0; i < MAX_MALLOCS && ptr_list [i] != NULL; i++) /*nil*/;
   if (i >= MAX_MALLOCS)
     {
	fprintf (stderr, "MAX_MALLOCS exceeded. Please increase\n");
     }
   else
     {
	ptr_list [i] = ptr;
	ptr_name [i] = id;
	check_all_mem ("malloc", id);
     }

   return ((void *)((int *)ptr+NPAD));
#else
   if ((ptr = malloc (size)) == NULL)
     abort ();
   return ptr;
#endif
}

/*
 * Free command good for use with above malloc, checks for out-of-bounds
 * before freeing.
 */
void
safe_free (void *ptr, char *id1, char *id2)
{
#ifdef DEBUG_MALLOC
   int i;

   if (ptr == NULL)
     return;
   ptr = (void *)((int *)ptr - NPAD);

   fprintf (stderr, "Freeing memory at 0x%08X", ptr);
   for (i = 0; i < MAX_MALLOCS && ptr_list [i] != ptr; i++) /*nil*/;
   if (i < MAX_MALLOCS)
     {
	fprintf (stderr,": \"%s\"\n", ptr_name [i]);
	check_all_mem (id1, id2);
	ptr_list [i] = NULL;
     }

   free (ptr);
#else
   if (ptr != NULL)
     free (ptr);
#endif
}

/*
 * Check all allocated memory for out-of-bounds memory usage
 */
void
check_all_mem (char *id1, char *id2)
{
#ifdef DEBUG_MALLOC
   int i, j, l, fail = 0;
   static char *prev_id1 = NULL, *prev_id2 = NULL;

   if (!alloc_init)
     return;

   for (i = 0; i < MAX_MALLOCS; i++)
     {
	/* Check each memory region */
	char * base = ptr_list [i];
	if (base == NULL)
	  continue;
	for (j = 0; j < ((NPAD-1)*sizeof(int)); j++)
	  {
	     if (base [j] != PAD1)
	       {
		  fprintf (stderr, "Ouch! ptr = 0x%08X [j = %d]: \"%s\" (%s/%s)\n",
			   base, j, ptr_name [i], id1, id2);
		  if (prev_id1 != NULL)
		    fprintf (stderr, "Last successful check (%s/%s)\n",
			     prev_id1, prev_id2);
		  fail = 1;
	       }
	  }

	l = *((int *) (&base [(NPAD-1)*sizeof(int)]));
	for (j = l + NPAD * sizeof(int);
	     j < l + (2*NPAD) * sizeof(int); j++)
	  {
	     if (base[j] != PAD2)
	       {
		  fprintf (stderr,"Ouch2! ptr = 0x%08X [k = %d]: \"%s\" (%s/%s)\n",
			   base, j, ptr_name [i], id1, id2);
		  if (prev_id1 != NULL)
		    fprintf (stderr, "Last successful check (%s/%s)\n",
			     prev_id1, prev_id2);
		  fail = 1;
	       }
	  }
     }

   if (!fail)
     {
	prev_id1 = id1;
	prev_id2 = id2;
     }
#endif
}

#endif	/* DEBUG_MALLOC */
/*----------------------- end-of-file (C source) -----------------------*/
