/*--------------------------------*-C-*---------------------------------*
 * File:	command.c
 *
 * Copyright 1992 John Bovey, University of Kent at Canterbury.
 *
 * You can do what you like with this source code as long as
 * you don't try to make money out of it and you include an
 * unaltered copy of this message (including the copyright).
 *
 * This module has been very heavily modified by R. Nation
 * <nation@rocket.sanders.lockheed.com>
 * No additional restrictions are applied
 *
 * Additional modification by Garrett D'Amore <garrett@netcom.com> to
 * allow vt100 printing.  No additional restrictions are applied.
 *
 * Integrated modifications by Steven Hirsch <hirsch@emba.uvm.edu> to
 * properly support X11 mouse report mode and support for DEC
 * "private mode" save/restore functions.
 *
 * Integrated key-related changes by Jakub Jelinek <jj@gnu.ai.mit.edu>
 * to handle Shift+function keys properly.
 * Should be used with enclosed termcap / terminfo database.
 *
 * Additional modifications by mj olesen <olesen@me.QueensU.CA>
 * No additional restrictions.
 *
 * Further modification and cleanups for Solaris 2.x and Linux 1.2.x
 * by Raul Garcia Garcia <rgg@tid.es>. No additional restrictions.
 *
 * As usual, the author accepts no responsibility for anything, nor does
 * he guarantee anything whatsoever.
 *----------------------------------------------------------------------*/
#include "rxvt.h"

#include <ctype.h>
#include <signal.h>
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#ifdef HAVE_FCNTL_H
# include <fcntl.h>
#endif
#ifdef HAVE_SYS_IOCTL_H
# include <sys/ioctl.h>
#endif
#include <sys/types.h>
#include <grp.h>

#ifdef TIME_WITH_SYS_TIME
# include <sys/time.h>
# include <time.h>
#else
# ifdef HAVE_SYS_TIME_H
#  include <sys/time.h>
# else
#  include <time.h>
# endif
#endif

/* I don't see that these are needed */
/* #ifdef SVR4 */
/* # include <sys/stropts.h> */
/* # include <sys/resource.h> */
/* #endif */

#ifdef HAVE_SYS_SELECT_H
# include <sys/select.h>
#endif

#ifdef HAVE_TERMIOS_H
# include <termios.h>
# undef HAVE_SGTTY_H
#else
# ifdef HAVE_SGTTY_H
#   include <sgtty.h>
# endif
#endif

#include <sys/wait.h>
#include <sys/stat.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>

#if defined (__sun__) && !defined (atexit)
# define atexit(x)	on_exit(x,NULL)		/* no ANSI exit handler */
#endif

#include "screen.h"
#include "sbar.h"
#include "xsetup.h"
#include "command.h"
#include "debug.h"
#include "graphics.h"
#include "grkelot.h"
#include "keys.h"

/* the fastest possible baudrate */
#ifdef B38400
# define FAST_BAUD	B38400
#else
# define FAST_BAUD	B9600
#endif

#ifndef MAX_REFRESH_PERIOD
# define MAX_REFRESH_PERIOD	10
#endif
#if !defined (NO_MULTIPLE_CLICK) && !defined (MULTICLICK_TIME)
# define MULTICLICK_TIME	500
#endif

/* time factor to slow down a `jumpy' mouse */
#define MOUSE_THRESHOLD		50

#define PAGE_AMOUNT	0.8	/* amount for page scrolling */
#define CONSOLE		"/dev/console"	/* console device -- rgg 16/12/95 */

/* #define DEBUG_CMD */
/*----------------------------------------------------------------------*
 * extern variables referenced
 */

/*----------------------------------------------------------------------*
 * extern variables declared here
 */
#ifdef WITH_7BIT_MODE
unsigned char hibit_mask = 0xFF;
#else
# define hibit_mask	0xFF
#endif
#ifdef REFRESH_RESOURCE
int refresh_period = MAX_REFRESH_PERIOD;
#else
#define refresh_period	MAX_REFRESH_PERIOD
#endif

#ifdef KEYSYM_RESOURCE
unsigned char * KeySym_map [256];	/* probably mostly empty */
#endif

/*----------------------------------------------------------------------*
 * local variables
 */
static int comm_fd = -1;	/* file descriptor connected to the command */
static pid_t comm_pid = -1;	/* process id if child */
static int x_fd;		/* file descriptor of X server connection */
static int fd_width;		/* width of file descriptors being used */
static struct stat ttyfd_stat;	/* original status of the tty we will use */
static int real_bs = 1;		/* backspace is backspace */

static char *ttynam;		/* tty name */

static unsigned char apl_CUR = 0;   /* application cursor keys */
static unsigned char apl_KP = 1;    /* application keypad */

static int refresh_nl_count = 0;
static int refresh_nl_limit = 1;
static int refresh_type = SLOW_REFRESH;

static Atom wm_del_win;

#ifndef NO_MOUSE_REPORT
#define X10_MOUSE	1
#define X11_MOUSE	2
/* preliminary code for highlight tracking, but too annoying to implement */
/*#define X11_TRACK	4*/
/*#define X11_HILITE	8*/
static int Mouse_Report = 0;
/*static int wait_for_reply = 0;*/
#endif
/* buffered command input */
static unsigned char inbuf [2048], *inbuf_next, *inbuf_top;

/*----------------------------------------------------------------------*
 * extern functions referenced
 */

/*----------------------------------------------------------------------*
 * local functions
 */
static void	privileges (int mode);

static void	catch_child (int);
static void	catch_sig (int);
static void	set_ttymodes (void);
static char *	get_ptty (int *pfd);
static int	get_tty (void);
static int	run_command (char *argv[]);
static void	lookup_key (XEvent *);
static unsigned char get_com_char (void);
static void	get_X_event (void);
/*static void	process_string (int);*/
#ifdef PRINTPIPE
static void	process_print_pipe (void);
#endif
static void	process_escape_seq (void);
static void	process_csi_seq (void);
static void	process_xterm_seq (void);
static void	process_terminal_mode (unsigned char mode, unsigned char priv,
				       int nargs, int arg[]);
static void	process_sgr_mode (int nargs, int arg[]);
static void	process_rxvt_graphics (void);

static void	stty_size (int fd, int width, int height);

/*----------------------------------------------------------------------*/
#define VT100_ANS	"\033[?1;2c"	/* vt100 answerback */
#ifdef ESCZ_RXVT_ANSWER		/* what to do with obsolete ESC-Z */
# define RXVT_ANS	"\033[?1;2C"	/* almost the same */
#else
# define RXVT_ANS	VT100_ANS	/* obsolete ANSI ESC[c */
#endif
/*
 * ESC-Z processing:
 *
 * By stealing a sequence to which other xterms respond, and sending the
 * same number of characters, but having a distinguishable sequence,
 * we can avoid having a timeout (when not under an rxvt) for every login
 * shell to auto-set its DISPLAY.
 *
 * This particular sequence is even explicitly stated as obsolete since
 * about 1985, so only very old software is likely to be confused, a
 * confusion which can likely be remedied through termcap or TERM.
 * Frankly, I doubt anyone will even notice.  We provide a #ifdef just
 * in case they don't care about auto-display setting.  Just in case the
 * ancient software in question is broken enough to be case insensitive
 * to the 'c' character in the answerback string, we make the
 * distinguishing characteristic be capitalization of that character.
 * The length of the two strings should be the same so that identical
 * read(2) calls may be used.
 */

#define KBUFSZ		8	/* size of keyboard mapping buffer */
#define STRING_MAX	128	/* max string size for process_xterm_seq() */
#define ESC_ARGS	32	/* max # of args for esc sequences */

/* Terminal mode structures. */
#ifndef _POSIX_VDISABLE
# define _POSIX_VDISABLE 0	/* Disable special character functions */
#endif

#ifndef CINTR
# if !defined (__alpha) && !defined (FREEBSD)
#  ifdef SVR4		/* really? */
#   define CINTR	'\177'		/* ^? */
#   define CQUIT	'\025'		/* ^U */
#   define CERASE	'\010'		/* ^H */
#  else
#   define CINTR	'\003'		/* ^C */
#   define CQUIT	'\034'		/* ^\ */
#   define CERASE	'\010'		/* ^H */
#  endif
#  define CKILL		'\025'		/* ^U */
#  define CEOF		'\004'		/* ^D */
#  define CSTART	'\021'		/* ^Q */
#  define CSTOP		'\023'		/* ^S */
#  define CSUSP		'\032'		/* ^Z */
#  define CDSUSP	'\031'		/* ^Y */
#  define CRPRNT	'\022'		/* ^R */
#  define CFLUSH	'\017'		/* ^O */
#  define CWERASE	'\027'		/* ^W */
#  define CLNEXT	'\026'		/* ^V */
# endif
#endif

#ifdef hpux			/* really? */
# undef CSUSP
# define CSUSP	CSWTCH
#endif

#if !defined (_POSIX_SOURCE) && defined (SVR4)
static int
getdtablesize (void)
{
   struct rlimit rlp;
   getrlimit (RLIMIT_NOFILE, &rlp);
   return rlp.rlim_cur;
}
#endif

/*
 * take care of suid/sgid privileges
 */
static void
privileges (int mode)
{
   static uid_t saved_uid;
   static gid_t saved_gid;

   switch (mode) {
    case RESTORE:
      seteuid (saved_uid);
      seteuid (saved_gid);
      break;
    case SAVE:
      saved_uid = geteuid ();
      saved_gid = getegid ();
      break;
    case IGNORE:
    default:
      seteuid (getuid ());
      setegid (getgid ());
      break;
   }
}

/*
 * Catch a SIGCHLD signal and exit if the direct child has died
 */
static RETSIGTYPE
catch_child (int unused)
{
   if (wait (NULL) == comm_pid)
     exit (EXIT_SUCCESS);
   signal (SIGCHLD, catch_child);	/* rgg */
}

/*
 * Catch a fatal signal and tidy up before quitting
 */
static RETSIGTYPE
catch_sig (int sig)
{
#ifdef DEBUG_CMD
   print_error ("signal %d", sig);
#endif
   signal (sig, SIG_DFL);

   privileges (RESTORE);
   cleanutent ();
   privileges (IGNORE);

   setuid (getuid ());
   kill (getpid (), sig);
}

/*
 * Exit gracefully, clearing the utmp entry and restoring tty attributes
 */
static void
clean_exit (void)
{
#ifdef DEBUG_CMD
   fprintf (stderr, "Restoring %s to mode %03o, uid %d, gid %d\n",
	    ttynam, ttyfd_stat.st_mode, ttyfd_stat.st_uid, ttyfd_stat.st_gid);
#endif
   privileges (RESTORE);
   chmod (ttynam, ttyfd_stat.st_mode);
   chown (ttynam, ttyfd_stat.st_uid, ttyfd_stat.st_gid);
   cleanutent ();
   privileges (IGNORE);
}

/*
 * Acquire a pseudo-teletype from the system.
 * The return value is the name of the slave part of the pair or
 * NULL if unsucsessful.  If successful, the file descriptor is
 * returned via the argument.
 */
static char *
get_ptty (int *pfd)
{
   int ptyfd = -1;
#ifdef SVR4
   extern char *ptsname();
   char *tty;

   ptyfd = open ("/dev/ptmx",O_RDWR);
   if (ptyfd < 0)
     {
	print_error ("can't open pseudo tty");
	return NULL;
     }
   grantpt (ptyfd);
   unlockpt (ptyfd);
   tty = ptsname (ptyfd);
#else
   static char tty[] = "/dev/tty??";
   char *c8, *c9, pty[] = "/dev/pty??";

   for (c8 = "pqrstuvwxyz"; *c8; c8++) {
      for (c9 = "0123456789abcdef"; *c9; c9++) {
	 pty [8] = tty [8] = *c8;
	 pty [9] = tty [9] = *c9;
	 if ((ptyfd = open (pty, O_RDWR)) >= 0) {
	    if (geteuid () == 0 || access (tty, R_OK|W_OK) == 0)
	      goto Found_pty;
	    close (ptyfd);
	    ptyfd = -1;
	 }
      }
   }

   if (ptyfd < 0)
     {
	print_error ("can't open pseudo tty");
	return NULL;
     }
#endif
   Found_pty:
   /* 9term uses fcntl (ptyfd, F_SETFL, O_NONBLOCK); -- rgg 12/12/95 */
   fcntl (ptyfd, F_SETFL, O_NDELAY);

   *pfd = ptyfd;
   return tty;
}

static int
get_tty (void)
{
   uid_t uid;
   gid_t gid;
   int i, ttyfd;

   if ((ttyfd = open (ttynam, O_RDWR)) < 0)
     {
	print_error ("can't open slave tty %s", ttynam);
	exit (EXIT_FAILURE);
     }
   uid = getuid ();
#ifdef SVR4
   ioctl (ttyfd, I_PUSH, "ptem");
   ioctl (ttyfd, I_PUSH, "ldterm");
#else
     {
	struct group *gr = getgrnam ("tty");
	gid = (gr == NULL ? -1 : gr->gr_gid);
     }
   /* 9term uses: gid = getgid (); -- rgg 12/12/95 */

   privileges (RESTORE);
   fchown (ttyfd, uid, gid);
   fchmod (ttyfd, 0600);
   privileges (IGNORE);

   setuid (uid);
   setgid (getgid ());
#endif	/* SVR4 */

   /*
    * Redirect critical fd's and close the rest.
    *
    * If only stdin, stdout, and stderr are closed, all children
    * processes remain alive upon deletion of the window, so it
    * is necessary to close fd's not only < 3, but all of them,
    * that is, fd's < fd_width -- rgg 10/12/95
    */
   for (i = 0; i < fd_width; i++)
     if (i != ttyfd)
       close (i);

   dup (ttyfd);
   dup (ttyfd);
   dup (ttyfd);

   if (ttyfd > 2)
     close (ttyfd);

#ifdef ultrix
   if ((ttyfd = open ("/dev/tty", O_RDONLY)) >= 0)
     {
	ioctl (ttyfd, TIOCNOTTY, 0);
	close (ttyfd);
     }
   else
     setpgrp (0, 0);
   /* no error, we could run with no tty to begin with */
#else	/* ultrix */
#ifdef NO_SETSID
   if ((gid = setpgrp (0, 0)) < 0)
     print_error ("failed to set process group");
#else
   if ((gid = setsid ()) < 0)
     print_error ("failed to start session");
#endif

#ifdef TIOCSCTTY
   ioctl (0, TIOCSCTTY, 0);
#endif

#ifdef TIOCSPGRP
     {
	int pid = getpid ();
#ifdef _POSIX_SOURCE
	tcsetpgrp (0, pid);
#else
	ioctl (0, TIOCSPGRP, (char *)&pid);	/* set process gorup */
#endif
	close (open (ttynam, O_WRONLY, 0));
     }
#elif defined (SVR4)
   tcsetpgrp (0, gid);
#endif
#endif	/* ultrix */

   setuid (uid);
   setgid (getgid ());

   return ttyfd;
}

/*
 * Initialize terminal attributes
 */
static void
set_ttymodes (void)
{
#ifdef HAVE_TERMIOS_H
   /* Set the terminal using the standard System V termios interface */
   struct termios ttmode;

   /* ways to deal with getting/setting termios structure */

   /* HPUX uses POSIX terminal I/O */
#ifdef TCSANOW                  /* POSIX */
# define GET_TERMIOS(fd,x)      tcgetattr (fd, x)
# define SET_TERMIOS(fd,x)      tcsetattr (fd, TCSANOW, x)
#else
# ifdef TIOCSETA
#  define GET_TERMIOS(fd,x)     ioctl (fd, TIOCGETA, (char *)(x))
#  define SET_TERMIOS(fd,x)     ioctl (fd, TIOCSETA, (char *)(x))
# else
#  define GET_TERMIOS(fd,x)     ioctl (fd, TCGETS, (char *)(x))
#  define SET_TERMIOS(fd,x)     ioctl (fd, TCSETS, (char *)(x))
# endif
#endif

   GET_TERMIOS (0, &ttmode);	/* init termios structure */

   ttmode.c_iflag = (BRKINT | IGNPAR | ICRNL | IXON
#ifdef IMAXBEL
		     | IMAXBEL
#endif
		     );
   ttmode.c_lflag = (ISIG | ICANON | IEXTEN | ECHO | ECHOE | ECHOK
#if defined (ECHOCTL) && defined (ECHOKE)
		     | ECHOCTL | ECHOKE
#endif
		     );
   ttmode.c_oflag = OPOST | ONLCR;
   ttmode.c_cflag = FAST_BAUD | CS8 | CREAD;

   ttmode.c_cc [VINTR]	= CINTR;
   ttmode.c_cc [VQUIT]	= CQUIT;
   ttmode.c_cc [VERASE]	= CERASE;
   ttmode.c_cc [VKILL]	= CKILL;
   ttmode.c_cc [VEOF]	= CEOF;
   ttmode.c_cc [VEOL]	= _POSIX_VDISABLE;
#ifdef VEOL2
   ttmode.c_cc [VEOL2]	= _POSIX_VDISABLE;
#endif
#ifdef VSWTC
   ttmode.c_cc [VSWTC]	= _POSIX_VDISABLE;
#endif
#ifdef VSWTCH
   ttmode.c_cc [VSWTCH]	= _POSIX_VDISABLE;
#endif
   ttmode.c_cc [VSTART]	= CSTART;
   ttmode.c_cc [VSTOP]	= CSTOP;
   ttmode.c_cc [VSUSP]	= CSUSP;
#ifdef VREPRINT
   ttmode.c_cc [VREPRINT] = CRPRNT;
#endif

#if defined (VDISCRD) && !defined (VDISCARD)
# define VDISCARD	VDISCRD
#endif
#ifdef VDISCARD
   ttmode.c_cc [VDISCARD] = CFLUSH;
#endif
#if defined (VWERSE) && !defined (VWERASE)
# define VWERASE	VWERSE
#endif
#ifdef VWERASE
   ttmode.c_cc [VWERASE] = CWERASE;
#endif

#ifdef VLNEXT
   ttmode.c_cc [VLNEXT] = CLNEXT;
#endif

#ifdef VDSUSP
   ttmode.c_cc [VDSUSP]	= CDSUSP;
#endif

#if VMIN != VEOF
   ttmode.c_cc [VMIN] = 1;
#endif
#if VTIME != VEOL
   ttmode.c_cc [VTIME] = 0;
#endif

#ifdef WITH_7BIT_MODE
   if (hibit_mask == 0x7F)
     {
	ttmode.c_cflag &= ~CS8;
	ttmode.c_cflag |= CS7 | PARENB;
     }
#endif

   SET_TERMIOS (0, &ttmode);

#else /* HAVE_TERMIOS_H */

   /*
    * Use sgtty rather than termios interface to configure the terminal
    */
   int ldisc, lmode;
   struct sgttyb tty;
   struct tchars tc;
   struct ltchars ltc;

#ifdef NTTYDISC
   ldisc = NTTYDISC;
   ioctl (0, TIOCSETD, &ldisc);
#endif /* NTTYDISC */
   tty.sg_ispeed = B9600;
   tty.sg_ospeed = B9600;
   tty.sg_erase = CINTR;
   tty.sg_kill = CKILL;		/* ^U */
   tty.sg_flags = CRMOD | ECHO | EVENP | ODDP;
   ioctl (0, TIOCSETP, &tty);

   tc.t_intrc = CINTR;		/* ^C */
   tc.t_quitc = CQUIT;		/* ^\ */
   tc.t_startc = CSTART;	/* ^Q */
   tc.t_stopc = CSTOP;		/* ^S */
   tc.t_eofc = CEOF;		/* ^D */
   tc.t_brkc = -1;
   ioctl (0, TIOCSETC, &tc);

   ltc.t_suspc  = CSUSP;	/* ^Z */
   ltc.t_dsuspc = CDSUSP;	/* ^Y */
   ltc.t_rprntc = CRPRNT;	/* ^R */
   ltc.t_flushc = CFLUSH;	/* ^O */
   ltc.t_werasc = CWERASE;	/* ^W */
   ltc.t_lnextc = CLNEXT;	/* ^V */
   ioctl (0, TIOCSLTC, &ltc);

   lmode = LCRTBS | LCRTERA | LCTLECH | LPASS8 | LCRTKIL;
   ioctl (0, TIOCLSET, &lmode);
#endif /* HAVE_TERMIOS_H */

   /* common part: set window size */
   stty_size (0, TermWin.ncol, TermWin.nrow);

    /* become virtual console, if required and possible */
   if (console)
     {
	int tmp;
#ifdef TIOCCONS
 	tmp = 1;		/* on */

 	/* if (ioctl (ttyfd, TIOCCONS, (unsigned char *)&on) == -1) */
 	if (ioctl (0, TIOCCONS, (unsigned char *)&tmp) == -1)
 	  {
	     print_error ("can't open console");
	     perror ("");
	  }
#elif defined (SRIOCREDIR)				/* rgg -- 16/12/95 */
	tmp = open (CONSOLE, O_WRONLY);

	if (tmp < 0 || ioctl (0, SRIOCSREDIR, tmp) == -1)
	  {
	     if (tmp >= 0)
	       close (tmp);
	     print_error ("can't open console");
	     perror ("");
	  }

#else
	print_error ("can't open console");
#endif	/* TIOCCONS */
     }
}

/*
 * Run the command in a subprocess and return a file descriptor for the
 * master end of the pseudo-teletype pair with the command talking to
 * the slave.
 */
static int
run_command (char *argv[])
{
   int i, ptyfd;
   char *cmd, argv0 [256];

   /* Save and then give up any suid/sgid privileges */
   privileges (SAVE);
   privileges (IGNORE);

   if ((ttynam = get_ptty (&ptyfd)) == NULL)
     return -1;

   /* store original tty status for restoration clean_exit() -- rgg 04/12/95 */
   lstat (ttynam, &ttyfd_stat);
#ifdef DEBUG_CMD
   fprintf (stderr, "Original settings of %s are mode %o, uid %d, gid %d\n",
	    ttynam, ttyfd_stat.st_mode, ttyfd_stat.st_uid, ttyfd_stat.st_gid);
#endif

   atexit (clean_exit);		/* install exit handler with cleanup */

   /* get number of available file descriptors */
#ifdef _POSIX_VERSION
   fd_width = sysconf (_SC_OPEN_MAX);
#else
   fd_width = getdtablesize ();
#endif

    /* spin off the command interpreter */
   signal (SIGHUP, catch_sig);
   signal (SIGINT, catch_sig);
   signal (SIGQUIT, catch_sig);
   signal (SIGTERM, catch_sig);
   signal (SIGCHLD, catch_child);

   comm_pid = fork ();
   if (comm_pid < 0)
     {
	print_error ("can't fork");
	return -1;
     }
   if (comm_pid == 0)			/* command interpreter path */
     {
#ifdef HAVE_UNSETENV
	/*
	 * bash may set these, but they should be clear for the child or
	 * the old settings are passed on and will confuse term size,
	 * unsetenv may not be very portable, surround with HAVE_UNSETENV
	 */
	unsetenv ("LINES");
	unsetenv ("COLUMNS");
#endif	/* HAVE_UNSETENV */

	/*
	 *  Having started a new session, we need to establish
	 *  a controlling teletype for it.  On some systems
	 *  this can be done with an ioctl() but on others
	 *  we need to re-open the slave tty.
	 */
	get_tty ();
	set_ttymodes ();

	/* set window size */
	stty_size (0, TermWin.ncol, TermWin.nrow);

	/* reset signals and spin off the command interpreter */
	signal (SIGINT,  SIG_DFL);
	signal (SIGQUIT, SIG_DFL);
	signal (SIGCHLD, SIG_DFL);
#ifdef SIGTSTP
	/* mimick login's behavior by disabling the job control   */
	/* signals; a shell that wants them can turn them back on */
	signal (SIGTSTP, SIG_IGN);
	signal (SIGTTIN, SIG_IGN);
	signal (SIGTTOU, SIG_IGN);
#endif /* SIGTSTP */

	cmd = argv [0];
	if (login_shell
	    /* only use a login shell for commands that end with "sh" */
	    /* && (((i = strlen (argv[0])) >= 2) && !strcmp (&argv[i-2], "sh")) */
	    )
	  {
	     strcpy (&argv0[1], my_basename (argv[0]));
	     argv0[0] = '-';
	     argv [0] = argv0;
	  }
	else
	  {
	     argv [0] = my_basename (argv[0]);
	  }
#ifdef DEBUG_CMD
	fprintf (stderr, "cmd = \"%s\"\n", cmd);
	for (i = 0; argv [i]; i++)
	  fprintf (stderr, "argv[%d] = \"%s\"\n", i, argv [i]);
#endif
	execvp (cmd, argv);
	print_error ("can't execute \"%s\"", cmd);
	exit (EXIT_FAILURE);
     }

   privileges (RESTORE);
   makeutent (&ttynam[5]);	/* stamp /etc/utmp */
   privileges (IGNORE);

   return ptyfd;
}

/*
 * Tell the teletype handler what size the window is.
 * Called after a window size change.
 */
static void
stty_size (int fd, int width, int height)
{
   struct winsize wsize;

   if (fd < 0) return;

   wsize.ws_col = (unsigned short) width;
   wsize.ws_row = (unsigned short) height;
   wsize.ws_xpixel = wsize.ws_ypixel = 0;
   ioctl (fd, TIOCSWINSZ, (unsigned char *) &wsize);
}

void
tty_resize (void)
{
   stty_size (comm_fd, TermWin.ncol, TermWin.nrow);
}

/*
 * Initialize the command connection.
 * This should be called after the X server connection is established.
 */
void
init_command (char *argv[])
{
#ifdef GREEK_SUPPORT
   greek_init ();
#endif
   /* Enable the delete window protocol */
   wm_del_win = XInternAtom (Xdisplay, "WM_DELETE_WINDOW", False);
   XSetWMProtocols (Xdisplay, TermWin.parent, &wm_del_win, 1);

   if ((comm_fd = run_command (argv)) < 0)
     {
	print_error ("aborting");
	exit (EXIT_FAILURE);
     }

   x_fd = XConnectionNumber (Xdisplay);
   inbuf_next = inbuf_top = inbuf;
}

#ifdef DEBUG_CMD
static void
print_keystring (unsigned keysym, char *str, int len)
{
   int i;

   /* Display the keyboard buffer contents -- rgg 06/12/95 */
   fprintf (stderr, "key 0x%04X [%d]: `", keysym, len);
   for (i = 0; i < len; i++)
     {
	if (str [i] < ' ' || str [i] >= '\177')
	  fprintf (stderr, "\\%03o", str[i]);
	else
	  fprintf (stderr, "%c", str[i]);
     }
   fprintf (stderr, "'\n");
}
#endif /* DEBUG_CMD */

/*
 * Convert the keypress event into a string
 *
 * use `buffer' as a temporary space for mapping most keys, but
 * use the pointer kbuf to that it is easy to use the KEYSYM_RESOURCE
 * without worrying about length restrictions
 */
static void
lookup_key (XEvent *ev)
{
#ifdef DEBUG_CMD
   static int debug_key = 1;	/* accessible by a debugger only */
#endif
#ifdef GREEK_SUPPORT
   static int greek_mode = 0;
#endif
   static int numlock_state = 1;
   static XComposeStatus compose = {NULL, 0};
   static unsigned char buffer [KBUFSZ];
   unsigned char *kbuf;		/* pointer to key-buffer */
   int len, shift, ctrl, meta;
   KeySym keysym;

   /*
    * if numlock has been used -- allow it as a toggle.
    * Otherwise, respect the application keypad set by escape sequence
    * but allow shift to override
    */
   shift= (ev->xkey.state & ShiftMask);
   ctrl = (ev->xkey.state & ControlMask);
   meta = (ev->xkey.state & Mod1Mask);
   len  = (ev->xkey.state & Mod5Mask);	/* tmp value */
   if (numlock_state || len)
     {
	numlock_state = len;	/* numlock toggle */
	apl_KP = !len;		/* current state */
     }

   len = XLookupString (&ev->xkey, buffer, sizeof(buffer)-1,
			&keysym, &compose);

   kbuf = buffer;		/* it was copied to here */
#ifndef HOTKEY
# define HOTKEY	meta
#endif
   if (HOTKEY) {
      if (keysym == ks_pageup)
	{ scr_page (UP, TermWin.nrow * PAGE_AMOUNT); return; }
      else if (keysym == ks_pagedown)
	{ scr_page (DOWN, TermWin.nrow * PAGE_AMOUNT); return; }
#if (NFONTS > 1)
      else if (keysym == ks_bigfont)	{ new_font (UP); return; }
      else if (keysym == ks_smallfont)	{ new_font (DOWN); return; }
#endif
#ifdef MAPALERT
      else if (keysym == ks_alert)	{ map_alert = !map_alert; return; }
#endif
   }
#ifdef GREEK_SUPPORT
   if (keysym == ks_greektoggle)
     {
	greek_mode = !greek_mode;
	/* no consistent way to set/restore the title since it could have
	 * been set via an XTerm escape sequence, so don't try
	 */
	if (greek_mode) {
	   xterm_seq (NEW_TITLE_NAME,
		      (greek_getmode () == GREEK_ELOT928 ?
		       "[GRK elot]" : "[GRK 437]"));
	   greek_reset ();
	} else {
	  xterm_seq (NEW_TITLE_NAME, rs_title);
	}
	return;
     }
#endif
#ifdef PRINTPIPE
   if (keysym == ks_printscreen)
     { scr_printscreen (ctrl|shift); return; }
#else
#ifdef DEBUG_SELECTION
   if (keysym == XK_Print)
     { scr_debug_selection_state (); return; }	/* hidden debugging dump */
#endif
#endif

   if (keysym >= 0xFF00 && keysym <= 0xFFFF) {
      /* Shift + F1 - F10 generates F11 - F20 */
      if (shift && keysym >= XK_F1 && keysym <= XK_F10)
	{
	   shift = 0;		/* turn off */
	   keysym += (XK_F11 - XK_F1);
	}

#ifdef KEYSYM_RESOURCE
      if (KeySym_map [keysym - 0xFF00] != NULL)
	{
	   /* don't use static_kbuf -- allow arbitrary length strings */
	   shift = meta = ctrl = 0;	/* no post-processing */
	   kbuf = KeySym_map [keysym - 0xFF00];
	   len = strlen (kbuf);
	}
      else
#endif
      switch (keysym) {
       case XK_BackSpace:
	 len = 1;
	 kbuf[0] = ((shift|ctrl) ? real_bs : !real_bs) ? '\177' : '\b';
	 break;

       case XK_Tab:	if (shift) { len = 3; strcpy(kbuf,"\033[Z"); }	break;

       case XK_Up:
	 len = 3; strcpy(kbuf, "\033[A");
	 if (apl_CUR) kbuf [1] = 'O';
#ifdef CURSOR_SHIFT
	 else if (shift) kbuf [2] = 'a';
#endif
#ifdef CURSOR_CTRL
	 else if (ctrl) kbuf [2] = '\001';
#endif
	 break;
       case XK_Down:
	 len = 3; strcpy(kbuf, "\033[B");
	 if (apl_CUR) kbuf [1] = 'O';
#ifdef CURSOR_SHIFT
	 else if (shift) kbuf [2] = 'b';
#endif
#ifdef CURSOR_CTRL
	 else if (ctrl) kbuf [2] = '\002';
#endif
	 break;
       case XK_Right:
	 len = 3; strcpy(kbuf, "\033[C");
	 if (apl_CUR) kbuf [1] = 'O';
#ifdef CURSOR_SHIFT
	 else if (shift) kbuf [2] = 'c';
#endif
#ifdef CURSOR_CTRL
	 else if (ctrl) kbuf [2] = '\003';
#endif
	 break;
       case XK_Left:
	 len = 3; strcpy(kbuf, "\033[D");
	 if (apl_CUR) kbuf [1] = 'O';
#ifdef CURSOR_SHIFT
	 else if (shift) kbuf [2] = 'd';
#endif
#ifdef CURSOR_CTRL
	 else if (ctrl) kbuf [2] = '\004';
#endif
	 break;

       case XK_Find:	len = strlen(strcpy(kbuf,KS_FIND));	break;
       case XK_Insert:	len = strlen(strcpy(kbuf,KS_INSERT));	break;
       case XK_Execute:	len = strlen(strcpy(kbuf,KS_EXECUTE));	break;
       case XK_Select:	len = strlen(strcpy(kbuf,KS_SELECT));	break;
       case XK_Prior:	len = strlen(strcpy(kbuf,KS_PRIOR));	break;
       case XK_Next:	len = strlen(strcpy(kbuf,KS_NEXT));	break;
#ifdef KS_DELETE
       case XK_Delete:	len = strlen(strcpy(kbuf,KS_DELETE));	break;
#endif
       case XK_Home:	len = strlen(strcpy(kbuf,KS_HOME));	break;
       case XK_End:	len = strlen(strcpy(kbuf,KS_END));	break;

       case XK_KP_Enter:
	 if (apl_KP ? !shift : shift)	/* allow shift to override */
	   {
	      len = 3; strcpy(kbuf, "\033OM");
	   }
	 else
	   {
	      len = 1; kbuf[0] = '\r';
	   }
	 break;

       case XK_KP_F1:		/* "\033OP" */
       case XK_KP_F2:		/* "\033OQ" */
       case XK_KP_F3:		/* "\033OR" */
       case XK_KP_F4:		/* "\033OS" */
	 len = 3; strcpy(kbuf, "\033OP");
	 kbuf[2] += (keysym - XK_KP_F1);
	 break;

       case XK_KP_Multiply:	/* "\033Oj" : "*" */
       case XK_KP_Add:		/* "\033Ok" : "+" */
       case XK_KP_Separator:	/* "\033Ol" : "," */
       case XK_KP_Subtract:	/* "\033Om" : "-" */
       case XK_KP_Decimal:	/* "\033On" : "." */
       case XK_KP_Divide:	/* "\033Oo" : "/" */
       case XK_KP_0:		/* "\033Op" : "0" */
       case XK_KP_1:		/* "\033Oq" : "1" */
       case XK_KP_2:		/* "\033Or" : "2" */
       case XK_KP_3:		/* "\033Os" : "3" */
       case XK_KP_4:		/* "\033Ot" : "4" */
       case XK_KP_5:		/* "\033Ou" : "5" */
       case XK_KP_6:		/* "\033Ov" : "6" */
       case XK_KP_7:		/* "\033Ow" : "7" */
       case XK_KP_8:		/* "\033Ox" : "8" */
       case XK_KP_9:		/* "\033Oy" : "9" */
	 if (apl_KP ? !shift : shift)	/* allow shift to override */
	   {
	      len = 3; strcpy(kbuf, "\033Oj");
	      kbuf[2] += (keysym - XK_KP_Multiply);
	   }
	 else
	   {
	      len = 1; kbuf[0] = ('*' + (keysym - XK_KP_Multiply));
	   }
	 break;

       case XK_F1:		/* "\033[11~" */
       case XK_F2:		/* "\033[12~" */
       case XK_F3:		/* "\033[13~" */
       case XK_F4:		/* "\033[14~" */
       case XK_F5:		/* "\033[15~" */
	 len = 5; sprintf(kbuf, "\033[%2d~", (11 + (keysym - XK_F1)));	break;

       case XK_F6:		/* "\033[17~" */
       case XK_F7:		/* "\033[18~" */
       case XK_F8:		/* "\033[19~" */
       case XK_F9:		/* "\033[20~" */
       case XK_F10:		/* "\033[21~" */
	 len = 5; sprintf(kbuf, "\033[%2d~", (17 + (keysym - XK_F6)));	break;

       case XK_F11:		/* "\033[23~" */
       case XK_F12:		/* "\033[24~" */
       case XK_F13:		/* "\033[25~" */
       case XK_F14:		/* "\033[26~" */
	 len = 5; sprintf(kbuf, "\033[%2d~", (23 + (keysym - XK_F11)));	break;

       case XK_Help:
       case XK_F15:	len = 5; strcpy(kbuf,"\033[28~");	break;
       case XK_Menu:
       case XK_F16:	len = 5; strcpy(kbuf,"\033[29~");	break;
       case XK_F17:		/* "\033[31~" */
       case XK_F18:		/* "\033[32~" */
       case XK_F19:		/* "\033[33~" */
       case XK_F20:		/* "\033[34~" */
	 len = 5; sprintf(kbuf, "\033[%2d~", (31 + (keysym - XK_F17)));	break;
      }
   }
   else if (ctrl && keysym == XK_minus) {
      len = 1; kbuf [0] = '\037';	/* Ctrl-Minus generates ^_ (31) */
   }
   else {
#ifndef META_ESCAPE
      /* set 8-bit on */
      if (meta && (meta_char == 0x80))
	{
	   unsigned char *c;
	   for (c = kbuf; c < kbuf + len; c++)
	     *c |= 0x80;
	   meta = 0;
	}
#endif
#ifdef GREEK_SUPPORT
      if (greek_mode)
	len = greek_xlat (kbuf, len);
#endif
      /*nil*/;
   }

   if (len <= 0) return;	/* not mapped */

   /*
    * these modifications only affect the static keybuffer
    * pass Shift/Control indicators for function keys ending with `~'
    */
   if (len > 3 && kbuf [len-1] == '~')
     {
	if (shift)
	  kbuf [len-1] = '$';
	else if (ctrl)
	  kbuf [len-1] = '^';
     }

   /* escape prefix */
   if (meta && (len < sizeof(buffer)-1)
#ifndef META_ESCAPE
       && (meta_char == 033)
#endif
       )
     {
	unsigned char *c;
	for (c = kbuf + len; c > kbuf ; c--)
	  *c = *(c-1);

	kbuf [0] = '\033';
	len++;
     }

   tty_write (kbuf, len);
#ifdef DEBUG_CMD
   if (debug_key)
     print_keystring (keysym, kbuf, len);
#endif
}

/*
 * Return the next input character after first passing any keyboard input
 * to the command.
 */
static unsigned char
get_com_char (void)
{
   static int refreshed = 0;
   fd_set in_fdset;
   int retval;
   struct itimerval value;

   /* If there have been a lot of new lines, then update the screen
    * What the heck I'll cheat and only refresh less than every page-full.
    * the number of pages between refreshes is refresh_nl_limit, which
    * is incremented here because we must be doing flat-out scrolling.
    *
    * refreshing should be correct for small scrolls, because of the
    * time-out */
   if (refresh_nl_count > refresh_nl_limit * TermWin.nrow)
     {
	if (refresh_nl_limit < refresh_period)
	  refresh_nl_limit++;
	refresh_nl_count = 0;
	refreshed = 1;
	scr_refresh (refresh_type);
     }

   /* If we have characters already read in. return one */
   if (inbuf_top > inbuf_next)
     {
	refreshed = 0;
	return (*inbuf_next++ & hibit_mask);
     }

   while (1)
     {
	/* process any X events that are pending */
	while (XPending (Xdisplay))
	  {
	     refreshed = 0;
	     get_X_event ();
	  }

	/* Nothing to do! */
	FD_ZERO (&in_fdset);
	FD_SET (comm_fd, &in_fdset);
	FD_SET (x_fd, &in_fdset);
	value.it_value.tv_usec = 5000;
	value.it_value.tv_sec = 0;

	retval = select (fd_width, &in_fdset,
			 NULL, NULL, (refreshed ? NULL : &value.it_value));

	/* See if we can read from the application */
	if (FD_ISSET (comm_fd, &in_fdset))
	  {
	     int total = sizeof(inbuf);
	     inbuf_next = inbuf_top = inbuf;
	     while (total > sizeof(inbuf) / 2)
	       {
		  int cnt = read (comm_fd, inbuf_top, total);
		  if (cnt <= 0)
		    break;
		  inbuf_top += cnt;
		  total -= cnt;
	       }
	     if (inbuf_top > inbuf_next)
	       {
		  refreshed = 0;
		  return (*inbuf_next++ & hibit_mask);
	       }
	  }
	/* If the select statement timed out, we better update the screen */
	if (retval == 0)
	  {
	     refresh_nl_count = 0;
	     refresh_nl_limit = 1;
	     if (!refreshed)
	       scr_refresh (refresh_type);
	     refreshed = 1;
#ifndef SCROLLBAR_NONE
	     sbar_show (TermWin.hist_count + (TermWin.nrow-1),
			TermWin.hist_start,
			(TermWin.nrow-1));
#endif
	     /* XFlush (Xdisplay);*/
	  }
     }
   return 0;
}

/*
 * Receive and process an X event
 */
static void
get_X_event (void)
{
   XEvent ev;
   Window root, child;

#ifndef NO_MOUSE_REPORT
   static int last_keystate = 0;
#endif
#ifndef NO_MULTIPLE_CLICK
   static Time buttonpress_time;
   static int clicks = 0;
#endif
   static int size_set = 0;	/* flag that window size has been set */

   int x, y, root_x, root_y;
   unsigned int mods;
   XEvent dummy;

   XNextEvent (Xdisplay, &ev);

   switch (ev.type) {
    case KeyPress:
      lookup_key (&ev);
      break;

    case ClientMessage:
      if (ev.xclient.format == 32 && ev.xclient.data.l[0] == wm_del_win)
	exit (EXIT_SUCCESS);
      break;

    case MappingNotify:
      XRefreshKeyboardMapping (&ev.xmapping);
      break;

    case GraphicsExpose:
    case Expose:
      if (!size_set)
	{
	   /*
	    * Force resize if an exposure event arrives before the first
	    * resize event
	    */
	   resize_window ();
	   size_set = 1;
	}
      if (ev.xany.window == TermWin.vt)
	{
	   scr_touch (ev.xexpose.x, ev.xexpose.y,
		      ev.xexpose.width, ev.xexpose.height);
	}
      else
	{
	   while (XCheckTypedWindowEvent (Xdisplay, ev.xany.window,
					  Expose, &dummy));
	   while (XCheckTypedWindowEvent (Xdisplay, ev.xany.window,
					  GraphicsExpose, &dummy));
#ifndef SCROLLBAR_NONE
	   if (ev.xany.window == sbar.sb.win)
	     sbar_show (-1,-1,-1);
#ifndef SCROLLBAR_XTERM
	   else if (ev.xany.window == sbar.up.win)
	     sbar_up_reset ();
	   else if (ev.xany.window == sbar.dn.win)
	     sbar_down_reset ();
#endif	/* ! SCROLLBAR_XTERM */
	   else
#endif
	     Gr_expose (ev.xany.window);
	}
      break;

    case VisibilityNotify:
      /* Here's my conclusiion:
       * If the window is completely unobscured, use bitblt's
       * to scroll. Even then, they're only used when doing partial
       * screen scrolling. When partially obscured, we have to fill
       * in the GraphicsExpose parts, which means that after each refresh,
       * we need to wait for the graphics expose or Noexpose events,
       * which ought to make things real slow! */
      switch (ev.xvisibility.state) {
       case VisibilityUnobscured:
	 refresh_type = FAST_REFRESH;
	 /* scr_touch (0, 0, TermWin.width, TermWin.height);*/
	 break;
       case VisibilityPartiallyObscured:
	 refresh_type = SLOW_REFRESH;
	 /* scr_touch (0, 0, TermWin.width, TermWin.height);*/
	 break;
       default:
	 refresh_type = NO_REFRESH;
	 break;
      }
      break;

    case FocusIn:	scr_focus (1);	break;
    case FocusOut:	scr_focus (0);	break;
    case ConfigureNotify:
      resize_window ();
      size_set = 1;
      break;

    case SelectionClear:
      scr_selection_delete ();
      break;

    case SelectionNotify:
      scr_paste_primary (ev.xselection.requestor,
			 ev.xselection.property, True);
      break;

    case SelectionRequest:
      scr_selection_send (&(ev.xselectionrequest));
      break;

    case ButtonPress:
#ifndef NO_MOUSE_REPORT
      last_keystate = ev.xbutton.state & (Mod1Mask|ShiftMask);
#endif
      if (ev.xany.window == TermWin.vt)
	{
	   if (ev.xbutton.subwindow != None)
	     Gr_ButtonPress (ev.xbutton.x, ev.xbutton.y);
	   else
	     {
#ifndef NO_MOUSE_REPORT
		if (Mouse_Report && !(last_keystate))
		  {
#ifndef NO_MULTIPLE_CLICK
		     clicks = 0;
#endif
		     if (Mouse_Report & X10_MOUSE)
		       ev.xbutton.state = 0;	/* no state info allowed */
		     mouse_report (&(ev.xbutton));
#ifdef X11_TRACK
		     if (Mouse_Report & X11_TRACK)
		       {
			  /* wait for and get reply
			   * "ESC [ Ps ; Ps ; Ps ; Ps ; Ps T"
			   * Parameters =
			   *  func, startx, starty, firstrow, lastrow.
			   * func is non-zero to initiate hilite tracking
			   */
			  if (func)
			    Mouse_Report |= X11_HILITE;
			  else
			    Mouse_Report &= ~X11_HILITE;

			  mouse_tracking (0, x, y, r1, r2);
		       }
#endif
		     return;
		  }
#endif
		switch (ev.xbutton.button) {
		 case Button1:
#ifndef NO_MULTIPLE_CLICK
		   if (ev.xbutton.time - buttonpress_time < MULTICLICK_TIME)
		     clicks++;
		   else
		     clicks = 1;
		   buttonpress_time = ev.xbutton.time;
		   scr_selection_click (clicks,
					ev.xbutton.x, ev.xbutton.y);
#else
		   scr_selection_start (ev.xbutton.x, ev.xbutton.y);
#endif
		   break;

		 case Button3:
		   scr_selection_extend (1, ev.xbutton.x, ev.xbutton.y);
		   break;
		}
		return;
	     }
	}
#ifndef SCROLLBAR_NONE
#ifndef NO_MOUSE_REPORT
      /* disabled scrollbar
       * arrow buttons - send up/down
       * click on scrollbar - send pageup/down
       */
      if (Mouse_Report && !(last_keystate))
	{
	   if (ev.xany.window == sbar.sb.win)
	     switch (ev.xbutton.button) {
	      case Button1: tty_printf ("\033[6~");	break;
	      case Button2: tty_printf ("\014");	break;
	      case Button3: tty_printf ("\033[5~");	break;
	     }
#ifndef SCROLLBAR_XTERM
	   else if (ev.xany.window == sbar.up.win) tty_printf ("\033[A");
	   else if (ev.xany.window == sbar.dn.win) tty_printf ("\033[B");
#endif	/* ! SCROLLBAR_XTERM */
	   return;
	}
#endif	/* NO_MOUSE_REPORT */
      if (ev.xany.window == sbar.sb.win)
	{
#ifndef SCROLLBAR_ARROWS
	   if (
#ifdef SCROLLBAR_ANY
	       (sbar.type == SBAR_XTERM) &&
#endif	/* SCROLLBAR_ANY */
	       ev.xbutton.button != Button2)
	     /*
	      * Move display proportional to location of the pointer
	      * If pointer is near bottom, scroll full page.
	      * If pointer is near top, scroll one line.
	      */
	     switch (ev.xbutton.button) {
	      case Button1:
		scr_page (DOWN, TermWin.nrow *
			  (double)(ev.xbutton.y) / (double)(sbar.h-1));
		break;
	      case Button3:
		scr_page (UP, TermWin.nrow *
			  (double)(ev.xbutton.y) / (double)(sbar.h-1));
		break;
	     }
	   else
#endif	/* ! SCROLLBAR_ARROWS */
	     scr_move_to (ev.xbutton.y);
	}
#ifndef SCROLLBAR_XTERM
      else if (ev.xany.window == sbar.dn.win)
	scr_page (DOWN, TermWin.nrow * PAGE_AMOUNT);
      else if (ev.xany.window == sbar.up.win)
	scr_page (UP, TermWin.nrow * PAGE_AMOUNT);
#endif	/* ! SCROLLBAR_XTERM */
      break;
#endif	/* SCROLLBAR_NONE */

    case ButtonRelease:
      if (ev.xany.window == TermWin.vt)
	{
	   if (ev.xbutton.subwindow != None)
	     Gr_ButtonRelease (ev.xbutton.x, ev.xbutton.y);
	   else
	     {
#ifndef NO_MOUSE_REPORT
		if (Mouse_Report && !(last_keystate))
		  {
		     switch (Mouse_Report) {
		      case X10_MOUSE:
			break;

		      case X11_MOUSE:
#ifndef NO_MULTIPLE_CLICK
			clicks = 0;
#endif
			ev.xbutton.state = last_keystate;
			ev.xbutton.button = AnyButton;
			mouse_report (&(ev.xbutton));
			break;

#ifdef X11_TRACK
		      case X11_TRACK:
			/* if start/end are  valid text locations:
			 * "ESC [ t CxCy"
			 * If either coordinate is past the end of the line:
			 * "ESC [ T CxCyCxCyCxCy"
			 * The parameters are startx, starty, endx, endy,
			 * mousex, and mousey */
			mouse_tracking (1, ev.xbutton.x, ev.xbutton.y, 0, 0);
			break;
#endif
		       }
		     return;
		  }
#endif

#ifndef NO_MOUSE_REPORT
		/*
		 * dumb hack to compensate for the failure of click-and-drag
		 * when overriding mouse reporting
		 */
		if (Mouse_Report && (ev.xbutton.state&(Mod1Mask|ShiftMask))
		    && (ev.xbutton.button == Button1)
#ifndef NO_MULTIPLE_CLICK
		    && (clicks <= 1)
#endif
		    )
		  scr_selection_extend (1, ev.xbutton.x, ev.xbutton.y);
#endif

		switch (ev.xbutton.button)
		  {
		   case Button1:
		   case Button3:
		     scr_selection_make (ev.xbutton.time);
		     break;

		   case Button2:
		     scr_selection_request (ev.xbutton.time,
					    ev.xbutton.x,
					    ev.xbutton.y);
		     break;
		  }
	     }
	}
      break;

    case MotionNotify:
#ifndef NO_MOUSE_REPORT
      if (Mouse_Report &&
#ifdef X11_TRACK
	  (Mouse_Report ^ X11_TRACK) &&
#endif
	  !(last_keystate))
	return;
#endif
      if (ev.xany.window == TermWin.vt)
	{
	   if ((ev.xbutton.state == Button1Mask ||
		ev.xbutton.state == Button3Mask)
#ifndef NOMULTIPLE_CLICK
	       && (clicks <= 1)
#endif
	       )
	     {
		while (XCheckTypedWindowEvent (Xdisplay, TermWin.vt,
					       MotionNotify, &ev));
		XQueryPointer (Xdisplay, TermWin.vt, &root, &child,
			      &root_x, &root_y, &x, &y, &mods);
#ifdef MOUSE_THRESHOLD
		/* deal with a `jumpy' mouse */
		if ((ev.xmotion.time - buttonpress_time) > MOUSE_THRESHOLD)
#endif
		  {
#ifdef X11_TRACK
		     if (Mouse_Report & X11_TRACK)
		       {
			  if (Mouse_Report & X11_HILITE)
			    scr_selection_extend (1, x, y);
			  else
			    scr_selection_extend (0, x, y);
		       }
		     else
#endif
		       scr_selection_extend (1, x, y);
		  }

	     }
	}
#ifndef SCROLLBAR_NONE
      else if ((ev.xany.window == sbar.sb.win) &&
	       (ev.xbutton.state&(Button1Mask|Button2Mask|Button3Mask)))
	{
	   while (XCheckTypedWindowEvent (Xdisplay, sbar.sb.win,
					  MotionNotify, &ev));
	   XQueryPointer (Xdisplay, sbar.sb.win, &root, &child,
			  &root_x, &root_y, &x, &y, &mods);
#ifndef SCROLLBAR_ARROWS
	   if (
#ifdef SCROLLBAR_ANY
	       (sbar.type == SBAR_ARROWS) ||
#endif	/* ANY */
	       (ev.xbutton.state & Button2Mask)
	       )
#endif	/* ! ARROWS */
	     {
		scr_move_to (y);
		scr_refresh (refresh_type);
		refresh_nl_count = refresh_nl_limit = 0;
		sbar_show (TermWin.hist_count + (TermWin.nrow-1),
			   TermWin.hist_start,
			   (TermWin.nrow-1));
	     }
	}
#endif	/* SCROLLBAR_NONE */
      break;
   }
}

/*
 * Send count characters directly to the command
 */
void
tty_write (unsigned char *buf, int count)
{
   while (count > 0)
     {
	int n = write (comm_fd, buf, count);
	if (n > 0)
	  {
	     count -= n;
	     buf += n;
	  }
     }
}

/*
 * Send printf() formatted output to the command.
 * Only used for small ammounts of data.
 */
void
tty_printf (unsigned char *fmt, ...)
{
   static unsigned char buf [256];
   va_list arg_ptr;

   va_start (arg_ptr, fmt);
   vsprintf (buf, fmt, arg_ptr);
   va_end (arg_ptr);
   tty_write (buf, strlen (buf));
}

/*----------------------------------------------------------------------*/
#ifdef PRINTPIPE
FILE *
popen_printer (void)
{
   FILE *stream = popen (rs_print_pipe, "w");
   if (stream == NULL)
     print_error ("can't open printer pipe");
   return stream;
}

int
pclose_printer (FILE *stream)
{
   fflush (stream);
   /* pclose() reported not to work on SunOS 4.1.3 */
   return pclose (stream);
}

/*
 * simulate attached vt100 printer
 */
static void
process_print_pipe (void)
{
   const char * escape_seq = "\033[4i";
   const char * rev_escape_seq = "i4[\033";
   int c, index;
   FILE *fd;

   fd = popen_printer ();
   if (fd == NULL)
     return;

   for (index = 0; index < 4; /* nil*/)
     {
	c = get_com_char ();

	if (c == escape_seq [index]) index++;
	else if (index)
	  for (/*nil*/; index > 0; index--)
	  fputc (rev_escape_seq [index-1], fd);

	if (index == 0) fputc (c, fd);
     }
   pclose_printer (fd);
}
#endif

/*
 * process escape sequences
 */
static void
process_escape_seq (void)
{
   switch (get_com_char ()) {
#if 0
    case 1:	do_tek_mode ();	break;
#endif
    case '#':	if ('8' == get_com_char ()) scr_E ();	break;
    case '(':	scr_charset_set (0, get_com_char ());	break;
    case ')':	scr_charset_set (1, get_com_char ());	break;
    case '*':	scr_charset_set (2, get_com_char ());	break;
    case '+':	scr_charset_set (3, get_com_char ());	break;
#ifdef KANJI
    case '$':	scr_charset_set (-2, get_com_char ());	break;
#endif
    case '7':	scr_cursor (SAVE);	break;
    case '8':	scr_cursor (RESTORE);	break;
    case '=':	apl_KP = 1;		break;
    case '>':	apl_KP = 0;		break;
    case '@':	(void)get_com_char ();	break;
    case 'D':	scr_index (UP);		break;
    case 'E':	scr_add_lines ("\n\r", 1, 2);	break;
    case 'G':	process_rxvt_graphics ();	break;
    case 'H':	scr_set_tab (1);	break;
    case 'M':	scr_index (DOWN);	break;
    /*case 'N':	scr_single_shift (2);	break;*/
    /*case 'O':	scr_single_shift (3);	break;*/
    case 'Z':	tty_printf (RXVT_ANS);	break;	/* steal obsolete ESC [ c */
    case '[':	process_csi_seq ();	break;
    case ']':	process_xterm_seq ();	break;
    case 'c':	scr_poweron ();		break;
    case 'n':	scr_charset_choose (2);	break;
    case 'o':	scr_charset_choose (3);	break;
   }
}

/*
 * process CSI (code sequence introducer) sequences `ESC [ '
 */
static void
process_csi_seq (void)
{
   unsigned char c, priv = 0;
   int nargs, arg [ESC_ARGS];

   nargs = 0;
   arg [0] = 0;
   arg [1] = 0;

   c = get_com_char ();
   if (c >= '<' && c <= '?')
     {
	priv = c;
	c = get_com_char ();
     }

   /* read any numerical arguments */
   do
     {
	int n;
	for (n = 0; isdigit (c); c = get_com_char ())
	  n = n * 10 + (c - '0');

	if (nargs < ESC_ARGS)
	  arg [nargs++] = n;
	if (c == '\b')
	  {
	     scr_backspace ();
	  }
	else if (c == 033)
	  {
	     process_escape_seq ();
	     return;
	  }
     	else if (c < ' ')
	  {
	     scr_add_lines (&c, 0, 1);
	     return;
	  }

	if (c < '@')
	  c = get_com_char ();
     }
   while (c >= ' ' && c < '@');
   if (c == 033)
     {
	process_escape_seq ();
	return;
     }
   else if (c < ' ')
     return;

   switch (c) {
#ifdef PRINTPIPE
    case 'i':			/* printing */
      switch (arg[0]) {
       case 0: scr_printscreen (0);		break;
       case 5: process_print_pipe ();		break;
      }
      break;
#endif
    case 'A': case 'e':		/* up <n> */
      scr_gotorc ((arg[0] ? -arg[0] : -1), 0, RELATIVE);	break;
    case 'B':			/* down <n> */
      scr_gotorc ((arg[0] ? +arg[0] : +1), 0, RELATIVE);	break;
    case 'C': case 'a':		/* right <n> */
      scr_gotorc (0, (arg[0] ? +arg[0] : +1), RELATIVE);	break;
    case 'D':			/* left <n> */
      scr_gotorc (0, (arg[0] ? -arg[0] : -1), RELATIVE);	break;
    case 'E':			/* down <n> & to first column */
      scr_gotorc ((arg[0] ? +arg[0] : +1), 0, R_RELATIVE);	break;
    case 'F':			/* up <n> & to first column */
      scr_gotorc ((arg[0] ? -arg[0] : -1), 0, R_RELATIVE);	break;
    case 'G': case '`':		/* move to col <n> */
      scr_gotorc (0, (arg[0] ? +arg[0] : +1), R_RELATIVE);	break;
    case 'd':			/* move to row <n> */
      scr_gotorc ((arg[0] ? +arg[0] : +1), 0, C_RELATIVE);	break;
    case 'H': case 'f':		/* position cursor */
      switch (nargs) {
       case 0: scr_gotorc (0, 0, 0); break;
       case 1: scr_gotorc ((arg[0] ? arg[0]-1 : 0), 0, 0);	break;
       default: scr_gotorc (arg[0]-1, arg[1]-1, 0);	break;
      }
      break;
    case 'I':	scr_tab (arg[0] ? +arg [0] : +1);	break;
    case 'Z':	scr_tab (arg[0] ? -arg [0] : -1);	break;
    case 'J':	scr_erase_screen (arg[0]);	break;
    case 'K':	scr_erase_line (arg[0]);	break;
    case '@':	scr_insdel_chars ((arg[0] ? arg [0] : 1), INSERT);	break;
    case 'L':	scr_insdel_lines ((arg[0] ? arg [0] : 1), INSERT);	break;
    case 'M':	scr_insdel_lines ((arg[0] ? arg [0] : 1), DELETE);	break;
    case 'X':	scr_insdel_chars ((arg[0] ? arg [0] : 1), ERASE);	break;
    case 'P':	scr_insdel_chars ((arg[0] ? arg [0] : 1), DELETE);	break;

    case 'c':	tty_printf (VT100_ANS);		break;
    case 'm':	process_sgr_mode (nargs, arg);	break;
    case 'n':			/* request for information */
      switch (arg[0]) {
       case 5: tty_printf ("\033[0n");		break;	/* ready */
       case 6: scr_report_position ();			break;
#if defined (ENABLE_DISPLAY_ANSWER) || defined (DISPLAY_ENV_AND_ANSWER_IS_IP)
       case 7: tty_printf ("%s\n", display_name);	break;
#endif
       case 8:
	 xterm_seq (NEW_TITLE_NAME, APL_NAME "-" VERSION);
	 break;
      }
      break;
    case 'r':			/* set top and bottom margins */
      /* what's this about? something to do with vi on ESIX systems */
      if (priv != '?')
	{
	   if (nargs < 2 || arg[0] >= arg[1])
	     scr_scroll_region (0, 10000);
	   else
	     scr_scroll_region (arg[0]-1, arg[1]-1);
	   break;
	}
      /* drop */
    case 's':
    case 'h':
    case 'l':
      process_terminal_mode (c, priv, nargs, arg);
      break;
    case 'g':
      switch (arg[0]) {
       case 0: scr_set_tab (0);		break;	/* delete tab */
       case 3: scr_set_tab (-1);	break;	/* clear all tabs */
      }
      break;
    case 'W':
      switch (arg [0]) {
       case 0: scr_set_tab (1);		break;	/* = ESC H */
       case 2: scr_set_tab (0);		break;	/* = ESC [ 0 g */
       case 5: scr_set_tab (-1);	break;	/* = ESC [ 3 g */
      }
      break;
   }
}

/*
 * process xterm text parameters sequences
 * `ESC ] Ps ; Pt BEL'
 */
static void
process_xterm_seq (void)
{
   unsigned char c, string [STRING_MAX];
   int n, arg;

   c = get_com_char ();
   for (n = 0; isdigit (c); c = get_com_char ())
     n = n * 10 + (c - '0');
   arg = n;

   n = 0;
   while ((c = get_com_char ()) != 007)
     if ((n < sizeof(string)-1) && (c >= ' '))
       string [n++] = c;

   string [n] = '\0';
   xterm_seq (arg, string);
}

/*
 * process DEC private mode sequences `ESC [ ? Ps mode'
 *
 * mode can only have the following values:
 *	'l' = low
 *	'h' = high
 *	's' = save
 *	'r' = restore
 * so no need for fancy checking
 */
static void
process_terminal_mode (unsigned char mode, unsigned char priv,
		       int nargs, int arg[])
{
   /* current and saved mode values */
   static unsigned char apl_CUR_s = 0, apl_KP_s = 0;
   static unsigned char mode132ok = 1, mode132ok_s = 1;
   static unsigned char mode132 = 0, mode132_s = 0;
   static unsigned char rvideo = 0, rvideo_s = 0;
   static unsigned char rel_origin = 0, rel_origin_s = 0;
   static unsigned char autowrap = 1, autowrap_s = 1;
   static unsigned char screen = 0, screen_s = 0;
#ifndef NO_MOUSE_REPORT
   static unsigned char mouse_s = 0;
#endif
   int i;

   /* make lo/hi boolean */
   switch (mode) {
    case 'l': mode = 0; break;
    case 'h': mode = 1; break;
   }

   if (nargs == 0) return;

   switch (priv) {
    case 0:
      for (i = 0; i < nargs; i++)
	switch (arg[i]) {
	 case 4:  scr_insert_mode (mode); break;
	 case 36: real_bs = mode; break;
	   /* case 38:	TEK mode */
	}
      break;
    case '?':
      for (i = 0; i < nargs; i++)
	switch (arg[i]) {
	 case 1:			/* application cursor keys */
	   switch (mode) {
	    case 's': apl_CUR_s = apl_CUR; break;
	    case 'r': mode = apl_CUR_s; /*drop*/
	    default:  apl_CUR = mode;
	   }
	   break;

	 case 3:			/* 80/132 */
	   switch (mode) {
	    case 's': mode132_s = mode132; break;
	    case 'r': mode = mode132_s;	/*drop*/
	    default:  mode132 = mode;
	      if (mode132ok)
		set_width (mode ? 132 : 80);
	   }
	   break;

	   /* case 4:	- smooth scrolling */
	 case 5:
	   switch (mode) {
	    case 's': rvideo_s = rvideo; break;
	    case 'r': mode = rvideo_s;	/*drop*/
	    default:  rvideo = mode; scr_rvideo_mode (mode);
	   }
	   break;

	 case 6:
	   switch (mode) {
	    case 's': rel_origin_s = rel_origin; break;
	    case 'r': mode = rel_origin_s;	/*drop*/
	    default:  rel_origin = mode; scr_relative_origin (mode);
	   }
	   break;

	 case 7:
	   switch (mode) {
	    case 's': autowrap_s = autowrap; break;
	    case 'r': mode = autowrap_s;	/*drop*/
	    default:  autowrap = mode; scr_autowrap (mode);
	   }
	   break;
	   /* case 8:	- auto repeat, can't do on a per window basis */
#ifndef NO_MOUSE_REPORT
	 case 9:			/* X10 mouse reporting */
	   switch (mode) {
	    case 's': mouse_s = (Mouse_Report & X10_MOUSE); break;
	    case 'r': mode = (mouse_s & X10_MOUSE);	/*drop*/
	    default:
	      if (mode)
		Mouse_Report = X10_MOUSE;
	      else
		Mouse_Report &= ~X10_MOUSE;
	   }
	   break;

	 case 40:			/* 80 <--> 132 mode */
	   switch (mode) {
	    case 's': mode132ok_s = mode132ok; break;
	    case 'r': mode = mode132ok_s;	/*drop*/
	    default:  mode132ok = mode;
	   }
	   break;

#endif	/* NO_MOUSE_REPORT */
	 case 47:
	   switch (mode) {
	    case 's': screen_s = screen; break;
	    case 'r': mode = screen_s;	/*drop*/
	    default:  screen = mode; scr_change_screen (mode);
	   }
	 case 66:			/* application key pad */
	   switch (mode) {
	    case 's': apl_KP_s = apl_KP; break;
	    case 'r': mode = apl_KP_s;	/*drop*/
	    default:  apl_KP = mode;
	   }
	   break;

#ifndef NO_MOUSE_REPORT
	 case 1000:		/* X11 mouse reporting */
	   switch (mode) {
	    case 's': mouse_s = (Mouse_Report & X11_MOUSE); break;
	    case 'r': mode = (mouse_s & X11_MOUSE); /*drop*/
	    default:
	      if (mode)
		Mouse_Report = X11_MOUSE;
	      else
		Mouse_Report &= ~X11_MOUSE;
	   }
	   break;

	 case 1001:		/* X11 mouse highlighting */
#ifdef X11_TRACK		/* not yet! */
	   switch (mode) {
	    case 's': mouse_s = (Mouse_Report & X11_TRACK); break;
	    case 'r': mode = (mouse_s & X11_TRACK);		/*drop*/
	    default:
	      if (mode)
		Mouse_Report = X11_TRACK;
	      else
		Mouse_Report &= ~X11_TRACK;
	   }
#endif	/* X11_TRACK */
	   break;
#endif	/* NO_MOUSE_REPORT */
	}
      break;
   }
}

/*
 * process sgr sequences
 */
static void
process_sgr_mode (int nargs, int arg[])
{
   int i;

   if (nargs == 0)
     {
	scr_rendition (0, ~RS_NONE);
	return;
     }
   for (i = 0; i < nargs; i++)
     switch (arg [i]) {
      case 0:	scr_rendition (0, ~RS_NONE);	break;
      case 1:	scr_rendition (1, RS_BOLD);	break;
      case 4:	scr_rendition (1, RS_ULINE);	break;
      case 5:	scr_rendition (1, RS_BLINK);	break;
      case 7:	scr_rendition (1, RS_RVID);	break;
      case 22:	scr_rendition (0, RS_BOLD);	break;
      case 24:	scr_rendition (0, RS_ULINE);	break;
      case 25:	scr_rendition (0, RS_BLINK);	break;
      case 27:	scr_rendition (0, RS_RVID);	break;
      case 30: case 31:		/* set fg color */
      case 32: case 33:
      case 34: case 35:
      case 36: case 37:
	scr_fgcolor (arg [i] - 30);
	break;
      case 39:			/* set default fg color */
	scr_fgcolor (RESTORE_FG);
	break;
      case 40: case 41:		/* set bg color */
      case 42: case 43:
      case 44: case 45:
      case 46: case 47:
	scr_bgcolor (arg [i] - 40);
	break;
      case 49:			/* set default bg color */
	scr_bgcolor (RESTORE_BG);
	break;
     }
}

/*
 * process Rob Nation's own graphics mode sequences
 */
static void
process_rxvt_graphics (void)
{
   unsigned char c, cmd = get_com_char ();
#ifndef RXVT_GRAPHICS
   if (cmd == 'Q')			/* query graphics */
     {
	tty_printf ("\033G0\n");	/* no graphics */
	return;
     }
   /* swallow other graphics sequences until terminating ':' */
   do c = get_com_char (); while (c != ':');
#else
   int nargs;
   long args [NGRX_PTS];
   unsigned char *text = NULL;

   if (cmd == 'Q')			/* query graphics */
     {
	tty_printf ("\033G1\n");	/* yes, graphics (color) */
	return;
     }

   for (nargs = 0; nargs < (sizeof(args)/sizeof(args[0]))-1; /*nil*/)
     {
	int neg;

	c = get_com_char ();
	neg = (c == '-');
	if (neg || c == '+')
	  c = get_com_char ();

	for (args [nargs] = 0; isdigit (c); c = get_com_char ())
	  args [nargs] = args [nargs] * 10 + (c - '0');
	if (neg) args [nargs] = -args[nargs];

	nargs++;
	args [nargs] = 0;
	if (c !=  ';')
	  break;
     }

   if ((cmd == 'T') && (nargs >= 5))
     {
	int i, len = args [4];

	text = MALLOC ((len + 1) * sizeof(char), "text");
	if (text != NULL)
	  {
	     for (i = 0; i < len; i++)
	       text [i] = get_com_char ();
	     text [len] = '\0';
	  }
     }

   Gr_do_graphics (cmd, nargs, args, text);
#endif
}

/*
 * Read and process output from the application
 */
void
main_loop (void)
{
   int c;

   /* determine appropriate value for Backspace key */
   real_bs = (CERASE == '\b');

   do {
      while ((c = get_com_char ()) == 0);	/* wait for something */
      if (c >= ' ' || c == '\t' || c == '\n' || c == '\r') /* (isprint (c)) */
	{
	   /* Read a text string from the input buffer */
	   unsigned char *str;
	   int nl_count = 0;

	   inbuf_next--;	/* decr, already did get_com_char () */
	   str = inbuf_next;	/* point to the start of the string */

	   while (inbuf_next < inbuf_top)
	     {
		c = *inbuf_next;
		if (c >= ' ' || c == '\t' || c == '\n' || c == '\r') /* (isprint (c))*/
		  {
		     inbuf_next++;
		     if (c == '\n')
		       {
			  nl_count++;
			  refresh_nl_count++;

			  if (refresh_nl_count >
			      refresh_nl_limit * TermWin.nrow)
			    break;
		       }
		  }
		else			/* unprintable */
		  {
		     break;
		  }
	     }
	   scr_add_lines (str, nl_count, inbuf_next - str);
	}
      else
	{
	   switch (c) {
	    case 005: tty_printf (VT100_ANS);	break;	/* terminal Status */
	    case 007: scr_bell ();	break;		/* bell */
	    case '\b': scr_backspace ();	break;	/* backspace */
	    case 013:
	    case 014: scr_index (1);	break;	/* vertical tab, form feed */
	    case 016: scr_charset_choose (1);	break;	/* shift out - acs */
	    case 017: scr_charset_choose (0);	break;	/* shift in - acs */
	    case 033: process_escape_seq ();	break;
	   }
	}
   } while (c != EOF);
}
/*----------------------- end-of-file (C source) -----------------------*/
