/*
 * move_prob.c --
 * Chris Hyde
 * created: 
 *
 * Modified: Dean Collins Sat Feb 12 13:46:12 1994
 * Added unistd.h inclusion for SunOS as requested by
 * magnush@isy.liu.se (Magnus Hammerin).
 * Also fixed several typos.
 *
 * Modified: Dean Collins Wed Feb 23 17:00:42 1994
 * Changed PTSROOT (a constant) to PtsRoot (a variable).
 *
 * Modified: Chris Hyde 1/10/95
 * Removed include <stdio.h>, since it's now included in zdbm.h.
 * Made changes to I/O to reflect zdbm names (zread(), zwrite(), etc.)
 * Replaced direct references to the index records with functions/macros
 *   defined for that purpose.
 *
 * Modified: Dean Collins Thu Jun  8 13:31:27 PDT 1995
 * Replaced #ifdef DEBUG method of debug output with calls to
 * the zdebug macros.
 *
 */

/*
 * Copyright (c) 1992,1993,1994,1995 Christopher G. Hyde
 * Copyright (c) 1992 University of Idaho, Moscow, Idaho.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation free of charge for any purpose is hereby granted without
 * fee, provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the University of Idaho
 * not be used in advertising or publicity pertaining to distribution of
 * the software without specific, written prior permission.  The University
 * of Idaho makes no representations about the suitability of this software
 * for any purpose.  It is provided "as is" without express or implied warranty.
 *
 * THE UNIVERSITY OF IDAHO DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 * SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS.
 * IN NO EVENT SHALL THE UNIVERSITY OF IDAHO BE LIABLE FOR ANY SPECIAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
 * OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */

#include <string.h>
#include <errno.h>
#ifdef SUNOS
#include <unistd.h>
#endif /*SUNOS*/

#include "zdbm.h"

int move_problem(char *old_db_path, char *new_db_path, char *prid)
{
 index_record index;
 char *tprid;
 ZFILE *temp;

 zdebug("\nIn move_problem()\n");
 zdebug2("\tOld DB path - %s\n\tNew DB path - %s\n", old_db_path, new_db_path);
 zdebug1("\tPRID - %s\n", prid);

 db_errorno=0;

 if (strcmp(old_db_path, new_db_path)==0)
 {  /* can't move a problem to the same leaf */
    db_errorno=SAMELEAF;
    return(FAIL);
 }

 if ((temp=zopen(old_db_path, ".index", "r+", INDEX))==NULL)
    return(FAIL);

 if (find_problem(temp, prid, &index)==NULL)
 {  /* error tring to find the problem to move */
    zclose(temp);
    return(FAIL);
 }

 if (is_in_edit(&index)) /* the problem is currently being edited */
 {
    db_errorno=ETXTBSY; /* the problem is locked as far as the user is
             * concerned */
    zclose(temp);
    return(FAIL);
 }

 if (is_gone(&index))
 {  /* can't move a deleted problem */
    db_errorno=DELETED;
    zclose(temp);
    return(FAIL);
 }

 if (is_linked(&index))
    if (strcmp(new_db_path, get_ln_path(&index))==0)
    {   /* can't move problem to the leaf that holds original link */
       db_errorno=SAMELEAF;
       zclose(temp);
       return(FAIL);
    }


 zdebug("move_problem() - Found problem to move, now begin moving it\n");

 if (zlock(temp)==FAIL)
 {  /* the lock failed */
    zclose(temp);
    return(FAIL);
 }

/* now physically move the problem requested */
 if ((tprid=move_record(old_db_path, new_db_path, &index))==NULL)
 {  /* the move failed */
    zunlock(temp);
    zclose(temp);
    return(FAIL);
 }

 zdebug("move_problem() - The problem has been moved, now mark original problem\n\t as deleted\n");

/* now move back to the begining of the index entry that was moved */
 zseek(temp, -(long)(sizeof(index_record)), SEEK_CUR);

 if (is_org_link(&index) || is_reopen_prob(&index)) 
 { /* the problem to move is either an original link or has been requested
    * for reopening so set up index entry so the move will travel down the 
    * links */
    set_moved(&index);
    set_ln_path(&index, new_db_path, DBPATHLEN);
    set_ln_prid(&index, tprid, MAXPRID);
 }
 else /* the problem either resides here or is just a plain link */
    set_gone(&index); /* so set the index entry to reflect the move */

/* now write out the updated index entry */
 if (zwrite(&index, sizeof(index_record), 1, temp)<1)
 {  /* the write failed */
    db_errorno=errno;
    zunlock(temp);
    zclose(temp);
    return(FAIL);
 }
 zunlock(temp);
 zclose(temp);
 return(SUCESS);
} /* end move_problem */

 
char *move_record(char *old_db_path, char *new_db_path, index_record *index)
{
 char tprid[MAXPRID];
 char *tptr;
 int tmrd;
 long size;
 ZFILE *temp;

 zdebug("\nIn move_record()\n");
 zdebug2("\tOld DB path - %s\n\tNew DB path - %s\n", old_db_path, new_db_path);

 tmrd=get_mrd(index); /* store the old value of this field for replacement */
 nl_strncpy(tprid, get_prid(index), MAXPRID);

/* we need to open the index for write rather than append, since we overwrite
 * the index entry when finished.
 */
 if ((temp=zopen(new_db_path, ".index", "w", INDEX))==NULL)
    if (errno==ENOENT) /* the index does not exist */
       if (make_branch(new_db_path)==FAIL) /* so create it */
          return(NULL); /* there was a problem creating the new index */
       else /* we  created the index, so open it for writing */
          if ((temp=zopen(new_db_path, ".index", "w", INDEX))==NULL)
              return(NULL);
          else; /* dummy else to go with if above */
    else /* we couldn't open the file for some other reason, so quit */
       return(NULL);

/* we want to append to the new index, so seek to it. */
 zseek(temp, 0L, SEEK_END);

 if (zlock(temp)==FAIL)/* lock the new index before making the new record */ {  /* the lock failed */
    zclose(temp);
    return(NULL);
 }
 if (is_org_link(index)) /* check to see if a link already exists in the
              * new leaf */
    if (double_problem(temp, old_db_path, get_prid(index)))
    { /* the problem already exists in the new leaf */
       db_errorno=SAMELEAF;
       zclose(temp);
       return(FAIL);
    }

 if (is_linked(index)) /* check to see if a link already exists in the
             * new leaf */
    if (double_problem(temp, get_ln_path(index), get_ln_prid(index)))
    { /* the problem already exists in the new leaf */
       db_errorno=SAMELEAF;
       zclose(temp);
       return(FAIL);
    }

/* get the size of the index file, in records */
 if ((size=file_size(temp, sizeof(index_record)))==ERROR)
 {  /* there was an error getting the file size */
    zunlock(temp);
    zclose(temp);
    return(NULL);
 }

/* now create a new problem id and stuff it in the index entry */
 if ((tptr=new_prid(size))==NULL)
 {  /* an error occured getting the new prid */
    zunlock(temp);
    zclose(temp);
    return(NULL);
 }
 set_prid(index, tptr, MAXPRID);
 set_gone(index); /* will keep the new entry from being read if the move
              * fails */


/* now write out the new index entry */
 if (zwrite(index, sizeof(index_record), 1, temp)<1)
 {  /* the write failed, so restore the index entry before returning */
    db_errorno=errno;
    set_mrd(index, tmrd);
    set_prid(index, tprid, MAXPRID);
    zunlock(temp);
    zclose(temp);
    return(NULL);
 }
 zunlock(temp); /* we don't need to keep the index locked */

 zdebug("move_record() - Created new index entry, and wrote it to the new index\n");
 zdebug("\tNow move over the associated files\n");

/* now move the files over to the new leaf */
 if (exists(PtsRoot, old_db_path, tprid, ".rep")) /* if the reporters file
                               * exists then copy it over */
    if (copy_file(old_db_path, tprid, ".rep", new_db_path, tptr, ".rep")==FAIL)
    {   /* the copy failed, so restore the index entry before returning */
       set_mrd(index, tmrd);
       set_prid(index, tprid, MAXPRID);
       zclose(temp);
       return(NULL);
    }

 if (exists(PtsRoot, old_db_path, tprid, ".rep_old")) /* if the rep_old file
                          * exists, copy it over to the 
                          * new leaf */
    if (copy_file(old_db_path, tprid, ".rep_old",
           new_db_path, tptr, ".rep_old")==FAIL)
    {   /* the copy failed, so restore the index entry before returning */
       set_mrd(index, tmrd);
       set_prid(index, tprid, MAXPRID);
       zclose(temp);
       remove_file(new_db_path, tptr, ".rep");
       return(NULL);
    }

 if (exists(PtsRoot, old_db_path, tprid, ".log")) /* if the log file exists,
                           * copy it over to the new leaf */
    if (copy_file(old_db_path, tprid, ".log", 
          new_db_path, tptr, ".log")==FAIL)
    {   /* the copy failed, so restore the index entry before returning */
       set_mrd(index, tmrd);
       set_prid(index, tprid, MAXPRID);
       zclose(temp);
       remove_file(new_db_path, tptr, ".rep");
       remove_file(new_db_path, tptr, ".rep_old");
       return(NULL);
    }

 zdebug("move_record() - The files were moved sucessfully, now delete the originals\n");

/* the file moves succeeded so mark the new index entry as before and overwrite
 * the old one, remove the old files and return the old index entry to its
 * original values 
 */
 zseek(temp, -(long)(sizeof(index_record)), SEEK_CUR);
 set_mrd(index, tmrd);
 if (zwrite(index, sizeof(index_record), 1, temp)<1)
 {   /* the write failed, so restore index entry and remove the cpoied files */
    db_errorno=errno;
    zclose(temp);
    set_prid(index, tprid, MAXPRID);
    remove_file(new_db_path, tptr, ".rep");
    remove_file(new_db_path, tptr, ".rep_old");
    remove_file(new_db_path, tptr, ".log");
    return(NULL);
 }
/* else everything worked so restore the index entry and remove the original
 * files
 */
 zclose(temp);
 set_prid(index, tprid, MAXPRID);
 remove_file(old_db_path, tprid, ".rep");
 remove_file(old_db_path, tprid, ".rep_old");
 remove_file(old_db_path, tprid, ".log");
 return(tptr);
} /* end move_record */

 
int link_problem(char *old_db_path, char *new_db_path, char *prid)
{
 index_record index,
          new_index;
 long size;
 ZFILE *temp;
 char *tptr;

  zdebug("\nIn link_problem()\n");
  zdebug1("\told_db_path - %s\n", old_db_path);
  zdebug2("\tnew_db_path - %s\n\tprid - %s\n", new_db_path, prid);

 db_errorno=0;

 if (strcmp(old_db_path, new_db_path)==0)
 {  /* can't link a problem to the same leaf */
    db_errorno=SAMELEAF;
    return(FAIL);
 }

 if ((temp=zopen(old_db_path, ".index", "r+", INDEX))==NULL)
    return(FAIL);

 if (find_problem(temp, prid, &index)==NULL)
 {  /* could not find problem requested */
     zclose(temp);
     return(FAIL);
 }

  zdebug("link_problem() - Found problem desired\n");
  zdebug("\tChecking problem found.\n");

 if (is_gone(&index)) /* can't link a deleted problem */
 {
    db_errorno=DELETED;
    zclose(temp);
    return(FAIL);
 }

 if (is_linked(&index))/* problem does not reside in leaf given so set up the */
 {  /* new index record using the original problem as the problem that
     * will be linked to
     */
    if (strcmp(new_db_path, get_ln_path(&index))==0)
    {   /* can't link to a leaf in which the problem already resides */
       db_errorno=SAMELEAF;
       zclose(temp);
       return(FAIL);
    }

    zdebug("link_problem() - Problem is linked, and is OK for linking\n");

    set_ln_path(&new_index, get_ln_path(&index), DBPATHLEN);
    set_ln_prid(&new_index, get_ln_prid(&index), MAXPRID);
    zclose(temp);
 } /* end if LINKED */
 else /* problem resides in the leaf given, so set up the new index record */
 {    /* using the problem from this leaf. If the problem is already an original
       * link then don't change the original index entry, otherwise this is the
       * first link to this problem so mark it as the original link in a series
       * of links and update the index entry
       */

    zdebug("link_problem() - Problem is not linked, and is OK for linking\n");

    if (is_org_link(&index))
    {
       set_org_link(&index);
       zseek(temp, -(long)(sizeof(index_record)), SEEK_CUR);
       if (zlock(temp)==FAIL)
       {    /* the lock failed */
          zclose(temp);
          return(FAIL);
       }
       if (zwrite(&index, sizeof(index_record), 1, temp)<1)
       {    /* error writing to file */
          db_errorno=errno;
          zunlock(temp);
          zclose(temp);
          return(FAIL);
       }
    } /* end not ORGLINK */
/* else the update succeed so set up the new index record */
    zunlock(temp);
    zclose(temp);
    set_ln_path(&new_index, old_db_path, DBPATHLEN);
    set_ln_prid(&new_index, get_prid(&index), MAXPRID);
 } /* end else problem resides in this leaf */

/* finish assigning values to the new index record */
 set_linked(&new_index); /* mark new problem as liked */
 set_sf(&new_index, get_sf(&index));
 set_edit(&new_index, get_edit(&index));
 set_account_name(&new_index, get_acc_name(&index), MAXANAMELEN);
 set_node_name(&new_index, get_node_name(&index), MAXHOSTNAMELEN);
 set_pid(&new_index, get_pid(&index));
 set_sd(&new_index, get_sd(&index), MAXSUMLEN);
 set_fr(&new_index, get_fr(&index), PADLEN);

 zdebug("link_problem() - Adding new index entry\n");

/* add new index entry to the new index */
 if ((temp=zopen(new_db_path, ".index", "a", INDEX))==NULL)
    if (errno==ENOENT) /* the index does not exist */
       if (make_branch(new_db_path)==FAIL) /* so create it */
          return(0); /* there was a problem creating the new index */
       else /* we  created the index, so open it for update */
          if ((temp=zopen(new_db_path, ".index", "a", INDEX))==NULL)
              return(0);
          else; /* dummy else to go with if above */
    else /* we couldn't open the file for some other reason, so quit */
       return(0);

 if (zlock(temp)==FAIL)
 {  /* the lock failed */
    zclose(temp);
    return(FAIL);
 }
 if (is_org_link(&index)) /* check the new leaf to see if the problem is
              * already in the new leaf */
    if (double_problem(temp, old_db_path, get_prid(&index)))
    { /* problem is already in new leaf */
       db_errorno=SAMELEAF;
       zclose(temp);
       return(FAIL);
    }
 if (is_linked(&index)) /* check the new leaf to see if the problem is
             * already in the new leaf */
    if (double_problem(temp, get_ln_path(&index), get_ln_prid(&index)))
    { /* problem is already in new leaf */
       db_errorno=SAMELEAF;
       zclose(temp);
       return(FAIL);
    }

 if ((size=file_size(temp, sizeof(index_record)))==ERROR)
 {  /* could not get file size */
    zunlock(temp);
    zclose(temp);
    return(FAIL);
 }
 if ((tptr=new_prid(size))==NULL)
 {  /* error finding new prid */
    zunlock(temp);
    zclose(temp);
    return(FAIL);
 }
 set_prid(&new_index, tptr, MAXPRID);

 if (zwrite(&new_index, sizeof(index_record), 1, temp)<1)
 {  /* error writing the new record */
    db_errorno=errno;
    zunlock(temp);
    zclose(temp);
    return(FAIL);
 }
/* else the problem was written OK so signal sucess */

 zdebug1("link_problem() - New prid of problem - %s\n", tptr);

 zunlock(temp);
 zclose(temp);
 return(SUCESS);
} /* end link_problem */

/* end of move_prob.c */
