/* web.c --  The Problem Tracking System Web (HTTP/WAIS) setup utility.
 *           This pre-formats all the problem logs into HTML files ready
 *           for building the WAIS index.
 *
 * Author: Dean Collins
 *
 */

/*
 * Copyright (c) 1995 Dean Collins.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation free of charge for any purpose is hereby granted without
 * fee, provided that the above copyright notices appear in all copies and
 * that both those copyright notices and this permission notice appear in
 * supporting documentation, and that the name of Dean Collins not
 * be used in advertising or publicity pertaining to distribution of
 * the software without specific, written prior permission.  Dean Collins
 * makes no representations about the suitability of this software for
 * any purpose.  It is provided "as is" without express or implied warranty.
 *
 * DEAN COLLINS DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS.
 * IN NO EVENT SHALL DEAN COLLINS BE LIABLE FOR ANY SPECIAL, INDIRECT
 * OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
 * WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdlib.h>		/* General C utilities header file */
#include <stdio.h>		/* Standard input/output utilities hdr. file */
#include <errno.h>		/* Error utilities header file */
#ifndef SYSV
#include <sys/types.h>		/* Common data types */
#endif
#include <sys/stat.h>		/* File status utilities */
#include "pts.h"
#include "zdbm.h"		/* Zombie Database Manager header file */
#include "cloud.h"		/* Nebulous Cloud header file */
#include "clouderror.h"		/* Nebulous Cloud error rtn. header file */
#include "config.h"		/* Config. file parser header file */
#include "html.h"		/* A couple of HTML-related routines */
#include "fmtprobhtml.h"	/* Hdr for FormatProblemHTML() */


/* FUNCTION PROTOTYPES */
int main(int argc, char *argv[]) ;
int
DoSummaryList( Summary     *Summaries,
               ProblemTree *ProbTree,
               char        *HTMLindex,
               char        *HTMLindexdir,
               int         *SolvedCount,
               int         *UnsolvedCount) ;
int mkdirhier(char *p) ;


/******************** MODULES *************************************/

/*
______________________________________________________________________
main()

FUNCTION DESCRIPTION:

	This is the main function for the PTS WAIS utility.
______________________________________________________________________
REVISION HISTORY:
     Author:  Dean Collins                     Date: 3/6/95
______________________________________________________________________
*/

int
main(int   argc,
     char *argv[])

     /* Interface description:
         INPUT:
          argc  - The number of command-line arguments.
          argv  - An array of command-line arguments.
         OUTPUT:
          None.
         RETURN VALUE:
          int   - Status flag.  Non-zero is error.
     */
      
{
     /* Internal variables: */
          /* Major */
   UserInfo UserInfoRec ;		/* Information about the user */
   Summary *Summaries = NULL ;		/* The list of problem summaries */
   char **SummaryItems=NULL ;		/* Summary strings */
   ProblemTree *ProbTree = NULL ;	/* The problem types tree */
   struct Full_path pathlist;		/* The problem tree, linked list */
   struct Full_path *path=NULL;		/* A path in the problem tree */
   char HTMLindex[COMPLETEPATHLEN+1] ;	/* File name of index file for a leaf*/
   char HTMLindexdir[COMPLETEPATHLEN+1];/* Dir. name of index file for a leaf*/
   char fname[COMPLETEPATHLEN+1] ;      /* File name */
   FILE *fp ;                           /* File pointer */
   int SolvedCount=0 ;			/* Number of solved problems encountered */
   int UnsolvedCount=0 ;		/* Number of solved problems encountered */

          /* Minor */
   Summary *s = NULL ;			/* A problem summary */
   int i=0 ;				/* Counter */
   int errcnt=0 ;			/* Number of errors encountered */
   char tmppath[COMPLETEPATHLEN+1] ;	/* A temporary pathname */


   /*---------------------- Start of main() routine -----------------------*/

     /* Get information about the user and read the problem hierarchy. */

   zdebug("calling Setup()...\n") ;

/*CLD*/   if (Setup(&UserInfoRec, &ProbTree) == False)
   {  fprintf(stderr, "Error in configuration file %s !\n", CONFIGFILE) ;
      exit(1);
   }

   zdebug("returned from Setup()...\n") ;
   zdebug1("\tUserInfoRec.sysopflag = %d\n", UserInfoRec.sysopflag);
   zdebug1("\tUserInfoRec.username  = %s\n", UserInfoRec.username) ;
   zdebug1("\tUserInfoRec.realname  = %s\n", UserInfoRec.realname) ;


        /* Get the problem tree into a list of leaves more easily printed.*/
        /* Then print the list of problem types (leaves only.) if DEBUGging */
/*CLD*/  TraverseLeaves(ProbTree, &pathlist) ;
#ifdef DEBUG
/*CLD*/  TraverseList(stderr,&pathlist) ;
#endif


	/* Print the header for the unsolved list */
      sprintf(fname,"%s/unsolved.html",PTSROOT);
   if (fp=(fopen(fname, "w")))
   {
      HTMLHeader(fp, "Problem Tracking System", "Unsolved Problems", HTML_REFRESH) ;
      fclose (fp) ;
   }
   else
   {
      fprintf(stderr, "Error -- cannot write %s\n", fname) ;
   }

	/* Print the header for the solved list */
   sprintf(fname,"%s/solved.html",PTSROOT);
   if (fp=(fopen(fname, "w")))
   {
      HTMLHeader(fp, "Problem Tracking System", "Solved Problems", HTML_REFRESH) ;
      fclose (fp) ;
   }
   else
   {
      fprintf(stderr, "Error -- cannot write %s\n", fname) ;
   }


   /* For each problem type leaf
    *	get the list of summaries for that leaf
    *	for each summary record
    *		print the summary record
    *	done
    * Done
    */
   for (path=&pathlist; path != NULL; path=path->next)
   {

      printf("%s\n", path->path) ;

      if (strlen(path->path) > 0)
      {
         sprintf(tmppath, "%s%s", PTSROOT, path->path) ;
         mkdirhier(tmppath) ;


/*CLD*/  Summaries = ReadSummaries(ProbTree, path->path, &SummaryItems) ;

         sprintf(HTMLindexdir, "%s", path->path) ;
         strcpy(HTMLindex, "index.html") ;
         errcnt += DoSummaryList(Summaries, ProbTree, 
                                 HTMLindexdir, HTMLindex,
				 &SolvedCount, &UnsolvedCount) ;
         FreeList(SummaryItems) ;

         SummaryItems=NULL ;

      }
   }


	/* Print the footer for the unsolved list */
   sprintf(fname,"%s/unsolved.html",PTSROOT);
   if (fp=(fopen(fname, "a")))
   {
      if (UnsolvedCount == 0)	/* None were found -- Good! No work!  */
      {
         fprintf (fp, "<I>No unsolved problems were found.</I>%c", 10) ;
      }

      HTMLFooter(fp) ;
   }

	/* Print the footer for the unsolved list */
   sprintf(fname,"%s/solved.html",PTSROOT);
   if (fp=(fopen(fname, "a")))
   {
      if (SolvedCount == 0)	/* None were found -- uh-oh, get to work! */
      {
         fprintf (fp, "<I>No solved problems were found.</I>%c", 10) ;
      }

      HTMLFooter(fp) ;
   }


   exit (errcnt) ;

}  /*-------------------------- End of main() -----------------------------*/


/*
______________________________________________________________________
DoSummaryList()

FUNCTION DESCRIPTION:

     This function will spit out the HTML files for the listed problems.
______________________________________________________________________
UNIT TESTING:
     This function will be tested by using a combination of white-box and 
black-box tests.
______________________________________________________________________
REVISION HISTORY:
     Author:  Dean Collins      Date:  Wed Mar 15 09:21:00 PST 1995
______________________________________________________________________
*/

int
DoSummaryList( Summary     *Summaries,
               ProblemTree *ProbTree,
               char        *HTMLindexdir,
               char        *HTMLindex,
               int         *SolvedCount,
               int         *UnsolvedCount)

     /* Interface description:
         INPUT:
          Summaries      - The list of summaries to deal with.  This could
			   be from ReadSummaries() or ReadUnsolvedSummaries().
          ProbTree       - The problem types tree.
          HTMLindexdir   - The name of directory the HTML index file
			   (below) is to be placed in.  This is relative
                           to PTSROOT.
          HTMLindex      - The name of the HTML index file to update.
			   This is in HTMLindexdir above.
          SolvedCount    - Number of solved problems encountered.
          UnsolvedCount  - Number of unsolved problems encountered.
         OUTPUT:
          None.
         RETURN VALUE:
          int		 - Number of errors encountered.
     */
      
{
     /* Internal variables: */
          /* Major */
   problem_record *ProblemRecord=NULL ; /* A full problem record */
   char fname[COMPLETEPATHLEN+1] ;      /* File name */
   char fname2[COMPLETEPATHLEN+1] ;     /* File name */
   FILE *fp ;                           /* File pointer */
   FILE *fp2 ;                          /* File pointer */
   char tmpstr[DBPATHLEN+1] ;		/* Used to hold a temporary string */

          /* Minor */
   Summary *s = NULL ;                  /* A problem summary */
   int i=0 ;                            /* Counter */
   int errcnt=0 ;                       /* Number of errors encountered */

   /*----------------- Start of DoSummaryList() routine ------------------*/

   sprintf(fname2, "/%s%s/%s", PTSROOT, HTMLindexdir, HTMLindex) ;
   fp2 = fopen(fname2,"w") ;
   sprintf (tmpstr,"Index of %s", HTMLindexdir) ;
   HTMLHeader (fp2, "Problem Tracking System", tmpstr, HTML_REFRESH) ;

   if (Summaries != NULL)
   {
      for (s=Summaries, i=0; s != NULL; s = s->next, ++i)
      {
         printf("\t%s/%s.html\n", s->path,s->prid) ;
         
         ProblemRecord = ReadProblem(ProbTree, s->path, i, Summaries) ;

         if (s->status == UNSOLVED)
         {  fprintf(fp2,"<IMG SRC=\"%s/%s\"    ALT=\"*  \">%c", IMAGES, UNSOLVEDIMAGE, 10) ;
            ++(*UnsolvedCount) ;
         }
         else if (s->status == SOLVED)
         {  fprintf(fp2,"<IMG SRC=\"%s/%s\"   ALT=\"o  \">%c", IMAGES, SOLVEDIMAGE, 10) ;
            ++(*SolvedCount) ;
         }
         else if (s->status == REOPENED)
            fprintf(fp2,"<IMG SRC=\"%s/%s\" ALT=\"R  \">%c", IMAGES, REOPENIMAGE, 10) ;


/*
 *Don't show the PRID since it will confuse most users
 *
 *         fprintf(fp2,"<A HREF=\"%s.html\">%s</A> -- %c",
 *	         s->prid, s->prid, 10) ;
 *
 */

	 fprintf(fp2,"<A HREF=\"%s.html\">%s</A> <BR>%c%c",
	         s->prid, s->summary, 10, 10) ;

         sprintf(fname,"%s%s/%s.html",PTSROOT, s->path, s->prid);

         if (fp=fopen(fname,"w"))
         {  
			/*THE CRUX OF IT ALL*/

#           ifndef WAIS_TEXT_ONLY
            FormatProblemHTML(fp, ProblemRecord, s->path) ;
#           else
            fprintf(fp,"%s\n", FormatProblem(ProblemRecord, s->path)) ;
#           endif /*WAIS_TEXT_ONLY*/

            fclose(fp) ;
         }
         else
         {  fprintf(stderr,"ERROR -- %s\n",strerror(errno)) ;
            errcnt++ ;
         }

		/* Update unsolved list */
         if (s->status == UNSOLVED || s->status == SOLVED)
         {  if (s->status == UNSOLVED)	sprintf(fname,"%s/unsolved.html",PTSROOT);
            if (s->status == SOLVED)	sprintf(fname,"%s/solved.html",PTSROOT);

			/* Write record to solved or unsolved list */
	    if ((fp=fopen(fname, "a")) != NULL)
            {  
               if (s->status == UNSOLVED)
                  fprintf(fp,"<IMG SRC=\"%s/%s\"    ALT=\"*  \">%c", IMAGES, UNSOLVEDIMAGE, 10) ;
               if (s->status == SOLVED)
                  fprintf(fp,"<IMG SRC=\"%s/%s\"   ALT=\"o  \">%c", IMAGES, SOLVEDIMAGE, 10) ;

	       fprintf(fp,"<A HREF=\"%s/db/%s/%s.html\">%s</A> -- ",
			HTDOCS_URL, s->path, s->prid, s->summary) ;
	       fprintf(fp,"<I>(<A HREF=\"%s/db/%s/index.html\">/%s</A>)</I>",
			HTDOCS_URL, s->path, s->path) ;
               fprintf(fp,"<BR>%c%c",10,10) ;

               fclose(fp) ;
	    }
         } 
      }
   }
   else /* Summaries == NULL */
   {
      fprintf (fp2,"No problems found.<BR>%c", 10, 10) ;
   }

  fprintf(fp2, "<FORM METHOD=\"POST\" ACTION=\"%s/pts-post\">%c",
          CGIBIN_URL, 10) ;
  fprintf(fp2, "<INPUT TYPE=\"submit\" VALUE=\"New Problem\">%c", 10) ;
  fprintf(fp2, "</FORM>%c", 10) ;

   HTMLFooter(fp2) ;
   fclose(fp2) ;
   return (errcnt) ;


}  /*--------------------- End of DoSummaryList() ------------------------*/

/*
______________________________________________________________________
mkdirhier()

FUNCTION DESCRIPTION:

        This function makes a directory.  If any components of the
directory pathname do not exist they are created as well.  (Rather
like "mkdir -p".  If any component of the directory already exists
it is not considered an error, however the directory component's
permissions are NOT set.
______________________________________________________________________
REVISION HISTORY:
     Author:  Dean Collins                     Date: 3/6/95
______________________________________________________________________
*/

int
mkdirhier(char *p)

     /* Interface description:
         INPUT:
          p     - The path to create.
         OUTPUT:
          None.
         RETURN VALUE:
          int   - Status flag.  Non-zero is error.
     */

{
     /* Internal variables: */
          /* Major */
   char s[COMPLETEPATHLEN+1] ;
          /* Minor */
   int i=0 ;
   int stat=0 ;

   /*------------------ Start of mkdirhier() routine -------------------*/

   for (i=0; i<strlen(p); ++i)
   {
      if (p[i] == '/' && i != 0)
      {  s[i] = '\0' ;
         if (mkdir(s, DIR_PERMS) == -1)
            if (errno != EEXIST)
               fprintf(stderr,"Error %d -- %s\n",errno, strerror(errno)) ;
      }
      s[i]=p[i] ;
   }

   if (mkdir(p,DIR_PERMS) == -1)
      if (errno != EEXIST)
         fprintf(stderr,"Error %d -- %s\n",errno, strerror(errno)) ;

   return (stat) ;

} /*------------------- End of mkdirhier() routine --------------------*/

/* End of wais.c */
