#include <sys/soundcard.h>	//OS-specific
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <cstdio>
#include <unistd.h>
#include <signal.h>
#include <iostream>

#include "akApp.h"
#include "akSound.h"

// MicroSoft WAV file header format
typedef struct {
	char		main_chunk[4];	// 'RIFF'
	unsigned long	length;		// 'file' length
	char		chunk_type[4];	// 'WAVE'
	char		sub_chunk[4];	// 'fmt'
	unsigned long	length_chunk;	// length sub_chunk, always 16 bytes
	unsigned short	format;		// always 1 (=PCM-Code)
	unsigned short	modus;		// 1=Mono, 2=Stereo
	unsigned long	sample_fq;	// Sample frequency
	unsigned long	byte_p_sec;	// Data/sec
	unsigned short	byte_p_spl;	// bytes/sample, 1=8 bit,2=16 bit(mono)
					//		 2=8 bit,4=16 bit(stereo)
	unsigned short	bit_p_spl;	// bits/sample, 8,12,16
	char		data_chunk[4];	// 'data'
	unsigned long	data_length;	// 'data' length
} wavHeader;

// Creative Labs VOC file and block header formats
typedef struct
{
	unsigned char	main_chunk[20];	// 'Creative Voice File'
	unsigned short	block_offset;	// Offset to first block from top of file
	unsigned short	version;	// VOC version
	unsigned short	id_code;	// Complement of version+0x1234 !!
} vocHeader;
typedef struct
{
	unsigned char	BlockID;
	unsigned char	BlockLen[3];	/* low, mid, high byte of length of rest of block */
} vocBlockHeader;
typedef struct
{
	unsigned char	TimeConstant;
	unsigned char	PackMethod;
} vocBlockType1;
typedef struct
{
	unsigned short	TimeConstant;
	unsigned char	PackMethod;
	unsigned char	VoiceMode;
} vocBlockType8;
typedef struct
{
	unsigned int	SamplesPerSec;
	unsigned char	BitsPerSample;
	unsigned char	Channels;
	unsigned short	Format;
	unsigned char	reserved[4];
} vocBlockType9;

////////////////////////////////////////////////////////////////////////////////
//
//	Constructor
//
////////////////////////////////////////////////////////////////////////////////
akSound::akSound(string filename)
	:_filename(filename),
	 _paused(false),
	 _pid(0)
{
	checkType();
}
////////////////////////////////////////////////////////////////////////////////
//
//	Destructor
//
////////////////////////////////////////////////////////////////////////////////
akSound::~akSound()
{
	if (_pid > 0)
	  if (kill(_pid,0) != -1)
	    kill(_pid,SIGKILL);
}
////////////////////////////////////////////////////////////////////////////////
//
//	pause - pause the sound
//
////////////////////////////////////////////////////////////////////////////////
bool	akSound::pause()
{
	if (_pid == 0 || kill(_pid,0) == -1)
	  {
	   _paused = false;
	   return false;
	  }
	if (_paused)
	  {
	   _paused = false;
	   kill(_pid,SIGCONT);
	  }
	else
	  {
	   _paused = true;
	   kill(_pid,SIGSTOP);
	  }
	return true;
}
////////////////////////////////////////////////////////////////////////////////
//
//	stop - stop the sound
//
////////////////////////////////////////////////////////////////////////////////
bool	akSound::stop()
{
	if (_pid > 0)
	  if (kill(_pid,0) != -1)
	    kill(_pid,SIGKILL);
	_paused = false;

	return false;
}
////////////////////////////////////////////////////////////////////////////////
//
//	playing - is the sound playing ?
//
////////////////////////////////////////////////////////////////////////////////
bool	akSound::playing()
{
	if (_pid > 0)
	  if (kill(_pid,0) != -1)
	    if (!_paused)
	      return true;

	return false;
}
////////////////////////////////////////////////////////////////////////////////
//
//	play - play the sound using the audio-device
//
////////////////////////////////////////////////////////////////////////////////
bool	akSound::play()
{
	int		pid;

//	Check for erroneous data

	if (_filename.empty() || _type == AK_SOUND_NONE)
	  return false;

//	Check if paused - continue if so

	if (_paused)
	  return pause();

//	Fork the process to play the sound

	signal(SIGCHLD,SIG_IGN);
	if (pid=fork(),pid == 0)
	  {
	   int		file_fd;
	   int		audio_fd;
	   int		bytes_read;

	   file_fd = open((char *)_filename.c_str(),O_RDONLY,0);
	   if (file_fd >= 0)
	     {

	      // Open audio device, sync the dsp device and write data

	      audio_fd = open("/dev/dsp",O_WRONLY,0);
	      ioctl(audio_fd,SNDCTL_DSP_SYNC,NULL);
	      if (audio_fd >= 0)
		{
		 int	nbytes;
		 int	count=0;
		 int	speed=0;
		 int	sample_size=0;
		 int	stereo=0;
		 char	*buffer;
		 int	buffer_size=0;

		 ///////////////////////////////////////////////////////////////
		 // Find and read BLOCK header info to set parameters and play block data
		 ///////////////////////////////////////////////////////////////

		 if (_type == AK_SOUND_WAV)
		   {
		    wavHeader	wav_hdr;

		    nbytes = read(file_fd,&wav_hdr,sizeof(wav_hdr));
		    count       = wav_hdr.data_length;
		    speed       = wav_hdr.sample_fq;
		    sample_size = wav_hdr.bit_p_spl;
		    stereo      = (wav_hdr.modus == 2);
		   }
	         else if (_type == AK_SOUND_VOC)
		   {
		    vocHeader		voc_hdr;
		    vocBlockHeader	block;
		    int			lastblocktype=-1;

		    nbytes = read(file_fd,&voc_hdr,sizeof(voc_hdr));
		    if ((read(file_fd,(char *)&block,sizeof(block))) != -1)
		      {
		       if ((int)block.BlockID == 1)
		         {
		          vocBlockType1		type1;

		          read(file_fd,(char *)&type1,sizeof(type1));
			  if (type1.PackMethod != 0) goto end_of_play;
			  if (lastblocktype != 8)
			    {
			     speed = 256000000/(65536 - ((int)type1.TimeConstant<<8));
			     sample_size = 8;
			     stereo = 0;
			    }
			  count = ((unsigned long)(block.BlockLen[0]) |
			          ((unsigned long)(block.BlockLen[1])<<8) |
			          ((unsigned long)(block.BlockLen[2])<<16)) - 2;
			  lastblocktype = 1;
		         }
		       else if ((int)block.BlockID == 0)
		         {
			  goto end_of_play;
			 }
		       else if ((int)block.BlockID == 6)
		         {
			  // Used for start of a loop - not supported
			 }
		       else if ((int)block.BlockID == 7)
		         {
			  // Used for end of a loop - not supported
			 }
		       else if ((int)block.BlockID == 8)
		         {
		          vocBlockType8		type8;

		          read(file_fd,(char *)&type8,sizeof(type8));
			  if (type8.PackMethod != 0) goto end_of_play;
			  speed = 256000000/(65536 - type8.TimeConstant);
			  sample_size = 8;
			  stereo = type8.VoiceMode;
			  if (stereo) speed >>=1;
			  count = 0;
			  lastblocktype = 8;
		         }
		       else if ((int)block.BlockID == 9)
		         {
		          vocBlockType9		type9;

		          read(file_fd,(char *)&type9,sizeof(type9));
			  if (type9.Format != 0 && type9.Format != 4) goto end_of_play;
			  speed = type9.SamplesPerSec;
			  sample_size = type9.BitsPerSample;
			  stereo = type9.Channels-1;
			  count = ((unsigned long)(block.BlockLen[0]) |
				  ((unsigned long)(block.BlockLen[1])<<8) |
				  ((unsigned long)(block.BlockLen[2])<<16)) - 12;
			  lastblocktype = 9;
		         }
		       else
			 {
			  count = ((unsigned long)(block.BlockLen[0]) |
				  ((unsigned long)(block.BlockLen[1])<<8) |
				  ((unsigned long)(block.BlockLen[2])<<16));
			  if (lseek(file_fd,count,SEEK_CUR) == -1)
			    goto end_of_play;
			  count = 0;
			  lastblocktype = block.BlockID;
			 }
		      }
		   }
		 else
		   goto end_of_play;

		 ///////////////////////////////////////////////////////////////
		 // Set Sampling Size, Stereo, and Speed
		 ///////////////////////////////////////////////////////////////

		 ioctl(audio_fd,SNDCTL_DSP_SAMPLESIZE,&sample_size);
		 ioctl(audio_fd,SNDCTL_DSP_STEREO,&stereo);
		 ioctl(audio_fd,SNDCTL_DSP_SPEED,&speed);

		 ///////////////////////////////////////////////////////////////
		 // Set up Sampling buffer
		 ///////////////////////////////////////////////////////////////

		 ioctl(audio_fd,SNDCTL_DSP_GETBLKSIZE,&buffer_size);
		 buffer = new char[buffer_size];

		 ///////////////////////////////////////////////////////////////
		 // Read and play this BLOCK - 'count' bytes
		 ///////////////////////////////////////////////////////////////

		 while (count > 0)
		   {
		    int	cnt;

		    if (count > buffer_size)
		      cnt = buffer_size;
		    else
		      cnt = count;
		    nbytes = read(file_fd,buffer,cnt);
		    if (nbytes > 0)
		      {
		       int	n = write(audio_fd,buffer,nbytes);
		       if (n != nbytes)
		         {
		          delete buffer;
		          goto end_of_play;
		         }
		       count = count - nbytes;
		      }
		    else	// Error reading sound data
		      {
		       delete buffer;
		       goto end_of_play;
		      }
		   }
	         delete buffer;
	        }
	     }

end_of_play:
	   if (audio_fd >= 0)
	     close(audio_fd);
	   if (file_fd >= 0)
	     close(file_fd);

	   _exit(255);
	  }
	else if (pid > 0)
	  {
	   _pid = pid;
	  }
	else
	  {
	  }

	return true;
}
////////////////////////////////////////////////////////////////////////////////
//
//	checkType - Find the sound file type
//
////////////////////////////////////////////////////////////////////////////////
void	akSound::checkType()
{
	int		file_fd;
	int		nbytes;

	wavHeader	wav_hdr;
	vocHeader	voc_hdr;

	file_fd = open((char *)_filename.c_str(),O_RDONLY,0);
	if (file_fd >= 0)
	  {

	   // Check for WAV file

	   lseek(file_fd,0,SEEK_SET);
	   nbytes = read(file_fd,&wav_hdr,sizeof(wav_hdr));
	   if (nbytes == sizeof(wav_hdr))
	     {
	      if (strncmp(wav_hdr.main_chunk,"RIFF",4) == 0 &&
		  strncmp(wav_hdr.chunk_type,"WAVE",4) == 0 &&
		  strncmp(wav_hdr.sub_chunk,"fmt ",4)  == 0 &&
		  wav_hdr.sample_fq*wav_hdr.byte_p_spl == wav_hdr.byte_p_sec)
		{
		 close(file_fd);
		 _type = AK_SOUND_WAV;
		 return;
		}
	     }

	   // Check for VOC file

	   lseek(file_fd,0,SEEK_SET);
	   nbytes = read(file_fd,&voc_hdr,sizeof(voc_hdr));
	   if (nbytes == sizeof(voc_hdr))
	     {
	      if (strncmp((char *)voc_hdr.main_chunk,"Creative Voice File",19) == 0)
		{
		 close(file_fd);
		 _type = AK_SOUND_VOC;
		 return;
		}
	     }

	   // File type not found

	   close(file_fd);
	   _type = AK_SOUND_NONE;
	   return;
	  }
	else
	  {
	   _type = AK_SOUND_NONE;
	   return;
	  }
}
