////////////////////////////////////////////////////////////////////////////////
//
//	Class : mxMsgSummary
//
//	Author : Andy Jefferson
//
//	Description :
//		mxMsgSummary is a summary representation of a message.
//		A message is split into a header tags and the message itself.
//		The mxMsgSummary object splits off the author,subject,
//		addressee, and other tags from the actual message.
//		The position refers to the position of the message in
//		the box where the message is stored (e.g see mxMailBox).
//
//	Public Methods :
//		Fields stored are only representative of 'Mail' currently. They
//		need broadening to also represent 'News'.
//
////////////////////////////////////////////////////////////////////////////////
#ifndef mxMsgSummary_h
#define mxMsgSummary_h
#ifdef __cplusplus

#include <std/string.h>
#include <bool.h>

class mxMsgSummary
{
	public:

		// Constructor, assignment operator, etc

		mxMsgSummary()
		  :_status(false),_position(0),_length(0) {};
		mxMsgSummary(const mxMsgSummary&);
		mxMsgSummary& operator=(const mxMsgSummary&);

		// Update functions - setting headers from message text

		void	setMessageHeaders(string actual_text);

		// Update functions - setting bit by bit

		void	setAuthor(string);
		void	setOrganisation(string);
		void	setSubject(string);
		void	setDate(string);
		void	setToAddressee(string);
		void	setCcAddressee(string);
		void	setBccAddressee(string);
		void	setMailid(string);
		void	setMessageId(string);
		void	setLength(int);

		// Update functions for message in relation to its box

		void	setStatus(bool);
		void	setPosition(int position);
		void	setActualText(string text);

		// Access functions

		const string&	actualText() const;
		const string&	author() const;
		const string&	organisation() const;
		const string&	date() const;
		const string&	toAddressee() const;
		const string&	ccAddressee() const;
		const string&	bccAddressee() const;
		const string&	subject() const;
		const string&	mailid() const;
		const string&	messageId() const;
		bool		status() const;
		int		position() const;
		int		length() const;

		// Control functions

		void	clear();

		friend ostream&	operator<< (ostream &s,mxMsgSummary& msg);

	protected:

		// Message data items

		string	_actualText;	// Copy of the actual message
		string 	_author;	// Author's name
		string	_mailid; 	// Mail-address of author
		string	_organisation;	// Author's Organisation
		string 	_date;		// Date message was sent
		string 	_toAddressee; 	// To Addressee's name
		string 	_ccAddressee; 	// Cc Addressee's name
		string 	_bccAddressee; 	// Bcc Addressee's name
		string 	_subject;	// Subject of the message
		string	_messageId; 	// Message-Id of message
		bool	_status;	// Has message been read ?
		int	_position;	// Location of position in mail box
		int	_length;	// Length of message (in bytes)

		// Class methods for extracting tags from a message

		static string	tagValue(string msg,string tag);
		static int	bodyStart(string msg);
		static int	nextTagStart(string msg,string tag,int pos=0);
};


inline
const string&	mxMsgSummary::actualText() const
	{return _actualText;}
inline
const string&	mxMsgSummary::author() const
	{return _author;}
inline
const string&	mxMsgSummary::organisation() const
	{return _organisation;}
inline
const string&	mxMsgSummary::date() const
	{return _date;}
inline
const string&	mxMsgSummary::toAddressee() const
	{return _toAddressee;}
inline
const string&	mxMsgSummary::ccAddressee() const
	{return _ccAddressee;}
inline
const string&	mxMsgSummary::bccAddressee() const
	{return _bccAddressee;}
inline
const string&	mxMsgSummary::subject() const
	{return _subject;}
inline
const string&	mxMsgSummary::mailid() const
	{return _mailid;}
inline
const string&	mxMsgSummary::messageId() const
	{return _messageId;}
inline
bool		mxMsgSummary::status() const
	{return _status;}
inline
int		mxMsgSummary::position() const
	{return _position;}
inline
int		mxMsgSummary::length() const
	{return _length;}
inline
void	mxMsgSummary::setAuthor(string author)
	{_author = author;}
inline
void	mxMsgSummary::setOrganisation(string org)
	{_organisation = org;}
inline
void	mxMsgSummary::setSubject(string subject)
	{_subject = subject;}
inline
void	mxMsgSummary::setDate(string date)
	{_date = date;}
inline
void	mxMsgSummary::setToAddressee(string toAddressee)
	{_toAddressee = toAddressee;}
inline
void	mxMsgSummary::setCcAddressee(string ccAddressee)
	{_ccAddressee = ccAddressee;}
inline
void	mxMsgSummary::setBccAddressee(string bccAddressee)
	{_bccAddressee = bccAddressee;}
inline
void	mxMsgSummary::setMailid(string mailid)
	{_mailid = mailid;}
inline
void	mxMsgSummary::setMessageId(string messageId)
	{_messageId = messageId;}
inline
void	mxMsgSummary::setLength(int length)
	{_length = length;}
inline
void	mxMsgSummary::setActualText(string the_text)
	{_actualText = the_text;}
inline
void	mxMsgSummary::setStatus(bool read)
	{_status = read;}
inline
void	mxMsgSummary::setPosition(int the_position)
	{_position = the_position;}
inline
void	mxMsgSummary::clear()
{
	_actualText   = "";
	_author       = "";
	_date         = "";
	_toAddressee  = "";
	_ccAddressee  = "";
	_bccAddressee = "";
	_subject      = "";
	_mailid       = "";
	_status       = false;
	_position     = 0;
	_length       = 0;
}
inline
ostream&	operator<<(ostream& s,mxMsgSummary& msg)
{
	return s <<
		"mxMsgSummary" << endl <<
		"  Subject:      " << msg._subject << endl <<
		"  Author:       " << msg._author << endl <<
		"  ToAddressee:  " << msg._toAddressee << endl <<
		"  CcAddressee:  " << msg._ccAddressee << endl <<
		"  BccAddressee: " << msg._bccAddressee << endl <<
		"  Mailid:       " << msg._mailid << endl <<
		"  Text:         " << msg._actualText << endl <<
		"  Start:        " << msg._position << endl <<
		"  Length:       " << msg._length << endl <<
		"  Status:       " << msg._status << endl;
}

#endif
#endif
