/*
 * log.c
 *	Log dialogs.  This file is part of wlib.
 *
 * Copyright (C) 1996  Matthew D. Francey
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   Authors:	Matthew D. Francey
 *		Eric A. Howe (mu@echo-on.net)
 */
#include	<wlib/rcs.h>
MU_ID("$Mu: wlib/log.c,v 1.45 $")

#include	<unistd.h>
#include	<errno.h>
#include	<stdarg.h>
#include	<stdlib.h>
#include	<assert.h>
#include	<Xm/Text.h>
#include	<Xm/FileSB.h>

#include	<wlib/wlib.h>
#include	<wlib/wlibP.h>
#include	<wlib/typesanity.h>

/*
 * user data for the log
 */
typedef struct {
	Widget	form;
	Widget	fs;
	int	hidden;
} LOG;

/*
 * log->clear
 */
static void
clear(Widget w, XtPointer closure, XtPointer call)
{
	LOG	*lg = (LOG *)closure;

	USEUP(call);

	w = wl_find1(lg->form, "*textSW*text");
	assert("log->clear(): can't find *textSW*text in the log!" != NULL);
	XmTextSetString(w, "");
	XmTextSetCursorPosition(w, (XmTextPosition)0);
	XmTextSetInsertionPosition(w, (XmTextPosition)0);
	XmTextShowPosition(w, (XmTextPosition)0);
}

/*
 * log->hide
 */
static void
hide(Widget w, XtPointer closure, XtPointer call)
{
	LOG	*lg = (LOG *)closure;

	USEUP(w); USEUP(call);

	lg->hidden = TRUE;
	XtUnmanageChild(lg->form);
}

/*
 * log->fs->ok
 */
static void
save_ok(Widget w, XtPointer closure, XtPointer call)
{
	XmFSS	*cbs = (XmFSS *)call;
	LOG	*lg  = (LOG *)closure;
	Widget	text;
	char	*file, *value;
	FILE	*fp;

	file = NULL;
	text = wl_find1(lg->form, "*textSW*text");
	assert("log->save_ok(): can't find *textSW*text in the log!" != NULL);

	if(!XmStringGetLtoR(cbs->value, XmFONTLIST_DEFAULT_TAG, &file)) {
		wl_error(w, "%s: %s", XtName(w), wlp_s(w, WlpScantGet));
		return;
	}
	if((value = XmTextGetString(text)) == NULL
	|| value[0] == '\0') {
		wl_warn(lg->form, "%s", wlp_s(w, WlpSemptyLog));
		XtFree((XtPointer)value);
		XtUnmanageChild(w);
		return;
	}
	if((fp = fopen(file, "w")) == NULL
	|| fwrite(value, 1, strlen(value), fp) != strlen(value)) {
		wl_error(w, "%s: %s", file, strerror(errno));
		unlink(file);
	}
	else {
		XtUnmanageChild(w);
	}
	if(fp != NULL)
		fclose(fp);
	XtFree((XtPointer)value);
}

static WLW log_save[] = {
	{"logSave",	wlFileSBDialog,	0,	NULL},
	{NULL}
};
static WLC cbs[] = {
	{"*logSave",	XmNcancelCallback,	0, NULL,	wl_dismiss},
	{"*logSave",	XmNokCallback,		0, NULL,	save_ok},
	{NULL}
};

/*
 * log->save
 */
static void
save(Widget w, XtPointer closure, XtPointer call)
{
	LOG	*lg = (LOG *)closure;

	USEUP(call);

	if(lg->fs == NULL) {
		if((lg->fs = wl_create(XtParent(w), &log_save[0])) == NULL)
			return;
		wl_callback(XtParent(lg->fs), &cbs[0], (XtPointer)lg);
	}
	else {
		XmFileSelectionDoSearch(lg->fs, NULL);
	}
	XtManageChild(lg->fs);
	XtPopup(XtParent(lg->fs), XtGrabNone);
}

/*
 * the log dialog itself
 */
static WLW log_form[] = {
	{"hide",		wlPushB,	0,		NULL},
	{"save",		wlPushB,	0,		NULL},
	{"clear",		wlPushB,	0,		NULL},
	{"help",		wlPushB,	0,		NULL},
	{"sep",			wlSeparatorG,	0,		NULL},
	{"text",		wlScrolledText,	WL_MANAGE,	NULL},
	{NULL}
};
static WLC log_cb[] = {
	{"*hide",	XmNactivateCallback, 0, NULL, hide},
	{"*save",	XmNactivateCallback, 0, NULL, save},
	{"*clear",	XmNactivateCallback, 0, NULL, clear},
	{"*help",	XmNactivateCallback, 0, NULL, wl_help},
	{NULL}
};

/*
 * create a log
 */
Widget
wl_log_open(Widget parent, char *name)
{
	WLW	ws[2];
	LOG	*lg;
	Widget	w;

	if((lg = (LOG *)calloc(1, sizeof(LOG))) == NULL)
		return NULL;
	ws[0].name  = name;
	ws[0].type  = wlFormDialog;
	ws[0].flags = 0;
	ws[0].kids  = log_form;
	ws[1].name  = NULL;
	if((lg->form = wl_create(parent, &ws[0])) == NULL) {
		free((void *)lg);
		return NULL;
	}
	wl_callback(lg->form, &log_cb[0], (XtPointer)lg);
	w = wl_find1(lg->form, "*hide");
	XtVaSetValues(lg->form,
		XmNuserData,		(XtPointer)lg,
		XmNdefaultButton,	w,
		XmNcancelButton,	w,
		NULL);

	return lg->form;
}

/*
 * make log appear or disappear
 */
void
wl_log_show(Widget w, int show)
{
	LOG	*lg;

	XtVaGetValues(w, XmNuserData, &lg, NULL);
	lg->hidden = !show;
	if(show) {
		XtManageChild(lg->form);
		XtPopup(XtParent(lg->form), XtGrabNone);
	}
	else {
		XtUnmanageChild(lg->form);
	}
}

/*
 * make an entry in a log
 */
void
wl_log(Widget w, char *fmt, ...)
{
	Widget	text;
	char	*s1, *s2;
	va_list	ap;
	char	msg[1024];
	int	n;
	LOG	*lg;

	if(w   == NULL
	|| fmt == NULL
	|| (text = wl_find1(w, "*textSW*text")) == NULL)
		return;
	va_start(ap, fmt);
	vsprintf(&msg[0], fmt, ap);
	va_end(ap);

	s1 = XmTextGetString(text);
	s2 = XtRealloc((XtPointer)s1, strlen(s1) + strlen(msg) + 1);
	if(s2 != NULL)
		strcat(s1 = s2, msg);
	XmTextSetString(text, s1);
	XmTextSetCursorPosition(text, (XmTextPosition)(n = strlen(s1)));
	XmTextSetInsertionPosition(text, (XmTextPosition)n);
	XmTextShowPosition(text, (XmTextPosition)n);
	XtFree((XtPointer)s1);

	XtVaGetValues(w, XmNuserData, &lg, NULL);
	if(lg->hidden)
		return;
	XtManageChild(lg->form);
	XtPopup(XtParent(lg->form), XtGrabNone);
}
