/*
 * WlAppShell.c
 *	Implementation of the WlAppShell widget.
 *
 * Copyright (C) 1997 Eric A. Howe
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Library General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *   Authors:	Matthew D. Francey
 *		Eric A. Howe (mu@echo-on.net)
 *
 * This widget is loosely based an the AppPlusS widget by John L. Cwikla
 * <cwikla@wri.com>.
 */
#include <wlib/rcs.h>
MU_ID("$Mu: wlib/WlAppShell.c,v 1.8 $")

#include	<string.h>
#include	<ctype.h>
#include	<stdlib.h>
#include	<X11/IntrinsicP.h>
#include	<X11/Intrinsic.h>
#include	<X11/StringDefs.h>
#include	<X11/Xmu/Editres.h>
#include	<Xm/Xm.h>

#include	<wlib/WlAppShellP.h>

static void class_init(void);
static void init(Widget, Widget, ArgList, Cardinal *);
static void realize(Widget, XtValueMask *, XSetWindowAttributes *);

#define	core(a)		((a)->core)
#define	ashell(a)	((a)->wl_app_shell)
#define	shell(a)	((a)->shell)
#define	classpart	(wlAppShellClassRec.wl_app_shell_class)
#define	visualid(a)	((a)->wl_app_shell.vid)

#define DEFAULT -1

#define OFFSET(field) XtOffset(WlAppShellWidget, wl_app_shell.field)
static XtResource resources[] =
{
	{
		XwlNvisualClass, XwlCVisualClass,
		XwlRVisualClass, sizeof(int), OFFSET(vclass),
		XtRImmediate, (XtPointer)DEFAULT
	}, {
		XwlNusePrivateColormap, XwlCUsePrivateColormap,
		XtRBoolean, sizeof(Boolean), OFFSET(use_private_colormap),
		XtRImmediate, (XtPointer)False
	}, {
		XwlNvisualID, XwlCVisualID,
		XwlRVisualID, sizeof(VisualID), OFFSET(vid),
		XtRImmediate, (XtPointer)0
	}, {
		XwlNdepth, XwlCDepth,
		XtRInt, sizeof(int), OFFSET(depth),
		XtRImmediate, (XtPointer)DEFAULT
	}
};
#undef OFFSET

WlAppShellClassRec wlAppShellClassRec = 
{
	{
		/*
		 * Core
		 */
		(WidgetClass)&applicationShellClassRec,
		"WlAppShell",
		sizeof(WlAppShellRec),
		class_init,
		NULL,
		False,
		init,
		NULL,
		realize,
		NULL,
		0,
		resources,
		XtNumber(resources),
		NULLQUARK,
		False,
		False,
		False,
		False,
		NULL,	/* nothing to destroy	*/
		XtInheritResize,
		XtInheritExpose,
		NULL,
		NULL,
		XtInheritSetValuesAlmost,
		NULL,
		XtInheritAcceptFocus,
		XtVersion,
		NULL,
		NULL,
		XtInheritQueryGeometry,	
		XtInheritDisplayAccelerator,
		NULL,
	}, {
		/*
		 * Composite
		 */
		XtInheritGeometryManager,
		XtInheritChangeManaged,
		XtInheritInsertChild,
		XtInheritDeleteChild,
		NULL,
	}, {
		/*
		 * Shell
		 */
		0,
	}, {
		/*
		 * WindowManagerShell
		 */
		0,
	}, {
		/*
		 * VendorShell
		 */
		NULL,
	}, {
		/*
		 * TopLevelShell
		 */
		NULL,
	}, {
		/*
		 * ApplicationShell
		 */
		NULL,
	}, {
		/*
		 * WlAppShell
		 */
		(Colormap)None,
		NULL,
		0,	
		0,
	}
};

WidgetClass wlAppShellWidgetClass = (WidgetClass)&wlAppShellClassRec;

static char *
stolower(char *s)
{
	char	*p;
	int	i;

	for(p = s, i = 0; *p != '\0'; ++p, ++i)
		s[i] = tolower(*p);
	return s;
}

static struct {
	char	*s;	/* string version of visual class	*/
	size_t	n;	/* size of `s'				*/
	int	v;	/* visual class				*/
} vis_names[] = {
#	define	BLECH(a,b)	{a, sizeof(a) - 1, b}
	BLECH("staticgray",	StaticGray),
	BLECH("staticcolor",	StaticColor),
	BLECH("pseudocolor",	PseudoColor),
	BLECH("grayscale",	GrayScale),
	BLECH("greyscale",	GrayScale),
	BLECH("truecolor",	TrueColor),
	BLECH("directcolor",	DirectColor)
#	undef	BLECH
};
#define	N_VNAMES	(int)(sizeof(vis_names)/sizeof(vis_names[0]))

static Boolean
string2visualclass(Display *dpy, XrmValue *a, Cardinal *n,
				XrmValue *from, XrmValue *to, XtPointer *data)
{
static	int	cls;
	char	*name;
	Boolean	found;
	size_t	i;

	if(*n != 0) {
		XtAppWarningMsg(XtDisplayToApplicationContext(dpy),
				"string2visualclass", "wrongParamaters",
				"ResourceError",
				"string2visualclass needs no arguments.",
				NULL, NULL);
	}

	if(to->addr != NULL && to->size < sizeof(cls)) {
		to->size = sizeof(cls);
		return False;
	}

	name = stolower(XtNewString(from->addr));
	cls  = DEFAULT;
	for(i = 0, found = False; i < N_VNAMES; ++i) {
		if(strncmp(name, vis_names[i].s, vis_names[i].n) != 0)
			continue;
		found = True;
		cls   = vis_names[i].v;
		break;
	}
	XtFree(name);

	if(!found) {
		XtDisplayStringConversionWarning(dpy, from->addr,
							XwlRVisualClass);
		return False;
	}

	if(to->addr == NULL)
		to->addr = (XtPointer)&cls;
	else
		*(int *)to->addr = cls;
	to->size = sizeof(cls);

	return True;
}

static Boolean
string2visualid(Display *dpy, XrmValue *a, Cardinal *n,
				XrmValue *from, XrmValue *to, XtPointer *data)
{
static	VisualID	vid;
	XVisualInfo	tmp, *v;
	int		n_v;

	if(*n != 0) {
		XtAppWarningMsg(XtDisplayToApplicationContext(dpy),
					"string2visualid", "wrongParamaters",
					"ResourceError",
					"string2visualid needs no arguments.",
					NULL, NULL);
	}

	if(to->addr != NULL && to->size < sizeof(vid)) {
		to->size = sizeof(vid);
		return False;
	}

	vid = (VisualID)strtol(from->addr, NULL, 0);
	tmp.visualid = vid;
	if((v = XGetVisualInfo(dpy, VisualIDMask, &tmp, &n_v)) == NULL) {
		XtDisplayStringConversionWarning(dpy, from->addr, XwlRVisualID);
		return False;
	}
	XFree((char *)v);

	if(to->addr == NULL)
		to->addr = (XtPointer)&vid;
	else
		*(VisualID *)to->addr = vid;
	to->size = sizeof(vid);

	return True;
}

static void
class_init(void)
{
	XtSetTypeConverter(XtRString, XwlRVisualClass, string2visualclass,
						NULL, 0, XtCacheAll, NULL);
	XtSetTypeConverter(XtRString, XwlRVisualID, string2visualid,
						NULL, 0, XtCacheAll, NULL);
}

static void
defaults(WlAppShellWidget a, XVisualInfo *vi)
{
	Display	*dpy = XtDisplay((Widget)a);

	vi->depth  = DefaultDepth(dpy, DefaultScreen(dpy));
	vi->visual = DefaultVisual(dpy, DefaultScreen(dpy));
	vi->class  = vi->visual->class;
}

static Boolean
valid_vid(WlAppShellWidget a, XVisualInfo *vi)
{
	XVisualInfo	*vis, tmp;
	int		n;

	if(visualid(a) == 0)
		return False;

	tmp.visualid = visualid(a);
	if((vis = XGetVisualInfo(XtDisplay(a), VisualIDMask, &tmp, &n)) == NULL)
		return False;

	/*
	 * We'll assume that there is only one.
	 */
	*vi = *vis;
	XFree((char *)vis);
	return True;
}

static Boolean
use_defaults(WlAppShellWidget a, XVisualInfo *vi)
{
	if(ashell(a).depth != DEFAULT || ashell(a).vclass != DEFAULT)
		return False;
	defaults(a, vi);

	return True;
}

static void
init(Widget inw, Widget outw, ArgList a, Cardinal *n)
{
	WlAppShellWidget	out = (WlAppShellWidget)outw;
	Display			*dpy;
	int			sn;
	Visual			*def;
	XVisualInfo		vi;

	dpy  = XtDisplay(out);
	sn   = DefaultScreen(dpy);
	def  = DefaultVisual(dpy, sn);

	if(classpart.colormap != None) {
		shell(out).visual  = classpart.visual;
		ashell(out).vclass = classpart.vclass;
		core(out).depth    = classpart.depth;
		core(out).colormap = classpart.colormap;
		return;
	}
	if(!valid_vid(out, &vi)
	&& !use_defaults(out, &vi)) {
		if(ashell(out).depth == DEFAULT)
			core(out).depth = DefaultDepth(dpy, sn);
		else
			core(out).depth = ashell(out).depth;

		if(ashell(out).vclass == DEFAULT)
			ashell(out).vclass = def->class;

		/*
		 * Try and match the user's request; if that doesn't work,
		 * try and match the depth; if that doesn't work, try and
		 * match the visual class (with the best depth); if that
		 * doesn't work, we just give up and use the defaults
		 * that Xt has given us.
		 */
		if(XMatchVisualInfo(dpy, sn, core(out).depth,
						ashell(out).vclass, &vi) == 0) {
			XVisualInfo	tmp;
			XVisualInfo	*v  = NULL;
			int		n_v = 0;
			int		n   = 0;

			if(ashell(out).depth != DEFAULT) {
				/*
				 * We'll ignore n_v here to avoid some
				 * hassle, if the user doesn't like the
				 * first visual we get, then they can
				 * always use wlVisualID.
				 */
				tmp.depth = core(out).depth;
				v = XGetVisualInfo(dpy, VisualDepthMask, &tmp,
									&n_v);
			}

			if(v != NULL) {
				vi = *v;
			}
			else {
				tmp.class = ashell(out).vclass;
				v = XGetVisualInfo(dpy, VisualClassMask, &tmp,
									&n_v);
				if(v != NULL) {
					int	i;
					for(i = 1, n = 0; i < n_v; ++i)
						if(v[i].depth > v[n].depth)
							n = i;
					vi = v[n];
				}
			}

			if(v == NULL)
				defaults(out, &vi);
			else
				XFree((char *)v);
		}
	}

	core(out).depth    = vi.depth;
	shell(out).visual  = vi.visual;
	ashell(out).vclass = vi.class;
	if(shell(out).visual->visualid == def->visualid
	&& !ashell(out).use_private_colormap)
		core(out).colormap = DefaultColormap(dpy, sn);
	else
		core(out).colormap = XCreateColormap(dpy, RootWindow(dpy, sn), 
						shell(out).visual, AllocNone);

	/*
	 * And finally, cache the results for future use.
	 */
	classpart.visual   = shell(out).visual;
	classpart.colormap = core(out).colormap;
	classpart.vclass   = ashell(out).vclass;
	classpart.depth    = core(out).depth;
}

static void
realize(Widget w, XtValueMask *m, XSetWindowAttributes *wa)
{
	WlAppShellWidget	a = (WlAppShellWidget)w;

	wa->colormap         = core(a).colormap;
	wa->background_pixel =
	wa->border_pixel     = 0;
	*m |= CWColormap | CWBorderPixel | CWBackPixel;

	wlAppShellWidgetClass->core_class.superclass->core_class.realize(w,m,wa);
	XSetWindowColormap(XtDisplay(w), XtWindow(w), core(a).colormap);
}
