/*****************************************************************************

	xpmsg.c

	Environment:    Unix R40V3/Solaris2/Linux.

	Revision history:	@(#)xpmsg.c	1.9	97/06/23


	DESCRIPTION: Part of the Mdb Application.
			Generalized error/info dialogs.

        COPYRIGHT NOTICE:
        Permission to use,  copy,  modify,  and  distribute  this
        software  and  its    documentation   is  hereby  granted
        without fee, provided that  the  above  copyright  notice
        appear  in all copies and that both that copyright notice
        and  this  permission   notice   appear   in   supporting
        documentation.  The   author  makes  no   representations
        about   the   suitability   of   this  software  for  any
        purpose.  It  is  provided  "as  is"  without  express or
        implied warranty.

        THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD  TO  THIS
        SOFTWARE,    INCLUDING    ALL   IMPLIED   WARRANTIES   OF
        MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL THE AUTHOR
        BE  LIABLE  FOR  ANY  SPECIAL,  INDIRECT OR CONSEQUENTIAL
        DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS  OF
        USE, DATA OR PROFITS, WHETHER IN AN ACTION  OF  CONTRACT,
        NEGLIGENCE  OR  OTHER  TORTIOUS   ACTION,   ARISING   OUT
        OF   OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
        SOFTWARE.

******************************************************************************/
/******************************************************************************/
#pragma ident "@(#)xpmsg.c      1.9		97/06/23"

#include "mdb.h"

/*
 * Externals.
 */
extern char	*sys_errlist[];

/*
 * Local Variables.
 */
static int	IntCnt;
static int	InUse = 0;
static Boolean	Stopped;

/*
 * Forward declarations.
 */
static void	ForceUpdate(Widget w);
static Widget	Busy(Widget parent, Boolean on, XmString str);
static Widget	MsgW;

/*
 * Dialog type.
 */
#define ERROR	1
#define INFO	2
#define WORK	3
#define FATAL	4


/*
 * Take down an info/error/work Dialog.
 */
/*ARGSUSED*/
static void clear_xpmsg_cb(Widget w, int typ, XmAnyCallbackStruct *cbs )
{

	if ( InUse ) {
		XmUpdateDisplay(w);
		XtUnmanageChild(w);
	}

	InUse = 0;

	if ( typ == WORK ) {
		Stopped = True;
		IntCnt = 0;
		(void)Busy( NULL, False, NULL );
	}
}

void xpmsg_close(void)
{
	/*
	 * External close request.
	 */
	clear_xpmsg_cb( MsgW, InUse, NULL );
}


/*
 * Generalized routine to launch an error/info/work Dialog.
 * Synopsis:
 * Widget xpmsg( w, "fmt", ... );
 * Where:
 *	w	Is the widget to expose the message for, or NULL to
 *	        force mdb's Top shell as the parent.
 *	fmt	printf() style format string where a pattern terminated
 *		by a `:' should discribe the type of Dialog, i.e:
 *		"error:", "fatal:", "info:" and "work:" are valid patterns.
 *	...	Variable Argument list
 *
 * Returns:
 *	Widget I.D of the Dialog.
 *
 * When an error Dialog is launched, then errno will be honored as a
 * parameter to form the message string. The caller must ensure that errno
 * is set to 0 if the error message is not due to a oprating system error.
 *
 * When a work Dialog is launched, then the caller (user) has the possibility
 * to abort the current operation by polling the CheckForInterrupt() routine
 * defined below. A work Dialog must be un-posted by a extra call to xpmsg()
 * with a "fmt" set to "work:", i.e, no message string post the colon.
 *
 * xpmsg() does not return unless the Dialog is mapped on the screen.
 *
 * When a fatal Dialog is launched then the program will terminate after
 * 5 seconds.
 */
/*VARARGS*/
Widget xpmsg( Widget w, ... )
{

	Widget m_w = NULL;
	int n;
	int err = errno;
	int typ = 0;
	Dimension posx, posy, wd, hi;
	char dbuff[1024];
	char sys_err[1024];
	char *fmt, *ptr, *title;
	unsigned char dtype;
	XmString str = NULL;
	va_list ap;
	Arg args[10];

	va_start(ap, w);
	fmt = va_arg( ap, char * );
	(void)vsprintf( dbuff, fmt, ap );
	va_end(ap);

	if      ( ! strncmp( dbuff, "error:", 6 ) )   typ = ERROR;
	else if ( ! strncmp( dbuff, "fatal:", 6 ) )   typ = FATAL;
	else if ( ! strncmp( dbuff, "info:",  5 ) ) { typ = INFO; err = 0; }
	else if ( ! strncmp( dbuff, "work:",  5 ) ) { typ = WORK; err = 0; }

	if ( ! typ )
		return(NULL);	/* invalid request */

	if ( w == NULL )
		w = GetTopWidget();
	else
		/*
		 * Get the closest shell.
		 */
		for(;;) {
			if ( XmIsText(w) == True )
				break;
			if ( XmIsRowColumn(w) == True )
				break;
			if ( XmIsForm(w) == True )
				break;
			if ( XtIsWMShell(w) == True )
				break;
			if ( w == GetTopWidget() )
				break;
			w = XtParent(w);
		}
		
	if ( err ) {
#if defined(SVR4) || defined(linux)
		/*
		 * If we are lucky, and mdb is
		 * linked with -lintl, we will
		 * get the error message in
		 * native language.
		 */
		if ( (ptr = strerror(err)) == NULL )
			ptr = "";
#else
		ptr = sys_errlist[err];
#endif
		(void)sprintf( sys_err, ": %s", ptr );
		(void)strcat( dbuff, sys_err );
	}

	n = 0;
	ptr = strchr( dbuff, ':' ); if ( isspace(ptr[1]) ) ptr++;
	if ( *++ptr ) {
		str = XmStringCreateLtoR( ptr, XmSTRING_DEFAULT_CHARSET );
		XtSetArg( args[n], XmNmessageString, str ); n++;
	}

	if ( InUse && (InUse != WORK) && (typ != WORK) && (InUse != FATAL) ) {

		/*
		 * Reuse a mapped Dialog.
		 */
		dtype = typ == INFO ? XmDIALOG_INFORMATION : XmDIALOG_ERROR;

		title = typ == INFO ?
				GetRes("*infoDialog.dialogTitle") :
				GetRes("*errorDialog.dialogTitle");

		XtSetArg( args[n], XmNdialogType, dtype ); n++;
		XtSetValues( MsgW, args, n );

		XtVaSetValues( XtParent(MsgW), XtNtitle, title, NULL );

		/*
		 * Center map the Dialog.
		 */
		if ( XtIsWMShell(w) == False )
			while ( XtIsWMShell( (w = XtParent(w)) ) == False );

		XtVaGetValues( w, XmNx, &posx, XmNy, &posy,
			XmNwidth, &wd, XmNheight, &hi, NULL );

		posx += wd/2; posy += hi/2;

		XtVaGetValues( MsgW, XmNwidth, &wd, XmNheight, &hi, NULL );

		posx -= wd/2; posy -= hi/2;

		XtVaSetValues( MsgW, XmNx, posx, XmNy, posy, NULL );

		return(MsgW);
	}

	xpmsg_close();

	if ( (typ == ERROR) || (typ == FATAL ) )
		m_w = XmCreateErrorDialog( w, "errorDialog", args, n );
	else if (typ == INFO)
		m_w = XmCreateInformationDialog( w,"infoDialog", args, n );
	else if (typ == WORK) {
		if ( ! *ptr  ) {
			(void)Busy( w, False, NULL );
			return(NULL);
		} else
			m_w = Busy( w, True, str );
	}

	if ( m_w == NULL )
		return(NULL);

	XtUnmanageChild( XmMessageBoxGetChild( m_w, XmDIALOG_HELP_BUTTON) );

	if ( typ == INFO )
		XtUnmanageChild( XmMessageBoxGetChild( m_w,
			XmDIALOG_CANCEL_BUTTON) );
	else
		XtUnmanageChild( XmMessageBoxGetChild( m_w,
			XmDIALOG_OK_BUTTON) );

	n = 0;
	XtSetArg( args[n], XmNdeleteResponse, XmDO_NOTHING ); n++;
	XtSetValues( XtParent(m_w), args, n );

	XtAddCallback( m_w, XmNcancelCallback,
		(XtCP)clear_xpmsg_cb, (XtPointer)typ );

	XtAddCallback( m_w, XmNokCallback,
		(XtCP)clear_xpmsg_cb, (XtPointer)typ );


	XtManageChild(m_w);

	ForceUpdate(m_w);

	if ( *ptr )
		XmStringFree(str);

	if ( typ == FATAL ) {
		/*
		 * May not happen.
		 */
		XBell( XtDisplay(GetTopWidget()), 0 );
		Wait(5);
		main_cb( (Widget)NULL, ABORT, NULL );
	}


	InUse = typ;

	MsgW = m_w;

	return(m_w);
}


/*ARGSUSED*/
static Widget Busy( Widget parent, Boolean on, XmString str )
{

	static int locked;
	static Cursor cursor;
	XSetWindowAttributes attrs;
	Display *dpy = XtDisplay(GetTopWidget());
	Widget wrk_w = NULL;
	Widget tmp_w;
	Arg args[1];

	/*
	 * Launch a work Dialog and alter the cursor.
	 */
	on ? locked++ : locked--;
	if ( locked >1 || (locked == 1 && on == False) )
		return(NULL);	/* Already locked. */


	if ( ! cursor )
		cursor = XCreateFontCursor(XtDisplay(GetTopWidget()), XC_watch);

	attrs.cursor = on ? cursor : None;


	XChangeWindowAttributes(dpy,XtWindow(GetTopWidget()), CWCursor, &attrs);

	if ( parent && (parent != GetTopWidget()) ) {
		XChangeWindowAttributes( dpy, XtWindow(parent),
				CWCursor, &attrs );

		tmp_w = parent;
		while( ( tmp_w = XtParent( tmp_w )) != GetTopWidget() )
			XChangeWindowAttributes(dpy, XtWindow(tmp_w),
					CWCursor, &attrs);
	}

	if ( on ) {

		Stopped = False;

		XtSetArg( args[0], XmNmessageString, str );
		wrk_w = XmCreateWorkingDialog( parent, "workDialog", args, 1 );

	}

	IntCnt = 0;
	return(wrk_w);
}


/*ARGSUSED*/
Boolean CheckForInterrupt( int cnt )
{

	/*
	 * A work Dialog is launched. Check for interrupt.
	 */
	Display *dpy;
	Window win;
	XEvent event;

	if ( ! InUse || (IntCnt++ < cnt) )
		return(False);

	IntCnt = 0;

	dpy = XtDisplay(GetTopWidget());
	win = XtWindow(MsgW);

	XFlush(dpy);
	XmUpdateDisplay(GetTopWidget());

	while( XCheckMaskEvent( dpy,
			ButtonPressMask | ButtonReleaseMask |
			ButtonMotionMask | PointerMotionMask |
			KeyPressMask | KeyReleaseMask, &event ) ) {
		if ( event.xany.window == win )
			XtDispatchEvent(&event);
	}

	return(Stopped);
}


/*ARGSUSED*/
static void ForceUpdate( Widget w )
{

	/*
	 * Do not return unless the Dialog is mapped on the screen.
	 */

	Widget dia, top;
	Window diaw, topw;
	XtAppContext cxt = XtWidgetToApplicationContext(w);
	Display *dpy;
	XWindowAttributes xwa;
	XEvent event;

	for( dia = w; !XtIsShell(dia); dia = XtParent(dia) )
		;

	for( top = dia; !XtIsTopLevelShell(top); top = XtParent(top) )
		;

	if ( XtIsRealized(dia) && XtIsRealized(top) ) {

		dpy = XtDisplay(top);
		diaw = XtWindow(dia);
		topw = XtWindow(top);

		while( XGetWindowAttributes( dpy, diaw, &xwa ) &&
				xwa.map_state != IsViewable ) {
			if ( XGetWindowAttributes( dpy, topw, &xwa ) &&
					xwa.map_state != IsViewable )
				break;

			XtAppNextEvent( cxt, &event );
			XtDispatchEvent(&event);
		}
	}
	XmUpdateDisplay(top);
}
