/* -*-Mode: C; -*- */

#ifndef _TTY_FREEBSD_INCLUDE
#define _TTY_FREEBSD_INCLUDE

/* $Id: tty_freebsd.i,v 1.3 1995/02/26 14:42:35 me Exp $ */

/*
 * This file is part of the Emu system.
 *
 * Copyright 1990 by PCS Computer Systeme, GmbH. Munich, West Germany.
 * 
 * Copyright 1994 by Jordan K. Hubbard and Michael W. Elbel
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL PCS, THE AUTHORS, OR THEIR HOUSEPETS BE LIABLE FOR
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE. SO DON'T SUE US.
 * THANK YOU.
 */

/*
 * Tty service routines for FreeBSD systems.
 *
 * Author: Michael Elbel
 * Date: Feb 20 1995
 * Description: FreeBSD tty handling
 *
 * Revision History:
 * $Log: tty_freebsd.i,v $
# Revision 1.3  1995/02/26  14:42:35  me
# FreeBSD support enhanced
#
 */

/* Mandatory */
#ifndef _TTY_INCLUDED
#define _TTY_INCLUDED
#endif

#include <utmp.h>
#include <pwd.h>
#include <sys/stat.h>

/* Sets no-blocking I/O on a file descriptor */
void
tty_set_nonblocking(TermWidget w, int fd)
{
     int mode = 1;

     if (ioctl(fd, FIONBIO, &mode) == -1)
	  perror("Couldn't FIONBIO pty");
}

/* Set a new tty size */
void
tty_set_size(TermWidget w, int rows, int cols, int width, int height)
{
     struct winsize wz;
     int pgrp;

     wz.ws_row = rows;
     wz.ws_col = cols;
     wz.ws_xpixel = width;
     wz.ws_ypixel = height;
     ioctl(w->term.slave, TIOCSWINSZ, &wz);
     if (ioctl(w->term.slave, TIOCGPGRP, &pgrp) != -1)
	  killpg(pgrp, SIGWINCH);
}


/* Add a utmp/wtmp entry for a tty */
void
tty_add_utmp(w)
TermWidget w;
{
     struct utmp utmp;
     struct passwd *pw;
     char *name;
     int fd;
     int tty;

     /* who's there? */
     pw = getpwuid(w->term.uid);

     /* we're somebody.. */
     /* or are we? */
     name = (pw && (int) pw->pw_name) ? pw->pw_name : "(unknown)";

     /* copy over name information */
     (void)strncpy(utmp.ut_name, name, sizeof(utmp.ut_name));
     (void)strncpy(utmp.ut_line, w->term.tname + 5, /* 5 is len of "/dev/" */
		   sizeof(utmp.ut_line));

     /* current time */
     (void)time(&utmp.ut_time);

     /* Display connection */
     (void)strncpy(utmp.ut_host, DisplayString(XtDisplay((Widget)w)), 
		   sizeof(utmp.ut_host));

     /* 
      * The rest of the function is shamelessly stolen from 
      * FreeBSD's /usr/src/libutil/login.c
      * I hope it's reasonably portable to other bsd derived systems.
      */

     if ((fd = open(_PATH_UTMP, O_RDWR|O_CREAT, 0644)) >= 0) {
	  if ((tty = ttyslot()) > 0) {
	       (void)lseek(fd, (long)(tty * sizeof(struct utmp)), L_SET);
	       (void)write(fd, (char *)&utmp, sizeof(struct utmp));
	       (void)close(fd);
	  } else {
	       setttyent();
	       for (tty = 0; getttyent(); tty++)
		    ;
	       endttyent();
	       (void)lseek(fd, (long)(tty * sizeof(struct utmp)), L_SET);
	       while (read(fd, (char *)&utmp,
			   sizeof(struct utmp)) == sizeof(struct utmp)) {
		    if (!utmp.ut_name[0]) {
			 (void)lseek(fd, -(long)sizeof(struct utmp), L_INCR);
			 break;
		    }
	       }
	       (void)write(fd, (char *)&utmp, sizeof(struct utmp));
	       (void)close(fd);
	  }
     }
#if 0
     /*
      * I'm not sure, wtmp handling us such a good idea, if you don't want
      * to clog your wtmp with loads of entries
      */
     if ((fd = open(_PATH_WTMP, O_WRONLY|O_APPEND, 0)) >= 0) {
	  (void)write(fd, (char *)&utmp, sizeof(struct utmp));
	  (void)close(fd);
     }
#endif
}

/* Remove a utmp/wtmp entry for a tty */
void
tty_remove_utmp(w)
TermWidget w;
{
     /*
      * Again, about everything is stolen from FreeBSD
      * /usr/src/libutil/logout.c and logwtmp.c
      */
     int fd;
     struct utmp ut;
     off_t lseek();
     time_t time();
     struct stat buf;
     
     if ((fd = open(_PATH_UTMP, O_RDWR)) < 0)
	  return;
     while (read(fd, (char *)&ut, 
		 sizeof(struct utmp)) == sizeof(struct utmp)) {
	  if (!ut.ut_name[0] || strncmp(ut.ut_line, w->term.tname + 5,
					sizeof(ut.ut_line)))
	       continue;
	  bzero(ut.ut_name, sizeof(ut.ut_name));
	  bzero(ut.ut_host, sizeof(ut.ut_host));
	  (void)time(&ut.ut_time);
	  (void)lseek(fd, -(long)sizeof(struct utmp), L_INCR);
	  (void)write(fd, (char *)&ut, sizeof(struct utmp));
     }
     (void)close(fd);

#if 0
     /*
      * wtmp handling commented out for now
      */
     if ((fd = open(_PATH_WTMP, O_WRONLY|O_APPEND, 0)) < 0)
	  return;
     if (fstat(fd, &buf) == 0) {
	  if (write(fd, (char *)&ut, sizeof(struct utmp)) !=
	      sizeof(struct utmp))
	       (void) ftruncate(fd, buf.st_size);
     }
     (void) close(fd);
#endif
     
     return;
}

/*
 * These next three routines complement eachother by getting and setting
 * some number of tty values using the TtyStuff structure.
 */

#define TTYDEFCHARS /* we want the default chars */
#include <termios.h>

typedef struct termios TtyStuff;

/* File descriptors to try */
Local int Fd_try[] = { 0, 1, 2, -1, 0, -1 };

/* If w is NULL, get tty values for invoking tty, not pty */
Generic
tty_get_values(w)
TermWidget w;
{
     register int i, *try;
     static TtyStuff ret;
     Import int errno;

     if (w == NULL)
	  try = Fd_try;
     else {
	  try = Fd_try + 4;
	  try[0] = w->term.slave;
     }
     for (i = 0; try[i] >= 0; i++) {
	  int r;

	  r = tcgetattr(try[i], &ret);
	  if (!r)	/* success? */
	       break;
     }

     if (try[i] == -1)
	  warn("tty_get_values: Couldn't tcgetattr, errno = %d", errno);
     return (Generic)&ret;
}

/* Some termios.h files don't define this, and we need it for a fall-back */
#ifndef CNUL
#define CNUL	'\0'
#endif

/* Must return some "sane" set of hardwired values */
Generic
tty_get_sane(w)
TermWidget w;
{
     static struct termios sane;

     sane.c_iflag = TTYDEF_IFLAG;
     sane.c_oflag = TTYDEF_OFLAG;
     sane.c_cflag = TTYDEF_CFLAG;
     sane.c_lflag = TTYDEF_LFLAG;
     sane.c_ispeed = TTYDEF_SPEED;
     sane.c_ospeed = TTYDEF_SPEED;
     bcopy(ttydefchars, sane.c_cc, sizeof(sane.c_cc));
     return (Generic)&sane;
}

/* Whap some saved values onto the tty */
void
tty_set_values(w, val)
TermWidget w;
Generic val;
{

     register int i, *try;
     Import int errno;

     if (w == NULL)
	  try = Fd_try;
     else {
	  try = Fd_try + 4;
	  try[0] = w->term.slave;
     }
     for (i = 0; try[i] >= 0; i++) {
	  int r;
	  r = tcsetattr(try[i], TCSADRAIN, (struct termios *)val);
	  if (!r)	/* success? */
	       break;
     }
     if (try[i] == -1)
	  warn("tty_set_values: Couldn't tcsetattr, errno = %d.", errno);
}

Export String
tty_find_pty(w)
TermWidget w;
{
     char ttyname[80];
     struct stat st;
     register char *c1, *c2;
     Import void tty_set_nonblocking();
 
     /* Use user provided search loop or our own to grub through ptys */
#ifdef PTY_ITERATION
     PTY_ITERATION
#else
     for (c1 = PTYCHAR1; *c1; ++c1) {
	  for (c2 = PTYCHAR2; *c2; ++c2) {
#endif
	       /* Go for the master side */
	       sprintf(ttyname, PTYDEV, *c1, *c2);
#ifdef DEBUG
	       debug("unix_tty_find_pty: Trying to open %s", ttyname);
#endif
	       if (stat(ttyname, &st) < 0)
		    continue;
	       if ((w->term.master = open(ttyname, O_RDWR, 0)) >= 0) {
		    /*
		     * Now make sure that the slave side is available.
		     * This allows for a bug in some versions of rlogind
		     * and gives us a handle for ioctl() ops.
		     */
		    sprintf(ttyname, TTYDEV, *c1, *c2);
#ifdef DEBUG
		    debug("unix_tty_find_pty: Checking slave side %s",
			  ttyname);
#endif
		    if ((w->term.slave = open(ttyname, O_RDWR, 0)) >= 0) {
			 /* All is casual, dude. */
#ifdef DEBUG
			 debug("unix_tty_find_pty: Found pty %s", ttyname);
#endif
#ifdef NOTDEF
			 /*
			  * This probably was the single reason, emu
			  * didn't work on BSD systems
			  */
			 tty_set_nonblocking(w, w->term.slave);
#endif
			 return XtNewString(ttyname);
                    }
		    else
			 close(w->term.master);
	       }
	  }
     }
     pmessage("No free ptys.  Sorry mate.");
     return NULL;
}

#endif	/* _TTY_FREEBSD_INCLUDE */
