/**
*** XPG - Graphical User Interface for Postgres
*** Copyright (C) 1993  Ranen Goren (ranen@cs.huji.ac.il).

*** This program is free software; you can redistribute it and/or modify
*** it under the terms of the GNU General Public License as published by
*** the Free Software Foundation; either version 2 of the License, or
*** (at your option) any later version.

*** This program is distributed in the hope that it will be useful,
*** but WITHOUT ANY WARRANTY; without even the implied warranty of
*** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*** GNU General Public License for more details.

*** You should have received a copy of the GNU General Public License
*** along with this program; if not, write to the Free Software
*** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
**/


#include <stdio.h>
#include <varargs.h>
#include <X11/StringDefs.h>
#include <X11/Intrinsic.h> 
#include <X11/cursorfont.h>
#include <Xm/Xm.h>
#include <Xm/MessageB.h>
#include <Xm/Form.h>
#include <Xm/Label.h>
#include <Xm/PushB.h>
#include <Xm/DialogS.h>
#include <Xm/PanedW.h>
#include "lib.h"

#ifdef MEM_DEBUG
#include "/CS/system/ranen/Src/Lib/Malloc/malloc.h"
#endif

int setCursor();
void warn();
void warnOk();
void insertCallback();
Pixmap createIcon();
void paneSetFixSize();

/*extern*/ XtAppContext app;

void dummy();

Widget toplevel;



/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/* climb widget tree until we get to the top.  Return the Shell */
Widget GetTopShell(w)
Widget w;
{
    while (w && !XtIsWMShell(w))
        w = XtParent(w);
    return w;
}





/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/* changes the cursor shape related to <widget>, to <shape>. 
   If shape == None, changes the cursor to the default cursor */

int setCursor(widget, shape)
  Widget        widget;
  unsigned int  shape;
{  
    XSetWindowAttributes attrs;
    
    if (XtWindow(widget) == 0)
	return 1;                     /* no window for widget! */   
    if (shape != None)
	attrs.cursor = XCreateFontCursor(XtDisplay(widget), shape);
    else
	attrs.cursor = None;
    XChangeWindowAttributes(XtDisplay(widget), XtWindow(widget),
			    CWCursor, &attrs);
    XFlush(XtDisplay(widget));
    return 0;
}    





/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/* creates and pops-up a warning dialog (with a single button - "Ok").
   The message is specified by <message>, and the dialog is created as a 
   child of the widget <parent>.                                          */

void warn(parent, va_alist)
  Widget parent;
  va_dcl
{
    char *format;
    va_list list;
    char buf[BUFSIZ];
    Arg wargs[10];
    int n;
    Widget   warnDialog;
    XmString compStr;

    va_start(list);
    format = va_arg(list, char *);
    vsprintf(buf, format, list);
    compStr = XmStringCreateLtoR(buf, XmSTRING_DEFAULT_CHARSET);
    n=0;
    XtSetArg(wargs[n], XmNdialogStyle, XmDIALOG_FULL_APPLICATION_MODAL);  n++;
    XtSetArg(wargs[n], XmNmessageString, compStr);                        n++;
    warnDialog = XmCreateErrorDialog(parent, "warnDialog", wargs, n);
    XtUnmanageChild(XmMessageBoxGetChild(warnDialog, XmDIALOG_HELP_BUTTON));
    XtUnmanageChild(XmMessageBoxGetChild(warnDialog, XmDIALOG_CANCEL_BUTTON));
    XtAddCallback(warnDialog, XmNokCallback, warnOk, NULL);
    XtManageChild(warnDialog);
    XmStringFree(compStr);
    va_end(args);
}





/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/* (callback) the "ok" button callback of the warnDialog widget. Should only
   destroy warnDialog (passed as <w>), as each call to warn() creates a new
   dialog.   */

void warnOk(w, clientData, cbs)
  Widget w;
  XtPointer clientData, cbs;
{
    XtDestroyWidget(w);
}





/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/* insert a callback to the beginning of a callback list,
   ie. a callback which will be executed first.
   The function interface is similar to XtAddCallback()     */

void insertCallback(w, callback_name, callback, client_data)
  Widget w;
  String callback_name;
  XtCallbackProc callback;
  XtPointer client_data;
{
    XtCallbackList list, ptr;
    XtCallbackRec  prevRec;
    int n;
    
    if (!w)
	return;
    XtAddCallback(w, callback_name, callback, client_data);
    XtVaGetValues(w, callback_name, &list, NULL);
    for (ptr=list+1, n=0;  (*ptr).callback != NULL;  ptr++, n++)
    {
	prevRec = *ptr;
	*ptr = *(ptr-1);
    }
    if (ptr != list+1)   /* true if list contained elements before calling */
	*list = prevRec;   /* last assignment to it was of the LAST
			      callback, ie. the one to be inserted */
    /* printf("cb had %d\n", n); */
}





/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/* calcs the position (posX,posY) which is related to the widget w:
     posX = ax *  width_of_w  +  bx  +  x_coor_of_w
     posY = ay * height_of_w  +  by  +  y_coor_of_w
   returns posX and posY using their passed pointers  */

widgetPos(w, posX, posY, ax, bx, ay, by)
  Widget w;
  Position *posX, *posY;
  float ax, ay;
  int   bx, by;
{
    Dimension width, height;
    
    XtVaGetValues(w,
		  XtNx,      posX,
		  XtNy,      posY,
		  XtNwidth,  &width,
		  XtNheight, &height,
		  NULL);
    *posX += ax*width  + bx;
    *posY += ay*height + by;
}    





/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/* sets the (x,y) placement of widget w1 to be relative to w2,
   as calculated by widgetPos() - see above */

setWidgetPos(w1, w2, ax, bx, ay, by)
  Widget w1, w2;
  float ax, ay;
  int   bx, by;
{
    Position posX, posY;
    
    widgetPos(w2, &posX, &posY, ax, bx, ay, by);
    XtVaSetValues(w1, XtNx, posX, XtNy, posY, NULL);
}





/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/* sets the already created children of a form widget in a column form.
   The last widget to be attached is the 1st created (if dir>=0) or the last
   otherwise. */

setFormColumn(w, dir)
  Widget w;
  int dir;      /* dir>=0 for top-down setting, dir<0 for bottom-up. */
{
    int num;
    WidgetList list;
    int i, first, stop, inc;
    Widget prev;
    
    XtVaGetValues(w, 
		  XmNnumChildren, &num,
		  XmNchildren,    &list,
		  NULL);
    if (num == 0)
	return;
    if (dir >= 0)
    {
	first = 0;
	stop = num;
	inc = 1;
    }
    else
    {
	first = num-1;
	stop = -1;
	inc = -1;
    }
    prev = NULL;
    for (i=first;  i!=stop;  i+=inc)
    {
	if (dir >= 0)
	    XtVaSetValues(list[i],
			  XmNtopAttachment, 
			    (prev ? XmATTACH_WIDGET:XmATTACH_FORM),
			  XmNtopWidget, prev,
			  XmNrightAttachment,  XmATTACH_FORM,
			  XmNleftAttachment,   XmATTACH_FORM,
			  XmNbottomAttachment, XmATTACH_NONE,
			  NULL);
	else
	    XtVaSetValues(list[i],
			  XmNbottomAttachment, 
			    (prev ? XmATTACH_WIDGET:XmATTACH_FORM),
			  XmNbottomWidget, prev,
			  XmNtopAttachment,    XmATTACH_NONE,
			  XmNrightAttachment,  XmATTACH_FORM,
			  XmNleftAttachment,   XmATTACH_FORM,
			  NULL);
	prev = list[i];
    }
    if (dir >= 0)
	XtVaSetValues(list[num-1], XmNbottomAttachment, XmATTACH_FORM, NULL);
    else
	XtVaSetValues(list[0], XmNtopAttachment, XmATTACH_FORM, NULL);
}





/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/* (by Dan Heller)
 * yesNo() -- a generalized routine that asks the user a question
 * and returns a response.  Parameters are: the question, the labels
 * for the "Yes" and "No" buttons, and the default selection to use.
 */
int yesNo(parent, question, ans1, ans2, default_ans)
  char *question, *ans1, *ans2;
  int default_ans;
{
    Widget dialog;
    XmString text, yes, no;
    static int answer = 0;
    int tmpAnswer;
    extern void yesNoResponse();

    dialog = XmCreateQuestionDialog(parent, "dialog", NULL, 0);
    XtVaSetValues(dialog,
		  XmNdialogStyle,        XmDIALOG_FULL_APPLICATION_MODAL,
		  XmNdeleteResponse,     XmDESTROY,
		  NULL);
    XtUnmanageChild(XmMessageBoxGetChild(dialog, XmDIALOG_HELP_BUTTON));
    XtAddCallback(dialog, XmNokCallback, yesNoResponse, &answer);
    XtAddCallback(dialog, XmNcancelCallback, yesNoResponse, &answer);

    text = XmStringCreateLtoR(question, XmSTRING_DEFAULT_CHARSET);
    yes = XmStringCreateLtoR(ans1, XmSTRING_DEFAULT_CHARSET);
    no = XmStringCreateLtoR(ans2, XmSTRING_DEFAULT_CHARSET);
    XtVaSetValues(dialog,
		  XmNmessageString,      text,
		  XmNokLabelString,      yes,
		  XmNcancelLabelString,  no,
		  XmNdefaultButtonType,  default_ans == 1 ?
		  XmDIALOG_OK_BUTTON : XmDIALOG_CANCEL_BUTTON,
		  NULL);
    XmStringFree(text);
    XmStringFree(yes);
    XmStringFree(no);
    XtManageChild(dialog);
    
    while (answer == 0) {
        XtAppProcessEvent(app, XtIMAll);
        XSync(XtDisplay(dialog), 0);
    }
    
    XtUnmanageChild(dialog);
    XSync(XtDisplay(dialog), 0);
    XmUpdateDisplay(dialog);
    
    tmpAnswer = answer;
    answer = 0;

    return tmpAnswer;
}





/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/* yesNoResponse() --The user made some sort of response to the
 * question posed in yesNo().  Set the answer (client_data)
 * accordingly.
 */
void yesNoResponse(w, answer, cbs)
  Widget w;
  int *answer;
  XmAnyCallbackStruct *cbs;
{
    if (cbs->reason == XmCR_OK)
        *answer = 1;
    else if (cbs->reason == XmCR_CANCEL)
        *answer = 2;
}





/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/* based on Heller's function, with modifications */
#define TIGHTNESS 20

Widget CreateActionArea(parent, actions, num_actions)
  Widget parent;
  ActionAreaItem *actions;
  int num_actions;
{
    Widget action_area, widget, defaultBut;
    Dimension height, h, maxHeight=0;
    int i;
    char wname[BUFSIZ];
    XmString compStr;
    
    action_area = XtVaCreateWidget("action_area", xmFormWidgetClass, parent,
				   XmNfractionBase, TIGHTNESS*num_actions - 1,
				   XmNleftOffset,   10,
				   XmNrightOffset,  10,
				   NULL);
    
    for (i = 0; i < num_actions; i++) {
	sprintf(wname, "actionAreaButton_%d", i);
	compStr = XmStringCreateLtoR(actions[i].label, 
				     XmSTRING_DEFAULT_CHARSET);
        widget = XtVaCreateManagedWidget(wname,
		 xmPushButtonWidgetClass, action_area,
		 XmNlabelString,          compStr,
		 XmNleftAttachment,       i? XmATTACH_POSITION : XmATTACH_FORM,
		 XmNleftPosition,         TIGHTNESS*i,
		 XmNtopAttachment,        XmATTACH_FORM,
		 XmNbottomAttachment,     XmATTACH_FORM,
		 XmNrightAttachment,
		     i != num_actions-1? XmATTACH_POSITION : XmATTACH_FORM,
		 XmNrightPosition,        TIGHTNESS*i + (TIGHTNESS-1),
                 XmNshowAsDefault,        actions[i].isDefault,
		 XmNsensitive,            actions[i].isSensitive,
                 XmNdefaultButtonShadowThickness, 1,
                 NULL);
	XmStringFree(compStr);
        if (actions[i].callback)
            XtAddCallback(widget, XmNactivateCallback,
			  actions[i].callback, actions[i].data);

	/* set the pane window constraint for max and min heights 
	 * so this particular pane in the PanedWindow is not resizable.
	 * we always reset this value when a taller button is found.
	 */
	XtVaGetValues(widget, XmNheight, &height, NULL);
	if (height > maxHeight)
	{
	    maxHeight = height;
	    XtVaGetValues(action_area, XmNmarginHeight, &h, NULL);
	    height += 2 * h;
	    XtVaSetValues(action_area,
			  XmNpaneMaximum,   height,
			  XmNpaneMinimum,   height,
			  NULL);
	}
	
	/* Set the action_area's default button */
        if (actions[i].isDefault)
            XtVaSetValues(action_area, XmNdefaultButton, widget, NULL);

	actions[i].widget = widget;
    }

    XtManageChild(action_area);

    return action_area;
}





/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/* 
 * askUser() -- a generalized routine that creates a dialog, using 
 * createCustomMessageDialog(), and loops on the dialog's action buttons'
 * events. When a button is pressed, the dialog is destroyed, and the
 * button's number (in the action area) is returned by askUser().
 * askUser() registers its response-callback (askUserResponse()) *before*
 * the callbacks specified in the actionAreaItem, and that's the reason
 * to the 'strange' malloc in this function.
 */
askUser(parent, message, pixBits, pixWidth, pixHeight, actions, num_actions)
  Widget parent;
  char *message;
  char *pixBits;
  int  pixWidth, pixHeight;
  ActionAreaItem *actions;
  int num_actions;
{
    int answer = -1;
    int i;
    extern void askUserResponse();
    ActionAreaItem *preAction;

    preAction = 
	(ActionAreaItem *)XtMalloc(sizeof(ActionAreaItem) * num_actions);
    memcpy(preAction, actions, sizeof(ActionAreaItem)*num_actions);
    for (i=0; i<num_actions; i++)
    {
	preAction[i].callback = askUserResponse;
	preAction[i].data     = (caddr_t)(&answer);
    }	
    createCustomMessageDialog(parent, message, pixBits, pixWidth, pixHeight, 
			      preAction, num_actions);
    for (i=0; i<num_actions; i++)
    {
	XtAddCallback(preAction[i].widget, XmNactivateCallback, 
		      actions[i].callback, actions[i].data);
	actions[i].widget = preAction[i].widget;
    }	
    XtFree(preAction);

    while (answer == -1) {
        XtAppProcessEvent(app, XtIMAll);
        XSync(XtDisplay(parent), 0);
    }
    XSync(XtDisplay(parent), 0);
    XmUpdateDisplay(parent);
    return answer;
}





/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/* askUserResponse() --The user made some sort of response to the
 * message posed in askUser().  Set the answer (client_data)
 * accordingly.
 */
void askUserResponse(w, answer, cbs)
Widget w;
int *answer;
XmAnyCallbackStruct *cbs;
{
    WidgetList list;
    int num;
    Widget sh, parent;

    XtVaGetValues(XtParent(w), 
		  XmNnumChildren, &num,
		  XmNchildren,    &list,
		  NULL);
    if (num == 0)
	return;
    while (num)
	if (list[--num] == w)
	    *answer = num;
    sh = GetTopShell(w);
    parent = XtParent(sh);
    XtDestroyWidget(sh);
    XSync(XtDisplay(parent), 0);
    XmUpdateDisplay(parent);
    printf("dia dead\n");
}





/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/* 
 *  Builds a custom message dialog, consisting of a text label,
 *  a pixmap, and an action-area. (if pixBits is NULL, no pixmap
 *  is created).
 */
createCustomMessageDialog(parent, message, pixBits, pixWidth, pixHeight, 
			  actions, num_actions)
  Widget parent;
  char *message;
  char *pixBits;
  int  pixWidth, pixHeight;
  ActionAreaItem *actions;
  int num_actions;
{
    Widget dialog, pane, form, pixW;
    XmString text;
    int i;
    Arg args[10];
    Pixmap pix;

    dialog = XtCreatePopupShell("msgDialog", xmDialogShellWidgetClass, 
				parent, NULL, 0);
    XtVaSetValues(dialog,
		  XmNdialogStyle,        XmDIALOG_FULL_APPLICATION_MODAL,
		  XmNdeleteResponse,     XmDESTROY,
		  NULL);
    pane = XtVaCreateWidget("msgPane", xmPanedWindowWidgetClass, dialog, 
			    XmNsashWidth,  1,
			    XmNsashHeight, 1,
			    NULL);
    form = XmCreateForm(pane, "msgForm", NULL, 0);
    if (pixBits != NULL)
    {
	pix = createIcon(pane, pixBits, pixWidth, pixHeight);
	pixW = XtVaCreateManagedWidget("msgPix", xmLabelWidgetClass, form,
				       XmNlabelType,        XmPIXMAP,
				       XmNlabelPixmap,      pix,
				       XmNleftAttachment,   XmATTACH_FORM,
				       XmNtopAttachment,    XmATTACH_FORM,
				       XmNbottomAttachment, XmATTACH_FORM,
				       NULL);
    }
    text = XmStringCreateLtoR(message, XmSTRING_DEFAULT_CHARSET);
    XtVaCreateManagedWidget("msgText", xmLabelWidgetClass, form,
			    XmNlabelString,      text,
			    XmNrightAttachment,  XmATTACH_FORM,
			    XmNtopAttachment,    XmATTACH_FORM,
			    XmNbottomAttachment, XmATTACH_FORM,
			    XmNleftAttachment,   
			        (pixBits ? XmATTACH_WIDGET : XmATTACH_FORM),
			    XmNleftWidget,       pixW,
			    NULL);
    XmStringFree(text);    
    XtManageChild(form);
    CreateActionArea(pane, actions, num_actions);
    XtManageChild(pane);
    XtManageChild(dialog);
}





/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
/*
main(argc, argv)
  int argc;
  char *argv[];
{
    int ans;
    
    toplevel = XtAppInitialize(&app, "Xpg", NULL, 0, 
                               &argc, argv, NULL, NULL, 0);
    XtVaCreateManagedWidget("label", xmLabelWidgetClass, toplevel, NULL);
    XtRealizeWidget(toplevel);

    askUser(toplevel, "What... is your favourite color", 
	    NULL, error_width, error_height, 
	    actions, XtNumber(actions));
    XtAppMainLoop(app);
    printf("Answer: %d\n", ans);
}
*/




/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
Pixmap createIcon(w, data, width, height)
  Widget w;
  char *data;
  unsigned int width, height;
{
    Pixmap result;
    Pixel fg, bg;
    
    XtVaGetValues(w, 
                  XmNforeground, &fg,
                  XmNbackground, &bg, 
                  NULL);
    result = XCreatePixmapFromBitmapData(XtDisplay(toplevel), 
                                RootWindowOfScreen(XtScreen(toplevel)),
                                data, width, height, fg, bg, 
                                DefaultDepthOfScreen(XtScreen(toplevel)));
    return result;
}





/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
char *skipSpaces(s)
    char *s;
{
    while (*s==' ' || *s=='\t')
	s++;
    return s;
}





/*%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%*/
void paneSetFixSize(paneW)
  Widget paneW;
{
    Dimension height;
    
    XtVaGetValues(paneW, XmNheight, &height, NULL);
    XtVaSetValues(paneW,
		  XmNpaneMaximum, height,
		  XmNpaneMinimum, height,
		  NULL);
}
