/*
 *   xmcd - Motif(tm) CD Audio Player
 *
 *   Copyright (C) 1994  Ti Kan
 *   E-mail: ti@amb.org
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
#ifndef LINT
static char *_os_svr4_c_ident_ = "@(#)os_svr4.c	2.9 94/01/21";
#endif

#include <Xm/Xm.h>
#include "xmcd.h"
#include "ac_util.h"
#include "ac_cdfunc.h"
#include "di_scsipt.h"

#ifndef OSI_VERS
#define OSI_VERS	"1.1"		/* Version */
#endif

#if defined(SVR4) && !defined(sun) && !defined(SIMULATED_CDROM)

extern AppData		app_data;
extern bool_t		notrom_error;

STATIC int		fd = -1;	/* Passthrough device file desc */


#ifdef i386
/*
 *   Intel x86 UNIX SVR4 support
 *   Portable Device Interface/SCSI Device Interface
 *
 *   This software fragment contains code that interfaces xmcd to
 *   the UNIX System V Release 4 operating system for the Intel
 *   x86 hardware platforms from UNIX System Laboratories.
 *   The name "USL", "UNIX" and "Intel" are used here for
 *   identification purposes.  This software and its author are
 *   not affiliated with USL or Intel.
 */


STATIC char		ptpath[FILE_PATH_SZ] = { '\0' };
					/* Passthrough device path */
STATIC req_sense_data_t	sense_data;	/* Request sense data buffer */


/*
 * pthru_send
 *	Build SCSI CDB and sent command to the device.
 *
 * Args:
 *	opcode - SCSI command opcode
 *	addr - The "address" portion of the SCSI CDB
 *	buf - Pointer to data buffer
 *	size - Number of bytes to transfer
 *	rsvd - The "reserved" portion of the SCSI CDB
 *	length - The "length" portion of the SCSI CDB
 *	param - The "param" portion of the SCSI CDB
 *	control - The "control" portion of the SCSI CDB
 *	rw - Data transfer direction flag (READ_OP or WRITE_OP)
 *
 * Return:
 *	TRUE - command completed successfully
 *	FALSE - command failed
 */
bool_t
pthru_send(
	byte_t		opcode,
	word32_t	addr,
	byte_t		*buf,
	word32_t	size,
	byte_t		rsvd,
	word32_t	length,
	byte_t		param,
	byte_t		control,
	byte_t		rw
)
{
	struct sb	sb;
	struct scb	*scbp;
	union scsi_cdb	cdb;
	
	if (fd < 0 || notrom_error)
		return(FALSE);

	memset(&sense_data, 0, sizeof(sense_data));
	memset(&sb, 0, sizeof(sb));
	memset(&cdb, 0, sizeof(cdb));

	sb.sb_type = ISCB_TYPE;
	scbp = &sb.SCB;

	/* set up SCSI CDB */
	switch (opcode & 0xf0) {
	case 0xa0:
	case 0xe0:
		/* 12-byte commands */
		cdb.scl.sl_op = opcode;
		cdb.scl.sl_res1 = param;
		cdb.scl.sl_lun = 0;
		CDB12_BLK(&cdb.scl, bswap32(addr));
		CDB12_LEN(&cdb.scl, bswap32(length));
		CDB12_RSV(&cdb.scl, rsvd);
		CDB12_CTL(&cdb.scl, control);

		scbp->sc_cmdpt = (caddr_t) SCL_AD(&cdb);
		scbp->sc_cmdsz = SCL_SZ;
		break;

	case 0xc0:
	case 0xd0:
	case 0x20:
	case 0x30:
	case 0x40:
		/* 10-byte commands */
		cdb.scm.sm_op = opcode;
		cdb.scm.sm_res1 = param;
		cdb.scm.sm_lun = 0;
		CDB10_BLK(&cdb.scm, bswap32(addr));
		CDB10_LEN(&cdb.scm, bswap16((word16_t) length));
		CDB10_RSV(&cdb.scm, rsvd);
		CDB10_CTL(&cdb.scm, control);

		scbp->sc_cmdpt = (caddr_t) SCM_AD(&cdb);
		scbp->sc_cmdsz = SCM_SZ;
		break;

	case 0x00:
	case 0x10:
		/* 6-byte commands */
		cdb.scs.ss_op = opcode;
		cdb.scs.ss_addr1 = param;
		cdb.scs.ss_lun = 0;
		CDB6_BLK(&cdb.scs, bswap16((word16_t) addr));
		CDB6_LEN(&cdb.scs, (byte_t) length);
		CDB6_CTL(&cdb.scs, control);

		scbp->sc_cmdpt = (caddr_t) SCS_AD(&cdb);
		scbp->sc_cmdsz = SCS_SZ;
		break;

	default:
		if (app_data.scsierr_msg)
			fprintf(stderr, "0x%02x: Unknown SCSI opcode\n",
				opcode);
		return(FALSE);
	}

	DBGDUMP("SCSI CDB bytes", (byte_t *) scbp->sc_cmdpt, scbp->sc_cmdsz);

	/* set up scsicmd */
	scbp->sc_datapt = (caddr_t) buf;
	scbp->sc_datasz = size;
	scbp->sc_mode = (rw == READ_OP) ? SCB_READ : SCB_WRITE;

	/* Send the command down via the "pass-through" interface */
	if (ioctl(fd, SDI_SEND, &sb) < 0) {
		perror("SDI_SEND ioctl failed");
		return(FALSE);
	}

	if (scbp->sc_comp_code != SDI_ASW) {
		if (opcode != OP_S_TEST && app_data.scsierr_msg) {
			fprintf(stderr, "%s: %s %s:\n%s=0x%x %s=0x%x %s=0x%x",
				PROGNAME,
				"SCSI command fault on",
				app_data.device,
				"Opcode",
				opcode,
				"Completion_code",
				scbp->sc_comp_code,
				"Target_status",
				scbp->sc_status);
		}

		/* Send Request Sense command */
		cdb.scs.ss_op = OP_S_RSENSE;
		cdb.scs.ss_addr1 = 0;
		cdb.scs.ss_lun = 0;
		CDB6_BLK(&cdb.scs, 0);
		CDB6_LEN(&cdb.scs, SZ_RSENSE);
		CDB6_CTL(&cdb.scs, 0);
		scbp->sc_datapt = (caddr_t) &sense_data;
		scbp->sc_datasz = SZ_RSENSE;
		scbp->sc_mode = SCB_READ;
		scbp->sc_cmdpt = (caddr_t) SCS_AD(&cdb);
		scbp->sc_cmdsz = SCS_SZ;

		if (ioctl(fd, SDI_SEND, &sb) < 0 || sense_data.valid == 0) {
			if (opcode != OP_S_TEST && app_data.scsierr_msg)
				fprintf(stderr, "\n");
		}
		else if (opcode != OP_S_TEST && app_data.scsierr_msg) {
			fprintf(stderr, " Key=0x%x Code=0x%x Qual=0x%x\n",
				sense_data.key,
				sense_data.code,
				sense_data.qual);
		}

		return(FALSE);
	}
	return(TRUE);
}


/*
 * pthru_open
 *	Open SCSI passthrough device
 *
 * Args:
 *	path - device path name string
 *
 * Return:
 *	TRUE - open successful
 *	FALSE - open failed
 */
bool_t
pthru_open(char *path)
{
	int		devfd;
	dev_t		ptdev;
	struct stat	stbuf;
	char		errstr[STR_BUF_SZ];

	/* Check for validity of device node */
	if (stat(path, &stbuf) < 0) {
		sprintf(errstr, app_data.str_staterr, path);
		cd_fatal_popup(app_data.str_fatal, errstr);
		return(FALSE);
	}
	if ((stbuf.st_mode & S_IFMT) != S_IFCHR) {
		sprintf(errstr, app_data.str_noderr, path);
		cd_fatal_popup(app_data.str_fatal, errstr);
		return(FALSE);
	}

	/* Check for another copy of xmcd running on the same
	 * CD-ROM device.
	 */
	if (!cd_devlock(path))
		return(FALSE);

	/* Open CD-ROM device */
	if ((devfd = open(path, O_RDONLY)) < 0)
		return(FALSE);

	/* Get passthrough interface device number */
	if (ioctl(devfd, B_GETDEV, &ptdev) < 0) {
		close(devfd);
		return(FALSE);
	}

	close(devfd);

	/* Make passthrough interface device node */
	sprintf(ptpath, "/tmp/.cdpt.%x", ptdev);

	if (mknod(ptpath, S_IFCHR | 0700, ptdev) < 0) {
		if (errno == EEXIST) {
			unlink(ptpath);

			if (mknod(ptpath, S_IFCHR | 0700, ptdev) < 0)
				return(FALSE);
		}
		else
			return(FALSE);
	}

	/* Open passthrough device node */
	if ((fd = open(ptpath, O_RDONLY)) < 0)
		return(FALSE);

	return(TRUE);
}


/*
 * pthru_close
 *	Close SCSI passthrough device
 *
 * Args:
 *	Nothing.
 *
 * Return:
 *	Nothing.
 */
void
pthru_close(void)
{
	if (fd >= 0) {
		close(fd);
		fd = -1;
	}

	if (ptpath[0] != '\0')
		unlink(ptpath);
}


/*
 * pthru_vers
 *	Return OS Interface Module version string
 *
 * Args:
 *	Nothing.
 *
 * Return:
 *	Module version text string.
 */
char *
pthru_vers(void)
{
	static char	vers[STR_BUF_SZ];

	sprintf(vers, "OS Interface module v%s (for UNIX SVR4-PDI/x86)\n",
		OSI_VERS);
	return(vers);
}

#endif	/* i386 */

#ifdef MOTOROLA
/*
 *   Motorola 88k UNIX SVR4 support
 *
 *   Contributing author: Mark Scott
 *   E-mail: mscott@urbana.mcd.mot.com
 *
 *   Note: Audio CDs sometimes produce "Blank check" warnings on the console, 
 *         just ignore these.
 *
 *   This software fragment contains code that interfaces xmcd to
 *   the System V Release 4 operating system from Motorola.
 *   The name "Motorola" is used here for identification purposes.
 */


/*
 * pthru_send
 *	Build SCSI CDB and sent command to the device.
 *
 * Args:
 *	opcode - SCSI command opcode
 *	addr - The "address" portion of the SCSI CDB
 *	buf - Pointer to data buffer
 *	size - Number of bytes to transfer
 *	rsvd - The "reserved" portion of the SCSI CDB
 *	length - The "length" portion of the SCSI CDB
 *	param - The "param" portion of the SCSI CDB
 *	control - The "control" portion of the SCSI CDB
 *	rw - Data transfer direction flag (READ_OP or WRITE_OP)
 *
 * Return:
 *	TRUE - command completed successfully
 *	FALSE - command failed
 */
bool_t
pthru_send(
	byte_t		opcode,
	word32_t	addr,
	byte_t		*buf,
	word32_t	size,
	byte_t		rsvd,
	word32_t	length,
	byte_t		param,
	byte_t		control,
	byte_t		rw
)
{
	char			scsistat = '\0',
				*tmpbuf;
	int			cdb_l = 0,
				i;
	long			residual = 0L;
	unsigned long		errinfo  = 0L,
				ccode = 0L;
	struct scsi_pass	spass,
				*sp = &spass;
	struct ext_sense	sense,
				*esp = &sense;

	if (fd < 0 || notrom_error)
		return(FALSE);

	/* Zero out the struct */
	memset(sp, 0, sizeof(struct scsi_pass));
	memset(esp, 0, sizeof(struct ext_sense));

	/* Setup passthru structure */
	sp->resid = &residual;
	sp->sense_data = esp;
	sp->status = &scsistat;
	sp->error_info = &errinfo;
	sp->ctlr_code = &ccode;
	sp->xfer_len = (unsigned long) size;

	/* Align on a page boundary */
	tmpbuf = NULL;
	if (sp->xfer_len > 0) {
		tmpbuf = (char *) MEM_ALLOC(2 * NBPP);
		sp->data = tmpbuf;
		sp->data += NBPP - ((unsigned int) sp->data & (NBPP - 1));
	}
	else
		sp->data = tmpbuf;


	if (rw == WRITE_OP && sp->xfer_len > 0)	/* Write operation */
		memcpy(sp->data, buf, sp->xfer_len);

	/* Set up SCSI CDB */
	switch (opcode & SPT_CDB_LEN) {
	case 0xa0:
	case 0xe0:
		/* 12-byte commands */
		cdb_l = 0xc0;
		sp->cdb[0] = opcode;
		sp->cdb[1] = param;
		sp->cdb[2] = (addr >> 24) & 0xff;
		sp->cdb[3] = (addr >> 16) & 0xff;
		sp->cdb[4] = (addr >> 8) & 0xff;
		sp->cdb[5] = (addr & 0xff);
		sp->cdb[6] = (length >> 24) & 0xff;
		sp->cdb[7] = (length >> 16) & 0xff;
		sp->cdb[8] = (length >> 8) & 0xff;
		sp->cdb[9] = length & 0xff;
		sp->cdb[10] = rsvd;
		sp->cdb[11] = control;
		break;

	case 0xc0:
	case 0xd0:
	case 0x20:
	case 0x30:
	case 0x40:
		/* 10-byte commands */
		cdb_l = 0xa0;
		sp->cdb[0] = opcode;
		sp->cdb[1] = param;
		sp->cdb[2] = (addr >> 24) & 0xff;
		sp->cdb[3] = (addr >> 16) & 0xff;
		sp->cdb[4] = (addr >> 8) & 0xff;
		sp->cdb[5] = addr & 0xff;
		sp->cdb[6] = rsvd;
		sp->cdb[7] = (length >> 8) & 0xff;
		sp->cdb[8] = length & 0xff;
		sp->cdb[9] = control;
		break;

	case 0x00:
	case 0x10:
		/* 6-byte commands */
		cdb_l = 0x60;
		sp->cdb[0] = opcode;
		sp->cdb[1] = param;
		sp->cdb[2] = (addr >> 8) & 0xff;
		sp->cdb[3] = addr & 0xff;
		sp->cdb[4] = length & 0xff;
		sp->cdb[5] = control;
		break;

	default:
		if (app_data.scsierr_msg)
			fprintf(stderr, "0x%02x: Unknown SCSI opcode\n",
				opcode);
		if (tmpbuf != NULL)
			MEM_FREE(tmpbuf);

		return(FALSE);
	}


	/* Check CDB length & flags */

	if (!SPT_CHK_CDB_LEN(cdb_l))
		fprintf(stderr, "%d: invalid CDB length\n", cdb_l);

	sp->flags = cdb_l | SPT_ERROR_QUIET;
	if (rw == READ_OP)
		sp->flags |= SPT_READ;

	if (SPT_CHK_FLAGS(cdb_l))
		fprintf(stderr, "0x%2x: bad CDB flags\n", sp->flags);

	DBGDUMP("SCSI CDB bytes", (byte_t *) sp->cdb, cdb_l);

	/* Send the command down via the "pass-through" interface */
	if (ioctl(fd, DKPASSTHRU, sp) < 0) {
		if (opcode != OP_S_TEST)
			perror("DKPASSTHRU ioctl failed");

		if (tmpbuf != NULL)
			MEM_FREE(tmpbuf);

		return(FALSE);
	}

	if (*sp->error_info != SPTERR_NONE) {
		if (*sp->error_info != SPTERR_SCSI &&
		    *sp->status != 2 &&
		    opcode != OP_S_TEST) {
			/* Request Sense is done automatically by the driver */

			fprintf(stderr, "%s %s\n",
				"xmcd SCSI command fault on",
				app_data.device);

			fprintf(stderr,
				"\nxfer_len=%d errinfo=%d ctlr_code=%d ",
				sp->xfer_len,
				*sp->error_info,
				*sp->ctlr_code);
			fprintf(stderr, "status=%d resid=%d\n",
				*sp->status,
				*sp->resid);
		}
		if (tmpbuf != NULL)
			MEM_FREE(tmpbuf);

		return(FALSE);
	}

	/* pass the data back to caller */
	if (sp->xfer_len > 0 && rw == READ_OP)	/* read operation */
		memcpy(buf, sp->data, sp->xfer_len);

	if (tmpbuf != NULL)
		MEM_FREE(tmpbuf);

	return(TRUE);
}


/*
 * pthru_open
 *	Open SCSI passthrough device
 *
 * Args:
 *	path - device path name string
 *
 * Return:
 *	TRUE - open successful
 *	FALSE - open failed
 */
bool_t
pthru_open(char *path)
{
	struct stat	stbuf;
	char		errstr[STR_BUF_SZ];

	/* Check for validity of device node */
	if (stat(path, &stbuf) < 0) {
		sprintf(errstr, app_data.str_staterr, path);
		cd_fatal_popup(app_data.str_fatal, errstr);
		return(FALSE);
	}
	if ((stbuf.st_mode & S_IFMT) != S_IFCHR) {
		sprintf(errstr, app_data.str_noderr, path);
		cd_fatal_popup(app_data.str_fatal, errstr);
		return(FALSE);
	}

	/* Check for another copy of xmcd running on the same CD-ROM device */
	if (!cd_devlock(path))
		return(FALSE);

	/* Open CD-ROM device */
	if ((fd = open(path, O_RDONLY | O_NDELAY | O_EXCL)) < 0)
		return(FALSE);

	return(TRUE);
}


/*
 * pthru_close
 *	Close SCSI passthrough device
 *
 * Args:
 *	Nothing.
 *
 * Return:
 *	Nothing.
 */
void
pthru_close(void)
{
	if (fd >= 0) {
		close(fd);
		fd = -1;
	}
}


/*
 * pthru_vers
 *	Return OS Interface Module version string
 *
 * Args:
 *	Nothing.
 *
 * Return:
 *	Module version text string.
 */
char *
pthru_vers(void)
{
	static char	vers[STR_BUF_SZ];

	sprintf(vers, "OS Interface module v%s (for UNIX SVR4-m88k)\n",
		OSI_VERS);
	return(vers);
}

#endif	/* MOTOROLA */

#endif	/* SVR4 sun SIMULATED_CDROM */

