/* semaphore management routines.

There are four sets of semaphores as follows:

SET 0  Control set consisting of three flags. ID name is sid_control.
      SEM 0   ServerAlive flag.    Set to 1 by server only using semop with UNDO. Set to 0 when server exits or crashes.
      SEM 1   numClients counter.  Incremented by client on process invocation using semop with UNDO.
                                   Decremented by UNDO when client exits.
      SEM 2   shmenRegCount.       Set to number of active regions (buffers) by server. Meaningless if controlSEM0 = 0.
                                   Read by clients to determine how many buffers (shmem regions) are
				   being maintained by the server.

SET 1  readReady set consisting of MAX_READERS flags (one for each active client).  ID name is sid_readReady.
      SEM 0 - n                    Set to 1 by server using semctl after change to any buffer.
                                   SET to 0 by client owner after finishing scan of all buffers.
				   Flags are for client info only and are not read by server.
				   Client identifies which sem of set belongs to it using clientNum.
				   Number of flags meaningful at any given time = controlSEM1.

SET 2  writeLock set consisting of MAX_BUFFERS flags (one for each active buffer). Number of flags
                                   meaningful at any given time = controlSEM2, i.e. controlSEM2 # of flags are
       SEM 0 - n                   Initialized by server to 1 using semctl.
                                   Decremented using semop with UNDO by any client or by the server to guarantee
				   exclusive access to the associated buffer. 
				   Incremented using semop with UNDO by same to restore the lock.
				   Both clients and server should attempt the decrement operation with the
				   IPC_NOWAIT flag set and skip processing of that buffer if the operation
				   returns with the EAGAIN errno.

SET 3 clientAlive set consisting of MAX_READERS flags (one for each active client). Number of flags
                                   meaningful at any given time = controlSEM1. Used to associate each
				   client with a unique clientNum which is valid for the life of that client.
				   When client is first invoked, it signals the server. The server responds
				   by sending the client a message giving it the number of the first 
				   available clientAlive flag. This number becomes that clients 
				   clientNum, and is its index into the clientAlive set and the readReady set. 
				   The client responds
				   by incrementing the clientAlive[clientNum] flag using semop with UNDO.
				   The flag is then only decremented when the client exits.

This module codes the following major actions on flags:

semset(semid, semnum)              unconditionally sets referenced flag to 1 using semctl.
                                   Returns -1 if operation fails.

semclr(semid, semnum)              unconditionally sets referenced flag to 0 using semctl.
                                   Returns -1 if operation fails.

semsetall(semid)                   unconditionally sets all flags in referenced group to 1 using semctl.
                                   Returns -1 if operation fails.

semclrall(semid)                   unconditionally clrs all flags in referenced group to 0 using semctl.
                                   Returns -1 if operation fails.

seminc(semid, semnum)              increments referenced flag using semop with UNDO.
                                   Returns -1 if operation fails.

semdec(semid, semnum)              decrements referenced flag using semop with UNDO and IPC_NOWAIT.
                                   returns -1 if operations returns with the EAGAIN errno.

semwait(semid, semnum)             Attempts to decrement referenced flag using semop with UNDO.
                                   Blocks until operation succeeds.

semgetval(semid, semnum)           returns the current value of the referenced flag.

*/
#include <stdio.h>
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/sem.h>

#ifndef ENOENT
#include <errno.h>
#endif
#ifndef NULL
#define NULL 0
#endif

extern int sid;

struct sembuf incflag, decflag;
struct semid_ds semid_ds;
union mysemun {
    int val;
    struct semid_ds *buf;
    ushort * array;
};

seminc(semid, semnum)
int semid, semnum;
{
    incflag.sem_op = 1;
    incflag.sem_flg = SEM_UNDO;

    incflag.sem_num = (short)semnum;
    return(semop(semid, &incflag, 1));
}

semdec(semid, semnum)
int semid, semnum;
{
    decflag.sem_op = -1;
    decflag.sem_flg = SEM_UNDO|IPC_NOWAIT;
    decflag.sem_num = (short)semnum;
    return(semop(semid, &decflag, 1));
}

semwait(semid, semnum)
int semid, semnum;
{
    decflag.sem_op = -1;
    decflag.sem_flg = SEM_UNDO;
    decflag.sem_num = (short)semnum;
    return(semop(semid, &decflag, 1));
}

semclrall(semid)
int semid;
{
/* 
  get number of semaphores in the semid set.
 */
    union mysemun arg;
    int i, retval;

    arg.buf = &semid_ds;
    if ((semctl(semid, 0, IPC_STAT, arg)) != 0)
	syserr("semctl in semclrall", NULL, NULL);
/* 
  allocate a buffer for the semaphore values. 
*/
    if (!(arg.array = (ushort *)malloc((unsigned)(semid_ds.sem_nsems * sizeof(ushort)))))
	syserr("malloc in semclrall", NULL, NULL);
/*
  set all the flags to 0.
*/
    for (i=0; i<semid_ds.sem_nsems; i++)
	arg.array[i] = 0;
/*
  issue the call.
*/
    retval = semctl(semid, 0, SETALL, arg);
    free(arg.array);
    return(retval);
}

semsetall(semid)
int semid;
{
/* 
  get number of semaphores in the semid set.
 */
    union mysemun arg;
    int i, retval;

    arg.buf = &semid_ds;
    if ((semctl(semid, 0, IPC_STAT, arg)) != 0)
	syserr("semctl in semsetall", NULL, NULL);
/* 
  allocate a buffer for the semaphore values. 
*/
    if (!(arg.array = (ushort *)malloc((unsigned)(semid_ds.sem_nsems * sizeof(ushort)))))
	syserr("malloc in semsetall", NULL, NULL);
/*
  set all the flags to 1.
*/
    for (i=0; i<semid_ds.sem_nsems; i++)
	arg.array[i] = 1;
/*
  issue the call.
*/
    retval = semctl(semid, 0, SETALL, arg);
    free(arg.array);
    return(retval);
}

semset(semid, semnum)
int semid, semnum;
{
    return(semctl(semid, semnum, SETVAL, 1));
}

semclr(semid, semnum)
int semid, semnum;
{
    return(semctl(semid, semnum, SETVAL, 0));
}

semgetval(semid, semnum)
int semid, semnum;
{
    return(semctl(semid, semnum, GETVAL, 0));
}
int openset(key)
long key;
{
    if ((sid = semget((key_t)key, 1, IPC_NOWAIT)) == -1 && errno != ENOENT){
	syserr("semget", NULL, NULL);
	return(-1);
    }
    return(sid);
}
