/* plot.c -- routines for manipulating and dealing with the plot window	*/
/*
 * Copyright (c) 1993  Leon Avery
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Send questions or comments on xdatplot to:
 *
 * Leon Avery
 * Department of Biochemistry
 * University of Texas Southwestern Medical Center
 * 5323 Harry Hines Blvd
 * Dallas, TX  75235-9038
 *
 * leon@eatworms.swmed.edu
 */

#include "xdatplot.h"

#define	BIGITVL	0x7fff

typedef	struct INTERVAL {
    struct INTERVAL	*i_nxt;
    long		x, width;
}		INTERVAL;

#ifdef	__STDC__
static	Boolean	redraw(INTERVAL **);
static	void	plot_line(int, int, int, int);
static	void	set_plot_region(void);
static	void	plot_axes(void);
static	void	plot_cursor(TIME, TIME);
static	Pixel	color_of_mark(MARK *);
static	void	plot_marks(TIME, TIME);
#ifdef	HAS_FUNC_ATTR
static	int	ntics(double, double, double) __attribute__ ((const));
#else
static	int	ntics(double, double, double);
#endif
static	void	plot_expose(Widget, XtPointer, XmDrawingAreaCallbackStruct *);
static	void	plot_input(Widget, XtPointer, XmDrawingAreaCallbackStruct *);
static	void	plot_key_press(XKeyEvent *);
static	void	plot_key(int, unsigned int, KeySym);
static	void	delete_every_mark(int);
static	void	delete_every_mark_cb(Widget, int, XmPushButtonCallbackStruct *);
static	void	plot_resize(Widget, XtPointer, XmDrawingAreaCallbackStruct *);
static	void	hsb_changed(Widget, XtPointer, XmScrollBarCallbackStruct *);
static	void	vsb_changed(Widget, XtPointer, XmScrollBarCallbackStruct *);
static	void	create_plot_GC(void);
static	void	create_axes_GC(void);
static	void	merge_interval(Dimension, Dimension, INTERVAL **);
static	INTERVAL *alloc_interval(void);
static	void	free_intervals(INTERVAL *);
static	void	free_interval(INTERVAL *);
static	void	plot_grexp(Widget, XGraphicsExposeEvent *, String *, int *);
static	int	adjust_displace(TIME, double, TIME, Dimension *);
#else	__STDC__
static	Boolean	redraw();
static	void	plot_line();
static	void	set_plot_region();
static	void	plot_axes();
static	void	plot_cursor();
static	Pixel	color_of_mark();
static	void	plot_marks();
static	int	ntics();
static	void	plot_expose();
static	void	plot_input();
static	void	plot_key_press();
static	void	plot_key();
static	void	delete_every_mark();
static	void	delete_every_mark_cb();
static	void	plot_resize();
static	void	hsb_changed();
static	void	vsb_changed();
static	void	create_plot_GC();
static	void	create_axes_GC();
static	void	merge_interval();
static	INTERVAL *alloc_interval();
static	void	free_intervals();
static	void	free_interval();
static	void	plot_grexp();
static	int	adjust_displace();
#endif	__STDC__

static	INTERVAL	*exposed = NULL;
static	XtWorkProcId	redrawID = NO_WORKID;
static	INTERVAL	*redraw_itvls = NULL;
static	INTERVAL	**redraw_end = &redraw_itvls;
/*
 * Tic spacing must be one of these times a power of ten
 */
static	int		ticmuls[] = {1, 2, 5};

void
create_plot_window(main_w)
Widget		main_w;
{
    XtActionsRec	actions;
    String		trans = "<GrExp>: plot_grexp()";

    plot = XtVaCreateManagedWidget("plot",
	xmDrawingAreaWidgetClass, main_w,
    NULL);
    AddCallback(plot, XmNexposeCallback, plot_expose, NULL);
    AddCallback(plot, XmNinputCallback, plot_input, NULL);
    AddCallback(plot, XmNresizeCallback, plot_resize, NULL);
    actions.string = "plot_grexp";
    actions.proc = (XtActionProc) plot_grexp;
    XtAppAddActions(app, &actions, 1);
    XtOverrideTranslations(plot, XtParseTranslationTable(trans));
    XtVaGetValues(plot,
	XmNwidth, &PWW,
	XmNheight, &PWH,
    NULL);
    set_plot_region();
    PFINFO = XQueryFont(XtDisplay(plot), PLOT_FONT);
    PHFINFO = XQueryFont(XtDisplay(plot), PLOT_HEADING_FONT);
    hsb = XtVaCreateManagedWidget("hsb",
	xmScrollBarWidgetClass, main_w,
	XmNorientation, XmHORIZONTAL,
	XmNprocessingDirection, XmMAX_ON_RIGHT,
    NULL);
    AddCallback(hsb, XmNvalueChangedCallback, hsb_changed, NULL);
    vsb = XtVaCreateManagedWidget("vsb",
	xmScrollBarWidgetClass, main_w,
	XmNorientation, XmVERTICAL,
	XmNprocessingDirection, XmMAX_ON_TOP,
    NULL);
    AddCallback(vsb, XmNvalueChangedCallback, vsb_changed, NULL);
/*    XmScrolledWindowSetAreas(main_w, hsb, vsb, plot); */
}

/* redo_whole -- redo whole plot, including scroll bars and scaling	*/
void
redo_whole()
{
    init_plot_params(FALSE);
    if (AUTOSCALE_ALWAYS) set_auto_scale();
    replot_whole();
}

/* set_auto_scale -- set VB and VT automatically, TRUE if changed	*/
Bool
set_auto_scale()
{
    register TIME	t;
    FILTER_VAL		vn, vx;
    Bool		change;

    if (TR <= TL) return(FALSE);
    vn = vx = Vf(TL);
    for(t = TL+1; t < TR; t++) {
	if (vn > Vf(t)) vn = Vf(t);
	if (vx < Vf(t)) vx = Vf(t);
    }
    change = (max(vn, VMIN) != VB) || (min(vx, VMAX) != VT);
    VB = max(vn, VMIN);
    VT = min(vx, VMAX);
    VT = max(VT, VB+1);
    if (change) adjust_sb();
    return(change);
}

/* adjust_sb -- adjust scrollbars and zoom buttons after changing sizes	*/
void
adjust_sb()
{
    TIME		tinc;
    TIME		tss;
    TIME		tpinc;
    TIME		tmin;
    TIME		tmax;
    VOLTAGE		vinc;
    VOLTAGE		vss;
    VOLTAGE		vpinc;
    VOLTAGE		vmin;
    VOLTAGE		vmax;

    tmin = TMIN;
    tmax = TMAX;
    if (tmin >= tmax) return;
    tinc = max(SBINC/TIME_to_Dim_sf, 1);
    tss = max(TR - TL, 1);
    tpinc = max(TR - TL - tinc, 1);
    XtVaSetValues(hsb,
	XmNminimum, tmin,
	XmNmaximum, tmax,
	XmNvalue, (int) TL,
	XmNsliderSize, tss,
	XmNincrement, tinc,
	XmNpageIncrement, tpinc,
    NULL);
    t_is_full_scale((TL == tmin) && (TR == tmax));
    vmin = VMIN;
    vmax = VMAX;
    if (vmin < vmax) {
	vinc = max(SBINC/VOLTAGE_to_Dim_sf, 1);
	vss = max(VT - VB, 1);
	vpinc = max(VT - VB - vinc, 1);
	XtVaSetValues(vsb,
	    XmNminimum, vmin,
	    XmNmaximum, vmax,
	    XmNvalue, (int) VB,
	    XmNsliderSize, vss,
	    XmNincrement, vinc,
	    XmNpageIncrement, vpinc,
	NULL);
	v_is_full_scale((VB == vmin) && (VT == vmax));
    }
    set_zoom_fields();
}

/* replot_part -- redo part of plot
 *
 * int	x, y, width, height;
 * replot_part(x, y, width, height);
 *
 * The parameters are the bounding box of the area to be replotted in
 * window coordinates, as they would be specified in an Expose event.
 */
void
replot_part(x, y, width, height)
int		x;
int		y;
int		width;
int		height;
{
    Display		*d = XtDisplay(plot);
    Window		w = XtWindow(plot);
    register TIME	t;
    TIME		t0, t1;
    FILTER_VAL		vlast;
    INTERVAL		*ip;

    if (F_DF != filter_type(&DF_FIL)) {	/* no file currently open	*/
	XClearWindow(d, w);
	return;
    }
    XClearArea(d, w,
	       x - (MARK_SIZE + MARK_THICKNESS)/2, PRY,
	       width + MARK_SIZE + MARK_THICKNESS, PRH,
	       FALSE);
    t0 = max(TL, Map_Dim_to_TIME(x - (MARK_SIZE + MARK_THICKNESS)/2 - 1) - 1);
    t1 = min(TR-1,
	     Map_Dim_to_TIME(x + width + MARK_SIZE + MARK_THICKNESS  + 1) + 1);
    if (t1 <= t0) return;
    plot_cursor(t0, t1);
    *redraw_end = ip = alloc_interval();
    redraw_end = &ip->i_nxt;
    ip->x = t0;
    ip->width = t1 - t0;
    if (NO_WORKID == redrawID)
	redrawID = XtAppAddWorkProc(app, (XtWorkProc) redraw, &redraw_itvls);
}

/* plot_line.c -- compress unnecessary XDrawLine calls
 *
 * int	x1, y1, x2, y2;
 * plot_line(x1, y1, x2, y2);
 *
 * plot_line_finish();
 *
 * plot_line caches requests to draw lines for which x1 == x2 == x2
 * from previous call, merely remembering the x and accumulating the
 * maximum and minimum y's.  When it receives a call with a new x, it
 * then writes a single vertical line from the maximum to the minimum
 * y.  To make plot_line emit any remaining data and reset itself,
 * call with x1 == -1.
 */
static INLINE void
plot_line(x1, y1, x2, y2)
int		x1, y1;
int		x2, y2;
{
    Display		*d = XtDisplay(plot);
    Window		w = XtWindow(plot);
    static int		oldx = -1;
    static int		ymin, ymax;

    if (-1 == x1) {
	if (-1 != oldx) XDrawLine(d, w, PLOT_GC, oldx, ymin, oldx, ymax);
	oldx = -1;
    }
    else if ((x1 == x2) && (x1 == oldx)) {
	ymin = min(y1, min(y2, ymin));
	ymax = max(y1, max(y2, ymax));
    }
    else if ((x1 == x2) && (x1 != oldx)) {
	if (-1 != oldx) XDrawLine(d, w, PLOT_GC, oldx, ymin, oldx, ymax);
	oldx = x1;
	ymin = min(y1, y2);
	ymax = max(y1, y2);
    }
    else /* (-1 != x1) || (x1 != x2) */ {
	if (-1 != oldx) XDrawLine(d, w, PLOT_GC, oldx, ymin, oldx, ymax);
	oldx = -1;
	XDrawLine(d, w, PLOT_GC, x1, y1, x2, y2);
    }
}

static Pixel
color_of_mark(mp)
MARK		*mp;
{
    if (NULL != STRCHR(app_data.color_2_marks, mp->val))
	return(app_data.mark_color_2);
    else if (NULL != STRCHR(app_data.color_3_marks, mp->val))
	return(app_data.mark_color_3);
    else if (NULL != STRCHR(app_data.color_4_marks, mp->val))
	return(app_data.mark_color_4);
    else return(app_data.mark_color);
}

static void
plot_marks(t0, t1)
TIME		t0;
TIME		t1;
{
    XGCValues		oldGCvals;
    XGCValues		newGCvals;
    Display		*d = XtDisplay(plot);
    Window		w = XtWindow(plot);
    MARK		*mp;
    int			n;
    register int	i;
    int			x, y;
    int			st, en;
    char		buf[2] = "a";

    t0 = Map_Dim_to_TIME(Map_TIME_to_Dim(t0) -
			 (MARK_SIZE + MARK_THICKNESS)/2 - 1);
    t1 = Map_Dim_to_TIME(Map_TIME_to_Dim(t1) +
			 (MARK_SIZE + MARK_THICKNESS)/2 + 1);
    n = lookup_interval(Map_TIME_to_MARK(t0), Map_TIME_to_MARK(t1), &mp);
    if (n > 0) {
	XGetGCValues(d, PLOT_GC, GCLineWidth | GCForeground, &oldGCvals);
	newGCvals.line_width = MARK_THICKNESS;
	XChangeGC(d, PLOT_GC, GCLineWidth, &newGCvals);
	st = -MARK_SIZE / 2;
	en = st + MARK_SIZE;
	for(i=0; i<n; i++) {
	    newGCvals.foreground = color_of_mark(&mp[i]);
	    XChangeGC(d, PLOT_GC, GCForeground, &newGCvals);
	    x = Map_MARK_to_Dim(mp[i].t);
	    y = Map_VOLTAGE_to_Dim(Vf(Map_MARK_to_TIME(mp[i].t)));
	    XDrawLine(d, w, PLOT_GC, x + st, y + st, x + en, y + en);
	    XDrawLine(d, w, PLOT_GC, x + en, y + st, x + st, y + en);
	    buf[0] = mp[i].val;
	    x -= XTextWidth(PFINFO, buf, 1) / 2;
	    y = PRY + PFINFO->ascent + 2;
	    XDrawString(d, w, PLOT_GC, x, y, buf, 1);
	}
	XChangeGC(d, PLOT_GC, GCLineWidth | GCForeground, &oldGCvals);
    }
}

static Boolean
redraw(ipp)
INTERVAL	**ipp;
{
    INTERVAL		*ip = *ipp;
    FILTER_VAL		vlast;
    TIME		t;
    TIME		t0;
    TIME		t1;
    Bool		whole;

    t0 = ip->x;
    whole = (ip->width <= BIGITVL);
    t1 = t0 + (whole ? ip->width : BIGITVL+1);
    vlast = Vf(t0);
    for(t=t0+1; t<=t1; t++) {
	plot_line(Map_TIME_to_Dim(t-1), Map_VOLTAGE_to_Dim(vlast),
		  Map_TIME_to_Dim(t), Map_VOLTAGE_to_Dim(Vf(t)));
	vlast = Vf(t);
    }
    plot_line(-1, -1, -1, -1);
    plot_marks(t0, t1);
    if (whole) {
	plot_axes();
	XFlush(XtDisplay(plot));
	if (&ip->i_nxt == redraw_end) {
	    if (NULL != ip->i_nxt) error("redraw: can't happen");
	    redraw_end = ipp;
	    redrawID = NO_WORKID;
	}
	ip = ip->i_nxt;
	free_interval(*ipp);
	*ipp = ip;
	return(NULL == ip);
    }	
    else {
	ip->x += BIGITVL;
	ip->width -= BIGITVL;
	return(False);
    }
}

void
cancel_redraw()
{
    if (NO_WORKID != redrawID) XtRemoveWorkProc(redrawID);
    redrawID = NO_WORKID;
    if (NULL != redraw_itvls) free_intervals(redraw_itvls);
    redraw_itvls = NULL;
    redraw_end = &redraw_itvls;
    return;
}

/* replot_whole -- redo entire plot					*/
void
replot_whole()
{
    cancel_redraw();
    if (TR - TL > BIGITVL) plot_axes();
    replot_part(PRX, PRY, PRW, PRH);
}

static void
plot_cursor(t0, t1)
TIME		t0;
TIME		t1;
{
    Display		*disp = XtDisplay(plot);
    Window		win = XtWindow(plot);
    XGCValues		oldGCvals;
    XGCValues		newGCvals;

    if (IS_CURSOR && !IS_CURSOR_END && (CURSOR >= t0) && (CURSOR < t1)) {
	XGetGCValues(disp, PLOT_GC, GCForeground, &oldGCvals);
	newGCvals.foreground = CURSOR_COLOR;
	XChangeGC(disp, PLOT_GC, GCForeground, &newGCvals);
	XDrawLine(disp, win, PLOT_GC,
		  Map_TIME_to_Dim(CURSOR), PRY,
		  Map_TIME_to_Dim(CURSOR), PRY + PRH - 1);
	XChangeGC(disp, PLOT_GC, GCForeground, &oldGCvals);
    }
    if (IS_CURSOR && IS_CURSOR_END &&(CURSOR < t1) && (CURSOR_END >= t0)) {
	Dimension		xl, xr;

	XGetGCValues(disp, PLOT_GC, GCForeground, &oldGCvals);
	newGCvals.foreground = CURSOR_COLOR;
	XChangeGC(disp, PLOT_GC, GCForeground, &newGCvals);
	xl = Map_TIME_to_Dim(max(t0, CURSOR));
	xr = Map_TIME_to_Dim(min(t1, CURSOR_END));
	XFillRectangle(disp, win, PLOT_GC,
		       xl, PRY, xr - xl + 1, PRY + PRH - 1);
	XChangeGC(disp, PLOT_GC, GCForeground, &oldGCvals);
    }
}

static void
plot_axes()
{
    register int	i;
    register double	tp;
    Display		*d = XtDisplay(plot);
    Window		w = XtWindow(plot);
    TICS		tics;
    int			x, y;
    int			xmin;
    char		lbuf[LLEN];
    String		heading;

    XDrawRectangle(d, w, AXES_GC, PRX-1, PRY-1, PRW+1, PRH+1);
    XClearArea(d, w, 0, 0, 0, PRY-1, FALSE);
    XClearArea(d, w, 0, PRY + PRH + 1, 0, 0, FALSE);
    XClearArea(d, w, 0, 0, PRX-1, 0, FALSE);
    XClearArea(d, w, PRX + PRW + 1, 0, 0, 0, FALSE);
    /*
     * put tic marks on the t axis
     */
    y = PRY + PRH + PFINFO->ascent + 2;
    tics.tcmin = T_TICS;
    tics.lo = Map_TIME_to_t(TL + V_TICLEN / TIME_to_Dim_sf);
    tics.hi = Map_TIME_to_t(TR - V_TICLEN / TIME_to_Dim_sf);
    find_tics(&tics);
    for(i = tics.first; i < tics.n + tics.first; i++) {
	tp = i * tics.space;
	sprintf(lbuf, "%g", tp);
	x = Map_t_to_Dim(tp) - XTextWidth(PFINFO, lbuf, strlen(lbuf)) / 2;
	if (x < 0) x = 0;
	XDrawImageString(d, w, AXES_GC, x, y, lbuf, strlen(lbuf));
	XDrawLine(d, w, AXES_GC,
		  Map_t_to_Dim(tp), PRY,
		  Map_t_to_Dim(tp), PRY + T_TICLEN);
	XDrawLine(d, w, AXES_GC,
		  Map_t_to_Dim(tp), PRY + PRH - 1,
		  Map_t_to_Dim(tp), PRY + PRH - 1 - T_TICLEN);
    }
    /*
     * put tic marks on the v axis
     */
    y = (PFINFO->ascent - PFINFO->descent)/2;
    tics.tcmin = V_TICS;
    tics.lo = Map_VOLTAGE_to_v(VB + T_TICLEN / VOLTAGE_to_Dim_sf);
    tics.hi = Map_VOLTAGE_to_v(VT - T_TICLEN / VOLTAGE_to_Dim_sf);
    find_tics(&tics);
    xmin = PRX;
    for(i = tics.first; i < tics.n + tics.first; i++) {
	tp = i * tics.space;
	sprintf(lbuf, "%g", tp);
	x = PRX - PFINFO->min_bounds.width -
	    XTextWidth(PFINFO, lbuf, strlen(lbuf));
	xmin = min(x, xmin);
	if (x < 0) x = 0;
	XDrawImageString(d, w, AXES_GC,
			 x, Map_v_to_Dim(tp) + y, lbuf, strlen(lbuf));
	XDrawLine(d, w, AXES_GC,
		  PRX, Map_v_to_Dim(tp),
		  PRX + V_TICLEN, Map_v_to_Dim(tp));
	XDrawLine(d, w, AXES_GC,
		  PRX + PRW - 1, Map_v_to_Dim(tp),
		  PRX + PRW - 1 - V_TICLEN, Map_v_to_Dim(tp));
    }
    /*
     * place all the various headers
     */
    x = PRX + PRW/2 - XTextWidth(PFINFO, T_UNITS, strlen(T_UNITS)) / 2;
    x = max(x, 0);
    y = PRY + PRH + PFINFO->ascent + 2 + PFINFO->ascent + PFINFO->descent;
    y = max(y, 0);
    XDrawImageString(d, w, AXES_GC, x, y, T_UNITS, strlen(T_UNITS));
    x = xmin - PFINFO->min_bounds.width -
	XTextWidth(PFINFO, V_UNITS, strlen(V_UNITS));
    x = max(x, 0);
    y = PRY + PRH/2 + (PFINFO->ascent - PFINFO->descent)/2;
    y = max(y, 0);
    XDrawImageString(d, w, AXES_GC, x, y, V_UNITS, strlen(V_UNITS));
    heading = (NULL != app_data.xdp_file) ?
	      app_data.xdp_file :
	      app_data.data_file;
    x = PRX + PRW/2 - XTextWidth(PHFINFO, heading, strlen(heading)) / 2;
    x = max(x, 0);
    y = PRY - PHFINFO->descent - 2;
    y = max(y, 0);
    if (PLOT_HEADING_FONT != PLOT_FONT) {
	XSetFont(d, AXES_GC, PLOT_HEADING_FONT);
	XDrawImageString(d, w, AXES_GC, x, y, heading, strlen(heading));
	XSetFont(d, AXES_GC, PLOT_FONT);
    }
    else {
	XDrawImageString(d, w, AXES_GC, x, y, heading, strlen(heading));
    }
}

/* find_tics -- figure out where to put tics marks
 *
 * TICS		tics;
 * find_tics(&tics);
 *
 * The caller must fill in the min, lo, and hi fields of the tics
 * structure.  find_tics fills in the n, first, and space fields,
 * which tell where the actual tics should go.
 *
 */
void
find_tics(t)
TICS		*t;
{
    double		sp;
    double		lsp;
    int			i;
    int			e;

    if (t->hi <= t->lo) error("find_tics: can't happen");
    i = 0;				/* guess at a resonable space	*/
    e = floor(log10((t->hi - t->lo) / (t->tcmin * ticmuls[i])));
    sp = ticmuls[i] * exp10((double) e);
    if (ntics(sp, t->lo, t->hi) >= t->tcmin) {
	do {
	    lsp = sp;
	    i++;
	    if (i >= XtNumber(ticmuls)) {
		e++;
		i = 0;
	    }
	    sp = ticmuls[i] * exp10((double) e);
	} while(ntics(sp, t->lo, t->hi) >= t->tcmin);
	sp = lsp;
    }
    else {
	do {
	    lsp = sp;
	    i--;
	    if (i < 0) {
		e--;
		i = XtNumber(ticmuls) - 1;
	    }
	    sp = ticmuls[i] * exp10((double) e);
	} while(ntics(sp, t->lo, t->hi) < t->tcmin);
    }
    t->space = sp;
    t->n = ntics(sp, t->lo, t->hi);
    t->first = ceil(t->lo / sp);
}

/* ntics -- find number of tics with given spacing in given interval	*/
static int
ntics(sp, lo, hi)
double		sp;
double		lo;
double		hi;
{
    int			ilo, ihi;

    ilo = ceil(lo / sp);
    ihi = floor(hi / sp);
    return(ihi - ilo + 1);
}

/* set_plot_region -- set plot region when window changes		*/
static void
set_plot_region()
{
    XRectangle		rect;
    String		s;

    PRX = LEFT_MARGIN;
    PRY = TOP_MARGIN;
    PRW = PWW - PRX - RIGHT_MARGIN;
    PRH = PWH - PRY - BOTTOM_MARGIN;
    if ((GC) NULL != PLOT_GC) {
	rect.x = PRX;
	rect.y = PRY;
	rect.width = PRW;
	rect.height = PRH;
	XSetClipRectangles(XtDisplay(plot), PLOT_GC, 0, 0, &rect, 1, Unsorted);
    }
}

/* plot was exposed							*/
static void
plot_expose(widget, data, cbs)
Widget				widget;
XtPointer			data;
XmDrawingAreaCallbackStruct	*cbs;
{
    XExposeEvent		*xex = &cbs->event->xexpose;
    INTERVAL			*ip;

    dprintf("plot_expose\n");
    merge_interval(xex->x, xex->width, &exposed);
    if (0 != xex->count) return;
    if (NULL == PLOT_GC) create_plot_GC();
    if (NULL == AXES_GC) create_axes_GC();
    create_axes_GC();
    for(ip = exposed; NULL != ip; ip = ip->i_nxt) {
	replot_part(ip->x, 0, ip->width, PWH);
    }
    free_intervals(exposed);
    exposed = NULL;
}

/* plot window input							*/
static void
plot_input(widget, data, cbs)
Widget				widget;
XtPointer			data;
XmDrawingAreaCallbackStruct	*cbs;
{
    dprintf("plot_input\n");
    switch(cbs->event->type) {
    case ButtonPress:
	{
	    int			ret;
	    XButtonEvent	*bev = (XButtonEvent *) cbs->event;

	    ret = XmProcessTraversal(widget, XmTRAVERSE_CURRENT);
	    if (
		(Button1 == bev->button) &&
		(XtWindow(plot) == bev->window) &&
		(bev->x >= PRX) &&
		(bev->x < PRX + PRW)
	    ) {
		if (TMIN >= TMAX) {
		    Bellret;
		}
		else {
		    set_cursor(Map_Dim_to_TIME(bev->x));
		}
	    }
	    else if (
		(Button3 == bev->button) &&
		(XtWindow(plot) == bev->window) &&
		(bev->x >= PRX) &&
		(bev->x < PRX + PRW)
	    ) {
		if ((TMIN >= TMAX) || !IS_CURSOR) {
		    Bellret;
		}
		else {
		    set_cursor_end(Map_Dim_to_TIME(bev->x));
		}
	    }
	}
	break;

    case ButtonRelease:
	break;

    case KeyPress:
	plot_key_press(&cbs->event->xkey);
	break;

    case KeyRelease:
	break;
    }
}

static void
plot_key_press(kev)
XKeyEvent	*kev;
{
    register int	i;
    char		lbuf[LLEN];
    int			count;
    KeySym		key;
    XComposeStatus	compose;

    count = XLookupString(kev, lbuf, LLEN, &key, &compose);
    if ((key >= XK_Shift_L) && (key <= XK_Hyper_R)) {
	return;				/* modifier: ignore		*/
    }
    else if (0 == count) {		/* no mapping for this		*/
	plot_key(EOF, kev->state, key);
	return;
    }
    else {
	for(i=0; i<count; i++) {	/* handle each char		*/
	    plot_key((int) lbuf[i], kev->state, key);
	}
    }
}

#define	MAKE_MARK(t, c)\
{							\
    char		lbuf[LLEN];			\
    MARK		*mp;				\
							\
    mp = find_mark(Map_TIME_to_MARK(t), c);		\
    sprintf(lbuf, app_data.mark_comment_format_2,			\
	    (double) Map_TIME_to_t_c(t),		\
	    (double) Map_VOLTAGE_to_v_c(Vf(t)));	\
    Nfree(mp->comment);					\
    mp->comment = XtNewString(lbuf);			\
}

/* plot_key -- deal with a single char input in the plot window		*/
static void
plot_key(c, state, key)
int		c;
unsigned int	state;
KeySym		key;
{
    String		s = NULL;
    String		sh_dig = ")!@#$%^&*("; /* shifted digits	*/
    TIME		t;
    MARK		*m;
    int			CtrlA = 001;
    int			CtrlZ = 032;
    int			shifted = state & ShiftMask;

    if (TMIN >= TMAX) Bellret;
    if (				/* kludge for Ctrl digits	*/
	isdigit(key) && !isdigit(c) &&
	(0 != (ControlMask & state))
    ) c = key;
    state &= ~ShiftMask;		/* info already in c and key	*/
    if (EOF == c) {			/* non-ASCII event		*/
	switch(key) {
	case XK_Left:			/* move CURSOR left		*/
	    if (!IS_CURSOR) {		/* no CURSOR: place one		*/
		t = (TL + TR) / 2;
		set_cursor(t);
	    }
	    else switch(state) {
	    case 0:			/* one pixel			*/
		t = Map_Dim_to_TIME(-1 + Map_TIME_to_Dim(CURSOR));
		break;

	    case Mod1Mask:		/* ten pixels			*/
		t = Map_Dim_to_TIME(-10 + Map_TIME_to_Dim(CURSOR));
		break;
		
	    case ControlMask:		/* 1 t-value			*/
		t = CURSOR - 1;
		break;

	    case Mod1Mask | ControlMask:
		if (0 == prev_marks(Map_TIME_to_MARK(CURSOR), &m)) Bellret;
		t = Map_MARK_to_TIME(m->t);
		break;

	    default:
		Bellret;
	    }
	    t = min(t, CURSOR-1);
	    if (t < TMIN) Bellret;
	    set_cursor(t);
	    return;

	case XK_Right:			/* move CURSOR left		*/
	    if (!IS_CURSOR) {		/* no CURSOR: place one		*/
		t = (TL + TR) / 2;
		set_cursor(t);
	    }
	    else switch(state) {
	    case 0:			/* one pixel			*/
		t = Map_Dim_to_TIME(1 + Map_TIME_to_Dim(CURSOR));
		break;

	    case Mod1Mask:		/* ten pixels			*/
		t = Map_Dim_to_TIME(10 + Map_TIME_to_Dim(CURSOR));
		break;
		
	    case ControlMask:		/* 1 t-value			*/
		t = CURSOR + 1;
		break;

	    case Mod1Mask | ControlMask:
		if (0 == next_marks(Map_TIME_to_MARK(CURSOR+1), &m)) Bellret;
		t = Map_MARK_to_TIME(m->t);
		break;

	    default:
		Bellret;
	    }
	    t = max(t, CURSOR+1);
	    if (t >= TMAX) Bellret;
	    set_cursor(t);
	    return;

	default:
	    Bellret;
	}
    }
    else {				/* ASCII			*/
	if (				/* lowercase or digit		*/
	    (0 == state) &&
	    (islower(c) || isdigit(c))
	) {
	    if (!IS_CURSOR) set_cursor((TL + TR) / 2);
	    MAKE_MARK(CURSOR, c);	/* insert a new mark		*/
	    replot_part(Map_TIME_to_Dim(CURSOR), PRY, 1, PRH);
	}
	else if (			/* uppercase or shifted digit	*/
	     (0 == state) &&
	     (isupper(c) || (NULL != (s = STRCHR(sh_dig, c))))
	) {				/* replace marks at CURSOR	*/
	    if (!IS_CURSOR) set_cursor((TL + TR) / 2);
	    if (isupper(c)) c = tolower(c);
	    else c = '0' + s - sh_dig;
	    delete_marks(Map_TIME_to_MARK(CURSOR));
	    MAKE_MARK(CURSOR, c);
	    replot_part(Map_TIME_to_Dim(CURSOR), PRY, 1, PRH);
	}
	else if (			/* Ctrl letter or digit		*/
	    (
	        (0 == (state & ~ControlMask)) && (!shifted) &&
	        (CtrlA <= c) &&
	        (CtrlZ >= c)
	    ) ||
	    (
	        (ControlMask == state) &&
	        (isalpha(c) || isdigit(c))
	    )
	) {				/* move forward to next match	*/
	    if (!IS_CURSOR) set_cursor((TL + TR) / 2);
	    if (isalpha(c)) c = tolower(c);
	    if ((CtrlA <= c) && (CtrlZ >= c)) c += 'a' - CtrlA;
	    m = next_mark(Map_TIME_to_MARK(CURSOR+1), c);
	    if (NULL == m) Bellret;
	    set_cursor(Map_MARK_to_TIME(m->t));
	}
	else if (			/* Ctrl Shift letter or digit	*/
	    (
	        (0 == (state & ~ControlMask)) && (shifted) &&
	        (CtrlA <= c) &&
	        (CtrlZ >= c)
	    ) ||
	    (
	        (ControlMask == state) && (shifted) &&
	        (isalpha(c) || isdigit(c) || (NULL != (s = STRCHR(sh_dig, c))))
	    )
	) {				/* delete all matching marks	*/
	    if (isalpha(c)) c = tolower(c);
	    else if ((CtrlA <= c) && (CtrlZ >= c)) c += 'a' - CtrlA;
	    else if (!isdigit(c)) c = '0' + s - sh_dig;
	    delete_every_mark(c);
	}
	else if (			/* Ctrl Alt letter or digit	*/
	    (
	        (Mod1Mask == (state & ~ControlMask)) &&
	        (CtrlA <= c) &&
	        (CtrlZ >= c)
	    ) ||
	    (
	        ((ControlMask | Mod1Mask) == state) &&
	        (isalpha(c) || isdigit(c))
	    )
	) {				/* move back to previous match	*/
	    if (!IS_CURSOR) set_cursor((TL + TR) / 2);
	    if (isalpha(c)) c = tolower(c);
	    if ((CtrlA <= c) && (CtrlZ >= c)) c += 'a' - CtrlA;
	    m = prev_mark(Map_TIME_to_MARK(CURSOR), c);
	    if (NULL == m) Bellret;
	    set_cursor(Map_MARK_to_TIME(m->t));
	}
	else switch(key) {
	case ' ':
	    replot_whole();
	    break;

	case XK_Delete:
	case XK_BackSpace:
	    switch(state | shifted) {
	    case 0:
		if (!IS_CURSOR) set_cursor((TL + TR) / 2);
		delete_marks(Map_TIME_to_MARK(CURSOR));
		replot_part(Map_TIME_to_Dim(CURSOR), PRY, 1, PRH);
		break;

	    case ControlMask | ShiftMask:
		delete_every_mark(0);
		break;

	    default:
		Bellret;
	    }
	    break;

	case '=':			/* center cursor in display	*/
	    t = (TL + TR) / 2;
	    set_cursor(t);
	    break;

	case '\\':			/* display cursor		*/
	    if (IS_CURSOR)
		set_cursor(CURSOR);
	    else
		Bellret;
	    break;

	case ',':			/* scroll left or down		*/
	case '.':			/* scroll right or up		*/
	    if (0 != (state & ~ControlMask & ~Mod1Mask)) Bellret;
	    if (0 == (state & Mod1Mask)) {
		int		inc;
		TIME		ntl, ntr;

		if (0 == (state & ControlMask)) {
		    XtVaGetValues(hsb, XmNincrement, &inc, NULL);
		}
		else {
		    XtVaGetValues(hsb, XmNpageIncrement, &inc, NULL);
		}
		if (key == ',') {
		    if (TL <= TMIN) Bellret;
		    inc = -inc;
		}
		else {
		    if (TR >= TMAX) Bellret;
		}
		ntl = TL + inc;
		ntr = TR + inc;
		FORCE_RANGE(ntl, ntr, TMIN, TMAX);
		scroll_plot(ntl);
	    }
	    else {			/* Alt: up or down		*/
		int		inc;

		if (0 == (state & ControlMask)) {
		    XtVaGetValues(vsb, XmNincrement, &inc, NULL);
		}
		else {
		    XtVaGetValues(vsb, XmNpageIncrement, &inc, NULL);
		}
		if (key == ',') {
		    if (VB <= VMIN) Bellret;
		    inc = -inc;
		}
		else {
		    if (VT >= VMAX) Bellret;
		}
		VB += inc;
		VT += inc;
		FORCE_RANGE(VB, VT, VMIN, VMAX);
		replot_whole();
	    }
	    adjust_sb();
	    break;

	case '<':			/* zoom in			*/
	    switch(state) {
	    case 0:			/* small zoom, t axis		*/
		t_zoom_in(ZOOMFAC);
		break;

	    case ControlMask:		/* big zoom, t axis		*/
		t_zoom_in(BIGZOOMFAC);
		break;

	    case Mod1Mask:		/* small zoom, v axis		*/
		v_zoom_in(ZOOMFAC);
		break;

	    case ControlMask | Mod1Mask: /* big zoom, v-axis		*/
		v_zoom_in(BIGZOOMFAC);
		break;

	    default:
		Bellret;
	    }
	    adjust_sb();
	    break;

	case '>':			/* zoom out			*/
	    switch(state) {
	    case 0:			/* small zoom, t axis		*/
		t_zoom_out(ZOOMFAC);
		break;

	    case ControlMask:		/* big zoom, t axis		*/
		t_zoom_out(BIGZOOMFAC);
		break;

	    case Mod1Mask:		/* small zoom, v axis		*/
		v_zoom_out(ZOOMFAC);
		break;

	    case ControlMask | Mod1Mask: /* big zoom, v-axis		*/
		v_zoom_out(BIGZOOMFAC);
		break;

	    default:
		Bellret;
	    }
	    adjust_sb();
	    break;

	case '-':			/* delete CURSOR		*/
	    reset_cursor();
	    break;

	default:
	    Bellret;
	}
    }
}

/* delete_every_mark -- delete every mark matching a value		*/
static void
delete_every_mark(val)
int		val;
{
    char		lbuf[LLEN];
    char		tbuf[LLEN];
    String		s;
    Widget		popup;
    Arg			args[1];

    XtSetArg(args[0], XmNautoUnmanage, True);
    popup = XmCreateQuestionDialog(toplevel, "deleteMarks", args, 1);
    XtVaGetValues(toplevel, XmNtitle, &s, NULL);
    sprintf(tbuf, "%s Delete Marks", s);
    if (0 == val) {
	if (IS_CURSOR && IS_CURSOR_END) {
	    sprintf(lbuf, "Delete all marks from %g to %g?",
		    Map_TIME_to_t(CURSOR), Map_TIME_to_t(CURSOR_END));
	}
	else {
	    sprintf(lbuf, "Delete all marks?");
	}
    }
    else {
	if (IS_CURSOR && IS_CURSOR_END) {
	    sprintf(lbuf, "Delete all %c marks from %g to %g?", val,
		    Map_TIME_to_t(CURSOR), Map_TIME_to_t(CURSOR_END));
	}
	else {
	    sprintf(lbuf, "Delete all %c marks?", val);
	}
    }
    XtVaSetValues(popup,
	XmNdeleteResponse, XmDESTROY,
	XmNdialogStyle, XmDIALOG_PRIMARY_APPLICATION_MODAL,
	XtVaTypedArg, XmNmessageString, XmRString,
	lbuf, strlen(lbuf)+1,
	XtVaTypedArg, XmNdialogTitle, XmRString,
	tbuf, strlen(tbuf)+1,
    NULL);
    XtSetSensitive(XmMessageBoxGetChild(popup, XmDIALOG_HELP_BUTTON),
		   FALSE);
    AddCallback(popup, XmNokCallback, delete_every_mark_cb, val);
    XtManageChild(popup);
    XtPopup(XtParent(popup), XtGrabNone);
    Bell;
}

static void
delete_every_mark_cb(dialog, val, cbs)
Widget				dialog;
int				val;
XmPushButtonCallbackStruct	*cbs;
{
    MARK		*mp;
    TIME		t;
    TIME		tmax;
    int			n;

    if (IS_CURSOR && IS_CURSOR_END) {
	t = Map_TIME_to_MARK(CURSOR);
	tmax = Map_TIME_to_MARK(CURSOR_END);
    }
    else {
	t = Map_TIME_to_MARK(TMIN);
	tmax = Map_TIME_to_MARK(TMAX);
    }
    if (0 == val) {
	if (!IS_CURSOR || !IS_CURSOR_END) {
	    clear_all_marks();
	    replot_whole();
	    return;
	}
	else {
	    delete_interval(t, tmax);
	}
    }
    n = lookup_interval(t, tmax, &mp);
    while(n > 0) {
	if (val == mp->val) {
	    t = mp->t;
	    delete_mark(mp->t, val);
	    n = lookup_interval(t, tmax, &mp);
	}
	else {
	    mp++;
	    n--;
	}
    }
    replot_whole();
}

/* set_cursor -- set cursor at particular time				*/
void
set_cursor(t)
TIME		t;
{
    char		lbuf[LLEN];
    TIME		w;

    if (IS_CURSOR && (CURSOR == t)) return;
    reset_cursor();
    IS_CURSOR = TRUE;
    CURSOR = t;
    sprintf(lbuf, "%g", (double) Map_TIME_to_t(t));
    XtVaSetValues(t_field, XmNvalue, lbuf, NULL);
    sprintf(lbuf, "%g", (double) Map_VOLTAGE_to_v(Vf(t)));
    XtVaSetValues(v_field, XmNvalue, lbuf, NULL);
    if ((t >= TL) && (t < TR)) {
	replot_part(Map_TIME_to_Dim(CURSOR), PRY, 1, PRH);
    }
    else {
	w = (TR - TL) / 2;
	app_data.tl = t - w;
	app_data.tlfrac = 0.0;
	TR = t + w;
	FORCE_RANGE(app_data.tl, TR, TMIN, TMAX);
	if (AUTOSCALE_ALWAYS) set_auto_scale();
	adjust_sb();
	replot_whole();
    }
}

/* set_cursor_end -- set cursor region					*/
void
set_cursor_end(t)
TIME		t;
{
    char		lbuf[LLEN];
    TIME		oc = CURSOR;
    TIME		w;

    if (oc < t) {
	set_cursor(oc);
	CURSOR_END = t;
    }
    else {
	set_cursor(t);
	CURSOR_END = oc;
    }
    IS_CURSOR_END = TRUE;
    if ((CURSOR_END >= TL) && (CURSOR < TR)) {
	Dimension		xl = Map_TIME_to_Dim(max(TL, CURSOR));
	Dimension		xr = Map_TIME_to_Dim(min(TR, CURSOR_END));

	replot_part(xl, PRY, xr - xl + 1, PRH);
    }
}

void
reset_cursor()
{
    if (IS_CURSOR) {
	IS_CURSOR = FALSE;
	XtVaSetValues(t_field, XmNvalue, "", NULL);
	XtVaSetValues(v_field, XmNvalue, "", NULL);
	if (!IS_CURSOR_END && (CURSOR >= TL) && (CURSOR < TR)) {
	    replot_part(Map_TIME_to_Dim(CURSOR), PRY, 1, PRH);
	}
	if (IS_CURSOR_END && (CURSOR < TR) && (CURSOR_END >= TL)) {
	    Dimension		xl = Map_TIME_to_Dim(max(CURSOR, TL));
	    Dimension		xr = Map_TIME_to_Dim(min(CURSOR_END, TR));

	    IS_CURSOR_END = FALSE;
	    replot_part(xl, PRY, xr - xl + 1, PRH);
	}
    }
    IS_CURSOR_END = FALSE;
}

/* plot was resized							*/
static void
plot_resize(widget, data, cbs)
Widget				widget;
XtPointer			data;
XmDrawingAreaCallbackStruct	*cbs;
{
    register double	d;
    Bool		change = FALSE;
    Dimension		w, h;
    int			old;

    dprintf("plot_resize\n");
    XtVaGetValues(plot,
	XmNwidth, &w,
	XmNheight, &h,
    NULL);
    if (PWW != w) {
	old = PRW;
	PWW = w;
	change = TRUE;
	set_plot_region();
	d = TR - TL; d /= old; d *= PRW;
	TR = TL + d;
	FORCE_RANGE(app_data.tl, TR, TMIN, TMAX);
    }
    if (AUTOSCALE_ALWAYS) {
	PWH = h;
	change = TRUE;
	set_plot_region();
	set_auto_scale();
    }
    else {
	if (PWH != h) {
	    old = PRH;
	    PWH = h;
	    change = TRUE;
	    set_plot_region();
	    d = VT - VB; d /= old; d *= PRH;
	    VB = (VB + VT - d)/2;
	    VT = VB + d;
	    FORCE_RANGE(VB, VT, VMIN, VMAX);
	}
    }
    if (change) {
	adjust_sb();
	expose_whole();
    }
}

/* hsb_changed -- value changed callback for horizontal scrollbar	*/
static void
hsb_changed(widget, data, cbs)
Widget				widget;
XtPointer			data;
XmScrollBarCallbackStruct	*cbs;
{
    int			val = cbs->value;
    TIME		ntl, ntr;
    Bool		change = FALSE;

    if (TL == val) return;		/* no change, actually		*/
    ntl = val;
    ntr = TR + val - TL;
    if (ntr > TMAX) {
	ntl -= ntr - TMAX;
	ntr = TMAX;
	change = TRUE;
    }
    scroll_plot(ntl);
    if (change) adjust_sb();
}

/* vsb_changed -- value changed callback for vertical scrollbar		*/
static void
vsb_changed(widget, data, cbs)
Widget				widget;
XtPointer			data;
XmScrollBarCallbackStruct	*cbs;
{
    int			val = cbs->value;
    FILTER_VAL		nvb, nvt;
    Bool		change = FALSE;

    if (VB == val) return;		/* no change, actually		*/
    nvb = val;
    nvt = VT + val - VB;
    if (nvt > VMAX) {
	nvb -= nvt - VMAX;
	nvt = VMAX;
	change = TRUE;
    }
    VB = nvb;
    VT = nvt;
    replot_whole();
    if (change) adjust_sb();
}

/* scroll_plot -- scroll plot to new time location
 *
 * TIME	ntl;
 * scroll_plot(ntl);
 *
 * TL is changed to ntl, the plot is scrolled, and TR is changed
 * appropriately.  scroll_plot doesn't check if TL is reasonable;
 * that's the caller's responsibility.
 */
void
scroll_plot(ntl)
TIME	ntl;
{
    Display		*d = XtDisplay(plot);
    Window		w = XtWindow(plot);
    TIME		otl = app_data.tl;
    double		otlfrac = app_data.tlfrac;
    TIME		otr = TR;
    Dimension		displace;
    int			adjust;

    if (ntl == otl) return;
    TR += ntl - otl;
    app_data.tl = ntl;
    app_data.tlfrac = 0.0;
    adjust = adjust_displace(otl, otlfrac, otr, &displace);
    if (AUTOSCALE_ALWAYS && set_auto_scale()) {
	replot_whole();
    }
    else if (0 > adjust) {
	XCopyArea(d, w, w, PLOT_GC,
		  PRX + V_TICLEN + 1, PRY, PRW - displace - V_TICLEN - 1,
		  PRH, PRX + displace + V_TICLEN + 1, PRY);
	replot_part(PRX, PRY, displace + V_TICLEN + 1, PRH);
    }
    else if (0 < adjust) {
	XCopyArea(d, w, w, PLOT_GC,
		  PRX + displace, PRY, PRW - displace - V_TICLEN - 1, PRH,
		  PRX, PRY);
	replot_part(PRX + PRW - displace - V_TICLEN - 1, PRY,
		    displace + V_TICLEN + 1, PRH);
    }
    else {
	replot_whole();
    }
}

/* adjust_displace -- adjust displacement to pixel boundary
 *
 * TIME		old_tl;
 * double	old_tlfrac;
 * TIME		old_tr;
 * Dimension	displacement;
 * if (adjust_displace(old_tl, old_tlfrac, old_tr, &displacement)) adjusted...
 *
 * XCopyArea is used in scrolling operations to avoid unnecessary
 * plotting.  However, for XCopyArea to work right, the scroll must
 * correspond precisely to an integral number of pixels, or subsequent
 * plotting may not line up correctly.  adjust_displace determines
 * whether any adjustment to TL and TR is necessary, and does it if
 * so.  The scrollbar is also adjusted.  The function value is <0 for
 * an overlapping scroll to the left (meaning window contents need to
 * be copied to the right), >0 for an overlapping scroll to the right,
 * and 0 if the new window contents don't overlap the old.  If the
 * return value is non-zero, displacement is set to the number of
 * pixels current window contents need to be moved.
 */
static int
adjust_displace(otl, otlfrac, otr, dp)
TIME		otl;
double		otlfrac;
TIME		otr;
Dimension	*dp;
{
    TIME		stl = app_data.tl; /* save, just in case	*/
    double		stlfrac = app_data.tlfrac;
    TIME		str = TR;
    TIME		t;
    Bool		adjusted = FALSE;

    if ((TR < otr) && (TR > otl)) {
	/*
	 * scroll left: choose TL & TR so that otl begins a new pixel
	 */
	*dp = -floor(TIME_to_Dim_sf * (TL - otl - otlfrac));
	app_data.tl = floor(otl + otlfrac - (*dp)/TIME_to_Dim_sf);
	app_data.tlfrac = otl + otlfrac - (*dp)/TIME_to_Dim_sf - app_data.tl;
	if (app_data.tlfrac >= 1.0) {
	    app_data.tl++;
	    app_data.tlfrac--;
	}
	TR = otr - otl + app_data.tl;
	if (TL < TMIN) {
	    app_data.tl = stl;
	    app_data.tlfrac = stlfrac;
	    TR = str;
	    adjust_sb();
	    return(0);
	}
	adjust_sb();
	return(-1);
    }
    else if ((TL > otl) && (TL < otr)) {
	/*
	 * scroll right: choose TL and TR so that otr ends a pixel
	 */
	*dp = ceil(TIME_to_Dim_sf * (TL - otl - otlfrac));
	app_data.tl = floor(otl + otlfrac + (*dp)/TIME_to_Dim_sf);
	app_data.tlfrac = otl + otlfrac + (*dp)/TIME_to_Dim_sf - app_data.tl;
	if (app_data.tlfrac >= 1.0) {
	    app_data.tl++;
	    app_data.tlfrac--;
	}
	TR = otr - otl + app_data.tl;
	if (TR > TMAX) {
	    app_data.tl = stl;
	    app_data.tlfrac = stlfrac;
	    TR = str;
	    adjust_sb();
	    return(0);
	}
	adjust_sb();
	return(1);
    }
    else {
	return(0);
    }
}

/* generate expose event for whole plot window				*/
void
expose_whole()
{
    XEvent		xev;
    XExposeEvent	*xex = &xev.xexpose;

    while(				/* flush existing exposures	*/
	XCheckTypedWindowEvent(XtDisplay(plot), XtWindow(plot),
			       Expose, &xev)
    );
    xex->type = Expose;
    xex->display = XtDisplay(plot);
    xex->window = XtWindow(plot);
    xex->x = 0;
    xex->y = 0;
    xex->width = PWW;
    xex->height = PWH;
    xex->count = 0;
    ret = XSendEvent(XtDisplay(plot), XtWindow(plot),
		     TRUE, ExposureMask, &xev);
}

static void
create_plot_GC()
{
    Display		*d = XtDisplay(plot);
    Window		w = XtWindow(plot);
    XGCValues		vals;

    XtVaGetValues(plot,
	XmNforeground, &vals.foreground,
	XmNbackground, &vals.background,
    NULL);
    vals.cap_style = CapRound;
    vals.font = PLOT_FONT;
    PLOT_GC = XCreateGC(d, w,
			GCForeground | GCBackground | GCCapStyle | GCFont,
			&vals);
    set_plot_region();
}

static void
create_axes_GC()
{
    Display		*d = XtDisplay(plot);
    Window		w = XtWindow(plot);
    XGCValues		vals;

    XtVaGetValues(plot,
	XmNforeground, &vals.foreground,
	XmNbackground, &vals.background,
    NULL);
    vals.font = PLOT_FONT;
    AXES_GC = XCreateGC(d, w,
			GCForeground | GCBackground | GCFont,
			&vals);
}

static INTERVAL *
alloc_interval()
{
    INTERVAL		*ip;

    ip = XtNew(INTERVAL);
    ip->i_nxt = NULL;
    return(ip);
}

static void
free_interval(ip)
INTERVAL	*ip;
{
    Free(ip);
}

static void
free_intervals(ip)
INTERVAL	*ip;
{
    INTERVAL		*jp;

    while(NULL != ip) {
	jp = ip->i_nxt;
	Free(ip);
	ip = jp;
    }
}

static void
merge_interval(x, width, ipp)
Dimension	x;
Dimension	width;
INTERVAL	**ipp;
{
    INTERVAL		**fpp;
    INTERVAL		*lp;
    INTERVAL		*np;
    Dimension		xh = x+width;
    Dimension		ixh;
    
    np = alloc_interval();
    /*
     * find the first overlapping interval
     */
    for(
	fpp = ipp;
	(NULL != *fpp) && (x > (*fpp)->x + (*fpp)->width + 1);
	fpp = &(*fpp)->i_nxt
    );
    /*
     * if there is none, create new interval and link in
     */
    if (
	(NULL == *fpp) ||
	((*fpp)->x > xh+1)
    ) {
	np->x = x;
	np->width = width;
	np->i_nxt = *fpp;
	*fpp = np;
	return;
    }
    /*
     * There was one, *fpp points to it.  Now find the last
     */
    for(
	lp = *fpp;
	(NULL != lp->i_nxt) && (lp->i_nxt->x <= xh+1);
	lp = lp->i_nxt
    );
    /*
     * OK: create the composite interval and replace the others
     */
    np->x = min(x, (*fpp)->x);
    ixh = lp->x + lp->width;
    xh = max(xh, ixh);
    np->width = xh - np->x;
    np->i_nxt = lp->i_nxt;
    lp->i_nxt = NULL;
    lp = *fpp;
    *fpp = np;
    free_intervals(lp);
}

/* plot_grexp -- action procedure for Graphics Expose in plot widget	*/
static void
plot_grexp(widget, ev, args, nargs)
Widget			widget;
XGraphicsExposeEvent	*ev;
String			*args;
int			*nargs;
{
    XmDrawingAreaCallbackStruct	cbs;
    XExposeEvent		ex;

    if (0 != *nargs) error("can't happen: plot_grexp");
    ex.type = Expose;			/* just fake an Expose		*/
    ex.serial = ev->serial;
    ex.send_event = ev->send_event;
    ex.display = ev->display;
    ex.window = XtWindow(widget);
    ex.x = ev->x;
    ex.y = ev->y;
    ex.width = ev->width;
    ex.height = ev->height;
    ex.count = ev->count;
    cbs.event = (XEvent *) &ex;
    cbs.reason = XmCR_EXPOSE;
    cbs.window = XtWindow(widget);
    plot_expose(widget, NULL, &cbs);
}
