/* files.c -- file-handling stuff for xdatplot				*/
/*
 * Copyright (c) 1993  Leon Avery
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2, or (at
 * your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Send questions or comments on xdatplot to:
 *
 * Leon Avery
 * Department of Biochemistry
 * University of Texas Southwestern Medical Center
 * 5323 Harry Hines Blvd
 * Dallas, TX  75235-9038
 *
 * leon@eatworms.swmed.edu
 */

#include "xdatplot.h"

typedef	struct UCP {			/* compressed file conversions	*/
    String	ext;
    String	cmd;
}		UCP;

#ifdef	__STDC__
static	Bool	new_datafile(String);
static	int	compile_ucp(String, UCP **);
static	UCP	*lookup_ucp(String, UCP *, int);
#else	/* __STDC__ */
static	Bool	new_datafile();
static	int	compile_ucp();
static	UCP	*lookup_ucp();
#endif

UCP	*cucp = NULL;			/* current uncompression	*/

/* new_file -- open a new file, set globals appropriately		*/
void
new_file(name)
String		name;
{
					/* old destination filter	*/
    FILTER		*olddst = destination_filter(&DF_FIL);
    String		**params;	/* old filter parameters	*/
    String		xname;		/* xdp file name		*/
    String		dname;		/* data file name		*/
    Bool		ignored;

    dprintf1("new_file(\"%s\")\n", name);
    /*
     * Save filter chain, then close old file
     */
    params = save_filter_params(olddst);
    close_file(olddst);
    if (NULL == name) {			/* (program starting)		*/
	restore_filter_params(olddst, params);
	return;
    }
    /*
     * if it's an xdp file, get the data file name.  Otherwise, try using
     * name itself as the data_file.
     */
    if (NULL == (dname = xdp_get_datafile(name))) {
	xname = NULL;
	dname = XtNewString(name);
    }
    else {
	xname = XtNewString(name);
    }
    /*
     * Open the datafile
     */
    if (new_datafile(dname)) {
	Nfree(xname);
	Nfree(dname);
	return;				/* error already reported	*/
    }
    Nfree(dname);
    connect_filters(NULL, &DF_FIL, olddst);
    /*
     * read xdp file again, this time for widget vals.  If error, don't
     * report (xdp_get_values already did that), and don't abort,
     * since we HAVE managed to open a datafile.
     */
    if (
	(NULL != xname) &&
	xdp_get_values(xname)
    ) {}				/* error ignored		*/
    /*
     * restore or destroy old filter chain
     */
    if (				/* old chain disconnected	*/
	(NULL == olddst) ||
	(NULL == source_filter(olddst))
    ) {
	FILTER		*fp, *gp;	/* kill it			*/

	discard_filter_params(params);
	for(
	    fp = olddst;
	    NULL != fp;
	    fp = gp
	) {
	    gp = destination_filter(fp);
	    kill_filter(fp);
	    Free(fp);
	}
    }
					/* still connected: restore	*/
    else if (&DF_FIL == source_filter(olddst)) {
	restore_filter_params(olddst, params);
    }
    else
	error("new_file: can't happen\n");
    for(				/* C_FILTER -> end of chain	*/
	C_FILTER = &DF_FIL;
	NULL != destination_filter(C_FILTER);
	C_FILTER = destination_filter(C_FILTER)
    );
    /*
     * everything properly opened.  Update display.
     */
    Nfree(app_data.xdp_file);		/* save xdp file name		*/
    app_data.xdp_file = xname;
    init_plot_params(TRUE);
    make_current_filters();
    if (AUTOSCALE_ALWAYS) {
	set_auto_scale();
    }
}

static Bool
new_datafile(dname)
String		dname;
{
    static UCP		*fct = NULL;
    static int		nfct = -1;
    static UCP		*ucp = NULL;
    static int		nucp = -1;

    char		ebuf[LLEN];
    FILE		*inf;
    FILTER_DATA		data;
    DF_DATA		*dp = &data.df_data;
    char		cbuf[LLEN];
    String		tname = NULL;
    Bool		ok;
    V_INFO		*finfo;

    /*
     * try to open as data file filter
     */
    if (NULL == (inf = fopen(dname, "r"))) {
	sprintf(ebuf, "unable to open %s", dname);
	PU_error(ebuf, "formats.html");
	return(TRUE);
    }
    dp->stream = inf;
    dp->fname = dname;
    dp->bufsiz = XD_BUFSIZ;
    kill_filter(&DF_FIL);
    if (make_filter(F_DF, &data, &DF_FIL)) {
	strcpy(ebuf, V_errmsg);		/* save error msg; may need it	*/
	fclose(inf);
	/*
	 * Didn't work.  Maybe compressed or needs conversion?
	 */
	if (NULL == ucp) {
	    nucp = compile_ucp(app_data.uncompress, &ucp);
	    nfct = compile_ucp(app_data.file_convert, &fct);
	}
	if (
	    (NULL == (cucp = lookup_ucp(dname, fct, nfct))) &&
	    (NULL == (cucp = lookup_ucp(dname, ucp, nucp)))
	) {
	    PU_error(ebuf, "formats.html");
	    return(TRUE);
	}
	if (NULL == (tname = tempnam(NULL, "xdat.")))
	    error("no memory");
	sprintf(cbuf, cucp->cmd, dname, tname);
	ret = system(cbuf);
	if (ok = ((127 != ret) && (0 == (0xff00 & ret)))) {
	    if (ok = (NULL != (dp->stream = inf = fopen(tname, "r")))) {
		ok = !make_filter(F_DF, &data, &DF_FIL);
		if (!ok) fclose(inf);
	    }
	}
	if (!ok) {
	    ret = unlink(tname);
	    PU_error(ebuf, "formats.html");
	    return(TRUE);
	}
    }
    /*
     * a data file has been (apparently) correctly opened.  Set resources.
     */
    Nfree(DATA_FILE);
    DATA_FILE = XtNewString(dname);
    Nfree(TFNAME);
    TFNAME = tname;
    finfo = Vinfo();
    set_comment(finfo->V_comment);
    return(FALSE);
}

void
close_file(olddst)
FILTER		*olddst;
{
    char		lbuf[LLEN];
    FILE		*inf;
    DF_DATA		*dp;

    marks_filter_end();
    cancel_redraw();
    clear_all_marks();
    if (is_active_filter(&DF_FIL)) {
	dp = &(filter_data(&DF_FIL))->df_data;
	if (NULL != dp) {
	    inf = dp->stream;
	    kill_filter(&DF_FIL);
	    if ((NULL != inf) && (EOF == fclose(inf))) {
		sprintf(lbuf, "unable to close %s", DATA_FILE);
		error(lbuf);
	    }
	}
    }
    set_comment("");
    make_filter(F_EMPTY, NULL, &DF_FIL);
    connect_filters(NULL, &DF_FIL, olddst);
    IS_CURSOR_END = IS_CURSOR = FALSE;
    if (t_field != NO_WIDGET) XtVaSetValues(t_field, XmNvalue, "", NULL);
    if (v_field != NO_WIDGET) XtVaSetValues(v_field, XmNvalue, "", NULL);
    if (NULL != TFNAME) {
	unlink(TFNAME);
	free(TFNAME);
	TFNAME = NULL;
    }
    Nfree(DATA_FILE);
    Nfree(app_data.xdp_file);
    expose_whole();
}

void
kill_filters()
{
    FILTER	*gp;
    FILTER	*fp = destination_filter(&DF_FIL);

    while(NULL != fp) {
	gp = destination_filter(fp);
	kill_filter(fp);
	fp = gp;
    }
}
    
/* init_plot_params -- set up plot parameters for newly opened file	*/
void
init_plot_params(newfile)
Bool		newfile;
{
    TIME		otl = TL;
    TIME		otr = TR;
    double		ot_con = T_CON;
    TIME		tc;

    /*
     * set time and voltage scales
     */
    if (!T_MUL_DEF) set_tmul(T_MULTIPLIER);
    T_CON = filter_tmul(C_FILTER) / T_GAIN;
    if (T_TIME_DEF) {
	tc = TMIN * T_CON;
    }
    else {
	tc = T_TIME;
    }
    if (newfile) {
	app_data.tl = (tc - T_WIDTH/2.0) / T_CON;
	app_data.tlfrac = 0.0;
	app_data.tr = TL + T_WIDTH / T_CON;
    }
    else {
	app_data.tl = otl * ot_con / T_CON;
	app_data.tlfrac = 0.0;
	app_data.tr = otr * ot_con / T_CON;
    }
    FORCE_RANGE(app_data.tl, app_data.tr, TMIN, TMAX);

    if (!V_MUL_DEF) set_vmul(V_MULTIPLIER);
    V_CON = filter_fmul(C_FILTER) / V_GAIN;
    if (newfile) {
	VB = VMIN;
	VT = VMAX;
	if (!V_HEIGHT_DEF) {
	    VB = (VMIN + VMAX - V_HEIGHT)/2.0;
	    VT = (VMIN + VMAX + V_HEIGHT)/2.0;
	}
    }
    FORCE_RANGE(VB, VT, VMIN, VMAX);

    copy_resources_into_fields();
    adjust_sb();			/* adjust scrollbars		*/
}

String
default_xdp_file()
{
    char		lbuf[LLEN];
    String		dot;
    String		dir;

    if (NULL != app_data.xdp_file)
	return(XtNewString(app_data.xdp_file));
    if (NULL == DATA_FILE)
	return(XtNewString(""));
    strncpy(lbuf, DATA_FILE, LLEN-1);
    lbuf[LLEN-1] = '\0';
    if (NULL != TFNAME) {		/* compressed: delete extension	*/
	if (NULL != cucp)
	    lbuf[strlen(lbuf) - strlen(cucp->ext)] = '\0';
    }
    if (
	(NULL != (dot = STRRCHR(lbuf, EXTSEP))) &&
	((NULL == (dir = STRRCHR(lbuf, DIRSEP))) || (dir < dot))
    ) {
	strncpy(dot, ".xdp", lbuf+LLEN-dot-1);
    }
    else {
	strncat(lbuf, ".xdp", LLEN-strlen(lbuf)-1);
    }
    return(XtNewString(lbuf));
}

double
default_tmul()
{
    if (F_DF == filter_type(&DF_FIL))
	return(Vinfo()->V_tscale);
    else
	return(filter_tmul(&DF_FIL));
}

double
default_vmul()
{
    if (F_DF == filter_type(&DF_FIL))
	return(Vinfo()->V_vscale);
    else
	return(filter_fmul(&DF_FIL));
}

void
set_tmul(d)
double		d;
{
    if (isnan(d)) d = default_tmul();
    if (d != filter_tmul(&DF_FIL))
	force_filter_tmul(&DF_FIL, d);
}

void
set_vmul(d)
double		d;
{
    if (isnan(d)) d = default_vmul();
    if (d != filter_fmul(&DF_FIL))
	force_filter_fmul(&DF_FIL, d);
}

static int
compile_ucp(s, ucpp)
String		s;
UCP		**ucpp;
{
    char		lbuf[LLEN+1], ebuf[LLEN+1], cbuf[LLEN+1];
    register int	i;
    register String	t, u;
    int			n;
    UCP			*ucp;

    for(				/* count conversions		*/
	t = s, n = 0;
	NULL != (u = STRCHR(t, '\n'));
	t = u+1, n++
    );
    if ('\0' != *t) n++;
    ucp = *ucpp = (UCP *) XtMalloc(max(n, 1) * sizeof(UCP));
    for(				/* parse them			*/
	t = s;
	'\0' != *t;
	t = u
    ) {
        if (NULL == (u = STRCHR(t, '\n'))) {
	    u = t + strlen(t);
	}
	else {
	    u++;
	}
	if (LLEN <= u-t) error("uncompress line too long");
	strncpy(lbuf, t, u-t);
	lbuf[u-t] = '\0';
	for(t=lbuf; isspace(*t); t++);
	if (':' == *t) {
	    ebuf[0] = '\0';
	    if (1 != sscanf(t, ": %[^\n]", cbuf))
		error("invalid uncompress line");
	}
	else {
	    if (2 != sscanf(t, "%[^:\n]: %[^\n]", ebuf, cbuf))
		error("invalid uncompress line");
	}
	ucp->ext = XtNewString(ebuf);
	ucp->cmd = XtNewString(cbuf);
	ucp++;
    }
    return(n);
}

static UCP *
lookup_ucp(dname, ucp, nucp)
String	dname;
UCP	*ucp;
int	nucp;
{
    register int	i;

    for(i=0; i<nucp; i++) {
	if (0 == strcmp(ucp[i].ext,
			dname + strlen(dname) - strlen(ucp[i].ext))) break;
    }
    if (i >= nucp) return(NULL);
    return(&ucp[i]);
}
