/**********************************************************************
 *
 *	xtmexecute.c
 *
 **********************************************************************/

/*

Copyright 1986, 1987, 1988, 1989 by Hewlett-Packard Corporation
Copyright 1986, 1987, 1988, 1989 by the Massachusetts Institute of Technology

Permission to use, copy, modify, and distribute this
software and its documentation for any purpose and without
fee is hereby granted, provided that the above copyright
notice appear in all copies and that both that copyright
notice and this permission notice appear in supporting
documentation, and that the name of M.I.T. not be used in
advertising or publicity pertaining to distribution of the
software without specific, written prior permission.

Hewlett-Packard and M.I.T. make no representations about the 
suitability of this software for any purpose.  It is provided 
"as is" without express or implied warranty.

This software is not subject to any license of the American
Telephone and Telegraph Company or of the Regents of the
University of California.

*/

/**********************************************************************
 * include files
 **********************************************************************/

/*
 * contains NULL constant
 */
#include  <stdio.h>
/*
 * constants for open
 */
#include  <fcntl.h>
/*
 * for the isspace macro
 */
#include  <ctype.h>
/*
 * constants for ftw call
 */
#include  <ftw.h>
/*
 * ???
 */
#include  <sys/types.h>
/*
 * ???
 */
#include  <sys/stat.h>
/*
 * ???
 */
#include  <X11/Xmd.h>
/*
 * X library files
 */
#include  <X11/Xlib.h>
/*
 * ???
 */
#include  <X11/extensions/xtestext1.h>
/*
 * client exerciser constants
 */
#include  <xtm.h>

/**********************************************************************
 * external variables
 **********************************************************************/

extern int		XTestFakeAckType;
extern int		errno;
extern char		version[];

/**********************************************************************
 * variables
 **********************************************************************/

/*
 * deltas for input action timing
 */
struct playback_info	G_playback_info;
/*
 * ???
 */
unsigned long		G_max_input_actions;
/*
 * ???
 */
Display			*G_disp_ptr;
/*
 * Buffer for screen matches
 */
unsigned char          	*G_c_data_buffer;
/*
 * Buffer for screen matches
 */
unsigned char		*G_sb;
/*
 * size of matbuf
 */
unsigned int           	G_sb_size;
/*
 * invocation options
 */
int	           	G_update_flag;
int	           	G_continue_flag;
/*
 * undocumented debug invocation option
 */
int	           	G_debug_flag;
/*
 * file pointer for the error message file (stderr)
 */
FILE            	*G_msg_file;
/*
 * file pointer for the log file
 */
FILE            	*G_log_file;
/*
 * file descriptor for the failure file
 */
int			G_fail_file_fd;
/*
 * ???
 */
unsigned int		G_all_events;
/*
 * ???
 */
unsigned int		G_all_pkt;
/*
 * invocation option for executing entire directory trees
 */
int			G_Directory_ex_flag;
/*
 * if true, read all image data from the root window instead of from
 * the specified window
 */
int			G_read_from_root_flag;
/*
 * if true then use checksums for matches
 */
int			G_checksum_flag;
/*
 * if true then compare the color map information in matches
 */
int			G_compare_color_map_flag;

/**********************************************************************
 * function definitions
 **********************************************************************/

void			readchk();
void			wrchk();
void			wrchk2();
char			*strrchr();
void			send_input_actions();
void			check_send();
INT32			fix_time();
void			comment1();
void			sigcatch();
void			make_files();
int			xerror();
char			*getkeysym();
char                    *malloc();
void			parse_t_string();
void			parse_delta_factor();
void			xtmexecute_parse_err();
Display			*XOpenDisplay();
FILE			*makefp();

/**********************************************************************
 * functions
 **********************************************************************/

/**********************************************************************
 *
 *	execute
 *
 *	Processes a test script.  Types logged input actions to the process
 *	being tested, and compares the results with those previously saved
 *	in the test script file.
 */
static int
execute(script_fd, script_name, new_script_fd)
/*
 * file descriptor for internal format test script file
 */
int	script_fd;
/*
 * pointer to name of test script file
 */
char	*script_name;
/*
 * file descriptor for new test script file (used only in update mode)
 */
int	new_script_fd;
{
	/*
	 * character read
	 */
	char	c;
	/*
	 * holds which match this is in the test script
	 */
	int	match_index = 0;
	/*
	 * execute return code
	 */
	int	exret;
	/*
	 * 1 = test script failed
	 */
	int	faflg = 0;
	/*
	 * set up signal catching
	 */
	sigcatch();

	(void) fprintf(G_log_file,
		       "\nExecuting test script '%s'\n\n",
		       script_name);
	/*
	 * check for test script version
	 */
	if (read_version_number(script_fd, new_script_fd) != 0)
	{
		(void) fprintf(G_log_file,
			       "Missing or invalid version number in test script '%s' - test aborted.\n",
			       script_name);
		(void) fprintf(G_msg_file,
			       "Missing or invalid version number in test script '%s' - test aborted.\n",
			       script_name);
		return(-1);
	}
	/*
	 * read  indicator
	 */
	while (read(script_fd, &c, 1) > 0)
	{
		/*
		 * switch on indicator
		 */
		switch (c)
		{
		case 'c':
			/*
			 * process a comment
			 */
			comment1(script_fd, new_script_fd);
			break;

		case 'k':
			/*
			 * simulate input action input
			 */
			send_input_actions(script_fd, new_script_fd);
			break;

		case 'm':
			/*
			 * process a match request
			 */
			if ((exret = exmatch(script_fd,
					     new_script_fd,
					     script_name,
					     (++match_index))) != 0)
			{
				faflg = exret;
			}
			break;

		case 'T':
			/*
			 * toggle a mode
			 */
			faflg = toggle_mode(script_fd,
					    new_script_fd,
					    script_name);
			break;

		default: 
			(void) fprintf(G_log_file,
				       "Error in test script '%s' - test aborted.\n",
				       script_name);
			(void) fprintf(G_log_file,
			       "       Unrecognized character in test script is (decimal): %d\n",
			       c       );
			(void) fprintf(G_msg_file,
				       "Error in test script '%s' - test aborted.\n",
			               script_name);
			(void) fprintf(G_msg_file,
			       "       Unrecognized character in test script is (decimal): %d\n",
			       c       );
			return(-1);
		}
	}
	return(faflg);
}

/**********************************************************************
 *
 *      read_version_number
 *
 *      This function reads the version number from a test script.  It
 *	returns non-zero if the version number is missing or doesn't
 *	match the version number of xtmexecute.
 */
static int
read_version_number(script_fd, new_script_fd)
/*
 * file descriptor of internal format test script file
 */
int     script_fd;
/*
 * file descriptor for new test script file (used only in update mode)
 */
int	new_script_fd;
{
	/*
	 * holds a character read from the test script
	 */
	char	c;
	/*
	 * holds the version number read from the test script
	 */
	int	version_number;
        /*
         * holds the version number string
         */
        char    buf[WRITE_BUFFER_SIZE];

	/*
	 * read a character from the test script
	 */
	if (read(script_fd, &c, 1) != 1)
	{
		return(1);
	}
	/*
	 * if the character is not a 'v' for the version number, then error
	 */
	if (c != 'v')
	{
		return(1);
	}
	/*
	 * read the version number from the test script
	 */
	version_number = getint(script_fd);
	/*
	 * if the version number of the test script and xtmexecute don't
	 * match, then error.
	 */
	if (version_number != VERSION_NUMBER)
	{
		return(1);
	}
	/*
	 * if in update mode write the version number to the new test script
	 */
	if (G_update_flag)
	{
		/*
		 * write the version number to a string
		 */
		(void) sprintf(buf, "v%d ", version_number);
		/*
		 * write the string to the test script
		 */
		wrchk(new_script_fd, buf);
	}
	/*
	 * if we get to here, we found the correct version number
	 */
	return(0);
}

/*************************************************************************
 *
 *	send_input_actions
 *
 *	This routine sends the recorded input actions to the process being
 *	tested.
 */
static void
send_input_actions(script_fd, new_script_fd)
/*
 * file descriptor for internal format test script file
 */
int	script_fd;
/*
 * file descriptor for new test script file (used only in update mode)
 */
int	new_script_fd;
{
	/*
	 * ???
	 */
	char		buf[WRITE_BUFFER_SIZE];
	/*
	 * ???
	 */
	XEvent		event;
	/*
	 * ???
	 */
	int		i;
	/*
	 * data type
	 */
	unsigned char	type;
	/*
	 * ???
	 */
	int		pindex = 0;
	/*
	 * ???
	 */
	int		sindex = 0;
	/*
	 * ???
	 */
	int		size = 0;
	/*
	 * buffer for input actions
	 */
	unsigned char	input_action_buffer[INPUT_ACTION_BUFFER_SIZE];
	/*
	 * size of input action data
	 */
	int	input_action_count;
	/*
	 * number of events sent
	 */
	int		evcnt = 0;
	/*
	 * ???
	 */
	CARD8		code;
	/*
	 * ???
	 */
	int		xdisp;
	/*
	 * ???
	 */
	int		ydisp;
	/*
	 * ???
	 */
	XTestKeyInfo	*fk;
	/*
	 * ???
	 */
	XTestMotionInfo	*fm;
	/*
	 * ???
	 */
	XTestJumpInfo	*fj;
	/*
	 * ???
	 */
	XTestDelayInfo	*ft;
	/*
	 * holds a INT32 sized number for the call to fix_time
	 */
	INT32		int32_temp;

	input_action_count = getint(script_fd);
	/*
	 * read in saved input actions
	 */
	readchk(script_fd,
		(char *) input_action_buffer,
		(unsigned int) input_action_count,
		"Error reading test script file input actions.\n");
	if (G_debug_flag)
	{
		(void) fprintf(G_msg_file,
			       "input action count is %d.\n",
			       input_action_count);
	}
	for (evcnt = 0, pindex = 0; pindex < input_action_count; evcnt++)
	{
		/*
		 * get type of data
		 */
		type = input_action_buffer[pindex] & XTestACTION_TYPE_MASK;
		switch (type)
		{
		case XTestKEY_ACTION:
			fk = (XTestKeyInfo *) &input_action_buffer[pindex];
			/*
			 * the value returned from fix_time is guaranteed
			 * to fit in fk->delay_time
			 */
			int32_temp = fk->delay_time;
			fk->delay_time = fix_time(int32_temp, SHORT_DELAY_SIZE);
			if (G_debug_flag)
			{
				G_all_events++;
				code = (CARD8) fk->keycode;
				if ((fk->header & XTestKEY_STATE_MASK) ==
				    XTestKEY_DOWN)
				{
					(void) fprintf(G_msg_file,
						       "KEY_DOWN\tDEV[%01x]\t",
						       XTestUnpackDeviceID(fk->header));
				}
				else
				{
					(void) fprintf(G_msg_file,
						       "KEY_UP\t\tDEV[%01x]\t",
						       XTestUnpackDeviceID(fk->header));
				}
				(void) fprintf(G_msg_file,
					       "KEY[%s]\t\t",
					       getkeysym(code,
							 G_disp_ptr));
				(void) fprintf(G_msg_file,
					       "TIME (ms)[%d]\t",
					       fk->delay_time);
				(void) fprintf(G_msg_file,
					       "COUNT [%d]\n",
					       G_all_events);
			}
			check_send(sizeof(XTestKeyInfo),
				   &evcnt,
				   &sindex,
				   &pindex,
				   &size,
				   input_action_buffer);
			break;

		case XTestMOTION_ACTION:
			fm = (XTestMotionInfo *) &input_action_buffer[pindex];
			/*
			 * the value returned from fix_time is guaranteed
			 * to fit in fm->delay_time
			 */
			int32_temp = fm->delay_time;
			fm->delay_time = fix_time(int32_temp, SHORT_DELAY_SIZE);
			if (G_debug_flag)
			{
				G_all_events++;
				xdisp = XTestUnpackXMotionValue(fm->motion_data);
				ydisp = XTestUnpackYMotionValue(fm->motion_data);
				(void) fprintf(G_msg_file,
					       "MOTION\t\tDEV[%01x]\t",
					       XTestUnpackDeviceID(fm->header));
				if ((fm->header & XTestX_SIGN_BIT_MASK) ==
				    XTestX_NEGATIVE)
				{
					xdisp = -xdisp;
				}
				if ((fm->header & XTestY_SIGN_BIT_MASK) ==
				    XTestY_NEGATIVE)
				{
					ydisp = -ydisp;
				}
				(void) fprintf(G_msg_file, "X[%d]\t", xdisp);
				(void) fprintf(G_msg_file, "Y[%d]\t", ydisp);
				(void) fprintf(G_msg_file,
					       "TIME (ms)[%d]\t",
					       fm->delay_time);
				(void) fprintf(G_msg_file,
					       "COUNT [%d]\n",
					       G_all_events);
			}
			check_send(sizeof(XTestMotionInfo),
				   &evcnt,
				   &sindex,
				   &pindex,
				   &size,
				   input_action_buffer);
			break;

		case XTestJUMP_ACTION:
			fj = (XTestJumpInfo *) &input_action_buffer[pindex];
			/*
			 * the value returned from fix_time is guaranteed
			 * to fit in fj->delay_time
			 */
			int32_temp = fj->delay_time;
			fj->delay_time = fix_time(int32_temp, SHORT_DELAY_SIZE);
			if (G_debug_flag)
			{
				G_all_events++;
				(void) fprintf(G_msg_file,
					       "JUMP\t\tDEV[%01x]\t",
					       XTestUnpackDeviceID(fj->header));
				(void) fprintf(G_msg_file,
					       "X[%04d]\t",
					       fj->jumpx);
				(void) fprintf(G_msg_file,
					       "Y[%04d]\t",
					       fj->jumpy);
				(void) fprintf(G_msg_file,
					       "TIME (ms)[%d]\t",
					       fj->delay_time);
				(void) fprintf(G_msg_file,
					       "COUNT [%d]\n",
					       G_all_events);
			}
			check_send(sizeof(XTestJumpInfo),
				   &evcnt,
				   &sindex,
				   &pindex,
				   &size,
				   input_action_buffer);
			break;

		default:
			type = XTestUnpackDeviceID(input_action_buffer[pindex]);
			if (type == XTestDELAY_DEVICE_ID)
			{
				ft = (XTestDelayInfo *) &input_action_buffer[pindex];
				ft->delay_time = fix_time(ft->delay_time,
							  LONG_DELAY_SIZE);
				if (G_debug_flag)
				{
					G_all_events++;
					(void) fprintf(G_msg_file,
						       "TIME\t\t\t\t\t");
					(void) fprintf(G_msg_file,
						       "TIME (ms)[%ld]\t",
						       ft->delay_time);
					(void) fprintf(G_msg_file,
						       "COUNT [%d]\n",
						       G_all_events);
				}
				check_send(sizeof(XTestDelayInfo),
					   &evcnt,
					   &sindex,
					   &pindex,
					   &size,
					   input_action_buffer);
			}
			else
			{
				if (G_debug_flag)
				{
					(void) fprintf(G_msg_file,
						       "data of type %x on execute.\n",
						       type);
					(void) fprintf(G_msg_file,
						       "pindex = %d\n",
						       pindex);
					(void) fprintf(G_msg_file,
						       "sindex = %d\n",
						       sindex);
					(void) fprintf(G_msg_file,
						       "evcnt = %d\n",
						       evcnt);
					(void) fprintf(G_msg_file,
						       "input_action_count = %d\n",
						       input_action_count);
				}
				pindex++;
			}
			break;
		}
	}
	if (G_debug_flag)
	{
		(void) fprintf(G_msg_file, "Sending packet %d\n", G_all_pkt++);
		for (i = 0; i < size; i++)
		{
			(void) fprintf(G_msg_file,
				       "%x ",
				       input_action_buffer[sindex + i]);
		}
		(void) fprintf(G_msg_file, "\n");
	}
	/*
	 * this function is defined in the X testing extension to Xlib
	 */
	if (XTestFakeInput(G_disp_ptr,
		           (char *) &input_action_buffer[sindex],
		           size,
		           XTestFAKE_ACK_REQUEST) == -1)
	{
		(void) fprintf(G_msg_file, "XTestFakeInput failed\n");
		exit(1);
	}
	XFlush(G_disp_ptr);
	while (1)
	{
		XNextEvent(G_disp_ptr, &event);
		if (event.type  == XTestFakeAckType)
		{
			if (G_debug_flag)
			{
				(void) fprintf(G_msg_file,
					       "Received ack for packet %d\n",
				               G_all_pkt - 1);
			}
			break;
		}
		else
		{
			(void) fprintf(G_msg_file,
				       "got an event of type %x\n",
				       event.type);
		}
	}
	evcnt = 0;
	/*
	 * update mode
	 */
	if (G_update_flag)
	{
		(void) sprintf(buf, "k%d ", input_action_count);
		/*
		 * write indicator to file
		 */
		wrchk(new_script_fd, buf);
		/*
		 * write saved input actions
		 */
		wrchk2(new_script_fd,
		       (char *) input_action_buffer,
		       (unsigned int) input_action_count,
		       "Error writing test script file input actions.\n");
	}
}


/**********************************************************************
 *
 *	check_send
 *
 *	This routine checks for a full packet, and sends it if so.
 * 
 *	Also need to check for reaching the maximum number of input actions
 *	and send ack request before continuing.
 */
static void
check_send(size, evcnt, sindex, pindex, csize, buf)
/*
 * ???
 */
int			size;
/*
 * ???
 */
int			*evcnt;
/*
 * ???
 */
int			*sindex;
/*
 * ???
 */
int			*pindex;
/*
 * ???
 */
int			*csize;
/*
 * ???
 */
unsigned char		*buf;
{
	/*
	 * ???
	 */
	XEvent			event;
	/*
	 * ???
	 */
	int			i;
	/*
	 * ???
	 */
	int			ack_req;

	if (((*csize + size) <= XTestMAX_ACTION_LIST_SIZE) &&
	    (*evcnt < G_max_input_actions))
	{
		*pindex += size;
		*csize += size;
		return;
	}
	if (G_debug_flag)
	{
		(void) fprintf(G_msg_file, "Sending packet %d\n", G_all_pkt++);
		for (i = 0; i < *csize; i++)
		{
			(void) fprintf(G_msg_file, "%x ", buf[*sindex + i]);
		}
		(void) fprintf(G_msg_file, "\n");
	}
	if (*evcnt >= G_max_input_actions)
	{
		ack_req = XTestFAKE_ACK_REQUEST;
	}
	else
	{
		ack_req = XTestFAKE_ACK_NOT_NEEDED;
	}
	/*
	 * this function is defined in the X testing extension to Xlib
	 */
	if (XTestFakeInput(G_disp_ptr,
		           (char *) (&buf[*sindex]),
		           *csize,
		           ack_req) == -1)
	{
		(void) fprintf(G_msg_file, "XTestFakeInput failed\n");
		exit(1);
	}
	XFlush(G_disp_ptr);
	*sindex = *pindex;
	*pindex += size;
	*csize = size;
	if (ack_req == XTestFAKE_ACK_REQUEST)
	{
		while (1)
		{
			XNextEvent(G_disp_ptr, &event);
			if (event.type == XTestFakeAckType)
			{
				if (G_debug_flag)
				{
					(void) fprintf(G_msg_file,
						       "Received ack for packet %d\n",
						       G_all_pkt - 1);
				}
				break;
			}
		}
		*evcnt = 0;
	}
}

/**********************************************************************
 *
 *      toggle_mode
 *
 *      This function processes a mode toggle command (currently only
 *	checksum mode).  It returns -1 if an error occurred, 0 otherwise.
 */
int
toggle_mode(script_fd, new_script_fd, script_name)
/*
 * file descriptor for internal format test script file
 */
int	script_fd;
/*
 * file descriptor for new test script file (used only in update mode)
 */
int	new_script_fd;
/*
 * points to the name of the test script
 */
char	*script_name;
{
	/*
	 * holds a character of input
	 */
	char	c;

	/*
	 * read the type of mode to toggle
	 */
	readchk(script_fd,
		&c,
		1,
		"error reading toggle mode type from test script\n");
	/*
	 * process the mode type
	 */
	switch(c)
	{
	case 'c':
		/*
		 * checksum mode
		 */
		readchk(script_fd,
			&c,
			1,
			"error reading toggle mode type from test script\n");
		switch(c)
		{
		case 'd':
			/*
			 * disable checksum mode
			 */
			G_checksum_flag = 0;
			/*
			 * if in update mode, make sure it gets into the 
			 * new test script file
			 */
			if (G_update_flag)
			{
				wrchk(new_script_fd, "Tcd");
			}
			break;

		case 'e':
			/*
			 * enable checksum mode
			 */
			G_checksum_flag = 1;
			/*
			 * if in update mode, make sure it gets into the 
			 * new test script file
			 */
			if (G_update_flag)
			{
				wrchk(new_script_fd, "Tce");
			}
			break;

		default:
			(void) fprintf(G_log_file,
				       "Error in test script '%s' - test aborted.\n",
				       script_name);
			(void) fprintf(G_log_file,
			       "       Unrecognized character in test script is (decimal): %d\n",
			       c       );
			(void) fprintf(G_msg_file,
				       "Error in test script '%s' - test aborted.\n",
				       script_name);
			(void) fprintf(G_msg_file,
			       "       Unrecognized character in test script is (decimal): %d\n",
			       c       );
			return(-1);
			break;
		}
		break;

	default:
		(void) fprintf(G_log_file,
			       "Error in test script '%s' - test aborted.\n",
			       script_name);
		(void) fprintf(G_log_file,
		       "       Unrecognized character in test script is (decimal): %d\n",
		       c       );
		(void) fprintf(G_msg_file,
			       "Error in test script '%s' - test aborted.\n",
			       script_name);
		(void) fprintf(G_msg_file,
		       "       Unrecognized character in test script is (decimal): %d\n",
		       c       );
		return(-1);
		break;
	}
	/*
	 * if we get to here then eveything is ok
	 */
	return(0);
}

/**********************************************************************
 *
 *      fix_time
 *
 *      This function adjusts delay times in input actions.
 */
INT32
fix_time(delay_time, size_flag)
/*
 * the time to delay in milliseconds
 */
INT32	delay_time;
/*
 * if SHORT_DELAY_SIZE, then delay is limited to XTestSHORT_DELAY_TIME
 */
int	size_flag;
{
	/*
	 * holds the modified delay time
	 */
	INT32	temp;

	temp = delay_time;
	switch(G_playback_info.delta)
	{
	case DEFAULT_DELTA:
		/*
		 * don't fiddle with the time delay
		 */
		break;
	case ADD_TO_INPUT_ACTION_TIMES:
		/*
		 * add the delta factor to the time delays
		 */
		temp += G_playback_info.factor;
		break;
	case SUBTRACT_FROM_INPUT_ACTION_TIMES:
		/*
		 * subtract the delta factor from the time delays
		 */
		temp -= G_playback_info.factor;
		break;
	case MULTIPLY_INPUT_ACTION_TIMES:
		/*
		 * multiply the delta factor times the time delays
		 */
		temp *= G_playback_info.double_factor;
		break;
	case DIVIDE_INPUT_ACTION_TIMES:
		/*
		 * divide the delta factor into the time delays
		 */
		if (G_playback_info.double_factor == 0.0)
		{
			(void) fprintf(G_msg_file,
				       "can't divide time delay by 0\n");
			exit(1);
		}
		temp /= G_playback_info.double_factor;
		break;
	case MINIMUM_INPUT_ACTION_TIME:
		/*
		 * make sure that the time delay is at least as large
		 * as the delta factor
		 */
		if (temp < G_playback_info.factor)
		{
			temp = G_playback_info.factor;
		}
		break;
	default:
		(void) fprintf(G_msg_file,
			       "invalid time delay delta value\n");
		exit(1);
		break;
	}
	/*
	 * if the resulting time delay is less than 0, make it 0
	 */
	if (temp < 0)
	{
		temp = 0;
	}
	/*
	 * if the delay time has to fit in a short delay, then make it fit
	 */
	if ((size_flag == SHORT_DELAY_SIZE) &&
	    (temp > (INT32) XTestSHORT_DELAY_TIME))
	{
		temp = (INT32) XTestSHORT_DELAY_TIME;
	}
	/*
	 * if debugging and the delay time changed, tell user
	 */
	if (G_debug_flag && (temp != delay_time))
	{
		(void) fprintf(G_msg_file,
			       "time delay actually used: [%ld]\n",
			       temp);
	}
	/*
	 * return the (possibly) modified delay time
	 */
	return(temp);
}

/**********************************************************************
 *
 *      comment1
 *
 *      This function copies comments from the test script to
 *      the file pointed to by the file descriptor fd.
 */
static void
comment1(script_fd, new_script_fd)
/*
 * file descriptor for internal format test script file
 */
int	script_fd;
/*
 * file descriptor for new test script file (used only in update mode)
 */
int	new_script_fd;
{
        /*
         * holds the length of the comment string (including a '\0');
         */
        int     cnt;
        /*
         * holds the comment string
         */
        char    buf[WRITE_BUFFER_SIZE];
        /*
         * holds the count during update mode
         */
        char    cbuf[WRITE_BUFFER_SIZE];

        /*
         * get count of comment byte
         */
        cnt = getint(script_fd);
        /*
         * read the bytes
         */
        readchk(script_fd,
		buf,
		(unsigned int) cnt,
		"Error reading comment from test script file.\n");
        /*
         * write comment to log file
         */
        (void) fprintf(G_log_file, "\n%s\n\n", buf);
        /*
         * updating test script file
         */
        if (G_update_flag)
        {
                (void) sprintf(cbuf, "c%d ", cnt);
                /*
                 * write comment indicator
                 */
                wrchk(new_script_fd, cbuf);
                /*
                 * write chars to new test script
                 */
                wrchk2(new_script_fd,
		       buf,
		       (unsigned int) cnt,
		       "Error writing comment to test script file.\n");
        }
}

/**********************************************************************
 *
 *      endmsg
 *
 *      This function outputs the message of the test script's final status.
 */
static int
endmsg(ptr, ret, name)
/*
 * ???
 */
FILE            *ptr;
/*
 * ???
 */
register int    ret;
/*
 * ???
 */
char            *name;
{
        if (G_update_flag)
        {
                if (ret == 0)
                {
                        (void) fprintf(ptr, "\nTest script '%s' updated.\n", name);
                }
                else
                {
                        (void) fprintf(ptr,
				       "\nTest script '%s' updated - match(s) failed or error in test script.\n",
				       name);
                }
        }
        else if (ret == 0)
        {
                (void) fprintf(ptr, "\nTest script '%s' completed.\n", name);
        }
        else
        {
                (void) fprintf(ptr, "\nTest script '%s' failed.\n", name);
        }
}

/**********************************************************************
 *
 *      makefp
 *
 *      File creation routine.
 *      It checks to see if the file exists and deletes it if it does
 *      and the "r" flag was specified, unless the file was a directory.
 *      It returns a file pointer to its caller.
 */
static FILE *
makefp(name)
/*
 * ptr to desired file name
 */
char   *name;
{
        /*
         * file pointer
         */
        FILE            *ptr;

        /*
         * create the file
         */
        ptr = fopen(name, "w");
        /*
         * if fopen failed
         */
        if (ptr == (FILE *) 0)
        {
                /*
                 * write message and exit
                 */
                (void) fprintf(G_msg_file,
			       "Unable to create file '%s'.\n",
			       name);
		(void) fprintf(G_msg_file, "errno was %d\n", errno);
		exit(1);
        }
        /*
         * return the file descriptor
         */
        return(ptr);
}

/**********************************************************************
 *
 *      xtmexecute_close_files
 *
 * This routine closes all files.
 */
static void
xtmexecute_close_files(script_fd,
		       script_name,
		       fail_file_name,
		       log_file_name,
		       new_script_fd,
		       new_script_name)
/*
 * file descriptor for internal format test script file
 */
int	script_fd;
/*
 * pointer to name of test script file
 */
char	*script_name;
/*
 * pointer to name of failure file
 */
char	*fail_file_name;
/*
 * pointer to name of log file
 */
char	*log_file_name;
/*
 * file descriptor for new test script file (used only in update mode)
 */
int	new_script_fd;
/*
 * pointer to the file name for the new test script
 * (used only in update mode)
 */
char	*new_script_name;
{
	/*
	 * close the test script file
	 */
	if (close(script_fd) == -1)
	{
		(void) fprintf(G_msg_file,
			       "error %d while closing '%s'\n",
			       errno,
			       script_name);
		exit(1);
	}
	/*
	 * close the log file
	 */
	if (fclose(G_log_file) != 0)
	{
		(void) fprintf(G_msg_file,
			       "error %d while closing '%s'\n",
			       errno,
			       log_file_name);
		exit(1);
	}
	/*
	 * close the failure file
	 */
	if (close(G_fail_file_fd) == -1)
	{
		(void) fprintf(G_msg_file,
			       "error %d while closing '%s'\n",
			       errno,
			       fail_file_name);
		exit(1);
	}
	/*
	 * if in update mode, replace the old test script
	 * with the new test script
	 */
	if (G_update_flag)
	{
		/*
		 * close the new file
		 */
		if (close(new_script_fd) == -1)
		{
			(void) fprintf(G_msg_file,
				       "error %d while closing '%s'\n",
				       errno,
				       new_script_name);
			exit(1);
		}
		/*
		 * unlink the old file
		 */
		if (unlink(script_name) != 0)
		{
			(void) fprintf(G_msg_file,
				       "error %d while unlinking '%s'\n",
				       errno,
				       script_name);
			exit(1);
		}
		/*
		 * link the new to the old
		 */
		if (link(new_script_name, script_name) != 0)
		{
			(void) fprintf(G_msg_file,
				       "error %d while linking '%s' to '%s'\n",
				       errno,
				       new_script_name,
				       script_name);
			exit(1);
		}
		/*
		 * unlink the new
		 */
		if (unlink(new_script_name) != 0)
		{
			(void) fprintf(G_msg_file,
				       "error %d while unlinking '%s'\n",
				       errno,
				       new_script_name);
			exit(1);
		}
	}
}

/**********************************************************************
 *
 *      parse_t_option
 *
 *      This routine parses the "t" option syntax.
 *
 *	The 't' is followed by an optional time delay type character and
 *	a delta factor in floating point format.
 *
 *	There are at least five different acceptable syntaxes for this option:
 *
 *	    -t123     (run together, no time delay type character)
 *	    -t+123    (run together, with time delay type character)
 *	    -t +123   (partially separate, with time delay type character)
 *	    -t 123    (completely separate, no time delay type character)
 *	    -t + 123  (completely separate, with time delay type character)
 */
static void
parse_t_option(arg_string, argv, numOptions, argvIndex_ptr)
/*
 * points to any characters left after the "-t" in that argument
 */
char		*arg_string;
/*
 * argument vector pointer
 */
register char   *argv[];
/*
 * argument count
 */
register int    numOptions;
/*
 * pointer to the current argument number
 */
int		*argvIndex_ptr;
{
	/*
	 * if the current argument has more than the "-t" in it,
	 *	then assume that the arguments are run together
	 * else assume that the arguments are separate
	 */
	if (strlen(arg_string) > 0)
	{
		/*
		 * parse the rest of the argument string
		 */
		parse_t_string(arg_string);
	}
	else
	{
		/*
		 * advance the argument index to the next argument
		 */
		*argvIndex_ptr += 1;
		if (*argvIndex_ptr > numOptions)
		{
			xtmexecute_parse_err();
			exit(1);
		}
		/*
		 * try the first character of the next argument to see if
		 * it is a time delay type character
		 */
		if (parse_time_delay_type_char(*argv[*argvIndex_ptr]))
		{
			/*
			 * There was a time delay type character.  If there
			 * is more than one character in this argument try
			 * to use the rest of the argument as the delta factor.
			 */
			if (strlen(argv[*argvIndex_ptr]) > 1)
			{
				parse_delta_factor(argv[*argvIndex_ptr] + 1);
			}
			else
			{
				/*
				 * advance the argument index to the next
				 * argument and use it as the delta factor
				 */
				*argvIndex_ptr += 1;
				if (*argvIndex_ptr > numOptions)
				{
					xtmexecute_parse_err();
					exit(1);
				}
				parse_delta_factor(argv[*argvIndex_ptr]);
			}
		}
		else
		{
			/*
			 * The first character was not a time delay type
			 * character.  Try to use the argument as the delta 
			 * number.
			 */
			G_playback_info.delta = MINIMUM_INPUT_ACTION_TIME;
			parse_delta_factor(argv[*argvIndex_ptr]);
		}
	}
}

/**********************************************************************
 *
 *      parse_t_string
 *
 *	Parse a string of characters into an optional time delay type
 *	character followed by a delta factor.
 */
static void
parse_t_string(arg_string)
/*
 * points to the string to parse
 */
char	*arg_string;
{
	/*
	 * advance to the first non-space character
	 */
	while (isspace(*arg_string))
	{
		arg_string++;
	}
	/*
	 * try the first non-space character to see if it is a time delay
	 * type character
	 */
	if (parse_time_delay_type_char(*arg_string))
	{
		/*
		 * The character was a time delay type character.
		 * Advance past the character and try to use the rest of
		 * the string as the delta factor.
		 */
		arg_string++;
		parse_delta_factor(arg_string);
	}
	else
	{
		/*
		 * The character was not a time delay type character.
		 * Try to use the string as the delta factor.
		 */
		G_playback_info.delta = MINIMUM_INPUT_ACTION_TIME;
		parse_delta_factor(arg_string);
	}
}

/**********************************************************************
 *
 *      parse_time_delay_type_char
 *
 *      This routine processes the time delay type character.  If this
 *	routine recognizes the character, it sets the time delay (delta)
 *	type and returns non-zero.  If this routine does not recognize
 *	the character, it returns zero.
 */
static int
parse_time_delay_type_char(time_delay_type_char)
/*
 * string with timing factor
 */
char    time_delay_type_char;
{
        switch (time_delay_type_char)
        {
        case '+':
                /*
                 * add to each time delay
                 */
                G_playback_info.delta = ADD_TO_INPUT_ACTION_TIMES;
                break;

        case '-':
                /*
                 * subtract from each time delay
                 */
                G_playback_info.delta = SUBTRACT_FROM_INPUT_ACTION_TIMES;
                break;

        case '*':
        case 'x':
        case 'X':
                /*
                 * multiply each time delay
                 */
                G_playback_info.delta = MULTIPLY_INPUT_ACTION_TIMES;
                break;

        case '/':
                /*
                 * divide each time delay
                 */
                G_playback_info.delta = DIVIDE_INPUT_ACTION_TIMES;
                break;

        default:
		return(0);
        }
	return(1);
}

/**********************************************************************
 *
 *      parse_delta_factor
 *
 *	Parse a string of characters into a (integer or floating point)
 *	delta factor.
 */
static void
parse_delta_factor(delta_string)
/*
 * points to string to convert to a delta factor
 */
char	*delta_string;
{
	switch(G_playback_info.delta)
	{
	case ADD_TO_INPUT_ACTION_TIMES:
	case SUBTRACT_FROM_INPUT_ACTION_TIMES:
	case MINIMUM_INPUT_ACTION_TIME:
		/*
		 * parse an integer
		 */
		if (sscanf(delta_string,
			   "%ld",
			   &G_playback_info.factor) != 1)
		{
			(void) fprintf(G_msg_file,
				       "error while trying to parse '%s' as an integer time delay factor\n",
				       delta_string);
			exit(1);
		}
		break;
	case MULTIPLY_INPUT_ACTION_TIMES:
		/*
		 * parse a (double) floating point number
		 */
		if (sscanf(delta_string,
			   "%lf",
			   &G_playback_info.double_factor) != 1)
		{
			(void) fprintf(G_msg_file,
				       "error while trying to parse '%s' as a double time delay factor\n",
				       delta_string);
			exit(1);
		}
		break;
	case DIVIDE_INPUT_ACTION_TIMES:
		/*
		 * parse a (double) floating point number
		 */
		if (sscanf(delta_string,
			   "%lf",
			   &G_playback_info.double_factor) != 1)
		{
			(void) fprintf(G_msg_file,
				       "error while trying to parse '%s' as a double time delay factor\n",
				       delta_string);
			exit(1);
		}
		if (G_playback_info.double_factor == 0.0)
		{
			(void) fprintf(G_msg_file,
				       "can't divide time delay by 0\n");
			exit(1);
		}
		break;
	default:
		(void) fprintf(G_msg_file,
			       "invalid time delay delta value\n");
		exit(1);
		break;
	}
}

/**********************************************************************
 *
 *      xtmexecute_global_initialize
 *
 *      Initialization routine for global variables.
 */
static void
xtmexecute_global_initialize(argv)
/*
 * argument vector pointer
 */
char *argv[];
{
        /*
         * points to the string returned by a call to XGetDefault
         */
        char	*default_string;
	/*
	 * this holds the screen buffer size for error checking
	 */
	int	temp_scrnbuf;

        G_msg_file  = stderr;

	/*
	 * open the x server display
	 */
        if ((G_disp_ptr = XOpenDisplay((char *) 0)) == (Display *) 0)
        {
                (void) fprintf(G_msg_file,
                               "%s: Could not open Display %s\n",
                               argv[0],
                               XDisplayName((char *) 0));
                exit(1);
        }
	/*
	 * set up the match data buffers
	 */
        if ((default_string = XGetDefault(G_disp_ptr,
					  argv[0],
					  "MatchBufferSize"))
	    != (char *) 0)
        {
		temp_scrnbuf = atoi(default_string);
		/*
		 * the buffer size must be divisible by 8
		 */
		if ((temp_scrnbuf % 8) != 0)
		{
			(void) fprintf(G_msg_file,
				       "%s: The match buffer size must be divisible by 8\n",
				       argv[0]);
			exit(1);
		}
		/*
		 * twice the buffer size must fit in an unsigned int
		 */
		if (temp_scrnbuf > ((unsigned int) UNSIGNED_INT_MAX / 2))
		{
			(void) fprintf(G_msg_file,
				       "%s: The match buffer size may not be larger than %d\n",
				       argv[0],
				       ((unsigned int) UNSIGNED_INT_MAX / 2));
			exit(1);
		}
		/*
		 * the value has passed the tests, so use it
		 */
		G_sb_size = temp_scrnbuf;
	}
        else
        {
		G_sb_size = DEFAULT_SCREEN_BUFFER_SIZE;
        }
        if ((G_c_data_buffer = (unsigned char *) malloc(G_sb_size * 2))
	    == (unsigned char *) 0)
        {
                (void) fprintf(G_msg_file,
			       "Unable to malloc space for match buffer.\n");
                exit(1);
        }
        if ((G_sb = (unsigned char *) malloc(G_sb_size)) == (unsigned char *) 0)
        {
                (void) fprintf(G_msg_file,
			       "Unable to malloc space for match buffer.\n");
                exit(1);
        }
	/*
	 * get the retry interval value, if any, from the .Xdefaults file
	 */
        if ((default_string = XGetDefault(G_disp_ptr,
					  argv[0],
					  "RetryInterval"))
	    != (char *) 0)
        {
		G_playback_info.interval = atol(default_string);
        }
        else
        {
		G_playback_info.interval = -1;
        }
	/*
	 * get the retry count value, if any, from the .Xdefaults file
	 */
        if ((default_string = XGetDefault(G_disp_ptr,
					  argv[0],
					  "MatchRetries"))
	    != (char *) 0)
        {
		G_playback_info.retries = atol(default_string);
        }
        else
        {
		G_playback_info.retries = -1;
        }
	/*
	 * get the action timing information, if any, from the .Xdefaults file
	 */
        if ((default_string = XGetDefault(G_disp_ptr,
					  argv[0],
					  "ActionTiming"))
	    != (char *) 0)
        {
                parse_t_string(default_string);
        }
        else
        {
		G_playback_info.delta = DEFAULT_DELTA;
        }
	/*
	 * get the read_from_root flag setting, if any, from the .Xdefaults file
	 */
        if ((default_string = XGetDefault(G_disp_ptr,
					  argv[0],
					  "ReadFromRoot"))
	    != (char *) 0)
        {
		if (strcmp(default_string, "yes") == 0)
		{
			G_read_from_root_flag = 1;
		}
		else
		{
			G_read_from_root_flag = 0;
		}
        }
        else
        {
		G_read_from_root_flag = 0;
        }
	/*
	 * get the compare_color_map flag setting, if any,
	 * from the .Xdefaults file
	 */
        if ((default_string = XGetDefault(G_disp_ptr,
					  argv[0],
					  "CompareColorMap"))
	    != (char *) 0)
        {
		if (strcmp(default_string, "yes") == 0)
		{
			G_compare_color_map_flag = 1;
		}
		else
		{
			G_compare_color_map_flag = 0;
		}
        }
        else
        {
		G_compare_color_map_flag = 1;
        }
	/*
	 * initialize flags
	 */
	G_update_flag = 0;
	G_continue_flag = 0;
	G_debug_flag = 0;
	G_Directory_ex_flag = 0;
}

/**********************************************************************
 *
 *      xtmexecute_parse_err
 *
 *      Output the revision date and usage info.
 */
static void
xtmexecute_parse_err()
{
	/*
	 * revision date
	 */
	(void) printf("\nusage:  xtmexecute [-options] filename\n");
	(void) printf("        The filename must end in \"%s\" unless the -d option is used.\n",
		      SCRIPT_ENDING);
	(void) printf("options:\n");
	(void) printf("        d - execute all test scripts in filename (it must be a directory)\n");
	(void) printf("        c - continue after match errors\n");
	(void) printf("        u - update mode\n");
	(void) printf("        t [+|-|*|/] number - modify playback timing\n");
	/*
	 * skip over the first 4 characters of the version string
	 */
	(void) printf("\n%s\n", &(version[4]));
}

/**********************************************************************
 *
 *      xtmexecute_parse_options1
 *
 *      parses the command line parameters
 */
static void
xtmexecute_parse_options1(numOptions, argv)
/*
 * argument count
 */
register int    numOptions;
/*
 * argument vector pointer
 */
register char   *argv[];
{
	int	argvIndex;
	char	*s;

	/*
	 * for each argument except for the file name, check and set the mode
	 */
	for (argvIndex = 1; argvIndex <= numOptions; argvIndex++)
	{
		s = argv[argvIndex];
		/*
		 * options must begin with a <->
		 */
		if (*s != '-')
		{
			xtmexecute_parse_err();
			exit(1);
		}
		/*
		 * for each character following a <-> set the appropiate mode
		 */
		s = argv[argvIndex] + 1;
		while (*s != '\0')
		{
			switch(*s)
			{
			case 'c':
				/*
				 * Continue Mode after match err
				 */
				G_continue_flag = 1;
				s++;
				break;

			case 'd':
				/*
				 * execute all test scripts in directory
				 */
				G_Directory_ex_flag = 1;
				s++;
				break;

			case 'D':
				/*
				 * Non-Documented Debug Mode
				 */
				G_debug_flag = 1;
				s++;
				break;

			case 't':
				/*
				 * Set timing interval and flag
				 */
				s++;
				parse_t_option(s, argv, numOptions, &argvIndex);
				/*
				 * skip past any characters following the <->
				 * because they have been parsed
				 */
				while (*s != '\0')
				{
					s++;
				}
				break;

			case 'u':
				/*
				 * Update mode
				 */
				G_update_flag = 1;
				s++;
				break;

			default:
				/*
				 * Bad command line given by user
				 */
				xtmexecute_parse_err();
				exit(1);
			}
		}
	}
	/*
	 * check for incompatable modes, error out if found
	 */
	if (G_continue_flag && G_update_flag)
	{
		xtmexecute_parse_err();
		(void) printf("\nNOTE: <c> and <u> options are incompatible\n");
		exit(1);
	}	
}

/**********************************************************************
 *
 *      xtmexecute_parse_options
 *
 *      Handle the command line parameters
 */
static void
xtmexecute_parse_options(argc, argv, script_name)
/*
 * argument count
 */
register int    argc;
/*
 * argument vector pointer
 */
register char   *argv[];
/*
 * RETURN parameter for actual test script file name
 */
char            *script_name;
{
        char		temp[MAXPATH];
        int		namePos;
	struct stat	stat_buf;

        (void) strcpy(temp, "");
	/*
	 * High level error check on number of args given
	 */
        switch (argc)
	{
	case 1:
		/*
		 * No parameters given - so give help
		 */
		xtmexecute_parse_err();
		exit(1);
		break;

	case 2:
		/*
		 * file name only given
		 */
		namePos = 1;
		break;

	/*
	 * Any of these indicate that parsing is to be done
	 */
	case 3:
	case 4:
	case 5:
	case 6:
		xtmexecute_parse_options1(argc - 2, argv);
		namePos = argc - 1;
		break;

	default:
		/*
		 * too many parameters, so give help
		 */
		xtmexecute_parse_err();
		exit(1);
		break;
	}
	if (G_Directory_ex_flag)
	{
		/*
		 * sanity check for directory name and assign it to temp
		 */
		if (stat(argv[namePos], &stat_buf) == -1)
		{
			/*
			 * couldn't stat name
			 */
			xtmexecute_parse_err();
			(void) printf("could not access \"%s\"\n",
				      argv[namePos]);
			exit(1);
		}
		else
		{
			if ((stat_buf.st_mode & S_IFMT) != S_IFDIR)
			{
				/*
				 * file is not a directory
				 */
				xtmexecute_parse_err();
				(void) printf("\nNOTE: \"%s\" is not a directory\n",
					      argv[namePos]);
				exit(1);
			}
			else
			{
				/*
				 * file is a directory
				 */
				(void) strcpy(temp, argv[namePos]);
			}
		}
	}
	else
	{
		if ((strlen(argv[namePos]) <= ENDING_LENGTH) ||
		    (strcmp(&(argv[namePos][strlen(argv[namePos]) - ENDING_LENGTH]),
			    SCRIPT_ENDING) != 0))
		{
			/*
			 * String length indicates that SCRIPT_ENDING must be
			 * added or SCRIPT_ENDING is not the last characters
			 */
			xtmexecute_parse_err();
			(void) printf("\nNOTE: '%s' filename extension is required\n",
				      SCRIPT_ENDING);
			exit(1);
		}
		else
		{
			/*
			 * String must already end in SCRIPT_ENDING
			 */
			(void) strcpy(temp, argv[namePos]);
		}
	}
	(void) strcpy(script_name, temp);
}

/**********************************************************************
 *
 *      make_files
 *
 *      File creation routine for 
 *        -- message file
 *        -- log file
 *        -- fail file
 *	  -- new test script file (in update mode)
 */
static void
make_files(script_fd_ptr,
	   script_name,
	   fail_file_name,
	   log_file_name,
	   new_script_fd_ptr,
	   new_script_name)
/*
 * pointer to the file descriptor for the test script
 */
int	*script_fd_ptr;
/*
 * pointer to name of test script file
 */
char	*script_name;
/*
 * pointer to name of failure file
 */
char	*fail_file_name;
/*
 * pointer to name of log file
 */
char	*log_file_name;
/*
 * pointer to the file descriptor for the new test script
 * (used only in update mode)
 */
int	*new_script_fd_ptr;
/*
 * pointer to the file name for the new test script
 * (used only in update mode)
 */
char	*new_script_name;
{
	/*
	 * string holding the test script name
	 */
	char	temp[MAXPATH];

	/*
	 * open the test script file
	 */
	if ((*script_fd_ptr = open(script_name, O_RDONLY)) < 0)
	{
		(void) fprintf(G_msg_file,
			       "Cannot open test script file '%s'.\n",
			       script_name);
		exit(1);
	}
	/*
	 * copy into temp the filename, less the SCRIPT_ENDING extension
	 */
	(void) strncpy(temp, script_name, strlen(script_name) - ENDING_LENGTH);
	/*
	 * make sure a null byte follows the base name
	 */
	temp[strlen(script_name) - ENDING_LENGTH] = '\0';
	/*
	 * build the log file name
	 */
	(void) strcpy(log_file_name, temp);
	(void) strcat(log_file_name, LOG_FILE_ENDING);
	/*
	 * open the log file
	 */
	G_log_file = makefp(log_file_name);  
	/*
	 * build the failure file name
	 */
	(void) strcpy(fail_file_name, temp);
	(void) strcat(fail_file_name, FAILURE_FILE_ENDING);
	/*
	 * open the failure file
	 */
	G_fail_file_fd = makef(fail_file_name);
	/*
	 * if in update mode, create the new test script file
	 */
	if (G_update_flag)
	{
		/*
		 * build the new test script file name
		 */
		(void) strcpy(new_script_name, temp);
		(void) strcat(new_script_name, NEW_ENDING);
		/*
		 * open the new test script file
		 */
		*new_script_fd_ptr = makef(new_script_name);
	}
}

/**********************************************************************
 *
 *      ex_file_filter
 *
 *
 */
static int
ex_file_filter(file_name, file_stat_ptr, file_type)
char		*file_name;
/*
 * ignored
 */
struct stat	*file_stat_ptr;
int		file_type;
{
	/*
	 * holds the test script name
	 */
	char	script_name[MAXPATH];
	/*
	 * holds the error flag returned from ex_one_script
	 */
	int	error_flag;

        if (G_debug_flag)
        {
		(void) fprintf(G_msg_file, "looking at %s\n", file_name);
	}
	/*
	 * if file_name is a file, then check it further
	 */
	if (file_type == FTW_F)
	{
		/*
		 * if file name ends in SCRIPT_ENDING, then execute it
		 */
                if ((strlen(file_name) > ENDING_LENGTH) &&
                    (strcmp(&(file_name[strlen(file_name) - ENDING_LENGTH]),
			    SCRIPT_ENDING) == 0))
                {
			/*
			 * set up the test script name
			 */
			(void) strcpy(script_name, file_name);
			/*
			 * call ex_one_script to execute the test script
			 */
			error_flag = ex_one_script(script_name);
			if (G_continue_flag)
			{
				/*
				 * if the "ignore errors" option was specified,
				 * return 0 to indicate to ftw
				 * to go on to the next file
				 */
				return(0);
			}
			else
			{
				/*
				 * return the error_flag
				 */
				return(error_flag);
			}
                }
	}
	/*
	 * if you get here, then the file is not of interest;
	 * return 0 to indicate to ftw to go on to the next file
	 */
	return(0);
}

/**********************************************************************
 *
 *      ex_all_scripts
 *
 *      This function executes all test scripts in the named directory
 *      tree; the path MUST be explicitly named in the parameters.
 *
 */
static void
ex_all_scripts(path_name)
/*
 * pointer to name of test script directory
 */
char	*path_name;
{
	int	error_flag;

	 /*
	  * walk the file tree rooted at path_name,
	  * calling ex_file_filter for each file in each directory
	  */
	 if ((error_flag = ftw(path_name, ex_file_filter, 1)) == -1)
	 {
		(void) fprintf(G_msg_file,
			       "fatal error %d in ftw, aborting\n",
			       errno);
		exit(1);
	 }
}

/**********************************************************************
 *
 *      ex_one_script
 *
 *      This function executes ONE test script.
 *
 *
 */
static int
ex_one_script(script_name)
/*
 * pointer to name of test script file
 */
char	*script_name;
{
	/*
	 * flag returned from execute
	 */
	int	ret;
	/*
	 * holds the file descriptor for the test script
	 */
	int	script_fd;
	/*
	 * holds the failure file name
	 */
	char	fail_file_name[MAXPATH];
	/*
	 * holds the log file name
	 */
	char	log_file_name[MAXPATH];
	/*
	 * holds the file descriptor for the new test script
	 * (used only in update mode)
	 */
	int	new_script_fd;
	/*
	 * holds the file name for the new test script
	 * (used only in update mode)
	 */
	char	new_script_name[MAXPATH];

	/*
	 * set the checksum flag to off (don't use checksums)
	 */
	G_checksum_flag = 0;
	/*
	 * Used for debug option only
	 */
	G_all_events = 0;
	/*
	 * Used for debug option only
	 */
	G_all_pkt = 0;
	/*
	 * Open the needed files and get the test script file descriptor
	 */
	make_files(&script_fd,
		   script_name,
		   fail_file_name,
		   log_file_name,
		   &new_script_fd,
		   new_script_name);
	/*
	 * Print some details if debugging is on
	 */
        if (G_debug_flag)
        {
                (void) fprintf(G_msg_file,
			       "test script is %s\n",
			       script_name);
                (void) fprintf(G_msg_file,
			       "retries is %ld\n",
			       G_playback_info.retries);
                (void) fprintf(G_msg_file,
			       "interval is %ld\n",
			       G_playback_info.interval);
        }
	/*
	 * execute the test script
	 */
        ret = execute(script_fd, script_name, new_script_fd);
	/*
	 * log the end of the test script
	 */
	endmsg(G_log_file, ret, script_name);
        endmsg(G_msg_file, ret, script_name);
	/*
	 * close test script, log, and failure files
	 */
	xtmexecute_close_files(script_fd,
			       script_name,
			       fail_file_name,
			       log_file_name,
			       new_script_fd,
			       new_script_name);
	/*
	 * If no error occurred delete the fail file
	 */
	if (ret == 0)
	{
		/*
		 * If the fail file exists after a test script is executed,
		 * it means that an error occurred.
		 */
		if (unlink(fail_file_name) != 0)
		{
			(void) fprintf(G_msg_file,
				       "error %d while unlinking '%s'\n",
				       errno,
				       fail_file_name);
		}
	}
	/*
	 * return the execute error status
	 */
	return(ret);
}

/**********************************************************************
 *
 *      main
 *
 *
 */
int
main(argc, argv)
/*
 * argument count
 */
register int    argc;
/*
 * argument vector pointer
 */
register char   *argv[];
{
	/*
	 * holds the test script name
	 */
	char	script_or_dir_name[MAXPATH];

	/*
	 * Initialize the zillion global variables
	 */
        xtmexecute_global_initialize(argv);
	/*
	 * Set the error handler
	 */
        XSetErrorHandler(xerror);
	/*
	 * Check to see if the extension exists
	 */
        if (XTestQueryInputSize(G_disp_ptr, &G_max_input_actions) != 0)
        {
                (void) fprintf(G_msg_file,
			       "Unable to query input action size.\n");
                (void) fprintf(G_msg_file,
			       "Does the server have the %s extension installed?\n",
                               XTestEXTENSION_NAME);
                exit(1);
        }
	/*
	 * parse the command line and set global flags
	 */
        xtmexecute_parse_options(argc, argv, script_or_dir_name);

	if (G_Directory_ex_flag)
	{
		ex_all_scripts(script_or_dir_name);
	}
	else
	{
        	(void) ex_one_script(script_or_dir_name);
	}
        return(0);
}
