/**********************************************************************
 *
 *      xtmconvert.c
 *
 **********************************************************************/

/*

Copyright 1986, 1987, 1988, 1989 by Hewlett-Packard Corporation
Copyright 1986, 1987, 1988, 1989 by the Massachusetts Institute of Technology

Permission to use, copy, modify, and distribute this
software and its documentation for any purpose and without
fee is hereby granted, provided that the above copyright
notice appear in all copies and that both that copyright
notice and this permission notice appear in supporting
documentation, and that the name of M.I.T. not be used in
advertising or publicity pertaining to distribution of the
software without specific, written prior permission.

Hewlett-Packard and M.I.T. make no representations about the 
suitability of this software for any purpose.  It is provided 
"as is" without express or implied warranty.

This software is not subject to any license of the American
Telephone and Telegraph Company or of the Regents of the
University of California.

*/

/**********************************************************************
 *      include files
 **********************************************************************/

/*
 * standard io defs
 */
#include  <stdio.h>
/*
 * standard file control defs
 */
#include  <fcntl.h>
/*
 * standard string defs
 */
#include  <string.h>
/*
 * X library defines
 */
#include  <X11/Xlib.h>
/*
 * more X library defines
 */
#include  <X11/Xmd.h>
/*
 * client exerciser defines
 */
#include  <xtm.h>

/***********************************************************************
 *	variables
 **********************************************************************/

/*
 * Buffer for compressing/uncompressing match image data
 */
unsigned char	*G_c_data_buffer;
/*
 * Buffer for match image data
 */
unsigned char	*G_sb;
/*
 * size of G_sb
 */
unsigned int	G_sb_size;
/*
 * non-zero if the debug option is on
 */
int		G_debug_flag;
/*
 * the file to write error messages to
 */
FILE		*G_msg_file;
/*
 * if non-zero then there has just been a newline (used 
 * when writing out single-character keysyms in summary format)
 */
int		G_newline_flag;
/*
 * holds the translation type
 */
int		G_translate_type;

/***********************************************************************
 *	external variables
 **********************************************************************/

extern int	errno;
extern char	version[];

/**********************************************************************
 *      function declarations
 **********************************************************************/

void		script_to_text();
void		open_files();
void		xtmconvert_close_files();
void		xtmconvert_parse_err();
void		parse_command_line();
char		*malloc();

/**********************************************************************
 *      functions
 **********************************************************************/

/***********************************************************************
 *
 *      main
 */
main(argc, argv)
/*
 * Number of command line arguments
 */
int	argc;
/*
 * Strings containing command line parameters
 */
char	*argv[];
{
	/*
	 * file descriptor of the "from" file
	 */
	int	from_fd;
	/*
	 * file descriptor of the "to" file
	 */
	int	to_fd;
	/*
	 * file pointer of the "from" file
	 */
	FILE	*from_fptr;
	/*
	 * file pointer of the "to" file
	 */
	FILE	*to_fptr;
	/*
	 * the name of the "from" file
	 */
	char   from_name[MAXPATH];
	/*
	 * the name of the "to" file
	 */
	char   to_name[MAXPATH];
	/*
	 * the base name of the "to" file
	 */
	char   to_basename[MAXPATH];
	/*
	 * if non-zero then any image data is stripped during translation
	 */
	int		stripflag;

	/*
	 * set the global error message file descriptor to stderr
	 */
	G_msg_file = stderr;

	/*
	 * set up some buffers for match image data
	 */
	G_sb_size = DEFAULT_SCREEN_BUFFER_SIZE;
        if ((G_c_data_buffer = (unsigned char *) malloc(G_sb_size * 2))
	    == (unsigned char *) 0)
        {
                (void) fprintf(G_msg_file,
			       "Unable to malloc space for match data buffer.\n");
                exit(1);
        }
        if ((G_sb = (unsigned char *) malloc(G_sb_size)) == (unsigned char *) 0)
        {
                (void) fprintf(G_msg_file,
			       "Unable to malloc space for match data buffer.\n");
                exit(1);
        }
	/*
	 * Parse the command line parameters
	 */
	parse_command_line(argc,
			   argv,
			   from_name,
			   to_name,
			   to_basename,
			   &stripflag);

	if (G_debug_flag)
	{
        	(void) fprintf(G_msg_file,
			       "From file: %s  To file: %s\n",
		               from_name,
		               to_name);
	}
	/*
	 * Open the files to be translated from and translated to
	 */
	open_files(&from_fd, &to_fd, &from_fptr, &to_fptr, from_name, to_name);
	/*
	 * Initialize the newline flag to true
	 */
	G_newline_flag = 1;
	/*
	 * Start the appropriate translation type
	 */
	switch (G_translate_type)
	{
	case SCRIPT_TO_OUTPUT:
		if (G_debug_flag)
		{
			(void) fprintf(G_msg_file,
				       "Translating from internal to output format\n");
		}
		script_to_text(from_fd,
			       to_fptr,
			       to_basename,
			       stripflag);
		break;

	case SCRIPT_TO_SUMMARY:
		if (G_debug_flag)
		{
			(void) fprintf(G_msg_file,
				       "Translating from internal to summary format\n");
		}
		script_to_text(from_fd,
			       to_fptr,
			       to_basename,
			       stripflag);
		break;

	case SUMMARY_TO_SCRIPT:
		if (G_debug_flag)
		{
			(void) fprintf(G_msg_file,
				       "Translating from summary to internal format\n");
		}
		summary_to_script(from_fptr, to_fd, stripflag, argv[0]);
		break;

	case OUTPUT_TO_SCRIPT:
		if (G_debug_flag)
		{
			(void) fprintf(G_msg_file,
				       "Translating from output to internal format\n");
		}
		output_to_script(from_fptr, to_fd, stripflag);
		break;

	default:
		/*
		 * It should never get here, for defensive programming..
		 */
		xtmconvert_parse_err();
		exit(1);
	}
	/*
	 * Close the <from> and <to> file and exit
	 */
	xtmconvert_close_files(from_fd,
			       to_fd,
			       from_fptr,
			       to_fptr,
			       from_name,
			       to_name);
	/*
	 * if we get here, then no errors occurred
	 */
	return(0);
}

/***********************************************************************
 *
 *      open_files
 */
static void
open_files(from_fd_ptr, to_fd_ptr, from_fptr_ptr, to_fptr_ptr, from_name, to_name)
/*
 * points to the file descriptor of the "from" file
 */
int	*from_fd_ptr;
/*
 * points to the file descriptor of the "to" file
 */
int	*to_fd_ptr;
/*
 * points to the file pointer of the "from" file
 */
FILE	**from_fptr_ptr;
/*
 * points to the file pointer of the "to" file
 */
FILE	**to_fptr_ptr;
/*
 * points to the the name of the "from" file
 */
char   *from_name;
/*
 * points to the the name of the "to" file
 */
char   *to_name;
{
	if (G_debug_flag)
	{
		(void) fprintf(G_msg_file,
			       "In open files, from file: %s, to file: %s\n",
		               from_name,
		               to_name);
	}
	/*
	 * open FROM file in read only mode
	 */
	if ((G_translate_type == SCRIPT_TO_OUTPUT) ||
	    (G_translate_type == SCRIPT_TO_SUMMARY))
	{

		*from_fd_ptr = open(from_name, O_RDONLY);
		if (*from_fd_ptr <= 2)
		{
			(void) fprintf(G_msg_file,
				       "Error in opening file: %s\n",
				       from_name);
			exit(1);
		}
		/*
		 * create to_fptr giving read/write permissions
		 */
		*to_fptr_ptr = fopen(to_name, "w");
		if (*to_fptr_ptr == (FILE *) 0)
		{
			(void) fprintf(G_msg_file,
				       "Error in creating file: %s\n",
				       to_name);
			exit(1);
		}
	}
	else
	{
		*to_fd_ptr = makef(to_name);
		/*
		 * create from_fptr giving read/write permissions
		 */
		*from_fptr_ptr = fopen(from_name, "r");
		if (*from_fptr_ptr == (FILE *) 0)
		{
			(void) fprintf(G_msg_file,
				       "Error in opening file: %s\n",
				       from_name);
			exit(1);
		}
	}
}

/***********************************************************************
 *
 *      xtmconvert_close_files
 */
static void
xtmconvert_close_files(from_fd, to_fd, from_fptr, to_fptr, from_name, to_name)
/*
 * file descriptor of the "from" file
 */
int	from_fd;
/*
 * file descriptor of the "to" file
 */
int	to_fd;
/*
 * file pointer of the "from" file
 */
FILE	*from_fptr;
/*
 * file pointer of the "to" file
 */
FILE	*to_fptr;
/*
 * the name of the "from" file
 */
char   *from_name;
/*
 * the name of the "to" file
 */
char   *to_name;
{
	if (G_debug_flag)
	{
		(void) fprintf(G_msg_file,
			       "In close files, from file: %s, to file: %s\n",
		               from_name,
		               to_name);
	}
	/*
	 * close the input (FROM) file
	 */
	if ((G_translate_type == SCRIPT_TO_OUTPUT) ||
	    (G_translate_type == SCRIPT_TO_SUMMARY))
	{
		if (from_fd > 2)
		{
			if (close(from_fd) == -1)
			{
				(void) fprintf(G_msg_file,
					       "error %d while closing \"from\" file\n",
					       errno);
			}
		}
		/*
		 * close the output (TO) file
		 */
		if (to_fptr != (FILE *) 0)
		{
			if (fclose(to_fptr) != 0)
			{
				(void) fprintf(G_msg_file,
					       "error %d while closing \"to\" file\n",
					       errno);
			}
		}
	}
	else
	{
		/*
		 * close the output (TO) file
		 */
		if (to_fd > 2)
		{
			if (close(to_fd) == -1)
			{
				(void) fprintf(G_msg_file,
					       "error %d while closing \"to\" file\n",
					       errno);
			}
			close(to_fd);
		}
		/*
		 * close the input (FROM) file
		 */
		if (from_fptr != (FILE *) 0)
		{
			if (fclose(from_fptr) != 0)
			{
				(void) fprintf(G_msg_file,
					       "error %d while closing \"from\" file\n",
					       errno);
			}
		}
	}
}

/***********************************************************************
 *
 *      xtmconvert_parse_err
 */
static void
xtmconvert_parse_err()
{
	/*
	 * print revision code and date
	 */
	(void) fprintf(G_msg_file, "\n");
	(void) fprintf(G_msg_file,
		       "usage:  xtmconvert [-s] <fromfile> <tofile>\n");
	(void) fprintf(G_msg_file, "options:\n");
	(void) fprintf(G_msg_file,
		       "        s - strip match data from <tofile>\n");
	(void) fprintf(G_msg_file,
		       "NOTE: file name extensions determine operation mode\n");
	/*
	 * skip over the first 4 characters of the version string
	 */
	(void) fprintf(G_msg_file, "\n%s\n", &(version[4]));
}

/***********************************************************************
 *
 *      xtmconvert_parse_options1
 */
static void
xtmconvert_parse_options1(numOptions, argv, stripflag_ptr)
/*
 * argument count
 */
int	numOptions;
/*
 * argument vector pointer
 */
char	*argv[];
/*
 * Points to strip flag.  If non-zero, then any image data is stripped
 * during the translation.
 */
int	*stripflag_ptr;
{
	int	argvIndex;
	char	*s;

	/*
	 * for each argument except for the 
	 * file names, check and set the mode
	 */
	for (argvIndex = 1; argvIndex <= numOptions; argvIndex++)
	{
		s = argv[argvIndex];
		/*
		 * options must begin with a <->
		 */
		if (*s != '-')
		{
			xtmconvert_parse_err();
			exit(1);
		}
		/*
		 * for each character following a <-> set the appropriate mode
		 */
		for (s = argv[argvIndex] + 1; *s != '\0'; s++)
		{
			switch(*s)
			{
			case 's':
				/*
				 * Strip match data flag
				 */
				*stripflag_ptr = 1;
				break;

			case 'D':
				/*
				 * Non-Documented Debug Mode
				 */
				G_debug_flag = 1;
				(void) fprintf(G_msg_file,
					       "Debugging turned on\n");
				break;

			default:
				/*
				 * Bad command line given by user
				 */
				xtmconvert_parse_err();
				exit(1);
			}
		}
	}
}

/***********************************************************************
 *
 *      parse_command_line
 */
static void
parse_command_line(argc, argv, from_name, to_name, to_basename, stripflag_ptr)
/*
 * argument count
 */
int    argc;
/*
 * argument vector pointer
 */
char   *argv[];
/*
 * points to where to put the name of the "from" file
 */
char   *from_name;
/*
 * points to where to put the name of the "to" file
 */
char   *to_name;
/*
 * points to where to put the name (without any extension) of the "to" file
 */
char   *to_basename;
/*
 * Points to strip flag.  If non-zero, then any image data is stripped
 * during the translation.
 */
int	*stripflag_ptr;
{
        int     fromPos,  toPos;
	char	*fromext, *toext;

	*stripflag_ptr = 0;
	/*
	 * High level error check on number of args given
	 */
        switch (argc)
	{
	case 1:
	case 2:
		/*
		 * Not enough parameters given - so give help
		 */
		xtmconvert_parse_err();
		exit(1);
		break;

	case 3:
		/*
		 * assume file names only given
		 */
		fromPos = 1;
		toPos   = 2;
		break;

	case 4:
	case 5:
		xtmconvert_parse_options1((argc - 3), argv, stripflag_ptr);
		fromPos = argc - 2;
		toPos   = argc - 1;
		break;

	default:
		/*
		 * too many parameters, so give help
		 */
		xtmconvert_parse_err();
		exit(1);
		break;
	}
	/*
	 * Determine what type of translation is to be done by looking at
	 * the file name extensions
	 */
	if ((strlen(argv[fromPos]) <= ENDING_LENGTH) || 
            (strlen(argv[toPos]) <= ENDING_LENGTH))
        {
		xtmconvert_parse_err();
                (void) fprintf(G_msg_file,
			       "\nNOTE: filename extensions are required\n");
                exit(1);
        }
        else
	{
		fromext = &(argv[fromPos][strlen(argv[fromPos]) -
					  ENDING_LENGTH]);
		toext   = &(argv[toPos][strlen(argv[toPos]) - ENDING_LENGTH]);
	}
	if (G_debug_flag)
	{
		(void) fprintf(G_msg_file,
			       "From extension: %s, To extension: %s\n",
			       fromext,
			       toext);
	}
	/*
	 * Determine operation type and assign flag value
	 */
	if ((strcmp(fromext, SCRIPT_ENDING) == 0) &&
	    (strcmp(toext, SUMMARY_ENDING) == 0))
	{
		G_translate_type = SCRIPT_TO_SUMMARY;
	}
	else if ((strcmp(fromext, SCRIPT_ENDING) == 0) &&
		 (strcmp(toext, OUTPUT_ENDING) == 0))
	{
                G_translate_type = SCRIPT_TO_OUTPUT;
	}
        else if ((strcmp(fromext, SUMMARY_ENDING) == 0) &&
		 (strcmp(toext, SCRIPT_ENDING) == 0))
	{
                G_translate_type = SUMMARY_TO_SCRIPT;
	}
        else if ((strcmp(fromext, OUTPUT_ENDING) == 0) &&
		 (strcmp(toext, SCRIPT_ENDING) == 0))
	{
                G_translate_type = OUTPUT_TO_SCRIPT;
	}
	else
	{
		/*
		 * Inproper file names given
		 */
		xtmconvert_parse_err();
		(void) fprintf(G_msg_file,
			       "\nError: File name extensions not valid\n");
		exit(1);
	}
	/*
	 * if we get here assign the file names
	 */
	(void) strcpy(from_name, argv[fromPos]);
	(void) strcpy(to_name, argv[toPos]);
	(void) strncpy(to_basename,
		       argv[toPos],
		       strlen(argv[toPos]) - ENDING_LENGTH);
	/*
	 * make sure that to_basename is null-terminated
	 */
	to_basename[strlen(argv[toPos]) - ENDING_LENGTH] = '\0';
}
