/***********************************************************************
 *
 *	tranin.c
 *
 **********************************************************************/

/*

Copyright 1986, 1987, 1988, 1989 by Hewlett-Packard Corporation
Copyright 1986, 1987, 1988, 1989 by the Massachusetts Institute of Technology

Permission to use, copy, modify, and distribute this
software and its documentation for any purpose and without
fee is hereby granted, provided that the above copyright
notice appear in all copies and that both that copyright
notice and this permission notice appear in supporting
documentation, and that the name of M.I.T. not be used in
advertising or publicity pertaining to distribution of the
software without specific, written prior permission.

Hewlett-Packard and M.I.T. make no representations about the 
suitability of this software for any purpose.  It is provided 
"as is" without express or implied warranty.

This software is not subject to any license of the American
Telephone and Telegraph Company or of the Regents of the
University of California.

*/

/***********************************************************************
 *	include files
 **********************************************************************/

/*
 * standard io defs
 */
#include  <stdio.h>
/*
 * standard string defs
 */
#include  <string.h>
/*
 * X library defines
 */
#include  <X11/Xlib.h>
/*
 * more X library type defs
 */
#include  <X11/Xmd.h>
/*
 * client exerciser defines
 */
#include  <xtm.h>
/*
 * input synthesis extension defines
 */
#include  <X11/extensions/xtestext1.h>

/***********************************************************************
 *	external variables
 **********************************************************************/

/*
 * errno
 */
extern int	errno;
/*
 * non-zero if debug option turned on
 */
extern int	G_debug_flag;
/*
 * the file to output error messages to
 */
extern FILE	*G_msg_file;
/*
 * the button names
 */
extern char	*G_button_names[];

/***********************************************************************
 *	variables
 **********************************************************************/

/*
 * holds input actions before they are written out to the test script
 */
unsigned char	G_ia_buffer[INPUT_ACTION_BUFFER_SIZE];
/*
 * the number of bytes in the input action buffer
 */
int		G_ia_count;
/*
 * if true then use checksums for match data
 */
int		G_checksum_flag;

/***********************************************************************
 *	function declarations
 **********************************************************************/

void	getkey();
void	chk_copy();
void	write_input_actions();
void	wrchk();
void	wrchk2();
void	outtoint();
void	sumtoint();
void	tran_comment();
void	tran_checksum_mode();
void	tran_version_number_from_output();
void	tran_version_number_from_summary();
CARD8	trankey();
Display	*XOpenDisplay();
KeyCode	XKeysumToKeycode();
char	*fgets();
char	*getkeysym();

/***********************************************************************
 * functions
 **********************************************************************/

/***********************************************************************
 *
 *	output_to_script
 *
 *	Convert a test script from output format to internal format.
 */
int
output_to_script(from_fptr, to_fd, stripflag)
/*
 * file pointer of output format test script file
 */
FILE	*from_fptr;
/*
 * file descriptor of internal format test script file
 */
int	to_fd;
/*
 * if non-zero strip any image data during the translation
 */
int	stripflag;
{
	/*
	 * used to hold the line type
	 */
	char		type[WRITE_BUFFER_SIZE];
	/*
	 * used to hold the device type value temporarily
	 */
	int		dev;
	/*
	 * used to hold the x offset/coordinate value temporarily
	 */
	int		x;
	/*
	 * used to hold the y offset/coordinate value temporarily
	 */
	int		y;
	/*
	 * used to hold the delay time value temporarily
	 */
	unsigned int	time;
	/*
	 * used to hold the keysym value temporarily
	 */
	char		key[WRITE_BUFFER_SIZE];
	/*
	 * this is read but ignored
	 */
	unsigned long	count;
	/*
	 * structure describing a key
	 */
	XTestKeyInfo	fk;
	/*
	 * describes a motion input action
	 */
	XTestMotionInfo	fm;
	/*
	 * describes a jump input action
	 */
	XTestJumpInfo	fj;
	/*
	 * describes a time input action
	 */
	XTestDelayInfo	ft;
	/*
	 * local pointer to X11 display
	 */
	Display		*display;

	/*
	 * initialize the checksum flag to false (not using checksums)
	 */
	G_checksum_flag = 0;
	/*
	 * initialize the input action count
	 */
	G_ia_count = 0;
	/*
	 * Extablish the connection to the X-Server
	 */
        if ((display = XOpenDisplay((char *) 0)) == (Display *) 0)
        {
                (void) fprintf(G_msg_file, "error on XOpenDisplay\n");
                exit(1);
        }
	if (G_debug_flag)
	{
		(void) fprintf(G_msg_file,
			       "Starting translate from output format to internal format\n");
	}
	/*
	 * translate and verify the version number
	 */
	tran_version_number_from_output(from_fptr, to_fd);
	/*
	 * read and process the rest of the test script
	 */
	while (fscanf(from_fptr, "%s", type) != EOF)
	{
                if (G_debug_flag)
		{
                        (void) fprintf(G_msg_file, "Translating: %s\n", type);
		}
		if (strcmp(type, "JUMP") == 0)
		{
			fscanf(from_fptr,
			       " DEV [%d] X [%d] Y [%d] TIME \(ms\) [%u] COUNT [%lu]",
			       &dev,
			       &x,
			       &y,
			       &time,
			       &count);
			fj.header = XTestPackDeviceID(dev) | XTestJUMP_ACTION;
			fj.jumpx = x;
			fj.jumpy = y;
			fj.delay_time = time;
			chk_copy(to_fd,
				 (unsigned char *) &fj,
				 sizeof(XTestJumpInfo));
		}
		else if (strcmp(type, "MOTION") == 0)
		{
			fscanf(from_fptr,
			       " DEV [%d] X [%d] Y [%d] TIME \(ms\) [%u] COUNT [%lu]",
			       &dev,
			       &x,
			       &y,
			       &time,
			       &count);
			fm.header = XTestPackDeviceID(dev) | XTestMOTION_ACTION;
			if (x < 0)
			{
				fm.header |= XTestX_NEGATIVE;
			}
			if (y < 0)
			{
				fm.header |= XTestY_NEGATIVE;
			}
			x = abs(x);
			y = abs(y);
			fm.delay_time = time;
			fm.motion_data = XTestPackXMotionValue(x) |
					 XTestPackYMotionValue(y);
			chk_copy(to_fd,
			 	 (unsigned char *) &fm,
			 	 sizeof(XTestMotionInfo));
		}
		else if (strcmp(type, "TIME") == 0)
		{
			fscanf(from_fptr,
			       " TIME \(ms\) [%lu] COUNT [%lu]",
			       &(ft.delay_time),
			       &count);
			ft.header = XTestPackDeviceID(XTestDELAY_DEVICE_ID);
			chk_copy(to_fd,
				 (unsigned char *) &ft,
				 sizeof(XTestDelayInfo));
		}
		else if (strcmp(type, "KEY_UP") == 0)
		{
			fscanf(from_fptr, " DEV [%d] KEY [", &dev);
			getkey(from_fptr, key, ']');
			fscanf(from_fptr,
			       " TIME \(ms\) [%u] COUNT [%lu]",
			       &time,
			       &count);
			fk.keycode = trankey(key, display);
			fk.header = XTestPackDeviceID(dev) | XTestKEY_ACTION;
			fk.header |= XTestKEY_UP;
			fk.delay_time = time;
			chk_copy(to_fd,
				 (unsigned char *) &fk,
				 sizeof(XTestKeyInfo));
		}
		else if (strcmp(type, "KEY_DOWN") == 0)
		{
			fscanf(from_fptr, " DEV [%d] KEY [", &dev);
			getkey(from_fptr, key, ']');
			fscanf(from_fptr,
			       " TIME \(ms\) [%u] COUNT [%lu]",
			       &time,
			       &count);
			fk.keycode = trankey(key, display);
			fk.header = XTestPackDeviceID(dev) | XTestKEY_ACTION;
			fk.header |= XTestKEY_DOWN;
			fk.delay_time = time;
			chk_copy(to_fd,
				 (unsigned char *) &fk,
				 sizeof(XTestKeyInfo));
		}
		else if (strcmp(type, "MATCH") == 0)
		{
			if (G_ia_count > 0)
			{
				/*
				 * write out the data
				 */
				write_input_actions(to_fd);
			}
			outtoint(from_fptr, to_fd, stripflag);
		}
		else if (strcmp(type, "COMMENT") == 0)
		{
			if (G_ia_count > 0)
			{
				/*
				 * write out the data
				 */
				write_input_actions(to_fd);
			}
			tran_comment(from_fptr, to_fd);
		}
		else if (strcmp(type, "CHECKSUMS") == 0)
		{
			if (G_ia_count > 0)
			{
				/*
				 * write out the data
				 */
				write_input_actions(to_fd);
			}
			/*
			 * read in the checksum mode
			 */
			if (fscanf(from_fptr, "%s", key) != 1)
			{
				(void) fprintf(G_msg_file,
					       "error %d reading checksum mode from test script\n",
					       errno);
				exit(1);
			}
			tran_checksum_mode(to_fd, key);
		}
		else
		{
			(void) fprintf(G_msg_file,
				       "Invalid data type = %s.\n",
				       type);
			exit(1);
		}
	}
	if (G_ia_count > 0)
	{
		/*
		 * write out the data
		 */
		write_input_actions(to_fd);
	}
	/*
	 * close the display, no longer needed
	 */
        XCloseDisplay(display);
}

/***********************************************************************
 *
 *	summary_to_script
 *
 *	Convert a test script from summary format to internal format.
 */
int
summary_to_script(from_fptr, to_fd, stripflag, program_name)
/*
 * file pointer of output format test script file
 */
FILE	*from_fptr;
/*
 * file descriptor of internal format test script file
 */
int	to_fd;
/*
 * if non-zero strip any image data during the translation
 */
int	stripflag;
/*
 * points to the name of this program
 */
char	*program_name;
{
	/*
	 * ???
	 */
	char			type[WRITE_BUFFER_SIZE];
	/*
	 * ???
	 */
	int			dev;
	/*
	 * ???
	 */
	int			x;
	/*
	 * ???
	 */
	int			y;
	/*
	 * ???
	 */
	unsigned int		time;
	/*
	 * ???
	 */
	char			key[WRITE_BUFFER_SIZE];
	/*
	 * ???
	 */
	char			cbuf[WRITE_BUFFER_SIZE];
	/*
	 * loop counter for key symbols
	 */
	int			i;
	/*
	 * structure describing a key
	 */
	XTestKeyInfo		fk;
	/*
	 * describes a motion input action
	 */
	XTestMotionInfo		fm;
	/*
	 * describes a jump input action
	 */
	XTestJumpInfo		fj;
	/*
	 * describes a time input action
	 */
	XTestDelayInfo		ft;
	/*
	 * argument point to X11 display
	 */
	Display			*display;
	/*
	 * points to the string returned from a XGetDefault call
	 */
	char			*default_string;
	/*
	 * holds the playback count and interval values
	 */
	struct playback_info	playback_info;
	/*
	 * the device number corresponding to the pointer/mouse
	 */
	int			motion_device;
	/*
	 * the device number corresponding to the keyboard
	 */
	int			keyboard_device;
	/*
	 * the interval between keys
	 */
	INT32			key_interval;
	/*
	 * the delay between a down transition and an up transition of a key
	 */
	INT32			stroke_interval;

	/*
	 * initialize the checksum flag to false (not using checksums)
	 */
	G_checksum_flag = 0;
	/*
	 * initialize the input action count
	 */
	G_ia_count = 0;
	/*
	 * Extablish the connection to the X Server
	 */
        if ((display = XOpenDisplay((char *) 0)) == (Display *) 0)
        {
                (void) fprintf(G_msg_file, "error on XOpenDisplay\n");
                exit(1);
        }
	if (G_debug_flag)
	{
		(void) fprintf(G_msg_file,
			       "Starting translate from summary format to internal format\n");
	}

	/*
	 * get the retry count from the .Xdefaults file, if there is one
	 */
        if ((default_string = XGetDefault(display,
					  program_name,
					  "MatchRetries")) != (char *) 0)
        {
                playback_info.retries = atoi(default_string);
        }
        else
        {
                playback_info.retries = DEFAULT_RETRY;
        }
	/*
	 * get the retry interval from the .Xdefaults file, if there is one
	 */
        if ((default_string = XGetDefault(display,
					  program_name,
					  "RetryInterval")) != (char *) 0)
        {
                playback_info.interval = atoi(default_string);
        }
        else
        {
                playback_info.interval = DEFAULT_INTERVAL;
        }
	/*
	 * get the motion device number from the .Xdefaults file,
	 * if there is one
	 */
        if ((default_string = XGetDefault(display,
					  program_name,
					  "PointerDevice")) != (char *) 0)
        {
                motion_device = atoi(default_string);
        }
        else
        {
                motion_device = MOTION_DEVICE;
        }
	/*
	 * get the keyboard device number from the .Xdefaults file,
	 * if there is one
	 */
        if ((default_string = XGetDefault(display,
					  program_name,
					  "KeyDevice")) != (char *) 0)
        {
                keyboard_device = atoi(default_string);
        }
        else
        {
                keyboard_device = KEYBOARD_DEVICE;
        }
	/*
	 * get the key interval from the .Xdefaults file, if there is one
	 */
        if ((default_string = XGetDefault(display,
					  program_name,
					  "KeyInterval")) != (char *) 0)
        {
                key_interval = atoi(default_string);
        }
        else
        {
                key_interval = KEY_INTERVAL;
        }
	/*
	 * get the stroke interval from the .Xdefaults file, if there is one
	 */
        if ((default_string = XGetDefault(display,
					  program_name,
					  "StrokeInterval")) != (char *) 0)
        {
                stroke_interval = atoi(default_string);
        }
        else
        {
                stroke_interval = STROKE_INTERVAL;
        }
	/*
	 * translate and verify the version number
	 */
	tran_version_number_from_summary(from_fptr, to_fd);
	/*
	 * read and process the rest of the test script
	 */
	while (fscanf(from_fptr, "%s", type) != EOF)
	{
                if (G_debug_flag)
		{
                        (void) fprintf(G_msg_file, "Translating: %s\n", type);
		}
		/*
		 * ...................JUMP INPUT ACTION.......................
		 */
		if (strcmp(type, "<JUMP") == 0)
		{
			/*
			 * Parse the rest of the line
			 */
			if (fscanf(from_fptr, " %d %d %u>", &x, &y, &time) != 3)
			{
				(void) fprintf(G_msg_file,
					       "error reading jump input action from summary format file\n");
				exit(1);
			}
			/*
			 * Assign values to input action structure
			 */
			fj.header = XTestPackDeviceID(motion_device) | 
                                    XTestJUMP_ACTION;
			fj.jumpx = x;
			fj.jumpy = y;
			fj.delay_time = time;
			if (G_debug_flag)
			{
				(void) fprintf(G_msg_file,
					       "JUMP x = %d, y = %d, time = %u\n",
				               fj.jumpx,
				               fj.jumpy,
				               fj.delay_time);
			}
			/*
			 * Write out input action structure to script file
			 */
			chk_copy(to_fd,
				 (unsigned char *) &fj,
				 sizeof(XTestJumpInfo));
		}
		/*
		 * ..................MOTION INPUT ACTION.......................
		 */
		else if (strcmp(type, "<MOTION") == 0)
		{
			/*
			 * Parse the rest of the line
			 */
			if (fscanf(from_fptr, " %d %d %u>", &x, &y, &time) != 3)
			{
				(void) fprintf(G_msg_file,
					       "error reading motion input action from summary format file\n");
				exit(1);
			}
			/*
			 * Assign values to input action structure
			 */
			fm.header = XTestPackDeviceID(motion_device) | 
                                    XTestMOTION_ACTION;
			if (x < 0)
			{
				fm.header |= XTestX_NEGATIVE;
			}
			if (y < 0)
			{
				fm.header |= XTestY_NEGATIVE;
			}
			x = abs(x);
			y = abs(y);
			fm.delay_time = time;
			fm.motion_data = XTestPackXMotionValue(x) |
					 XTestPackYMotionValue(y);
			if (G_debug_flag)
			{
				(void) fprintf(G_msg_file,
					       "MOTION x = %d, y = %d, time = %u\n",
				               x,
				               y,
				               fm.delay_time);
			}
			/*
			 * Write out input action structure to script file
			 */
			chk_copy(to_fd,
				 (unsigned char *) &fm,
				 sizeof(XTestMotionInfo));
		}
		/*
		 * ..................TIME INPUT ACTION........................
		 */
		else if (strcmp(type, "<TIME") == 0)
		{
			/*
			 * Parse the rest of the line
			 */
			if (fscanf(from_fptr, " %lu>", &(ft.delay_time)) != 1)
			{
				(void) fprintf(G_msg_file,
					       "error reading time input action from summary format file\n");
				exit(1);
			}
			/*
			 * Assign values to input action structure
			 */
			ft.header = XTestPackDeviceID(XTestDELAY_DEVICE_ID);
			if (G_debug_flag)
			{
				(void) fprintf(G_msg_file,
					       "TIME time = %lu\n",
					       ft.delay_time);
			}
			/*
			 * Write out input action structure to script file
			 */
			chk_copy(to_fd,
				 (unsigned char *) &ft,
				 sizeof(XTestDelayInfo));
		}
		/*
		 * ................KEYRELEASE INPUT ACTION.....................
		 */
		else if (strcmp(type, "<KEY_UP") == 0)
		{
			/*
			 * Parse the rest of the line
			 */
			getkey(from_fptr, key, '>');
			/*
			 * Assign values to input action structure
			 */
			fk.keycode = trankey(key, display);
			/*
			 * if the key is a mouse button, use the motion
			 * device id else use the keyboard device id
			 */
			if (fk.keycode < 8)
			{
				fk.header = XTestPackDeviceID(motion_device) |
					    XTestKEY_ACTION;
			}
			else
			{
				fk.header = XTestPackDeviceID(keyboard_device) |
					    XTestKEY_ACTION;
			}
			fk.header |= XTestKEY_UP;
			fk.delay_time = key_interval;
			if (G_debug_flag)
			{
				(void) fprintf(G_msg_file,
					       "KEY_UP Key = %s, code = %u\n",
				               key,
				               fk.keycode);
			}
			/*
			 * Write out input action structure to script file
			 */
			chk_copy(to_fd,
				 (unsigned char *) &fk,
				 sizeof(XTestKeyInfo));
		}
		/*
		 * ...............KEYPRESS INPUT ACTION......................
		 */
		else if (strcmp(type, "<KEY_DOWN") == 0)
		{
			/*
			 * Parse the rest of the line
			 */
			getkey(from_fptr, key, '>');
			/*
			 * Assign values to input action structure
			 */
			fk.keycode = trankey(key, display);
			/*
			 * if the key is a mouse button, use the motion
			 * device id else use the keyboard device id
			 */
			if (fk.keycode < 8)
			{
				fk.header = XTestPackDeviceID(motion_device) |
					    XTestKEY_ACTION;
			}
			else
			{
				fk.header = XTestPackDeviceID(keyboard_device) |
					    XTestKEY_ACTION;
			}
			fk.header |= XTestKEY_DOWN;
			fk.delay_time = key_interval;
			if (G_debug_flag)
			{
				(void) fprintf(G_msg_file,
					       "KEY_DOWN Key = %s, code = %u\n",
				               key,
				               fk.keycode);
			}
			/*
			 * Write out input action structure to script file
			 */
			chk_copy(to_fd,
				 (unsigned char *) &fk,
				 sizeof(XTestKeyInfo));
		}
		/*
		 * ..................MATCH INPUT ACTION........................
		 */
		else if (strcmp(type, "<MATCH") == 0)
		{
			if (G_ia_count > 0)
			{
				write_input_actions(to_fd);
			}
			sumtoint(from_fptr, to_fd, stripflag, &playback_info);
		}
		/*
		 * ................COMMENT INPUT ACTION........................
		 */
		else if (strcmp(type, "<COMMENT") == 0)
		{
			/*
			 * Parse the rest of the line
			 */
			getkey(from_fptr, key, '>');
			if (G_debug_flag)
			{
				(void) fprintf(G_msg_file,
					       "COMMENT = %s\n",
					       key);
			}
			if (G_ia_count > 0)
			{
				write_input_actions(to_fd);
			}
			(void) sprintf(cbuf,
				       "c%d ",
				       (strlen(key) + 1) * sizeof(char));
			wrchk(to_fd, cbuf);
			/*
			 * write comment to test script
			 */
			wrchk2(to_fd,
			       key,
			       (unsigned int) ((strlen(key) + 1) *
					       sizeof(char)),
			       "Error writing keys to test script.\n");
		}
		/*
		 * ................CHECKSUM MODE LINE........................
		 */
		else if (strcmp(type, "<CHECKSUMS") == 0)
		{
			if (G_ia_count > 0)
			{
				write_input_actions(to_fd);
			}
			/*
			 * Parse the rest of the line for the mode name
			 */
			getkey(from_fptr, key, '>');
			tran_checksum_mode(to_fd, key);
		}
		/*
		 * ..............KEY STROKE MULTI-CHAR SYMBOL.................
		 */
		else if (type[0] == '<') 
		{
			/*
			 * Parse the type string for Key Symbol
			 */
			if (sscanf(type, "<%s", key) != 1)
			{
				(void) fprintf(G_msg_file,
					       "error reading key name from summary format file\n");
				exit(1);
			}
			/*
			 * Remove the ending '>' from key
			 */
			key[strlen(key) - 1] = '\0';
			fk.keycode = trankey(key, display);
			/*
			 * if the key is a mouse button, use the motion
			 * device id else use the keyboard device id
			 */
			if (fk.keycode < 8)
			{
				fk.header = XTestPackDeviceID(motion_device) |
					    XTestKEY_ACTION;
			}
			else
			{
				fk.header = XTestPackDeviceID(keyboard_device) |
					    XTestKEY_ACTION;
			}
			fk.header |= XTestKEY_DOWN;
			fk.delay_time = key_interval;
			if (G_debug_flag)
			{
				(void) fprintf(G_msg_file,
					       "KeyStroke Key = %s, code = %u\n",
				               key,
				               fk.keycode);
			}
			/*
			 * Write out the key down input action structure to script file
			 */
			chk_copy(to_fd,
				 (unsigned char *) &fk,
				 sizeof(XTestKeyInfo));
			/*
			 * if the key is a mouse button, use the motion
			 * device id else use the keyboard device id
			 */
			if (fk.keycode < 8)
			{
				fk.header = XTestPackDeviceID(motion_device) |
					    XTestKEY_ACTION;
			}
			else
			{
				fk.header = XTestPackDeviceID(keyboard_device) |
					    XTestKEY_ACTION;
			}
			fk.header |= XTestKEY_UP;
			fk.delay_time = stroke_interval;
			/*
			 * Write out the key up input action structure to script file
			 */
			chk_copy(to_fd,
				 (unsigned char *) &fk,
				 sizeof(XTestKeyInfo));
		}
		/*
		 * .............KEY STROKE SINGLE-CHAR SEQUENCE................
		 */
		else
		{
			if (G_debug_flag)
			{
				(void) fprintf(G_msg_file,
					       "KEY SEQUENCE = %s\n",
					       type);
			}
			/*
			 * Parse sequence of key strokes
			 */
			for (i = 0; type[i] != '\0'; i++)
			{
				/*
				 * put one key stroke into key and
				 * the string terminator          
				 */ 
				key[0] = type[i];
				key[1] = '\0';
				/*
				 * translate key sym into key code
				 * and assign values to key input action 
				 */
				fk.keycode = trankey(key, display);
				/*
				 * if the key is a mouse button, use the motion
				 * device id else use the keyboard device id
				 */
				if (fk.keycode < 8)
				{
					fk.header = XTestPackDeviceID(motion_device) |
						    XTestKEY_ACTION;
				}
				else
				{
					fk.header = XTestPackDeviceID(keyboard_device) |
						    XTestKEY_ACTION;
				}
				fk.header |= XTestKEY_DOWN;
				fk.delay_time = key_interval;
				if (G_debug_flag)
				{
					(void) fprintf(G_msg_file,
						       "key sequence Key = %s, code = %u\n",
					               key,
					               fk.keycode);
				}
				/*
				 * Write out input action structure to script file
				 * Key down input action
				 */
				chk_copy(to_fd,
					 (unsigned char *) &fk,
					 sizeof(XTestKeyInfo));
				/*
				 * if the key is a mouse button, use the motion
				 * device id else use the keyboard device id
				 */
				if (fk.keycode < 8)
				{
					fk.header = XTestPackDeviceID(motion_device) |
						    XTestKEY_ACTION;
				}
				else
				{
					fk.header = XTestPackDeviceID(keyboard_device) |
						    XTestKEY_ACTION;
				}
				/*
				 * Key up input action
				 */
				fk.header |= XTestKEY_UP;
				fk.delay_time = stroke_interval;
				chk_copy(to_fd,
					 (unsigned char *) &fk,
					 sizeof(XTestKeyInfo));
			}
		}
	}
	if (G_ia_count > 0)
	{
		/*
		 * write out the data
		 */
		write_input_actions(to_fd);
	}
	/*
	 * close the display, no longer needed
	 */
        XCloseDisplay(display);
}

/***********************************************************************
 *
 *	This routine translates a string into the keycode that it represents.
 */
static CARD8
trankey(str, display)
/*
 * The key name to be converted to code
 */
char	*str;
/*
 * argument pointer to default X11 display
 */
Display	*display;
{
	/*
	 * local temp Key Symbol var
	 */
	KeySym	tempKeySymbol;
	/*
	 * local temp Key Code var
	 */
	KeyCode	tempKeyCode;
	/*
	 * local string for key name
	 */
	char    *keystring;
	/*
	 * used to index through the button names
	 */
	int	i;

	/*
	 * convert the key string to the generic key symbol code
	 */
	tempKeySymbol = XStringToKeysym(str);
	/*
	 * convert the generic code to keyboard scan code
	 */
	if (tempKeySymbol != NoSymbol)
	{
		tempKeyCode = XKeysymToKeycode(display, tempKeySymbol);
	}
	/*
	 * if we didn't get a key code, try comparing it to the button names
	 */
	if ((tempKeySymbol == NoSymbol) || (tempKeyCode == 0))
	{
		/*
		 * loop through the button names (the buttons have no keysyms,
		 * but have keycodes 0 through 7)
		 */
		for (i = 0; i < 8; i++)
		{
			/*
			 * if the string matches a button name, use
			 * the index into the button names as the key code
			 */
			if (strcmp(str, G_button_names[i]) == 0)
			{
				tempKeyCode = i;
				break;
			}
		}
		/*
		 * if the string didn't match a normal key or a button name,
		 * error
		 */
		if (i == 8)
		{
			(void) fprintf(G_msg_file, "invalid key '%s'\n", str);
			exit(1);
		}
	}
	/*
	 * Try converting the key code back to a key string.  There are
	 * some key strings that convert to a key code that does not
	 * correspond to a key string.  Find that out now rather than later.
	 *
	 * An example of this on my system is ")".
	 */
	if (tempKeyCode > 7)
	{
		/*
		 * Convert the key code to the key name
		 */
		tempKeySymbol = XKeycodeToKeysym(display, tempKeyCode, 0);
		/*
		 * if there is no keysym corresponding to the key name, error
		 */
		if (tempKeySymbol == NoSymbol)
		{
			(void) fprintf(G_msg_file, "invalid key '%s'\n", str);
			exit(1);
		}
		keystring = XKeysymToString(tempKeySymbol);
		/*
		 * if there is no string corresponding to the keysym, error
		 */
		if (keystring == (char *) 0)
		{
			(void) fprintf(G_msg_file, "invalid key '%s'\n", str);
			exit(1);
		}
		/*
		 * if the string does not match the original string, error
		 */
		if (strcmp(keystring, str) != 0)
		{
			(void) fprintf(G_msg_file, "invalid key '%s'\n", str);
			exit(1);
		}
	}
	/*
	 * if you get here, you have a good key code
	 */
	return((CARD8) tempKeyCode);
}

/***********************************************************************
 *
 *	tran_comment
 *
 *	Translate a comment.
 */
static void
tran_comment(fptr, ofd)
/*
 * points to output format file
 */
FILE		*fptr;
/*
 * file descriptor of internal format file
 */
register int	ofd;
{
	/*
	 * ???
	 */
	char	buf[WRITE_BUFFER_SIZE];
	/*
	 * ???
	 */
	char	cbuf[WRITE_BUFFER_SIZE];

	/*
	 * read in comment until a new-line character is seen
	 */
	if (fgets(cbuf, WRITE_BUFFER_SIZE, fptr) == (char *) 0)
	{
		(void) fprintf(G_msg_file,
			       "error %d reading comment from test script\n",
			       errno);
		exit(1);
	}
	/*
	 * strip off the new-line character if there is one
	 */
	if (cbuf[strlen(cbuf) - 1] == '\n')
	{
		cbuf[strlen(cbuf) - 1] = '\0';
	}
	/*
	 * skip over the first 4 characters (they are spaces)
	 */
	if (strlen(cbuf) < 4)
	{
		(void) fprintf(G_msg_file,
			       "error in comment from test script\n",
			       errno);
		exit(1);
	}
	/*
	 * write the count of characters in the comment to the test script
	 */
	(void) sprintf(buf, "c%d ", (strlen(cbuf) - 4 + 1) * sizeof(char));
	wrchk(ofd, buf);
	/*
	 * write comment to test script
	 */
	wrchk2(ofd,
	       cbuf + (4 * sizeof(char)),
	       (unsigned int) ((strlen(cbuf) - 4 + 1) * sizeof(char)),
	       "Error writing comment to test script.\n");
}

/***********************************************************************
 *
 *	tran_checksum_mode
 *
 *	Translate a checksum mode line from output format to internal format.
 */
static void
tran_checksum_mode(ofd, mode_string)
/*
 * file descriptor of internal format file
 */
register int	ofd;
/*
 * the mode string to parse
 */
char		*mode_string;
{
	if (strcmp(mode_string, "ON") == 0)
	{
		/*
		 * enable checksum mode
		 */
		G_checksum_flag = 1;
		wrchk(ofd, "Tce");
	}
	else if (strcmp(mode_string, "OFF") == 0)
	{
		/*
		 * disable checksum mode
		 */
		G_checksum_flag = 0;
		wrchk(ofd, "Tcd");
	}
	else
	{
		(void) fprintf(G_msg_file, "error in checksum mode\n");
		exit(1);
	}
}

/***********************************************************************
 *
 *	tran_version_number_from_output
 *
 *	Translate a version number line from output format to internal format.
 */
static void
tran_version_number_from_output(from_fptr, to_fd)
/*
 * points to the output format file
 */
FILE		*from_fptr;
/*
 * file descriptor of the internal format file
 */
register int	to_fd;
{
	/*
	 * holds the version title
	 */
	char	version_title[WRITE_BUFFER_SIZE];
	/*
	 * holds the version number
	 */
	int	version_number;

	/*
	 * read the version title
	 */
	if (fscanf(from_fptr, "%s", version_title) != 1)
	{
		(void) fprintf(G_msg_file,
			       "Error %d while reading the version title from the test script.\n",
			       errno);
		exit(1);
	}
	/*
	 * if debugging, say what we are translating
	 */
	if (G_debug_flag)
	{
		(void) fprintf(G_msg_file, "Translating: %s\n", version_title);
	}
	/*
	 * if the line is not a version number line, then error
	 */
	if (strcmp(version_title, "VERSION") != 0)
	{
		(void) fprintf(G_msg_file,
			       "Incorrect or missing version title in the test script.\n");
		exit(1);
	}
	/*
	 * read the version number
	 */
	if (fscanf(from_fptr, " [%d]", &version_number) != 1)
	{
		(void) fprintf(G_msg_file,
			       "Error %d while reading the version number from the test script.\n",
			       errno);
		exit(1);
	}
	/*
	 * if the version number is missing or doesn't match the version 
	 * of the program, then error
	 */
	if (version_number != VERSION_NUMBER)
	{
		(void) fprintf(G_msg_file,
			       "Incorrect or missing version number in the test script.\n");
		exit(1);
	}
        /*
	 * write the version number to a string
	 */
	(void) sprintf(version_title, "v%d ", version_number);
	/*
	 * write the string to the test script
	 */
	wrchk(to_fd, version_title);
}

/***********************************************************************
 *
 *	tran_version_number_from_summary
 *
 *	Translate a version number line from summary format to internal format.
 */
static void
tran_version_number_from_summary(from_fptr, to_fd)
/*
 * points to the summary format file
 */
FILE		*from_fptr;
/*
 * file descriptor of the internal format file
 */
register int	to_fd;
{
	/*
	 * holds the version title
	 */
	char	version_title[WRITE_BUFFER_SIZE];
	/*
	 * holds the version number
	 */
	int	version_number;

	/*
	 * read the version title
	 */
	if (fscanf(from_fptr, "%s", version_title) != 1)
	{
		(void) fprintf(G_msg_file,
			       "Error %d while reading the version title from the test script.\n",
			       errno);
		exit(1);
	}
	/*
	 * if debugging, say what we are translating
	 */
	if (G_debug_flag)
	{
		(void) fprintf(G_msg_file, "Translating: %s\n", version_title);
	}
	/*
	 * if the line is not a version number line, then error
	 */
	if (strcmp(version_title, "<VERSION") != 0)
	{
		(void) fprintf(G_msg_file,
			       "Incorrect or missing version title in the test script.\n");
		exit(1);
	}
	/*
	 * read the version number
	 */
	if (fscanf(from_fptr, " %d>", &version_number) != 1)
	{
		(void) fprintf(G_msg_file,
			       "Error %d while reading the version number from the test script.\n",
			       errno);
		exit(1);
	}
	/*
	 * if the version number is missing or doesn't match the version 
	 * of the program, then error
	 */
	if (version_number != VERSION_NUMBER)
	{
		(void) fprintf(G_msg_file,
			       "Incorrect or missing version number in the test script.\n");
		exit(1);
	}
        /*
	 * write the version number to a string
	 */
	(void) sprintf(version_title, "v%d ", version_number);
	/*
	 * write the string to the test script
	 */
	wrchk(to_fd, version_title);
}
