/**********************************************************************
 *
 *	getbox.c
 *
 **********************************************************************/

/*
 * Copyright 1989 by Hewlett-Packard Corporation
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.
 * 
 * Hewlett-Packard makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 * 
 * This software is not subject to any license of the American Telephone and
 * Telegraph Company or of the Regents of the University of California.
 * 
 */

/**********************************************************************
 * include files
 **********************************************************************/

/*
 * contains NULL constant
 */
#include  <stdio.h>
/*
 * X libraries
 */
#include <X11/X.h>
#include <X11/Xmd.h>
#include <X11/Intrinsic.h>
#include <X11/Shell.h>
#include <X11/cursorfont.h>

#define XK_MISCELLANY
#define XK_LATIN1
#include <X11/keysymdef.h>
#include  <sys/time.h>
/*
 * X input extension definitions
 */
#include <X11/extensions/xtestext1.h>
#include <xtm.h>

/**********************************************************************
 * external variables
 **********************************************************************/


extern Display *G_disp_ptr;	/* pointer to the display */

extern unsigned int G_sb_size;	/* size of screen-match buffer */

extern int      G_debug_flag;	/* debug invocation option flag *//* ..debug_f
				 * lag is undocumented */

extern FILE    *G_msg_file;	/* diagnostic/error messages */
extern int      errno;

extern int      XTestInputActionType;

extern KeyCode  G_acceptpartial_key;
extern KeyCode  G_rejectpartial_key;
/*********************************************************************
 * definitions
 **********************************************************************/
/*
/*
 * absolute value macro
 */
#ifndef ABS
#define ABS(x) (((x) > 0) ? (x) : (-(x)))
#endif

/*
 * function declarations
 */
int             init_getbox_params();
Bool            init_getbox();
Bool            get_box_posn();
int             box_key_interaction();
Bool            MakeMotion();
Bool            jump_box();
void            move_box();
Bool            flush_moves();
/*
 * functions for internal use only
 */
static void     box_restore_cursor();
int             DrawOutline();
static Bool     ForceOnWindow();
static int      SetPointerPosition();
static Bool     SwitchCorners();
static Bool     DoGrabs();
static void     UndoGrabs();
static Cursor   ConfigCursor();
static int      TWarpPointer();
static int      TSync();
/*
 * external functions
 */
extern void     gettmp();
extern char    *malloc();
extern XImage  *XGetImage();
extern GC       XCreateGC();
extern int      XGrabPointer();

/**********************************************************************
 * data
 **********************************************************************/
/*
 * items global to the module
 */
static GD       boxGD;		/* miscellaneous local params */

static int      cursposX;	/* saved cursor posn at start */
static int      cursposY;	/* of interaction.	     */

int             moveX = 0;	/* accumulated, but not yet  */
int             move16Y = 0;	/* issued, move values.	     */

static int      outerX;		/* .extreme values permitted */
static int      outerY;		/* .for place_ parameters.   */
static int      outerOX;	/* .  X,Y coords are ulc.    */
static int      outerOY;	/* .  XO,YO coords are lrc.  */

static int      placeX;		/* .			     */
static int      placeY;		/* .location and size of     */
static int      placeOX;	/* .box during interaction.  */
static int      placeOY;	/* .  X,Y coords are ulc.    */
static unsigned int placeWidth;	/* .  XO,YO coords are lrc.  */
static unsigned int placeHeight;

static int      placeCX;	/* location of cursor */
static int      placeCY;	/* during interaction.       */

static Bool     placeULC;	/* these are various flags   */
static Bool     placeResize;
static Bool     placementPending;
static Bool     placementDone;
static Bool     placementAbandoned;

/**********************************************************************
 * functions
 **********************************************************************/

/*************************************<->*************************************
 *
 *  int get_box_posn( &x, &y, &width, &height )
 *
 *
 *  Description:
 *  -----------
 *  Gets the final box size and position information
 *
 *
 *  Inputs:
 *  ------
 *
 *
 *  Outputs:
 *  -------
 *
 *
 *  Comments:
 *  --------
 *
 *************************************<->***********************************/
Bool
get_box_posn(x, y, width, height)
	int            *x, *y, *width, *height;
{
	*x = placeX;
	*y = placeY;
	*width = placeWidth;
	*height = placeHeight;

	box_restore_cursor();

	return !placementAbandoned;
}




/*************************************<->*************************************
 *
 *  init_getbox_params()
 *
 *
 *  Description:
 *  -----------
 *  Perform the initialization for interactive placement
 *
 *
 *  Inputs:
 *  ------
 *
 *  Outputs:
 *  -------
 *
 *
 *  Comments:
 *  --------
 * o sets up global data and puts initial display on the screen
 *
 *************************************<->***********************************/
int
init_getbox_params()
{
	XGCValues       gcv;
	XtGCMask        mask;
	XColor          cursor_color;
	Pixmap          cursor_pixmap;
	unsigned int    cursor_width, cursor_height;
	GC              cursorGC;
	Status          ok;


	/*
	 * Initialize workspace global data:
	 */

	boxGD.display = G_disp_ptr;
	boxGD.rootWindow = DefaultRootWindow(G_disp_ptr);
	boxGD.screen = get_current_screen();

	mask = GCFunction | GCLineWidth | GCSubwindowMode;
	gcv.function = GXinvert;
	gcv.line_width = 1;
	gcv.subwindow_mode = IncludeInferiors;

	boxGD.xorGC = XCreateGC(boxGD.display, boxGD.rootWindow, mask, &gcv);

	/*
	 * Create cursors (an amazing amount of effort is expended, the
	 * result being an invisible cursor)
	 */

	ok = XQueryBestCursor(G_disp_ptr, boxGD.rootWindow, 1, 1,
			      &cursor_width, &cursor_height);
	if (ok == True) {
		cursor_pixmap = XCreatePixmap(G_disp_ptr, boxGD.rootWindow,
					    cursor_width, cursor_height, 1);
		/*
		 * set up a GC for the 'empty' cursor definition.
		 */
		mask = GCFunction;
		gcv.function = GXclear;
		cursorGC = XCreateGC(boxGD.display, cursor_pixmap, mask, &gcv);
		XFillRectangle(G_disp_ptr, cursor_pixmap, cursorGC,
			       0, 0, cursor_width - 1, cursor_height - 1);
		cursor_color.pixel = BlackPixel(G_disp_ptr, boxGD.screen);
		XQueryColor(G_disp_ptr,
			    DefaultColormap(G_disp_ptr, boxGD.screen),
			    &cursor_color);
		/*
		 * blank cursor saved
		 */
		boxGD.ulcCursor = XCreatePixmapCursor(G_disp_ptr,
					       cursor_pixmap, cursor_pixmap,
					       &cursor_color, &cursor_color,
						      0, 0);
		boxGD.lrcCursor = boxGD.ulcCursor;
	}
	/*
	 * boxGD.ulcCursor = XCreateFontCursor(G_disp_ptr, XC_ul_angle);
	 * boxGD.lrcCursor = XCreateFontCursor(G_disp_ptr, XC_lr_angle);
	 */
}



/*************************************<->*************************************
 *
 *  init_getbox( window, p )
 *
 *
 *  Description:
 *  -----------
 *  Perform the initialization for interactive placement
 *
 *
 *  Inputs:
 *  ------
 *
 *  Outputs:
 *  -------
 *
 *
 *  Comments:
 *  --------
* o sets up global data and puts initial display on the screen
 *
 *************************************<->***********************************/
#define CURSOR_OFFSET	0
Bool
init_getbox(window, x, y, width, height)
	Window          window;
	int             x, y, width, height;
{
	int             junk;
	Window          junk_win;
	XWindowAttributes win_info;
	int             ok;

	ok = DoGrabs(boxGD.ulcCursor);
	if (!ok)
		return False;	/* unable to grab server */

	/* initial placement is at ulc */

	ok = XQueryPointer(G_disp_ptr, boxGD.rootWindow, &junk_win, &junk_win,
		&cursposX, &cursposY, &junk, &junk, (unsigned int *) &junk);
	if (!ok) {
		UndoGrabs();
		return False;	/* unable to examine pointer */
	}
	placeCX = cursposX;
	placeCY = cursposY;

	/*
	 * outer boundary allowed for rectangle
	 */

	if (!XGetWindowAttributes(G_disp_ptr, window, &win_info))
		return False;

	outerX = win_info.x;
	outerY = win_info.y;
	outerOX = win_info.x + win_info.width + 2 * win_info.border_width - 1;
	outerOY = win_info.y + win_info.height + 2 * win_info.border_width - 1;

	/*
	 * initial box position and size parameters
	 */

	if (width == 0 && height == 0) {
		placeX = cursposX - CURSOR_OFFSET;
		placeY = cursposY - CURSOR_OFFSET;
		ForceOnWindow(&placeX, &placeY);
		placeWidth = win_info.width / 10;
		placeHeight = win_info.height / 10;
		SetPointerPosition(cursposX, cursposY, &junk, &junk);
	} else {
		placeX = x;
		placeY = y;
		placeWidth = width;
		placeHeight = height;
		SetPointerPosition(x + CURSOR_OFFSET, y + CURSOR_OFFSET,
				   &junk, &junk);
	}

	/*
	 * adjust rectangle size parameters & set position of lower right
	 * corner
	 */
	if (placeWidth > win_info.width)
		placeWidth = win_info.width;
	if (placeHeight > win_info.height)
		placeHeight = win_info.height;

	placeOX = placeX + placeWidth - 1;
	placeOY = placeY + placeHeight - 1;

	if (placeOX > outerOX) {
		placeX -= (placeOX - outerOX);
		placeOX = outerOX;
	}
	if (placeOY > outerOY) {
		placeY -= (placeOY - outerOY);
		placeOY = outerOY;
	}
	/* in "position" mode to start with */
	placeULC = True;
	placeResize = False;

	placementDone = False;
        placementPending = False;
	placementAbandoned = False;

	/* set up initial visual feedback */
	DrawOutline(placeX, placeY, placeWidth, placeHeight);
	XSync(G_disp_ptr, FALSE);

	return True;		/* initialized normally */
}



/*************************************<->*************************************
 *
 *  box_key_interaction(keycode, dir)
 *
 *
 *  Description:
 *  -----------
 *  Handle key presses during interactive placement
 *
 *
 *  Inputs:
 *  ------
 *  keycode	- raw character code
 *  dir		- XTestKEY_UP or XTestKEY_DOWN
 *
 *  Outputs:
 *  -------
 *
 *
 *  Comments:
 *  --------
 *
 *************************************<->***********************************/
int
box_key_interaction(keycode, dir)
	KeyCode         keycode;
	int             dir;
{
	/* button press is (keycode < 8) */
	if (keycode < 8)
		if (dir == XTestKEY_DOWN)
			{
			if (!placeULC)
				{
				placementPending = True;
				UndoGrabs();
				printf("Accept window match? <y/n>\n");
				}
			else
				placeResize = True;
			SwitchCorners();
			}
	if (keycode == G_acceptpartial_key)
		if (dir == XTestKEY_UP)
			placementDone = True;
	if (keycode == G_rejectpartial_key)
		if (dir == XTestKEY_UP)
			{
			placementDone = True;
			placementAbandoned = True;
			}
	return !placementDone;
}


/*************************************<->*************************************
 *
 *  Bool MakeMotion ( )
 *
 *
 *  Description:
 *  -----------
 *  Draw outline at fixed offset from the cursor position, (placeCX, placeCY)
 *
 *  -------
 *
 *
 *  Comments:
 *  --------
 *
 *************************************<->***********************************/
static          Bool
MakeMotion()
{
	int             CornerX, CornerY;

        if (placementPending)
		return;
	if (placeULC) {
		CornerX = placeCX - CURSOR_OFFSET;
		CornerY = placeCY - CURSOR_OFFSET;
	} else {
		CornerX = placeCX + CURSOR_OFFSET;
		CornerY = placeCY + CURSOR_OFFSET;
	}

	if (placeULC) {		/* track upper left corner */
		if (placeResize) {
			/*
			 * leave opposite corner in place. (but keep the
			 * rectangle from becoming degenerate)
			 */
			if (CornerX >= placeOX)
				placeX = placeOX;
			else if (CornerX < outerX)
				placeX = outerX;
			else
				placeX = CornerX;

			if (CornerY >= placeOY)
				placeY = placeOY;
			else if (CornerY < outerY)
				placeY = outerY;
			else
				placeY = CornerY;

			placeWidth = placeOX - placeX + 1;
			placeHeight = placeOY - placeY + 1;
		} else {
			/* MODS HERE */
			if (CornerX < outerX)
				placeX = outerX;
			else
				placeX = CornerX;

			placeOX = placeX + placeWidth - 1;

			if (CornerY < outerY)
				placeY = outerY;
			else
				placeY = CornerY;

			placeOY = placeY + placeHeight - 1;
		}
	} else {		/* track lower right corner */
		if (placeResize) {
			/*
			 * logic as above
			 */
			if (CornerX <= placeX)
				placeOX = placeX;
			else
				placeOX = CornerX;

			if (CornerY <= placeY)
				placeOY = placeY;
			else
				placeOY = CornerY;

			placeWidth = placeOX - placeX + 1;
			placeHeight = placeOY - placeY + 1;
		} else {
			if (CornerX > outerOX)
				placeOX = outerOX;

			else if (CornerX - placeWidth + 1 < outerX)
				placeOX = outerX + placeWidth - 1;
			else
				placeOX = CornerX;

			placeX = placeOX - placeWidth + 1;

			if (CornerY > outerOY)
				placeOY = outerOY;
			else if (CornerY - placeHeight + 1 < outerY)
				placeOY = outerY + placeHeight - 1;
			else
				placeOY = CornerY;

			placeY = placeOY - placeHeight + 1;

		}
	}

	DrawOutline(placeX, placeY, placeWidth, placeHeight);

	return True;
}


/*************************************<->*************************************
 *
 *  int DrawOutline (x, y, width, height)
 *
 *
 *  Description:
 *  -----------
 *  Draw a window outline on the root window.
 *  (this version uses XFillRectangles, giving a solider-looking box)
 *
 *
 *  Inputs:
 *  ------
 *  x		- x coordinate (on root)
 *  y		- y coordinate (on root)
 *  width	- pixel width of frame
 *  height	- pixel height of frame
 *
 *  Outputs:
 *  -------
 *
 *
 *  Comments:
 *  --------
 *  o get display, root window ID, and xorGC out of global data.
 *
 *************************************<->***********************************/
int
DrawOutline(x, y, width, height)
	int             x, y;
	unsigned int    width, height;
{
	XRectangle      outline[8];
	XRectangle     *r = outline;

	static unsigned int minWidth, minHeight, maxHeight, maxWidth;

	static int      lastOutlineX = 0;
	static int      lastOutlineY = 0;
	static int      lastOutlineWidth = 0;
	static int      lastOutlineHeight = 0;
	int             fatness = MOVE_OUTLINE_WIDTH;

	if (x == lastOutlineX && y == lastOutlineY &&
	    width == lastOutlineWidth && height == lastOutlineHeight) {
		return 0;	/* no change */
	}
	if (lastOutlineWidth || lastOutlineHeight) {
		r->x = lastOutlineX - fatness;
		r->y = lastOutlineY - fatness;
		r->width = lastOutlineWidth + 2 * fatness;
		r++->height = fatness;
		r->x = lastOutlineX - fatness;
		r->y = lastOutlineY + lastOutlineHeight;
		r->width = lastOutlineWidth + 2 * fatness;;
		r++->height = fatness;
		r->x = lastOutlineX - fatness;
		r->y = lastOutlineY;
		r->width = fatness;
		r++->height = lastOutlineHeight;
		r->x = lastOutlineX + lastOutlineWidth;
		r->y = lastOutlineY;
		r->width = fatness;
		r++->height = lastOutlineHeight;
	}
	lastOutlineX = x;
	lastOutlineY = y;
	lastOutlineWidth = width;
	lastOutlineHeight = height;
	if (lastOutlineWidth || lastOutlineHeight) {
		r->x = lastOutlineX - fatness;
		r->y = lastOutlineY - fatness;
		r->width = lastOutlineWidth + 2 * fatness;
		r++->height = fatness;
		r->x = lastOutlineX - fatness;
		r->y = lastOutlineY + lastOutlineHeight;
		r->width = lastOutlineWidth + 2 * fatness;;
		r++->height = fatness;
		r->x = lastOutlineX - fatness;
		r->y = lastOutlineY;
		r->width = fatness;
		r++->height = lastOutlineHeight;
		r->x = lastOutlineX + lastOutlineWidth;
		r->y = lastOutlineY;
		r->width = fatness;
		r++->height = lastOutlineHeight;
	}
	if (r != outline) {
		XFillRectangles(G_disp_ptr, boxGD.rootWindow, boxGD.xorGC,
				outline, r - outline);
		return 1;
	}
	 /* box was drawn   */ 
	else
		return 0;	/* nothing to draw */
}




/*************************************<->*************************************
 *
 *  DoGrabs (cursor)
 *
 *
 *  Description:
 *  -----------
 *  Do the grabs for window configuration
 *
 *
 *  Inputs:
 *  ------
 *
 *
 *  Outputs:
 *  -------
 *
 *  Comments:
 *  --------
 *
 *************************************<->***********************************/
static          Bool
DoGrabs(cursor)
	Cursor          cursor;
{
	Bool            ok;

	ok = XGrabPointer(G_disp_ptr,
			  boxGD.rootWindow,
			  FALSE,/* owner_events */
			  0,	/* events via test extnsn */
			  GrabModeAsync,	/* pointer_mode */
			  GrabModeAsync,	/* keyboard_mode */
			  boxGD.rootWindow,	/* confine_to window */
			  cursor,
			  CurrentTime) == GrabSuccess;

	if (ok) {
		 XGrabServer(G_disp_ptr);
	}
	if (!ok)
		fprintf(G_msg_file, "DoGrabs...Unable to grab resources");

	return ok;
}


/*************************************<->*************************************
 *
 *  UndoGrabs ()
 *
 *
 *  Description:
 *  -----------
 *  Release the grabs
 *
 *
 *  Inputs:
 *  ------
 *
 *
 *  Outputs:
 *  -------
 *
 *  Comments:
 *  --------
 *
 *************************************<->***********************************/
static void
UndoGrabs()
{
	/* give up grabs */

	XUngrabServer(G_disp_ptr);

	XUngrabPointer(G_disp_ptr, CurrentTime);
}




/*************************************<->*************************************
 *
 *  ConfigCursor (frame_part)
 *
 *
 *  Description:
 *  -----------
 *  return the config cursor that goes with the config part specified
 *
 *
 *  Inputs:
 *  ------
 *  frame_part	- frame part id
 *
 *  Outputs:
 *  -------
 *  return	- cursor to use
 *
 *
 *  Comments:
 *  --------
 *
 *************************************<->***********************************/
static          Cursor
ConfigCursor(frame_part)
	int             frame_part;
{
	Cursor          cursor;
	switch (frame_part) {
	case ULC:
		cursor = boxGD.ulcCursor;
		break;
	case LRC:
		cursor = boxGD.lrcCursor;
		break;
	default:
		cursor = boxGD.ulcCursor;
	}

	return (cursor);
}



/*************************************<->*************************************
 *
 *  ForceOnWindow (pX, pY)
 *
 *
 *  Description:
 *  -----------
 *  Correct (if necessary) the coords specified to keep them in bounds
 *
 *
 *  Inputs:
 *  ------
 *  pX		- pointer to x-coord
 *  pY		- pointer to y-coord
 *
 *  Outputs:
 *  -------
 *  *pX		- x-coord (on screen)
 *  *pY		- y-coord (on screen)
 *
 *  Return value
 *  ------------
 *  True if a change is made, False otherwise.
 *
 *  Comments:
 *  --------
 *  init_getbox sets outerX, outerY, outerOX, outerOY, the boundary values,
 *
 *************************************<->***********************************/
static          Bool
ForceOnWindow(pX, pY)
	int            *pX, *pY;
{
	int             rval = False;

	if (*pX < outerX) {
		*pX = outerX;
		rval = True;
	} else if (*pX > outerOX) {
		*pX = outerOX;
		rval = True;
	}
	if (*pY < outerY) {
		*pY = outerY;
		rval = True;
	} else if (*pY > outerOY) {
		*pY = outerOY;
		rval = True;
	}
	return rval;
}



/*************************************<->*************************************
 *
 *  int SetPointerPosition (newX, newY, finalX, finalY)
 *
 *
 *  Description:
 *  -----------
 *  Attempt to set the pointer to position at newX, newY.
 *
 *
 *  Inputs:
 *  ------
 *  newX	- X-coordinate to set pointer at
 *  newY	- Y-coordinate to set pointer at
 *
 *
 *  Outputs:
 *  -------
 *
 *
 *  Comments:
 *  --------
 *
 *************************************<->***********************************/
static int
SetPointerPosition(newX, newY, actualX, actualY)
	int             newX, newY;
	int            *actualX, *actualY;
{
	int             junk;
	Window          junk_win;
	int             initX = newX, initY = newY;

	/*
	 * Warp pointer to position guaranteed to be on screen
	 */
	(void) ForceOnWindow(&initX, &initY);
	TWarpPointer(G_disp_ptr, None, boxGD.rootWindow,
		     0, 0, 0, 0, initX, initY);
	TSync(G_disp_ptr, False);

	*actualX = initX;
	*actualY = initY;

	return 0;		/* always claims success */
}

/*
 * jump_box moves "jumps" the box to a new location. Accumulated but unissued
 * moves are cleared..
 */
Bool
jump_box(action_ptr)
	XTestJumpInfo  *action_ptr;
{
	int             pointer_no;

	/*
	 * find out which pointer is involved. Note-- don't know what to do
	 * about multiple ptrs !!
	 */

	pointer_no = XTestUnpackDeviceID(action_ptr->header);
	if (G_debug_flag && pointer_no != POINTER_NO)
		fprintf(G_msg_file, "jump_box...pointer # %d", pointer_no);

	/*
	 * clear overridden move variables
	 */

	moveX = move16Y = 0;

	placeCX = action_ptr->jumpx;
	placeCY = action_ptr->jumpy;

	return MakeMotion();
}

/*
 * move_box accumulates a move value.  The move is actually accomplished by
 * flush_moves.
 */
void
move_box(action_ptr)
	XTestMotionInfo *action_ptr;
{
	/*
	 * find out which pointer is involved. NOTE -- don't know what to do
	 * about an unexpected pointer
	 */

	if (G_debug_flag &&
	    XTestUnpackDeviceID(action_ptr->header) != POINTER_NO)
		fprintf(G_msg_file, "move_box...pointer # %d",
			XTestUnpackDeviceID(action_ptr->header));

	/*
	 * get the motion from the packet
	 */
	if (action_ptr->header & XTestX_NEGATIVE)
		moveX -= action_ptr->motion_data & XTestX_MOTION_MASK;
	else
		moveX += action_ptr->motion_data & XTestX_MOTION_MASK;

	if (action_ptr->header & XTestY_NEGATIVE)
		move16Y -= action_ptr->motion_data & XTestY_MOTION_MASK;
	else
		move16Y += action_ptr->motion_data & XTestY_MOTION_MASK;

	return;
}

Bool
flush_moves()
{
	Bool            rval;

	if (moveX == 0 && move16Y == 0)
		rval = False;
	else {
		placeCX += moveX;
		placeCY += (move16Y >> 4);
		(void) MakeMotion();
		moveX = move16Y = 0;
		rval = True;
	}

	return rval;
}

static void
box_restore_cursor()
{
	/*
	 * remove window outline
	 */

	DrawOutline(0, 0, 0, 0);

	/*
	 * restore pointer position and appearance and Ungrab the server
	 */

	TWarpPointer(G_disp_ptr, None, boxGD.rootWindow,
		     0, 0, 0, 0, cursposX, cursposY);
	TSync(G_disp_ptr, False);
	UndoGrabs();
	return;
}

static          Bool
SwitchCorners()
{
	Cursor          current_cursor;

	placeULC = !placeULC;
	if (placeULC)
		current_cursor = ConfigCursor(ULC);
	else
		current_cursor = ConfigCursor(LRC);

	/* regrab pointer to change cursor */
	XChangeActivePointerGrab(G_disp_ptr, 0, current_cursor, CurrentTime);

	if (placeULC) {
		placeCX = placeX + CURSOR_OFFSET;
		placeCY = placeY + CURSOR_OFFSET;
	} else {
		placeCX = placeOX - CURSOR_OFFSET;
		placeCY = placeOY - CURSOR_OFFSET;
	}
	SetPointerPosition(placeCX, placeCY, &placeCX, &placeCY);

	/*
	 * Note that moving the cursor will generate a motion event, which
	 * will later be received by MakeMotion, which will expect to move
	 * the rectangle.  As long as the motion is a jump, not a move, the
	 * 'motion' should be a zero-move.
	 */

	return (1);
}

/*
 * A stripped-down TestInput version of XWarpPointer
 */
static int
TWarpPointer(disp_ptr, dummyw, window,
	     dummyp1, dummyp2, dummyp3, dummyp4, xposn, yposn)
	Display        *disp_ptr;
	Window          dummyw, window;
	int             dummyp1, dummyp2, dummyp3, dummyp4;	/* not used */
	int             xposn, yposn;
{
	static unsigned long zero = 0;
	static int      xval;
	static int      yval;

	xval = xposn;
	yval = yposn;
	/*
	 * XTestMovePointer( disp_ptr, POINTER_NO, &zero, &xval, &yval, 1 );
	 */
	XTestMovePointer(disp_ptr, 0, &zero, &xval, &yval, 1);
}

/*
 * A stripped-down TestInput version of XSync
 */
static int
TSync(disp_ptr, dummy1)
	Display        *disp_ptr;
	int             dummy1;
{
	XTestFlush(disp_ptr);
}


/*****************************************************************
 *
 *	delay
 *
 *	This function delays the specified amount of milliseconds.
 *
 *	This is Berkeley-derived, but there is no way
 *	common to SYSV and Berkeley to do time delays more accurately
 *	than the nearest second.  If desired this could just
 *	round the value up to the next second and wait that long.
 */

/*
 * constants for select
 */

void
delay(milli_sec)
/*
 * milliseconds to wait
 */
	INT32           milli_sec;
{
	/*
	 * holds the amount of time to delay for the select call
	 */
	struct timeval  pause;

	if (milli_sec == 0) {
		return;
	}
	/*
	 * get seconds part
	 */
	pause.tv_sec = milli_sec / 1000;
	/*
	 * get microseconds part
	 */
	pause.tv_usec = (milli_sec % 1000) * 1000;
	/*
	 * timeout on pause interval
	 */
	(void) select(0, (int *) 0, (int *) 0, (int *) 0, &pause);
}
