/**********************************************************************
 *
 *	exmatch.c
 *
 **********************************************************************/

/*

Copyright 1986, 1987, 1988, 1989 by Hewlett-Packard Corporation
Copyright 1986, 1987, 1988, 1989 by the Massachusetts Institute of Technology

Permission to use, copy, modify, and distribute this
software and its documentation for any purpose and without
fee is hereby granted, provided that the above copyright
notice appear in all copies and that both that copyright
notice and this permission notice appear in supporting
documentation, and that the name of M.I.T. not be used in
advertising or publicity pertaining to distribution of the
software without specific, written prior permission.

Hewlett-Packard and M.I.T. make no representations about the 
suitability of this software for any purpose.  It is provided 
"as is" without express or implied warranty.

This software is not subject to any license of the American
Telephone and Telegraph Company or of the Regents of the
University of California.

*/

/**********************************************************************
 * include files
 **********************************************************************/

/*
 * contains NULL constant
 */
#include  <stdio.h>
/*
 * constants for select
 */
#include  <sys/time.h>
/*
 * X library files
 */
#include  <X11/Xlib.h>
/*
 * ???
 */
#include  <X11/Xmd.h>
/*
 * client exerciser constants
 */
#include  <xtm.h>

/**********************************************************************
 * external variables
 **********************************************************************/

/*
 * points to the display structure
 */
extern Display			*G_disp_ptr;
/*
 * invocation options
 */
extern int           		G_update_flag;
extern int           		G_continue_flag;
/*
 * undocumented debug invocation option
 */
extern int           		G_debug_flag;
/*
 * device file pointers
 */
extern FILE            		*G_msg_file;
extern FILE	            	*G_log_file;
/*
 * this holds the size of the match data buffer
 */
extern unsigned int		G_sb_size;
/*
 * this buffer holds the match data
 */
extern unsigned char		*G_sb;
/*
 * this buffer holds the "compressed" data
 */
extern unsigned char		*G_c_data_buffer;
/*
 * this buffer holds the "compressed" data
 */
extern struct playback_info	G_playback_info;
/*
 * if true then use checksums for matches
 */
extern int			G_checksum_flag;
/*
 * if true then compare color map information
 */
extern int			G_compare_color_map_flag;

extern int			errno;

/**********************************************************************
 * function definitions
 **********************************************************************/

long	lseek();
void	report_mismatch();
void	delay();
void	wintotemp();
void	temptoint();
void	inttotemp();
void	read_match_data_from_internal();
void	write_match_data_to_xwd();
void	readchk();

/**********************************************************************
 * functions
 **********************************************************************/

/**********************************************************************
 *
 *	exmatch
 *
 *	Match a portion of the screen and compare the result to that
 *	obtained when the test script was created.
 *	
 *	Returns -1 if the match fails, and 0 if it succeeds.
 */
int
exmatch(script_fd, new_fd, script_name, match_index)
/*
 * file descriptor of internal format test script file
 */
int	script_fd;
/*
 * file descriptor of new test script file (only used in update mode)
 */
int	new_fd;
/*
 * points to the name of the test script file
 */
char	*script_name;
/*
 * holds which match this is in the test script
 */
int	match_index;
{
	/*
	 * for old match data in old test script
	 */
	struct match_data	old_match_data;
	/*
	 * If 0, then the update/match succeeded.
	 * If -1, then the update/match failed.
	 */
	int			fail_flag;
	/*
	 * holds the number of attempts to match
	 */
	int			attempt;

	/*
	 * if there are too many matches in this test script, error out now
	 * rather than realize that error file names won't fit later
	 */
	if (match_index > MATCH_LIMIT)
	{
		(void) fprintf(G_msg_file,
			       "too many matches in '%s'\n",
			       script_name);
		(void) fprintf(G_log_file,
			       "too many matches in '%s'\n",
			       script_name);
		exit(1);
	}
	/*
	 * Read the old match data structure information from the test script.
	 * This allocates space to hold color map entries, if needed.
	 */
	read_match_data_from_internal(script_fd, &old_match_data);
	/*
	 * set the file descriptor of the image data (if any)
	 * to the test script
	 */
	old_match_data.tfd = script_fd;
	/*
	 * adjust the repetition factor if necessary
	 */
	if (G_playback_info.retries != -1)
	{
		old_match_data.retry = G_playback_info.retries;
	}
	/*
	 * adjust the retry interval if necessary
	 */
	if (G_playback_info.interval != -1)
	{
		old_match_data.intv = G_playback_info.interval;
	}
	if (G_debug_flag)
	{
		(void) fprintf(G_msg_file,
			       "\nMATCH #%d TYPE[%c] WX[%d] WY[%d] X[%d] Y[%d] W[%d] H[%d] \n",
		              match_index,
		              old_match_data.type,
		              old_match_data.window_x,
		              old_match_data.window_y,
		              old_match_data.x,
		              old_match_data.y,
		              old_match_data.width,
		              old_match_data.height);
		(void) fprintf(G_msg_file,
			       "\t BO[%d] BU[%d] BBO[%d] BP[%d] D[%d] BPL[%d] BPP[%d] \n",
		              old_match_data.byte_order,
		              old_match_data.bitmap_unit,
		              old_match_data.bitmap_bit_order,
		              old_match_data.bitmap_pad,
		              old_match_data.depth,
		              old_match_data.bytes_per_line,
		              old_match_data.bits_per_pixel);
		(void) fprintf(G_msg_file,
			       "\t REP[%d] INTV[%d] CNT[%d]\n",
		              old_match_data.retry,
		              old_match_data.intv,
		              old_match_data.cnt);
	}
	/*
	 * read the new match data and compare it to
	 * the old match data
	 */
	fail_flag = do_match(&old_match_data,
			     script_name,
			     &attempt,
			     match_index,
			     new_fd);
	/*
	 * if the match succeeded, tell the user
	 */
	if (!fail_flag)
	{
		(void) fprintf(G_log_file,
			       "Match attempt %d of match %d succeeded.\n",
			       attempt,
			       match_index);
	}
	/*
	 * if there were any color map entries for the old match data,
	 * free their space
	 */
	if ((old_match_data.ncolors > 0) &&
	    (old_match_data.color_ptr != (XColor *) 0))
	{
		free((char *) (old_match_data.color_ptr));
	}
	return(fail_flag);
}

/**********************************************************************
 *
 *	do_match
 *
 *	Read the new match data from the server and compare it to the
 *	old match data.  If in update mode, just copy the new match data
 *	to the new test script.
 *
 *	Returns -1 if the comparison fails, and 0 if it succeeds.
 */
static int
do_match(old_match_data_ptr, script_name, attempt_ptr, match_index, new_fd)
/*
 * points to the information about the old match data in the old test script
 */
struct match_data	*old_match_data_ptr;
/*
 * points to the name of the test script
 */
char			*script_name;
/*
 * points to where to return the number of attempts it took before a success
 */
int			*attempt_ptr;
/*
 * holds which match this is in the test script
 */
int			match_index;
/*
 * file descriptor for the new test script file (only used in update mode)
 */
int			new_fd;
{
	/*
	 * for new match data from screen
	 */
	struct match_data	new_match_data;
	/*
	 * holds the window id to read new match data from
	 */
	Window			subw;
	/*
	 * holds the offset in the test script of the start
	 * of the old match data
	 */
	long			s_off;
	/*
	 * loop counter
	 */
	int			i;
	/*
	 * If 0, then the match succeeded.
	 * If -1, then the match failed.
	 */
	int			fail_flag;
	/*
	 * if non-zero says that a window was found
	 */
	int			found_win_flag;

	/*
	 * if we are not using checksums, remember where the old match
	 * data starts
	 */
	if (!G_checksum_flag)
	{
		/*
		 * remember the file offset of the old match data
		 * in the test script in case we need to try more than once
		 */
		if ((s_off = lseek(old_match_data_ptr->tfd, 0, 1)) == -1)
		{
			(void) fprintf(G_msg_file,
				       "error %d while seeking in old match data file\n",
				       errno);
			exit(1);
		}
	}
	/*
	 * set the fail flag to indicate that we have failed so far
	 */
	fail_flag = -1;
	/*
	 * Keep trying to match the data until we succeed or
	 * until we have tried enough.  The most common cause
	 * of a failure on the first try that succeeds on a later
	 * try is the test script not waiting long enough
	 * for the client to catch up before trying to match.
	 */
	for (i = 0; i <= old_match_data_ptr->retry; i++)
	{
		/*
		 * Get the window id of the new window corresponding to the
		 * one used by the old match data.  Also, set up some of the
		 * window information in the new match data structure.
		 */
		found_win_flag = get_match_window(old_match_data_ptr,
						  &new_match_data,
						  &subw);
		if (!found_win_flag)
		{
			/*
			 * didn't find a new window corresponding to the
			 * one used by the old match data, so fail
			 */
			fail_flag = -1;
		}
		else
		{
			/*
			 * Read the new match data from the screen area
			 * to be matched.  This creates a temporary file
			 * to hold the match data and allocates space
			 * to hold color map entries, if needed.
			 */
			wintotemp(subw, &new_match_data);
			/*
			 * if we are in update mode, the just skip over the
			 * old match data, otherwise compare the old match
			 * data with the new match data
			 */
			if (G_update_flag)
			{
				/*
				 * if not using checksums,
				 * skip over old match data
				 */
				if (!G_checksum_flag)
				{
					/*
					 * skip past the old match data
					 * in the old test script
					 */
					if (lseek(old_match_data_ptr->tfd,
						  (long) old_match_data_ptr->cnt,
						  1) == -1)
					{
						(void) fprintf(G_msg_file,
							       "error %d while seeking in old match data file\n",
							       errno);
						exit(1);
					}
				}
				/*
				 * the match cannot fail during an update
				 */
				fail_flag = 0;
			}
			else
			{
				/*
				 * compare the old and new match data
				 */
				fail_flag = compare_match_data(old_match_data_ptr,
							       &new_match_data);
			}
		}
		/*
		 * if the match succeeded, get out of the loop
		 */
		if (fail_flag != -1)
		{
			break;
		}
		/*
		 * log the failure
		 */
		(void) fprintf(G_log_file,
			       "Attempt %d of match %d failed.\n",
			       (i + 1),
			       match_index);
		/*
		 * if we are going to retry the match, clean up any
		 * un-needed new match data and delay the appropriate
		 * amount of time
		 */
		if (i < old_match_data_ptr->retry)
		{
			/*
			 * if we didn't find a corresponding window,
			 * we didn't read any new match data,
			 * so we don't need to clean it up
			 */
			if (found_win_flag)
			{
				if (!G_checksum_flag)
				{
					/*
					 * seek back to the beginning
					 * of the old match data
					 * in the test script
					 */
					if (lseek(old_match_data_ptr->tfd, s_off, 0)
					    == -1)
					{
						(void) fprintf(G_msg_file,
							       "error %d while seeking in old match data file\n",
							       errno);
						exit(1);
					}
					/*
					 * remove the temporary file that held the
					 * un-needed new match data
					 */
					if (close(new_match_data.tfd) == -1)
					{
						(void) fprintf(G_msg_file,
							       "error %d while closing match data temp file\n",
							       errno);
						exit(1);
					}
				}
				/*
				 * if there were any color map entries,
				 * free their space
				 */
				if ((new_match_data.ncolors > 0) &&
				    (new_match_data.color_ptr != (XColor *) 0))
				{
					free((char *) (new_match_data.color_ptr));
				}
			}
			/*
			 * delay the appropriate amount of time
			 */
			delay(old_match_data_ptr->intv);
		}
	}
	/*
	 * if the match failed, write out the actual and expected data
	 * and the appropriate error message
	 */
	if (fail_flag == -1)
	{
		/*
		 * If not using checksums, get ready for the next attempt.
		 * There is no actual and expected data to write out if
		 * using checksums.
		 */
		if (!G_checksum_flag)
		{
			/*
			 * write out the expected and actual data.
			 */
			output_match_failure_data(old_match_data_ptr,
						  &new_match_data,
						  script_name,
						  match_index,
						  s_off,
						  found_win_flag);
		}
		/*
		 * if we are not continuing in spite of errors,
		 * tell the user that the match failed and exit
		 */
		if (!G_continue_flag)
		{
			if (G_update_flag)
			{
				(void) fprintf(G_log_file,
					       "\nTest script '%s' failed - update aborted.\n",
					       script_name);
				(void) fprintf(G_msg_file,
					       "\nTest script '%s' failed - update aborted.\n",
					       script_name);
				exit(1);
			}
			else
			{
				(void) fprintf(G_log_file,
					       "\nTest script '%s' failed - test aborted.\n",
					       script_name);
				(void) fprintf(G_msg_file,
					       "\nTest script '%s' failed - test aborted.\n",
					       script_name);
				exit(1);
			}
		}
	}
	else
	{
		/*
		 * set the number of attempts it took before a successful match
		 */
		*attempt_ptr = i + 1;
	}
	/*
	 * if we are updating the test script, write out the new match data
	 * to the new test script else just clean up the new match data
	 */
	if (G_update_flag)
	{
		if (found_win_flag)
		{
			/*
			 * set the new match data type, retry count,
			 * and retry interval
			 */
			new_match_data.type = old_match_data_ptr->type;
			new_match_data.retry = old_match_data_ptr->retry;
			new_match_data.intv = old_match_data_ptr->intv;
			/*
			 * Write out the new match data to the new test script.
			 * This removes the temporary file that held the match
			 * data, if any and frees the space that held the color
			 * map entries.
			 */
			temptoint(new_fd, &new_match_data);
		}
		else
		{
			(void) fprintf(G_log_file,
				       "\nUnable to find a corresponding window , Update aborted\n");
			(void) fprintf(G_msg_file,
				       "\nUnable to find a corresponding window , Update aborted\n");
			exit(1);
		}
	}
	else
	{
		if (found_win_flag)
		{
			if (!G_checksum_flag)
			{
				/*
				 * remove the temporary file
				 * that held the new match data
				 */
				if (close(new_match_data.tfd) == -1)
				{
					(void) fprintf(G_msg_file,
						       "error %d while closing match data temp file\n",
						       errno);
					exit(1);
				}
			}
			/*
			 * if there were any color map entries, free their space
			 */
			if ((new_match_data.ncolors > 0) &&
			    (new_match_data.color_ptr != (XColor *) 0))
			{
				free((char *) (new_match_data.color_ptr));
			}
		}
	}
	return(fail_flag);
}

/**********************************************************************
 *
 *	get_match_window
 *
 *	Gets the window id of the new window corresponding to the one
 *	used by the old match data.  Also, set up the x, y, width, and
 *	height of the match area in the new match data structure.  If this
 *	routine returns non-zero, then a corresponding window was found.
 */
static int
get_match_window(old_match_data_ptr, new_match_data_ptr, window_ptr)
/*
 * points to the information about the old match data in the old test script
 */
struct match_data	*old_match_data_ptr;
/*
 * points to the information about the new match data in the temp file
 */
struct match_data	*new_match_data_ptr;
/*
 * points to where to return the window
 */
Window			*window_ptr;
{
	/*
	 * holds the return value
	 */
	int		found_win_flag;
	XRectangle      rect;
	int             computed_window_x, computed_window_y;

	switch(old_match_data_ptr->type)
	{
	case 't':
		/*
		 * Entire window match, so set the x and y offsets
		 * of the match area inside the window to 0.  They
		 * may get modified if the window is clipped by its
		 * parent window or the root window.
		 */
		new_match_data_ptr->x = 0;
		new_match_data_ptr->y = 0;
		/*
		 * get the window id, width, and height of the top window
		 */
		found_win_flag = gettop(new_match_data_ptr, window_ptr);
		break;
	case 'm':
		/*
		 * Entire window match, so set the x and y offsets
		 * of the match area inside the window to 0.  They
		 * may get modified if the window is clipped by its
		 * parent window or the root window.
		 */
		new_match_data_ptr->x = 0;
		new_match_data_ptr->y = 0;
		/*
		 * get the window id, width, and height of the window
		 * that the mouse is currently in
		 */
		found_win_flag = getmouse(new_match_data_ptr, window_ptr);
		break;
	case 's':
		/*
		 * entire screen match, so set the x and y offsets
		 * of the match area inside the root window to 0
		 */
		new_match_data_ptr->x = 0;
		new_match_data_ptr->y = 0;
		/*
		 * the window is the root window of the screen that
		 * the mouse is currently in
		 */
		found_win_flag = getroot(new_match_data_ptr, window_ptr);
		break;
	case 'p':
                /*
                 * Partial rectangle match. Set the x and y offsets and
                 * height and width of the match area inside the root
                 * to the explicit offsets.
                 */
                rect.x = old_match_data_ptr->x
                                + old_match_data_ptr->window_x
				+ old_match_data_ptr->window_border_width;
                rect.y = old_match_data_ptr->y
                                + old_match_data_ptr->window_y
				+ old_match_data_ptr->window_border_width;
                rect.width = old_match_data_ptr->width;
                rect.height = old_match_data_ptr->height;
                new_match_data_ptr->screen_number = get_current_screen();
                new_match_data_ptr->x = old_match_data_ptr->x;
                new_match_data_ptr->y = old_match_data_ptr->y;
                new_match_data_ptr->width = old_match_data_ptr->width;
                new_match_data_ptr->height = old_match_data_ptr->height;
                computed_window_x = 0;
		computed_window_y = 0;
                if (get_rect_window(RootWindow(G_disp_ptr, 
					       new_match_data_ptr->screen_number),
				    window_ptr,
                                    &rect,
				    &computed_window_x,
				    &computed_window_y)
		    != GoodCoverage)
		{
			/*
			 * report that we couldn't find a window that
			 * contained the unobscured rectangle 
			 */
			found_win_flag = 0;
			*window_ptr = (Window) 0;

		}
		else
		{
			/*
			 * report that a correct window was found
			 */
			found_win_flag = 1;
		}
                break;
	default:
		/*
		 * this should never happen
		 */
		(void) fprintf(G_msg_file,
			       "unknown match type %c\n",
			       (char) old_match_data_ptr->type);
		exit(1);
	}
	/*
	 * return the found window flag
	 */
	return(found_win_flag);
}

/**********************************************************************
 *
 *	compare_match_data
 *
 *	Compares the new and old match data.
 *
 *	Returns -1 if the comparison fails, and 0 if it succeeds.
 */
static int
compare_match_data(old_match_data_ptr, new_match_data_ptr)
/*
 * points to the information about the old match data in the old test script
 */
struct match_data	*old_match_data_ptr;
/*
 * points to the information about the new match data in the temp file
 */
struct match_data	*new_match_data_ptr;
{
	/*
	 * check the old match data description against the new match data
	 * description and fail if they are not the same
	 */
	if (!descriptions_match(old_match_data_ptr, new_match_data_ptr))
	{
		return(-1);
	}
	/*
	 * The descriptions matched.  If using checksums, compare the
	 * checksums, else compare the old and new match data
	 */
	if (G_checksum_flag)
	{
		if (report_ulong_mismatch(old_match_data_ptr->checksum,
					  new_match_data_ptr->checksum,
					  "checksum"))
		{
			return(-1);
		}
	}
	else
	{
		/*
		 * the descriptions matched, so compare the old compressed
		 * data and the new compressed data and fail if they are not
		 * the same
		 */
		if (!compare_data(old_match_data_ptr->tfd, new_match_data_ptr))
		{
			return(-1);
		}
	}
	/*
	 * if you get to here, then the comparison succeeded
	 */
	return(0);
}

/**********************************************************************
 *
 *	output_match_failure_data
 *
 *	The match has failed, write out the expected and actual data.
 */
static int
output_match_failure_data(old_match_data_ptr,
			  new_match_data_ptr,
			  script_name,
			  match_index,
			  old_match_data_seek_offset,
			  found_win_flag)
/*
 * points to the information about the old match data in the old test script
 */
struct match_data	*old_match_data_ptr;
/*
 * points to the information about the new match data in the temp file
 */
struct match_data	*new_match_data_ptr;
/*
 * points to the basename of the test script
 */
char			*script_name;
/*
 * holds which match this is in the test script
 */
int			match_index;
/*
 * holds the offset in the test script of the start
 * of the old match data
 */
long			old_match_data_seek_offset;
/*
 * if non-zero, found a window for the new match data
 */
int			found_win_flag;
{
	/*
	 * holds the base name of the test script
	 */
	char	test_script_basename[MAXPATH];
	/*
	 * holds the name of the xwd format file
	 */
	char	xwd_file_name[MAXPATH];
	/*
	 * holds the file descriptor for the xwd format file
	 */
	int	xwd_fd;

	/*
	 * output a failure message
	 */
	(void) fprintf(G_log_file,
		       "Match %d failed for all attempts.\n",
		       match_index);
	/*
	 * seek back to the beginning of the old match data
	 * in the test script
	 */
	if (lseek(old_match_data_ptr->tfd, old_match_data_seek_offset, 0) == -1)
	{
		(void) fprintf(G_msg_file,
			       "error %d while seeking in old match data file\n",
			       errno);
		exit(1);
	}
	/*
	 * figure out the base name of the test script
	 */
	(void) strncpy(test_script_basename,
		       script_name,
		       (strlen(script_name) - ENDING_LENGTH));
	/*
	 * make sure a null byte follows the base name
	 */
	test_script_basename[strlen(script_name) - ENDING_LENGTH] = '\0';
	/*
	 * create the file name for the expected data file
	 */
	(void) sprintf(xwd_file_name,
		       "%s.e%02d",
		       test_script_basename,
		       match_index);
	/*
	 * create the expected data file
	 */
	xwd_fd = makef(xwd_file_name);
	/*
	 * write the old match data out in xwd format to the expected data file
	 */
	write_match_data_to_xwd(xwd_fd,
				old_match_data_ptr->tfd,
				old_match_data_ptr);
	/*
	 * close the expected data file
	 */
	if (close(xwd_fd) == -1)
	{
		(void) fprintf(G_msg_file,
			       "error %d while closing expected data file\n",
			       errno);
		exit(1);
	}
	/*
	 * if a window was found for the new match data, then output it
	 */
	if (found_win_flag)
	{
		/*
		 * seek back to the beginning of the new match data
		 * in the temp file
		 */
		if (lseek(new_match_data_ptr->tfd, 0L, 0) == -1)
		{
			(void) fprintf(G_msg_file,
				       "error %d while seeking in new match data file\n",
				       errno);
			exit(1);
		}
		/*
		 * create the file name for the new (actual) data file
		 */
		(void) sprintf(xwd_file_name,
			       "%s.a%02d",
			       test_script_basename,
			       match_index);
		/*
		 * create the actual data file
		 */
		xwd_fd = makef(xwd_file_name);
		/*
		 * write the new match data out in xwd format to the actual data file
		 */
		write_match_data_to_xwd(xwd_fd,
					new_match_data_ptr->tfd,
					new_match_data_ptr);
		/*
		 * close the actual data file
		 */
		if (close(xwd_fd) == -1)
		{
			(void) fprintf(G_msg_file,
				       "error %d while closing actual data file\n",
				       errno);
			exit(1);
		}
	}
	else
	{
		/*
		 * output a failure message
		 */
		(void) fprintf(G_log_file,
			       "Unable to create an actual data file for match %d\n",
			       match_index);
		(void) fprintf(G_log_file,
			       "because no window corresponding to the old match data was found.\n");
	}
}

/**********************************************************************
 *
 *	 descriptions_match
 *
 *	Return non-zero (true) if the match data structures are the same.
 *	Return zero (false) if not.
 */
static int
descriptions_match(old_match_data_ptr, new_match_data_ptr)
/*
 * for match data in file
 */
struct match_data	*old_match_data_ptr;
/*
 * for new data from screen
 */
struct match_data	*new_match_data_ptr;
{
	/*
	 * check the screen_number parameter
	 */
	if (report_int_mismatch(old_match_data_ptr->screen_number,
				new_match_data_ptr->screen_number,
				"screen number"))
	{
		return(0);
	}
	/*
	 * check the window_x parameter
	 */
	/*
	if (report_int_mismatch(old_match_data_ptr->window_x,
				new_match_data_ptr->window_x,
				"window x"))
	{
		return(0);
	}
	*/
	/*
	 * check the window_y parameter
	 */
	/*
	if (report_int_mismatch(old_match_data_ptr->window_y,
				new_match_data_ptr->window_y,
				"window y"))
	{
		return(0);
	}
	*/
	/*
	 * check the window_width parameter
	 */
	if (report_int_mismatch(old_match_data_ptr->window_width,
				new_match_data_ptr->window_width,
				"window width"))
	{
		return(0);
	}
	/*
	 * check the window_height parameter
	 */
	if (report_int_mismatch(old_match_data_ptr->window_height,
				new_match_data_ptr->window_height,
				"window height"))
	{
		return(0);
	}
	/*
	 * check the window_border_width parameter
	 */
	if (report_int_mismatch(old_match_data_ptr->window_border_width,
				new_match_data_ptr->window_border_width,
				"window border width"))
	{
		return(0);
	}
	/*
	 * check the x parameter
	 */
	if (report_int_mismatch(old_match_data_ptr->x,
				new_match_data_ptr->x,
				"image data x"))
	{
		return(0);
	}
	/*
	 * check the y parameter
	 */
	if (report_int_mismatch(old_match_data_ptr->y,
				new_match_data_ptr->y,
				"image data y"))
	{
		return(0);
	}
	/*
	 * check the width parameter
	 */
	if (report_int_mismatch(old_match_data_ptr->width,
				new_match_data_ptr->width,
				"width"))
	{
		return(0);
	}
	/*
	 * check the height parameter
	 */
	if (report_int_mismatch(old_match_data_ptr->height,
				new_match_data_ptr->height,
				"height"))
	{
		return(0);
	}
	/*
	 * check the depth parameter
	 */
	if (report_int_mismatch(old_match_data_ptr->depth,
				new_match_data_ptr->depth,
				"depth"))
	{
		return(0);
	}
	/*
	 * check the format parameter
	 */
	if (report_int_mismatch(old_match_data_ptr->format,
				new_match_data_ptr->format,
				"format"))
	{
		return(0);
	}
	/*
	 * check the xoffset parameter
	 */
	if (report_int_mismatch(old_match_data_ptr->xoffset,
				new_match_data_ptr->xoffset,
				"xoffset"))
	{
		return(0);
	}
	/*
	 * check the byte_order parameter
	 */
	if (report_int_mismatch(old_match_data_ptr->byte_order,
				new_match_data_ptr->byte_order,
				"byte order"))
	{
		return(0);
	}
	/*
	 * check the bitmap_unit parameter
	 */
	if (report_int_mismatch(old_match_data_ptr->bitmap_unit,
				new_match_data_ptr->bitmap_unit,
				"bitmap unit"))
	{
		return(0);
	}
	/*
	 * check the bitmap_bit_order parameter
	 */
	if (report_int_mismatch(old_match_data_ptr->bitmap_bit_order,
				new_match_data_ptr->bitmap_bit_order,
				"bitmap bit order"))
	{
		return(0);
	}
	/*
	 * check the bitmap_pad parameter
	 */
	if (report_int_mismatch(old_match_data_ptr->bitmap_pad,
				new_match_data_ptr->bitmap_pad,
				"bitmap pad"))
	{
		return(0);
	}
	/*
	 * check the bytes_per_line parameter
	 */
	if (report_int_mismatch(old_match_data_ptr->bytes_per_line,
				new_match_data_ptr->bytes_per_line,
				"bytes per line"))
	{
		return(0);
	}
	/*
	 * check the bits_per_pixel parameter
	 */
	if (report_int_mismatch(old_match_data_ptr->bits_per_pixel,
				new_match_data_ptr->bits_per_pixel,
				"bits per pixel"))
	{
		return(0);
	}
	/*
	 * check the visual_class parameter
	 */
	if (report_int_mismatch(old_match_data_ptr->visual_class,
				new_match_data_ptr->visual_class,
				"visual class"))
	{
		return(0);
	}
	/*
	 * check the red_mask parameter
	 */
	if (report_ulong_mismatch(old_match_data_ptr->red_mask,
				  new_match_data_ptr->red_mask,
				  "red mask"))
	{
		return(0);
	}
	/*
	 * check the green_mask parameter
	 */
	if (report_ulong_mismatch(old_match_data_ptr->green_mask,
				  new_match_data_ptr->green_mask,
				  "green mask"))
	{
		return(0);
	}
	/*
	 * check the blue_mask parameter
	 */
	if (report_ulong_mismatch(old_match_data_ptr->blue_mask,
				  new_match_data_ptr->blue_mask,
				  "blue mask"))
	{
		return(0);
	}
	/*
	 * check the bits_per_rgb parameter
	 */
	if (report_int_mismatch(old_match_data_ptr->bits_per_rgb,
				new_match_data_ptr->bits_per_rgb,
				"bits per rgb"))
	{
		return(0);
	}
	/*
	 * check the colormap_entries parameter
	 */
	if (report_int_mismatch(old_match_data_ptr->colormap_entries,
				new_match_data_ptr->colormap_entries,
				"colormap entries"))
	{
		return(0);
	}
	/*
	 * check the ncolors parameter
	 */
	if (report_ulong_mismatch((unsigned long) old_match_data_ptr->ncolors,
				  (unsigned long) new_match_data_ptr->ncolors,
				  "number of colors"))
	{
		return(0);
	}
	/*
	 * check the window names
	 */
	if (strcmp(old_match_data_ptr->window_name,
		   new_match_data_ptr->window_name) != 0)
	{
		(void) fprintf(G_msg_file,
			       "match data %s was \"%s\", new data %s was \"%s\"\n",
			       "window name",
			       old_match_data_ptr->window_name,
			       "window name",
			       new_match_data_ptr->window_name);
		(void) fprintf(G_log_file,
			       "match data %s was \"%s\", new data %s was \"%s\"\n",
			       "window name",
			       old_match_data_ptr->window_name,
			       "window name",
			       new_match_data_ptr->window_name);
		return(0);
	}
	/*
	 * check the color structures
	 */
	if (check_color_structures(old_match_data_ptr->ncolors,
				   old_match_data_ptr->color_ptr,
				   new_match_data_ptr->color_ptr))
	{
		return(0);
	}
	if (!G_checksum_flag)
	{
		/*
		 * check the compressed image data count
		 */
		if (report_ulong_mismatch((unsigned long) old_match_data_ptr->cnt,
					  (unsigned long) new_match_data_ptr->cnt,
					  "image data count"))
		{
			return(0);
		}
	}
	/*
	 * if you made it to here, the descriptions match
	 */
	return(1);
}

/************************************************************************
 *
 *	report_int_mismatch
 *
 *	If the parameters don't match, report the mismatch and return 1.
 *	If the paramters match, return 0.
 */
static int
report_int_mismatch(old_match_data_value, new_match_data_value, param_name)
/*
 * holds the old match data parameter value
 */
int 	old_match_data_value;
/*
 * holds the new match data parameter value
 */
int	new_match_data_value;
/*
 * holds the name of the parameter
 */
char	*param_name;
{
	if (old_match_data_value != new_match_data_value)
	{
		(void) fprintf(G_msg_file,
			       "match data %s was %d, new data %s was %d\n",
			       param_name,
			       old_match_data_value,
			       param_name,
			       new_match_data_value);
		(void) fprintf(G_log_file,
			       "match data %s was %d, new data %s was %d\n",
			       param_name,
			       old_match_data_value,
			       param_name,
			       new_match_data_value);
		return(1);
	}
	else
	{
		return(0);
	}
}

/************************************************************************
 *
 *	report_ulong_mismatch
 *
 *	If the parameters don't match, report the mismatch and return 1.
 *	If the paramters match, return 0.
 */
static int
report_ulong_mismatch(old_match_data_value, new_match_data_value, param_name)
/*
 * holds the old match data parameter value
 */
unsigned long 	old_match_data_value;
/*
 * holds the new match data parameter value
 */
unsigned long	new_match_data_value;
/*
 * holds the name of the parameter
 */
char	*param_name;
{
	if (old_match_data_value != new_match_data_value)
	{
		(void) fprintf(G_msg_file,
			       "match data %s was %lu, new data %s was %lu\n",
			       param_name,
			       old_match_data_value,
			       param_name,
			       new_match_data_value);
		(void) fprintf(G_log_file,
			       "match data %s was %lu, new data %s was %lu\n",
			       param_name,
			       old_match_data_value,
			       param_name,
			       new_match_data_value);
		return(1);
	}
	else
	{
		return(0);
	}
}

/**********************************************************************
 *
 *	 check_color_structures
 *
 *	Return zero (false) if the match data color structures are
 *	the same.  Return non-zero (true) if not.
 */
static int
check_color_structures(color_cnt, old_color_ptr, new_color_ptr)
/*
 * used to step through the color structures
 */
INT32	color_cnt;
/*
 * used to step through the old match data color structures
 */
XColor	*old_color_ptr;
/*
 * used to step through the new match data color structures
 */
XColor	*new_color_ptr;
{
	/*
	 * used to step through the color structures
	 */
	INT32	i;

	/*
	 * compare one color struture at a time
	 */
	for (i = 0; i < color_cnt; i++)
	{
		/*
		 * compare the pixel value in the color structure
		 */
		if (report_color_mismatch(i,
					  old_color_ptr->pixel,
					  new_color_ptr->pixel,
					  "pixel value"))
		{
			return(1);
		}
		/*
		 * compare the red value in the color structure
		 */
		if (report_color_mismatch(i,
					  (unsigned long) old_color_ptr->red,
					  (unsigned long) new_color_ptr->red,
					  "red value"))
		{
			return(1);
		}
		/*
		 * compare the green value in the color structure
		 */
		if (report_color_mismatch(i,
					  (unsigned long) old_color_ptr->green,
					  (unsigned long) new_color_ptr->green,
					  "green value"))
		{
			return(1);
		}
		/*
		 * compare the blue value in the color structure
		 */
		if (report_color_mismatch(i,
					  (unsigned long) old_color_ptr->blue,
					  (unsigned long) new_color_ptr->blue,
					  "blue value"))
		{
			return(1);
		}
		/*
		 * compare the flags value in the color structure
		 */
		if (report_color_mismatch(i,
					  (unsigned long) old_color_ptr->flags,
					  (unsigned long) new_color_ptr->flags,
					  "flags value"))
		{
			return(1);
		}
		/*
		 * if they all matched, advance to the next color structure
		 */
		old_color_ptr++;
		new_color_ptr++;
	}
	/*
	 * if you made it to here, the color structures match
	 */
	return(0);
}

/************************************************************************
 *
 *	report_color_mismatch
 *
 *	If the paramters match, return 0.
 *
 *	If the parameters don't match and the compare_color_map flag is
 *	true, report the mismatch and return 1.
 *
 *	If the parameters don't match and the compare_color_map flag is
 *	false, put out warning messages and return 0.
 */
static int
report_color_mismatch(color_number,
		      old_color_struct_value,
		      new_color_struct_value,
		      param_name)
/*
 * holds the color structure number
 */
INT32		color_number;
/*
 * holds the old match data color structure parameter value
 */
unsigned long 	old_color_struct_value;
/*
 * holds the new match data color structure parameter value
 */
unsigned long	new_color_struct_value;
/*
 * holds the name of the parameter
 */
char	*param_name;
{
	if (old_color_struct_value != new_color_struct_value)
	{
		if (G_compare_color_map_flag)
		{
			(void) fprintf(G_msg_file,
				       "match data (color number %ld) %s was %d\n",
				       color_number,
				       param_name,
				       old_color_struct_value);
			(void) fprintf(G_msg_file,
				       "new data (color number %ld) %s was %d\n",
				       color_number,
				       param_name,
				       new_color_struct_value);
			(void) fprintf(G_log_file,
				       "match data (color number %ld) %s was %d\n",
				       color_number,
				       param_name,
				       old_color_struct_value);
			(void) fprintf(G_log_file,
				       "new data (color number %ld) %s was %d\n",
				       color_number,
				       param_name,
				       new_color_struct_value);
			return(1);
		}
		else
		{
			/* Id rather skip the warnings */
			/*
			(void) fprintf(G_msg_file,
				       "Warning: match data (color number %ld) %s was %d\n",
				       color_number,
				       param_name,
				       old_color_struct_value);
			(void) fprintf(G_msg_file,
				       "Warning: new data (color number %ld) %s was %d\n",
				       color_number,
				       param_name,
				       new_color_struct_value);
			(void) fprintf(G_log_file,
				       "Warning: match data (color number %ld) %s was %d\n",
				       color_number,
				       param_name,
				       old_color_struct_value);
			(void) fprintf(G_log_file,
				       "Warning: new data (color number %ld) %s was %d\n",
				       color_number,
				       param_name,
				       new_color_struct_value);
			*/
			return(0);
		}
	}
	else
	{
		return(0);
	}
}

/*****************************************************************
 *
 *	delay
 *
 *	This function delays the specified amount of milliseconds.
 *
 *	I know this is Berkeley-derived, but there is no way
 *	common to SYSV and Berkeley to do time delays more accurately
 *	than the nearest second.  If desired this could just
 *	round the value up to the next second and wait that long.
 */
static void
delay(milli_sec)
/*
 * milliseconds to wait
 */
INT32	milli_sec;
{
	/*
	 * holds the amount of time to delay for the select call
	 */
	struct timeval	pause;

	if (milli_sec == 0)
	{
		return;
	}
	/*
	 * get seconds part
	 */
	pause.tv_sec = milli_sec / 1000;
	/*
	 * get microseconds part
	 */
	pause.tv_usec = (milli_sec % 1000) * 1000;
	/*
	 * timeout on pause interval
	 */
	(void) select(0, (int *) 0, (int *) 0, (int *) 0, &pause);
}

/**********************************************************************
 *
 *      compare_data
 *
 *      Compares old and new match data in compressed format.  It
 *	returns 0 if the comparison fails and 1 if the comparison succeeds.
 *
 *	If the comparison fails, this routine also outputs the x and y
 *	offsets of the first pixel that doesn't match.
 */
static int
compare_data(old_match_data_fd, new_match_data_ptr)
/*
 * holds the file descriptor of the old match data
 */
int			old_match_data_fd;
/*
 * points to the new match data information structure
 */
struct match_data	*new_match_data_ptr;
{
        /*
         * holds the number of bytes of compressed data
	 * that can be read at one time
         */
        unsigned     size;
        /*
         * holds the number of bytes of compressed data left to be compared
         */
        long     compressed_cnt;
        /*
         * holds the number of bytes of uncompressed data that matched
         */
        long     byte_count;
        /*
         * used to step through the compressed data
         */
        long     i;
	/*
	 * holds the x offset of the failure point if the match fails
	 */
	int	x_failure_offset;
	/*
	 * holds the y offset of the failure point if the match fails
	 */
	int	y_failure_offset;

	/*
	 * reset the file pointer to the beginning of the new match data
	 */
        if (lseek(new_match_data_ptr->tfd, 0L, 0) == -1)
	{
		(void) fprintf(G_msg_file,
			       "error %d while seeking in new match data file\n",
			       errno);
		exit(1);
	}
	/*
	 * set the number of uncompressed data bytes that matched to 0
	 */
	byte_count = 0;
	/*
	 * start out with the total number of compressed bytes to compare
	 */
	compressed_cnt = new_match_data_ptr->cnt;
	/*
	 * loop, reading a chunk of compressed bytes at at time, until
	 * the comparison fails or were have compared all of the 
	 * compressed data
	 */
        while (compressed_cnt > 0)
        {
		/*
		 * if the number of compressed bytes left to compare won't
		 * fit in the ram buffers, then use the amount that will
		 */
                size = (compressed_cnt < G_sb_size) ? compressed_cnt :
						      G_sb_size;
		/*
		 * read some compressed bytes from the old match data
		 */
                readchk(old_match_data_fd,
			(char *) G_sb,
			size,
			"error while reading old match data\n");
		/*
		 * read some compressed bytes from the new match data
		 */
                readchk(new_match_data_ptr->tfd,
			(char *) G_c_data_buffer,
			size,
			"error while reading new match data\n");
		/*
		 * update our count of the number of compressed bytes
		 * left to be compared
		 */
                compressed_cnt -= size;
		/*
		 * compare the bytes in two-byte groups:  a byte of "count"
		 * followed by a byte of "value"
		 */
                for (i = 0; i < size; i += 2)
                {
			/*
			 * if the "value"s don't match, then drop out of 
			 * the comparison loop early
			 */
                        if (*(G_c_data_buffer + i + 1) != *(G_sb + i + 1))
                        {
                                break;
                        }
			/*
			 * if the "count"s don't match, then figure out
			 * how many bytes matched and update the byte_count
			 * before dropping out of the comparison loop early
			 */
                        if (*(G_c_data_buffer + i) != *(G_sb + i))
                        {
				/*
				 * the number of uncompressed bytes that
				 * matched is the lesser of the two "count"s
				 */
				byte_count += min((*(G_c_data_buffer + i)), (*(G_sb + i)));
                                break;
                        }
			/*
			 * the "count"s matched, so update the uncompressed
			 * byte count
			 */
			byte_count += (*(G_c_data_buffer + i));
                }
		/*
		 * If we got out of the compare loop early, then the
		 * comparison failed.
		 */
		if (i < size)
		{
			/*
			 * figure out the y offset inside the match data
			 * of the pixel that failed
			 */
			y_failure_offset = byte_count /
					   new_match_data_ptr->bytes_per_line;
			/*
			 * Figure out the x offset inside the match data 
			 * of the pixel that failed.  First get the number
			 * of bytes left on the current line.
			 */
			x_failure_offset = byte_count %
					   new_match_data_ptr->bytes_per_line;
			/*
			 * now convert the byte count to a bit count
			 */
			x_failure_offset *= 8;
			/*
			 * now convert the bit count to a pixel count
			 */
			x_failure_offset /= new_match_data_ptr->bits_per_pixel;
			/*
			 * output the failure point x and y offsets
			 */
			if (G_compare_color_map_flag)
				{
				(void) fprintf(G_log_file,
				       "Failure point x-offset was %d.\n",
				       x_failure_offset);
				(void) fprintf(G_log_file,
				       "Failure point y-offset was %d.\n",
				       y_failure_offset);
				return(0);
				}
			else
				return(1);
		}
        }
	/*
	 * if we got to here, then all of the bytes have been compared
	 * without a mismatch, so return success
	 */
	return(1);
}
