#!/usr/X/bin/wksh -motif

#
#	lpx:	Replacement for lp command.  Pops up an X window to allow you
#		to specify destination and other lp options.  Destination
#		may be a printer, file, or the DigiFAX facsimile system.
#
#	Rick Richardson, Digi Intl., rick@digibd.com
#

#
#	WKSH Programming notes:
#
#	- You cannot XtGetValues for menuHistory.  Instead, I set up
#	  a callback on each option menu button, which just sets
#	  a shell variable whenever the button is selected.
#	- You cannot use panedWindow - wksh doesn't know about it.
#

#########################################################################
#									#
#	Useful functions						#
#									#
#########################################################################
function postscript	# Return true or false if $* is a postscript file
{
	# A file is PS if it starts with "%!" or "^D%!"
	ctrl_d=`echo "\004\c"`
	cat $* | read a
        case "$a" in
        %!*)            return 0;;
        "$ctrl_d"%!*)   return 0;;
        *)              return 1;;
        esac
}

function pcl		# Return true or false if $* is a HP PLC file
{
	# A file is PCL if there are escape characters in the file
	esc=`cat $* | tr -d '[\001-\032][\034-\377]' | wc -c`
	if [ $esc -gt 0 ]; then return 0; else return 1; fi
}

#########################################################################
#									#
#	Callbacks 							#
#									#
#########################################################################
function ok_cb
{
	if [ "$Debug" = 1 ]; then
		cmd="echo lp"		# For debugging
	else
		cmd="lp"
	fi

	case "$Dest" in
	"")		;;
	DigiFAX)	do_fax;; 
	FILE)		do_file;; 
	*)		cmd="$cmd -d '$Dest'";;
	esac

	case "$Filetype" in
	ps)		cmd="$cmd -o ps";;
	pcl)		cmd="$cmd -o pcl";;
	esac

	XtGetValues $copies value:Copies
	if [ "$Copies" -lt 1 ]; then Copies=1; fi
	if [ "$Copies" != "" ]; then cmd="$cmd -n '$Copies'"; fi

	XtGetValues $title value:Title
	if [ "$Title" != "" ]; then cmd="$cmd -t '$Title'"; fi

	XtGetValues $other value:Other
	if [ "$Other" != "" ]; then cmd="$cmd $Other"; fi

	if [ "$TmpFile" != "" ]; then
		cmd="$cmd -c $Files"
		eval $cmd
		rm -f $TmpFile
	else
		cmd="$cmd $Files"
		eval $cmd
	fi
	exit 0
}

function cancel_cb
{
	if [ "$TmpFile" != "" ]; then
		rm -f $TmpFile
	fi
	exit
}

function destfile_cb
{
	if [ "$fsd" == "" ]; then
		XmCreateFileSelectionDialog fsd $topLevel fsd \
				dialogTitle:"Select File" \

		XtAddCallback $fsd okCallback fsd_ok_cb
		XtAddCallback $fsd cancelCallback "XtUnmanageChildren $fsd"
	fi

	XtManageChildren $fsd
}

function fsd_ok_cb
{
	typeset destFILENAME

	XtGetValues $fsd_TEXT value:Filename	# Get selected file

	Dest=FILE

	XtUnmanageChildren $destOptions

	XtCreateManagedWidget destFILENAME destFILENAME pushButton $destMenu \
			labelString:"$Filename" \
			activateCallback:"Dest=FILE; Filename=$Filename" \

	XtSetValues $destOptions menuHistory:$destFILENAME \

	XtManageChildren $destOptions

	XtUnmanageChildren $fsd
}

function new_copies
{
	XtGetValues $copies value:Copies
	((Copies=$Copies+$1))
	if [ "$Copies" -lt 1 ]; then Copies=1; fi
	XtSetValues $copies value:$Copies
	XtSetValues $copies cursorPosition:"${#Copies}"
}

function do_fax
{
	if [ "$TmpFile" != "" ]; then
		(xfaxsend $FFiles; rm -f $TmpFile) &
	else
		xfaxsend $FFiles &
	fi
	exit 0
}

function do_file
{
	if [ "$Filename" == "" ]; then
		return
	fi

	cat $Files > $Filename
	if [ "$TmpFile" != "" ]; then
		rm -f $TmpFile
	fi

	exit 0
}

#########################################################################
#									#
#	Program Starts here						#
#									#
#########################################################################

#
#	Parse the options accepted by "lp"
#
#	We ignore most, but you may want to add support for them yourself
#
Command="$0 $@"
case "$0" in
*.sh)	Debug=1;;
*)	Debug=0;;
esac
Copies=1
Other=
while getopts 'cd:f:H:mn:o:P:q:rsS:t:T:wy:' let
do
	case "$let" in
	c)	Copyit=1;;
	d)	Dest=$OPTARG;;
	f)	Form=$OPTARG;		Other="${Other}-f '$OPTARG' ";;
	H)	Handling=$OPTARG;	Other="${Other}-H '$OPTARG' ";;
	m)	Mail=1;			Other="${Other}-m ";;
	n)	Copies=$OPTARG;;
	o)	Options=$OPTARG;	Other="${Other}-o '$OPTARG' ";;
	P)	Pages=$OPTARG;		Other="${Other}-P '$OPTARG' ";;
	q)	Priority=$OPTARG;	Other="${Other}-q '$OPTARG' ";;
	r)	Reject=1;		Other="${Other}-r ";;
	s)	Supress=1;		Other="${Other}-s ";;
	S)	Priority=$OPTARG;	Other="${Other}-S '$OPTARG' ";;
	t)	Title=$OPTARG;;
	T)	Content=$OPTARG;	Other="${Other}-T '$OPTARG' ";;
	w)	Write=1;		Other="${Other}-w ";;
	y)	Modes=$OPTARG;		Other="${Other}-y '$OPTARG' ";;
	'?')	usage() ;;
	esac
done
shift `expr $OPTIND - 1`

if [ $# = 0 ]; then
	TmpFile=/tmp/lpx$$
	Files=$TmpFile
	FFiles="-f $TmpFile"
	cat > $Files
else
	TmpFile=
	Files=
	for i in $*
	do
		Files="$Files $i"
		FFiles="$FFiles -f $i"
	done
fi

#
#	Guess what type the file is
#
if postscript $Files; then
	Filetype=ps
elif pcl $Files; then
	Filetype=pcl
else
	Filetype=text
fi

#
#	Load the Table DSO, and load the xmpTable widget from the DSO
#
libload /usr/local/X/lib/Table.so
widload xmpTable

#
#	Create the widgets
#
XtAppInitialize topLevel lpx lpx "$@" \

XtSetValues $topLevel title:"$Command"

#
#	Create a control area and an action area, with a separator
#		NOTE: panedWindow is missing in the current wksh, so we use
#		a form, instead.
#
XmCreateForm pane $topLevel pane

	# This order of creation allows the form to stretch the way we want
	# it to (action area never grows in height)

XtCreateWidget aa aa form $pane  \
				leftAttachment:attach_form \
				rightAttachment:attach_form \
				bottomAttachment:attach_form \
				fractionBase:59 \
				# Fraction base is 20*nbuttons-1

XtCreateManagedWidget panesep panesep separator $pane \
				leftAttachment:attach_form \
				rightAttachment:attach_form \
				bottomAttachment:attach_widget \
				bottomWidget:$aa \

XtCreateManagedWidget ca ca xmpTable $pane \
				topAttachment:attach_form \
				leftAttachment:attach_form \
				rightAttachment:attach_form \
				bottomAttachment:attach_widget \
				bottomWidget:$panesep \

#
#	Create the destination option menu
#
XtCreateManagedWidget destLabel destLabel label $ca \
				labelString:"Destination" \

XmCreatePulldownMenu destMenu $ca destMenu \

XtCreateManagedWidget destDefault destDefault pushButton $destMenu \
				labelString:"System Default Printer" \
				activateCallback:"Dest=" \

XtCreateManagedWidget destFAX destFAX pushButton $destMenu \
				labelString:"FAX it using DigiFAX" \
				activateCallback:"Dest=DigiFAX" \

XtCreateManagedWidget destFILE destFILE pushButton $destMenu \
				labelString:"Print to File" \
				activateCallback:"destfile_cb" \

XtCreateManagedWidget destSEP destSEP separator $destMenu \

destHistory=$destDefault
lpstat -p | awk '{print $2}' |
while read name
do
	XtCreateManagedWidget printer printer pushButton $destMenu \
				labelString:"$name" \
				activateCallback:"Dest=$name" \

	if [ "$name" = "$Dest" ]; then
		destHistory=$printer
	fi
done

XmCreateOptionMenu destOptions $ca destOptions \
				subMenuId:$destMenu \
				menuHistory:$destHistory \

XtManageChildren $destOptions

#
#	Create the file type option menu
#
XtCreateManagedWidget typeLabel typeLabel label $ca \
				labelString:"File Contents" \

XmCreatePulldownMenu typeMenu $ca typeMenu

XtCreateManagedWidget textContents textContents pushButton $typeMenu \
				labelString:"Plain Text" \
				activateCallback:"Filetype=text" \

XtCreateManagedWidget psContents psContents pushButton $typeMenu \
				labelString:"PostScript" \
				activateCallback:"Filetype=ps" \

XtCreateManagedWidget pclContents pclContents pushButton $typeMenu \
				labelString:"HP LaserJet PCL" \
				activateCallback:"Filetype=pcl" \

eval typeHistory=\$${Filetype}Contents

XmCreateOptionMenu typeOptions $ca typeOptions \
				subMenuId:$typeMenu \
				menuHistory:$typeHistory \

XtManageChildren $typeOptions

#
#	Create the copies field
#
XtCreateManagedWidget copiesLabel copiesLabel label $ca \
				labelString:"Copies" \

XtCreateManagedWidget copiesForm copiesForm rowColumn $ca \
				orientation:horizontal \

XtCreateManagedWidget copies copies textField $copiesForm \
				value:"$Copies" \
				cursorPosition:"${#Copies}" \
				columns:2 \

XtCreateManagedWidget copiesDown copiesDown arrowButton $copiesForm \
				arrowDirection:arrow_down \
				activateCallback:"new_copies -1" \

XtCreateManagedWidget copiesUp copiesUp arrowButton $copiesForm \
				arrowDirection:arrow_up \
				activateCallback:"new_copies +1" \

#
#	Create the title field
#
XtCreateManagedWidget titleLabel titleLabel label $ca \
				labelString:"Title" \

XtCreateManagedWidget title title textField $ca \
				value:"$Title" \
				cursorPosition:"${#Title}" \
				columns:40 \

#
#	Create the other options field
#
XtCreateManagedWidget otherLabel otherLabel label $ca \
				labelString:"Other Options" \

XtCreateManagedWidget other other textField $ca \
				value:"$Other" \
				cursorPosition:"${#Other}" \
				columns:40 \

#
#	Some bogus widgets to allow stretching the way we want it
#
XtCreateManagedWidget rowdummy1 rowdummy1 form $ca
XtCreateManagedWidget rowdummy2 rowdummy2 form $ca

XtCreateManagedWidget coldummy coldummy form $ca
XtSetValues $coldummy	width:100

#
#	Now, specify the layout of the Table
#
#	Name		Col	Row	ColSpan	RowSpan	Options
XtSetValues $ca layout:"
	destLabel	0	1	1	1	rwWh;
	destOptions	2	1	1	1	lh;
	typeLabel	0	2	1	1	rwWh;
	typeOptions	2	2	1	1	lwh;
	copiesLabel	0	3	1	1	rwWh;
	copiesForm	2	3	1	1	lwh;
	titleLabel	0	4	1	1	rwWh;
	title		2	4	1	1	lwh;
	otherLabel	0	5	1	1	rwWh;
	other		2	5	1	1	lwh;
	coldummy1	1	0	1	7	wW;
	rowdummy1	0	0	2	1	;
	rowdummy2	0	6	2	1	lw;
"
#	Name		Col	Row	ColSpan	RowSpan	Options

#
#	Create the action area buttons
#
XtCreateManagedWidget okButton okButton pushButton $aa \
				labelString:"OK" \
				activateCallback:ok_cb \
				topAttachment:attach_form \
				bottomAttachment:attach_form \
				leftAttachment:attach_form \
				rightAttachment:attach_position \
				rightPosition:19 \

XtCreateManagedWidget cancelButton cancelButton pushButton $aa \
				labelString:"Cancel" \
				activateCallback:cancel_cb \
				topAttachment:attach_form \
				bottomAttachment:attach_form \
				leftAttachment:attach_position \
				leftPosition:20 \
				rightAttachment:attach_position \
				rightPosition:39 \

XtCreateManagedWidget helpButton helpButton pushButton $aa \
				labelString:"Help" \
				topAttachment:attach_form \
				bottomAttachment:attach_form \
				leftAttachment:attach_position \
				leftPosition:40 \
				rightAttachment:attach_form \

#
#	The main loop
#
XtManageChildren $ca
XtManageChildren $aa
XtManageChildren $pane
XtRealizeWidget $topLevel
XtMainLoop
